"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const codepipeline = require("../../../aws-codepipeline"); // Automatically re-written from '@aws-cdk/aws-codepipeline'
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const action_1 = require("../action");
const common_1 = require("../common");
/**
 * CodePipeline Action to deploy an ECS Service.
 */
class EcsDeployAction extends action_1.Action {
    constructor(props) {
        super({
            ...props,
            category: codepipeline.ActionCategory.DEPLOY,
            provider: 'ECS',
            artifactBounds: common_1.deployArtifactBounds(),
            inputs: [determineInputArtifact(props)],
            resource: props.service,
        });
        this.props = props;
    }
    bound(_scope, _stage, options) {
        // permissions based on CodePipeline documentation:
        // https://docs.aws.amazon.com/codepipeline/latest/userguide/how-to-custom-role.html#how-to-update-role-new-services
        options.role.addToPolicy(new iam.PolicyStatement({
            actions: [
                'ecs:DescribeServices',
                'ecs:DescribeTaskDefinition',
                'ecs:DescribeTasks',
                'ecs:ListTasks',
                'ecs:RegisterTaskDefinition',
                'ecs:UpdateService',
            ],
            resources: ['*'],
        }));
        options.role.addToPolicy(new iam.PolicyStatement({
            actions: ['iam:PassRole'],
            resources: ['*'],
            conditions: {
                StringEqualsIfExists: {
                    'iam:PassedToService': [
                        'ec2.amazonaws.com',
                        'ecs-tasks.amazonaws.com',
                    ],
                },
            },
        }));
        options.bucket.grantRead(options.role);
        return {
            configuration: {
                ClusterName: this.props.service.cluster.clusterName,
                ServiceName: this.props.service.serviceName,
                FileName: this.props.imageFile && this.props.imageFile.fileName,
            },
        };
    }
}
exports.EcsDeployAction = EcsDeployAction;
function determineInputArtifact(props) {
    if (props.imageFile && props.input) {
        throw new Error("Exactly one of 'input' or 'imageFile' can be provided in the ECS deploy Action");
    }
    if (props.imageFile) {
        return props.imageFile.artifact;
    }
    if (props.input) {
        return props.input;
    }
    throw new Error("Specifying one of 'input' or 'imageFile' is required for the ECS deploy Action");
}
//# sourceMappingURL=data:application/json;base64,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