"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const codebuild = require("../../../aws-codebuild"); // Automatically re-written from '@aws-cdk/aws-codebuild'
const codepipeline = require("../../../aws-codepipeline"); // Automatically re-written from '@aws-cdk/aws-codepipeline'
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const cdk = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const __1 = require("..");
const action_1 = require("../action");
/**
 * The type of the CodeBuild action that determines its CodePipeline Category -
 * Build, or Test.
 * The default is Build.
 */
var CodeBuildActionType;
(function (CodeBuildActionType) {
    /**
     * The action will have the Build Category.
     * This is the default.
     */
    CodeBuildActionType[CodeBuildActionType["BUILD"] = 0] = "BUILD";
    /**
     * The action will have the Test Category.
     */
    CodeBuildActionType[CodeBuildActionType["TEST"] = 1] = "TEST";
})(CodeBuildActionType = exports.CodeBuildActionType || (exports.CodeBuildActionType = {}));
/**
 * CodePipeline build action that uses AWS CodeBuild.
 */
class CodeBuildAction extends action_1.Action {
    constructor(props) {
        super({
            ...props,
            category: props.type === CodeBuildActionType.TEST
                ? codepipeline.ActionCategory.TEST
                : codepipeline.ActionCategory.BUILD,
            provider: 'CodeBuild',
            artifactBounds: { minInputs: 1, maxInputs: 5, minOutputs: 0, maxOutputs: 5 },
            inputs: [props.input, ...props.extraInputs || []],
            resource: props.project,
        });
        this.props = props;
    }
    /**
     * Reference a CodePipeline variable defined by the CodeBuild project this action points to.
     * Variables in CodeBuild actions are defined using the 'exported-variables' subsection of the 'env'
     * section of the buildspec.
     *
     * @param variableName the name of the variable to reference.
     *   A variable by this name must be present in the 'exported-variables' section of the buildspec
     *
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/build-spec-ref.html#build-spec-ref-syntax
     */
    variable(variableName) {
        return this.variableExpression(variableName);
    }
    bound(scope, _stage, options) {
        // check for a cross-account action if there are any outputs
        if ((this.actionProperties.outputs || []).length > 0) {
            const pipelineStack = cdk.Stack.of(scope);
            const projectStack = cdk.Stack.of(this.props.project);
            if (pipelineStack.account !== projectStack.account) {
                throw new Error('A cross-account CodeBuild action cannot have outputs. ' +
                    'This is a known CodeBuild limitation. ' +
                    'See https://github.com/aws/aws-cdk/issues/4169 for details');
            }
        }
        // grant the Pipeline role the required permissions to this Project
        options.role.addToPolicy(new iam.PolicyStatement({
            resources: [this.props.project.projectArn],
            actions: [
                'codebuild:BatchGetBuilds',
                'codebuild:StartBuild',
                'codebuild:StopBuild',
            ],
        }));
        // allow the Project access to the Pipeline's artifact Bucket
        // but only if the project is not imported
        // (ie., has a role) - otherwise, the IAM library throws an error
        if (this.props.project.role) {
            if ((this.actionProperties.outputs || []).length > 0) {
                options.bucket.grantReadWrite(this.props.project);
            }
            else {
                options.bucket.grantRead(this.props.project);
            }
        }
        if (this.props.project instanceof codebuild.Project) {
            this.props.project.bindToCodePipeline(scope, {
                artifactBucket: options.bucket,
            });
        }
        // if any of the inputs come from the BitBucketSourceAction
        // with codeBuildCloneOutput=true,
        // grant the Project's Role to use the connection
        for (const inputArtifact of this.actionProperties.inputs || []) {
            const connectionArn = inputArtifact.getMetadata(__1.BitBucketSourceAction._CONNECTION_ARN_PROPERTY);
            if (connectionArn) {
                this.props.project.addToRolePolicy(new iam.PolicyStatement({
                    actions: ['codestar-connections:UseConnection'],
                    resources: [connectionArn],
                }));
            }
        }
        const configuration = {
            ProjectName: this.props.project.projectName,
            EnvironmentVariables: this.props.environmentVariables &&
                cdk.Stack.of(scope).toJsonString(codebuild.Project.serializeEnvVariables(this.props.environmentVariables)),
        };
        if ((this.actionProperties.inputs || []).length > 1) {
            // lazy, because the Artifact name might be generated lazily
            configuration.PrimarySource = cdk.Lazy.stringValue({ produce: () => this.props.input.artifactName });
        }
        return {
            configuration,
        };
    }
}
exports.CodeBuildAction = CodeBuildAction;
//# sourceMappingURL=data:application/json;base64,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