"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cloudformation = require("../../../aws-cloudformation"); // Automatically re-written from '@aws-cdk/aws-cloudformation'
const codepipeline = require("../../../aws-codepipeline"); // Automatically re-written from '@aws-cdk/aws-codepipeline'
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const cdk = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const action_1 = require("../action");
/**
 * Base class for Actions that execute CloudFormation
 */
class CloudFormationAction extends action_1.Action {
    constructor(props, inputs) {
        super({
            ...props,
            provider: 'CloudFormation',
            category: codepipeline.ActionCategory.DEPLOY,
            artifactBounds: {
                minInputs: 0,
                maxInputs: 10,
                minOutputs: 0,
                maxOutputs: 1,
            },
            inputs,
            outputs: props.outputFileName
                ? [props.output || new codepipeline.Artifact(`${props.actionName}_${props.stackName}_Artifact`)]
                : undefined,
        });
        this.props = props;
    }
    bound(_scope, _stage, options) {
        const singletonPolicy = SingletonPolicy.forRole(options.role);
        if ((this.actionProperties.outputs || []).length > 0) {
            options.bucket.grantReadWrite(singletonPolicy);
        }
        else if ((this.actionProperties.inputs || []).length > 0) {
            options.bucket.grantRead(singletonPolicy);
        }
        return {
            configuration: {
                StackName: this.props.stackName,
                OutputFileName: this.props.outputFileName,
            },
        };
    }
}
/**
 * CodePipeline action to execute a prepared change set.
 */
class CloudFormationExecuteChangeSetAction extends CloudFormationAction {
    constructor(props) {
        super(props, undefined);
        this.props2 = props;
    }
    bound(scope, stage, options) {
        SingletonPolicy.forRole(options.role).grantExecuteChangeSet(this.props2);
        const actionConfig = super.bound(scope, stage, options);
        return {
            ...actionConfig,
            configuration: {
                ...actionConfig.configuration,
                ActionMode: 'CHANGE_SET_EXECUTE',
                ChangeSetName: this.props2.changeSetName,
            },
        };
    }
}
exports.CloudFormationExecuteChangeSetAction = CloudFormationExecuteChangeSetAction;
// tslint:enable:max-line-length
/**
 * Base class for all CloudFormation actions that execute or stage deployments.
 */
class CloudFormationDeployAction extends CloudFormationAction {
    constructor(props, inputs) {
        super(props, (props.extraInputs || []).concat(inputs || []));
        this.props2 = props;
    }
    /**
     * Add statement to the service role assumed by CloudFormation while executing this action.
     */
    addToDeploymentRolePolicy(statement) {
        return this.getDeploymentRole('method addToRolePolicy()').addToPolicy(statement);
    }
    get deploymentRole() {
        return this.getDeploymentRole('property role()');
    }
    bound(scope, stage, options) {
        if (this.props2.deploymentRole) {
            this._deploymentRole = this.props2.deploymentRole;
        }
        else {
            const roleStack = cdk.Stack.of(options.role);
            const pipelineStack = cdk.Stack.of(scope);
            if (roleStack.account !== pipelineStack.account) {
                // pass role is not allowed for cross-account access - so,
                // create the deployment Role in the other account!
                this._deploymentRole = new iam.Role(roleStack, `${stage.pipeline.node.uniqueId}-${stage.stageName}-${this.actionProperties.actionName}-DeploymentRole`, {
                    assumedBy: new iam.ServicePrincipal('cloudformation.amazonaws.com'),
                    roleName: cdk.PhysicalName.GENERATE_IF_NEEDED,
                });
            }
            else {
                this._deploymentRole = new iam.Role(scope, 'Role', {
                    assumedBy: new iam.ServicePrincipal('cloudformation.amazonaws.com'),
                });
            }
            // the deployment role might need read access to the pipeline's bucket
            // (for example, if it's deploying a Lambda function),
            // and even if it has admin permissions, it won't be enough,
            // as it needs to be added to the key's resource policy
            // (and the bucket's, if the access is cross-account)
            options.bucket.grantRead(this._deploymentRole);
            if (this.props2.adminPermissions) {
                this._deploymentRole.addToPolicy(new iam.PolicyStatement({
                    actions: ['*'],
                    resources: ['*'],
                }));
            }
        }
        SingletonPolicy.forRole(options.role).grantPassRole(this._deploymentRole);
        const capabilities = this.props2.adminPermissions && this.props2.capabilities === undefined
            ? [cloudformation.CloudFormationCapabilities.NAMED_IAM]
            : this.props2.capabilities;
        const actionConfig = super.bound(scope, stage, options);
        return {
            ...actionConfig,
            configuration: {
                ...actionConfig.configuration,
                // None evaluates to empty string which is falsey and results in undefined
                Capabilities: parseCapabilities(capabilities),
                RoleArn: this.deploymentRole.roleArn,
                ParameterOverrides: cdk.Stack.of(scope).toJsonString(this.props2.parameterOverrides),
                TemplateConfiguration: this.props2.templateConfiguration
                    ? this.props2.templateConfiguration.location
                    : undefined,
                StackName: this.props2.stackName,
            },
        };
    }
    getDeploymentRole(member) {
        if (this._deploymentRole) {
            return this._deploymentRole;
        }
        else {
            throw new Error(`Cannot use the ${member} before the Action has been added to a Pipeline`);
        }
    }
}
/**
 * CodePipeline action to prepare a change set.
 *
 * Creates the change set if it doesn't exist based on the stack name and template that you submit.
 * If the change set exists, AWS CloudFormation deletes it, and then creates a new one.
 */
class CloudFormationCreateReplaceChangeSetAction extends CloudFormationDeployAction {
    constructor(props) {
        super(props, props.templateConfiguration
            ? [props.templatePath.artifact, props.templateConfiguration.artifact]
            : [props.templatePath.artifact]);
        this.props3 = props;
    }
    bound(scope, stage, options) {
        // the super call order is to preserve the existing order of statements in policies
        const actionConfig = super.bound(scope, stage, options);
        SingletonPolicy.forRole(options.role).grantCreateReplaceChangeSet(this.props3);
        return {
            ...actionConfig,
            configuration: {
                ...actionConfig.configuration,
                ActionMode: 'CHANGE_SET_REPLACE',
                ChangeSetName: this.props3.changeSetName,
                TemplatePath: this.props3.templatePath.location,
            },
        };
    }
}
exports.CloudFormationCreateReplaceChangeSetAction = CloudFormationCreateReplaceChangeSetAction;
/**
 * CodePipeline action to deploy a stack.
 *
 * Creates the stack if the specified stack doesn't exist. If the stack exists,
 * AWS CloudFormation updates the stack. Use this action to update existing
 * stacks.
 *
 * AWS CodePipeline won't replace the stack, and will fail deployment if the
 * stack is in a failed state. Use `ReplaceOnFailure` for an action that
 * will delete and recreate the stack to try and recover from failed states.
 *
 * Use this action to automatically replace failed stacks without recovering or
 * troubleshooting them. You would typically choose this mode for testing.
 */
class CloudFormationCreateUpdateStackAction extends CloudFormationDeployAction {
    constructor(props) {
        super(props, props.templateConfiguration
            ? [props.templatePath.artifact, props.templateConfiguration.artifact]
            : [props.templatePath.artifact]);
        this.props3 = props;
    }
    bound(scope, stage, options) {
        // the super call order is to preserve the existing order of statements in policies
        const actionConfig = super.bound(scope, stage, options);
        SingletonPolicy.forRole(options.role).grantCreateUpdateStack(this.props3);
        return {
            ...actionConfig,
            configuration: {
                ...actionConfig.configuration,
                ActionMode: this.props3.replaceOnFailure ? 'REPLACE_ON_FAILURE' : 'CREATE_UPDATE',
                TemplatePath: this.props3.templatePath.location,
            },
        };
    }
}
exports.CloudFormationCreateUpdateStackAction = CloudFormationCreateUpdateStackAction;
/**
 * CodePipeline action to delete a stack.
 *
 * Deletes a stack. If you specify a stack that doesn't exist, the action completes successfully
 * without deleting a stack.
 */
class CloudFormationDeleteStackAction extends CloudFormationDeployAction {
    constructor(props) {
        super(props, undefined);
        this.props3 = props;
    }
    bound(scope, stage, options) {
        // the super call order is to preserve the existing order of statements in policies
        const actionConfig = super.bound(scope, stage, options);
        SingletonPolicy.forRole(options.role).grantDeleteStack(this.props3);
        return {
            ...actionConfig,
            configuration: {
                ...actionConfig.configuration,
                ActionMode: 'DELETE_ONLY',
            },
        };
    }
}
exports.CloudFormationDeleteStackAction = CloudFormationDeleteStackAction;
/**
 * Manages a bunch of singleton-y statements on the policy of an IAM Role.
 * Dedicated methods can be used to add specific permissions to the role policy
 * using as few statements as possible (adding resources to existing compatible
 * statements instead of adding new statements whenever possible).
 *
 * Statements created outside of this class are not considered when adding new
 * permissions.
 */
class SingletonPolicy extends cdk.Construct {
    constructor(role) {
        super(role, SingletonPolicy.UUID);
        this.role = role;
        this.statements = {};
        this.grantPrincipal = role;
    }
    /**
     * Obtain a SingletonPolicy for a given role.
     * @param role the Role this policy is bound to.
     * @returns the SingletonPolicy for this role.
     */
    static forRole(role) {
        const found = role.node.tryFindChild(SingletonPolicy.UUID);
        return found || new SingletonPolicy(role);
    }
    grantExecuteChangeSet(props) {
        this.statementFor({
            actions: [
                'cloudformation:DescribeStacks',
                'cloudformation:DescribeChangeSet',
                'cloudformation:ExecuteChangeSet',
            ],
            conditions: { StringEqualsIfExists: { 'cloudformation:ChangeSetName': props.changeSetName } },
        }).addResources(this.stackArnFromProps(props));
    }
    grantCreateReplaceChangeSet(props) {
        this.statementFor({
            actions: [
                'cloudformation:CreateChangeSet',
                'cloudformation:DeleteChangeSet',
                'cloudformation:DescribeChangeSet',
                'cloudformation:DescribeStacks',
            ],
            conditions: { StringEqualsIfExists: { 'cloudformation:ChangeSetName': props.changeSetName } },
        }).addResources(this.stackArnFromProps(props));
    }
    grantCreateUpdateStack(props) {
        const actions = [
            'cloudformation:DescribeStack*',
            'cloudformation:CreateStack',
            'cloudformation:UpdateStack',
            'cloudformation:GetTemplate*',
            'cloudformation:ValidateTemplate',
            'cloudformation:GetStackPolicy',
            'cloudformation:SetStackPolicy',
        ];
        if (props.replaceOnFailure) {
            actions.push('cloudformation:DeleteStack');
        }
        this.statementFor({ actions }).addResources(this.stackArnFromProps(props));
    }
    grantDeleteStack(props) {
        this.statementFor({
            actions: [
                'cloudformation:DescribeStack*',
                'cloudformation:DeleteStack',
            ],
        }).addResources(this.stackArnFromProps(props));
    }
    grantPassRole(role) {
        this.statementFor({ actions: ['iam:PassRole'] }).addResources(role.roleArn);
    }
    statementFor(template) {
        const key = keyFor(template);
        if (!(key in this.statements)) {
            this.statements[key] = new iam.PolicyStatement({ actions: template.actions });
            if (template.conditions) {
                this.statements[key].addConditions(template.conditions);
            }
            this.role.addToPolicy(this.statements[key]);
        }
        return this.statements[key];
        function keyFor(props) {
            const actions = `${props.actions.sort().join('\x1F')}`;
            const conditions = formatConditions(props.conditions);
            return `${actions}\x1D${conditions}`;
            function formatConditions(cond) {
                if (cond == null) {
                    return '';
                }
                let result = '';
                for (const op of Object.keys(cond).sort()) {
                    result += `${op}\x1E`;
                    const condition = cond[op];
                    for (const attribute of Object.keys(condition).sort()) {
                        const value = condition[attribute];
                        result += `${value}\x1F`;
                    }
                }
                return result;
            }
        }
    }
    stackArnFromProps(props) {
        return cdk.Stack.of(this).formatArn({
            region: props.region,
            service: 'cloudformation',
            resource: 'stack',
            resourceName: `${props.stackName}/*`,
        });
    }
}
SingletonPolicy.UUID = '8389e75f-0810-4838-bf64-d6f85a95cf83';
function parseCapabilities(capabilities) {
    if (capabilities === undefined) {
        return undefined;
    }
    else if (capabilities.length === 1) {
        const capability = capabilities.toString();
        return (capability === '') ? undefined : capability;
    }
    else if (capabilities.length > 1) {
        return capabilities.join(',');
    }
    return undefined;
}
//# sourceMappingURL=data:application/json;base64,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