"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const ec2 = require("../../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const s3 = require("../../../aws-s3"); // Automatically re-written from '@aws-cdk/aws-s3'
const cdk = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const codedeploy_generated_1 = require("../codedeploy.generated");
const utils_1 = require("../utils");
const application_1 = require("./application");
const deployment_config_1 = require("./deployment-config");
const load_balancer_1 = require("./load-balancer");
/**
 * Represents a reference to a CodeDeploy EC2/on-premise Deployment Group.
 *
 * If you're managing the Deployment Group alongside the rest of your CDK resources,
 * use the {@link ServerDeploymentGroup} class.
 *
 * If you want to reference an already existing Deployment Group,
 * or one defined in a different CDK Stack,
 * use the {@link #import} method.
 */
class ServerDeploymentGroupBase extends cdk.Resource {
    constructor(scope, id, deploymentConfig, props) {
        super(scope, id, props);
        this.deploymentConfig = deploymentConfig || deployment_config_1.ServerDeploymentConfig.ONE_AT_A_TIME;
    }
}
class ImportedServerDeploymentGroup extends ServerDeploymentGroupBase {
    constructor(scope, id, props) {
        super(scope, id, props.deploymentConfig);
        this.role = undefined;
        this.autoScalingGroups = undefined;
        this.application = props.application;
        this.deploymentGroupName = props.deploymentGroupName;
        this.deploymentGroupArn = utils_1.arnForDeploymentGroup(props.application.applicationName, props.deploymentGroupName);
    }
}
/**
 * Represents a set of instance tag groups.
 * An instance will match a set if it matches all of the groups in the set -
 * in other words, sets follow 'and' semantics.
 * You can have a maximum of 3 tag groups inside a set.
 */
class InstanceTagSet {
    constructor(...instanceTagGroups) {
        if (instanceTagGroups.length > 3) {
            throw new Error('An instance tag set can have a maximum of 3 instance tag groups, ' +
                `but ${instanceTagGroups.length} were provided`);
        }
        this._instanceTagGroups = instanceTagGroups;
    }
    get instanceTagGroups() {
        return this._instanceTagGroups.slice();
    }
}
exports.InstanceTagSet = InstanceTagSet;
/**
 * A CodeDeploy Deployment Group that deploys to EC2/on-premise instances.
 * @resource AWS::CodeDeploy::DeploymentGroup
 */
class ServerDeploymentGroup extends ServerDeploymentGroupBase {
    constructor(scope, id, props = {}) {
        super(scope, id, props.deploymentConfig, {
            physicalName: props.deploymentGroupName,
        });
        this.application = props.application || new application_1.ServerApplication(this, 'Application');
        this.role = props.role || new iam.Role(this, 'Role', {
            assumedBy: new iam.ServicePrincipal('codedeploy.amazonaws.com'),
            managedPolicies: [iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSCodeDeployRole')],
        });
        this._autoScalingGroups = props.autoScalingGroups || [];
        this.installAgent = props.installAgent === undefined ? true : props.installAgent;
        this.codeDeployBucket = s3.Bucket.fromBucketName(this, 'Bucket', `aws-codedeploy-${cdk.Stack.of(this).region}`);
        for (const asg of this._autoScalingGroups) {
            this.addCodeDeployAgentInstallUserData(asg);
        }
        this.alarms = props.alarms || [];
        const resource = new codedeploy_generated_1.CfnDeploymentGroup(this, 'Resource', {
            applicationName: this.application.applicationName,
            deploymentGroupName: this.physicalName,
            serviceRoleArn: this.role.roleArn,
            deploymentConfigName: props.deploymentConfig &&
                props.deploymentConfig.deploymentConfigName,
            autoScalingGroups: cdk.Lazy.listValue({ produce: () => this._autoScalingGroups.map(asg => asg.autoScalingGroupName) }, { omitEmpty: true }),
            loadBalancerInfo: this.loadBalancerInfo(props.loadBalancer),
            deploymentStyle: props.loadBalancer === undefined
                ? undefined
                : {
                    deploymentOption: 'WITH_TRAFFIC_CONTROL',
                },
            ec2TagSet: this.ec2TagSet(props.ec2InstanceTags),
            onPremisesTagSet: this.onPremiseTagSet(props.onPremiseInstanceTags),
            alarmConfiguration: cdk.Lazy.anyValue({ produce: () => utils_1.renderAlarmConfiguration(this.alarms, props.ignorePollAlarmsFailure) }),
            autoRollbackConfiguration: cdk.Lazy.anyValue({ produce: () => utils_1.renderAutoRollbackConfiguration(this.alarms, props.autoRollback) }),
        });
        this.deploymentGroupName = this.getResourceNameAttribute(resource.ref);
        this.deploymentGroupArn = this.getResourceArnAttribute(utils_1.arnForDeploymentGroup(this.application.applicationName, resource.ref), {
            service: 'codedeploy',
            resource: 'deploymentgroup',
            resourceName: `${this.application.applicationName}/${this.physicalName}`,
            sep: ':',
        });
    }
    /**
     * Import an EC2/on-premise Deployment Group defined either outside the CDK app,
     * or in a different region.
     *
     * @param scope the parent Construct for this new Construct
     * @param id the logical ID of this new Construct
     * @param attrs the properties of the referenced Deployment Group
     * @returns a Construct representing a reference to an existing Deployment Group
     */
    static fromServerDeploymentGroupAttributes(scope, id, attrs) {
        return new ImportedServerDeploymentGroup(scope, id, attrs);
    }
    /**
     * Adds an additional auto-scaling group to this Deployment Group.
     *
     * @param asg the auto-scaling group to add to this Deployment Group.
     * [disable-awslint:ref-via-interface] is needed in order to install the code
     * deploy agent by updating the ASGs user data.
     */
    addAutoScalingGroup(asg) {
        this._autoScalingGroups.push(asg);
        this.addCodeDeployAgentInstallUserData(asg);
    }
    /**
     * Associates an additional alarm with this Deployment Group.
     *
     * @param alarm the alarm to associate with this Deployment Group
     */
    addAlarm(alarm) {
        this.alarms.push(alarm);
    }
    get autoScalingGroups() {
        return this._autoScalingGroups.slice();
    }
    addCodeDeployAgentInstallUserData(asg) {
        if (!this.installAgent) {
            return;
        }
        this.codeDeployBucket.grantRead(asg.role, 'latest/*');
        switch (asg.osType) {
            case ec2.OperatingSystemType.LINUX:
                asg.addUserData('PKG_CMD=`which yum 2>/dev/null`', 'if [ -z "$PKG_CMD" ]; then', 'PKG_CMD=apt-get', 'else', 'PKG=CMD=yum', 'fi', '$PKG_CMD update -y', '$PKG_CMD install -y ruby2.0', 'if [ $? -ne 0 ]; then', '$PKG_CMD install -y ruby', 'fi', '$PKG_CMD install -y awscli', 'TMP_DIR=`mktemp -d`', 'cd $TMP_DIR', `aws s3 cp s3://aws-codedeploy-${cdk.Stack.of(this).region}/latest/install . --region ${cdk.Stack.of(this).region}`, 'chmod +x ./install', './install auto', 'rm -fr $TMP_DIR');
                break;
            case ec2.OperatingSystemType.WINDOWS:
                asg.addUserData('Set-Variable -Name TEMPDIR -Value (New-TemporaryFile).DirectoryName', `aws s3 cp s3://aws-codedeploy-${cdk.Stack.of(this).region}/latest/codedeploy-agent.msi $TEMPDIR\\codedeploy-agent.msi`, '$TEMPDIR\\codedeploy-agent.msi /quiet /l c:\\temp\\host-agent-install-log.txt');
                break;
        }
    }
    loadBalancerInfo(loadBalancer) {
        if (!loadBalancer) {
            return undefined;
        }
        switch (loadBalancer.generation) {
            case load_balancer_1.LoadBalancerGeneration.FIRST:
                return {
                    elbInfoList: [
                        { name: loadBalancer.name },
                    ],
                };
            case load_balancer_1.LoadBalancerGeneration.SECOND:
                return {
                    targetGroupInfoList: [
                        { name: loadBalancer.name },
                    ],
                };
        }
    }
    ec2TagSet(tagSet) {
        if (!tagSet || tagSet.instanceTagGroups.length === 0) {
            return undefined;
        }
        return {
            ec2TagSetList: tagSet.instanceTagGroups.map(tagGroup => {
                return {
                    ec2TagGroup: this.tagGroup2TagsArray(tagGroup),
                };
            }),
        };
    }
    onPremiseTagSet(tagSet) {
        if (!tagSet || tagSet.instanceTagGroups.length === 0) {
            return undefined;
        }
        return {
            onPremisesTagSetList: tagSet.instanceTagGroups.map(tagGroup => {
                return {
                    onPremisesTagGroup: this.tagGroup2TagsArray(tagGroup),
                };
            }),
        };
    }
    tagGroup2TagsArray(tagGroup) {
        const tagsInGroup = new Array();
        for (const tagKey in tagGroup) {
            if (tagGroup.hasOwnProperty(tagKey)) {
                const tagValues = tagGroup[tagKey];
                if (tagKey.length > 0) {
                    if (tagValues.length > 0) {
                        for (const tagValue of tagValues) {
                            tagsInGroup.push({
                                key: tagKey,
                                value: tagValue,
                                type: 'KEY_AND_VALUE',
                            });
                        }
                    }
                    else {
                        tagsInGroup.push({
                            key: tagKey,
                            type: 'KEY_ONLY',
                        });
                    }
                }
                else {
                    if (tagValues.length > 0) {
                        for (const tagValue of tagValues) {
                            tagsInGroup.push({
                                value: tagValue,
                                type: 'VALUE_ONLY',
                            });
                        }
                    }
                    else {
                        throw new Error('Cannot specify both an empty key and no values for an instance tag filter');
                    }
                }
            }
        }
        return tagsInGroup;
    }
}
exports.ServerDeploymentGroup = ServerDeploymentGroup;
//# sourceMappingURL=data:application/json;base64,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