"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const source_types_1 = require("./source-types");
/**
 * Source provider definition for a CodeBuild Project.
 */
class Source {
    constructor(props) {
        this.badgeSupported = false;
        this.identifier = props.identifier;
    }
    static s3(props) {
        return new S3Source(props);
    }
    static codeCommit(props) {
        return new CodeCommitSource(props);
    }
    static gitHub(props) {
        return new GitHubSource(props);
    }
    static gitHubEnterprise(props) {
        return new GitHubEnterpriseSource(props);
    }
    static bitBucket(props) {
        return new BitBucketSource(props);
    }
    /**
     * Called by the project when the source is added so that the source can perform
     * binding operations on the source. For example, it can grant permissions to the
     * code build project to read from the S3 bucket.
     */
    bind(_scope, _project) {
        return {
            sourceProperty: {
                sourceIdentifier: this.identifier,
                type: this.type,
            },
        };
    }
}
exports.Source = Source;
/**
 * A common superclass of all build sources that are backed by Git.
 */
class GitSource extends Source {
    constructor(props) {
        super(props);
        this.cloneDepth = props.cloneDepth;
        this.branchOrRef = props.branchOrRef;
    }
    bind(_scope, _project) {
        const superConfig = super.bind(_scope, _project);
        return {
            sourceVersion: this.branchOrRef,
            sourceProperty: {
                ...superConfig.sourceProperty,
                gitCloneDepth: this.cloneDepth,
            },
        };
    }
}
/**
 * The types of webhook event actions.
 */
var EventAction;
(function (EventAction) {
    /**
     * A push (of a branch, or a tag) to the repository.
     */
    EventAction["PUSH"] = "PUSH";
    /**
     * Creating a Pull Request.
     */
    EventAction["PULL_REQUEST_CREATED"] = "PULL_REQUEST_CREATED";
    /**
     * Updating a Pull Request.
     */
    EventAction["PULL_REQUEST_UPDATED"] = "PULL_REQUEST_UPDATED";
    /**
     * Merging a Pull Request.
     */
    EventAction["PULL_REQUEST_MERGED"] = "PULL_REQUEST_MERGED";
    /**
     * Re-opening a previously closed Pull Request.
     * Note that this event is only supported for GitHub and GitHubEnterprise sources.
     */
    EventAction["PULL_REQUEST_REOPENED"] = "PULL_REQUEST_REOPENED";
})(EventAction = exports.EventAction || (exports.EventAction = {}));
const FILE_PATH_WEBHOOK_COND = 'FILE_PATH';
/**
 * An object that represents a group of filter conditions for a webhook.
 * Every condition in a given FilterGroup must be true in order for the whole group to be true.
 * You construct instances of it by calling the {@link #inEventOf} static factory method,
 * and then calling various `andXyz` instance methods to create modified instances of it
 * (this class is immutable).
 *
 * You pass instances of this class to the `webhookFilters` property when constructing a source.
 */
class FilterGroup {
    constructor(actions, filters) {
        if (actions.size === 0) {
            throw new Error('A filter group must contain at least one event action');
        }
        this.actions = actions;
        this.filters = filters;
    }
    /**
     * Creates a new event FilterGroup that triggers on any of the provided actions.
     *
     * @param actions the actions to trigger the webhook on
     */
    static inEventOf(...actions) {
        return new FilterGroup(new Set(actions), []);
    }
    /**
     * Create a new FilterGroup with an added condition:
     * the event must affect the given branch.
     *
     * @param branchName the name of the branch (can be a regular expression)
     */
    andBranchIs(branchName) {
        return this.addHeadBranchFilter(branchName, true);
    }
    /**
     * Create a new FilterGroup with an added condition:
     * the event must not affect the given branch.
     *
     * @param branchName the name of the branch (can be a regular expression)
     */
    andBranchIsNot(branchName) {
        return this.addHeadBranchFilter(branchName, false);
    }
    /**
     * Create a new FilterGroup with an added condition:
     * the event must affect the given tag.
     *
     * @param tagName the name of the tag (can be a regular expression)
     */
    andTagIs(tagName) {
        return this.addHeadTagFilter(tagName, true);
    }
    /**
     * Create a new FilterGroup with an added condition:
     * the event must not affect the given tag.
     *
     * @param tagName the name of the tag (can be a regular expression)
     */
    andTagIsNot(tagName) {
        return this.addHeadTagFilter(tagName, false);
    }
    /**
     * Create a new FilterGroup with an added condition:
     * the event must affect a Git reference (ie., a branch or a tag)
     * that matches the given pattern.
     *
     * @param pattern a regular expression
     */
    andHeadRefIs(pattern) {
        return this.addHeadRefFilter(pattern, true);
    }
    /**
     * Create a new FilterGroup with an added condition:
     * the event must not affect a Git reference (ie., a branch or a tag)
     * that matches the given pattern.
     *
     * @param pattern a regular expression
     */
    andHeadRefIsNot(pattern) {
        return this.addHeadRefFilter(pattern, false);
    }
    /**
     * Create a new FilterGroup with an added condition:
     * the account ID of the actor initiating the event must match the given pattern.
     *
     * @param pattern a regular expression
     */
    andActorAccountIs(pattern) {
        return this.addActorAccountId(pattern, true);
    }
    /**
     * Create a new FilterGroup with an added condition:
     * the account ID of the actor initiating the event must not match the given pattern.
     *
     * @param pattern a regular expression
     */
    andActorAccountIsNot(pattern) {
        return this.addActorAccountId(pattern, false);
    }
    /**
     * Create a new FilterGroup with an added condition:
     * the Pull Request that is the source of the event must target the given base branch.
     * Note that you cannot use this method if this Group contains the `PUSH` event action.
     *
     * @param branchName the name of the branch (can be a regular expression)
     */
    andBaseBranchIs(branchName) {
        return this.addBaseBranchFilter(branchName, true);
    }
    /**
     * Create a new FilterGroup with an added condition:
     * the Pull Request that is the source of the event must not target the given base branch.
     * Note that you cannot use this method if this Group contains the `PUSH` event action.
     *
     * @param branchName the name of the branch (can be a regular expression)
     */
    andBaseBranchIsNot(branchName) {
        return this.addBaseBranchFilter(branchName, false);
    }
    /**
     * Create a new FilterGroup with an added condition:
     * the Pull Request that is the source of the event must target the given Git reference.
     * Note that you cannot use this method if this Group contains the `PUSH` event action.
     *
     * @param pattern a regular expression
     */
    andBaseRefIs(pattern) {
        return this.addBaseRefFilter(pattern, true);
    }
    /**
     * Create a new FilterGroup with an added condition:
     * the Pull Request that is the source of the event must not target the given Git reference.
     * Note that you cannot use this method if this Group contains the `PUSH` event action.
     *
     * @param pattern a regular expression
     */
    andBaseRefIsNot(pattern) {
        return this.addBaseRefFilter(pattern, false);
    }
    /**
     * Create a new FilterGroup with an added condition:
     * the push that is the source of the event must affect a file that matches the given pattern.
     * Note that you can only use this method if this Group contains only the `PUSH` event action,
     * and only for GitHub and GitHubEnterprise sources.
     *
     * @param pattern a regular expression
     */
    andFilePathIs(pattern) {
        return this.addFilePathFilter(pattern, true);
    }
    /**
     * Create a new FilterGroup with an added condition:
     * the push that is the source of the event must not affect a file that matches the given pattern.
     * Note that you can only use this method if this Group contains only the `PUSH` event action,
     * and only for GitHub and GitHubEnterprise sources.
     *
     * @param pattern a regular expression
     */
    andFilePathIsNot(pattern) {
        return this.addFilePathFilter(pattern, false);
    }
    /** @internal */
    get _actions() {
        return set2Array(this.actions);
    }
    /** @internal */
    get _filters() {
        return this.filters.slice();
    }
    /** @internal */
    _toJson() {
        const eventFilter = {
            type: 'EVENT',
            pattern: set2Array(this.actions).join(', '),
        };
        return [eventFilter].concat(this.filters);
    }
    addHeadBranchFilter(branchName, include) {
        return this.addHeadRefFilter(`refs/heads/${branchName}`, include);
    }
    addHeadTagFilter(tagName, include) {
        return this.addHeadRefFilter(`refs/tags/${tagName}`, include);
    }
    addHeadRefFilter(refName, include) {
        return this.addFilter('HEAD_REF', refName, include);
    }
    addActorAccountId(accountId, include) {
        return this.addFilter('ACTOR_ACCOUNT_ID', accountId, include);
    }
    addBaseBranchFilter(branchName, include) {
        return this.addBaseRefFilter(`refs/heads/${branchName}`, include);
    }
    addBaseRefFilter(refName, include) {
        if (this.actions.has(EventAction.PUSH)) {
            throw new Error('A base reference condition cannot be added if a Group contains a PUSH event action');
        }
        return this.addFilter('BASE_REF', refName, include);
    }
    addFilePathFilter(pattern, include) {
        if (this.actions.size !== 1 || !this.actions.has(EventAction.PUSH)) {
            throw new Error('A file path condition cannot be added if a Group contains any event action other than PUSH');
        }
        return this.addFilter(FILE_PATH_WEBHOOK_COND, pattern, include);
    }
    addFilter(type, pattern, include) {
        return new FilterGroup(this.actions, this.filters.concat([{
                type,
                pattern,
                excludeMatchedPattern: include ? undefined : true,
            }]));
    }
}
exports.FilterGroup = FilterGroup;
/**
 * A common superclass of all third-party build sources that are backed by Git.
 */
class ThirdPartyGitSource extends GitSource {
    constructor(props) {
        super(props);
        this.badgeSupported = true;
        this.webhook = props.webhook;
        this.reportBuildStatus = props.reportBuildStatus === undefined ? true : props.reportBuildStatus;
        this.webhookFilters = props.webhookFilters || [];
    }
    bind(_scope, _project) {
        const anyFilterGroupsProvided = this.webhookFilters.length > 0;
        const webhook = this.webhook === undefined ? (anyFilterGroupsProvided ? true : undefined) : this.webhook;
        const superConfig = super.bind(_scope, _project);
        return {
            sourceProperty: {
                ...superConfig.sourceProperty,
                reportBuildStatus: this.reportBuildStatus,
            },
            sourceVersion: superConfig.sourceVersion,
            buildTriggers: webhook === undefined ? undefined : {
                webhook,
                filterGroups: anyFilterGroupsProvided ? this.webhookFilters.map(fg => fg._toJson()) : undefined,
            },
        };
    }
}
/**
 * CodeCommit Source definition for a CodeBuild project.
 */
class CodeCommitSource extends GitSource {
    constructor(props) {
        super(props);
        this.badgeSupported = true;
        this.type = source_types_1.CODECOMMIT_SOURCE_TYPE;
        this.repo = props.repository;
    }
    bind(_scope, project) {
        // https://docs.aws.amazon.com/codebuild/latest/userguide/setting-up.html
        project.addToRolePolicy(new iam.PolicyStatement({
            actions: ['codecommit:GitPull'],
            resources: [this.repo.repositoryArn],
        }));
        const superConfig = super.bind(_scope, project);
        return {
            sourceProperty: {
                ...superConfig.sourceProperty,
                location: this.repo.repositoryCloneUrlHttp,
            },
            sourceVersion: superConfig.sourceVersion,
        };
    }
}
/**
 * S3 bucket definition for a CodeBuild project.
 */
class S3Source extends Source {
    constructor(props) {
        super(props);
        this.type = source_types_1.S3_SOURCE_TYPE;
        this.bucket = props.bucket;
        this.path = props.path;
        this.version = props.version;
    }
    bind(_scope, project) {
        this.bucket.grantRead(project);
        const superConfig = super.bind(_scope, project);
        return {
            sourceProperty: {
                ...superConfig.sourceProperty,
                location: `${this.bucket.bucketName}/${this.path}`,
            },
            sourceVersion: this.version,
        };
    }
}
/**
 * GitHub Source definition for a CodeBuild project.
 */
class GitHubSource extends ThirdPartyGitSource {
    constructor(props) {
        super(props);
        this.type = source_types_1.GITHUB_SOURCE_TYPE;
        this.httpsCloneUrl = `https://github.com/${props.owner}/${props.repo}.git`;
    }
    bind(_scope, project) {
        const superConfig = super.bind(_scope, project);
        return {
            sourceProperty: {
                ...superConfig.sourceProperty,
                location: this.httpsCloneUrl,
            },
            sourceVersion: superConfig.sourceVersion,
            buildTriggers: superConfig.buildTriggers,
        };
    }
}
/**
 * GitHub Enterprise Source definition for a CodeBuild project.
 */
class GitHubEnterpriseSource extends ThirdPartyGitSource {
    constructor(props) {
        super(props);
        this.type = source_types_1.GITHUB_ENTERPRISE_SOURCE_TYPE;
        this.httpsCloneUrl = props.httpsCloneUrl;
        this.ignoreSslErrors = props.ignoreSslErrors;
    }
    bind(_scope, _project) {
        const superConfig = super.bind(_scope, _project);
        return {
            sourceProperty: {
                ...superConfig.sourceProperty,
                location: this.httpsCloneUrl,
                insecureSsl: this.ignoreSslErrors,
            },
            sourceVersion: superConfig.sourceVersion,
            buildTriggers: superConfig.buildTriggers,
        };
    }
}
/**
 * BitBucket Source definition for a CodeBuild project.
 */
class BitBucketSource extends ThirdPartyGitSource {
    constructor(props) {
        super(props);
        this.type = source_types_1.BITBUCKET_SOURCE_TYPE;
        this.httpsCloneUrl = `https://bitbucket.org/${props.owner}/${props.repo}.git`;
    }
    bind(_scope, _project) {
        // BitBucket sources don't support the PULL_REQUEST_REOPENED event action
        if (this.anyWebhookFilterContainsPrReopenedEventAction()) {
            throw new Error('BitBucket sources do not support the PULL_REQUEST_REOPENED webhook event action');
        }
        // they also don't support file path conditions
        if (this.anyWebhookFilterContainsFilePathConditions()) {
            throw new Error('BitBucket sources do not support file path conditions for webhook filters');
        }
        const superConfig = super.bind(_scope, _project);
        return {
            sourceProperty: {
                ...superConfig.sourceProperty,
                location: this.httpsCloneUrl,
            },
            sourceVersion: superConfig.sourceVersion,
            buildTriggers: superConfig.buildTriggers,
        };
    }
    anyWebhookFilterContainsPrReopenedEventAction() {
        return this.webhookFilters.findIndex(fg => {
            return fg._actions.findIndex(a => a === EventAction.PULL_REQUEST_REOPENED) !== -1;
        }) !== -1;
    }
    anyWebhookFilterContainsFilePathConditions() {
        return this.webhookFilters.findIndex(fg => {
            return fg._filters.findIndex(f => f.type === FILE_PATH_WEBHOOK_COND) !== -1;
        }) !== -1;
    }
}
function set2Array(set) {
    const ret = [];
    set.forEach(el => ret.push(el));
    return ret;
}
//# sourceMappingURL=data:application/json;base64,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