"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
/**
 * Local cache modes to enable for the CodeBuild Project
 */
var LocalCacheMode;
(function (LocalCacheMode) {
    /**
     * Caches Git metadata for primary and secondary sources
     */
    LocalCacheMode["SOURCE"] = "LOCAL_SOURCE_CACHE";
    /**
     * Caches existing Docker layers
     */
    LocalCacheMode["DOCKER_LAYER"] = "LOCAL_DOCKER_LAYER_CACHE";
    /**
     * Caches directories you specify in the buildspec file
     */
    LocalCacheMode["CUSTOM"] = "LOCAL_CUSTOM_CACHE";
})(LocalCacheMode = exports.LocalCacheMode || (exports.LocalCacheMode = {}));
/**
 * Cache options for CodeBuild Project.
 * A cache can store reusable pieces of your build environment and use them across multiple builds.
 * @see https://docs.aws.amazon.com/codebuild/latest/userguide/build-caching.html
 */
class Cache {
    static none() {
        return { _toCloudFormation: () => undefined, _bind: () => { return; } };
    }
    /**
     * Create a local caching strategy.
     * @param modes the mode(s) to enable for local caching
     */
    static local(...modes) {
        return {
            _toCloudFormation: () => ({
                type: 'LOCAL',
                modes,
            }),
            _bind: () => { return; },
        };
    }
    /**
     * Create an S3 caching strategy.
     * @param bucket the S3 bucket to use for caching
     * @param options additional options to pass to the S3 caching
     */
    static bucket(bucket, options) {
        return {
            _toCloudFormation: () => ({
                type: 'S3',
                location: core_1.Fn.join('/', [bucket.bucketName, options && options.prefix || core_1.Aws.NO_VALUE]),
            }),
            _bind: (project) => {
                bucket.grantReadWrite(project);
            },
        };
    }
}
exports.Cache = Cache;
//# sourceMappingURL=data:application/json;base64,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