"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
/**
 * BuildSpec for CodeBuild projects
 */
class BuildSpec {
    constructor() {
    }
    static fromObject(value) {
        return new ObjectBuildSpec(value);
    }
    /**
     * Use a file from the source as buildspec
     *
     * Use this if you want to use a file different from 'buildspec.yml'`
     */
    static fromSourceFilename(filename) {
        return new FilenameBuildSpec(filename);
    }
}
exports.BuildSpec = BuildSpec;
/**
 * BuildSpec that just returns the input unchanged
 */
class FilenameBuildSpec extends BuildSpec {
    constructor(filename) {
        super();
        this.filename = filename;
        this.isImmediate = false;
    }
    toBuildSpec() {
        return this.filename;
    }
    toString() {
        return `<buildspec file: ${this.filename}>`;
    }
}
/**
 * BuildSpec that understands about structure
 */
class ObjectBuildSpec extends BuildSpec {
    constructor(spec) {
        super();
        this.spec = spec;
        this.isImmediate = true;
    }
    toBuildSpec() {
        // We have to pretty-print the buildspec, otherwise
        // CodeBuild will not recognize it as an inline buildspec.
        return core_1.Lazy.stringValue({ produce: (ctx) => core_1.Stack.of(ctx.scope).toJsonString(this.spec, 2),
        });
    }
}
/**
 * Merge two buildspecs into a new BuildSpec
 *
 * NOTE: will currently only merge commands, not artifact
 * declarations, environment variables, secrets, or any
 * other configuration elements.
 *
 * Internal for now because it's not complete/good enough
 * to expose on the objects directly, but we need to it to
 * keep feature-parity for Project.
 *
 * @internal
 */
function mergeBuildSpecs(lhs, rhs) {
    if (!(lhs instanceof ObjectBuildSpec) || !(rhs instanceof ObjectBuildSpec)) {
        throw new Error('Can only merge buildspecs created using BuildSpec.fromObject()');
    }
    return new ObjectBuildSpec(copyCommands(lhs.spec, rhs.spec));
}
exports.mergeBuildSpecs = mergeBuildSpecs;
/**
 * Extend buildSpec phases with the contents of another one
 */
function copyCommands(buildSpec, extend) {
    if (buildSpec.version === '0.1') {
        throw new Error('Cannot extend buildspec at version "0.1". Set the version to "0.2" or higher instead.');
    }
    const ret = Object.assign({}, buildSpec); // Return a copy
    ret.phases = Object.assign({}, ret.phases);
    for (const phaseName of Object.keys(extend.phases)) {
        const phase = ret.phases[phaseName] = Object.assign({}, ret.phases[phaseName]);
        phase.commands = [...phase.commands || [], ...extend.phases[phaseName].commands];
    }
    return ret;
}
//# sourceMappingURL=data:application/json;base64,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