"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const drop_empty_object_at_the_end_of_an_array_token_1 = require("./drop-empty-object-at-the-end-of-an-array-token");
const env_tokens_1 = require("./env-tokens");
const metric_util_1 = require("./metric-util");
const object_1 = require("./object");
/**
 * Return the JSON structure which represents these metrics in a graph.
 *
 * Depending on the metric type (stat or expression), one `Metric` object
 * can render to multiple time series.
 *
 * - Top-level metrics will be rendered visibly, additionally added metrics will
 *   be rendered invisibly.
 * - IDs used in math expressions need to be either globally unique, or refer to the same
 *   metric object.
 *
 * This will be called by GraphWidget, no need for clients to call this.
 */
function allMetricsGraphJson(left, right) {
    // Add metrics to a set which will automatically expand them recursively,
    // making sure to retain conflicting the visible one on conflicting metrics objects.
    const mset = new MetricSet();
    mset.addTopLevel('left', ...left);
    mset.addTopLevel('right', ...right);
    // Render all metrics from the set.
    return mset.entries.map(entry => new drop_empty_object_at_the_end_of_an_array_token_1.DropEmptyObjectAtTheEndOfAnArray(metricGraphJson(entry.metric, entry.tag, entry.id)));
}
exports.allMetricsGraphJson = allMetricsGraphJson;
function metricGraphJson(metric, yAxis, id) {
    const config = metric.toMetricConfig();
    const ret = [];
    const options = { ...config.renderingProperties };
    metric_util_1.dispatchMetric(metric, {
        withStat(stat) {
            ret.push(stat.namespace, stat.metricName);
            // Dimensions
            for (const dim of (stat.dimensions || [])) {
                ret.push(dim.name, dim.value);
            }
            // Metric attributes that are rendered to graph options
            if (stat.account) {
                options.accountId = env_tokens_1.accountIfDifferentFromStack(stat.account);
            }
            if (stat.region) {
                options.region = env_tokens_1.regionIfDifferentFromStack(stat.region);
            }
            if (stat.period && stat.period.toSeconds() !== 300) {
                options.period = stat.period.toSeconds();
            }
            if (stat.statistic && stat.statistic !== 'Average') {
                options.stat = stat.statistic;
            }
        },
        withExpression(expr) {
            options.expression = expr.expression;
        },
    });
    // Options
    if (!yAxis) {
        options.visible = false;
    }
    if (yAxis !== 'left') {
        options.yAxis = yAxis;
    }
    if (id) {
        options.id = id;
    }
    // If math expressions don't have a label (or an ID), they'll render with an unelegant
    // autogenerated id ("metric_alias0"). Our ids may in the future also be autogenerated,
    // so if an ME doesn't have a label, use its toString() as the label (renders the expression).
    if (options.visible !== false && options.expression && !options.label) {
        options.label = metric.toString();
    }
    const renderedOpts = object_1.dropUndefined(options);
    if (Object.keys(renderedOpts).length !== 0) {
        ret.push(renderedOpts);
    }
    return ret;
}
/**
 * Contain a set of metrics, expanding math expressions
 *
 * "Primary" metrics (added via a top-level call) can be tagged with an additional value.
 */
class MetricSet {
    constructor() {
        this.metrics = new Array();
        this.metricById = new Map();
        this.metricByKey = new Map();
    }
    /**
     * Add the given set of metrics to this set
     */
    addTopLevel(tag, ...metrics) {
        for (const metric of metrics) {
            this.addOne(metric, tag);
        }
    }
    /**
     * Access all the accumulated timeseries entries
     */
    get entries() {
        return this.metrics;
    }
    /**
     * Add a metric into the set
     *
     * The id may not be the same as a previous metric added, unless it's the same metric.
     *
     * It can be made visible, in which case the new "metric" object replaces the old
     * one (and the new ones "renderingPropertieS" will be honored instead of the old
     * one's).
     */
    addOne(metric, tag, id) {
        const key = metric_util_1.metricKey(metric);
        let existingEntry;
        // Try lookup existing by id if we have one
        if (id) {
            existingEntry = this.metricById.get(id);
            if (existingEntry && metric_util_1.metricKey(existingEntry.metric) !== key) {
                throw new Error(`Cannot have two different metrics share the same id ('${id}') in one Alarm or Graph. Rename one of them.`);
            }
        }
        if (!existingEntry) {
            // Try lookup by metric if we didn't find one by id
            existingEntry = this.metricByKey.get(key);
            // If the one we found already has an id, it must be different from the id
            // we're trying to add and we want to add a new metric. Pretend we didn't
            // find one.
            if ((existingEntry === null || existingEntry === void 0 ? void 0 : existingEntry.id) && id) {
                existingEntry = undefined;
            }
        }
        // Create a new entry if we didn't find one so far
        let entry;
        if (existingEntry) {
            entry = existingEntry;
        }
        else {
            entry = { metric };
            this.metrics.push(entry);
            this.metricByKey.set(key, entry);
        }
        // If it didn't have an id but now we do, add one
        if (!entry.id && id) {
            entry.id = id;
            this.metricById.set(id, entry);
        }
        // If it didn't have a tag but now we do, add one
        if (!entry.tag && tag) {
            entry.tag = tag;
        }
        // Recurse and add children
        const conf = metric.toMetricConfig();
        if (conf.mathExpression) {
            for (const [subId, subMetric] of Object.entries(conf.mathExpression.usingMetrics)) {
                this.addOne(subMetric, undefined, subId);
            }
        }
    }
}
exports.MetricSet = MetricSet;
//# sourceMappingURL=data:application/json;base64,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