"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const events = require("../../aws-events"); // Automatically re-written from '@aws-cdk/aws-events'
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const logs = require("../../aws-logs"); // Automatically re-written from '@aws-cdk/aws-logs'
const s3 = require("../../aws-s3"); // Automatically re-written from '@aws-cdk/aws-s3'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const cloudtrail_generated_1 = require("./cloudtrail.generated");
/**
 * Types of events that CloudTrail can log
 */
var ReadWriteType;
(function (ReadWriteType) {
    /**
     * Read-only events include API operations that read your resources,
     * but don't make changes.
     * For example, read-only events include the Amazon EC2 DescribeSecurityGroups
     * and DescribeSubnets API operations.
     */
    ReadWriteType["READ_ONLY"] = "ReadOnly";
    /**
     * Write-only events include API operations that modify (or might modify)
     * your resources.
     * For example, the Amazon EC2 RunInstances and TerminateInstances API
     * operations modify your instances.
     */
    ReadWriteType["WRITE_ONLY"] = "WriteOnly";
    /**
     * All events
     */
    ReadWriteType["ALL"] = "All";
    /**
     * No events
     */
    ReadWriteType["NONE"] = "None";
})(ReadWriteType = exports.ReadWriteType || (exports.ReadWriteType = {}));
/**
 * Cloud trail allows you to log events that happen in your AWS account
 * For example:
 *
 * import { CloudTrail } from '@aws-cdk/aws-cloudtrail'
 *
 * const cloudTrail = new CloudTrail(this, 'MyTrail');
 *
 * NOTE the above example creates an UNENCRYPTED bucket by default,
 * If you are required to use an Encrypted bucket you can supply a preconfigured bucket
 * via TrailProps
 *
 */
class Trail extends core_1.Resource {
    constructor(scope, id, props = {}) {
        var _a, _b, _c;
        super(scope, id, {
            physicalName: props.trailName,
        });
        this.eventSelectors = [];
        const cloudTrailPrincipal = new iam.ServicePrincipal('cloudtrail.amazonaws.com');
        this.s3bucket = props.bucket || new s3.Bucket(this, 'S3', { encryption: s3.BucketEncryption.UNENCRYPTED });
        this.s3bucket.addToResourcePolicy(new iam.PolicyStatement({
            resources: [this.s3bucket.bucketArn],
            actions: ['s3:GetBucketAcl'],
            principals: [cloudTrailPrincipal],
        }));
        this.s3bucket.addToResourcePolicy(new iam.PolicyStatement({
            resources: [this.s3bucket.arnForObjects(`${props.s3KeyPrefix ? `${props.s3KeyPrefix}/` : ''}AWSLogs/${core_1.Stack.of(this).account}/*`)],
            actions: ['s3:PutObject'],
            principals: [cloudTrailPrincipal],
            conditions: {
                StringEquals: { 's3:x-amz-acl': 'bucket-owner-full-control' },
            },
        }));
        let logsRole;
        if (props.sendToCloudWatchLogs) {
            if (props.cloudWatchLogGroup) {
                this.logGroup = props.cloudWatchLogGroup;
            }
            else {
                this.logGroup = new logs.LogGroup(this, 'LogGroup', {
                    retention: (_a = props.cloudWatchLogsRetention) !== null && _a !== void 0 ? _a : logs.RetentionDays.ONE_YEAR,
                });
            }
            logsRole = new iam.Role(this, 'LogsRole', { assumedBy: cloudTrailPrincipal });
            logsRole.addToPolicy(new iam.PolicyStatement({
                actions: ['logs:PutLogEvents', 'logs:CreateLogStream'],
                resources: [this.logGroup.logGroupArn],
            }));
        }
        if (props.managementEvents) {
            let managementEvent;
            if (props.managementEvents === ReadWriteType.NONE) {
                managementEvent = {
                    includeManagementEvents: false,
                };
            }
            else {
                managementEvent = {
                    includeManagementEvents: true,
                    readWriteType: props.managementEvents,
                };
            }
            this.eventSelectors.push(managementEvent);
        }
        // TODO: not all regions support validation. Use service configuration data to fail gracefully
        const trail = new cloudtrail_generated_1.CfnTrail(this, 'Resource', {
            isLogging: true,
            enableLogFileValidation: props.enableFileValidation == null ? true : props.enableFileValidation,
            isMultiRegionTrail: props.isMultiRegionTrail == null ? true : props.isMultiRegionTrail,
            includeGlobalServiceEvents: props.includeGlobalServiceEvents == null ? true : props.includeGlobalServiceEvents,
            trailName: this.physicalName,
            kmsKeyId: props.kmsKey && props.kmsKey.keyArn,
            s3BucketName: this.s3bucket.bucketName,
            s3KeyPrefix: props.s3KeyPrefix,
            cloudWatchLogsLogGroupArn: (_b = this.logGroup) === null || _b === void 0 ? void 0 : _b.logGroupArn,
            cloudWatchLogsRoleArn: logsRole === null || logsRole === void 0 ? void 0 : logsRole.roleArn,
            snsTopicName: (_c = props.snsTopic) === null || _c === void 0 ? void 0 : _c.topicName,
            eventSelectors: this.eventSelectors,
        });
        this.trailArn = this.getResourceArnAttribute(trail.attrArn, {
            service: 'cloudtrail',
            resource: 'trail',
            resourceName: this.physicalName,
        });
        this.trailSnsTopicArn = trail.attrSnsTopicArn;
        // Add a dependency on the bucket policy being updated, CloudTrail will test this upon creation.
        if (this.s3bucket.policy) {
            trail.node.addDependency(this.s3bucket.policy);
        }
        // If props.sendToCloudWatchLogs is set to true then the trail needs to depend on the created logsRole
        // so that it can create the log stream for the log group. This ensures the logsRole is created and propagated
        // before the trail tries to create the log stream.
        if (logsRole !== undefined) {
            trail.node.addDependency(logsRole);
        }
    }
    /**
     * Create an event rule for when an event is recorded by any Trail in the account.
     *
     * Note that the event doesn't necessarily have to come from this Trail, it can
     * be captured from any one.
     *
     * Be sure to filter the event further down using an event pattern.
     */
    static onEvent(scope, id, options = {}) {
        const rule = new events.Rule(scope, id, options);
        rule.addTarget(options.target);
        rule.addEventPattern({
            detailType: ['AWS API Call via CloudTrail'],
        });
        return rule;
    }
    /**
     * When an event occurs in your account, CloudTrail evaluates whether the event matches the settings for your trails.
     * Only events that match your trail settings are delivered to your Amazon S3 bucket and Amazon CloudWatch Logs log group.
     *
     * This method adds an Event Selector for filtering events that match either S3 or Lambda function operations.
     *
     * Data events: These events provide insight into the resource operations performed on or within a resource.
     * These are also known as data plane operations.
     *
     * @param dataResourceValues the list of data resource ARNs to include in logging (maximum 250 entries).
     * @param options the options to configure logging of management and data events.
     */
    addEventSelector(dataResourceType, dataResourceValues, options = {}) {
        if (dataResourceValues.length > 250) {
            throw new Error('A maximum of 250 data elements can be in one event selector');
        }
        if (this.eventSelectors.length > 5) {
            throw new Error('A maximum of 5 event selectors are supported per trail.');
        }
        this.eventSelectors.push({
            dataResources: [{
                    type: dataResourceType,
                    values: dataResourceValues,
                }],
            includeManagementEvents: options.includeManagementEvents,
            readWriteType: options.readWriteType,
        });
    }
    /**
     * When an event occurs in your account, CloudTrail evaluates whether the event matches the settings for your trails.
     * Only events that match your trail settings are delivered to your Amazon S3 bucket and Amazon CloudWatch Logs log group.
     *
     * This method adds a Lambda Data Event Selector for filtering events that match Lambda function operations.
     *
     * Data events: These events provide insight into the resource operations performed on or within a resource.
     * These are also known as data plane operations.
     *
     * @param handlers the list of lambda function handlers whose data events should be logged (maximum 250 entries).
     * @param options the options to configure logging of management and data events.
     */
    addLambdaEventSelector(handlers, options = {}) {
        if (handlers.length === 0) {
            return;
        }
        const dataResourceValues = handlers.map((h) => h.functionArn);
        return this.addEventSelector(DataResourceType.LAMBDA_FUNCTION, dataResourceValues, options);
    }
    /**
     * Log all Lamda data events for all lambda functions the account.
     * @see https://docs.aws.amazon.com/awscloudtrail/latest/userguide/logging-data-events-with-cloudtrail.html
     * @default false
     */
    logAllLambdaDataEvents(options = {}) {
        return this.addEventSelector(DataResourceType.LAMBDA_FUNCTION, ['arn:aws:lambda'], options);
    }
    /**
     * When an event occurs in your account, CloudTrail evaluates whether the event matches the settings for your trails.
     * Only events that match your trail settings are delivered to your Amazon S3 bucket and Amazon CloudWatch Logs log group.
     *
     * This method adds an S3 Data Event Selector for filtering events that match S3 operations.
     *
     * Data events: These events provide insight into the resource operations performed on or within a resource.
     * These are also known as data plane operations.
     *
     * @param s3Selector the list of S3 bucket with optional prefix to include in logging (maximum 250 entries).
     * @param options the options to configure logging of management and data events.
     */
    addS3EventSelector(s3Selector, options = {}) {
        if (s3Selector.length === 0) {
            return;
        }
        const dataResourceValues = s3Selector.map((sel) => { var _a; return `${sel.bucket.bucketArn}/${(_a = sel.objectPrefix) !== null && _a !== void 0 ? _a : ''}`; });
        return this.addEventSelector(DataResourceType.S3_OBJECT, dataResourceValues, options);
    }
    /**
     * Log all S3 data events for all objects for all buckets in the account.
     * @see https://docs.aws.amazon.com/awscloudtrail/latest/userguide/logging-data-events-with-cloudtrail.html
     * @default false
     */
    logAllS3DataEvents(options = {}) {
        return this.addEventSelector(DataResourceType.S3_OBJECT, ['arn:aws:s3:::'], options);
    }
    /**
     * Create an event rule for when an event is recorded by any Trail in the account.
     *
     * Note that the event doesn't necessarily have to come from this Trail, it can
     * be captured from any one.
     *
     * Be sure to filter the event further down using an event pattern.
     *
     * @deprecated - use Trail.onEvent()
     */
    onCloudTrailEvent(id, options = {}) {
        return Trail.onEvent(this, id, options);
    }
}
exports.Trail = Trail;
/**
 * Resource type for a data event
 */
var DataResourceType;
(function (DataResourceType) {
    /**
     * Data resource type for Lambda function
     */
    DataResourceType["LAMBDA_FUNCTION"] = "AWS::Lambda::Function";
    /**
     * Data resource type for S3 objects
     */
    DataResourceType["S3_OBJECT"] = "AWS::S3::Object";
})(DataResourceType = exports.DataResourceType || (exports.DataResourceType = {}));
//# sourceMappingURL=data:application/json;base64,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