"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const certificatemanager_generated_1 = require("./certificatemanager.generated");
const util_1 = require("./util");
/**
 * A certificate managed by AWS Certificate Manager
 *
 * IMPORTANT: if you are creating a certificate as part of your stack, the stack
 * will not complete creating until you read and follow the instructions in the
 * email that you will receive.
 *
 * ACM will send validation emails to the following addresses:
 *
 *  admin@domain.com
 *  administrator@domain.com
 *  hostmaster@domain.com
 *  postmaster@domain.com
 *  webmaster@domain.com
 *
 * For every domain that you register.
 */
class Certificate extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        const allDomainNames = [props.domainName].concat(props.subjectAlternativeNames || []);
        const cert = new certificatemanager_generated_1.CfnCertificate(this, 'Resource', {
            domainName: props.domainName,
            subjectAlternativeNames: props.subjectAlternativeNames,
            domainValidationOptions: allDomainNames.map(domainValidationOption),
            validationMethod: props.validationMethod,
        });
        this.certificateArn = cert.ref;
        /**
         * Return the domain validation options for the given domain
         *
         * Closes over props.
         */
        function domainValidationOption(domainName) {
            let validationDomain = props.validationDomains && props.validationDomains[domainName];
            if (validationDomain === undefined) {
                if (core_1.Token.isUnresolved(domainName)) {
                    throw new Error('When using Tokens for domain names, \'validationDomains\' needs to be supplied');
                }
                validationDomain = util_1.apexDomain(domainName);
            }
            return { domainName, validationDomain };
        }
    }
    /**
     * Import a certificate
     */
    static fromCertificateArn(scope, id, certificateArn) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.certificateArn = certificateArn;
            }
        }
        return new Import(scope, id);
    }
}
exports.Certificate = Certificate;
/**
 * Method used to assert ownership of the domain
 */
var ValidationMethod;
(function (ValidationMethod) {
    /**
     * Send email to a number of email addresses associated with the domain
     *
     * @see https://docs.aws.amazon.com/acm/latest/userguide/gs-acm-validate-email.html
     */
    ValidationMethod["EMAIL"] = "EMAIL";
    /**
     * Validate ownership by adding appropriate DNS records
     *
     * @see https://docs.aws.amazon.com/acm/latest/userguide/gs-acm-validate-dns.html
     */
    ValidationMethod["DNS"] = "DNS";
})(ValidationMethod = exports.ValidationMethod || (exports.ValidationMethod = {}));
//# sourceMappingURL=data:application/json;base64,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