"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const batch_generated_1 = require("./batch.generated");
/**
 * Batch Job Queue.
 *
 * Defines a batch job queue to define how submitted batch jobs
 * should be ran based on specified batch compute environments.
 */
class JobQueue extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.jobQueueName,
        });
        if (props.computeEnvironments.length === 0) {
            throw new Error('computeEnvironments must be non-empty');
        }
        const jobQueue = new batch_generated_1.CfnJobQueue(this, 'Resource', {
            computeEnvironmentOrder: props.computeEnvironments.map(cp => ({
                computeEnvironment: cp.computeEnvironment.computeEnvironmentArn,
                order: cp.order,
            })),
            jobQueueName: this.physicalName,
            priority: props.priority || 1,
            state: props.enabled === undefined ? 'ENABLED' : (props.enabled ? 'ENABLED' : 'DISABLED'),
        });
        this.jobQueueArn = this.getResourceArnAttribute(jobQueue.ref, {
            service: 'batch',
            resource: 'job-queue',
            resourceName: this.physicalName,
        });
        this.jobQueueName = this.getResourceNameAttribute(jobQueue.ref);
    }
    /**
     * Fetches an existing batch job queue by its amazon resource name.
     *
     * @param scope
     * @param id
     * @param jobQueueArn
     */
    static fromJobQueueArn(scope, id, jobQueueArn) {
        const stack = core_1.Stack.of(scope);
        const jobQueueName = stack.parseArn(jobQueueArn).resourceName;
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.jobQueueArn = jobQueueArn;
                this.jobQueueName = jobQueueName;
            }
        }
        return new Import(scope, id);
    }
}
exports.JobQueue = JobQueue;
//# sourceMappingURL=data:application/json;base64,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