"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const batch_generated_1 = require("./batch.generated");
const job_definition_image_config_1 = require("./job-definition-image-config");
/**
 * Batch Job Definition.
 *
 * Defines a batch job definition to execute a specific batch job.
 */
class JobDefinition extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.jobDefinitionName,
        });
        this.imageConfig = new job_definition_image_config_1.JobDefinitionImageConfig(this, props.container);
        const jobDef = new batch_generated_1.CfnJobDefinition(this, 'Resource', {
            jobDefinitionName: props.jobDefinitionName,
            containerProperties: this.buildJobContainer(props.container),
            type: 'container',
            nodeProperties: props.nodeProps
                ? { mainNode: props.nodeProps.mainNode,
                    nodeRangeProperties: this.buildNodeRangeProps(props.nodeProps),
                    numNodes: props.nodeProps.count }
                : undefined,
            parameters: props.parameters,
            retryStrategy: {
                attempts: props.retryAttempts || 1,
            },
            timeout: {
                attemptDurationSeconds: props.timeout ? props.timeout.toSeconds() : undefined,
            },
        });
        this.jobDefinitionArn = this.getResourceArnAttribute(jobDef.ref, {
            service: 'batch',
            resource: 'job-definition',
            resourceName: this.physicalName,
        });
        this.jobDefinitionName = this.getResourceNameAttribute(jobDef.ref);
    }
    /**
     * Imports an existing batch job definition by its amazon resource name.
     *
     * @param scope
     * @param id
     * @param jobDefinitionArn
     */
    static fromJobDefinitionArn(scope, id, jobDefinitionArn) {
        const stack = core_1.Stack.of(scope);
        const jobDefName = stack.parseArn(jobDefinitionArn).resourceName;
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.jobDefinitionArn = jobDefinitionArn;
                this.jobDefinitionName = jobDefName;
            }
        }
        return new Import(scope, id);
    }
    deserializeEnvVariables(env) {
        const vars = new Array();
        if (env === undefined) {
            return undefined;
        }
        Object.keys(env).map((name) => {
            vars.push({ name, value: env[name] });
        });
        return vars;
    }
    buildJobContainer(container) {
        if (container === undefined) {
            return undefined;
        }
        return {
            command: container.command,
            environment: this.deserializeEnvVariables(container.environment),
            image: this.imageConfig.imageName,
            instanceType: container.instanceType && container.instanceType.toString(),
            jobRoleArn: container.jobRole && container.jobRole.roleArn,
            linuxParameters: container.linuxParams
                ? { devices: container.linuxParams.renderLinuxParameters().devices }
                : undefined,
            memory: container.memoryLimitMiB || 4,
            mountPoints: container.mountPoints,
            privileged: container.privileged || false,
            resourceRequirements: container.gpuCount
                ? [{ type: 'GPU', value: String(container.gpuCount) }]
                : undefined,
            readonlyRootFilesystem: container.readOnly || false,
            ulimits: container.ulimits,
            user: container.user,
            vcpus: container.vcpus || 1,
            volumes: container.volumes,
        };
    }
    buildNodeRangeProps(multiNodeProps) {
        const rangeProps = new Array();
        for (const prop of multiNodeProps.rangeProps) {
            rangeProps.push({
                container: this.buildJobContainer(prop.container),
                targetNodes: `${prop.fromNodeIndex || 0}:${prop.toNodeIndex || multiNodeProps.count}`,
            });
        }
        return rangeProps;
    }
}
exports.JobDefinition = JobDefinition;
//# sourceMappingURL=data:application/json;base64,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