"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const backup_generated_1 = require("./backup.generated");
const backupable_resources_collector_1 = require("./backupable-resources-collector");
const resource_1 = require("./resource");
/**
 * A backup selection
 */
class BackupSelection extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        this.listOfTags = [];
        this.resources = [];
        this.backupableResourcesCollector = new backupable_resources_collector_1.BackupableResourcesCollector();
        const role = props.role || new iam.Role(this, 'Role', {
            assumedBy: new iam.ServicePrincipal('backup.amazonaws.com'),
        });
        role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSBackupServiceRolePolicyForBackup'));
        if (props.allowRestores) {
            role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSBackupServiceRolePolicyForRestores'));
        }
        this.grantPrincipal = role;
        const selection = new backup_generated_1.CfnBackupSelection(this, 'Resource', {
            backupPlanId: props.backupPlan.backupPlanId,
            backupSelection: {
                iamRoleArn: role.roleArn,
                selectionName: props.backupSelectionName || this.node.id,
                listOfTags: core_1.Lazy.anyValue({
                    produce: () => this.listOfTags,
                }, { omitEmptyArray: true }),
                resources: core_1.Lazy.listValue({
                    produce: () => [...this.resources, ...this.backupableResourcesCollector.resources],
                }, { omitEmpty: true }),
            },
        });
        this.backupPlanId = selection.attrBackupPlanId;
        this.selectionId = selection.attrSelectionId;
        for (const resource of props.resources) {
            this.addResource(resource);
        }
    }
    addResource(resource) {
        if (resource.tagCondition) {
            this.listOfTags.push({
                conditionKey: resource.tagCondition.key,
                conditionType: resource.tagCondition.operation || resource_1.TagOperation.STRING_EQUALS,
                conditionValue: resource.tagCondition.value,
            });
        }
        if (resource.resource) {
            this.resources.push(resource.resource);
        }
        if (resource.construct) {
            resource.construct.node.applyAspect(this.backupableResourcesCollector);
            // Cannot push `this.backupableResourcesCollector.resources` to
            // `this.resources` here because it has not been evaluated yet.
            // Will be concatenated to `this.resources` in a `Lazy.listValue`
            // in the constructor instead.
        }
    }
}
exports.BackupSelection = BackupSelection;
//# sourceMappingURL=data:application/json;base64,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