"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const events = require("../../aws-events"); // Automatically re-written from '@aws-cdk/aws-events'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
/**
 * A backup plan rule
 */
class BackupPlanRule {
    /** @param props Rule properties */
    constructor(props) {
        this.props = props;
        if (props.deleteAfter && props.moveToColdStorageAfter &&
            props.deleteAfter.toSeconds() < props.moveToColdStorageAfter.toSeconds()) {
            throw new Error('`deleteAfter` must be greater than `moveToColdStorageAfter`');
        }
        if (props.scheduleExpression && !/^cron/.test(props.scheduleExpression.expressionString)) {
            throw new Error('`scheduleExpression` must be of type `cron`');
        }
    }
    /**
     * Daily with 35 days retention
     */
    static daily(backupVault) {
        return new BackupPlanRule({
            backupVault,
            ruleName: 'Daily',
            scheduleExpression: events.Schedule.cron({
                hour: '5',
                minute: '0',
            }),
            deleteAfter: core_1.Duration.days(35),
        });
    }
    /**
     * Weekly with 3 months retention
     */
    static weekly(backupVault) {
        return new BackupPlanRule({
            backupVault,
            ruleName: 'Weekly',
            scheduleExpression: events.Schedule.cron({
                hour: '5',
                minute: '0',
                weekDay: 'SAT',
            }),
            deleteAfter: core_1.Duration.days(30 * 3),
        });
    }
    /**
     * Monthly 1 year retention, move to cold storage after 1 month
     */
    static monthly1Year(backupVault) {
        return new BackupPlanRule({
            backupVault,
            ruleName: 'Monthly1Year',
            scheduleExpression: events.Schedule.cron({
                day: '1',
                hour: '5',
                minute: '0',
            }),
            moveToColdStorageAfter: core_1.Duration.days(30),
            deleteAfter: core_1.Duration.days(365),
        });
    }
    /**
     * Monthly 5 year retention, move to cold storage after 3 months
     */
    static monthly5Year(backupVault) {
        return new BackupPlanRule({
            backupVault,
            ruleName: 'Monthly5Year',
            scheduleExpression: events.Schedule.cron({
                day: '1',
                hour: '5',
                minute: '0',
            }),
            moveToColdStorageAfter: core_1.Duration.days(30 * 3),
            deleteAfter: core_1.Duration.days(365 * 5),
        });
    }
    /**
     * Monthly 7 year retention, move to cold storage after 3 months
     */
    static monthly7Year(backupVault) {
        return new BackupPlanRule({
            backupVault,
            ruleName: 'Monthly7Year',
            scheduleExpression: events.Schedule.cron({
                day: '1',
                hour: '5',
                minute: '0',
            }),
            moveToColdStorageAfter: core_1.Duration.days(30 * 3),
            deleteAfter: core_1.Duration.days(365 * 7),
        });
    }
}
exports.BackupPlanRule = BackupPlanRule;
//# sourceMappingURL=data:application/json;base64,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