"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
/**
 * An operation that is applied to a key-value pair
 */
var TagOperation;
(function (TagOperation) {
    /**
     * StringEquals
     */
    TagOperation["STRING_EQUALS"] = "STRINGEQUALS";
    /**
     * Dummy member
     */
    TagOperation["DUMMY"] = "dummy";
})(TagOperation = exports.TagOperation || (exports.TagOperation = {}));
/**
 * A resource to backup
 */
class BackupResource {
    constructor(resource, tagCondition, construct) {
        this.resource = resource;
        this.tagCondition = tagCondition;
        this.construct = construct;
    }
    /**
     * Adds all supported resources in a construct
     *
     * @param construct The construct containing resources to backup
     */
    static fromConstruct(construct) {
        return new BackupResource(undefined, undefined, construct);
    }
    /**
     * A DynamoDB table
     */
    static fromDynamoDbTable(table) {
        return BackupResource.fromArn(table.tableArn);
    }
    /**
     * An EC2 instance
     */
    static fromEc2Instance(instance) {
        return BackupResource.fromArn(core_1.Stack.of(instance).formatArn({
            service: 'ec2',
            resource: 'instance',
            resourceName: instance.instanceId,
        }));
    }
    /**
     * An EFS file system
     */
    static fromEfsFileSystem(fileSystem) {
        return BackupResource.fromArn(core_1.Stack.of(fileSystem).formatArn({
            service: 'elasticfilesystem',
            resource: 'file-system',
            resourceName: fileSystem.fileSystemId,
        }));
    }
    /**
     * A RDS database instance
     */
    static fromRdsDatabaseInstance(instance) {
        return BackupResource.fromArn(instance.instanceArn);
    }
    /**
     * A list of ARNs or match patterns such as
     * `arn:aws:ec2:us-east-1:123456789012:volume/*`
     */
    static fromArn(arn) {
        return new BackupResource(arn);
    }
    /**
     * A tag condition
     */
    static fromTag(key, value, operation) {
        return new BackupResource(undefined, {
            key,
            value,
            operation,
        });
    }
}
exports.BackupResource = BackupResource;
//# sourceMappingURL=data:application/json;base64,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