"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const backup_generated_1 = require("./backup.generated");
const rule_1 = require("./rule");
const selection_1 = require("./selection");
const vault_1 = require("./vault");
/**
 * A backup plan
 */
class BackupPlan extends core_1.Resource {
    constructor(scope, id, props = {}) {
        super(scope, id);
        this.rules = [];
        const plan = new backup_generated_1.CfnBackupPlan(this, 'Resource', {
            backupPlan: {
                backupPlanName: props.backupPlanName || id,
                backupPlanRule: core_1.Lazy.anyValue({ produce: () => this.rules }, { omitEmptyArray: true }),
            },
        });
        this.backupPlanId = plan.attrBackupPlanId;
        this.backupPlanArn = plan.attrBackupPlanArn;
        this.versionId = plan.attrVersionId;
        this._backupVault = props.backupVault;
        for (const rule of props.backupPlanRules || []) {
            this.addRule(rule);
        }
    }
    /**
     * Import an existing backup plan
     */
    static fromBackupPlanId(scope, id, backupPlanId) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.backupPlanId = backupPlanId;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Daily with 35 day retention
     */
    static daily35DayRetention(scope, id, backupVault) {
        const plan = new BackupPlan(scope, id, { backupVault });
        plan.addRule(rule_1.BackupPlanRule.daily());
        return plan;
    }
    /**
     * Daily and monthly with 1 year retention
     */
    static dailyMonthly1YearRetention(scope, id, backupVault) {
        const plan = new BackupPlan(scope, id, { backupVault });
        plan.addRule(rule_1.BackupPlanRule.daily());
        plan.addRule(rule_1.BackupPlanRule.monthly1Year());
        return plan;
    }
    /**
     * Daily, weekly and monthly with 5 year retention
     */
    static dailyWeeklyMonthly5YearRetention(scope, id, backupVault) {
        const plan = new BackupPlan(scope, id, { backupVault });
        plan.addRule(rule_1.BackupPlanRule.daily());
        plan.addRule(rule_1.BackupPlanRule.weekly());
        plan.addRule(rule_1.BackupPlanRule.monthly5Year());
        return plan;
    }
    /**
     * Daily, weekly and monthly with 7 year retention
     */
    static dailyWeeklyMonthly7YearRetention(scope, id, backupVault) {
        const plan = new BackupPlan(scope, id, { backupVault });
        plan.addRule(rule_1.BackupPlanRule.daily());
        plan.addRule(rule_1.BackupPlanRule.weekly());
        plan.addRule(rule_1.BackupPlanRule.monthly7Year());
        return plan;
    }
    /**
     * Adds a rule to a plan
     *
     * @param rule the rule to add
     */
    addRule(rule) {
        var _a, _b, _c, _d, _e, _f;
        let vault;
        if (rule.props.backupVault) {
            vault = rule.props.backupVault;
        }
        else if (this._backupVault) {
            vault = this._backupVault;
        }
        else {
            this._backupVault = new vault_1.BackupVault(this, 'Vault');
            vault = this._backupVault;
        }
        this.rules.push({
            completionWindowMinutes: (_a = rule.props.completionWindow) === null || _a === void 0 ? void 0 : _a.toMinutes(),
            lifecycle: (rule.props.deleteAfter || rule.props.moveToColdStorageAfter) && {
                deleteAfterDays: (_b = rule.props.deleteAfter) === null || _b === void 0 ? void 0 : _b.toDays(),
                moveToColdStorageAfterDays: (_c = rule.props.moveToColdStorageAfter) === null || _c === void 0 ? void 0 : _c.toDays(),
            },
            ruleName: (_d = rule.props.ruleName) !== null && _d !== void 0 ? _d : `${this.node.id}Rule${this.rules.length}`,
            scheduleExpression: (_e = rule.props.scheduleExpression) === null || _e === void 0 ? void 0 : _e.expressionString,
            startWindowMinutes: (_f = rule.props.startWindow) === null || _f === void 0 ? void 0 : _f.toMinutes(),
            targetBackupVault: vault.backupVaultName,
        });
    }
    /**
     * The backup vault where backups are stored if not defined at
     * the rule level
     */
    get backupVault() {
        if (!this._backupVault) {
            // This cannot happen but is here to make TypeScript happy
            throw new Error('No backup vault!');
        }
        return this._backupVault;
    }
    /**
     * Adds a selection to this plan
     */
    addSelection(id, options) {
        return new selection_1.BackupSelection(this, id, {
            backupPlan: this,
            ...options,
        });
    }
    validate() {
        if (this.rules.length === 0) {
            return ['A backup plan must have at least 1 rule.'];
        }
        return [];
    }
}
exports.BackupPlan = BackupPlan;
//# sourceMappingURL=data:application/json;base64,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