"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * Normalize the given interval set to cover the complete number line and make sure it has at most one gap
 */
function normalizeIntervals(intervals, changesAreAbsolute) {
    // Make intervals a complete numberline
    const full = orderAndCompleteIntervals(intervals);
    // Add 'undefined's in uncovered areas of the number line
    makeGapsUndefined(full);
    // In case of relative changes, treat 0-change also as 'undefined' (= no change action)
    if (!changesAreAbsolute) {
        makeZerosUndefined(full);
    }
    // Combine adjacent undefines and make sure there's at most one of them
    combineUndefineds(full);
    validateAtMostOneUndefined(full);
    return full;
}
exports.normalizeIntervals = normalizeIntervals;
/**
 * Completely order scaling intervals, making their lower and upper bounds concrete.
 */
function orderAndCompleteIntervals(intervals) {
    if (intervals.length < 2) {
        throw new Error('Require at least 2 intervals');
    }
    for (const interval of intervals) {
        if (interval.lower === undefined && interval.upper === undefined) {
            throw new Error(`Must supply at least one of 'upper' or 'lower', got: ${JSON.stringify(interval)}`);
        }
    }
    // Make a copy
    intervals = intervals.map(x => ({ ...x }));
    // Sort by whatever number we have for each interval
    intervals.sort(comparatorFromKey((x) => x.lower !== undefined ? x.lower : x.upper));
    // Propagate boundaries until no more change
    while (propagateBounds(intervals)) { /* Repeat */ }
    const lastIndex = intervals.length - 1;
    // Validate that no intervals have undefined bounds now, which must mean they're complete.
    if (intervals[0].lower === undefined) {
        intervals[0] = { ...intervals[0], lower: 0 };
    }
    if (intervals[lastIndex].upper === undefined) {
        intervals[lastIndex] = { ...intervals[lastIndex], upper: Infinity };
    }
    for (const interval of intervals) {
        if (interval.lower === undefined || interval.upper === undefined) {
            throw new Error(`Could not determine the lower and upper bounds for ${JSON.stringify(interval)}`);
        }
    }
    const completeIntervals = intervals;
    // Validate that we have nonoverlapping intervals now.
    for (let i = 0; i < completeIntervals.length - 1; i++) {
        if (overlap(completeIntervals[i], completeIntervals[i + 1])) {
            throw new Error(`Two intervals overlap: ${JSON.stringify(completeIntervals[i])} and ${JSON.stringify(completeIntervals[i + 1])}`);
        }
    }
    // Fill up the gaps
    return completeIntervals;
}
/**
 * Make the intervals cover the complete number line
 *
 * This entails adding intervals with an 'undefined' change to fill up the gaps.
 *
 * Since metrics have a halfopen interval, the first one will get a lower bound
 * of 0, the last one will get an upper bound of +Infinity.
 *
 * In case of absolute adjustments, the lower number of the adjacent bound will
 * be used, which means conservative change. In case of relative adjustments,
 * we'll use relative adjusment 0 (which means no change).
 */
function makeGapsUndefined(intervals) {
    // Add edge intervals if necessary, but only for relative adjustments. Since we're
    // going to make scaling intervals extend all the way out to infinity on either side,
    // the result is the same for absolute adjustments anyway.
    if (intervals[0].lower !== 0) {
        intervals.splice(0, 1, {
            lower: 0,
            upper: intervals[0].lower,
            change: undefined,
        });
    }
    if (last(intervals).upper !== Infinity) {
        intervals.push({
            lower: last(intervals).upper,
            upper: Infinity,
            change: undefined,
        });
    }
    let i = 1;
    while (i < intervals.length) {
        if (intervals[i - 1].upper < intervals[i].lower) {
            intervals.splice(i, 0, {
                lower: intervals[i - 1].upper,
                upper: intervals[i].lower,
                change: undefined,
            });
        }
        else {
            i++;
        }
    }
}
/**
 * Turn zero changes into undefined, in-place
 */
function makeZerosUndefined(intervals) {
    for (let i = 0; i < intervals.length; ++i) {
        const interval = intervals[i];
        if (interval.change === 0) {
            intervals[i] = { ...interval, change: undefined };
        }
    }
}
/**
 * If there are adjacent "undefined" intervals, combine them
 */
function combineUndefineds(intervals) {
    let i = 0;
    while (i < intervals.length - 1) {
        if (intervals[i].change === undefined && intervals[i + 1].change === undefined) {
            intervals[i] = { ...intervals[i], upper: intervals[i + 1].upper };
            intervals.splice(i + 1, 1);
        }
        else {
            i++;
        }
    }
}
function validateAtMostOneUndefined(intervals) {
    const undef = intervals.filter(x => x.change === undefined);
    if (undef.length > 1) {
        throw new Error(`Can have at most one no-change interval, got ${JSON.stringify(undef)}`);
    }
}
function comparatorFromKey(keyFn) {
    return (a, b) => {
        const keyA = keyFn(a);
        const keyB = keyFn(b);
        if (keyA < keyB) {
            return -1;
        }
        if (keyA === keyB) {
            return 0;
        }
        return 1;
    };
}
function propagateBounds(intervals) {
    let ret = false;
    // Propagate upper bounds upwards
    for (let i = 0; i < intervals.length - 1; i++) {
        if (intervals[i].upper !== undefined && intervals[i + 1].lower === undefined) {
            intervals[i + 1] = { ...intervals[i + 1], lower: intervals[i].upper };
            ret = true;
        }
    }
    // Propagate lower bounds downwards
    for (let i = intervals.length - 1; i >= 1; i--) {
        if (intervals[i].lower !== undefined && intervals[i - 1].upper === undefined) {
            intervals[i - 1] = { ...intervals[i - 1], upper: intervals[i].lower };
            ret = true;
        }
    }
    return ret;
}
/**
 * Whether two intervals overlap
 */
function overlap(a, b) {
    return a.lower < b.upper && a.upper > b.lower;
}
function last(xs) {
    return xs[xs.length - 1];
}
/**
 * Locate the intervals that should have the alarm thresholds, by index.
 *
 * Pick the intervals on either side of the singleton "undefined" interval, or
 * pick the middle interval if there's no such interval.
 */
function findAlarmThresholds(intervals) {
    const gapIndex = intervals.findIndex(x => x.change === undefined);
    if (gapIndex !== -1) {
        return {
            lowerAlarmIntervalIndex: gapIndex > 0 ? gapIndex - 1 : undefined,
            upperAlarmIntervalIndex: gapIndex < intervals.length - 1 ? gapIndex + 1 : undefined,
        };
    }
    if (intervals.length === 1) {
        return { upperAlarmIntervalIndex: 0 };
    }
    const middleIndex = Math.floor(intervals.length / 2);
    return {
        lowerAlarmIntervalIndex: middleIndex - 1,
        upperAlarmIntervalIndex: middleIndex,
    };
}
exports.findAlarmThresholds = findAlarmThresholds;
//# sourceMappingURL=data:application/json;base64,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