"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const aws_iam_1 = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const fs_1 = require("fs");
const appsync_generated_1 = require("./appsync.generated");
/**
 * enum with all possible values for Cognito user-pool default actions
 */
var UserPoolDefaultAction;
(function (UserPoolDefaultAction) {
    /**
     * ALLOW access to API
     */
    UserPoolDefaultAction["ALLOW"] = "ALLOW";
    /**
     * DENY access to API
     */
    UserPoolDefaultAction["DENY"] = "DENY";
})(UserPoolDefaultAction = exports.UserPoolDefaultAction || (exports.UserPoolDefaultAction = {}));
function isUserPoolConfig(obj) {
    return obj.userPool !== undefined;
}
function isApiKeyConfig(obj) {
    return obj.apiKeyDesc !== undefined;
}
/**
 * log-level for fields in AppSync
 */
var FieldLogLevel;
(function (FieldLogLevel) {
    /**
     * No logging
     */
    FieldLogLevel["NONE"] = "NONE";
    /**
     * Error logging
     */
    FieldLogLevel["ERROR"] = "ERROR";
    /**
     * All logging
     */
    FieldLogLevel["ALL"] = "ALL";
})(FieldLogLevel = exports.FieldLogLevel || (exports.FieldLogLevel = {}));
/**
 * An AppSync GraphQL API
 */
class GraphQLApi extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        let apiLogsRole;
        if (props.logConfig) {
            apiLogsRole = new aws_iam_1.Role(this, 'ApiLogsRole', { assumedBy: new aws_iam_1.ServicePrincipal('appsync') });
            apiLogsRole.addManagedPolicy(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSAppSyncPushToCloudWatchLogs'));
        }
        this.api = new appsync_generated_1.CfnGraphQLApi(this, 'Resource', {
            name: props.name,
            authenticationType: 'API_KEY',
            ...props.logConfig && {
                logConfig: {
                    cloudWatchLogsRoleArn: apiLogsRole ? apiLogsRole.roleArn : undefined,
                    excludeVerboseContent: props.logConfig.excludeVerboseContent,
                    fieldLogLevel: props.logConfig.fieldLogLevel ? props.logConfig.fieldLogLevel.toString() : undefined,
                },
            },
        });
        this.apiId = this.api.attrApiId;
        this.arn = this.api.attrArn;
        this.graphQlUrl = this.api.attrGraphQlUrl;
        this.name = this.api.name;
        if (props.authorizationConfig) {
            this.setupAuth(props.authorizationConfig);
        }
        let definition;
        if (props.schemaDefinition) {
            definition = props.schemaDefinition;
        }
        else if (props.schemaDefinitionFile) {
            definition = fs_1.readFileSync(props.schemaDefinitionFile).toString('UTF-8');
        }
        else {
            throw new Error('Missing Schema definition. Provide schemaDefinition or schemaDefinitionFile');
        }
        this.schema = new appsync_generated_1.CfnGraphQLSchema(this, 'Schema', {
            apiId: this.apiId,
            definition,
        });
    }
    /**
     * the configured API key, if present
     */
    get apiKey() {
        return this._apiKey;
    }
    /**
     * add a new dummy data source to this API
     * @param name The name of the data source
     * @param description The description of the data source
     */
    addNoneDataSource(name, description) {
        return new NoneDataSource(this, `${name}DS`, {
            api: this,
            description,
            name,
        });
    }
    /**
     * add a new DynamoDB data source to this API
     * @param name The name of the data source
     * @param description The description of the data source
     * @param table The DynamoDB table backing this data source [disable-awslint:ref-via-interface]
     */
    addDynamoDbDataSource(name, description, table) {
        return new DynamoDbDataSource(this, `${name}DS`, {
            api: this,
            description,
            name,
            table,
        });
    }
    /**
     * add a new Lambda data source to this API
     * @param name The name of the data source
     * @param description The description of the data source
     * @param lambdaFunction The Lambda function to call to interact with this data source
     */
    addLambdaDataSource(name, description, lambdaFunction) {
        return new LambdaDataSource(this, `${name}DS`, {
            api: this,
            description,
            name,
            lambdaFunction,
        });
    }
    setupAuth(auth) {
        if (isUserPoolConfig(auth.defaultAuthorization)) {
            const { authenticationType, userPoolConfig } = this.userPoolDescFrom(auth.defaultAuthorization);
            this.api.authenticationType = authenticationType;
            this.api.userPoolConfig = userPoolConfig;
        }
        else if (isApiKeyConfig(auth.defaultAuthorization)) {
            this.api.authenticationType = this.apiKeyDesc(auth.defaultAuthorization).authenticationType;
        }
        this.api.additionalAuthenticationProviders = [];
        for (const mode of (auth.additionalAuthorizationModes || [])) {
            if (isUserPoolConfig(mode)) {
                this.api.additionalAuthenticationProviders.push(this.userPoolDescFrom(mode));
            }
            else if (isApiKeyConfig(mode)) {
                this.api.additionalAuthenticationProviders.push(this.apiKeyDesc(mode));
            }
        }
    }
    userPoolDescFrom(upConfig) {
        return {
            authenticationType: 'AMAZON_COGNITO_USER_POOLS',
            userPoolConfig: {
                appIdClientRegex: upConfig.appIdClientRegex,
                userPoolId: upConfig.userPool.userPoolId,
                awsRegion: upConfig.userPool.stack.region,
                defaultAction: upConfig.defaultAction ? upConfig.defaultAction.toString() : 'ALLOW',
            },
        };
    }
    apiKeyDesc(akConfig) {
        let expires;
        if (akConfig.expires) {
            expires = new Date(akConfig.expires).valueOf();
            const now = Date.now();
            const days = (d) => now + core_1.Duration.days(d).toMilliseconds();
            if (expires < days(1) || expires > days(365)) {
                throw Error('API key expiration must be between 1 and 365 days.');
            }
            expires = Math.round(expires / 1000);
        }
        const key = new appsync_generated_1.CfnApiKey(this, `${akConfig.apiKeyDesc || ''}ApiKey`, {
            expires,
            description: akConfig.apiKeyDesc,
            apiId: this.apiId,
        });
        this._apiKey = key.attrApiKey;
        return { authenticationType: 'API_KEY' };
    }
}
exports.GraphQLApi = GraphQLApi;
/**
 * Abstract AppSync datasource implementation. Do not use directly but use subclasses for concrete datasources
 */
class BaseDataSource extends core_1.Construct {
    constructor(scope, id, props, extended) {
        var _a;
        super(scope, id);
        if (extended.type !== 'NONE') {
            this.serviceRole = props.serviceRole || new aws_iam_1.Role(this, 'ServiceRole', { assumedBy: new aws_iam_1.ServicePrincipal('appsync') });
        }
        this.ds = new appsync_generated_1.CfnDataSource(this, 'Resource', {
            apiId: props.api.apiId,
            name: props.name,
            description: props.description,
            serviceRoleArn: (_a = this.serviceRole) === null || _a === void 0 ? void 0 : _a.roleArn,
            ...extended,
        });
        this.name = props.name;
        this.api = props.api;
    }
    /**
     * creates a new resolver for this datasource and API using the given properties
     */
    createResolver(props) {
        return new Resolver(this, `${props.typeName}${props.fieldName}Resolver`, {
            api: this.api,
            dataSource: this,
            ...props,
        });
    }
}
exports.BaseDataSource = BaseDataSource;
/**
 * Abstract AppSync datasource implementation. Do not use directly but use subclasses for resource backed datasources
 */
class BackedDataSource extends BaseDataSource {
    constructor(scope, id, props, extended) {
        super(scope, id, props, extended);
        this.grantPrincipal = this.serviceRole;
    }
}
exports.BackedDataSource = BackedDataSource;
/**
 * An AppSync dummy datasource
 */
class NoneDataSource extends BaseDataSource {
    constructor(scope, id, props) {
        super(scope, id, props, {
            type: 'NONE',
        });
    }
}
exports.NoneDataSource = NoneDataSource;
/**
 * An AppSync datasource backed by a DynamoDB table
 */
class DynamoDbDataSource extends BackedDataSource {
    constructor(scope, id, props) {
        super(scope, id, props, {
            type: 'AMAZON_DYNAMODB',
            dynamoDbConfig: {
                tableName: props.table.tableName,
                awsRegion: props.table.stack.region,
                useCallerCredentials: props.useCallerCredentials,
            },
        });
        if (props.readOnlyAccess) {
            props.table.grantReadData(this);
        }
        else {
            props.table.grantReadWriteData(this);
        }
    }
}
exports.DynamoDbDataSource = DynamoDbDataSource;
/**
 * An AppSync datasource backed by a Lambda function
 */
class LambdaDataSource extends BackedDataSource {
    constructor(scope, id, props) {
        super(scope, id, props, {
            type: 'AWS_LAMBDA',
            lambdaConfig: {
                lambdaFunctionArn: props.lambdaFunction.functionArn,
            },
        });
        props.lambdaFunction.grantInvoke(this);
    }
}
exports.LambdaDataSource = LambdaDataSource;
function concatAndDedup(left, right) {
    return left.concat(right).filter((elem, index, self) => {
        return index === self.indexOf(elem);
    });
}
/**
 * Utility class to represent DynamoDB key conditions.
 */
class BaseKeyCondition {
    and(cond) {
        return new (class extends BaseKeyCondition {
            constructor(left, right) {
                super();
                this.left = left;
                this.right = right;
            }
            renderCondition() {
                return `${this.left.renderCondition()} AND ${this.right.renderCondition()}`;
            }
            keyNames() {
                return concatAndDedup(this.left.keyNames(), this.right.keyNames());
            }
            args() {
                return concatAndDedup(this.left.args(), this.right.args());
            }
        })(this, cond);
    }
    renderExpressionNames() {
        return this.keyNames()
            .map((keyName) => {
            return `"#${keyName}" : "${keyName}"`;
        })
            .join(', ');
    }
    renderExpressionValues() {
        return this.args()
            .map((arg) => {
            return `":${arg}" : $util.dynamodb.toDynamoDBJson($ctx.args.${arg})`;
        })
            .join(', ');
    }
}
/**
 * Utility class to represent DynamoDB "begins_with" key conditions.
 */
class BeginsWith extends BaseKeyCondition {
    constructor(keyName, arg) {
        super();
        this.keyName = keyName;
        this.arg = arg;
    }
    renderCondition() {
        return `begins_with(#${this.keyName}, :${this.arg})`;
    }
    keyNames() {
        return [this.keyName];
    }
    args() {
        return [this.arg];
    }
}
/**
 * Utility class to represent DynamoDB binary key conditions.
 */
class BinaryCondition extends BaseKeyCondition {
    constructor(keyName, op, arg) {
        super();
        this.keyName = keyName;
        this.op = op;
        this.arg = arg;
    }
    renderCondition() {
        return `#${this.keyName} ${this.op} :${this.arg}`;
    }
    keyNames() {
        return [this.keyName];
    }
    args() {
        return [this.arg];
    }
}
/**
 * Utility class to represent DynamoDB "between" key conditions.
 */
class Between extends BaseKeyCondition {
    constructor(keyName, arg1, arg2) {
        super();
        this.keyName = keyName;
        this.arg1 = arg1;
        this.arg2 = arg2;
    }
    renderCondition() {
        return `#${this.keyName} BETWEEN :${this.arg1} AND :${this.arg2}`;
    }
    keyNames() {
        return [this.keyName];
    }
    args() {
        return [this.arg1, this.arg2];
    }
}
/**
 * Factory class for DynamoDB key conditions.
 */
class KeyCondition {
    constructor(cond) {
        this.cond = cond;
    }
    /**
     * Condition k = arg, true if the key attribute k is equal to the Query argument
     */
    static eq(keyName, arg) {
        return new KeyCondition(new BinaryCondition(keyName, '=', arg));
    }
    /**
     * Condition k < arg, true if the key attribute k is less than the Query argument
     */
    static lt(keyName, arg) {
        return new KeyCondition(new BinaryCondition(keyName, '<', arg));
    }
    /**
     * Condition k <= arg, true if the key attribute k is less than or equal to the Query argument
     */
    static le(keyName, arg) {
        return new KeyCondition(new BinaryCondition(keyName, '<=', arg));
    }
    /**
     * Condition k > arg, true if the key attribute k is greater than the the Query argument
     */
    static gt(keyName, arg) {
        return new KeyCondition(new BinaryCondition(keyName, '>', arg));
    }
    /**
     * Condition k >= arg, true if the key attribute k is greater or equal to the Query argument
     */
    static ge(keyName, arg) {
        return new KeyCondition(new BinaryCondition(keyName, '>=', arg));
    }
    /**
     * Condition (k, arg). True if the key attribute k begins with the Query argument.
     */
    static beginsWith(keyName, arg) {
        return new KeyCondition(new BeginsWith(keyName, arg));
    }
    /**
     * Condition k BETWEEN arg1 AND arg2, true if k >= arg1 and k <= arg2.
     */
    static between(keyName, arg1, arg2) {
        return new KeyCondition(new Between(keyName, arg1, arg2));
    }
    /**
     * Conjunction between two conditions.
     */
    and(keyCond) {
        return new KeyCondition(this.cond.and(keyCond.cond));
    }
    /**
     * Renders the key condition to a VTL string.
     */
    renderTemplate() {
        return `"query" : {
            "expression" : "${this.cond.renderCondition()}",
            "expressionNames" : {
                ${this.cond.renderExpressionNames()}
            },
            "expressionValues" : {
                ${this.cond.renderExpressionValues()}
            }
        }`;
    }
}
exports.KeyCondition = KeyCondition;
/**
 * Utility class representing the assigment of a value to an attribute.
 */
class Assign {
    constructor(attr, arg) {
        this.attr = attr;
        this.arg = arg;
    }
    /**
     * Renders the assignment as a VTL string.
     */
    renderAsAssignment() {
        return `"${this.attr}" : $util.dynamodb.toDynamoDBJson(${this.arg})`;
    }
    /**
     * Renders the assignment as a map element.
     */
    putInMap(map) {
        return `$util.qr($${map}.put("${this.attr}", "${this.arg}"))`;
    }
}
exports.Assign = Assign;
/**
 * Utility class to allow assigning a value or an auto-generated id
 * to a partition key.
 */
class PartitionKeyStep {
    constructor(key) {
        this.key = key;
    }
    /**
     * Assign an auto-generated value to the partition key.
     */
    is(val) {
        return new PartitionKey(new Assign(this.key, `$ctx.args.${val}`));
    }
    /**
     * Assign an auto-generated value to the partition key.
     */
    auto() {
        return new PartitionKey(new Assign(this.key, '$util.autoId()'));
    }
}
exports.PartitionKeyStep = PartitionKeyStep;
/**
 * Utility class to allow assigning a value or an auto-generated id
 * to a sort key.
 */
class SortKeyStep {
    constructor(pkey, skey) {
        this.pkey = pkey;
        this.skey = skey;
    }
    /**
     * Assign an auto-generated value to the sort key.
     */
    is(val) {
        return new PrimaryKey(this.pkey, new Assign(this.skey, `$ctx.args.${val}`));
    }
    /**
     * Assign an auto-generated value to the sort key.
     */
    auto() {
        return new PrimaryKey(this.pkey, new Assign(this.skey, '$util.autoId()'));
    }
}
exports.SortKeyStep = SortKeyStep;
/**
 * Specifies the assignment to the primary key. It either
 * contains the full primary key or only the partition key.
 */
class PrimaryKey {
    constructor(pkey, skey) {
        this.pkey = pkey;
        this.skey = skey;
    }
    /**
     * Allows assigning a value to the partition key.
     */
    static partition(key) {
        return new PartitionKeyStep(key);
    }
    /**
     * Renders the key assignment to a VTL string.
     */
    renderTemplate() {
        const assignments = [this.pkey.renderAsAssignment()];
        if (this.skey) {
            assignments.push(this.skey.renderAsAssignment());
        }
        return `"key" : {
            ${assignments.join(',')}
        }`;
    }
}
exports.PrimaryKey = PrimaryKey;
/**
 * Specifies the assignment to the partition key. It can be
 * enhanced with the assignment of the sort key.
 */
class PartitionKey extends PrimaryKey {
    constructor(pkey) {
        super(pkey);
    }
    /**
     * Allows assigning a value to the sort key.
     */
    sort(key) {
        return new SortKeyStep(this.pkey, key);
    }
}
exports.PartitionKey = PartitionKey;
/**
 * Specifies the attribute value assignments.
 */
class AttributeValues {
    constructor(container, assignments = []) {
        this.container = container;
        this.assignments = assignments;
    }
    /**
     * Allows assigning a value to the specified attribute.
     */
    attribute(attr) {
        return new AttributeValuesStep(attr, this.container, this.assignments);
    }
    /**
     * Renders the attribute value assingments to a VTL string.
     */
    renderTemplate() {
        return `
            #set($input = ${this.container})
            ${this.assignments.map(a => a.putInMap('input')).join('\n')}
            "attributeValues": $util.dynamodb.toMapValuesJson($input)`;
    }
}
exports.AttributeValues = AttributeValues;
/**
 * Utility class to allow assigning a value to an attribute.
 */
class AttributeValuesStep {
    constructor(attr, container, assignments) {
        this.attr = attr;
        this.container = container;
        this.assignments = assignments;
    }
    /**
     * Assign the value to the current attribute.
     */
    is(val) {
        this.assignments.push(new Assign(this.attr, val));
        return new AttributeValues(this.container, this.assignments);
    }
}
exports.AttributeValuesStep = AttributeValuesStep;
/**
 * Factory class for attribute value assignments.
 */
class Values {
    /**
     * Treats the specified object as a map of assignments, where the property
     * names represent attribute names. It’s opinionated about how it represents
     * some of the nested objects: e.g., it will use lists (“L”) rather than sets
     * (“SS”, “NS”, “BS”). By default it projects the argument container ("$ctx.args").
     */
    static projecting(arg) {
        return new AttributeValues('$ctx.args' + (arg ? `.${arg}` : ''));
    }
    /**
     * Allows assigning a value to the specified attribute.
     */
    static attribute(attr) {
        return new AttributeValues('{}').attribute(attr);
    }
}
exports.Values = Values;
/**
 * MappingTemplates for AppSync resolvers
 */
class MappingTemplate {
    /**
     * Create a mapping template from the given string
     */
    static fromString(template) {
        return new StringMappingTemplate(template);
    }
    /**
     * Create a mapping template from the given file
     */
    static fromFile(fileName) {
        return new StringMappingTemplate(fs_1.readFileSync(fileName).toString('UTF-8'));
    }
    /**
     * Mapping template for a result list from DynamoDB
     */
    static dynamoDbResultList() {
        return this.fromString('$util.toJson($ctx.result.items)');
    }
    /**
     * Mapping template for a single result item from DynamoDB
     */
    static dynamoDbResultItem() {
        return this.fromString('$util.toJson($ctx.result)');
    }
    /**
     * Mapping template to scan a DynamoDB table to fetch all entries
     */
    static dynamoDbScanTable() {
        return this.fromString('{"version" : "2017-02-28", "operation" : "Scan"}');
    }
    /**
     * Mapping template to query a set of items from a DynamoDB table
     *
     * @param cond the key condition for the query
     */
    static dynamoDbQuery(cond) {
        return this.fromString(`{"version" : "2017-02-28", "operation" : "Query", ${cond.renderTemplate()}}`);
    }
    /**
     * Mapping template to get a single item from a DynamoDB table
     *
     * @param keyName the name of the hash key field
     * @param idArg the name of the Query argument
     */
    static dynamoDbGetItem(keyName, idArg) {
        return this.fromString(`{"version": "2017-02-28", "operation": "GetItem", "key": {"${keyName}": $util.dynamodb.toDynamoDBJson($ctx.args.${idArg})}}`);
    }
    /**
     * Mapping template to delete a single item from a DynamoDB table
     *
     * @param keyName the name of the hash key field
     * @param idArg the name of the Mutation argument
     */
    static dynamoDbDeleteItem(keyName, idArg) {
        return this.fromString(`{"version": "2017-02-28", "operation": "DeleteItem", "key": {"${keyName}": $util.dynamodb.toDynamoDBJson($ctx.args.${idArg})}}`);
    }
    /**
     * Mapping template to save a single item to a DynamoDB table
     *
     * @param key the assigment of Mutation values to the primary key
     * @param values the assignment of Mutation values to the table attributes
     */
    static dynamoDbPutItem(key, values) {
        return this.fromString(`{
            "version" : "2017-02-28",
            "operation" : "PutItem",
            ${key.renderTemplate()},
            ${values.renderTemplate()}
        }`);
    }
    /**
     * Mapping template to invoke a Lambda function
     *
     * @param payload the VTL template snippet of the payload to send to the lambda.
     * If no payload is provided all available context fields are sent to the Lambda function
     */
    static lambdaRequest(payload = '$util.toJson($ctx)') {
        return this.fromString(`{"version": "2017-02-28", "operation": "Invoke", "payload": ${payload}}`);
    }
    /**
     * Mapping template to return the Lambda result to the caller
     */
    static lambdaResult() {
        return this.fromString('$util.toJson($ctx.result)');
    }
}
exports.MappingTemplate = MappingTemplate;
class StringMappingTemplate extends MappingTemplate {
    constructor(template) {
        super();
        this.template = template;
    }
    renderTemplate() {
        return this.template;
    }
}
/**
 * An AppSync resolver
 */
class Resolver extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.resolver = new appsync_generated_1.CfnResolver(this, 'Resource', {
            apiId: props.api.apiId,
            typeName: props.typeName,
            fieldName: props.fieldName,
            dataSourceName: props.dataSource ? props.dataSource.name : undefined,
            kind: props.pipelineConfig ? 'PIPELINE' : 'UNIT',
            requestMappingTemplate: props.requestMappingTemplate ? props.requestMappingTemplate.renderTemplate() : undefined,
            responseMappingTemplate: props.responseMappingTemplate ? props.responseMappingTemplate.renderTemplate() : undefined,
        });
        this.resolver.addDependsOn(props.api.schema);
        if (props.dataSource) {
            this.resolver.addDependsOn(props.dataSource.ds);
        }
        this.arn = this.resolver.attrResolverArn;
    }
}
exports.Resolver = Resolver;
//# sourceMappingURL=data:application/json;base64,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