"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const appmesh_generated_1 = require("./appmesh.generated");
const shared_interfaces_1 = require("./shared-interfaces");
class VirtualNodeBase extends cdk.Resource {
    constructor() {
        super(...arguments);
        this.backends = new Array();
        this.listeners = new Array();
    }
    /**
     * Add a Virtual Services that this node is expected to send outbound traffic to
     */
    addBackends(...props) {
        for (const s of props) {
            this.backends.push({
                virtualService: {
                    virtualServiceName: s.virtualServiceName,
                },
            });
        }
    }
    /**
     * Utility method to add an inbound listener for this virtual node
     */
    addListeners(...listeners) {
        if (this.listeners.length + listeners.length > 1) {
            throw new Error('VirtualNode may have at most one listener');
        }
        for (const listener of listeners) {
            const portMapping = listener.portMapping || { port: 8080, protocol: shared_interfaces_1.Protocol.HTTP };
            this.listeners.push({
                portMapping,
                healthCheck: renderHealthCheck(listener.healthCheck, portMapping),
            });
        }
    }
}
/**
 * Minimum and maximum thresholds for HeathCheck numeric properties
 *
 * @see https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_HealthCheckPolicy.html
 */
const HEALTH_CHECK_PROPERTY_THRESHOLDS = {
    healthyThreshold: [2, 10],
    intervalMillis: [5000, 300000],
    port: [1, 65535],
    timeoutMillis: [2000, 60000],
    unhealthyThreshold: [2, 10],
};
function renderHealthCheck(hc, pm) {
    if (hc === undefined) {
        return undefined;
    }
    if (hc.protocol === shared_interfaces_1.Protocol.TCP && hc.path) {
        throw new Error('The path property cannot be set with Protocol.TCP');
    }
    const healthCheck = {
        healthyThreshold: hc.healthyThreshold || 2,
        intervalMillis: (hc.interval || cdk.Duration.seconds(5)).toMilliseconds(),
        path: hc.path || (hc.protocol === shared_interfaces_1.Protocol.HTTP ? '/' : undefined),
        port: hc.port || pm.port,
        protocol: hc.protocol || pm.protocol,
        timeoutMillis: (hc.timeout || cdk.Duration.seconds(2)).toMilliseconds(),
        unhealthyThreshold: hc.unhealthyThreshold || 2,
    };
    Object.keys(healthCheck)
        .filter((key) => HEALTH_CHECK_PROPERTY_THRESHOLDS[key] &&
        typeof healthCheck[key] === 'number' &&
        !cdk.Token.isUnresolved(healthCheck[key])).map((key) => {
        const [min, max] = HEALTH_CHECK_PROPERTY_THRESHOLDS[key];
        const value = healthCheck[key];
        if (value < min) {
            throw new Error(`The value of '${key}' is below the minimum threshold (expected >=${min}, got ${value})`);
        }
        if (value > max) {
            throw new Error(`The value of '${key}' is above the maximum threshold (expected <=${max}, got ${value})`);
        }
    });
    return healthCheck;
}
/**
 * VirtualNode represents a newly defined AppMesh VirtualNode
 *
 * Any inbound traffic that your virtual node expects should be specified as a
 * listener. Any outbound traffic that your virtual node expects to reach
 * should be specified as a backend.
 *
 * @see https://docs.aws.amazon.com/app-mesh/latest/userguide/virtual_nodes.html
 */
class VirtualNode extends VirtualNodeBase {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.virtualNodeName || cdk.Lazy.stringValue({ produce: () => this.node.uniqueId }),
        });
        this.mesh = props.mesh;
        this.addBackends(...props.backends || []);
        this.addListeners(...props.listener ? [props.listener] : []);
        const node = new appmesh_generated_1.CfnVirtualNode(this, 'Resource', {
            virtualNodeName: this.physicalName,
            meshName: this.mesh.meshName,
            spec: {
                backends: cdk.Lazy.anyValue({ produce: () => this.backends }, { omitEmptyArray: true }),
                listeners: cdk.Lazy.anyValue({ produce: () => this.listeners }, { omitEmptyArray: true }),
                serviceDiscovery: {
                    dns: props.dnsHostName !== undefined ? { hostname: props.dnsHostName } : undefined,
                    awsCloudMap: props.cloudMapService !== undefined ? {
                        serviceName: props.cloudMapService.serviceName,
                        namespaceName: props.cloudMapService.namespace.namespaceName,
                        attributes: renderAttributes(props.cloudMapServiceInstanceAttributes),
                    } : undefined,
                },
                logging: {
                    accessLog: {
                        file: {
                            path: '/dev/stdout',
                        },
                    },
                },
            },
        });
        this.virtualNodeName = this.getResourceNameAttribute(node.attrVirtualNodeName);
        this.virtualNodeArn = this.getResourceArnAttribute(node.ref, {
            service: 'appmesh',
            resource: `mesh/${props.mesh.meshName}/virtualNode`,
            resourceName: this.physicalName,
        });
    }
    /**
     * Import an existing VirtualNode given an ARN
     */
    static fromVirtualNodeArn(scope, id, virtualNodeArn) {
        return new ImportedVirtualNode(scope, id, { virtualNodeArn });
    }
    /**
     * Import an existing VirtualNode given its name
     */
    static fromVirtualNodeName(scope, id, meshName, virtualNodeName) {
        return new ImportedVirtualNode(scope, id, {
            meshName,
            virtualNodeName,
        });
    }
}
exports.VirtualNode = VirtualNode;
function renderAttributes(attrs) {
    if (attrs === undefined) {
        return undefined;
    }
    return Object.entries(attrs).map(([key, value]) => ({ key, value }));
}
/**
 * Used to import a VirtualNode and read it's properties
 */
class ImportedVirtualNode extends VirtualNodeBase {
    constructor(scope, id, props) {
        super(scope, id);
        if (props.virtualNodeArn) {
            this.virtualNodeArn = props.virtualNodeArn;
            this.virtualNodeName = cdk.Fn.select(2, cdk.Fn.split('/', cdk.Stack.of(scope).parseArn(props.virtualNodeArn).resourceName));
        }
        else if (props.virtualNodeName && props.meshName) {
            this.virtualNodeName = props.virtualNodeName;
            this.virtualNodeArn = cdk.Stack.of(this).formatArn({
                service: 'appmesh',
                resource: `mesh/${props.meshName}/virtualNode`,
                resourceName: this.virtualNodeName,
            });
        }
        else {
            throw new Error('Need either arn or both names');
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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