"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const applicationautoscaling_generated_1 = require("./applicationautoscaling.generated");
/**
 * Define a step scaling action
 *
 * This kind of scaling policy adjusts the target capacity in configurable
 * steps. The size of the step is configurable based on the metric's distance
 * to its alarm threshold.
 *
 * This Action must be used as the target of a CloudWatch alarm to take effect.
 */
class StepScalingAction extends cdk.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.adjustments = new Array();
        // Cloudformation requires either the ResourceId, ScalableDimension, and ServiceNamespace
        // properties, or the ScalingTargetId property, but not both.
        // https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-applicationautoscaling-scalingpolicy.html
        const resource = new applicationautoscaling_generated_1.CfnScalingPolicy(this, 'Resource', {
            policyName: props.policyName || this.node.uniqueId,
            policyType: 'StepScaling',
            scalingTargetId: props.scalingTarget.scalableTargetId,
            stepScalingPolicyConfiguration: {
                adjustmentType: props.adjustmentType,
                cooldown: props.cooldown && props.cooldown.toSeconds(),
                minAdjustmentMagnitude: props.minAdjustmentMagnitude,
                metricAggregationType: props.metricAggregationType,
                stepAdjustments: cdk.Lazy.anyValue({ produce: () => this.adjustments }),
            },
        });
        this.scalingPolicyArn = resource.ref;
    }
    /**
     * Add an adjusment interval to the ScalingAction
     */
    addAdjustment(adjustment) {
        if (adjustment.lowerBound === undefined && adjustment.upperBound === undefined) {
            throw new Error('At least one of lowerBound or upperBound is required');
        }
        this.adjustments.push({
            metricIntervalLowerBound: adjustment.lowerBound,
            metricIntervalUpperBound: adjustment.upperBound,
            scalingAdjustment: adjustment.adjustment,
        });
    }
}
exports.StepScalingAction = StepScalingAction;
/**
 * How adjustment numbers are interpreted
 */
var AdjustmentType;
(function (AdjustmentType) {
    /**
     * Add the adjustment number to the current capacity.
     *
     * A positive number increases capacity, a negative number decreases capacity.
     */
    AdjustmentType["CHANGE_IN_CAPACITY"] = "ChangeInCapacity";
    /**
     * Add this percentage of the current capacity to itself.
     *
     * The number must be between -100 and 100; a positive number increases
     * capacity and a negative number decreases it.
     */
    AdjustmentType["PERCENT_CHANGE_IN_CAPACITY"] = "PercentChangeInCapacity";
    /**
     * Make the capacity equal to the exact number given.
     */
    AdjustmentType["EXACT_CAPACITY"] = "ExactCapacity";
})(AdjustmentType = exports.AdjustmentType || (exports.AdjustmentType = {}));
/**
 * How the scaling metric is going to be aggregated
 */
var MetricAggregationType;
(function (MetricAggregationType) {
    /**
     * Average
     */
    MetricAggregationType["AVERAGE"] = "Average";
    /**
     * Minimum
     */
    MetricAggregationType["MINIMUM"] = "Minimum";
    /**
     * Maximum
     */
    MetricAggregationType["MAXIMUM"] = "Maximum";
})(MetricAggregationType = exports.MetricAggregationType || (exports.MetricAggregationType = {}));
//# sourceMappingURL=data:application/json;base64,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