"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const applicationautoscaling_generated_1 = require("./applicationautoscaling.generated");
const step_scaling_policy_1 = require("./step-scaling-policy");
const target_tracking_scaling_policy_1 = require("./target-tracking-scaling-policy");
/**
 * Define a scalable target
 */
class ScalableTarget extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        this.actions = new Array();
        core_1.withResolved(props.maxCapacity, max => {
            if (max < 0) {
                throw new RangeError(`maxCapacity cannot be negative, got: ${props.maxCapacity}`);
            }
        });
        core_1.withResolved(props.minCapacity, min => {
            if (min < 0) {
                throw new RangeError(`minCapacity cannot be negative, got: ${props.minCapacity}`);
            }
        });
        core_1.withResolved(props.minCapacity, props.maxCapacity, (min, max) => {
            if (max < min) {
                throw new RangeError(`minCapacity (${props.minCapacity}) should be lower than maxCapacity (${props.maxCapacity})`);
            }
        });
        this.role = props.role || new iam.Role(this, 'Role', {
            assumedBy: new iam.ServicePrincipal('application-autoscaling.amazonaws.com'),
        });
        const resource = new applicationautoscaling_generated_1.CfnScalableTarget(this, 'Resource', {
            maxCapacity: props.maxCapacity,
            minCapacity: props.minCapacity,
            resourceId: props.resourceId,
            roleArn: this.role.roleArn,
            scalableDimension: props.scalableDimension,
            scheduledActions: core_1.Lazy.anyValue({ produce: () => this.actions }, { omitEmptyArray: true }),
            serviceNamespace: props.serviceNamespace,
        });
        this.scalableTargetId = resource.ref;
    }
    static fromScalableTargetId(scope, id, scalableTargetId) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.scalableTargetId = scalableTargetId;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Add a policy statement to the role's policy
     */
    addToRolePolicy(statement) {
        this.role.addToPolicy(statement);
    }
    /**
     * Scale out or in based on time
     */
    scaleOnSchedule(id, action) {
        if (action.minCapacity === undefined && action.maxCapacity === undefined) {
            throw new Error(`You must supply at least one of minCapacity or maxCapacity, got ${JSON.stringify(action)}`);
        }
        this.actions.push({
            scheduledActionName: id,
            schedule: action.schedule.expressionString,
            startTime: action.startTime,
            endTime: action.endTime,
            scalableTargetAction: {
                maxCapacity: action.maxCapacity,
                minCapacity: action.minCapacity,
            },
        });
    }
    /**
     * Scale out or in, in response to a metric
     */
    scaleOnMetric(id, props) {
        return new step_scaling_policy_1.StepScalingPolicy(this, id, { ...props, scalingTarget: this });
    }
    /**
     * Scale out or in in order to keep a metric around a target value
     */
    scaleToTrackMetric(id, props) {
        return new target_tracking_scaling_policy_1.TargetTrackingScalingPolicy(this, id, { ...props, scalingTarget: this });
    }
}
exports.ScalableTarget = ScalableTarget;
/**
 * The service that supports Application AutoScaling
 */
var ServiceNamespace;
(function (ServiceNamespace) {
    /**
     * Elastic Container Service
     */
    ServiceNamespace["ECS"] = "ecs";
    /**
     * Elastic Map Reduce
     */
    ServiceNamespace["ELASTIC_MAP_REDUCE"] = "elasticmapreduce";
    /**
     * Elastic Compute Cloud
     */
    ServiceNamespace["EC2"] = "ec2";
    /**
     * App Stream
     */
    ServiceNamespace["APPSTREAM"] = "appstream";
    /**
     * Dynamo DB
     */
    ServiceNamespace["DYNAMODB"] = "dynamodb";
    /**
     * Relational Database Service
     */
    ServiceNamespace["RDS"] = "rds";
    /**
     * SageMaker
     */
    ServiceNamespace["SAGEMAKER"] = "sagemaker";
    /**
     * Custom Resource
     */
    ServiceNamespace["CUSTOM_RESOURCE"] = "custom-resource";
    /**
     * Lambda
     */
    ServiceNamespace["LAMBDA"] = "lambda";
    /**
     * Comprehend
     */
    ServiceNamespace["COMPREHEND"] = "comprehend";
})(ServiceNamespace = exports.ServiceNamespace || (exports.ServiceNamespace = {}));
//# sourceMappingURL=data:application/json;base64,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