"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const apigateway_generated_1 = require("./apigateway.generated");
const util_1 = require("./util");
/**
 * Time period for which quota settings apply.
 */
var Period;
(function (Period) {
    Period["DAY"] = "DAY";
    Period["WEEK"] = "WEEK";
    Period["MONTH"] = "MONTH";
})(Period = exports.Period || (exports.Period = {}));
class UsagePlan extends core_1.Resource {
    constructor(scope, id, props = {}) {
        super(scope, id);
        this.apiStages = new Array();
        let resource;
        resource = new apigateway_generated_1.CfnUsagePlan(this, 'Resource', {
            apiStages: core_1.Lazy.anyValue({ produce: () => this.renderApiStages(this.apiStages) }),
            description: props.description,
            quota: this.renderQuota(props),
            throttle: this.renderThrottle(props.throttle),
            usagePlanName: props.name,
        });
        this.apiStages.push(...(props.apiStages || []));
        this.usagePlanId = resource.ref;
        // Add ApiKey when
        if (props.apiKey) {
            this.addApiKey(props.apiKey);
        }
    }
    /**
     * Adds an ApiKey.
     *
     * @param apiKey
     */
    addApiKey(apiKey) {
        const prefix = 'UsagePlanKeyResource';
        // Postfixing apikey id only from the 2nd child, to keep physicalIds of UsagePlanKey for existing CDK apps unmodifed.
        const id = this.node.tryFindChild(prefix) ? `${prefix}:${apiKey.node.uniqueId}` : prefix;
        new apigateway_generated_1.CfnUsagePlanKey(this, id, {
            keyId: apiKey.keyId,
            keyType: "API_KEY" /* API_KEY */,
            usagePlanId: this.usagePlanId,
        });
    }
    /**
     * Adds an apiStage.
     * @param apiStage
     */
    addApiStage(apiStage) {
        this.apiStages.push(apiStage);
    }
    /**
     *
     * @param props
     */
    renderApiStages(apiStages) {
        if (apiStages && apiStages.length > 0) {
            const stages = [];
            apiStages.forEach((apiStage) => {
                stages.push(this.createStage(apiStage));
            });
            return stages;
        }
        return undefined;
    }
    createStage(apiStage) {
        const stage = apiStage.stage ? apiStage.stage.stageName.toString() : undefined;
        const apiId = apiStage.stage ? apiStage.stage.restApi.restApiId : undefined;
        const throttle = this.renderThrottlePerMethod(apiStage.throttle);
        return {
            apiId,
            stage,
            throttle,
        };
    }
    renderQuota(props) {
        if (props.quota === undefined) {
            return undefined;
        }
        else {
            const limit = props.quota ? props.quota.limit : undefined;
            util_1.validateInteger(limit, 'Throttle quota limit');
            const ret = {
                limit: limit ? limit : undefined,
                offset: props.quota ? props.quota.offset : undefined,
                period: props.quota ? props.quota.period : undefined,
            };
            return ret;
        }
    }
    renderThrottle(props) {
        let ret;
        if (props !== undefined) {
            const burstLimit = props.burstLimit;
            util_1.validateInteger(burstLimit, 'Throttle burst limit');
            const rateLimit = props.rateLimit;
            util_1.validateInteger(rateLimit, 'Throttle rate limit');
            ret = {
                burstLimit: burstLimit ? burstLimit : undefined,
                rateLimit: rateLimit ? rateLimit : undefined,
            };
        }
        return ret;
    }
    renderThrottlePerMethod(throttlePerMethod) {
        const ret = {};
        if (throttlePerMethod && throttlePerMethod.length > 0) {
            throttlePerMethod.forEach((value) => {
                const method = value.method;
                // this methodId is resource path and method for example /GET or /pets/GET
                const methodId = `${method.resource.path}/${method.httpMethod}`;
                ret[methodId] = this.renderThrottle(value.throttle);
            });
        }
        return ret;
    }
}
exports.UsagePlan = UsagePlan;
//# sourceMappingURL=data:application/json;base64,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