"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const access_log_1 = require("./access-log");
const apigateway_generated_1 = require("./apigateway.generated");
const util_1 = require("./util");
var MethodLoggingLevel;
(function (MethodLoggingLevel) {
    MethodLoggingLevel["OFF"] = "OFF";
    MethodLoggingLevel["ERROR"] = "ERROR";
    MethodLoggingLevel["INFO"] = "INFO";
})(MethodLoggingLevel = exports.MethodLoggingLevel || (exports.MethodLoggingLevel = {}));
class Stage extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        this.enableCacheCluster = props.cacheClusterEnabled;
        const methodSettings = this.renderMethodSettings(props); // this can mutate `this.cacheClusterEnabled`
        // custom access logging
        let accessLogSetting;
        const accessLogDestination = props.accessLogDestination;
        const accessLogFormat = props.accessLogFormat;
        if (!accessLogDestination && !accessLogFormat) {
            accessLogSetting = undefined;
        }
        else {
            if (accessLogFormat !== undefined && !/.*\$context.requestId.*/.test(accessLogFormat.toString())) {
                throw new Error('Access log must include at least `AccessLogFormat.contextRequestId()`');
            }
            if (accessLogFormat !== undefined && accessLogDestination === undefined) {
                throw new Error('Access log format is specified without a destination');
            }
            accessLogSetting = {
                destinationArn: accessLogDestination === null || accessLogDestination === void 0 ? void 0 : accessLogDestination.bind(this).destinationArn,
                format: (accessLogFormat === null || accessLogFormat === void 0 ? void 0 : accessLogFormat.toString()) ? accessLogFormat === null || accessLogFormat === void 0 ? void 0 : accessLogFormat.toString() : access_log_1.AccessLogFormat.clf().toString(),
            };
        }
        // enable cache cluster if cacheClusterSize is set
        if (props.cacheClusterSize !== undefined) {
            if (this.enableCacheCluster === undefined) {
                this.enableCacheCluster = true;
            }
            else if (this.enableCacheCluster === false) {
                throw new Error(`Cannot set "cacheClusterSize" to ${props.cacheClusterSize} and "cacheClusterEnabled" to "false"`);
            }
        }
        const cacheClusterSize = this.enableCacheCluster ? (props.cacheClusterSize || '0.5') : undefined;
        const resource = new apigateway_generated_1.CfnStage(this, 'Resource', {
            stageName: props.stageName || 'prod',
            accessLogSetting,
            cacheClusterEnabled: this.enableCacheCluster,
            cacheClusterSize,
            clientCertificateId: props.clientCertificateId,
            deploymentId: props.deployment.deploymentId,
            restApiId: props.deployment.api.restApiId,
            description: props.description,
            documentationVersion: props.documentationVersion,
            variables: props.variables,
            tracingEnabled: props.tracingEnabled,
            methodSettings,
        });
        this.stageName = resource.ref;
        this.restApi = props.deployment.api;
    }
    /**
     * Returns the invoke URL for a certain path.
     * @param path The resource path
     */
    urlForPath(path = '/') {
        if (!path.startsWith('/')) {
            throw new Error(`Path must begin with "/": ${path}`);
        }
        return `https://${this.restApi.restApiId}.execute-api.${core_1.Stack.of(this).region}.${core_1.Stack.of(this).urlSuffix}/${this.stageName}${path}`;
    }
    renderMethodSettings(props) {
        const settings = new Array();
        const self = this;
        // extract common method options from the stage props
        const commonMethodOptions = {
            metricsEnabled: props.metricsEnabled,
            loggingLevel: props.loggingLevel,
            dataTraceEnabled: props.dataTraceEnabled,
            throttlingBurstLimit: props.throttlingBurstLimit,
            throttlingRateLimit: props.throttlingRateLimit,
            cachingEnabled: props.cachingEnabled,
            cacheTtl: props.cacheTtl,
            cacheDataEncrypted: props.cacheDataEncrypted,
        };
        // if any of them are defined, add an entry for '/*/*'.
        const hasCommonOptions = Object.keys(commonMethodOptions).map(v => commonMethodOptions[v]).filter(x => x).length > 0;
        if (hasCommonOptions) {
            settings.push(renderEntry('/*/*', commonMethodOptions));
        }
        if (props.methodOptions) {
            for (const path of Object.keys(props.methodOptions)) {
                settings.push(renderEntry(path, props.methodOptions[path]));
            }
        }
        return settings.length === 0 ? undefined : settings;
        function renderEntry(path, options) {
            if (options.cachingEnabled) {
                if (self.enableCacheCluster === undefined) {
                    self.enableCacheCluster = true;
                }
                else if (self.enableCacheCluster === false) {
                    throw new Error(`Cannot enable caching for method ${path} since cache cluster is disabled on stage`);
                }
            }
            const { httpMethod, resourcePath } = util_1.parseMethodOptionsPath(path);
            return {
                httpMethod, resourcePath,
                cacheDataEncrypted: options.cacheDataEncrypted,
                cacheTtlInSeconds: options.cacheTtl && options.cacheTtl.toSeconds(),
                cachingEnabled: options.cachingEnabled,
                dataTraceEnabled: options.dataTraceEnabled,
                loggingLevel: options.loggingLevel,
                metricsEnabled: options.metricsEnabled,
                throttlingBurstLimit: options.throttlingBurstLimit,
                throttlingRateLimit: options.throttlingRateLimit,
            };
        }
    }
}
exports.Stage = Stage;
//# sourceMappingURL=data:application/json;base64,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