"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const apigateway_generated_1 = require("./apigateway.generated");
const authorizer_1 = require("./authorizer");
const integration_1 = require("./integration");
const mock_1 = require("./integrations/mock");
const util_1 = require("./util");
class Method extends core_1.Resource {
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        this.resource = props.resource;
        this.restApi = props.resource.restApi;
        this.httpMethod = props.httpMethod.toUpperCase();
        util_1.validateHttpMethod(this.httpMethod);
        const options = props.options || {};
        const defaultMethodOptions = props.resource.defaultMethodOptions || {};
        const authorizer = options.authorizer || defaultMethodOptions.authorizer;
        const authorizerId = authorizer === null || authorizer === void 0 ? void 0 : authorizer.authorizerId;
        const authorizationTypeOption = options.authorizationType || defaultMethodOptions.authorizationType;
        const authorizationType = (authorizer === null || authorizer === void 0 ? void 0 : authorizer.authorizationType) || authorizationTypeOption || AuthorizationType.NONE;
        // if the authorizer defines an authorization type and we also have an explicit option set, check that they are the same
        if ((authorizer === null || authorizer === void 0 ? void 0 : authorizer.authorizationType) && authorizationTypeOption && (authorizer === null || authorizer === void 0 ? void 0 : authorizer.authorizationType) !== authorizationTypeOption) {
            throw new Error(`${this.resource}/${this.httpMethod} - Authorization type is set to ${authorizationTypeOption} ` +
                `which is different from what is required by the authorizer [${authorizer.authorizationType}]`);
        }
        if (authorizer_1.Authorizer.isAuthorizer(authorizer)) {
            authorizer._attachToApi(this.restApi);
        }
        const methodProps = {
            resourceId: props.resource.resourceId,
            restApiId: this.restApi.restApiId,
            httpMethod: this.httpMethod,
            operationName: options.operationName || defaultMethodOptions.operationName,
            apiKeyRequired: options.apiKeyRequired || defaultMethodOptions.apiKeyRequired,
            authorizationType,
            authorizerId,
            requestParameters: options.requestParameters || defaultMethodOptions.requestParameters,
            integration: this.renderIntegration(props.integration),
            methodResponses: this.renderMethodResponses(options.methodResponses),
            requestModels: this.renderRequestModels(options.requestModels),
            requestValidatorId: this.requestValidatorId(options),
            authorizationScopes: (_a = options.authorizationScopes) !== null && _a !== void 0 ? _a : defaultMethodOptions.authorizationScopes,
        };
        const resource = new apigateway_generated_1.CfnMethod(this, 'Resource', methodProps);
        this.methodId = resource.ref;
        props.resource.restApi._attachMethod(this);
        const deployment = props.resource.restApi.latestDeployment;
        if (deployment) {
            deployment.node.addDependency(resource);
            deployment.addToLogicalId({ method: methodProps });
        }
    }
    /**
     * Returns an execute-api ARN for this method:
     *
     *   arn:aws:execute-api:{region}:{account}:{restApiId}/{stage}/{method}/{path}
     *
     * NOTE: {stage} will refer to the `restApi.deploymentStage`, which will
     * automatically set if auto-deploy is enabled.
     *
     * @attribute
     */
    get methodArn() {
        if (!this.restApi.deploymentStage) {
            throw new Error(`Unable to determine ARN for method "${this.node.id}" since there is no stage associated with this API.\n` +
                'Either use the `deploy` prop or explicitly assign `deploymentStage` on the RestApi');
        }
        const stage = this.restApi.deploymentStage.stageName.toString();
        return this.restApi.arnForExecuteApi(this.httpMethod, this.resource.path, stage);
    }
    /**
     * Returns an execute-api ARN for this method's "test-invoke-stage" stage.
     * This stage is used by the AWS Console UI when testing the method.
     */
    get testMethodArn() {
        return this.restApi.arnForExecuteApi(this.httpMethod, this.resource.path, 'test-invoke-stage');
    }
    renderIntegration(integration) {
        if (!integration) {
            // use defaultIntegration from API if defined
            if (this.resource.defaultIntegration) {
                return this.renderIntegration(this.resource.defaultIntegration);
            }
            // fallback to mock
            return this.renderIntegration(new mock_1.MockIntegration());
        }
        integration.bind(this);
        const options = integration._props.options || {};
        let credentials;
        if (options.credentialsPassthrough !== undefined && options.credentialsRole !== undefined) {
            throw new Error('\'credentialsPassthrough\' and \'credentialsRole\' are mutually exclusive');
        }
        if (options.connectionType === integration_1.ConnectionType.VPC_LINK && options.vpcLink === undefined) {
            throw new Error('\'connectionType\' of VPC_LINK requires \'vpcLink\' prop to be set');
        }
        if (options.connectionType === integration_1.ConnectionType.INTERNET && options.vpcLink !== undefined) {
            throw new Error('cannot set \'vpcLink\' where \'connectionType\' is INTERNET');
        }
        if (options.credentialsRole) {
            credentials = options.credentialsRole.roleArn;
        }
        else if (options.credentialsPassthrough) {
            // arn:aws:iam::*:user/*
            // tslint:disable-next-line:max-line-length
            credentials = core_1.Stack.of(this).formatArn({ service: 'iam', region: '', account: '*', resource: 'user', sep: '/', resourceName: '*' });
        }
        return {
            type: integration._props.type,
            uri: integration._props.uri,
            cacheKeyParameters: options.cacheKeyParameters,
            cacheNamespace: options.cacheNamespace,
            contentHandling: options.contentHandling,
            integrationHttpMethod: integration._props.integrationHttpMethod,
            requestParameters: options.requestParameters,
            requestTemplates: options.requestTemplates,
            passthroughBehavior: options.passthroughBehavior,
            integrationResponses: options.integrationResponses,
            connectionType: options.connectionType,
            connectionId: options.vpcLink ? options.vpcLink.vpcLinkId : undefined,
            credentials,
        };
    }
    renderMethodResponses(methodResponses) {
        if (!methodResponses) {
            // Fall back to nothing
            return undefined;
        }
        return methodResponses.map(mr => {
            let responseModels;
            if (mr.responseModels) {
                responseModels = {};
                for (const contentType in mr.responseModels) {
                    if (mr.responseModels.hasOwnProperty(contentType)) {
                        responseModels[contentType] = mr.responseModels[contentType].modelId;
                    }
                }
            }
            const methodResponseProp = {
                statusCode: mr.statusCode,
                responseParameters: mr.responseParameters,
                responseModels,
            };
            return methodResponseProp;
        });
    }
    renderRequestModels(requestModels) {
        if (!requestModels) {
            // Fall back to nothing
            return undefined;
        }
        const models = {};
        for (const contentType in requestModels) {
            if (requestModels.hasOwnProperty(contentType)) {
                models[contentType] = requestModels[contentType].modelId;
            }
        }
        return models;
    }
    requestValidatorId(options) {
        var _a;
        if (options.requestValidator && options.requestValidatorOptions) {
            throw new Error('Only one of \'requestValidator\' or \'requestValidatorOptions\' must be specified.');
        }
        if (options.requestValidatorOptions) {
            const validator = this.restApi.addRequestValidator('validator', options.requestValidatorOptions);
            return validator.requestValidatorId;
        }
        // For backward compatibility
        return (_a = options.requestValidator) === null || _a === void 0 ? void 0 : _a.requestValidatorId;
    }
}
exports.Method = Method;
var AuthorizationType;
(function (AuthorizationType) {
    /**
     * Open access.
     */
    AuthorizationType["NONE"] = "NONE";
    /**
     * Use AWS IAM permissions.
     */
    AuthorizationType["IAM"] = "AWS_IAM";
    /**
     * Use a custom authorizer.
     */
    AuthorizationType["CUSTOM"] = "CUSTOM";
    /**
     * Use an AWS Cognito user pool.
     */
    AuthorizationType["COGNITO"] = "COGNITO_USER_POOLS";
})(AuthorizationType = exports.AuthorizationType || (exports.AuthorizationType = {}));
//# sourceMappingURL=data:application/json;base64,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