"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const aws_1 = require("./aws");
/**
 * Integrates an AWS Lambda function to an API Gateway method.
 *
 * @example
 *
 *    const handler = new lambda.Function(this, 'MyFunction', ...);
 *    api.addMethod('GET', new LambdaIntegration(handler));
 *
 */
class LambdaIntegration extends aws_1.AwsIntegration {
    constructor(handler, options = {}) {
        const proxy = options.proxy === undefined ? true : options.proxy;
        super({
            proxy,
            service: 'lambda',
            path: `2015-03-31/functions/${handler.functionArn}/invocations`,
            options,
        });
        this.handler = handler;
        this.enableTest = options.allowTestInvoke === undefined ? true : false;
    }
    bind(method) {
        super.bind(method);
        const principal = new iam.ServicePrincipal('apigateway.amazonaws.com');
        const desc = `${method.restApi.node.uniqueId}.${method.httpMethod}.${method.resource.path.replace(/\//g, '.')}`;
        this.handler.addPermission(`ApiPermission.${desc}`, {
            principal,
            scope: method,
            sourceArn: core_1.Lazy.stringValue({ produce: () => method.methodArn }),
        });
        // add permission to invoke from the console
        if (this.enableTest) {
            this.handler.addPermission(`ApiPermission.Test.${desc}`, {
                principal,
                scope: method,
                sourceArn: method.testMethodArn,
            });
        }
    }
}
exports.LambdaIntegration = LambdaIntegration;
//# sourceMappingURL=data:application/json;base64,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