"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const s3_assets = require("../../aws-s3-assets"); // Automatically re-written from '@aws-cdk/aws-s3-assets'
/**
 * Represents an OpenAPI definition asset.
 * @experimental
 */
class ApiDefinition {
    /**
     * Creates an API definition from a specification file in an S3 bucket
     * @experimental
     */
    static fromBucket(bucket, key, objectVersion) {
        return new S3ApiDefinition(bucket, key, objectVersion);
    }
    /**
     * Create an API definition from an inline object. The inline object must follow the
     * schema of OpenAPI 2.0 or OpenAPI 3.0
     *
     * @example
     *   ApiDefinition.fromInline({
     *     openapi: '3.0.2',
     *     paths: {
     *       '/pets': {
     *         get: {
     *           'responses': {
     *             200: {
     *               content: {
     *                 'application/json': {
     *                   schema: {
     *                     $ref: '#/components/schemas/Empty',
     *                   },
     *                 },
     *               },
     *             },
     *           },
     *           'x-amazon-apigateway-integration': {
     *             responses: {
     *               default: {
     *                 statusCode: '200',
     *               },
     *             },
     *             requestTemplates: {
     *               'application/json': '{"statusCode": 200}',
     *             },
     *             passthroughBehavior: 'when_no_match',
     *             type: 'mock',
     *           },
     *         },
     *       },
     *     },
     *     components: {
     *       schemas: {
     *         Empty: {
     *           title: 'Empty Schema',
     *           type: 'object',
     *         },
     *       },
     *     },
     *   });
     */
    static fromInline(definition) {
        return new InlineApiDefinition(definition);
    }
    /**
     * Loads the API specification from a local disk asset.
     * @experimental
     */
    static fromAsset(file, options) {
        return new AssetApiDefinition(file, options);
    }
}
exports.ApiDefinition = ApiDefinition;
/**
 * OpenAPI specification from an S3 archive.
 * @experimental
 */
class S3ApiDefinition extends ApiDefinition {
    constructor(bucket, key, objectVersion) {
        super();
        this.key = key;
        this.objectVersion = objectVersion;
        if (!bucket.bucketName) {
            throw new Error('bucketName is undefined for the provided bucket');
        }
        this.bucketName = bucket.bucketName;
    }
    bind(_scope) {
        return {
            s3Location: {
                bucket: this.bucketName,
                key: this.key,
                version: this.objectVersion,
            },
        };
    }
}
exports.S3ApiDefinition = S3ApiDefinition;
/**
 * OpenAPI specification from an inline JSON object.
 * @experimental
 */
class InlineApiDefinition extends ApiDefinition {
    constructor(definition) {
        super();
        this.definition = definition;
        if (typeof (definition) !== 'object') {
            throw new Error('definition should be of type object');
        }
        if (Object.keys(definition).length === 0) {
            throw new Error('JSON definition cannot be empty');
        }
    }
    bind(_scope) {
        return {
            inlineDefinition: this.definition,
        };
    }
}
exports.InlineApiDefinition = InlineApiDefinition;
/**
 * OpenAPI specification from a local file.
 * @experimental
 */
class AssetApiDefinition extends ApiDefinition {
    constructor(path, options = {}) {
        super();
        this.path = path;
        this.options = options;
    }
    bind(scope) {
        // If the same AssetAPIDefinition is used multiple times, retain only the first instantiation.
        if (this.asset === undefined) {
            this.asset = new s3_assets.Asset(scope, 'APIDefinition', {
                path: this.path,
                ...this.options,
            });
        }
        if (this.asset.isZipArchive) {
            throw new Error(`Asset cannot be a .zip file or a directory (${this.path})`);
        }
        return {
            s3Location: {
                bucket: this.asset.s3BucketName,
                key: this.asset.s3ObjectKey,
            },
        };
    }
}
exports.AssetApiDefinition = AssetApiDefinition;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYXBpLWRlZmluaXRpb24uanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJhcGktZGVmaW5pdGlvbi50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOztBQUNBLGlEQUFpRCxDQUFDLHlEQUF5RDtBQUUzRzs7O0dBR0c7QUFDSCxNQUFzQixhQUFhO0lBQy9COzs7T0FHRztJQUNJLE1BQU0sQ0FBQyxVQUFVLENBQUMsTUFBa0IsRUFBRSxHQUFXLEVBQUUsYUFBc0I7UUFDNUUsT0FBTyxJQUFJLGVBQWUsQ0FBQyxNQUFNLEVBQUUsR0FBRyxFQUFFLGFBQWEsQ0FBQyxDQUFDO0lBQzNELENBQUM7SUFDRDs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7O09BNkNHO0lBQ0ksTUFBTSxDQUFDLFVBQVUsQ0FBQyxVQUFlO1FBQ3BDLE9BQU8sSUFBSSxtQkFBbUIsQ0FBQyxVQUFVLENBQUMsQ0FBQztJQUMvQyxDQUFDO0lBQ0Q7OztPQUdHO0lBQ0ksTUFBTSxDQUFDLFNBQVMsQ0FBQyxJQUFZLEVBQUUsT0FBZ0M7UUFDbEUsT0FBTyxJQUFJLGtCQUFrQixDQUFDLElBQUksRUFBRSxPQUFPLENBQUMsQ0FBQztJQUNqRCxDQUFDO0NBU0o7QUF4RUQsc0NBd0VDO0FBa0NEOzs7R0FHRztBQUNILE1BQWEsZUFBZ0IsU0FBUSxhQUFhO0lBRTlDLFlBQVksTUFBa0IsRUFBVSxHQUFXLEVBQVUsYUFBc0I7UUFDL0UsS0FBSyxFQUFFLENBQUM7UUFENEIsUUFBRyxHQUFILEdBQUcsQ0FBUTtRQUFVLGtCQUFhLEdBQWIsYUFBYSxDQUFTO1FBRS9FLElBQUksQ0FBQyxNQUFNLENBQUMsVUFBVSxFQUFFO1lBQ3BCLE1BQU0sSUFBSSxLQUFLLENBQUMsaURBQWlELENBQUMsQ0FBQztTQUN0RTtRQUNELElBQUksQ0FBQyxVQUFVLEdBQUcsTUFBTSxDQUFDLFVBQVUsQ0FBQztJQUN4QyxDQUFDO0lBQ00sSUFBSSxDQUFDLE1BQXFCO1FBQzdCLE9BQU87WUFDSCxVQUFVLEVBQUU7Z0JBQ1IsTUFBTSxFQUFFLElBQUksQ0FBQyxVQUFVO2dCQUN2QixHQUFHLEVBQUUsSUFBSSxDQUFDLEdBQUc7Z0JBQ2IsT0FBTyxFQUFFLElBQUksQ0FBQyxhQUFhO2FBQzlCO1NBQ0osQ0FBQztJQUNOLENBQUM7Q0FDSjtBQWxCRCwwQ0FrQkM7QUFDRDs7O0dBR0c7QUFDSCxNQUFhLG1CQUFvQixTQUFRLGFBQWE7SUFDbEQsWUFBb0IsVUFBZTtRQUMvQixLQUFLLEVBQUUsQ0FBQztRQURRLGVBQVUsR0FBVixVQUFVLENBQUs7UUFFL0IsSUFBSSxPQUFPLENBQUMsVUFBVSxDQUFDLEtBQUssUUFBUSxFQUFFO1lBQ2xDLE1BQU0sSUFBSSxLQUFLLENBQUMscUNBQXFDLENBQUMsQ0FBQztTQUMxRDtRQUNELElBQUksTUFBTSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsQ0FBQyxNQUFNLEtBQUssQ0FBQyxFQUFFO1lBQ3RDLE1BQU0sSUFBSSxLQUFLLENBQUMsaUNBQWlDLENBQUMsQ0FBQztTQUN0RDtJQUNMLENBQUM7SUFDTSxJQUFJLENBQUMsTUFBcUI7UUFDN0IsT0FBTztZQUNILGdCQUFnQixFQUFFLElBQUksQ0FBQyxVQUFVO1NBQ3BDLENBQUM7SUFDTixDQUFDO0NBQ0o7QUFmRCxrREFlQztBQUNEOzs7R0FHRztBQUNILE1BQWEsa0JBQW1CLFNBQVEsYUFBYTtJQUVqRCxZQUE2QixJQUFZLEVBQW1CLFVBQWtDLEVBQUU7UUFDNUYsS0FBSyxFQUFFLENBQUM7UUFEaUIsU0FBSSxHQUFKLElBQUksQ0FBUTtRQUFtQixZQUFPLEdBQVAsT0FBTyxDQUE2QjtJQUVoRyxDQUFDO0lBQ00sSUFBSSxDQUFDLEtBQW9CO1FBQzVCLDhGQUE4RjtRQUM5RixJQUFJLElBQUksQ0FBQyxLQUFLLEtBQUssU0FBUyxFQUFFO1lBQzFCLElBQUksQ0FBQyxLQUFLLEdBQUcsSUFBSSxTQUFTLENBQUMsS0FBSyxDQUFDLEtBQUssRUFBRSxlQUFlLEVBQUU7Z0JBQ3JELElBQUksRUFBRSxJQUFJLENBQUMsSUFBSTtnQkFDZixHQUFHLElBQUksQ0FBQyxPQUFPO2FBQ2xCLENBQUMsQ0FBQztTQUNOO1FBQ0QsSUFBSSxJQUFJLENBQUMsS0FBSyxDQUFDLFlBQVksRUFBRTtZQUN6QixNQUFNLElBQUksS0FBSyxDQUFDLCtDQUErQyxJQUFJLENBQUMsSUFBSSxHQUFHLENBQUMsQ0FBQztTQUNoRjtRQUNELE9BQU87WUFDSCxVQUFVLEVBQUU7Z0JBQ1IsTUFBTSxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsWUFBWTtnQkFDL0IsR0FBRyxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsV0FBVzthQUM5QjtTQUNKLENBQUM7SUFDTixDQUFDO0NBQ0o7QUF2QkQsZ0RBdUJDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0ICogYXMgczMgZnJvbSBcIi4uLy4uL2F3cy1zM1wiOyAvLyBBdXRvbWF0aWNhbGx5IHJlLXdyaXR0ZW4gZnJvbSAnQGF3cy1jZGsvYXdzLXMzJ1xuaW1wb3J0ICogYXMgczNfYXNzZXRzIGZyb20gXCIuLi8uLi9hd3MtczMtYXNzZXRzXCI7IC8vIEF1dG9tYXRpY2FsbHkgcmUtd3JpdHRlbiBmcm9tICdAYXdzLWNkay9hd3MtczMtYXNzZXRzJ1xuaW1wb3J0ICogYXMgY2RrIGZyb20gXCIuLi8uLi9jb3JlXCI7IC8vIEF1dG9tYXRpY2FsbHkgcmUtd3JpdHRlbiBmcm9tICdAYXdzLWNkay9jb3JlJ1xuLyoqXG4gKiBSZXByZXNlbnRzIGFuIE9wZW5BUEkgZGVmaW5pdGlvbiBhc3NldC5cbiAqIEBleHBlcmltZW50YWxcbiAqL1xuZXhwb3J0IGFic3RyYWN0IGNsYXNzIEFwaURlZmluaXRpb24ge1xuICAgIC8qKlxuICAgICAqIENyZWF0ZXMgYW4gQVBJIGRlZmluaXRpb24gZnJvbSBhIHNwZWNpZmljYXRpb24gZmlsZSBpbiBhbiBTMyBidWNrZXRcbiAgICAgKiBAZXhwZXJpbWVudGFsXG4gICAgICovXG4gICAgcHVibGljIHN0YXRpYyBmcm9tQnVja2V0KGJ1Y2tldDogczMuSUJ1Y2tldCwga2V5OiBzdHJpbmcsIG9iamVjdFZlcnNpb24/OiBzdHJpbmcpOiBTM0FwaURlZmluaXRpb24ge1xuICAgICAgICByZXR1cm4gbmV3IFMzQXBpRGVmaW5pdGlvbihidWNrZXQsIGtleSwgb2JqZWN0VmVyc2lvbik7XG4gICAgfVxuICAgIC8qKlxuICAgICAqIENyZWF0ZSBhbiBBUEkgZGVmaW5pdGlvbiBmcm9tIGFuIGlubGluZSBvYmplY3QuIFRoZSBpbmxpbmUgb2JqZWN0IG11c3QgZm9sbG93IHRoZVxuICAgICAqIHNjaGVtYSBvZiBPcGVuQVBJIDIuMCBvciBPcGVuQVBJIDMuMFxuICAgICAqXG4gICAgICogQGV4YW1wbGVcbiAgICAgKiAgIEFwaURlZmluaXRpb24uZnJvbUlubGluZSh7XG4gICAgICogICAgIG9wZW5hcGk6ICczLjAuMicsXG4gICAgICogICAgIHBhdGhzOiB7XG4gICAgICogICAgICAgJy9wZXRzJzoge1xuICAgICAqICAgICAgICAgZ2V0OiB7XG4gICAgICogICAgICAgICAgICdyZXNwb25zZXMnOiB7XG4gICAgICogICAgICAgICAgICAgMjAwOiB7XG4gICAgICogICAgICAgICAgICAgICBjb250ZW50OiB7XG4gICAgICogICAgICAgICAgICAgICAgICdhcHBsaWNhdGlvbi9qc29uJzoge1xuICAgICAqICAgICAgICAgICAgICAgICAgIHNjaGVtYToge1xuICAgICAqICAgICAgICAgICAgICAgICAgICAgJHJlZjogJyMvY29tcG9uZW50cy9zY2hlbWFzL0VtcHR5JyxcbiAgICAgKiAgICAgICAgICAgICAgICAgICB9LFxuICAgICAqICAgICAgICAgICAgICAgICB9LFxuICAgICAqICAgICAgICAgICAgICAgfSxcbiAgICAgKiAgICAgICAgICAgICB9LFxuICAgICAqICAgICAgICAgICB9LFxuICAgICAqICAgICAgICAgICAneC1hbWF6b24tYXBpZ2F0ZXdheS1pbnRlZ3JhdGlvbic6IHtcbiAgICAgKiAgICAgICAgICAgICByZXNwb25zZXM6IHtcbiAgICAgKiAgICAgICAgICAgICAgIGRlZmF1bHQ6IHtcbiAgICAgKiAgICAgICAgICAgICAgICAgc3RhdHVzQ29kZTogJzIwMCcsXG4gICAgICogICAgICAgICAgICAgICB9LFxuICAgICAqICAgICAgICAgICAgIH0sXG4gICAgICogICAgICAgICAgICAgcmVxdWVzdFRlbXBsYXRlczoge1xuICAgICAqICAgICAgICAgICAgICAgJ2FwcGxpY2F0aW9uL2pzb24nOiAne1wic3RhdHVzQ29kZVwiOiAyMDB9JyxcbiAgICAgKiAgICAgICAgICAgICB9LFxuICAgICAqICAgICAgICAgICAgIHBhc3N0aHJvdWdoQmVoYXZpb3I6ICd3aGVuX25vX21hdGNoJyxcbiAgICAgKiAgICAgICAgICAgICB0eXBlOiAnbW9jaycsXG4gICAgICogICAgICAgICAgIH0sXG4gICAgICogICAgICAgICB9LFxuICAgICAqICAgICAgIH0sXG4gICAgICogICAgIH0sXG4gICAgICogICAgIGNvbXBvbmVudHM6IHtcbiAgICAgKiAgICAgICBzY2hlbWFzOiB7XG4gICAgICogICAgICAgICBFbXB0eToge1xuICAgICAqICAgICAgICAgICB0aXRsZTogJ0VtcHR5IFNjaGVtYScsXG4gICAgICogICAgICAgICAgIHR5cGU6ICdvYmplY3QnLFxuICAgICAqICAgICAgICAgfSxcbiAgICAgKiAgICAgICB9LFxuICAgICAqICAgICB9LFxuICAgICAqICAgfSk7XG4gICAgICovXG4gICAgcHVibGljIHN0YXRpYyBmcm9tSW5saW5lKGRlZmluaXRpb246IGFueSk6IElubGluZUFwaURlZmluaXRpb24ge1xuICAgICAgICByZXR1cm4gbmV3IElubGluZUFwaURlZmluaXRpb24oZGVmaW5pdGlvbik7XG4gICAgfVxuICAgIC8qKlxuICAgICAqIExvYWRzIHRoZSBBUEkgc3BlY2lmaWNhdGlvbiBmcm9tIGEgbG9jYWwgZGlzayBhc3NldC5cbiAgICAgKiBAZXhwZXJpbWVudGFsXG4gICAgICovXG4gICAgcHVibGljIHN0YXRpYyBmcm9tQXNzZXQoZmlsZTogc3RyaW5nLCBvcHRpb25zPzogczNfYXNzZXRzLkFzc2V0T3B0aW9ucyk6IEFzc2V0QXBpRGVmaW5pdGlvbiB7XG4gICAgICAgIHJldHVybiBuZXcgQXNzZXRBcGlEZWZpbml0aW9uKGZpbGUsIG9wdGlvbnMpO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBDYWxsZWQgd2hlbiB0aGUgc3BlY2lmaWNhdGlvbiBpcyBpbml0aWFsaXplZCB0byBhbGxvdyB0aGlzIG9iamVjdCB0byBiaW5kXG4gICAgICogdG8gdGhlIHN0YWNrLCBhZGQgcmVzb3VyY2VzIGFuZCBoYXZlIGZ1bi5cbiAgICAgKlxuICAgICAqIEBwYXJhbSBzY29wZSBUaGUgYmluZGluZyBzY29wZS4gRG9uJ3QgYmUgc21hcnQgYWJvdXQgdHJ5aW5nIHRvIGRvd24tY2FzdCBvclxuICAgICAqIGFzc3VtZSBpdCdzIGluaXRpYWxpemVkLiBZb3UgbWF5IGp1c3QgdXNlIGl0IGFzIGEgY29uc3RydWN0IHNjb3BlLlxuICAgICAqL1xuICAgIHB1YmxpYyBhYnN0cmFjdCBiaW5kKHNjb3BlOiBjZGsuQ29uc3RydWN0KTogQXBpRGVmaW5pdGlvbkNvbmZpZztcbn1cbi8qKlxuICogUzMgbG9jYXRpb24gb2YgdGhlIEFQSSBkZWZpbml0aW9uIGZpbGVcbiAqIEBleHBlcmltZW50YWxcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBBcGlEZWZpbml0aW9uUzNMb2NhdGlvbiB7XG4gICAgLyoqIFRoZSBTMyBidWNrZXQgKi9cbiAgICByZWFkb25seSBidWNrZXQ6IHN0cmluZztcbiAgICAvKiogVGhlIFMzIGtleSAqL1xuICAgIHJlYWRvbmx5IGtleTogc3RyaW5nO1xuICAgIC8qKlxuICAgICAqIEFuIG9wdGlvbmFsIHZlcnNpb25cbiAgICAgKiBAZGVmYXVsdCAtIGxhdGVzdCB2ZXJzaW9uXG4gICAgICovXG4gICAgcmVhZG9ubHkgdmVyc2lvbj86IHN0cmluZztcbn1cbi8qKlxuICogUG9zdC1CaW5kaW5nIENvbmZpZ3VyYXRpb24gZm9yIGEgQ0RLIGNvbnN0cnVjdFxuICogQGV4cGVyaW1lbnRhbFxuICovXG5leHBvcnQgaW50ZXJmYWNlIEFwaURlZmluaXRpb25Db25maWcge1xuICAgIC8qKlxuICAgICAqIFRoZSBsb2NhdGlvbiBvZiB0aGUgc3BlY2lmaWNhdGlvbiBpbiBTMyAobXV0dWFsbHkgZXhjbHVzaXZlIHdpdGggYGlubGluZURlZmluaXRpb25gKS5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gQVBJIGRlZmluaXRpb24gaXMgbm90IGFuIFMzIGxvY2F0aW9uXG4gICAgICovXG4gICAgcmVhZG9ubHkgczNMb2NhdGlvbj86IEFwaURlZmluaXRpb25TM0xvY2F0aW9uO1xuICAgIC8qKlxuICAgICAqIElubGluZSBzcGVjaWZpY2F0aW9uIChtdXR1YWxseSBleGNsdXNpdmUgd2l0aCBgczNMb2NhdGlvbmApLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSBBUEkgZGVmaW5pdGlvbiBpcyBub3QgZGVmaW5lZCBpbmxpbmVcbiAgICAgKi9cbiAgICByZWFkb25seSBpbmxpbmVEZWZpbml0aW9uPzogYW55O1xufVxuLyoqXG4gKiBPcGVuQVBJIHNwZWNpZmljYXRpb24gZnJvbSBhbiBTMyBhcmNoaXZlLlxuICogQGV4cGVyaW1lbnRhbFxuICovXG5leHBvcnQgY2xhc3MgUzNBcGlEZWZpbml0aW9uIGV4dGVuZHMgQXBpRGVmaW5pdGlvbiB7XG4gICAgcHJpdmF0ZSBidWNrZXROYW1lOiBzdHJpbmc7XG4gICAgY29uc3RydWN0b3IoYnVja2V0OiBzMy5JQnVja2V0LCBwcml2YXRlIGtleTogc3RyaW5nLCBwcml2YXRlIG9iamVjdFZlcnNpb24/OiBzdHJpbmcpIHtcbiAgICAgICAgc3VwZXIoKTtcbiAgICAgICAgaWYgKCFidWNrZXQuYnVja2V0TmFtZSkge1xuICAgICAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdidWNrZXROYW1lIGlzIHVuZGVmaW5lZCBmb3IgdGhlIHByb3ZpZGVkIGJ1Y2tldCcpO1xuICAgICAgICB9XG4gICAgICAgIHRoaXMuYnVja2V0TmFtZSA9IGJ1Y2tldC5idWNrZXROYW1lO1xuICAgIH1cbiAgICBwdWJsaWMgYmluZChfc2NvcGU6IGNkay5Db25zdHJ1Y3QpOiBBcGlEZWZpbml0aW9uQ29uZmlnIHtcbiAgICAgICAgcmV0dXJuIHtcbiAgICAgICAgICAgIHMzTG9jYXRpb246IHtcbiAgICAgICAgICAgICAgICBidWNrZXQ6IHRoaXMuYnVja2V0TmFtZSxcbiAgICAgICAgICAgICAgICBrZXk6IHRoaXMua2V5LFxuICAgICAgICAgICAgICAgIHZlcnNpb246IHRoaXMub2JqZWN0VmVyc2lvbixcbiAgICAgICAgICAgIH0sXG4gICAgICAgIH07XG4gICAgfVxufVxuLyoqXG4gKiBPcGVuQVBJIHNwZWNpZmljYXRpb24gZnJvbSBhbiBpbmxpbmUgSlNPTiBvYmplY3QuXG4gKiBAZXhwZXJpbWVudGFsXG4gKi9cbmV4cG9ydCBjbGFzcyBJbmxpbmVBcGlEZWZpbml0aW9uIGV4dGVuZHMgQXBpRGVmaW5pdGlvbiB7XG4gICAgY29uc3RydWN0b3IocHJpdmF0ZSBkZWZpbml0aW9uOiBhbnkpIHtcbiAgICAgICAgc3VwZXIoKTtcbiAgICAgICAgaWYgKHR5cGVvZiAoZGVmaW5pdGlvbikgIT09ICdvYmplY3QnKSB7XG4gICAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ2RlZmluaXRpb24gc2hvdWxkIGJlIG9mIHR5cGUgb2JqZWN0Jyk7XG4gICAgICAgIH1cbiAgICAgICAgaWYgKE9iamVjdC5rZXlzKGRlZmluaXRpb24pLmxlbmd0aCA9PT0gMCkge1xuICAgICAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdKU09OIGRlZmluaXRpb24gY2Fubm90IGJlIGVtcHR5Jyk7XG4gICAgICAgIH1cbiAgICB9XG4gICAgcHVibGljIGJpbmQoX3Njb3BlOiBjZGsuQ29uc3RydWN0KTogQXBpRGVmaW5pdGlvbkNvbmZpZyB7XG4gICAgICAgIHJldHVybiB7XG4gICAgICAgICAgICBpbmxpbmVEZWZpbml0aW9uOiB0aGlzLmRlZmluaXRpb24sXG4gICAgICAgIH07XG4gICAgfVxufVxuLyoqXG4gKiBPcGVuQVBJIHNwZWNpZmljYXRpb24gZnJvbSBhIGxvY2FsIGZpbGUuXG4gKiBAZXhwZXJpbWVudGFsXG4gKi9cbmV4cG9ydCBjbGFzcyBBc3NldEFwaURlZmluaXRpb24gZXh0ZW5kcyBBcGlEZWZpbml0aW9uIHtcbiAgICBwcml2YXRlIGFzc2V0PzogczNfYXNzZXRzLkFzc2V0O1xuICAgIGNvbnN0cnVjdG9yKHByaXZhdGUgcmVhZG9ubHkgcGF0aDogc3RyaW5nLCBwcml2YXRlIHJlYWRvbmx5IG9wdGlvbnM6IHMzX2Fzc2V0cy5Bc3NldE9wdGlvbnMgPSB7fSkge1xuICAgICAgICBzdXBlcigpO1xuICAgIH1cbiAgICBwdWJsaWMgYmluZChzY29wZTogY2RrLkNvbnN0cnVjdCk6IEFwaURlZmluaXRpb25Db25maWcge1xuICAgICAgICAvLyBJZiB0aGUgc2FtZSBBc3NldEFQSURlZmluaXRpb24gaXMgdXNlZCBtdWx0aXBsZSB0aW1lcywgcmV0YWluIG9ubHkgdGhlIGZpcnN0IGluc3RhbnRpYXRpb24uXG4gICAgICAgIGlmICh0aGlzLmFzc2V0ID09PSB1bmRlZmluZWQpIHtcbiAgICAgICAgICAgIHRoaXMuYXNzZXQgPSBuZXcgczNfYXNzZXRzLkFzc2V0KHNjb3BlLCAnQVBJRGVmaW5pdGlvbicsIHtcbiAgICAgICAgICAgICAgICBwYXRoOiB0aGlzLnBhdGgsXG4gICAgICAgICAgICAgICAgLi4udGhpcy5vcHRpb25zLFxuICAgICAgICAgICAgfSk7XG4gICAgICAgIH1cbiAgICAgICAgaWYgKHRoaXMuYXNzZXQuaXNaaXBBcmNoaXZlKSB7XG4gICAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYEFzc2V0IGNhbm5vdCBiZSBhIC56aXAgZmlsZSBvciBhIGRpcmVjdG9yeSAoJHt0aGlzLnBhdGh9KWApO1xuICAgICAgICB9XG4gICAgICAgIHJldHVybiB7XG4gICAgICAgICAgICBzM0xvY2F0aW9uOiB7XG4gICAgICAgICAgICAgICAgYnVja2V0OiB0aGlzLmFzc2V0LnMzQnVja2V0TmFtZSxcbiAgICAgICAgICAgICAgICBrZXk6IHRoaXMuYXNzZXQuczNPYmplY3RLZXksXG4gICAgICAgICAgICB9LFxuICAgICAgICB9O1xuICAgIH1cbn1cbiJdfQ==