"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * Use CloudWatch Logs as a custom access log destination for API Gateway.
 */
class LogGroupLogDestination {
    constructor(logGroup) {
        this.logGroup = logGroup;
    }
    /**
     * Binds this destination to the CloudWatch Logs.
     */
    bind(_stage) {
        return {
            destinationArn: this.logGroup.logGroupArn,
        };
    }
}
exports.LogGroupLogDestination = LogGroupLogDestination;
/**
 * $context variables that can be used to customize access log pattern.
 */
class AccessLogField {
    /**
     * The API owner's AWS account ID.
     */
    static contextAccountId() {
        return '$context.identity.accountId';
    }
    /**
     * The identifier API Gateway assigns to your API.
     */
    static contextApiId() {
        return '$context.apiId';
    }
    /**
     * A property of the claims returned from the Amazon Cognito user pool after the method caller is successfully authenticated.
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-integrate-with-cognito.html
     *
     * @param property A property key of the claims.
     */
    static contextAuthorizerClaims(property) {
        return `$context.authorizer.claims.${property}`;
    }
    /**
     * The principal user identification associated with the token sent by the client and returned
     * from an API Gateway Lambda authorizer (formerly known as a custom authorizer).
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-use-lambda-authorizer.html
     */
    static contextAuthorizerPrincipalId() {
        return '$context.authorizer.principalId';
    }
    /**
     * The stringified value of the specified key-value pair of the `context` map returned from an API Gateway Lambda authorizer function.
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-use-lambda-authorizer.html
     * @param property key of the context map.
     */
    static contextAuthorizer(property) {
        return `$context.authorizer.${property}`;
    }
    /**
     * The AWS endpoint's request ID.
     */
    static contextAwsEndpointRequestId() {
        return '$context.awsEndpointRequestId';
    }
    /**
     * The full domain name used to invoke the API. This should be the same as the incoming `Host` header.
     */
    static contextDomainName() {
        return '$context.domainName';
    }
    /**
     * The first label of the `$context.domainName`. This is often used as a caller/customer identifier.
     */
    static contextDomainPrefix() {
        return '$context.domainPrefix';
    }
    /**
     * A string containing an API Gateway error message.
     */
    static contextErrorMessage() {
        return '$context.error.message';
    }
    /**
     * The quoted value of $context.error.message, namely "$context.error.message".
     */
    static contextErrorMessageString() {
        return '$context.error.messageString';
    }
    /**
     * A type of GatewayResponse. This variable can only be used for simple variable substitution in a GatewayResponse body-mapping template,
     * which is not processed by the Velocity Template Language engine, and in access logging.
     *
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-websocket-api-logging.html
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/customize-gateway-responses.html
     */
    static contextErrorResponseType() {
        return '$context.error.responseType';
    }
    /**
     * A string containing a detailed validation error message.
     */
    static contextErrorValidationErrorString() {
        return '$context.error.validationErrorString';
    }
    /**
     * The extended ID that API Gateway assigns to the API request, which contains more useful information for debugging/troubleshooting.
     */
    static contextExtendedRequestId() {
        return '$context.extendedRequestId';
    }
    /**
     * The HTTP method used. Valid values include: `DELETE`, `GET`, `HEAD`, `OPTIONS`, `PATCH`, `POST`, and `PUT`.
     */
    static contextHttpMethod() {
        return '$context.httpMethod';
    }
    /**
     * The AWS account ID associated with the request.
     */
    static contextIdentityAccountId() {
        return '$context.identity.accountId';
    }
    /**
     * For API methods that require an API key, this variable is the API key associated with the method request.
     * For methods that don't require an API key, this variable is
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-api-usage-plans.html
     */
    static contextIdentityApiKey() {
        return '$context.identity.apiKey';
    }
    /**
     * The API key ID associated with an API request that requires an API key.
     */
    static contextIdentityApiKeyId() {
        return '$context.identity.apiKeyId';
    }
    /**
     * The principal identifier of the caller making the request.
     */
    static contextIdentityCaller() {
        return '$context.identity.caller';
    }
    /**
     * The Amazon Cognito authentication provider used by the caller making the request.
     * Available only if the request was signed with Amazon Cognito credentials.
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-identity.html
     */
    static contextIdentityCognitoAuthenticationProvider() {
        return '$context.identity.cognitoAuthenticationProvider';
    }
    /**
     * The Amazon Cognito authentication type of the caller making the request.
     * Available only if the request was signed with Amazon Cognito credentials.
     */
    static contextIdentityCognitoAuthenticationType() {
        return '$context.identity.cognitoAuthenticationType';
    }
    /**
     * The Amazon Cognito identity ID of the caller making the request. Available only if the request was signed with Amazon Cognito credentials.
     */
    static contextIdentityCognitoIdentityId() {
        return '$context.identity.cognitoIdentityId';
    }
    /**
     * The Amazon Cognito identity pool ID of the caller making the request.
     * Available only if the request was signed with Amazon Cognito credentials.
     */
    static contextIdentityCognitoIdentityPoolId() {
        return '$context.identity.cognitoIdentityPoolId';
    }
    /**
     * The AWS organization ID.
     */
    static contextIdentityPrincipalOrgId() {
        return '$context.identity.principalOrgId';
    }
    /**
     * The source IP address of the TCP connection making the request to API Gateway.
     * Warning: You should not trust this value if there is any chance that the `X-Forwarded-For` header could be forged.
     */
    static contextIdentitySourceIp() {
        return '$context.identity.sourceIp';
    }
    /**
     * The principal identifier of the user making the request. Used in Lambda authorizers.
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-lambda-authorizer-output.html
     */
    static contextIdentityUser() {
        return '$context.identity.user';
    }
    /**
     * The User-Agent header of the API caller.
     */
    static contextIdentityUserAgent() {
        return '$context.identity.userAgent';
    }
    /**
     * The Amazon Resource Name (ARN) of the effective user identified after authentication.
     * @see https://docs.aws.amazon.com/IAM/latest/UserGuide/id_users.html
     */
    static contextIdentityUserArn() {
        return '$context.identity.userArn';
    }
    /**
     * The request path.
     * For example, for a non-proxy request URL of https://{rest-api-id.execute-api.{region}.amazonaws.com/{stage}/root/child,
     * this value is /{stage}/root/child.
     */
    static contextPath() {
        return '$context.path';
    }
    /**
     * The request protocol, for example, HTTP/1.1.
     */
    static contextProtocol() {
        return '$context.protocol';
    }
    /**
     * The ID that API Gateway assigns to the API request.
     */
    static contextRequestId() {
        return '$context.requestId';
    }
    /**
     * The request header override.
     * If this parameter is defined, it contains the headers to be used instead of the HTTP Headers that are defined in the Integration Request pane.
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-override-request-response-parameters.html
     *
     * @param headerName
     */
    static contextRequestOverrideHeader(headerName) {
        return `$context.requestOverride.header.${headerName}`;
    }
    /**
     * The request path override. If this parameter is defined,
     * it contains the request path to be used instead of the URL Path Parameters that are defined in the Integration Request pane.
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-override-request-response-parameters.html
     *
     * @param pathName
     */
    static contextRequestOverridePath(pathName) {
        return `$context.requestOverride.path.${pathName}`;
    }
    /**
     * The request query string override.
     * If this parameter is defined, it contains the request query strings to be used instead
     * of the URL Query String Parameters that are defined in the Integration Request pane.
     *
     * @param querystringName
     */
    static contextRequestOverrideQuerystring(querystringName) {
        return `$context.requestOverride.querystring.${querystringName}`;
    }
    /**
     * The response header override.
     * If this parameter is defined, it contains the header to be returned instead of the Response header
     * that is defined as the Default mapping in the Integration Response pane.
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-override-request-response-parameters.html
     *
     * @param headerName
     */
    static contextResponseOverrideHeader(headerName) {
        return `$context.responseOverride.header.${headerName}`;
    }
    /**
     * The response status code override.
     * If this parameter is defined, it contains the status code to be returned instead of the Method response status
     * that is defined as the Default mapping in the Integration Response pane.
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-override-request-response-parameters.html
     */
    static contextResponseOverrideStatus() {
        return '$context.responseOverride.status';
    }
    /**
     * The CLF-formatted request time (dd/MMM/yyyy:HH:mm:ss +-hhmm).
     */
    static contextRequestTime() {
        return '$context.requestTime';
    }
    /**
     * The Epoch-formatted request time.
     */
    static contextRequestTimeEpoch() {
        return '$context.requestTimeEpoch';
    }
    /**
     * The identifier that API Gateway assigns to your resource.
     */
    static contextResourceId() {
        return '$context.resourceId';
    }
    /**
     * The path to your resource.
     * For example, for the non-proxy request URI of `https://{rest-api-id.execute-api.{region}.amazonaws.com/{stage}/root/child`,
     * The $context.resourcePath value is `/root/child`.
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-create-api-step-by-step.html
     */
    static contextResourcePath() {
        return '$context.resourcePath';
    }
    /**
     * The deployment stage of the API request (for example, `Beta` or `Prod`).
     */
    static contextStage() {
        return '$context.stage';
    }
    /**
     * The response received from AWS WAF: `WAF_ALLOW` or `WAF_BLOCK`. Will not be set if the stage is not associated with a web ACL.
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-control-access-aws-waf.html
     */
    static contextWafResponseCode() {
        return '$context.wafResponseCode';
    }
    /**
     * The complete ARN of the web ACL that is used to decide whether to allow or block the request.
     * Will not be set if the stage is not associated with a web ACL.
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-control-access-aws-waf.html
     */
    static contextWebaclArn() {
        return '$context.webaclArn';
    }
    /**
     * The trace ID for the X-Ray trace.
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-enabling-xray.html
     */
    static contextXrayTraceId() {
        return '$context.xrayTraceId';
    }
    /**
     * The authorizer latency in ms.
     */
    static contextAuthorizerIntegrationLatency() {
        return '$context.authorizer.integrationLatency';
    }
    /**
     * The integration latency in ms.
     */
    static contextIntegrationLatency() {
        return '$context.integrationLatency';
    }
    /**
     * For Lambda proxy integration, this parameter represents the status code returned from AWS Lambda,
     * not from the backend Lambda function.
     */
    static contextIntegrationStatus() {
        return '$context.integrationStatus.';
    }
    /**
     * The response latency in ms.
     */
    static contextResponseLatency() {
        return '$context.responseLatency';
    }
    /**
     * The response payload length.
     */
    static contextResponseLength() {
        return '$context.responseLength';
    }
    /**
     * The method response status.
     */
    static contextStatus() {
        return '$context.status';
    }
}
exports.AccessLogField = AccessLogField;
/**
 * factory methods for access log format.
 */
class AccessLogFormat {
    constructor(format) {
        this.format = format;
    }
    /**
     * Custom log format.
     * You can create any log format string. You can easily get the $ context variable by using the methods of AccessLogField.
     * @param format
     * @example custom(JSON.stringify({
     *  requestId: AccessLogField.contextRequestId(),
     *  sourceIp: AccessLogField.contextIdentitySourceIp(),
     *  method: AccessLogFiled.contextHttpMethod(),
     *  userContext: {
     *    sub: AccessLogField.contextAuthorizerClaims('sub'),
     *    email: AccessLogField.contextAuthorizerClaims('email')
     *  }
     * }))
     */
    static custom(format) {
        return new AccessLogFormat(format);
    }
    /**
     * Generate Common Log Format.
     */
    static clf() {
        const requester = [AccessLogField.contextIdentitySourceIp(), AccessLogField.contextIdentityCaller(), AccessLogField.contextIdentityUser()].join(' ');
        const requestTime = AccessLogField.contextRequestTime();
        const request = [AccessLogField.contextHttpMethod(), AccessLogField.contextResourcePath(), AccessLogField.contextProtocol()].join(' ');
        const status = [AccessLogField.contextStatus(), AccessLogField.contextResponseLength(), AccessLogField.contextRequestId()].join(' ');
        return new AccessLogFormat(`${requester} [${requestTime}] "${request}" ${status}`);
    }
    /**
     * Access log will be produced in the JSON format with a set of fields most useful in the access log. All fields are turned on by default with the
     * option to turn off specific fields.
     */
    static jsonWithStandardFields(fields = {
        ip: true,
        user: true,
        caller: true,
        requestTime: true,
        httpMethod: true,
        resourcePath: true,
        status: true,
        protocol: true,
        responseLength: true,
    }) {
        return this.custom(JSON.stringify({
            requestId: AccessLogField.contextRequestId(),
            ip: fields.ip ? AccessLogField.contextIdentitySourceIp() : undefined,
            user: fields.user ? AccessLogField.contextIdentityUser() : undefined,
            caller: fields.caller ? AccessLogField.contextIdentityCaller() : undefined,
            requestTime: fields.requestTime ? AccessLogField.contextRequestTime() : undefined,
            httpMethod: fields.httpMethod ? AccessLogField.contextHttpMethod() : undefined,
            resourcePath: fields.resourcePath ? AccessLogField.contextResourcePath() : undefined,
            status: fields.status ? AccessLogField.contextStatus() : undefined,
            protocol: fields.protocol ? AccessLogField.contextProtocol() : undefined,
            responseLength: fields.responseLength ? AccessLogField.contextResponseLength() : undefined,
        }));
    }
    /**
     * Output a format string to be used with CloudFormation.
     */
    toString() {
        return this.format;
    }
}
exports.AccessLogFormat = AccessLogFormat;
//# sourceMappingURL=data:application/json;base64,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