"use strict";
const cdk = require("@aws-cdk/core");
const appmesh = require("../lib");
let idCounter = 0;
const getNode = (stack) => {
    const mesh = new appmesh.Mesh(stack, `mesh-${++idCounter}`, {
        meshName: 'test-mesh',
    });
    return mesh.addVirtualNode(`virtual-node-${idCounter}`, {
        dnsHostName: 'test-node',
    });
};
module.exports = {
    'interval'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const [min, max] = [5000, 300000];
        // WHEN
        const toThrow = (millis) => getNode(stack).addListeners({
            healthCheck: { interval: cdk.Duration.millis(millis) },
        });
        // THEN
        test.doesNotThrow(() => toThrow(min));
        test.doesNotThrow(() => toThrow(max));
        test.throws(() => toThrow(min - 1), /below the minimum threshold/);
        test.throws(() => toThrow(max + 1), /above the maximum threshold/);
        test.done();
    },
    'timeout'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const [min, max] = [2000, 60000];
        // WHEN
        const toThrow = (millis) => getNode(stack).addListeners({
            healthCheck: { timeout: cdk.Duration.millis(millis) },
        });
        // THEN
        test.doesNotThrow(() => toThrow(min));
        test.doesNotThrow(() => toThrow(max));
        test.throws(() => toThrow(min - 1), /below the minimum threshold/);
        test.throws(() => toThrow(max + 1), /above the maximum threshold/);
        test.done();
    },
    'port'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const [min, max] = [1, 65535];
        // WHEN
        const toThrow = (port) => getNode(stack).addListeners({
            healthCheck: { port },
        });
        // THEN
        test.doesNotThrow(() => toThrow(min));
        test.doesNotThrow(() => toThrow(max));
        // falsy, falls back to portMapping.port
        // test.throws(() => toThrow(min - 1), /below the minimum threshold/);
        test.throws(() => toThrow(max + 1), /above the maximum threshold/);
        test.done();
    },
    'healthyThreshold'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const [min, max] = [2, 10];
        // WHEN
        const toThrow = (healthyThreshold) => getNode(stack).addListeners({
            healthCheck: { healthyThreshold },
        });
        // THEN
        test.doesNotThrow(() => toThrow(min));
        test.doesNotThrow(() => toThrow(max));
        test.throws(() => toThrow(min - 1), /below the minimum threshold/);
        test.throws(() => toThrow(max + 1), /above the maximum threshold/);
        test.done();
    },
    'unhealthyThreshold'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const [min, max] = [2, 10];
        // WHEN
        const toThrow = (unhealthyThreshold) => getNode(stack).addListeners({
            healthCheck: { unhealthyThreshold },
        });
        // THEN
        test.doesNotThrow(() => toThrow(min));
        test.doesNotThrow(() => toThrow(max));
        test.throws(() => toThrow(min - 1), /below the minimum threshold/);
        test.throws(() => toThrow(max + 1), /above the maximum threshold/);
        test.done();
    },
    'throws if path and Protocol.TCP'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        const toThrow = (protocol) => getNode(stack).addListeners({
            healthCheck: {
                protocol,
                path: '/',
            },
        });
        // THEN
        test.doesNotThrow(() => toThrow(appmesh.Protocol.HTTP));
        test.throws(() => toThrow(appmesh.Protocol.TCP), /The path property cannot be set with Protocol.TCP/);
        test.done();
    },
    'throws if path and Protocol.GRPC'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        const toThrow = (protocol) => getNode(stack).addListeners({
            healthCheck: {
                protocol,
                path: '/',
            },
        });
        // THEN
        test.doesNotThrow(() => toThrow(appmesh.Protocol.HTTP));
        test.throws(() => toThrow(appmesh.Protocol.GRPC), /The path property cannot be set with Protocol.GRPC/);
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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