"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.VirtualService = void 0;
const cdk = require("@aws-cdk/core");
const appmesh_generated_1 = require("./appmesh.generated");
/**
 * VirtualService represents a service inside an AppMesh
 *
 * It routes traffic either to a Virtual Node or to a Virtual Router.
 *
 * @see https://docs.aws.amazon.com/app-mesh/latest/userguide/virtual_services.html
 */
class VirtualService extends cdk.Resource {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.virtualServiceName || cdk.Lazy.stringValue({ produce: () => this.node.uniqueId }),
        });
        if (props.virtualNode && props.virtualRouter) {
            throw new Error('Must provide only one of virtualNode or virtualRouter for the provider');
        }
        this.mesh = props.mesh;
        // Check which provider to use node or router (or neither)
        if (props.virtualRouter) {
            this.virtualServiceProvider = this.addVirtualRouter(props.virtualRouter.virtualRouterName);
        }
        if (props.virtualNode) {
            this.virtualServiceProvider = this.addVirtualNode(props.virtualNode.virtualNodeName);
        }
        const svc = new appmesh_generated_1.CfnVirtualService(this, 'Resource', {
            meshName: this.mesh.meshName,
            virtualServiceName: this.physicalName,
            spec: {
                provider: this.virtualServiceProvider,
            },
        });
        this.virtualServiceName = this.getResourceNameAttribute(svc.attrVirtualServiceName);
        this.virtualServiceArn = this.getResourceArnAttribute(svc.ref, {
            service: 'appmesh',
            resource: `mesh/${props.mesh.meshName}/virtualService`,
            resourceName: this.physicalName,
        });
    }
    /**
     * Import an existing VirtualService given an ARN
     */
    static fromVirtualServiceArn(scope, id, virtualServiceArn) {
        return new ImportedVirtualService(scope, id, {
            virtualServiceArn,
        });
    }
    /**
     * Import an existing VirtualService given mesh and service names
     */
    static fromVirtualServiceName(scope, id, meshName, virtualServiceName) {
        return new ImportedVirtualService(scope, id, {
            meshName,
            virtualServiceName,
        });
    }
    addVirtualRouter(name) {
        return {
            virtualRouter: {
                virtualRouterName: name,
            },
        };
    }
    addVirtualNode(name) {
        return {
            virtualNode: {
                virtualNodeName: name,
            },
        };
    }
}
exports.VirtualService = VirtualService;
/**
 * Returns properties that allows a VirtualService to be imported
 */
class ImportedVirtualService extends cdk.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        if (props.virtualServiceArn) {
            this.virtualServiceArn = props.virtualServiceArn;
            this.virtualServiceName = cdk.Fn.select(2, cdk.Fn.split('/', cdk.Stack.of(scope).parseArn(props.virtualServiceArn).resourceName));
        }
        else if (props.virtualServiceName && props.meshName) {
            this.virtualServiceName = props.virtualServiceName;
            this.virtualServiceArn = cdk.Stack.of(this).formatArn({
                service: 'appmesh',
                resource: `mesh/${props.meshName}/virtualService`,
                resourceName: this.virtualServiceName,
            });
        }
        else {
            throw new Error('Need either arn or both names');
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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