"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
require("@aws-cdk/assert/jest");
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const stepfunctions = require("../lib");
describe('State Machine Resources', () => {
    test('Tasks can add permissions to the execution role', () => {
        // GIVEN
        const stack = new cdk.Stack();
        const task = new stepfunctions.Task(stack, 'Task', {
            task: {
                bind: () => ({
                    resourceArn: 'resource',
                    policyStatements: [new iam.PolicyStatement({
                            actions: ['resource:Everything'],
                            resources: ['resource'],
                        })],
                }),
            },
        });
        // WHEN
        new stepfunctions.StateMachine(stack, 'SM', {
            definition: task,
        });
        // THEN
        expect(stack).toHaveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Version: '2012-10-17',
                Statement: [
                    {
                        Action: 'resource:Everything',
                        Effect: 'Allow',
                        Resource: 'resource',
                    },
                ],
            },
        });
    }),
        test('Tasks hidden inside a Parallel state are also included', () => {
            // GIVEN
            const stack = new cdk.Stack();
            const task = new stepfunctions.Task(stack, 'Task', {
                task: {
                    bind: () => ({
                        resourceArn: 'resource',
                        policyStatements: [
                            new iam.PolicyStatement({
                                actions: ['resource:Everything'],
                                resources: ['resource'],
                            }),
                        ],
                    }),
                },
            });
            const para = new stepfunctions.Parallel(stack, 'Para');
            para.branch(task);
            // WHEN
            new stepfunctions.StateMachine(stack, 'SM', {
                definition: para,
            });
            // THEN
            expect(stack).toHaveResource('AWS::IAM::Policy', {
                PolicyDocument: {
                    Version: '2012-10-17',
                    Statement: [
                        {
                            Action: 'resource:Everything',
                            Effect: 'Allow',
                            Resource: 'resource',
                        },
                    ],
                },
            });
        }),
        test('Task should render InputPath / Parameters / OutputPath correctly', () => {
            // GIVEN
            const stack = new cdk.Stack();
            const task = new stepfunctions.Task(stack, 'Task', {
                inputPath: '$',
                outputPath: '$.state',
                task: {
                    bind: () => ({
                        resourceArn: 'resource',
                        parameters: {
                            'input.$': '$',
                            'stringArgument': 'inital-task',
                            'numberArgument': 123,
                            'booleanArgument': true,
                            'arrayArgument': ['a', 'b', 'c'],
                        },
                    }),
                },
            });
            // WHEN
            const taskState = task.toStateJson();
            // THEN
            expect(taskState).toStrictEqual({ End: true,
                Retry: undefined,
                Catch: undefined,
                InputPath: '$',
                Parameters: { 'input.$': '$',
                    'stringArgument': 'inital-task',
                    'numberArgument': 123,
                    'booleanArgument': true,
                    'arrayArgument': ['a', 'b', 'c'] },
                OutputPath: '$.state',
                Type: 'Task',
                Comment: undefined,
                Resource: 'resource',
                ResultPath: undefined,
                TimeoutSeconds: undefined,
                HeartbeatSeconds: undefined,
            });
        }),
        test('Task combines taskobject parameters with direct parameters', () => {
            // GIVEN
            const stack = new cdk.Stack();
            const task = new stepfunctions.Task(stack, 'Task', {
                inputPath: '$',
                outputPath: '$.state',
                task: {
                    bind: () => ({
                        resourceArn: 'resource',
                        parameters: {
                            a: 'aa',
                        },
                    }),
                },
                parameters: {
                    b: 'bb',
                },
            });
            // WHEN
            const taskState = task.toStateJson();
            // THEN
            expect(taskState).toStrictEqual({ End: true,
                Retry: undefined,
                Catch: undefined,
                InputPath: '$',
                Parameters: { a: 'aa',
                    b: 'bb' },
                OutputPath: '$.state',
                Type: 'Task',
                Comment: undefined,
                Resource: 'resource',
                ResultPath: undefined,
                TimeoutSeconds: undefined,
                HeartbeatSeconds: undefined,
            });
        }),
        test('Created state machine can grant start execution to a role', () => {
            // GIVEN
            const stack = new cdk.Stack();
            const task = new stepfunctions.Task(stack, 'Task', {
                task: {
                    bind: () => ({ resourceArn: 'resource' }),
                },
            });
            const stateMachine = new stepfunctions.StateMachine(stack, 'StateMachine', {
                definition: task,
            });
            const role = new iam.Role(stack, 'Role', {
                assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            });
            // WHEN
            stateMachine.grantStartExecution(role);
            // THEN
            expect(stack).toHaveResourceLike('AWS::IAM::Policy', {
                PolicyDocument: {
                    Statement: assert_1.arrayWith(assert_1.objectLike({
                        Action: 'states:StartExecution',
                        Effect: 'Allow',
                        Resource: {
                            Ref: 'StateMachine2E01A3A5',
                        },
                    })),
                },
            });
        }),
        test('Created state machine can grant read access to a role', () => {
            // GIVEN
            const stack = new cdk.Stack();
            const task = new stepfunctions.Task(stack, 'Task', {
                task: {
                    bind: () => ({ resourceArn: 'resource' }),
                },
            });
            const stateMachine = new stepfunctions.StateMachine(stack, 'StateMachine', {
                definition: task,
            });
            const role = new iam.Role(stack, 'Role', {
                assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            });
            // WHEN
            stateMachine.grantRead(role);
            // THEN
            expect(stack).toHaveResourceLike('AWS::IAM::Policy', {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: [
                                'states:ListExecutions',
                                'states:ListStateMachines',
                            ],
                            Effect: 'Allow',
                            Resource: {
                                Ref: 'StateMachine2E01A3A5',
                            },
                        },
                        {
                            Action: [
                                'states:DescribeExecution',
                                'states:DescribeStateMachineForExecution',
                                'states:GetExecutionHistory',
                            ],
                            Effect: 'Allow',
                            Resource: {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':states:',
                                        {
                                            Ref: 'AWS::Region',
                                        },
                                        ':',
                                        {
                                            Ref: 'AWS::AccountId',
                                        },
                                        ':execution:',
                                        {
                                            'Fn::Select': [
                                                6,
                                                {
                                                    'Fn::Split': [
                                                        ':',
                                                        {
                                                            Ref: 'StateMachine2E01A3A5',
                                                        },
                                                    ],
                                                },
                                            ],
                                        },
                                        ':*',
                                    ],
                                ],
                            },
                        },
                        {
                            Action: [
                                'states:ListActivities',
                                'states:DescribeStateMachine',
                                'states:DescribeActivity',
                            ],
                            Effect: 'Allow',
                            Resource: '*',
                        },
                    ],
                }
            });
        }),
        test('Created state machine can grant task response actions to the state machine', () => {
            // GIVEN
            const stack = new cdk.Stack();
            const task = new stepfunctions.Task(stack, 'Task', {
                task: {
                    bind: () => ({ resourceArn: 'resource' }),
                },
            });
            const stateMachine = new stepfunctions.StateMachine(stack, 'StateMachine', {
                definition: task,
            });
            const role = new iam.Role(stack, 'Role', {
                assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            });
            // WHEN
            stateMachine.grantTaskResponse(role);
            // THEN
            expect(stack).toHaveResourceLike('AWS::IAM::Policy', {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: [
                                'states:SendTaskSuccess',
                                'states:SendTaskFailure',
                                'states:SendTaskHeartbeat',
                            ],
                            Effect: 'Allow',
                            Resource: {
                                Ref: 'StateMachine2E01A3A5',
                            },
                        },
                    ],
                },
            });
        }),
        test('Created state machine can grant actions to the executions', () => {
            // GIVEN
            const stack = new cdk.Stack();
            const task = new stepfunctions.Task(stack, 'Task', {
                task: {
                    bind: () => ({ resourceArn: 'resource' }),
                },
            });
            const stateMachine = new stepfunctions.StateMachine(stack, 'StateMachine', {
                definition: task,
            });
            const role = new iam.Role(stack, 'Role', {
                assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            });
            // WHEN
            stateMachine.grantExecution(role, 'states:GetExecutionHistory');
            // THEN
            expect(stack).toHaveResourceLike('AWS::IAM::Policy', {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: 'states:GetExecutionHistory',
                            Effect: 'Allow',
                            Resource: {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':states:',
                                        {
                                            Ref: 'AWS::Region',
                                        },
                                        ':',
                                        {
                                            Ref: 'AWS::AccountId',
                                        },
                                        ':execution:',
                                        {
                                            'Fn::Select': [
                                                6,
                                                {
                                                    'Fn::Split': [
                                                        ':',
                                                        {
                                                            Ref: 'StateMachine2E01A3A5',
                                                        },
                                                    ],
                                                },
                                            ],
                                        },
                                        ':*',
                                    ],
                                ],
                            },
                        },
                    ],
                },
            });
        }),
        test('Created state machine can grant actions to a role', () => {
            // GIVEN
            const stack = new cdk.Stack();
            const task = new stepfunctions.Task(stack, 'Task', {
                task: {
                    bind: () => ({ resourceArn: 'resource' }),
                },
            });
            const stateMachine = new stepfunctions.StateMachine(stack, 'StateMachine', {
                definition: task,
            });
            const role = new iam.Role(stack, 'Role', {
                assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            });
            // WHEN
            stateMachine.grant(role, 'states:ListExecution');
            // THEN
            expect(stack).toHaveResourceLike('AWS::IAM::Policy', {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: 'states:ListExecution',
                            Effect: 'Allow',
                            Resource: {
                                Ref: 'StateMachine2E01A3A5',
                            },
                        },
                    ],
                },
            });
        }),
        test('Imported state machine can grant start execution to a role', () => {
            // GIVEN
            const stack = new cdk.Stack();
            const stateMachineArn = 'arn:aws:states:::my-state-machine';
            const stateMachine = stepfunctions.StateMachine.fromStateMachineArn(stack, 'StateMachine', stateMachineArn);
            const role = new iam.Role(stack, 'Role', {
                assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            });
            // WHEN
            stateMachine.grantStartExecution(role);
            // THEN
            expect(stack).toHaveResource('AWS::IAM::Policy', {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: 'states:StartExecution',
                            Effect: 'Allow',
                            Resource: stateMachineArn,
                        },
                    ],
                    Version: '2012-10-17',
                },
                PolicyName: 'RoleDefaultPolicy5FFB7DAB',
                Roles: [
                    {
                        Ref: 'Role1ABCC5F0',
                    },
                ],
            });
        }),
        test('Imported state machine can grant read access to a role', () => {
            // GIVEN
            const stack = new cdk.Stack();
            const stateMachineArn = 'arn:aws:states:::my-state-machine';
            const stateMachine = stepfunctions.StateMachine.fromStateMachineArn(stack, 'StateMachine', stateMachineArn);
            const role = new iam.Role(stack, 'Role', {
                assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            });
            // WHEN
            stateMachine.grantRead(role);
            // THEN
            expect(stack).toHaveResourceLike('AWS::IAM::Policy', {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: [
                                'states:ListExecutions',
                                'states:ListStateMachines',
                            ],
                            Effect: 'Allow',
                            Resource: stateMachineArn,
                        },
                        {
                            Action: [
                                'states:DescribeExecution',
                                'states:DescribeStateMachineForExecution',
                                'states:GetExecutionHistory',
                            ],
                            Effect: 'Allow',
                            Resource: {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':states:',
                                        {
                                            Ref: 'AWS::Region',
                                        },
                                        ':',
                                        {
                                            Ref: 'AWS::AccountId',
                                        },
                                        ':execution:*',
                                    ],
                                ],
                            },
                        },
                        {
                            Action: [
                                'states:ListActivities',
                                'states:DescribeStateMachine',
                                'states:DescribeActivity',
                            ],
                            Effect: 'Allow',
                            Resource: '*',
                        },
                    ],
                }
            });
        }),
        test('Imported state machine can task response permissions to the state machine', () => {
            // GIVEN
            const stack = new cdk.Stack();
            const stateMachineArn = 'arn:aws:states:::my-state-machine';
            const stateMachine = stepfunctions.StateMachine.fromStateMachineArn(stack, 'StateMachine', stateMachineArn);
            const role = new iam.Role(stack, 'Role', {
                assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            });
            // WHEN
            stateMachine.grantTaskResponse(role);
            // THEN
            expect(stack).toHaveResourceLike('AWS::IAM::Policy', {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: [
                                'states:SendTaskSuccess',
                                'states:SendTaskFailure',
                                'states:SendTaskHeartbeat',
                            ],
                            Effect: 'Allow',
                            Resource: stateMachineArn,
                        },
                    ],
                },
            });
        }),
        test('Imported state machine can grant access to a role', () => {
            // GIVEN
            const stack = new cdk.Stack();
            const stateMachineArn = 'arn:aws:states:::my-state-machine';
            const stateMachine = stepfunctions.StateMachine.fromStateMachineArn(stack, 'StateMachine', stateMachineArn);
            const role = new iam.Role(stack, 'Role', {
                assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            });
            // WHEN
            stateMachine.grant(role, 'states:ListExecution');
            // THEN
            expect(stack).toHaveResourceLike('AWS::IAM::Policy', {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: 'states:ListExecution',
                            Effect: 'Allow',
                            Resource: stateMachine.stateMachineArn,
                        },
                    ],
                },
            });
        }),
        test('Pass should render InputPath / Parameters / OutputPath correctly', () => {
            // GIVEN
            const stack = new cdk.Stack();
            const task = new stepfunctions.Pass(stack, 'Pass', {
                inputPath: '$',
                outputPath: '$.state',
                parameters: {
                    'input.$': '$',
                    'stringArgument': 'inital-task',
                    'numberArgument': 123,
                    'booleanArgument': true,
                    'arrayArgument': ['a', 'b', 'c'],
                },
            });
            // WHEN
            const taskState = task.toStateJson();
            // THEN
            expect(taskState).toStrictEqual({ End: true,
                InputPath: '$',
                OutputPath: '$.state',
                Parameters: { 'input.$': '$',
                    'stringArgument': 'inital-task',
                    'numberArgument': 123,
                    'booleanArgument': true,
                    'arrayArgument': ['a', 'b', 'c'] },
                Type: 'Pass',
                Comment: undefined,
                Result: undefined,
                ResultPath: undefined,
            });
        }),
        test('parameters can be selected from the input with a path', () => {
            // GIVEN
            const stack = new cdk.Stack();
            const task = new stepfunctions.Pass(stack, 'Pass', {
                parameters: {
                    input: stepfunctions.Data.stringAt('$.myField'),
                },
            });
            // WHEN
            const taskState = task.toStateJson();
            // THEN
            expect(taskState).toEqual({ End: true,
                Parameters: { 'input.$': '$.myField' },
                Type: 'Pass',
            });
        }),
        test('State machines must depend on their roles', () => {
            // GIVEN
            const stack = new cdk.Stack();
            const task = new stepfunctions.Task(stack, 'Task', {
                task: {
                    bind: () => ({
                        resourceArn: 'resource',
                        policyStatements: [
                            new iam.PolicyStatement({
                                resources: ['resource'],
                                actions: ['lambda:InvokeFunction'],
                            }),
                        ],
                    }),
                },
            });
            new stepfunctions.StateMachine(stack, 'StateMachine', {
                definition: task,
            });
            // THEN
            expect(stack).toHaveResource('AWS::StepFunctions::StateMachine', {
                DependsOn: [
                    'StateMachineRoleDefaultPolicyDF1E6607',
                    'StateMachineRoleB840431D',
                ],
            }, assert_1.ResourcePart.CompleteDefinition);
        });
});
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoic3RhdGUtbWFjaGluZS1yZXNvdXJjZXMudGVzdC5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbInN0YXRlLW1hY2hpbmUtcmVzb3VyY2VzLnRlc3QudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7QUFBQSw0Q0FBc0U7QUFDdEUsZ0NBQThCO0FBQzlCLHdDQUF3QztBQUN4QyxxQ0FBcUM7QUFDckMsd0NBQXdDO0FBRXhDLFFBQVEsQ0FBQyx5QkFBeUIsRUFBRSxHQUFHLEVBQUU7SUFDdkMsSUFBSSxDQUFDLGlEQUFpRCxFQUFFLEdBQUcsRUFBRTtRQUMzRCxRQUFRO1FBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7UUFDOUIsTUFBTSxJQUFJLEdBQUcsSUFBSSxhQUFhLENBQUMsSUFBSSxDQUFDLEtBQUssRUFBRSxNQUFNLEVBQUU7WUFDakQsSUFBSSxFQUFFO2dCQUNKLElBQUksRUFBRSxHQUFHLEVBQUUsQ0FBQyxDQUFDO29CQUNYLFdBQVcsRUFBRSxVQUFVO29CQUN2QixnQkFBZ0IsRUFBRSxDQUFDLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQzs0QkFDekMsT0FBTyxFQUFFLENBQUMscUJBQXFCLENBQUM7NEJBQ2hDLFNBQVMsRUFBRSxDQUFDLFVBQVUsQ0FBQzt5QkFDeEIsQ0FBQyxDQUFDO2lCQUNKLENBQUM7YUFDSDtTQUNGLENBQUMsQ0FBQztRQUVILE9BQU87UUFDUCxJQUFJLGFBQWEsQ0FBQyxZQUFZLENBQUMsS0FBSyxFQUFFLElBQUksRUFBRTtZQUMxQyxVQUFVLEVBQUUsSUFBSTtTQUNqQixDQUFDLENBQUM7UUFFSCxPQUFPO1FBQ1AsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLGNBQWMsQ0FBQyxrQkFBa0IsRUFBRTtZQUMvQyxjQUFjLEVBQUU7Z0JBQ2QsT0FBTyxFQUFFLFlBQVk7Z0JBQ3JCLFNBQVMsRUFBRTtvQkFDVDt3QkFDRSxNQUFNLEVBQUUscUJBQXFCO3dCQUM3QixNQUFNLEVBQUUsT0FBTzt3QkFDZixRQUFRLEVBQUUsVUFBVTtxQkFDckI7aUJBQ0Y7YUFDRjtTQUNGLENBQUMsQ0FBQztJQUNMLENBQUMsQ0FBQztRQUVGLElBQUksQ0FBQyx3REFBd0QsRUFBRSxHQUFHLEVBQUU7WUFDbEUsUUFBUTtZQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1lBQzlCLE1BQU0sSUFBSSxHQUFHLElBQUksYUFBYSxDQUFDLElBQUksQ0FBQyxLQUFLLEVBQUUsTUFBTSxFQUFFO2dCQUNqRCxJQUFJLEVBQUU7b0JBQ0osSUFBSSxFQUFFLEdBQUcsRUFBRSxDQUFDLENBQUM7d0JBQ1gsV0FBVyxFQUFFLFVBQVU7d0JBQ3ZCLGdCQUFnQixFQUFFOzRCQUNoQixJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUM7Z0NBQ3RCLE9BQU8sRUFBRSxDQUFDLHFCQUFxQixDQUFDO2dDQUNoQyxTQUFTLEVBQUUsQ0FBQyxVQUFVLENBQUM7NkJBQ3hCLENBQUM7eUJBQ0g7cUJBQ0YsQ0FBQztpQkFDSDthQUNGLENBQUMsQ0FBQztZQUVILE1BQU0sSUFBSSxHQUFHLElBQUksYUFBYSxDQUFDLFFBQVEsQ0FBQyxLQUFLLEVBQUUsTUFBTSxDQUFDLENBQUM7WUFDdkQsSUFBSSxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsQ0FBQztZQUVsQixPQUFPO1lBQ1AsSUFBSSxhQUFhLENBQUMsWUFBWSxDQUFDLEtBQUssRUFBRSxJQUFJLEVBQUU7Z0JBQzFDLFVBQVUsRUFBRSxJQUFJO2FBQ2pCLENBQUMsQ0FBQztZQUVILE9BQU87WUFDUCxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsY0FBYyxDQUFDLGtCQUFrQixFQUFFO2dCQUMvQyxjQUFjLEVBQUU7b0JBQ2QsT0FBTyxFQUFFLFlBQVk7b0JBQ3JCLFNBQVMsRUFBRTt3QkFDVDs0QkFDRSxNQUFNLEVBQUUscUJBQXFCOzRCQUM3QixNQUFNLEVBQUUsT0FBTzs0QkFDZixRQUFRLEVBQUUsVUFBVTt5QkFDckI7cUJBQ0Y7aUJBQ0Y7YUFDRixDQUFDLENBQUM7UUFDTCxDQUFDLENBQUM7UUFFRixJQUFJLENBQUMsa0VBQWtFLEVBQUUsR0FBRyxFQUFFO1lBQzVFLFFBQVE7WUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztZQUM5QixNQUFNLElBQUksR0FBRyxJQUFJLGFBQWEsQ0FBQyxJQUFJLENBQUMsS0FBSyxFQUFFLE1BQU0sRUFBRTtnQkFDakQsU0FBUyxFQUFFLEdBQUc7Z0JBQ2QsVUFBVSxFQUFFLFNBQVM7Z0JBQ3JCLElBQUksRUFBRTtvQkFDSixJQUFJLEVBQUUsR0FBRyxFQUFFLENBQUMsQ0FBQzt3QkFDWCxXQUFXLEVBQUUsVUFBVTt3QkFDdkIsVUFBVSxFQUFFOzRCQUNWLFNBQVMsRUFBRSxHQUFHOzRCQUNkLGdCQUFnQixFQUFFLGFBQWE7NEJBQy9CLGdCQUFnQixFQUFFLEdBQUc7NEJBQ3JCLGlCQUFpQixFQUFFLElBQUk7NEJBQ3ZCLGVBQWUsRUFBRSxDQUFDLEdBQUcsRUFBRSxHQUFHLEVBQUUsR0FBRyxDQUFDO3lCQUNqQztxQkFDRixDQUFDO2lCQUNIO2FBQ0YsQ0FBQyxDQUFDO1lBRUgsT0FBTztZQUNQLE1BQU0sU0FBUyxHQUFHLElBQUksQ0FBQyxXQUFXLEVBQUUsQ0FBQztZQUVyQyxPQUFPO1lBQ1AsTUFBTSxDQUFDLFNBQVMsQ0FBQyxDQUFDLGFBQWEsQ0FBQyxFQUFFLEdBQUcsRUFBRSxJQUFJO2dCQUN6QyxLQUFLLEVBQUUsU0FBUztnQkFDaEIsS0FBSyxFQUFFLFNBQVM7Z0JBQ2hCLFNBQVMsRUFBRSxHQUFHO2dCQUNkLFVBQVUsRUFDSCxFQUFFLFNBQVMsRUFBRSxHQUFHO29CQUNkLGdCQUFnQixFQUFFLGFBQWE7b0JBQy9CLGdCQUFnQixFQUFFLEdBQUc7b0JBQ3JCLGlCQUFpQixFQUFFLElBQUk7b0JBQ3ZCLGVBQWUsRUFBRSxDQUFFLEdBQUcsRUFBRSxHQUFHLEVBQUUsR0FBRyxDQUFFLEVBQUU7Z0JBQzdDLFVBQVUsRUFBRSxTQUFTO2dCQUNyQixJQUFJLEVBQUUsTUFBTTtnQkFDWixPQUFPLEVBQUUsU0FBUztnQkFDbEIsUUFBUSxFQUFFLFVBQVU7Z0JBQ3BCLFVBQVUsRUFBRSxTQUFTO2dCQUNyQixjQUFjLEVBQUUsU0FBUztnQkFDekIsZ0JBQWdCLEVBQUUsU0FBUzthQUM1QixDQUFDLENBQUM7UUFDTCxDQUFDLENBQUM7UUFFRixJQUFJLENBQUMsNERBQTRELEVBQUUsR0FBRyxFQUFFO1lBQ3RFLFFBQVE7WUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztZQUM5QixNQUFNLElBQUksR0FBRyxJQUFJLGFBQWEsQ0FBQyxJQUFJLENBQUMsS0FBSyxFQUFFLE1BQU0sRUFBRTtnQkFDakQsU0FBUyxFQUFFLEdBQUc7Z0JBQ2QsVUFBVSxFQUFFLFNBQVM7Z0JBQ3JCLElBQUksRUFBRTtvQkFDSixJQUFJLEVBQUUsR0FBRyxFQUFFLENBQUMsQ0FBQzt3QkFDWCxXQUFXLEVBQUUsVUFBVTt3QkFDdkIsVUFBVSxFQUFFOzRCQUNWLENBQUMsRUFBRSxJQUFJO3lCQUNSO3FCQUNGLENBQUM7aUJBQ0g7Z0JBQ0QsVUFBVSxFQUFFO29CQUNWLENBQUMsRUFBRSxJQUFJO2lCQUNSO2FBQ0YsQ0FBQyxDQUFDO1lBRUgsT0FBTztZQUNQLE1BQU0sU0FBUyxHQUFHLElBQUksQ0FBQyxXQUFXLEVBQUUsQ0FBQztZQUVyQyxPQUFPO1lBQ1AsTUFBTSxDQUFDLFNBQVMsQ0FBQyxDQUFDLGFBQWEsQ0FBQyxFQUFFLEdBQUcsRUFBRSxJQUFJO2dCQUN6QyxLQUFLLEVBQUUsU0FBUztnQkFDaEIsS0FBSyxFQUFFLFNBQVM7Z0JBQ2hCLFNBQVMsRUFBRSxHQUFHO2dCQUNkLFVBQVUsRUFDSCxFQUFFLENBQUMsRUFBRSxJQUFJO29CQUNQLENBQUMsRUFBRSxJQUFJLEVBQUU7Z0JBQ2xCLFVBQVUsRUFBRSxTQUFTO2dCQUNyQixJQUFJLEVBQUUsTUFBTTtnQkFDWixPQUFPLEVBQUUsU0FBUztnQkFDbEIsUUFBUSxFQUFFLFVBQVU7Z0JBQ3BCLFVBQVUsRUFBRSxTQUFTO2dCQUNyQixjQUFjLEVBQUUsU0FBUztnQkFDekIsZ0JBQWdCLEVBQUUsU0FBUzthQUM1QixDQUFDLENBQUM7UUFDTCxDQUFDLENBQUM7UUFFRixJQUFJLENBQUMsMkRBQTJELEVBQUUsR0FBRyxFQUFFO1lBQ3JFLFFBQVE7WUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztZQUM5QixNQUFNLElBQUksR0FBRyxJQUFJLGFBQWEsQ0FBQyxJQUFJLENBQUMsS0FBSyxFQUFFLE1BQU0sRUFBRTtnQkFDakQsSUFBSSxFQUFFO29CQUNKLElBQUksRUFBRSxHQUFHLEVBQUUsQ0FBQyxDQUFDLEVBQUUsV0FBVyxFQUFFLFVBQVUsRUFBRSxDQUFDO2lCQUMxQzthQUNGLENBQUMsQ0FBQztZQUNILE1BQU0sWUFBWSxHQUFHLElBQUksYUFBYSxDQUFDLFlBQVksQ0FBQyxLQUFLLEVBQUUsY0FBYyxFQUFFO2dCQUN6RSxVQUFVLEVBQUUsSUFBSTthQUNqQixDQUFDLENBQUM7WUFDSCxNQUFNLElBQUksR0FBRyxJQUFJLEdBQUcsQ0FBQyxJQUFJLENBQUMsS0FBSyxFQUFFLE1BQU0sRUFBRTtnQkFDdkMsU0FBUyxFQUFFLElBQUksR0FBRyxDQUFDLGdCQUFnQixDQUFDLHNCQUFzQixDQUFDO2FBQzVELENBQUMsQ0FBQztZQUVILE9BQU87WUFDUCxZQUFZLENBQUMsbUJBQW1CLENBQUMsSUFBSSxDQUFDLENBQUM7WUFFdkMsT0FBTztZQUNQLE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxrQkFBa0IsQ0FBQyxrQkFBa0IsRUFBRTtnQkFDbkQsY0FBYyxFQUFFO29CQUNkLFNBQVMsRUFBRSxrQkFBUyxDQUFDLG1CQUFVLENBQUM7d0JBQzlCLE1BQU0sRUFBRSx1QkFBdUI7d0JBQy9CLE1BQU0sRUFBRSxPQUFPO3dCQUNmLFFBQVEsRUFBRTs0QkFDUixHQUFHLEVBQUUsc0JBQXNCO3lCQUM1QjtxQkFDRixDQUFDLENBQUM7aUJBQ0o7YUFDRixDQUFDLENBQUM7UUFFTCxDQUFDLENBQUM7UUFFRixJQUFJLENBQUMsdURBQXVELEVBQUUsR0FBRyxFQUFFO1lBQ2pFLFFBQVE7WUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztZQUM5QixNQUFNLElBQUksR0FBRyxJQUFJLGFBQWEsQ0FBQyxJQUFJLENBQUMsS0FBSyxFQUFFLE1BQU0sRUFBRTtnQkFDakQsSUFBSSxFQUFFO29CQUNKLElBQUksRUFBRSxHQUFHLEVBQUUsQ0FBQyxDQUFDLEVBQUUsV0FBVyxFQUFFLFVBQVUsRUFBRSxDQUFDO2lCQUMxQzthQUNGLENBQUMsQ0FBQztZQUNILE1BQU0sWUFBWSxHQUFHLElBQUksYUFBYSxDQUFDLFlBQVksQ0FBQyxLQUFLLEVBQUUsY0FBYyxFQUFFO2dCQUN6RSxVQUFVLEVBQUUsSUFBSTthQUNqQixDQUFDLENBQUM7WUFDSCxNQUFNLElBQUksR0FBRyxJQUFJLEdBQUcsQ0FBQyxJQUFJLENBQUMsS0FBSyxFQUFFLE1BQU0sRUFBRTtnQkFDdkMsU0FBUyxFQUFFLElBQUksR0FBRyxDQUFDLGdCQUFnQixDQUFDLHNCQUFzQixDQUFDO2FBQzVELENBQUMsQ0FBQztZQUVILE9BQU87WUFDUCxZQUFZLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxDQUFDO1lBRTdCLE9BQU87WUFDUCxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsa0JBQWtCLENBQUMsa0JBQWtCLEVBQUU7Z0JBQ25ELGNBQWMsRUFBRTtvQkFDZCxTQUFTLEVBQUU7d0JBQ1Q7NEJBQ0UsTUFBTSxFQUFFO2dDQUNOLHVCQUF1QjtnQ0FDdkIsMEJBQTBCOzZCQUMzQjs0QkFDRCxNQUFNLEVBQUUsT0FBTzs0QkFDZixRQUFRLEVBQUU7Z0NBQ1IsR0FBRyxFQUFFLHNCQUFzQjs2QkFDNUI7eUJBQ0Y7d0JBQ0Q7NEJBQ0UsTUFBTSxFQUFFO2dDQUNOLDBCQUEwQjtnQ0FDMUIseUNBQXlDO2dDQUN6Qyw0QkFBNEI7NkJBQzdCOzRCQUNELE1BQU0sRUFBRSxPQUFPOzRCQUNmLFFBQVEsRUFBRTtnQ0FDUixVQUFVLEVBQUU7b0NBQ1YsRUFBRTtvQ0FDRjt3Q0FDRSxNQUFNO3dDQUNOOzRDQUNFLEdBQUcsRUFBRSxnQkFBZ0I7eUNBQ3RCO3dDQUNELFVBQVU7d0NBQ1Y7NENBQ0UsR0FBRyxFQUFFLGFBQWE7eUNBQ25CO3dDQUNELEdBQUc7d0NBQ0g7NENBQ0UsR0FBRyxFQUFFLGdCQUFnQjt5Q0FDdEI7d0NBQ0QsYUFBYTt3Q0FDYjs0Q0FDRSxZQUFZLEVBQUU7Z0RBQ1osQ0FBQztnREFDRDtvREFDRSxXQUFXLEVBQUU7d0RBQ1gsR0FBRzt3REFDSDs0REFDRSxHQUFHLEVBQUUsc0JBQXNCO3lEQUM1QjtxREFDRjtpREFDRjs2Q0FDRjt5Q0FDRjt3Q0FDRCxJQUFJO3FDQUNMO2lDQUNGOzZCQUNGO3lCQUNGO3dCQUNEOzRCQUNFLE1BQU0sRUFBRTtnQ0FDTix1QkFBdUI7Z0NBQ3ZCLDZCQUE2QjtnQ0FDN0IseUJBQXlCOzZCQUMxQjs0QkFDRCxNQUFNLEVBQUUsT0FBTzs0QkFDZixRQUFRLEVBQUUsR0FBRzt5QkFDZDtxQkFDRjtpQkFDRjthQUFDLENBQ0gsQ0FBQztRQUVKLENBQUMsQ0FBQztRQUVGLElBQUksQ0FBQyw0RUFBNEUsRUFBRSxHQUFHLEVBQUU7WUFDdEYsUUFBUTtZQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1lBQzlCLE1BQU0sSUFBSSxHQUFHLElBQUksYUFBYSxDQUFDLElBQUksQ0FBQyxLQUFLLEVBQUUsTUFBTSxFQUFFO2dCQUNqRCxJQUFJLEVBQUU7b0JBQ0osSUFBSSxFQUFFLEdBQUcsRUFBRSxDQUFDLENBQUMsRUFBRSxXQUFXLEVBQUUsVUFBVSxFQUFFLENBQUM7aUJBQzFDO2FBQ0YsQ0FBQyxDQUFDO1lBQ0gsTUFBTSxZQUFZLEdBQUcsSUFBSSxhQUFhLENBQUMsWUFBWSxDQUFDLEtBQUssRUFBRSxjQUFjLEVBQUU7Z0JBQ3pFLFVBQVUsRUFBRSxJQUFJO2FBQ2pCLENBQUMsQ0FBQztZQUNILE1BQU0sSUFBSSxHQUFHLElBQUksR0FBRyxDQUFDLElBQUksQ0FBQyxLQUFLLEVBQUUsTUFBTSxFQUFFO2dCQUN2QyxTQUFTLEVBQUUsSUFBSSxHQUFHLENBQUMsZ0JBQWdCLENBQUMsc0JBQXNCLENBQUM7YUFDNUQsQ0FBQyxDQUFDO1lBRUgsT0FBTztZQUNQLFlBQVksQ0FBQyxpQkFBaUIsQ0FBQyxJQUFJLENBQUMsQ0FBQztZQUVyQyxPQUFPO1lBQ1AsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLGtCQUFrQixDQUFDLGtCQUFrQixFQUFFO2dCQUNuRCxjQUFjLEVBQUU7b0JBQ2QsU0FBUyxFQUFFO3dCQUNUOzRCQUNFLE1BQU0sRUFBRTtnQ0FDTix3QkFBd0I7Z0NBQ3hCLHdCQUF3QjtnQ0FDeEIsMEJBQTBCOzZCQUMzQjs0QkFDRCxNQUFNLEVBQUUsT0FBTzs0QkFDZixRQUFRLEVBQUU7Z0NBQ1IsR0FBRyxFQUFFLHNCQUFzQjs2QkFDNUI7eUJBQ0Y7cUJBQ0Y7aUJBQ0Y7YUFDRixDQUFDLENBQUM7UUFDTCxDQUFDLENBQUM7UUFFRixJQUFJLENBQUMsMkRBQTJELEVBQUUsR0FBRyxFQUFFO1lBQ3JFLFFBQVE7WUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztZQUM5QixNQUFNLElBQUksR0FBRyxJQUFJLGFBQWEsQ0FBQyxJQUFJLENBQUMsS0FBSyxFQUFFLE1BQU0sRUFBRTtnQkFDakQsSUFBSSxFQUFFO29CQUNKLElBQUksRUFBRSxHQUFHLEVBQUUsQ0FBQyxDQUFDLEVBQUUsV0FBVyxFQUFFLFVBQVUsRUFBRSxDQUFDO2lCQUMxQzthQUNGLENBQUMsQ0FBQztZQUNILE1BQU0sWUFBWSxHQUFHLElBQUksYUFBYSxDQUFDLFlBQVksQ0FBQyxLQUFLLEVBQUUsY0FBYyxFQUFFO2dCQUN6RSxVQUFVLEVBQUUsSUFBSTthQUNqQixDQUFDLENBQUM7WUFDSCxNQUFNLElBQUksR0FBRyxJQUFJLEdBQUcsQ0FBQyxJQUFJLENBQUMsS0FBSyxFQUFFLE1BQU0sRUFBRTtnQkFDdkMsU0FBUyxFQUFFLElBQUksR0FBRyxDQUFDLGdCQUFnQixDQUFDLHNCQUFzQixDQUFDO2FBQzVELENBQUMsQ0FBQztZQUVILE9BQU87WUFDUCxZQUFZLENBQUMsY0FBYyxDQUFDLElBQUksRUFBRSw0QkFBNEIsQ0FBQyxDQUFDO1lBRWhFLE9BQU87WUFDUCxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsa0JBQWtCLENBQUMsa0JBQWtCLEVBQUU7Z0JBQ25ELGNBQWMsRUFBRTtvQkFDZCxTQUFTLEVBQUU7d0JBQ1Q7NEJBQ0UsTUFBTSxFQUFFLDRCQUE0Qjs0QkFDcEMsTUFBTSxFQUFFLE9BQU87NEJBQ2YsUUFBUSxFQUFFO2dDQUNSLFVBQVUsRUFBRTtvQ0FDVixFQUFFO29DQUNGO3dDQUNFLE1BQU07d0NBQ047NENBQ0UsR0FBRyxFQUFFLGdCQUFnQjt5Q0FDdEI7d0NBQ0QsVUFBVTt3Q0FDVjs0Q0FDRSxHQUFHLEVBQUUsYUFBYTt5Q0FDbkI7d0NBQ0QsR0FBRzt3Q0FDSDs0Q0FDRSxHQUFHLEVBQUUsZ0JBQWdCO3lDQUN0Qjt3Q0FDRCxhQUFhO3dDQUNiOzRDQUNFLFlBQVksRUFBRTtnREFDWixDQUFDO2dEQUNEO29EQUNFLFdBQVcsRUFBRTt3REFDWCxHQUFHO3dEQUNIOzREQUNFLEdBQUcsRUFBRSxzQkFBc0I7eURBQzVCO3FEQUNGO2lEQUNGOzZDQUNGO3lDQUNGO3dDQUNELElBQUk7cUNBQ0w7aUNBQ0Y7NkJBQ0Y7eUJBQ0Y7cUJBQ0Y7aUJBQ0Y7YUFDRixDQUFDLENBQUM7UUFDTCxDQUFDLENBQUM7UUFFRixJQUFJLENBQUMsbURBQW1ELEVBQUUsR0FBRyxFQUFFO1lBQzdELFFBQVE7WUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztZQUM5QixNQUFNLElBQUksR0FBRyxJQUFJLGFBQWEsQ0FBQyxJQUFJLENBQUMsS0FBSyxFQUFFLE1BQU0sRUFBRTtnQkFDakQsSUFBSSxFQUFFO29CQUNKLElBQUksRUFBRSxHQUFHLEVBQUUsQ0FBQyxDQUFDLEVBQUUsV0FBVyxFQUFFLFVBQVUsRUFBRSxDQUFDO2lCQUMxQzthQUNGLENBQUMsQ0FBQztZQUNILE1BQU0sWUFBWSxHQUFHLElBQUksYUFBYSxDQUFDLFlBQVksQ0FBQyxLQUFLLEVBQUUsY0FBYyxFQUFFO2dCQUN6RSxVQUFVLEVBQUUsSUFBSTthQUNqQixDQUFDLENBQUM7WUFDSCxNQUFNLElBQUksR0FBRyxJQUFJLEdBQUcsQ0FBQyxJQUFJLENBQUMsS0FBSyxFQUFFLE1BQU0sRUFBRTtnQkFDdkMsU0FBUyxFQUFFLElBQUksR0FBRyxDQUFDLGdCQUFnQixDQUFDLHNCQUFzQixDQUFDO2FBQzVELENBQUMsQ0FBQztZQUVILE9BQU87WUFDUCxZQUFZLENBQUMsS0FBSyxDQUFDLElBQUksRUFBRSxzQkFBc0IsQ0FBQyxDQUFDO1lBRWpELE9BQU87WUFDUCxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsa0JBQWtCLENBQUMsa0JBQWtCLEVBQUU7Z0JBQ25ELGNBQWMsRUFBRTtvQkFDZCxTQUFTLEVBQUU7d0JBQ1Q7NEJBQ0UsTUFBTSxFQUFFLHNCQUFzQjs0QkFDOUIsTUFBTSxFQUFFLE9BQU87NEJBQ2YsUUFBUSxFQUFFO2dDQUNSLEdBQUcsRUFBRSxzQkFBc0I7NkJBQzVCO3lCQUNGO3FCQUNGO2lCQUNGO2FBQ0YsQ0FBQyxDQUFDO1FBRUwsQ0FBQyxDQUFDO1FBRUYsSUFBSSxDQUFDLDREQUE0RCxFQUFFLEdBQUcsRUFBRTtZQUN0RSxRQUFRO1lBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7WUFDOUIsTUFBTSxlQUFlLEdBQUcsbUNBQW1DLENBQUM7WUFDNUQsTUFBTSxZQUFZLEdBQUcsYUFBYSxDQUFDLFlBQVksQ0FBQyxtQkFBbUIsQ0FBQyxLQUFLLEVBQUUsY0FBYyxFQUFFLGVBQWUsQ0FBQyxDQUFDO1lBQzVHLE1BQU0sSUFBSSxHQUFHLElBQUksR0FBRyxDQUFDLElBQUksQ0FBQyxLQUFLLEVBQUUsTUFBTSxFQUFFO2dCQUN2QyxTQUFTLEVBQUUsSUFBSSxHQUFHLENBQUMsZ0JBQWdCLENBQUMsc0JBQXNCLENBQUM7YUFDNUQsQ0FBQyxDQUFDO1lBRUgsT0FBTztZQUNQLFlBQVksQ0FBQyxtQkFBbUIsQ0FBQyxJQUFJLENBQUMsQ0FBQztZQUV2QyxPQUFPO1lBQ1AsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLGNBQWMsQ0FBQyxrQkFBa0IsRUFBRTtnQkFDL0MsY0FBYyxFQUFFO29CQUNkLFNBQVMsRUFBRTt3QkFDVDs0QkFDRSxNQUFNLEVBQUUsdUJBQXVCOzRCQUMvQixNQUFNLEVBQUUsT0FBTzs0QkFDZixRQUFRLEVBQUUsZUFBZTt5QkFDMUI7cUJBQ0Y7b0JBQ0QsT0FBTyxFQUFFLFlBQVk7aUJBQ3RCO2dCQUNELFVBQVUsRUFBRSwyQkFBMkI7Z0JBQ3ZDLEtBQUssRUFBRTtvQkFDTDt3QkFDRSxHQUFHLEVBQUUsY0FBYztxQkFDcEI7aUJBQ0Y7YUFDRixDQUFDLENBQUM7UUFDTCxDQUFDLENBQUM7UUFFRixJQUFJLENBQUMsd0RBQXdELEVBQUUsR0FBRyxFQUFFO1lBQ2xFLFFBQVE7WUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztZQUM5QixNQUFNLGVBQWUsR0FBRyxtQ0FBbUMsQ0FBQztZQUM1RCxNQUFNLFlBQVksR0FBRyxhQUFhLENBQUMsWUFBWSxDQUFDLG1CQUFtQixDQUFDLEtBQUssRUFBRSxjQUFjLEVBQUUsZUFBZSxDQUFDLENBQUM7WUFDNUcsTUFBTSxJQUFJLEdBQUcsSUFBSSxHQUFHLENBQUMsSUFBSSxDQUFDLEtBQUssRUFBRSxNQUFNLEVBQUU7Z0JBQ3ZDLFNBQVMsRUFBRSxJQUFJLEdBQUcsQ0FBQyxnQkFBZ0IsQ0FBQyxzQkFBc0IsQ0FBQzthQUM1RCxDQUFDLENBQUM7WUFFSCxPQUFPO1lBQ1AsWUFBWSxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsQ0FBQztZQUU3QixPQUFPO1lBQ1AsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLGtCQUFrQixDQUFDLGtCQUFrQixFQUFFO2dCQUNuRCxjQUFjLEVBQUU7b0JBQ2QsU0FBUyxFQUFFO3dCQUNUOzRCQUNFLE1BQU0sRUFBRTtnQ0FDTix1QkFBdUI7Z0NBQ3ZCLDBCQUEwQjs2QkFDM0I7NEJBQ0QsTUFBTSxFQUFFLE9BQU87NEJBQ2YsUUFBUSxFQUFFLGVBQWU7eUJBQzFCO3dCQUNEOzRCQUNFLE1BQU0sRUFBRTtnQ0FDTiwwQkFBMEI7Z0NBQzFCLHlDQUF5QztnQ0FDekMsNEJBQTRCOzZCQUM3Qjs0QkFDRCxNQUFNLEVBQUUsT0FBTzs0QkFDZixRQUFRLEVBQUU7Z0NBQ1IsVUFBVSxFQUFFO29DQUNWLEVBQUU7b0NBQ0Y7d0NBQ0UsTUFBTTt3Q0FDTjs0Q0FDRSxHQUFHLEVBQUUsZ0JBQWdCO3lDQUN0Qjt3Q0FDRCxVQUFVO3dDQUNWOzRDQUNFLEdBQUcsRUFBRSxhQUFhO3lDQUNuQjt3Q0FDRCxHQUFHO3dDQUNIOzRDQUNFLEdBQUcsRUFBRSxnQkFBZ0I7eUNBQ3RCO3dDQUNELGNBQWM7cUNBQ2Y7aUNBQ0Y7NkJBQ0Y7eUJBQ0Y7d0JBQ0Q7NEJBQ0UsTUFBTSxFQUFFO2dDQUNOLHVCQUF1QjtnQ0FDdkIsNkJBQTZCO2dDQUM3Qix5QkFBeUI7NkJBQzFCOzRCQUNELE1BQU0sRUFBRSxPQUFPOzRCQUNmLFFBQVEsRUFBRSxHQUFHO3lCQUNkO3FCQUNGO2lCQUNGO2FBQUMsQ0FDSCxDQUFDO1FBQ0osQ0FBQyxDQUFDO1FBRUYsSUFBSSxDQUFDLDJFQUEyRSxFQUFFLEdBQUcsRUFBRTtZQUNyRixRQUFRO1lBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7WUFDOUIsTUFBTSxlQUFlLEdBQUcsbUNBQW1DLENBQUM7WUFDNUQsTUFBTSxZQUFZLEdBQUcsYUFBYSxDQUFDLFlBQVksQ0FBQyxtQkFBbUIsQ0FBQyxLQUFLLEVBQUUsY0FBYyxFQUFFLGVBQWUsQ0FBQyxDQUFDO1lBQzVHLE1BQU0sSUFBSSxHQUFHLElBQUksR0FBRyxDQUFDLElBQUksQ0FBQyxLQUFLLEVBQUUsTUFBTSxFQUFFO2dCQUN2QyxTQUFTLEVBQUUsSUFBSSxHQUFHLENBQUMsZ0JBQWdCLENBQUMsc0JBQXNCLENBQUM7YUFDNUQsQ0FBQyxDQUFDO1lBRUgsT0FBTztZQUNQLFlBQVksQ0FBQyxpQkFBaUIsQ0FBQyxJQUFJLENBQUMsQ0FBQztZQUVyQyxPQUFPO1lBQ1AsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLGtCQUFrQixDQUFDLGtCQUFrQixFQUFFO2dCQUNuRCxjQUFjLEVBQUU7b0JBQ2QsU0FBUyxFQUFFO3dCQUNUOzRCQUNFLE1BQU0sRUFBRTtnQ0FDTix3QkFBd0I7Z0NBQ3hCLHdCQUF3QjtnQ0FDeEIsMEJBQTBCOzZCQUMzQjs0QkFDRCxNQUFNLEVBQUUsT0FBTzs0QkFDZixRQUFRLEVBQUUsZUFBZTt5QkFDMUI7cUJBQ0Y7aUJBQ0Y7YUFDRixDQUFDLENBQUM7UUFDTCxDQUFDLENBQUM7UUFFRixJQUFJLENBQUMsbURBQW1ELEVBQUUsR0FBRyxFQUFFO1lBQzdELFFBQVE7WUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztZQUM5QixNQUFNLGVBQWUsR0FBRyxtQ0FBbUMsQ0FBQztZQUM1RCxNQUFNLFlBQVksR0FBRyxhQUFhLENBQUMsWUFBWSxDQUFDLG1CQUFtQixDQUFDLEtBQUssRUFBRSxjQUFjLEVBQUUsZUFBZSxDQUFDLENBQUM7WUFDNUcsTUFBTSxJQUFJLEdBQUcsSUFBSSxHQUFHLENBQUMsSUFBSSxDQUFDLEtBQUssRUFBRSxNQUFNLEVBQUU7Z0JBQ3ZDLFNBQVMsRUFBRSxJQUFJLEdBQUcsQ0FBQyxnQkFBZ0IsQ0FBQyxzQkFBc0IsQ0FBQzthQUM1RCxDQUFDLENBQUM7WUFFSCxPQUFPO1lBQ1AsWUFBWSxDQUFDLEtBQUssQ0FBQyxJQUFJLEVBQUUsc0JBQXNCLENBQUMsQ0FBQztZQUVqRCxPQUFPO1lBQ1AsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLGtCQUFrQixDQUFDLGtCQUFrQixFQUFFO2dCQUNuRCxjQUFjLEVBQUU7b0JBQ2QsU0FBUyxFQUFFO3dCQUNUOzRCQUNFLE1BQU0sRUFBRSxzQkFBc0I7NEJBQzlCLE1BQU0sRUFBRSxPQUFPOzRCQUNmLFFBQVEsRUFBRSxZQUFZLENBQUMsZUFBZTt5QkFDdkM7cUJBQ0Y7aUJBQ0Y7YUFDRixDQUFDLENBQUM7UUFDTCxDQUFDLENBQUM7UUFFRixJQUFJLENBQUMsa0VBQWtFLEVBQUUsR0FBRyxFQUFFO1lBQzVFLFFBQVE7WUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztZQUM5QixNQUFNLElBQUksR0FBRyxJQUFJLGFBQWEsQ0FBQyxJQUFJLENBQUMsS0FBSyxFQUFFLE1BQU0sRUFBRTtnQkFDakQsU0FBUyxFQUFFLEdBQUc7Z0JBQ2QsVUFBVSxFQUFFLFNBQVM7Z0JBQ3JCLFVBQVUsRUFBRTtvQkFDVixTQUFTLEVBQUUsR0FBRztvQkFDZCxnQkFBZ0IsRUFBRSxhQUFhO29CQUMvQixnQkFBZ0IsRUFBRSxHQUFHO29CQUNyQixpQkFBaUIsRUFBRSxJQUFJO29CQUN2QixlQUFlLEVBQUUsQ0FBQyxHQUFHLEVBQUUsR0FBRyxFQUFFLEdBQUcsQ0FBQztpQkFDakM7YUFDRixDQUFDLENBQUM7WUFFSCxPQUFPO1lBQ1AsTUFBTSxTQUFTLEdBQUcsSUFBSSxDQUFDLFdBQVcsRUFBRSxDQUFDO1lBRXJDLE9BQU87WUFDUCxNQUFNLENBQUMsU0FBUyxDQUFDLENBQUMsYUFBYSxDQUFDLEVBQUUsR0FBRyxFQUFFLElBQUk7Z0JBQ3pDLFNBQVMsRUFBRSxHQUFHO2dCQUNkLFVBQVUsRUFBRSxTQUFTO2dCQUNyQixVQUFVLEVBQ0gsRUFBRSxTQUFTLEVBQUUsR0FBRztvQkFDZCxnQkFBZ0IsRUFBRSxhQUFhO29CQUMvQixnQkFBZ0IsRUFBRSxHQUFHO29CQUNyQixpQkFBaUIsRUFBRSxJQUFJO29CQUN2QixlQUFlLEVBQUUsQ0FBRSxHQUFHLEVBQUUsR0FBRyxFQUFFLEdBQUcsQ0FBRSxFQUFFO2dCQUM3QyxJQUFJLEVBQUUsTUFBTTtnQkFDWixPQUFPLEVBQUUsU0FBUztnQkFDbEIsTUFBTSxFQUFFLFNBQVM7Z0JBQ2pCLFVBQVUsRUFBRSxTQUFTO2FBQ3RCLENBQUMsQ0FBQztRQUNMLENBQUMsQ0FBQztRQUVGLElBQUksQ0FBQyx1REFBdUQsRUFBRSxHQUFHLEVBQUU7WUFDakUsUUFBUTtZQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1lBQzlCLE1BQU0sSUFBSSxHQUFHLElBQUksYUFBYSxDQUFDLElBQUksQ0FBQyxLQUFLLEVBQUUsTUFBTSxFQUFFO2dCQUNqRCxVQUFVLEVBQUU7b0JBQ1YsS0FBSyxFQUFFLGFBQWEsQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLFdBQVcsQ0FBQztpQkFDaEQ7YUFDRixDQUFDLENBQUM7WUFFSCxPQUFPO1lBQ1AsTUFBTSxTQUFTLEdBQUcsSUFBSSxDQUFDLFdBQVcsRUFBRSxDQUFDO1lBRXJDLE9BQU87WUFDUCxNQUFNLENBQUMsU0FBUyxDQUFDLENBQUMsT0FBTyxDQUFDLEVBQUUsR0FBRyxFQUFFLElBQUk7Z0JBQ25DLFVBQVUsRUFDVixFQUFFLFNBQVMsRUFBRSxXQUFXLEVBQUM7Z0JBQ3pCLElBQUksRUFBRSxNQUFNO2FBQ2IsQ0FBQyxDQUFDO1FBQ0wsQ0FBQyxDQUFDO1FBRUYsSUFBSSxDQUFDLDJDQUEyQyxFQUFFLEdBQUcsRUFBRTtZQUNyRCxRQUFRO1lBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7WUFDOUIsTUFBTSxJQUFJLEdBQUcsSUFBSSxhQUFhLENBQUMsSUFBSSxDQUFDLEtBQUssRUFBRSxNQUFNLEVBQUU7Z0JBQ2pELElBQUksRUFBRTtvQkFDSixJQUFJLEVBQUUsR0FBRyxFQUFFLENBQUMsQ0FBQzt3QkFDWCxXQUFXLEVBQUUsVUFBVTt3QkFDdkIsZ0JBQWdCLEVBQUU7NEJBQ2hCLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQztnQ0FDdEIsU0FBUyxFQUFFLENBQUMsVUFBVSxDQUFDO2dDQUN2QixPQUFPLEVBQUUsQ0FBQyx1QkFBdUIsQ0FBQzs2QkFDbkMsQ0FBQzt5QkFDSDtxQkFDRixDQUFDO2lCQUNIO2FBQ0YsQ0FBQyxDQUFDO1lBQ0gsSUFBSSxhQUFhLENBQUMsWUFBWSxDQUFDLEtBQUssRUFBRSxjQUFjLEVBQUU7Z0JBQ3BELFVBQVUsRUFBRSxJQUFJO2FBQ2pCLENBQUMsQ0FBQztZQUVILE9BQU87WUFDUCxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsY0FBYyxDQUFDLGtDQUFrQyxFQUFFO2dCQUMvRCxTQUFTLEVBQUU7b0JBQ1QsdUNBQXVDO29CQUN2QywwQkFBMEI7aUJBQzNCO2FBQ0YsRUFBRSxxQkFBWSxDQUFDLGtCQUFrQixDQUFDLENBQUM7UUFDdEMsQ0FBQyxDQUFDLENBQUM7QUFFTCxDQUFDLENBQUMsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IGFycmF5V2l0aCwgb2JqZWN0TGlrZSwgUmVzb3VyY2VQYXJ0IH0gZnJvbSAnQGF3cy1jZGsvYXNzZXJ0JztcbmltcG9ydCAnQGF3cy1jZGsvYXNzZXJ0L2plc3QnO1xuaW1wb3J0ICogYXMgaWFtIGZyb20gJ0Bhd3MtY2RrL2F3cy1pYW0nO1xuaW1wb3J0ICogYXMgY2RrIGZyb20gJ0Bhd3MtY2RrL2NvcmUnO1xuaW1wb3J0ICogYXMgc3RlcGZ1bmN0aW9ucyBmcm9tICcuLi9saWInO1xuXG5kZXNjcmliZSgnU3RhdGUgTWFjaGluZSBSZXNvdXJjZXMnLCAoKSA9PiB7XG4gIHRlc3QoJ1Rhc2tzIGNhbiBhZGQgcGVybWlzc2lvbnMgdG8gdGhlIGV4ZWN1dGlvbiByb2xlJywgKCkgPT4ge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gICAgY29uc3QgdGFzayA9IG5ldyBzdGVwZnVuY3Rpb25zLlRhc2soc3RhY2ssICdUYXNrJywge1xuICAgICAgdGFzazoge1xuICAgICAgICBiaW5kOiAoKSA9PiAoe1xuICAgICAgICAgIHJlc291cmNlQXJuOiAncmVzb3VyY2UnLFxuICAgICAgICAgIHBvbGljeVN0YXRlbWVudHM6IFtuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgICAgICBhY3Rpb25zOiBbJ3Jlc291cmNlOkV2ZXJ5dGhpbmcnXSxcbiAgICAgICAgICAgIHJlc291cmNlczogWydyZXNvdXJjZSddLFxuICAgICAgICAgIH0pXSxcbiAgICAgICAgfSksXG4gICAgICB9LFxuICAgIH0pO1xuXG4gICAgLy8gV0hFTlxuICAgIG5ldyBzdGVwZnVuY3Rpb25zLlN0YXRlTWFjaGluZShzdGFjaywgJ1NNJywge1xuICAgICAgZGVmaW5pdGlvbjogdGFzayxcbiAgICB9KTtcblxuICAgIC8vIFRIRU5cbiAgICBleHBlY3Qoc3RhY2spLnRvSGF2ZVJlc291cmNlKCdBV1M6OklBTTo6UG9saWN5Jywge1xuICAgICAgUG9saWN5RG9jdW1lbnQ6IHtcbiAgICAgICAgVmVyc2lvbjogJzIwMTItMTAtMTcnLFxuICAgICAgICBTdGF0ZW1lbnQ6IFtcbiAgICAgICAgICB7XG4gICAgICAgICAgICBBY3Rpb246ICdyZXNvdXJjZTpFdmVyeXRoaW5nJyxcbiAgICAgICAgICAgIEVmZmVjdDogJ0FsbG93JyxcbiAgICAgICAgICAgIFJlc291cmNlOiAncmVzb3VyY2UnLFxuICAgICAgICAgIH0sXG4gICAgICAgIF0sXG4gICAgICB9LFxuICAgIH0pO1xuICB9KSxcblxuICB0ZXN0KCdUYXNrcyBoaWRkZW4gaW5zaWRlIGEgUGFyYWxsZWwgc3RhdGUgYXJlIGFsc28gaW5jbHVkZWQnLCAoKSA9PiB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcbiAgICBjb25zdCB0YXNrID0gbmV3IHN0ZXBmdW5jdGlvbnMuVGFzayhzdGFjaywgJ1Rhc2snLCB7XG4gICAgICB0YXNrOiB7XG4gICAgICAgIGJpbmQ6ICgpID0+ICh7XG4gICAgICAgICAgcmVzb3VyY2VBcm46ICdyZXNvdXJjZScsXG4gICAgICAgICAgcG9saWN5U3RhdGVtZW50czogW1xuICAgICAgICAgICAgbmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgICAgICAgICBhY3Rpb25zOiBbJ3Jlc291cmNlOkV2ZXJ5dGhpbmcnXSxcbiAgICAgICAgICAgICAgcmVzb3VyY2VzOiBbJ3Jlc291cmNlJ10sXG4gICAgICAgICAgICB9KSxcbiAgICAgICAgICBdLFxuICAgICAgICB9KSxcbiAgICAgIH0sXG4gICAgfSk7XG5cbiAgICBjb25zdCBwYXJhID0gbmV3IHN0ZXBmdW5jdGlvbnMuUGFyYWxsZWwoc3RhY2ssICdQYXJhJyk7XG4gICAgcGFyYS5icmFuY2godGFzayk7XG5cbiAgICAvLyBXSEVOXG4gICAgbmV3IHN0ZXBmdW5jdGlvbnMuU3RhdGVNYWNoaW5lKHN0YWNrLCAnU00nLCB7XG4gICAgICBkZWZpbml0aW9uOiBwYXJhLFxuICAgIH0pO1xuXG4gICAgLy8gVEhFTlxuICAgIGV4cGVjdChzdGFjaykudG9IYXZlUmVzb3VyY2UoJ0FXUzo6SUFNOjpQb2xpY3knLCB7XG4gICAgICBQb2xpY3lEb2N1bWVudDoge1xuICAgICAgICBWZXJzaW9uOiAnMjAxMi0xMC0xNycsXG4gICAgICAgIFN0YXRlbWVudDogW1xuICAgICAgICAgIHtcbiAgICAgICAgICAgIEFjdGlvbjogJ3Jlc291cmNlOkV2ZXJ5dGhpbmcnLFxuICAgICAgICAgICAgRWZmZWN0OiAnQWxsb3cnLFxuICAgICAgICAgICAgUmVzb3VyY2U6ICdyZXNvdXJjZScsXG4gICAgICAgICAgfSxcbiAgICAgICAgXSxcbiAgICAgIH0sXG4gICAgfSk7XG4gIH0pLFxuXG4gIHRlc3QoJ1Rhc2sgc2hvdWxkIHJlbmRlciBJbnB1dFBhdGggLyBQYXJhbWV0ZXJzIC8gT3V0cHV0UGF0aCBjb3JyZWN0bHknLCAoKSA9PiB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcbiAgICBjb25zdCB0YXNrID0gbmV3IHN0ZXBmdW5jdGlvbnMuVGFzayhzdGFjaywgJ1Rhc2snLCB7XG4gICAgICBpbnB1dFBhdGg6ICckJyxcbiAgICAgIG91dHB1dFBhdGg6ICckLnN0YXRlJyxcbiAgICAgIHRhc2s6IHtcbiAgICAgICAgYmluZDogKCkgPT4gKHtcbiAgICAgICAgICByZXNvdXJjZUFybjogJ3Jlc291cmNlJyxcbiAgICAgICAgICBwYXJhbWV0ZXJzOiB7XG4gICAgICAgICAgICAnaW5wdXQuJCc6ICckJyxcbiAgICAgICAgICAgICdzdHJpbmdBcmd1bWVudCc6ICdpbml0YWwtdGFzaycsXG4gICAgICAgICAgICAnbnVtYmVyQXJndW1lbnQnOiAxMjMsXG4gICAgICAgICAgICAnYm9vbGVhbkFyZ3VtZW50JzogdHJ1ZSxcbiAgICAgICAgICAgICdhcnJheUFyZ3VtZW50JzogWydhJywgJ2InLCAnYyddLFxuICAgICAgICAgIH0sXG4gICAgICAgIH0pLFxuICAgICAgfSxcbiAgICB9KTtcblxuICAgIC8vIFdIRU5cbiAgICBjb25zdCB0YXNrU3RhdGUgPSB0YXNrLnRvU3RhdGVKc29uKCk7XG5cbiAgICAvLyBUSEVOXG4gICAgZXhwZWN0KHRhc2tTdGF0ZSkudG9TdHJpY3RFcXVhbCh7IEVuZDogdHJ1ZSxcbiAgICAgIFJldHJ5OiB1bmRlZmluZWQsXG4gICAgICBDYXRjaDogdW5kZWZpbmVkLFxuICAgICAgSW5wdXRQYXRoOiAnJCcsXG4gICAgICBQYXJhbWV0ZXJzOlxuICAgICAgICAgICAgIHsgJ2lucHV0LiQnOiAnJCcsXG4gICAgICAgICAgICAgICAnc3RyaW5nQXJndW1lbnQnOiAnaW5pdGFsLXRhc2snLFxuICAgICAgICAgICAgICAgJ251bWJlckFyZ3VtZW50JzogMTIzLFxuICAgICAgICAgICAgICAgJ2Jvb2xlYW5Bcmd1bWVudCc6IHRydWUsXG4gICAgICAgICAgICAgICAnYXJyYXlBcmd1bWVudCc6IFsgJ2EnLCAnYicsICdjJyBdIH0sXG4gICAgICBPdXRwdXRQYXRoOiAnJC5zdGF0ZScsXG4gICAgICBUeXBlOiAnVGFzaycsXG4gICAgICBDb21tZW50OiB1bmRlZmluZWQsXG4gICAgICBSZXNvdXJjZTogJ3Jlc291cmNlJyxcbiAgICAgIFJlc3VsdFBhdGg6IHVuZGVmaW5lZCxcbiAgICAgIFRpbWVvdXRTZWNvbmRzOiB1bmRlZmluZWQsXG4gICAgICBIZWFydGJlYXRTZWNvbmRzOiB1bmRlZmluZWQsXG4gICAgfSk7XG4gIH0pLFxuXG4gIHRlc3QoJ1Rhc2sgY29tYmluZXMgdGFza29iamVjdCBwYXJhbWV0ZXJzIHdpdGggZGlyZWN0IHBhcmFtZXRlcnMnLCAoKSA9PiB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcbiAgICBjb25zdCB0YXNrID0gbmV3IHN0ZXBmdW5jdGlvbnMuVGFzayhzdGFjaywgJ1Rhc2snLCB7XG4gICAgICBpbnB1dFBhdGg6ICckJyxcbiAgICAgIG91dHB1dFBhdGg6ICckLnN0YXRlJyxcbiAgICAgIHRhc2s6IHtcbiAgICAgICAgYmluZDogKCkgPT4gKHtcbiAgICAgICAgICByZXNvdXJjZUFybjogJ3Jlc291cmNlJyxcbiAgICAgICAgICBwYXJhbWV0ZXJzOiB7XG4gICAgICAgICAgICBhOiAnYWEnLFxuICAgICAgICAgIH0sXG4gICAgICAgIH0pLFxuICAgICAgfSxcbiAgICAgIHBhcmFtZXRlcnM6IHtcbiAgICAgICAgYjogJ2JiJyxcbiAgICAgIH0sXG4gICAgfSk7XG5cbiAgICAvLyBXSEVOXG4gICAgY29uc3QgdGFza1N0YXRlID0gdGFzay50b1N0YXRlSnNvbigpO1xuXG4gICAgLy8gVEhFTlxuICAgIGV4cGVjdCh0YXNrU3RhdGUpLnRvU3RyaWN0RXF1YWwoeyBFbmQ6IHRydWUsXG4gICAgICBSZXRyeTogdW5kZWZpbmVkLFxuICAgICAgQ2F0Y2g6IHVuZGVmaW5lZCxcbiAgICAgIElucHV0UGF0aDogJyQnLFxuICAgICAgUGFyYW1ldGVyczpcbiAgICAgICAgICAgICB7IGE6ICdhYScsXG4gICAgICAgICAgICAgICBiOiAnYmInIH0sXG4gICAgICBPdXRwdXRQYXRoOiAnJC5zdGF0ZScsXG4gICAgICBUeXBlOiAnVGFzaycsXG4gICAgICBDb21tZW50OiB1bmRlZmluZWQsXG4gICAgICBSZXNvdXJjZTogJ3Jlc291cmNlJyxcbiAgICAgIFJlc3VsdFBhdGg6IHVuZGVmaW5lZCxcbiAgICAgIFRpbWVvdXRTZWNvbmRzOiB1bmRlZmluZWQsXG4gICAgICBIZWFydGJlYXRTZWNvbmRzOiB1bmRlZmluZWQsXG4gICAgfSk7XG4gIH0pLFxuXG4gIHRlc3QoJ0NyZWF0ZWQgc3RhdGUgbWFjaGluZSBjYW4gZ3JhbnQgc3RhcnQgZXhlY3V0aW9uIHRvIGEgcm9sZScsICgpID0+IHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuICAgIGNvbnN0IHRhc2sgPSBuZXcgc3RlcGZ1bmN0aW9ucy5UYXNrKHN0YWNrLCAnVGFzaycsIHtcbiAgICAgIHRhc2s6IHtcbiAgICAgICAgYmluZDogKCkgPT4gKHsgcmVzb3VyY2VBcm46ICdyZXNvdXJjZScgfSksXG4gICAgICB9LFxuICAgIH0pO1xuICAgIGNvbnN0IHN0YXRlTWFjaGluZSA9IG5ldyBzdGVwZnVuY3Rpb25zLlN0YXRlTWFjaGluZShzdGFjaywgJ1N0YXRlTWFjaGluZScsIHtcbiAgICAgIGRlZmluaXRpb246IHRhc2ssXG4gICAgfSk7XG4gICAgY29uc3Qgcm9sZSA9IG5ldyBpYW0uUm9sZShzdGFjaywgJ1JvbGUnLCB7XG4gICAgICBhc3N1bWVkQnk6IG5ldyBpYW0uU2VydmljZVByaW5jaXBhbCgnbGFtYmRhLmFtYXpvbmF3cy5jb20nKSxcbiAgICB9KTtcblxuICAgIC8vIFdIRU5cbiAgICBzdGF0ZU1hY2hpbmUuZ3JhbnRTdGFydEV4ZWN1dGlvbihyb2xlKTtcblxuICAgIC8vIFRIRU5cbiAgICBleHBlY3Qoc3RhY2spLnRvSGF2ZVJlc291cmNlTGlrZSgnQVdTOjpJQU06OlBvbGljeScsIHtcbiAgICAgIFBvbGljeURvY3VtZW50OiB7XG4gICAgICAgIFN0YXRlbWVudDogYXJyYXlXaXRoKG9iamVjdExpa2Uoe1xuICAgICAgICAgIEFjdGlvbjogJ3N0YXRlczpTdGFydEV4ZWN1dGlvbicsXG4gICAgICAgICAgRWZmZWN0OiAnQWxsb3cnLFxuICAgICAgICAgIFJlc291cmNlOiB7XG4gICAgICAgICAgICBSZWY6ICdTdGF0ZU1hY2hpbmUyRTAxQTNBNScsXG4gICAgICAgICAgfSxcbiAgICAgICAgfSkpLFxuICAgICAgfSxcbiAgICB9KTtcblxuICB9KSxcblxuICB0ZXN0KCdDcmVhdGVkIHN0YXRlIG1hY2hpbmUgY2FuIGdyYW50IHJlYWQgYWNjZXNzIHRvIGEgcm9sZScsICgpID0+IHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuICAgIGNvbnN0IHRhc2sgPSBuZXcgc3RlcGZ1bmN0aW9ucy5UYXNrKHN0YWNrLCAnVGFzaycsIHtcbiAgICAgIHRhc2s6IHtcbiAgICAgICAgYmluZDogKCkgPT4gKHsgcmVzb3VyY2VBcm46ICdyZXNvdXJjZScgfSksXG4gICAgICB9LFxuICAgIH0pO1xuICAgIGNvbnN0IHN0YXRlTWFjaGluZSA9IG5ldyBzdGVwZnVuY3Rpb25zLlN0YXRlTWFjaGluZShzdGFjaywgJ1N0YXRlTWFjaGluZScsIHtcbiAgICAgIGRlZmluaXRpb246IHRhc2ssXG4gICAgfSk7XG4gICAgY29uc3Qgcm9sZSA9IG5ldyBpYW0uUm9sZShzdGFjaywgJ1JvbGUnLCB7XG4gICAgICBhc3N1bWVkQnk6IG5ldyBpYW0uU2VydmljZVByaW5jaXBhbCgnbGFtYmRhLmFtYXpvbmF3cy5jb20nKSxcbiAgICB9KTtcblxuICAgIC8vIFdIRU5cbiAgICBzdGF0ZU1hY2hpbmUuZ3JhbnRSZWFkKHJvbGUpO1xuXG4gICAgLy8gVEhFTlxuICAgIGV4cGVjdChzdGFjaykudG9IYXZlUmVzb3VyY2VMaWtlKCdBV1M6OklBTTo6UG9saWN5Jywge1xuICAgICAgUG9saWN5RG9jdW1lbnQ6IHtcbiAgICAgICAgU3RhdGVtZW50OiBbXG4gICAgICAgICAge1xuICAgICAgICAgICAgQWN0aW9uOiBbXG4gICAgICAgICAgICAgICdzdGF0ZXM6TGlzdEV4ZWN1dGlvbnMnLFxuICAgICAgICAgICAgICAnc3RhdGVzOkxpc3RTdGF0ZU1hY2hpbmVzJyxcbiAgICAgICAgICAgIF0sXG4gICAgICAgICAgICBFZmZlY3Q6ICdBbGxvdycsXG4gICAgICAgICAgICBSZXNvdXJjZToge1xuICAgICAgICAgICAgICBSZWY6ICdTdGF0ZU1hY2hpbmUyRTAxQTNBNScsXG4gICAgICAgICAgICB9LFxuICAgICAgICAgIH0sXG4gICAgICAgICAge1xuICAgICAgICAgICAgQWN0aW9uOiBbXG4gICAgICAgICAgICAgICdzdGF0ZXM6RGVzY3JpYmVFeGVjdXRpb24nLFxuICAgICAgICAgICAgICAnc3RhdGVzOkRlc2NyaWJlU3RhdGVNYWNoaW5lRm9yRXhlY3V0aW9uJyxcbiAgICAgICAgICAgICAgJ3N0YXRlczpHZXRFeGVjdXRpb25IaXN0b3J5JyxcbiAgICAgICAgICAgIF0sXG4gICAgICAgICAgICBFZmZlY3Q6ICdBbGxvdycsXG4gICAgICAgICAgICBSZXNvdXJjZToge1xuICAgICAgICAgICAgICAnRm46OkpvaW4nOiBbXG4gICAgICAgICAgICAgICAgJycsXG4gICAgICAgICAgICAgICAgW1xuICAgICAgICAgICAgICAgICAgJ2FybjonLFxuICAgICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgICBSZWY6ICdBV1M6OlBhcnRpdGlvbicsXG4gICAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgICAgJzpzdGF0ZXM6JyxcbiAgICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgICAgUmVmOiAnQVdTOjpSZWdpb24nLFxuICAgICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICAgICc6JyxcbiAgICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgICAgUmVmOiAnQVdTOjpBY2NvdW50SWQnLFxuICAgICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICAgICc6ZXhlY3V0aW9uOicsXG4gICAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICAgICdGbjo6U2VsZWN0JzogW1xuICAgICAgICAgICAgICAgICAgICAgIDYsXG4gICAgICAgICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgICAgICAgJ0ZuOjpTcGxpdCc6IFtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgJzonLFxuICAgICAgICAgICAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgUmVmOiAnU3RhdGVNYWNoaW5lMkUwMUEzQTUnLFxuICAgICAgICAgICAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICAgICc6KicsXG4gICAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgfSxcbiAgICAgICAgICB7XG4gICAgICAgICAgICBBY3Rpb246IFtcbiAgICAgICAgICAgICAgJ3N0YXRlczpMaXN0QWN0aXZpdGllcycsXG4gICAgICAgICAgICAgICdzdGF0ZXM6RGVzY3JpYmVTdGF0ZU1hY2hpbmUnLFxuICAgICAgICAgICAgICAnc3RhdGVzOkRlc2NyaWJlQWN0aXZpdHknLFxuICAgICAgICAgICAgXSxcbiAgICAgICAgICAgIEVmZmVjdDogJ0FsbG93JyxcbiAgICAgICAgICAgIFJlc291cmNlOiAnKicsXG4gICAgICAgICAgfSxcbiAgICAgICAgXSxcbiAgICAgIH19LFxuICAgICk7XG5cbiAgfSksXG5cbiAgdGVzdCgnQ3JlYXRlZCBzdGF0ZSBtYWNoaW5lIGNhbiBncmFudCB0YXNrIHJlc3BvbnNlIGFjdGlvbnMgdG8gdGhlIHN0YXRlIG1hY2hpbmUnLCAoKSA9PiB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcbiAgICBjb25zdCB0YXNrID0gbmV3IHN0ZXBmdW5jdGlvbnMuVGFzayhzdGFjaywgJ1Rhc2snLCB7XG4gICAgICB0YXNrOiB7XG4gICAgICAgIGJpbmQ6ICgpID0+ICh7IHJlc291cmNlQXJuOiAncmVzb3VyY2UnIH0pLFxuICAgICAgfSxcbiAgICB9KTtcbiAgICBjb25zdCBzdGF0ZU1hY2hpbmUgPSBuZXcgc3RlcGZ1bmN0aW9ucy5TdGF0ZU1hY2hpbmUoc3RhY2ssICdTdGF0ZU1hY2hpbmUnLCB7XG4gICAgICBkZWZpbml0aW9uOiB0YXNrLFxuICAgIH0pO1xuICAgIGNvbnN0IHJvbGUgPSBuZXcgaWFtLlJvbGUoc3RhY2ssICdSb2xlJywge1xuICAgICAgYXNzdW1lZEJ5OiBuZXcgaWFtLlNlcnZpY2VQcmluY2lwYWwoJ2xhbWJkYS5hbWF6b25hd3MuY29tJyksXG4gICAgfSk7XG5cbiAgICAvLyBXSEVOXG4gICAgc3RhdGVNYWNoaW5lLmdyYW50VGFza1Jlc3BvbnNlKHJvbGUpO1xuXG4gICAgLy8gVEhFTlxuICAgIGV4cGVjdChzdGFjaykudG9IYXZlUmVzb3VyY2VMaWtlKCdBV1M6OklBTTo6UG9saWN5Jywge1xuICAgICAgUG9saWN5RG9jdW1lbnQ6IHtcbiAgICAgICAgU3RhdGVtZW50OiBbXG4gICAgICAgICAge1xuICAgICAgICAgICAgQWN0aW9uOiBbXG4gICAgICAgICAgICAgICdzdGF0ZXM6U2VuZFRhc2tTdWNjZXNzJyxcbiAgICAgICAgICAgICAgJ3N0YXRlczpTZW5kVGFza0ZhaWx1cmUnLFxuICAgICAgICAgICAgICAnc3RhdGVzOlNlbmRUYXNrSGVhcnRiZWF0JyxcbiAgICAgICAgICAgIF0sXG4gICAgICAgICAgICBFZmZlY3Q6ICdBbGxvdycsXG4gICAgICAgICAgICBSZXNvdXJjZToge1xuICAgICAgICAgICAgICBSZWY6ICdTdGF0ZU1hY2hpbmUyRTAxQTNBNScsXG4gICAgICAgICAgICB9LFxuICAgICAgICAgIH0sXG4gICAgICAgIF0sXG4gICAgICB9LFxuICAgIH0pO1xuICB9KSxcblxuICB0ZXN0KCdDcmVhdGVkIHN0YXRlIG1hY2hpbmUgY2FuIGdyYW50IGFjdGlvbnMgdG8gdGhlIGV4ZWN1dGlvbnMnLCAoKSA9PiB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcbiAgICBjb25zdCB0YXNrID0gbmV3IHN0ZXBmdW5jdGlvbnMuVGFzayhzdGFjaywgJ1Rhc2snLCB7XG4gICAgICB0YXNrOiB7XG4gICAgICAgIGJpbmQ6ICgpID0+ICh7IHJlc291cmNlQXJuOiAncmVzb3VyY2UnIH0pLFxuICAgICAgfSxcbiAgICB9KTtcbiAgICBjb25zdCBzdGF0ZU1hY2hpbmUgPSBuZXcgc3RlcGZ1bmN0aW9ucy5TdGF0ZU1hY2hpbmUoc3RhY2ssICdTdGF0ZU1hY2hpbmUnLCB7XG4gICAgICBkZWZpbml0aW9uOiB0YXNrLFxuICAgIH0pO1xuICAgIGNvbnN0IHJvbGUgPSBuZXcgaWFtLlJvbGUoc3RhY2ssICdSb2xlJywge1xuICAgICAgYXNzdW1lZEJ5OiBuZXcgaWFtLlNlcnZpY2VQcmluY2lwYWwoJ2xhbWJkYS5hbWF6b25hd3MuY29tJyksXG4gICAgfSk7XG5cbiAgICAvLyBXSEVOXG4gICAgc3RhdGVNYWNoaW5lLmdyYW50RXhlY3V0aW9uKHJvbGUsICdzdGF0ZXM6R2V0RXhlY3V0aW9uSGlzdG9yeScpO1xuXG4gICAgLy8gVEhFTlxuICAgIGV4cGVjdChzdGFjaykudG9IYXZlUmVzb3VyY2VMaWtlKCdBV1M6OklBTTo6UG9saWN5Jywge1xuICAgICAgUG9saWN5RG9jdW1lbnQ6IHtcbiAgICAgICAgU3RhdGVtZW50OiBbXG4gICAgICAgICAge1xuICAgICAgICAgICAgQWN0aW9uOiAnc3RhdGVzOkdldEV4ZWN1dGlvbkhpc3RvcnknLFxuICAgICAgICAgICAgRWZmZWN0OiAnQWxsb3cnLFxuICAgICAgICAgICAgUmVzb3VyY2U6IHtcbiAgICAgICAgICAgICAgJ0ZuOjpKb2luJzogW1xuICAgICAgICAgICAgICAgICcnLFxuICAgICAgICAgICAgICAgIFtcbiAgICAgICAgICAgICAgICAgICdhcm46JyxcbiAgICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgICAgUmVmOiAnQVdTOjpQYXJ0aXRpb24nLFxuICAgICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICAgICc6c3RhdGVzOicsXG4gICAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICAgIFJlZjogJ0FXUzo6UmVnaW9uJyxcbiAgICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgICAnOicsXG4gICAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICAgIFJlZjogJ0FXUzo6QWNjb3VudElkJyxcbiAgICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgICAnOmV4ZWN1dGlvbjonLFxuICAgICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgICAnRm46OlNlbGVjdCc6IFtcbiAgICAgICAgICAgICAgICAgICAgICA2LFxuICAgICAgICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICdGbjo6U3BsaXQnOiBbXG4gICAgICAgICAgICAgICAgICAgICAgICAgICc6JyxcbiAgICAgICAgICAgICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgIFJlZjogJ1N0YXRlTWFjaGluZTJFMDFBM0E1JyxcbiAgICAgICAgICAgICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgICAnOionLFxuICAgICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICB9LFxuICAgICAgICAgIH0sXG4gICAgICAgIF0sXG4gICAgICB9LFxuICAgIH0pO1xuICB9KSxcblxuICB0ZXN0KCdDcmVhdGVkIHN0YXRlIG1hY2hpbmUgY2FuIGdyYW50IGFjdGlvbnMgdG8gYSByb2xlJywgKCkgPT4ge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gICAgY29uc3QgdGFzayA9IG5ldyBzdGVwZnVuY3Rpb25zLlRhc2soc3RhY2ssICdUYXNrJywge1xuICAgICAgdGFzazoge1xuICAgICAgICBiaW5kOiAoKSA9PiAoeyByZXNvdXJjZUFybjogJ3Jlc291cmNlJyB9KSxcbiAgICAgIH0sXG4gICAgfSk7XG4gICAgY29uc3Qgc3RhdGVNYWNoaW5lID0gbmV3IHN0ZXBmdW5jdGlvbnMuU3RhdGVNYWNoaW5lKHN0YWNrLCAnU3RhdGVNYWNoaW5lJywge1xuICAgICAgZGVmaW5pdGlvbjogdGFzayxcbiAgICB9KTtcbiAgICBjb25zdCByb2xlID0gbmV3IGlhbS5Sb2xlKHN0YWNrLCAnUm9sZScsIHtcbiAgICAgIGFzc3VtZWRCeTogbmV3IGlhbS5TZXJ2aWNlUHJpbmNpcGFsKCdsYW1iZGEuYW1hem9uYXdzLmNvbScpLFxuICAgIH0pO1xuXG4gICAgLy8gV0hFTlxuICAgIHN0YXRlTWFjaGluZS5ncmFudChyb2xlLCAnc3RhdGVzOkxpc3RFeGVjdXRpb24nKTtcblxuICAgIC8vIFRIRU5cbiAgICBleHBlY3Qoc3RhY2spLnRvSGF2ZVJlc291cmNlTGlrZSgnQVdTOjpJQU06OlBvbGljeScsIHtcbiAgICAgIFBvbGljeURvY3VtZW50OiB7XG4gICAgICAgIFN0YXRlbWVudDogW1xuICAgICAgICAgIHtcbiAgICAgICAgICAgIEFjdGlvbjogJ3N0YXRlczpMaXN0RXhlY3V0aW9uJyxcbiAgICAgICAgICAgIEVmZmVjdDogJ0FsbG93JyxcbiAgICAgICAgICAgIFJlc291cmNlOiB7XG4gICAgICAgICAgICAgIFJlZjogJ1N0YXRlTWFjaGluZTJFMDFBM0E1JyxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgfSxcbiAgICAgICAgXSxcbiAgICAgIH0sXG4gICAgfSk7XG5cbiAgfSksXG5cbiAgdGVzdCgnSW1wb3J0ZWQgc3RhdGUgbWFjaGluZSBjYW4gZ3JhbnQgc3RhcnQgZXhlY3V0aW9uIHRvIGEgcm9sZScsICgpID0+IHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuICAgIGNvbnN0IHN0YXRlTWFjaGluZUFybiA9ICdhcm46YXdzOnN0YXRlczo6Om15LXN0YXRlLW1hY2hpbmUnO1xuICAgIGNvbnN0IHN0YXRlTWFjaGluZSA9IHN0ZXBmdW5jdGlvbnMuU3RhdGVNYWNoaW5lLmZyb21TdGF0ZU1hY2hpbmVBcm4oc3RhY2ssICdTdGF0ZU1hY2hpbmUnLCBzdGF0ZU1hY2hpbmVBcm4pO1xuICAgIGNvbnN0IHJvbGUgPSBuZXcgaWFtLlJvbGUoc3RhY2ssICdSb2xlJywge1xuICAgICAgYXNzdW1lZEJ5OiBuZXcgaWFtLlNlcnZpY2VQcmluY2lwYWwoJ2xhbWJkYS5hbWF6b25hd3MuY29tJyksXG4gICAgfSk7XG5cbiAgICAvLyBXSEVOXG4gICAgc3RhdGVNYWNoaW5lLmdyYW50U3RhcnRFeGVjdXRpb24ocm9sZSk7XG5cbiAgICAvLyBUSEVOXG4gICAgZXhwZWN0KHN0YWNrKS50b0hhdmVSZXNvdXJjZSgnQVdTOjpJQU06OlBvbGljeScsIHtcbiAgICAgIFBvbGljeURvY3VtZW50OiB7XG4gICAgICAgIFN0YXRlbWVudDogW1xuICAgICAgICAgIHtcbiAgICAgICAgICAgIEFjdGlvbjogJ3N0YXRlczpTdGFydEV4ZWN1dGlvbicsXG4gICAgICAgICAgICBFZmZlY3Q6ICdBbGxvdycsXG4gICAgICAgICAgICBSZXNvdXJjZTogc3RhdGVNYWNoaW5lQXJuLFxuICAgICAgICAgIH0sXG4gICAgICAgIF0sXG4gICAgICAgIFZlcnNpb246ICcyMDEyLTEwLTE3JyxcbiAgICAgIH0sXG4gICAgICBQb2xpY3lOYW1lOiAnUm9sZURlZmF1bHRQb2xpY3k1RkZCN0RBQicsXG4gICAgICBSb2xlczogW1xuICAgICAgICB7XG4gICAgICAgICAgUmVmOiAnUm9sZTFBQkNDNUYwJyxcbiAgICAgICAgfSxcbiAgICAgIF0sXG4gICAgfSk7XG4gIH0pLFxuXG4gIHRlc3QoJ0ltcG9ydGVkIHN0YXRlIG1hY2hpbmUgY2FuIGdyYW50IHJlYWQgYWNjZXNzIHRvIGEgcm9sZScsICgpID0+IHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuICAgIGNvbnN0IHN0YXRlTWFjaGluZUFybiA9ICdhcm46YXdzOnN0YXRlczo6Om15LXN0YXRlLW1hY2hpbmUnO1xuICAgIGNvbnN0IHN0YXRlTWFjaGluZSA9IHN0ZXBmdW5jdGlvbnMuU3RhdGVNYWNoaW5lLmZyb21TdGF0ZU1hY2hpbmVBcm4oc3RhY2ssICdTdGF0ZU1hY2hpbmUnLCBzdGF0ZU1hY2hpbmVBcm4pO1xuICAgIGNvbnN0IHJvbGUgPSBuZXcgaWFtLlJvbGUoc3RhY2ssICdSb2xlJywge1xuICAgICAgYXNzdW1lZEJ5OiBuZXcgaWFtLlNlcnZpY2VQcmluY2lwYWwoJ2xhbWJkYS5hbWF6b25hd3MuY29tJyksXG4gICAgfSk7XG5cbiAgICAvLyBXSEVOXG4gICAgc3RhdGVNYWNoaW5lLmdyYW50UmVhZChyb2xlKTtcblxuICAgIC8vIFRIRU5cbiAgICBleHBlY3Qoc3RhY2spLnRvSGF2ZVJlc291cmNlTGlrZSgnQVdTOjpJQU06OlBvbGljeScsIHtcbiAgICAgIFBvbGljeURvY3VtZW50OiB7XG4gICAgICAgIFN0YXRlbWVudDogW1xuICAgICAgICAgIHtcbiAgICAgICAgICAgIEFjdGlvbjogW1xuICAgICAgICAgICAgICAnc3RhdGVzOkxpc3RFeGVjdXRpb25zJyxcbiAgICAgICAgICAgICAgJ3N0YXRlczpMaXN0U3RhdGVNYWNoaW5lcycsXG4gICAgICAgICAgICBdLFxuICAgICAgICAgICAgRWZmZWN0OiAnQWxsb3cnLFxuICAgICAgICAgICAgUmVzb3VyY2U6IHN0YXRlTWFjaGluZUFybixcbiAgICAgICAgICB9LFxuICAgICAgICAgIHtcbiAgICAgICAgICAgIEFjdGlvbjogW1xuICAgICAgICAgICAgICAnc3RhdGVzOkRlc2NyaWJlRXhlY3V0aW9uJyxcbiAgICAgICAgICAgICAgJ3N0YXRlczpEZXNjcmliZVN0YXRlTWFjaGluZUZvckV4ZWN1dGlvbicsXG4gICAgICAgICAgICAgICdzdGF0ZXM6R2V0RXhlY3V0aW9uSGlzdG9yeScsXG4gICAgICAgICAgICBdLFxuICAgICAgICAgICAgRWZmZWN0OiAnQWxsb3cnLFxuICAgICAgICAgICAgUmVzb3VyY2U6IHtcbiAgICAgICAgICAgICAgJ0ZuOjpKb2luJzogW1xuICAgICAgICAgICAgICAgICcnLFxuICAgICAgICAgICAgICAgIFtcbiAgICAgICAgICAgICAgICAgICdhcm46JyxcbiAgICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgICAgUmVmOiAnQVdTOjpQYXJ0aXRpb24nLFxuICAgICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICAgICc6c3RhdGVzOicsXG4gICAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICAgIFJlZjogJ0FXUzo6UmVnaW9uJyxcbiAgICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgICAnOicsXG4gICAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICAgIFJlZjogJ0FXUzo6QWNjb3VudElkJyxcbiAgICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgICAnOmV4ZWN1dGlvbjoqJyxcbiAgICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICB9LFxuICAgICAgICAgIHtcbiAgICAgICAgICAgIEFjdGlvbjogW1xuICAgICAgICAgICAgICAnc3RhdGVzOkxpc3RBY3Rpdml0aWVzJyxcbiAgICAgICAgICAgICAgJ3N0YXRlczpEZXNjcmliZVN0YXRlTWFjaGluZScsXG4gICAgICAgICAgICAgICdzdGF0ZXM6RGVzY3JpYmVBY3Rpdml0eScsXG4gICAgICAgICAgICBdLFxuICAgICAgICAgICAgRWZmZWN0OiAnQWxsb3cnLFxuICAgICAgICAgICAgUmVzb3VyY2U6ICcqJyxcbiAgICAgICAgICB9LFxuICAgICAgICBdLFxuICAgICAgfX0sXG4gICAgKTtcbiAgfSksXG5cbiAgdGVzdCgnSW1wb3J0ZWQgc3RhdGUgbWFjaGluZSBjYW4gdGFzayByZXNwb25zZSBwZXJtaXNzaW9ucyB0byB0aGUgc3RhdGUgbWFjaGluZScsICgpID0+IHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuICAgIGNvbnN0IHN0YXRlTWFjaGluZUFybiA9ICdhcm46YXdzOnN0YXRlczo6Om15LXN0YXRlLW1hY2hpbmUnO1xuICAgIGNvbnN0IHN0YXRlTWFjaGluZSA9IHN0ZXBmdW5jdGlvbnMuU3RhdGVNYWNoaW5lLmZyb21TdGF0ZU1hY2hpbmVBcm4oc3RhY2ssICdTdGF0ZU1hY2hpbmUnLCBzdGF0ZU1hY2hpbmVBcm4pO1xuICAgIGNvbnN0IHJvbGUgPSBuZXcgaWFtLlJvbGUoc3RhY2ssICdSb2xlJywge1xuICAgICAgYXNzdW1lZEJ5OiBuZXcgaWFtLlNlcnZpY2VQcmluY2lwYWwoJ2xhbWJkYS5hbWF6b25hd3MuY29tJyksXG4gICAgfSk7XG5cbiAgICAvLyBXSEVOXG4gICAgc3RhdGVNYWNoaW5lLmdyYW50VGFza1Jlc3BvbnNlKHJvbGUpO1xuXG4gICAgLy8gVEhFTlxuICAgIGV4cGVjdChzdGFjaykudG9IYXZlUmVzb3VyY2VMaWtlKCdBV1M6OklBTTo6UG9saWN5Jywge1xuICAgICAgUG9saWN5RG9jdW1lbnQ6IHtcbiAgICAgICAgU3RhdGVtZW50OiBbXG4gICAgICAgICAge1xuICAgICAgICAgICAgQWN0aW9uOiBbXG4gICAgICAgICAgICAgICdzdGF0ZXM6U2VuZFRhc2tTdWNjZXNzJyxcbiAgICAgICAgICAgICAgJ3N0YXRlczpTZW5kVGFza0ZhaWx1cmUnLFxuICAgICAgICAgICAgICAnc3RhdGVzOlNlbmRUYXNrSGVhcnRiZWF0JyxcbiAgICAgICAgICAgIF0sXG4gICAgICAgICAgICBFZmZlY3Q6ICdBbGxvdycsXG4gICAgICAgICAgICBSZXNvdXJjZTogc3RhdGVNYWNoaW5lQXJuLFxuICAgICAgICAgIH0sXG4gICAgICAgIF0sXG4gICAgICB9LFxuICAgIH0pO1xuICB9KSxcblxuICB0ZXN0KCdJbXBvcnRlZCBzdGF0ZSBtYWNoaW5lIGNhbiBncmFudCBhY2Nlc3MgdG8gYSByb2xlJywgKCkgPT4ge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gICAgY29uc3Qgc3RhdGVNYWNoaW5lQXJuID0gJ2Fybjphd3M6c3RhdGVzOjo6bXktc3RhdGUtbWFjaGluZSc7XG4gICAgY29uc3Qgc3RhdGVNYWNoaW5lID0gc3RlcGZ1bmN0aW9ucy5TdGF0ZU1hY2hpbmUuZnJvbVN0YXRlTWFjaGluZUFybihzdGFjaywgJ1N0YXRlTWFjaGluZScsIHN0YXRlTWFjaGluZUFybik7XG4gICAgY29uc3Qgcm9sZSA9IG5ldyBpYW0uUm9sZShzdGFjaywgJ1JvbGUnLCB7XG4gICAgICBhc3N1bWVkQnk6IG5ldyBpYW0uU2VydmljZVByaW5jaXBhbCgnbGFtYmRhLmFtYXpvbmF3cy5jb20nKSxcbiAgICB9KTtcblxuICAgIC8vIFdIRU5cbiAgICBzdGF0ZU1hY2hpbmUuZ3JhbnQocm9sZSwgJ3N0YXRlczpMaXN0RXhlY3V0aW9uJyk7XG5cbiAgICAvLyBUSEVOXG4gICAgZXhwZWN0KHN0YWNrKS50b0hhdmVSZXNvdXJjZUxpa2UoJ0FXUzo6SUFNOjpQb2xpY3knLCB7XG4gICAgICBQb2xpY3lEb2N1bWVudDoge1xuICAgICAgICBTdGF0ZW1lbnQ6IFtcbiAgICAgICAgICB7XG4gICAgICAgICAgICBBY3Rpb246ICdzdGF0ZXM6TGlzdEV4ZWN1dGlvbicsXG4gICAgICAgICAgICBFZmZlY3Q6ICdBbGxvdycsXG4gICAgICAgICAgICBSZXNvdXJjZTogc3RhdGVNYWNoaW5lLnN0YXRlTWFjaGluZUFybixcbiAgICAgICAgICB9LFxuICAgICAgICBdLFxuICAgICAgfSxcbiAgICB9KTtcbiAgfSksXG5cbiAgdGVzdCgnUGFzcyBzaG91bGQgcmVuZGVyIElucHV0UGF0aCAvIFBhcmFtZXRlcnMgLyBPdXRwdXRQYXRoIGNvcnJlY3RseScsICgpID0+IHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuICAgIGNvbnN0IHRhc2sgPSBuZXcgc3RlcGZ1bmN0aW9ucy5QYXNzKHN0YWNrLCAnUGFzcycsIHtcbiAgICAgIGlucHV0UGF0aDogJyQnLFxuICAgICAgb3V0cHV0UGF0aDogJyQuc3RhdGUnLFxuICAgICAgcGFyYW1ldGVyczoge1xuICAgICAgICAnaW5wdXQuJCc6ICckJyxcbiAgICAgICAgJ3N0cmluZ0FyZ3VtZW50JzogJ2luaXRhbC10YXNrJyxcbiAgICAgICAgJ251bWJlckFyZ3VtZW50JzogMTIzLFxuICAgICAgICAnYm9vbGVhbkFyZ3VtZW50JzogdHJ1ZSxcbiAgICAgICAgJ2FycmF5QXJndW1lbnQnOiBbJ2EnLCAnYicsICdjJ10sXG4gICAgICB9LFxuICAgIH0pO1xuXG4gICAgLy8gV0hFTlxuICAgIGNvbnN0IHRhc2tTdGF0ZSA9IHRhc2sudG9TdGF0ZUpzb24oKTtcblxuICAgIC8vIFRIRU5cbiAgICBleHBlY3QodGFza1N0YXRlKS50b1N0cmljdEVxdWFsKHsgRW5kOiB0cnVlLFxuICAgICAgSW5wdXRQYXRoOiAnJCcsXG4gICAgICBPdXRwdXRQYXRoOiAnJC5zdGF0ZScsXG4gICAgICBQYXJhbWV0ZXJzOlxuICAgICAgICAgICAgIHsgJ2lucHV0LiQnOiAnJCcsXG4gICAgICAgICAgICAgICAnc3RyaW5nQXJndW1lbnQnOiAnaW5pdGFsLXRhc2snLFxuICAgICAgICAgICAgICAgJ251bWJlckFyZ3VtZW50JzogMTIzLFxuICAgICAgICAgICAgICAgJ2Jvb2xlYW5Bcmd1bWVudCc6IHRydWUsXG4gICAgICAgICAgICAgICAnYXJyYXlBcmd1bWVudCc6IFsgJ2EnLCAnYicsICdjJyBdIH0sXG4gICAgICBUeXBlOiAnUGFzcycsXG4gICAgICBDb21tZW50OiB1bmRlZmluZWQsXG4gICAgICBSZXN1bHQ6IHVuZGVmaW5lZCxcbiAgICAgIFJlc3VsdFBhdGg6IHVuZGVmaW5lZCxcbiAgICB9KTtcbiAgfSksXG5cbiAgdGVzdCgncGFyYW1ldGVycyBjYW4gYmUgc2VsZWN0ZWQgZnJvbSB0aGUgaW5wdXQgd2l0aCBhIHBhdGgnLCAoKSA9PiB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcbiAgICBjb25zdCB0YXNrID0gbmV3IHN0ZXBmdW5jdGlvbnMuUGFzcyhzdGFjaywgJ1Bhc3MnLCB7XG4gICAgICBwYXJhbWV0ZXJzOiB7XG4gICAgICAgIGlucHV0OiBzdGVwZnVuY3Rpb25zLkRhdGEuc3RyaW5nQXQoJyQubXlGaWVsZCcpLFxuICAgICAgfSxcbiAgICB9KTtcblxuICAgIC8vIFdIRU5cbiAgICBjb25zdCB0YXNrU3RhdGUgPSB0YXNrLnRvU3RhdGVKc29uKCk7XG5cbiAgICAvLyBUSEVOXG4gICAgZXhwZWN0KHRhc2tTdGF0ZSkudG9FcXVhbCh7IEVuZDogdHJ1ZSxcbiAgICAgIFBhcmFtZXRlcnM6XG4gICAgICB7ICdpbnB1dC4kJzogJyQubXlGaWVsZCd9LFxuICAgICAgVHlwZTogJ1Bhc3MnLFxuICAgIH0pO1xuICB9KSxcblxuICB0ZXN0KCdTdGF0ZSBtYWNoaW5lcyBtdXN0IGRlcGVuZCBvbiB0aGVpciByb2xlcycsICgpID0+IHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuICAgIGNvbnN0IHRhc2sgPSBuZXcgc3RlcGZ1bmN0aW9ucy5UYXNrKHN0YWNrLCAnVGFzaycsIHtcbiAgICAgIHRhc2s6IHtcbiAgICAgICAgYmluZDogKCkgPT4gKHtcbiAgICAgICAgICByZXNvdXJjZUFybjogJ3Jlc291cmNlJyxcbiAgICAgICAgICBwb2xpY3lTdGF0ZW1lbnRzOiBbXG4gICAgICAgICAgICBuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgICAgICAgIHJlc291cmNlczogWydyZXNvdXJjZSddLFxuICAgICAgICAgICAgICBhY3Rpb25zOiBbJ2xhbWJkYTpJbnZva2VGdW5jdGlvbiddLFxuICAgICAgICAgICAgfSksXG4gICAgICAgICAgXSxcbiAgICAgICAgfSksXG4gICAgICB9LFxuICAgIH0pO1xuICAgIG5ldyBzdGVwZnVuY3Rpb25zLlN0YXRlTWFjaGluZShzdGFjaywgJ1N0YXRlTWFjaGluZScsIHtcbiAgICAgIGRlZmluaXRpb246IHRhc2ssXG4gICAgfSk7XG5cbiAgICAvLyBUSEVOXG4gICAgZXhwZWN0KHN0YWNrKS50b0hhdmVSZXNvdXJjZSgnQVdTOjpTdGVwRnVuY3Rpb25zOjpTdGF0ZU1hY2hpbmUnLCB7XG4gICAgICBEZXBlbmRzT246IFtcbiAgICAgICAgJ1N0YXRlTWFjaGluZVJvbGVEZWZhdWx0UG9saWN5REYxRTY2MDcnLFxuICAgICAgICAnU3RhdGVNYWNoaW5lUm9sZUI4NDA0MzFEJyxcbiAgICAgIF0sXG4gICAgfSwgUmVzb3VyY2VQYXJ0LkNvbXBsZXRlRGVmaW5pdGlvbik7XG4gIH0pO1xuXG59KTtcbiJdfQ==