"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CodeBuildImageBuilder = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const crypto = require("crypto");
const cdk = require("aws-cdk-lib");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_codebuild_1 = require("aws-cdk-lib/aws-codebuild");
const aws_ecr_1 = require("aws-cdk-lib/aws-ecr");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const constructs_1 = require("constructs");
const utils_1 = require("../../utils");
const common_1 = require("../common");
/**
 * An image builder that uses CodeBuild to build Docker images pre-baked with all the GitHub Actions runner requirements. Builders can be used with runner providers.
 *
 * Each builder re-runs automatically at a set interval to make sure the images contain the latest versions of everything.
 *
 * You can create an instance of this construct to customize the image used to spin-up runners. Each provider has its own requirements for what an image should do. That's why they each provide their own Dockerfile.
 *
 * For example, to set a specific runner version, rebuild the image every 2 weeks, and add a few packages for the Fargate provider, use:
 *
 * ```
 * const builder = new CodeBuildImageBuilder(this, 'Builder', {
 *     dockerfilePath: FargateProvider.LINUX_X64_DOCKERFILE_PATH,
 *     runnerVersion: RunnerVersion.specific('2.293.0'),
 *     rebuildInterval: Duration.days(14),
 * });
 * builder.setBuildArg('EXTRA_PACKAGES', 'nginx xz-utils');
 * new FargateRunner(this, 'Fargate provider', {
 *     label: 'customized-fargate',
 *     imageBuilder: builder,
 * });
 * ```
 */
class CodeBuildImageBuilder extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.props = props;
        this.preBuild = [];
        this.postBuild = [];
        this.buildArgs = new Map();
        this.policyStatements = [];
        this.secondaryAssets = new Map();
        if (props.subnetSelection?.subnetType == aws_cdk_lib_1.aws_ec2.SubnetType.PRIVATE_ISOLATED) {
            aws_cdk_lib_1.Annotations.of(this).addWarning('Private isolated subnets cannot pull from public ECR and VPC endpoint is not supported yet. ' +
                'See https://github.com/aws/containers-roadmap/issues/1160');
        }
        // set platform
        this.architecture = props.architecture ?? common_1.Architecture.X86_64;
        this.os = props.os ?? common_1.Os.LINUX;
        // create repository that only keeps one tag
        this.repository = new aws_cdk_lib_1.aws_ecr.Repository(this, 'Repository', {
            imageScanOnPush: true,
            imageTagMutability: aws_ecr_1.TagMutability.MUTABLE,
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
            lifecycleRules: [
                {
                    description: 'Remove untagged images that have been replaced by CodeBuild',
                    tagStatus: aws_ecr_1.TagStatus.UNTAGGED,
                    maxImageAge: aws_cdk_lib_1.Duration.days(1),
                },
            ],
        });
        // upload Dockerfile to S3 as an asset
        this.dockerfile = new aws_cdk_lib_1.aws_s3_assets.Asset(this, 'Dockerfile', {
            path: props.dockerfilePath,
        });
        // choose build image
        this.buildImage = this.getBuildImage();
    }
    /**
     * Uploads a folder to the build server at a given folder name.
     *
     * @param sourcePath path to source directory
     * @param destName name of destination folder
     */
    addFiles(sourcePath, destName) {
        if (this.boundImage) {
            throw new Error('Image is already bound. Use this method before passing the builder to a runner provider.');
        }
        const asset = new aws_cdk_lib_1.aws_s3_assets.Asset(this, destName, { path: sourcePath });
        this.secondaryAssets.set(destName, asset);
        this.preBuild.push(`rm -rf "${destName}" && cp -r "$CODEBUILD_SRC_DIR_${destName}" "${destName}"`); // symlinks don't work with docker
    }
    /**
     * Adds a command that runs before `docker build`.
     *
     * @param command command to add
     */
    addPreBuildCommand(command) {
        if (this.boundImage) {
            throw new Error('Image is already bound. Use this method before passing the builder to a runner provider.');
        }
        this.preBuild.push(command);
    }
    /**
     * Adds a command that runs after `docker build` and `docker push`.
     *
     * @param command command to add
     */
    addPostBuildCommand(command) {
        if (this.boundImage) {
            throw new Error('Image is already bound. Use this method before passing the builder to a runner provider.');
        }
        this.postBuild.push(command);
    }
    /**
     * Adds a build argument for Docker. See the documentation for the Dockerfile you're using for a list of supported build arguments.
     *
     * @param name build argument name
     * @param value build argument value
     */
    setBuildArg(name, value) {
        if (this.boundImage) {
            throw new Error('Image is already bound. Use this method before passing the builder to a runner provider.');
        }
        this.buildArgs.set(name, value);
    }
    /**
     * Add a policy statement to the builder to access resources required to the image build.
     *
     * @param statement IAM policy statement
     */
    addPolicyStatement(statement) {
        if (this.boundImage) {
            throw new Error('Image is already bound. Use this method before passing the builder to a runner provider.');
        }
        this.policyStatements.push(statement);
    }
    /**
     * Add extra trusted certificates. This helps deal with self-signed certificates for GitHub Enterprise Server.
     *
     * All first party Dockerfiles support this. Others may not.
     *
     * @param path path to directory containing a file called certs.pem containing all the required certificates
     */
    addExtraCertificates(path) {
        if (this.boundImage) {
            throw new Error('Image is already bound. Use this method before passing the builder to a runner provider.');
        }
        this.addFiles(path, 'extra_certs');
    }
    /**
     * Called by IRunnerProvider to finalize settings and create the image builder.
     */
    bind() {
        if (this.boundImage) {
            return this.boundImage;
        }
        // log group for the image builds
        const logGroup = new aws_cdk_lib_1.aws_logs.LogGroup(this, 'Logs', {
            retention: this.props.logRetention ?? aws_logs_1.RetentionDays.ONE_MONTH,
            removalPolicy: this.props.logRemovalPolicy ?? aws_cdk_lib_1.RemovalPolicy.DESTROY,
        });
        // generate buildSpec
        const buildSpec = this.getBuildSpec(this.repository, logGroup, this.props.runnerVersion);
        // create CodeBuild project that builds Dockerfile and pushes to repository
        const project = new aws_cdk_lib_1.aws_codebuild.Project(this, 'CodeBuild', {
            description: `Build docker image for self-hosted GitHub runner ${this.node.path} (${this.os.name}/${this.architecture.name})`,
            buildSpec: aws_cdk_lib_1.aws_codebuild.BuildSpec.fromObject(buildSpec),
            source: aws_cdk_lib_1.aws_codebuild.Source.s3({
                bucket: this.dockerfile.bucket,
                path: this.dockerfile.s3ObjectKey,
            }),
            vpc: this.props.vpc,
            securityGroups: this.props.securityGroup ? [this.props.securityGroup] : undefined,
            subnetSelection: this.props.subnetSelection,
            timeout: this.props.timeout ?? aws_cdk_lib_1.Duration.hours(1),
            environment: {
                buildImage: this.buildImage,
                computeType: this.props.computeType ?? aws_codebuild_1.ComputeType.SMALL,
                privileged: true,
            },
            logging: {
                cloudWatch: {
                    logGroup,
                },
            },
        });
        // permissions
        this.repository.grantPullPush(project);
        this.policyStatements.forEach(project.addToRolePolicy);
        // call CodeBuild during deployment and delete all images from repository during destruction
        const cr = this.customResource(project);
        // rebuild image on a schedule
        this.rebuildImageOnSchedule(project, this.props.rebuildInterval);
        for (const [assetPath, asset] of this.secondaryAssets.entries()) {
            project.addSecondarySource(aws_cdk_lib_1.aws_codebuild.Source.s3({
                identifier: assetPath,
                bucket: asset.bucket,
                path: asset.s3ObjectKey,
            }));
        }
        this.boundImage = {
            imageRepository: aws_cdk_lib_1.aws_ecr.Repository.fromRepositoryAttributes(this, 'Dependable Image', {
                // There are simpler ways to get name and ARN, but we want an image object that depends on the custom resource.
                // We want whoever is using this image to automatically wait for CodeBuild to start and finish through the custom resource.
                repositoryName: cr.getAttString('Name'),
                repositoryArn: cr.ref,
            }),
            imageTag: 'latest',
            architecture: this.architecture,
            os: this.os,
            logGroup,
        };
        return this.boundImage;
    }
    getBuildImage() {
        if (this.os.is(common_1.Os.LINUX)) {
            if (this.architecture.is(common_1.Architecture.X86_64)) {
                return aws_cdk_lib_1.aws_codebuild.LinuxBuildImage.STANDARD_4_0;
            }
            else if (this.architecture.is(common_1.Architecture.ARM64)) {
                return aws_cdk_lib_1.aws_codebuild.LinuxArmBuildImage.AMAZON_LINUX_2_STANDARD_2_0;
            }
        }
        if (this.os.is(common_1.Os.WINDOWS)) {
            throw new Error('CodeBuild cannot be used to build Windows Docker images https://github.com/docker-library/docker/issues/49');
        }
        throw new Error(`Unable to find CodeBuild image for ${this.os.name}/${this.architecture.name}`);
    }
    getBuildSpec(repository, logGroup, runnerVersion) {
        // don't forget to change BUILDSPEC_VERSION when the buildSpec changes, and you want to trigger a rebuild on deploy
        let buildArgs = '';
        for (const [name, value] of this.buildArgs.entries()) {
            buildArgs += ` --build-arg "${name}"="${value}"`;
        }
        buildArgs += ` --build-arg RUNNER_VERSION="${runnerVersion ? runnerVersion.version : common_1.RunnerVersion.latest().version}"`;
        return {
            version: '0.2',
            env: {
                variables: {
                    REPO_ARN: repository.repositoryArn,
                    REPO_URI: repository.repositoryUri,
                    STACK_ID: 'unspecified',
                    REQUEST_ID: 'unspecified',
                    LOGICAL_RESOURCE_ID: 'unspecified',
                    RESPONSE_URL: 'unspecified',
                    RUNNER_VERSION: runnerVersion ? runnerVersion.version : common_1.RunnerVersion.latest().version,
                },
            },
            phases: {
                pre_build: {
                    commands: this.preBuild.concat([
                        'mkdir -p extra_certs',
                        '$(aws ecr get-login --no-include-email --region "$AWS_DEFAULT_REGION")',
                    ]),
                },
                build: {
                    commands: [
                        `docker build . -t "$REPO_URI" ${buildArgs}`,
                        'docker push "$REPO_URI"',
                    ],
                },
                post_build: {
                    commands: this.postBuild.concat([
                        'STATUS="SUCCESS"',
                        'if [ $CODEBUILD_BUILD_SUCCEEDING -ne 1 ]; then STATUS="FAILED"; fi',
                        'cat <<EOF > /tmp/payload.json\n' +
                            '{\n' +
                            '  "StackId": "$STACK_ID",\n' +
                            '  "RequestId": "$REQUEST_ID",\n' +
                            '  "LogicalResourceId": "$LOGICAL_RESOURCE_ID",\n' +
                            '  "PhysicalResourceId": "$REPO_ARN",\n' +
                            '  "Status": "$STATUS",\n' +
                            `  "Reason": "See logs in ${logGroup.logGroupName}/$CODEBUILD_LOG_PATH (deploy again with \'cdk deploy -R\' or logRemovalPolicy=RemovalPolicy.RETAIN if they are already deleted)",\n` +
                            `  "Data": {"Name": "${repository.repositoryName}"}\n` +
                            '}\n' +
                            'EOF',
                        'if [ "$RESPONSE_URL" != "unspecified" ]; then jq . /tmp/payload.json; curl -fsSL -X PUT -H "Content-Type:" -d "@/tmp/payload.json" "$RESPONSE_URL"; fi',
                    ]),
                },
            },
        };
    }
    customResource(project) {
        const crHandler = utils_1.BundledNodejsFunction.singleton(this, 'build-image', {
            description: 'Custom resource handler that triggers CodeBuild to build runner images, and cleans-up images on deletion',
            timeout: cdk.Duration.minutes(3),
        });
        const policy = new aws_cdk_lib_1.aws_iam.Policy(this, 'CR Policy', {
            statements: [
                new aws_cdk_lib_1.aws_iam.PolicyStatement({
                    actions: ['codebuild:StartBuild'],
                    resources: [project.projectArn],
                }),
                new aws_cdk_lib_1.aws_iam.PolicyStatement({
                    actions: ['ecr:BatchDeleteImage', 'ecr:ListImages'],
                    resources: [this.repository.repositoryArn],
                }),
            ],
        });
        crHandler.role?.attachInlinePolicy(policy);
        const cr = new aws_cdk_lib_1.CustomResource(this, 'Builder', {
            serviceToken: crHandler.functionArn,
            resourceType: 'Custom::ImageBuilder',
            properties: {
                RepoName: this.repository.repositoryName,
                ProjectName: project.projectName,
                // We include a hash so the image is built immediately on changes, and we don't have to wait for its scheduled build.
                // This also helps make sure the changes are good. If they have a bug, the deployment will fail instead of just the scheduled build.
                BuildHash: this.hashBuildSettings(),
            },
        });
        // add dependencies to make sure resources are there when we need them
        cr.node.addDependency(project);
        cr.node.addDependency(policy);
        cr.node.addDependency(crHandler);
        return cr;
    }
    /**
     * Return hash of all settings that can affect the result image so we can trigger the build when it changes.
     * @private
     */
    hashBuildSettings() {
        // main Dockerfile
        let components = [this.dockerfile.assetHash];
        // all additional files
        for (const [name, asset] of this.secondaryAssets.entries()) {
            components.push(name);
            components.push(asset.assetHash);
        }
        // buildspec.yml version
        components.push(`v${CodeBuildImageBuilder.BUILDSPEC_VERSION}`);
        // runner version
        components.push(this.props.runnerVersion?.version ?? common_1.RunnerVersion.latest().version);
        // user commands
        components = components.concat(this.preBuild);
        components = components.concat(this.postBuild);
        for (const [name, value] of this.buildArgs.entries()) {
            components.push(name);
            components.push(value);
        }
        // hash it
        const all = components.join('-');
        return crypto.createHash('md5').update(all).digest('hex');
    }
    rebuildImageOnSchedule(project, rebuildInterval) {
        rebuildInterval = rebuildInterval ?? aws_cdk_lib_1.Duration.days(7);
        if (rebuildInterval.toMilliseconds() != 0) {
            const scheduleRule = new aws_cdk_lib_1.aws_events.Rule(this, 'Build Schedule', {
                description: `Rebuild runner image for ${this.repository.repositoryName}`,
                schedule: aws_cdk_lib_1.aws_events.Schedule.rate(rebuildInterval),
            });
            scheduleRule.addTarget(new aws_cdk_lib_1.aws_events_targets.CodeBuildProject(project));
        }
    }
}
exports.CodeBuildImageBuilder = CodeBuildImageBuilder;
_a = JSII_RTTI_SYMBOL_1;
CodeBuildImageBuilder[_a] = { fqn: "@cloudsnorkel/cdk-github-runners.CodeBuildImageBuilder", version: "0.5.4" };
/**
 * Bump this number every time the buildspec or any important setting of the project changes. It will force a rebuild of the image.
 * @private
 */
CodeBuildImageBuilder.BUILDSPEC_VERSION = 2;
//# sourceMappingURL=data:application/json;base64,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