import attr
import logging
import os
import toml
from typing import List, Optional

from telegram_rss.utils import get_default_directory as _get_default_directory
from telegram_rss.utils import save_as

logger = logging.getLogger(__name__)


@attr.dataclass
class FeedConfig:
    name: str
    source: str
    save_bandwith: bool = True
    footer: bool = True
    footer_name: Optional[str] = None
    footer_link: Optional[str] = None
    etag: Optional[str] = None
    modified: Optional[str] = None
    web_page_preview: Optional[bool] = None
    read_more_button: Optional[str] = None
    message_delay: Optional[float] = None
    cache: Optional[str] = None

    def __str__(self):
        return self.name


@attr.dataclass
class Config:
    bot_token: str = ""
    env_token: str = "TOKEN"
    web_page_preview: bool = True
    message_delay: float = 0.05
    author_text: str = "Author"
    read_more_button: str = "Read more..."
    channel_text: str = "Channel"
    config_dir: str = _get_default_directory()
    data_dir: str = _get_default_directory("data")
    users: List[int] = attr.ib(factory=list)
    channels: List[int] = attr.ib(factory=list)
    feeds: List[FeedConfig] = attr.ib(factory=list)

    def __attrs_post_init__(self) -> None:
        new_rss: List[FeedConfig] = list()
        for rss in self.feeds:
            if isinstance(rss, FeedConfig):
                new_rss.append(rss)
            elif isinstance(rss, dict):
                new_rss.append(FeedConfig(**rss))
        self.feeds = new_rss
        logger.debug(f"Loaded config {self}")

    @property
    def token(self) -> str:
        if self.bot_token:
            return self.bot_token
        elif self.env_token:
            return os.environ.get(self.env_token, "")
        else:
            raise ValueError(
                f"either bot_token or env_token must be filled in the config ({self.config_dir})"
            )

    get_default_directory = staticmethod(_get_default_directory)

    @classmethod
    def exist(cls, directory: Optional[str] = None) -> bool:
        filepath = cls._config_file(directory)
        return os.path.isfile(filepath)

    @classmethod
    def read(cls, directory: Optional[str] = None) -> "Config":
        config_file = cls._config_file(directory)
        if not os.path.isfile(config_file):
            cls._create(config_file)
            print(f"Please edit the configuration file at {config_file}")
            exit()
        configs = toml.load(config_file)
        configs["config_dir"] = config_file
        return cls(**configs)  # type: ignore

    @classmethod
    def _create(cls, directory: Optional[str] = None) -> "Config":
        config_file = cls._config_file(directory)
        config = Config(config_dir=config_file)
        configs = attr.asdict(config, recurse=True)
        save_as(configs, config_file)
        return config

    @classmethod
    def _config_file(cls, directory: Optional[str] = None, name="config.toml") -> str:
        if directory and not directory.endswith(".toml"):
            directory = os.path.join(directory, name)
        return directory or os.path.join(cls.get_default_directory(), name)

    def save(self, directory: Optional[str] = None):
        configs = attr.asdict(self, recurse=True)
        config_file = self._config_file(directory)
        save_as(configs, config_file)

    def __del__(self):
        return self.save(self.config_dir)
