# AUTOGENERATED! DO NOT EDIT! File to edit: ../00_ipyleaflet.ipynb.

# %% auto 0
__all__ = ['DEFAULT_MAP_HEIGHT', 'SCALE_LEVEL_0', 'TileLayerEE', 'Map', 'Inspector']

# %% ../00_ipyleaflet.ipynb 4
import ee
import ipyleaflet
import ipytree
import ipywidgets as widgets
import traitlets

# %% ../00_ipyleaflet.ipynb 6
DEFAULT_MAP_HEIGHT = '400px'

class TileLayerEE(ipyleaflet.TileLayer):
  """Class for a tile layer generated by Earth Engine.
  
  Attributes:
    ee_object: An Earth Engine object.
  """
  def __init__(self,
               ee_object:ee.Image, # An Earth Engine Image object
               *args,
               **kwargs):

    self.ee_object = ee_object

    super(TileLayerEE, self).__init__(*args, **kwargs)


class Map(ipyleaflet.Map):
  """An interactive map class for Jupyter clients.
  
  Attributes:
    layers_control: a boolean indicating whether to display a layers control.
  """

  layers_control = traitlets.Bool(True)

  def __init__(self, *args, **kwargs):

    self.layers_control_instance = None

    # Set default values for the map.
    if 'zoom' not in kwargs:
      kwargs['zoom'] = 4
    
    if 'basemap' not in kwargs:
      kwargs['basemap'] = ipyleaflet.basemap_to_tiles(ipyleaflet.basemaps.Stamen.Watercolor)

    if 'height' not in kwargs:
      kwargs['height'] = DEFAULT_MAP_HEIGHT
    
    super(Map, self).__init__(*args, **kwargs)
    
    if self.layers_control:
      self.layers_control_instance = ipyleaflet.LayersControl(position='topright')
      self.add_control(self.layers_control_instance)
        
    self.default_style.cursor = 'crosshair'
        
  def addLayer(self, eeObject, visParams={}, name=None, shown=True, opacity=1):
    """Adds a layer for an Earth Engine object."""
    
    if name is None:
      # Provide a default name for the layer in the form "Layer ##"
      name = f'Layer {len(self.layers)}'

    def get_tile_layer_url(ee_image_object):
      map_id_dict = ee.Image(ee_image_object).getMapId()
      return map_id_dict['tile_fetcher'].url_format

    # Assume that the eeObject is an ee.Image.
    # TODO: Generalize this to other EE objects.
    ee_image = eeObject

    tile_url = get_tile_layer_url(
      ee_image.visualize(**visParams)
    )
    self.add_layer(TileLayerEE(ee_object=eeObject, url=tile_url, name=name, visible=shown))

# %% ../00_ipyleaflet.ipynb 14
# Map scale at Level 0 in meters/pixel
SCALE_LEVEL_0 = 156543.03392

class Inspector(ipytree.Tree):
    """Class representing an inspector tool that responds to map events."""
    
    def __init__(self,
                 map_object=None, # An Earth Engine Image object
                 *args,
                 **kwargs):

        point_folder = ipytree.Node('Point', icon='map')
        pixels_folder = ipytree.Node('Pixels', icon='archive')
        objects_folder = ipytree.Node('Objects', icon='archive')
        
        self.map_object = map_object
        self.layout.width = '40%'
        self.layout.max_height = '400px'
        self.layout.border = 'solid'
        self.layout.overflow = 'scroll'

        super(Inspector, self).__init__(
            nodes=[point_folder, pixels_folder, objects_folder],
            *args, 
            **kwargs)

        if map_object:
            self.set_map(map_object)
            
        self.update_inspector()
    
    @property
    def point_node(self):
        return self.nodes[0]
    
    @point_node.setter
    def point_node(self, new_point_node):
        #(lat, lon) = new_coords
        _temp_nodes = list(self.nodes)
        _temp_nodes[0] = new_point_node
        self.nodes = _temp_nodes
    
    @property
    def pixels_node(self):
        return self.nodes[1]
    
    @property
    def objects_node(self):
        return self.nodes[2]
    
    def update_inspector(self, coords=None):
        """Update information in the inspector tree."""
            
        def _order_items(item_dict, ordering_list):
            """Orders dictionary items in a specified order."""
            list_of_tuples = [(key, item_dict[key]) for key in [x for x in ordering_list if x in item_dict.keys()]]
            return dict(list_of_tuples)  
    
        def _process_info(info):
            node_list = []    
            if isinstance(info, list):
                for count, item in enumerate(info):
                    if isinstance(item, (list, dict)):
                        node_list.append(ipytree.Node(f'{count}', nodes = _process_info(item), opened=False))
                    else:
                        node_list.append(ipytree.Node(f'{count}: {item}'))
            elif isinstance(info, dict):
                for k,v in info.items():
                    if isinstance(v, (list, dict)):
                        node_list.append(ipytree.Node(f'{k}', nodes = _process_info(v), opened=False))
                    else:
                        node_list.append(ipytree.Node(f'{k}: {v}'))
            else:
                node_list.append(ipytree.Node(f'{info}'))
            return node_list
    
        # Disable the Pixels and Objects folders if the map does not have any
        # layers. This assumes the map has a single basemap layer.
        if self.map_object:
            if len(self.map_object.layers) > 1:
                self.pixels_node.disabled = False
                self.objects_node.disabled = False
            else:
                self.pixels_node.disabled = True
                self.objects_node.disabled = True
        else:
            self.pixels_node.disabled = True
            self.objects_node.disabled = True
        
        if coords:
            (lat, lon) = coords
            
            # Clear the inspector folders before recalculating outputs.
            self.point_node.nodes = []
            self.pixels_node.nodes = []
            self.objects_node.nodes = []
            
            # Update the Point folder
            point_nodes = [
                ipytree.Node(f'Longitude: {lon:.6f}'),
                ipytree.Node(f'Latitutde: {lat:.6f}'),
                ipytree.Node(
                    f'Zoom Level: {self.map_object.zoom:.0f}'
                ),
                ipytree.Node(
                    f'Scale (approx. m/px): '
                    f'{SCALE_LEVEL_0 / 2**self.map_object.zoom:.2f}'
                )
            ]
            _point_node = ipytree.Node(f'Point ({lon:.2f}, {lat:.2f})', nodes=point_nodes)
            self.point_node = _point_node

            
            # Update the Pixels folder
            pixel_nodes = []
            for layer in self.map_object.layers:
                if not layer.base:
                    ee_type = ee.Algorithms.ObjectType(layer.ee_object).getInfo()

                    if ee_type == 'Image':
                        value_dict = layer.ee_object.reduceRegion(
                                reducer=ee.Reducer.mean(),
                                geometry=ee.Geometry.Point(lon, lat),
                                scale=30,
                                bestEffort=True
                            ).getInfo()
                        num_bands = len(value_dict.keys())

                        layer_node = ipytree.Node(f'{layer.name}: Image ({num_bands} bands)')

                        has_unmasked_pixel = False
                        for bandname in layer.ee_object.bandNames().getInfo():          
                            if value_dict[bandname] is not None:
                                has_unmasked_pixel = True
                            layer_node.add_node(
                              ipytree.Node(f'{bandname}: {value_dict[bandname]}')
                            )
                    
                        if not has_unmasked_pixel:
                            layer_node.nodes = [
                                ipytree.Node(f'No unmasked pixels at clicked point.'),
                            ] 
                        pixel_nodes.append(layer_node)        
            self.pixels_node.nodes = pixel_nodes
            
            # Update the Objects folder
            object_nodes = []
            for layer in self.map_object.layers:
                if not layer.base:
                    
                    ee_type = ee.Algorithms.ObjectType(layer.ee_object).getInfo()                    
                    layer_info = layer.ee_object.getInfo()
                    
                    # Order the layer information.
                    ordering_list = ['type', 'id', 'version', 'bands', 'properties']
                    layer_info = _order_items(layer_info, ordering_list)
                    
                    layer_node = ipytree.Node(f'{layer.name}: {ee_type} ({len(layer_info["bands"])} bands)', nodes=_process_info(layer_info))
                    
                    object_nodes.append(layer_node)
                    
            self.objects_node.nodes = object_nodes
                
    
    def register_map(self, map_object):
        def handle_interaction(type, event, coordinates):
            if type == 'click':
                self.update_inspector(coordinates)
        map_object.on_interaction(handle_interaction)
            
    def set_map(self, map_object):
        self.map_object = map_object
        self.register_map(map_object)

    def get_map(self):
        return self.map_object
