'''
# AWS Lambda Construct Library

<!--BEGIN STABILITY BANNER-->---


![cfn-resources: Stable](https://img.shields.io/badge/cfn--resources-stable-success.svg?style=for-the-badge)

![cdk-constructs: Stable](https://img.shields.io/badge/cdk--constructs-stable-success.svg?style=for-the-badge)

---
<!--END STABILITY BANNER-->

This construct library allows you to define AWS Lambda Functions.

```python
fn = lambda_.Function(self, "MyFunction",
    runtime=lambda_.Runtime.NODEJS_16_X,
    handler="index.handler",
    code=lambda_.Code.from_asset(path.join(__dirname, "lambda-handler"))
)
```

## Handler Code

The `lambda.Code` class includes static convenience methods for various types of
runtime code.

* `lambda.Code.fromBucket(bucket, key[, objectVersion])` - specify an S3 object
  that contains the archive of your runtime code.
* `lambda.Code.fromInline(code)` - inline the handle code as a string. This is
  limited to supported runtimes and the code cannot exceed 4KiB.
* `lambda.Code.fromAsset(path)` - specify a directory or a .zip file in the local
  filesystem which will be zipped and uploaded to S3 before deployment. See also
  [bundling asset code](#bundling-asset-code).
* `lambda.Code.fromDockerBuild(path, options)` - use the result of a Docker
  build as code. The runtime code is expected to be located at `/asset` in the
  image and will be zipped and uploaded to S3 as an asset.

The following example shows how to define a Python function and deploy the code
from the local directory `my-lambda-handler` to it:

```python
lambda_.Function(self, "MyLambda",
    code=lambda_.Code.from_asset(path.join(__dirname, "my-lambda-handler")),
    handler="index.main",
    runtime=lambda_.Runtime.PYTHON_3_9
)
```

When deploying a stack that contains this code, the directory will be zip
archived and then uploaded to an S3 bucket, then the exact location of the S3
objects will be passed when the stack is deployed.

During synthesis, the CDK expects to find a directory on disk at the asset
directory specified. Note that we are referencing the asset directory relatively
to our CDK project directory. This is especially important when we want to share
this construct through a library. Different programming languages will have
different techniques for bundling resources into libraries.

## Docker Images

Lambda functions allow specifying their handlers within docker images. The docker
image can be an image from ECR or a local asset that the CDK will package and load
into ECR.

The following `DockerImageFunction` construct uses a local folder with a
Dockerfile as the asset that will be used as the function handler.

```python
lambda_.DockerImageFunction(self, "AssetFunction",
    code=lambda_.DockerImageCode.from_image_asset(path.join(__dirname, "docker-handler"))
)
```

You can also specify an image that already exists in ECR as the function handler.

```python
import aws_cdk.aws_ecr as ecr

repo = ecr.Repository(self, "Repository")

lambda_.DockerImageFunction(self, "ECRFunction",
    code=lambda_.DockerImageCode.from_ecr(repo)
)
```

The props for these docker image resources allow overriding the image's `CMD`, `ENTRYPOINT`, and `WORKDIR`
configurations as well as choosing a specific tag or digest. See their docs for more information.

## Execution Role

Lambda functions assume an IAM role during execution. In CDK by default, Lambda
functions will use an autogenerated Role if one is not provided.

The autogenerated Role is automatically given permissions to execute the Lambda
function. To reference the autogenerated Role:

```python
fn = lambda_.Function(self, "MyFunction",
    runtime=lambda_.Runtime.NODEJS_16_X,
    handler="index.handler",
    code=lambda_.Code.from_asset(path.join(__dirname, "lambda-handler"))
)

role = fn.role
```

You can also provide your own IAM role. Provided IAM roles will not automatically
be given permissions to execute the Lambda function. To provide a role and grant
it appropriate permissions:

```python
my_role = iam.Role(self, "My Role",
    assumed_by=iam.ServicePrincipal("sns.amazonaws.com")
)

fn = lambda_.Function(self, "MyFunction",
    runtime=lambda_.Runtime.NODEJS_16_X,
    handler="index.handler",
    code=lambda_.Code.from_asset(path.join(__dirname, "lambda-handler")),
    role=my_role
)

my_role.add_managed_policy(iam.ManagedPolicy.from_aws_managed_policy_name("service-role/AWSLambdaBasicExecutionRole"))
my_role.add_managed_policy(iam.ManagedPolicy.from_aws_managed_policy_name("service-role/AWSLambdaVPCAccessExecutionRole"))
```

## Function Timeout

AWS Lambda functions have a default timeout of 3 seconds, but this can be increased
up to 15 minutes. The timeout is available as a property of `Function` so that
you can reference it elsewhere in your stack. For instance, you could use it to create
a CloudWatch alarm to report when your function timed out:

```python
import aws_cdk.core as cdk
import aws_cdk.aws_cloudwatch as cloudwatch


fn = lambda_.Function(self, "MyFunction",
    runtime=lambda_.Runtime.NODEJS_16_X,
    handler="index.handler",
    code=lambda_.Code.from_asset(path.join(__dirname, "lambda-handler")),
    timeout=cdk.Duration.minutes(5)
)

if fn.timeout:
    cloudwatch.Alarm(self, "MyAlarm",
        metric=fn.metric_duration().with(
            statistic="Maximum"
        ),
        evaluation_periods=1,
        datapoints_to_alarm=1,
        threshold=fn.timeout.to_milliseconds(),
        treat_missing_data=cloudwatch.TreatMissingData.IGNORE,
        alarm_name="My Lambda Timeout"
    )
```

## Resource-based Policies

AWS Lambda supports resource-based policies for controlling access to Lambda
functions and layers on a per-resource basis. In particular, this allows you to
give permission to AWS services and other AWS accounts to modify and invoke your
functions. You can also restrict permissions given to AWS services by providing
a source account or ARN (representing the account and identifier of the resource
that accesses the function or layer).

```python
# fn: lambda.Function

principal = iam.ServicePrincipal("my-service")

fn.grant_invoke(principal)

# Equivalent to:
fn.add_permission("my-service Invocation",
    principal=principal
)
```

For more information, see [Resource-based
policies](https://docs.aws.amazon.com/lambda/latest/dg/access-control-resource-based.html)
in the AWS Lambda Developer Guide.

Providing an unowned principal (such as account principals, generic ARN
principals, service principals, and principals in other accounts) to a call to
`fn.grantInvoke` will result in a resource-based policy being created. If the
principal in question has conditions limiting the source account or ARN of the
operation (see above), these conditions will be automatically added to the
resource policy.

```python
# fn: lambda.Function

service_principal = iam.ServicePrincipal("my-service")
source_arn = "arn:aws:s3:::my-bucket"
source_account = "111122223333"
service_principal_with_conditions = service_principal.with_conditions({
    "ArnLike": {
        "aws:SourceArn": source_arn
    },
    "StringEquals": {
        "aws:SourceAccount": source_account
    }
})

fn.grant_invoke(service_principal_with_conditions)

# Equivalent to:
fn.add_permission("my-service Invocation",
    principal=service_principal,
    source_arn=source_arn,
    source_account=source_account
)
```

## Versions

You can use
[versions](https://docs.aws.amazon.com/lambda/latest/dg/configuration-versions.html)
to manage the deployment of your AWS Lambda functions. For example, you can
publish a new version of a function for beta testing without affecting users of
the stable production version.

The function version includes the following information:

* The function code and all associated dependencies.
* The Lambda runtime that executes the function.
* All of the function settings, including the environment variables.
* A unique Amazon Resource Name (ARN) to identify this version of the function.

You could create a version to your lambda function using the `Version` construct.

```python
# fn: lambda.Function

version = lambda_.Version(self, "MyVersion",
    lambda_=fn
)
```

The major caveat to know here is that a function version must always point to a
specific 'version' of the function. When the function is modified, the version
will continue to point to the 'then version' of the function.

One way to ensure that the `lambda.Version` always points to the latest version
of your `lambda.Function` is to set an environment variable which changes at
least as often as your code does. This makes sure the function always has the
latest code. For instance -

```python
code_version = "stringOrMethodToGetCodeVersion"
fn = lambda_.Function(self, "MyFunction",
    runtime=lambda_.Runtime.NODEJS_16_X,
    handler="index.handler",
    code=lambda_.Code.from_asset(path.join(__dirname, "lambda-handler")),
    environment={
        "CodeVersionString": code_version
    }
)
```

The `fn.latestVersion` property returns a `lambda.IVersion` which represents
the `$LATEST` pseudo-version.

However, most AWS services require a specific AWS Lambda version,
and won't allow you to use `$LATEST`. Therefore, you would normally want
to use `lambda.currentVersion`.

The `fn.currentVersion` property can be used to obtain a `lambda.Version`
resource that represents the AWS Lambda function defined in your application.
Any change to your function's code or configuration will result in the creation
of a new version resource. You can specify options for this version through the
`currentVersionOptions` property.

NOTE: The `currentVersion` property is only supported when your AWS Lambda function
uses either `lambda.Code.fromAsset` or `lambda.Code.fromInline`. Other types
of code providers (such as `lambda.Code.fromBucket`) require that you define a
`lambda.Version` resource directly since the CDK is unable to determine if
their contents had changed.

### `currentVersion`: Updated hashing logic

To produce a new lambda version each time the lambda function is modified, the
`currentVersion` property under the hood, computes a new logical id based on the
properties of the function. This informs CloudFormation that a new
`AWS::Lambda::Version` resource should be created pointing to the updated Lambda
function.

However, a bug was introduced in this calculation that caused the logical id to
change when it was not required (ex: when the Function's `Tags` property, or
when the `DependsOn` clause was modified). This caused the deployment to fail
since the Lambda service does not allow creating duplicate versions.

This has been fixed in the AWS CDK but *existing* users need to opt-in via a
[feature flag](https://docs.aws.amazon.com/cdk/latest/guide/featureflags.html). Users who have run `cdk init` since this fix will be opted in,
by default.

Otherwise, you will need to enable the [feature flag](https://docs.aws.amazon.com/cdk/latest/guide/featureflags.html)
`@aws-cdk/aws-lambda:recognizeVersionProps`. Since CloudFormation does not
allow duplicate versions, you will also need to make some modification to
your function so that a new version can be created. To efficiently and trivially
modify all your lambda functions at once, you can attach the
`FunctionVersionUpgrade` aspect to the stack, which slightly alters the
function description. This aspect is intended for one-time use to upgrade the
version of all your functions at the same time, and can safely be removed after
deploying once.

```python
stack = Stack()
Aspects.of(stack).add(lambda_.FunctionVersionUpgrade(LAMBDA_RECOGNIZE_VERSION_PROPS))
```

When the new logic is in effect, you may rarely come across the following error:
`The following properties are not recognized as version properties`. This will
occur, typically when [property overrides](https://docs.aws.amazon.com/cdk/latest/guide/cfn_layer.html#cfn_layer_raw) are used, when a new property
introduced in `AWS::Lambda::Function` is used that CDK is still unaware of.

To overcome this error, use the API `Function.classifyVersionProperty()` to
record whether a new version should be generated when this property is changed.
This can be typically determined by checking whether the property can be
modified using the *[UpdateFunctionConfiguration](https://docs.aws.amazon.com/lambda/latest/dg/API_UpdateFunctionConfiguration.html)* API or not.

### `currentVersion`: Updated hashing logic for layer versions

An additional update to the hashing logic fixes two issues surrounding layers.
Prior to this change, updating the lambda layer version would have no effect on
the function version. Also, the order of lambda layers provided to the function
was unnecessarily baked into the hash.

This has been fixed in the AWS CDK starting with version 2.27. If you ran
`cdk init` with an earlier version, you will need to opt-in via a [feature flag](https://docs.aws.amazon.com/cdk/latest/guide/featureflags.html).
If you run `cdk init` with v2.27 or later, this fix will be opted in, by default.

Existing users will need to enable the [feature flag](https://docs.aws.amazon.com/cdk/latest/guide/featureflags.html)
`@aws-cdk/aws-lambda:recognizeLayerVersion`. Since CloudFormation does not
allow duplicate versions, they will also need to make some modification to
their function so that a new version can be created. To efficiently and trivially
modify all your lambda functions at once, users can attach the
`FunctionVersionUpgrade` aspect to the stack, which slightly alters the
function description. This aspect is intended for one-time use to upgrade the
version of all your functions at the same time, and can safely be removed after
deploying once.

```python
stack = Stack()
Aspects.of(stack).add(lambda_.FunctionVersionUpgrade(LAMBDA_RECOGNIZE_LAYER_VERSION))
```

## Aliases

You can define one or more
[aliases](https://docs.aws.amazon.com/lambda/latest/dg/configuration-aliases.html)
for your AWS Lambda function. A Lambda alias is like a pointer to a specific
Lambda function version. Users can access the function version using the alias
ARN.

The `version.addAlias()` method can be used to define an AWS Lambda alias that
points to a specific version.

The following example defines an alias named `live` which will always point to a
version that represents the function as defined in your CDK app. When you change
your lambda code or configuration, a new resource will be created. You can
specify options for the current version through the `currentVersionOptions`
property.

```python
fn = lambda_.Function(self, "MyFunction",
    current_version_options=lambda.VersionOptions(
        removal_policy=RemovalPolicy.RETAIN,  # retain old versions
        retry_attempts=1
    ),
    runtime=lambda_.Runtime.NODEJS_16_X,
    handler="index.handler",
    code=lambda_.Code.from_asset(path.join(__dirname, "lambda-handler"))
)

fn.add_alias("live")
```

## Function URL

A function URL is a dedicated HTTP(S) endpoint for your Lambda function. When you create a function URL, Lambda automatically generates a unique URL endpoint for you. Function URLs can be created for the latest version Lambda Functions, or Function Aliases (but not for Versions).

Function URLs are dual stack-enabled, supporting IPv4 and IPv6, and cross-origin resource sharing (CORS) configuration. After you configure a function URL for your function, you can invoke your function through its HTTP(S) endpoint via a web browser, curl, Postman, or any HTTP client. To invoke a function using IAM authentication your HTTP client must support SigV4 signing.

See the [Invoking Function URLs](https://docs.aws.amazon.com/lambda/latest/dg/urls-invocation.html) section of the AWS Lambda Developer Guide
for more information on the input and output payloads of Functions invoked in this way.

### IAM-authenticated Function URLs

To create a Function URL which can be called by an IAM identity, call `addFunctionUrl()`, followed by `grantInvokeFunctionUrl()`:

```python
# Can be a Function or an Alias
# fn: lambda.Function
# my_role: iam.Role


fn_url = fn.add_function_url()
fn_url.grant_invoke_url(my_role)

CfnOutput(self, "TheUrl",
    # The .url attributes will return the unique Function URL
    value=fn_url.url
)
```

Calls to this URL need to be signed with SigV4.

### Anonymous Function URLs

To create a Function URL which can be called anonymously, pass `authType: FunctionUrlAuthType.NONE` to `addFunctionUrl()`:

```python
# Can be a Function or an Alias
# fn: lambda.Function


fn_url = fn.add_function_url(
    auth_type=lambda_.FunctionUrlAuthType.NONE
)

CfnOutput(self, "TheUrl",
    value=fn_url.url
)
```

### CORS configuration for Function URLs

If you want your Function URLs to be invokable from a web page in browser, you
will need to configure cross-origin resource sharing to allow the call (if you do
not do this, your browser will refuse to make the call):

```python
# fn: lambda.Function


fn.add_function_url(
    auth_type=lambda_.FunctionUrlAuthType.NONE,
    cors=lambda.FunctionUrlCorsOptions(
        # Allow this to be called from websites on https://example.com.
        # Can also be ['*'] to allow all domain.
        allowed_origins=["https://example.com"]
    )
)
```

## Layers

The `lambda.LayerVersion` class can be used to define Lambda layers and manage
granting permissions to other AWS accounts or organizations.

```python
layer = lambda_.LayerVersion(stack, "MyLayer",
    code=lambda_.Code.from_asset(path.join(__dirname, "layer-code")),
    compatible_runtimes=[lambda_.Runtime.NODEJS_14_X],
    license="Apache-2.0",
    description="A layer to test the L2 construct"
)

# To grant usage by other AWS accounts
layer.add_permission("remote-account-grant", account_id=aws_account_id)

# To grant usage to all accounts in some AWS Ogranization
# layer.grantUsage({ accountId: '*', organizationId });

lambda_.Function(stack, "MyLayeredLambda",
    code=lambda_.InlineCode("foo"),
    handler="index.handler",
    runtime=lambda_.Runtime.NODEJS_14_X,
    layers=[layer]
)
```

By default, updating a layer creates a new layer version, and CloudFormation will delete the old version as part of the stack update.

Alternatively, a removal policy can be used to retain the old version:

```python
lambda_.LayerVersion(self, "MyLayer",
    removal_policy=RemovalPolicy.RETAIN,
    code=lambda_.Code.from_asset(path.join(__dirname, "lambda-handler"))
)
```

## Architecture

Lambda functions, by default, run on compute systems that have the 64 bit x86 architecture.

The AWS Lambda service also runs compute on the ARM architecture, which can reduce cost
for some workloads.

A lambda function can be configured to be run on one of these platforms:

```python
lambda_.Function(self, "MyFunction",
    runtime=lambda_.Runtime.NODEJS_16_X,
    handler="index.handler",
    code=lambda_.Code.from_asset(path.join(__dirname, "lambda-handler")),
    architecture=lambda_.Architecture.ARM_64
)
```

Similarly, lambda layer versions can also be tagged with architectures it is compatible with.

```python
lambda_.LayerVersion(self, "MyLayer",
    removal_policy=RemovalPolicy.RETAIN,
    code=lambda_.Code.from_asset(path.join(__dirname, "lambda-handler")),
    compatible_architectures=[lambda_.Architecture.X86_64, lambda_.Architecture.ARM_64]
)
```

## Lambda Insights

Lambda functions can be configured to use CloudWatch [Lambda Insights](https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/Lambda-Insights.html)
which provides low-level runtime metrics for a Lambda functions.

```python
lambda_.Function(self, "MyFunction",
    runtime=lambda_.Runtime.NODEJS_16_X,
    handler="index.handler",
    code=lambda_.Code.from_asset(path.join(__dirname, "lambda-handler")),
    insights_version=lambda_.LambdaInsightsVersion.VERSION_1_0_98_0
)
```

If the version of insights is not yet available in the CDK, you can also provide the ARN directly as so -

```python
layer_arn = "arn:aws:lambda:us-east-1:580247275435:layer:LambdaInsightsExtension:14"
lambda_.Function(self, "MyFunction",
    runtime=lambda_.Runtime.NODEJS_16_X,
    handler="index.handler",
    code=lambda_.Code.from_asset(path.join(__dirname, "lambda-handler")),
    insights_version=lambda_.LambdaInsightsVersion.from_insight_version_arn(layer_arn)
)
```

If you are deploying an ARM_64 Lambda Function, you must specify a
Lambda Insights Version >= `1_0_119_0`.

```python
lambda_.Function(self, "MyFunction",
    runtime=lambda_.Runtime.NODEJS_16_X,
    handler="index.handler",
    architecture=lambda_.Architecture.ARM_64,
    code=lambda_.Code.from_asset(path.join(__dirname, "lambda-handler")),
    insights_version=lambda_.LambdaInsightsVersion.VERSION_1_0_119_0
)
```

## Event Rule Target

You can use an AWS Lambda function as a target for an Amazon CloudWatch event
rule:

```python
import aws_cdk.aws_events as events
import aws_cdk.aws_events_targets as targets

# fn: lambda.Function

rule = events.Rule(self, "Schedule Rule",
    schedule=events.Schedule.cron(minute="0", hour="4")
)
rule.add_target(targets.LambdaFunction(fn))
```

## Event Sources

AWS Lambda supports a [variety of event sources](https://docs.aws.amazon.com/lambda/latest/dg/invoking-lambda-function.html).

In most cases, it is possible to trigger a function as a result of an event by
using one of the `add<Event>Notification` methods on the source construct. For
example, the `s3.Bucket` construct has an `onEvent` method which can be used to
trigger a Lambda when an event, such as PutObject occurs on an S3 bucket.

An alternative way to add event sources to a function is to use `function.addEventSource(source)`.
This method accepts an `IEventSource` object. The module **@aws-cdk/aws-lambda-event-sources**
includes classes for the various event sources supported by AWS Lambda.

For example, the following code adds an SQS queue as an event source for a function:

```python
import aws_cdk.aws_lambda_event_sources as eventsources
import aws_cdk.aws_sqs as sqs

# fn: lambda.Function

queue = sqs.Queue(self, "Queue")
fn.add_event_source(eventsources.SqsEventSource(queue))
```

The following code adds an S3 bucket notification as an event source:

```python
import aws_cdk.aws_lambda_event_sources as eventsources
import aws_cdk.aws_s3 as s3

# fn: lambda.Function

bucket = s3.Bucket(self, "Bucket")
fn.add_event_source(eventsources.S3EventSource(bucket,
    events=[s3.EventType.OBJECT_CREATED, s3.EventType.OBJECT_REMOVED],
    filters=[s3.NotificationKeyFilter(prefix="subdir/")]
))
```

See the documentation for the **@aws-cdk/aws-lambda-event-sources** module for more details.

## Imported Lambdas

When referencing an imported lambda in the CDK, use `fromFunctionArn()` for most use cases:

```python
fn = lambda_.Function.from_function_arn(self, "Function", "arn:aws:lambda:us-east-1:123456789012:function:MyFn")
```

The `fromFunctionAttributes()` API is available for more specific use cases:

```python
fn = lambda_.Function.from_function_attributes(self, "Function",
    function_arn="arn:aws:lambda:us-east-1:123456789012:function:MyFn",
    # The following are optional properties for specific use cases and should be used with caution:

    # Use Case: imported function is in the same account as the stack. This tells the CDK that it
    # can modify the function's permissions.
    same_environment=True,

    # Use Case: imported function is in a different account and user commits to ensuring that the
    # imported function has the correct permissions outside the CDK.
    skip_permissions=True
)
```

If `fromFunctionArn()` causes an error related to having to provide an account and/or region in a different construct,
and the lambda is in the same account and region as the stack you're importing it into,
you can use `Function.fromFunctionName()` instead:

```python
fn = lambda_.Function.from_function_name(self, "Function", "MyFn")
```

## Lambda with DLQ

A dead-letter queue can be automatically created for a Lambda function by
setting the `deadLetterQueueEnabled: true` configuration. In such case CDK creates
a `sqs.Queue` as `deadLetterQueue`.

```python
fn = lambda_.Function(self, "MyFunction",
    runtime=lambda_.Runtime.NODEJS_16_X,
    handler="index.handler",
    code=lambda_.Code.from_inline("exports.handler = function(event, ctx, cb) { return cb(null, \"hi\"); }"),
    dead_letter_queue_enabled=True
)
```

It is also possible to provide a dead-letter queue instead of getting a new queue created:

```python
import aws_cdk.aws_sqs as sqs


dlq = sqs.Queue(self, "DLQ")
fn = lambda_.Function(self, "MyFunction",
    runtime=lambda_.Runtime.NODEJS_16_X,
    handler="index.handler",
    code=lambda_.Code.from_inline("exports.handler = function(event, ctx, cb) { return cb(null, \"hi\"); }"),
    dead_letter_queue=dlq
)
```

You can also use a `sns.Topic` instead of an `sqs.Queue` as dead-letter queue:

```python
import aws_cdk.aws_sns as sns


dlt = sns.Topic(self, "DLQ")
fn = lambda_.Function(self, "MyFunction",
    runtime=lambda_.Runtime.NODEJS_16_X,
    handler="index.handler",
    code=lambda_.Code.from_inline("// your code here"),
    dead_letter_topic=dlt
)
```

See [the AWS documentation](https://docs.aws.amazon.com/lambda/latest/dg/dlq.html)
to learn more about AWS Lambdas and DLQs.

## Lambda with X-Ray Tracing

```python
fn = lambda_.Function(self, "MyFunction",
    runtime=lambda_.Runtime.NODEJS_16_X,
    handler="index.handler",
    code=lambda_.Code.from_inline("exports.handler = function(event, ctx, cb) { return cb(null, \"hi\"); }"),
    tracing=lambda_.Tracing.ACTIVE
)
```

See [the AWS documentation](https://docs.aws.amazon.com/lambda/latest/dg/lambda-x-ray.html)
to learn more about AWS Lambda's X-Ray support.

## Lambda with Profiling

The following code configures the lambda function with CodeGuru profiling. By default, this creates a new CodeGuru
profiling group -

```python
fn = lambda_.Function(self, "MyFunction",
    runtime=lambda_.Runtime.PYTHON_3_9,
    handler="index.handler",
    code=lambda_.Code.from_asset("lambda-handler"),
    profiling=True
)
```

The `profilingGroup` property can be used to configure an existing CodeGuru profiler group.

CodeGuru profiling is supported for all Java runtimes and Python3.6+ runtimes.

See [the AWS documentation](https://docs.aws.amazon.com/codeguru/latest/profiler-ug/setting-up-lambda.html)
to learn more about AWS Lambda's Profiling support.

## Lambda with Reserved Concurrent Executions

```python
fn = lambda_.Function(self, "MyFunction",
    runtime=lambda_.Runtime.NODEJS_16_X,
    handler="index.handler",
    code=lambda_.Code.from_inline("exports.handler = function(event, ctx, cb) { return cb(null, \"hi\"); }"),
    reserved_concurrent_executions=100
)
```

See [the AWS documentation](https://docs.aws.amazon.com/lambda/latest/dg/concurrent-executions.html)
managing concurrency.

## AutoScaling

You can use Application AutoScaling to automatically configure the provisioned concurrency for your functions. AutoScaling can be set to track utilization or be based on a schedule. To configure AutoScaling on a function alias:

```python
import aws_cdk.aws_autoscaling as autoscaling

# fn: lambda.Function

alias = fn.add_alias("prod")

# Create AutoScaling target
as = alias.add_auto_scaling(max_capacity=50)

# Configure Target Tracking
as.scale_on_utilization(
    utilization_target=0.5
)

# Configure Scheduled Scaling
as.scale_on_schedule("ScaleUpInTheMorning",
    schedule=autoscaling.Schedule.cron(hour="8", minute="0"),
    min_capacity=20
)
```

```python
import aws_cdk.aws_applicationautoscaling as appscaling
import aws_cdk.core as cdk
from aws_cdk.cx_api import LAMBDA_RECOGNIZE_LAYER_VERSION
import aws_cdk.aws_lambda as lambda_

#
# Stack verification steps:
# aws application-autoscaling describe-scalable-targets --service-namespace lambda --resource-ids function:<function name>:prod
# has a minCapacity of 3 and maxCapacity of 50
#
class TestStack(cdk.Stack):
    def __init__(self, scope, id):
        super().__init__(scope, id)

        fn = lambda_.Function(self, "MyLambda",
            code=lambda_.InlineCode("exports.handler = async () => { console.log('hello world'); };"),
            handler="index.handler",
            runtime=lambda_.Runtime.NODEJS_14_X
        )

        version = fn.current_version

        alias = lambda_.Alias(self, "Alias",
            alias_name="prod",
            version=version
        )

        scaling_target = alias.add_auto_scaling(min_capacity=3, max_capacity=50)

        scaling_target.scale_on_utilization(
            utilization_target=0.5
        )

        scaling_target.scale_on_schedule("ScaleUpInTheMorning",
            schedule=appscaling.Schedule.cron(hour="8", minute="0"),
            min_capacity=20
        )

        scaling_target.scale_on_schedule("ScaleDownAtNight",
            schedule=appscaling.Schedule.cron(hour="20", minute="0"),
            max_capacity=20
        )

        cdk.CfnOutput(self, "FunctionName",
            value=fn.function_name
        )

app = cdk.App()

stack = TestStack(app, "aws-lambda-autoscaling")

# Changes the function description when the feature flag is present
# to validate the changed function hash.
cdk.Aspects.of(stack).add(lambda_.FunctionVersionUpgrade(LAMBDA_RECOGNIZE_LAYER_VERSION))

app.synth()
```

See [the AWS documentation](https://docs.aws.amazon.com/lambda/latest/dg/invocation-scaling.html) on autoscaling lambda functions.

## Log Group

Lambda functions automatically create a log group with the name `/aws/lambda/<function-name>` upon first execution with
log data set to never expire.

The `logRetention` property can be used to set a different expiration period.

It is possible to obtain the function's log group as a `logs.ILogGroup` by calling the `logGroup` property of the
`Function` construct.

By default, CDK uses the AWS SDK retry options when creating a log group. The `logRetentionRetryOptions` property
allows you to customize the maximum number of retries and base backoff duration.

*Note* that, if either `logRetention` is set or `logGroup` property is called, a [CloudFormation custom
resource](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cfn-customresource.html) is added
to the stack that pre-creates the log group as part of the stack deployment, if it already doesn't exist, and sets the
correct log retention period (never expire, by default).

*Further note* that, if the log group already exists and the `logRetention` is not set, the custom resource will reset
the log retention to never expire even if it was configured with a different value.

## FileSystem Access

You can configure a function to mount an Amazon Elastic File System (Amazon EFS) to a
directory in your runtime environment with the `filesystem` property. To access Amazon EFS
from lambda function, the Amazon EFS access point will be required.

The following sample allows the lambda function to mount the Amazon EFS access point to `/mnt/msg` in the runtime environment and access the filesystem with the POSIX identity defined in `posixUser`.

```python
import aws_cdk.aws_ec2 as ec2
import aws_cdk.aws_efs as efs


# create a new VPC
vpc = ec2.Vpc(self, "VPC")

# create a new Amazon EFS filesystem
file_system = efs.FileSystem(self, "Efs", vpc=vpc)

# create a new access point from the filesystem
access_point = file_system.add_access_point("AccessPoint",
    # set /export/lambda as the root of the access point
    path="/export/lambda",
    # as /export/lambda does not exist in a new efs filesystem, the efs will create the directory with the following createAcl
    create_acl=efs.Acl(
        owner_uid="1001",
        owner_gid="1001",
        permissions="750"
    ),
    # enforce the POSIX identity so lambda function will access with this identity
    posix_user=efs.PosixUser(
        uid="1001",
        gid="1001"
    )
)

fn = lambda_.Function(self, "MyLambda",
    # mount the access point to /mnt/msg in the lambda runtime environment
    filesystem=lambda_.FileSystem.from_efs_access_point(access_point, "/mnt/msg"),
    runtime=lambda_.Runtime.NODEJS_16_X,
    handler="index.handler",
    code=lambda_.Code.from_asset(path.join(__dirname, "lambda-handler")),
    vpc=vpc
)
```

## Ephemeral Storage

You can configure ephemeral storage on a function to control the amount of storage it gets for reading
or writing data, allowing you to use AWS Lambda for ETL jobs, ML inference, or other data-intensive workloads.
The ephemeral storage will be accessible in the functions' `/tmp` directory.

```python
from aws_cdk.core import Size


fn = lambda_.Function(self, "MyFunction",
    runtime=lambda_.Runtime.NODEJS_16_X,
    handler="index.handler",
    code=lambda_.Code.from_asset(path.join(__dirname, "lambda-handler")),
    ephemeral_storage_size=Size.mebibytes(1024)
)
```

Read more about using this feature in [this AWS blog post](https://aws.amazon.com/blogs/aws/aws-lambda-now-supports-up-to-10-gb-ephemeral-storage/).

## Singleton Function

The `SingletonFunction` construct is a way to guarantee that a lambda function will be guaranteed to be part of the stack,
once and only once, irrespective of how many times the construct is declared to be part of the stack. This is guaranteed
as long as the `uuid` property and the optional `lambdaPurpose` property stay the same whenever they're declared into the
stack.

A typical use case of this function is when a higher level construct needs to declare a Lambda function as part of it but
needs to guarantee that the function is declared once. However, a user of this higher level construct can declare it any
number of times and with different properties. Using `SingletonFunction` here with a fixed `uuid` will guarantee this.

For example, the `LogRetention` construct requires only one single lambda function for all different log groups whose
retention it seeks to manage.

## Bundling Asset Code

When using `lambda.Code.fromAsset(path)` it is possible to bundle the code by running a
command in a Docker container. The asset path will be mounted at `/asset-input`. The
Docker container is responsible for putting content at `/asset-output`. The content at
`/asset-output` will be zipped and used as Lambda code.

Example with Python:

```python
lambda_.Function(self, "Function",
    code=lambda_.Code.from_asset(path.join(__dirname, "my-python-handler"),
        bundling=BundlingOptions(
            image=lambda_.Runtime.PYTHON_3_9.bundling_image,
            command=["bash", "-c", "pip install -r requirements.txt -t /asset-output && cp -au . /asset-output"
            ]
        )
    ),
    runtime=lambda_.Runtime.PYTHON_3_9,
    handler="index.handler"
)
```

Runtimes expose a `bundlingImage` property that points to the [AWS SAM](https://github.com/awslabs/aws-sam-cli) build image.

Use `cdk.DockerImage.fromRegistry(image)` to use an existing image or
`cdk.DockerImage.fromBuild(path)` to build a specific image:

```python
lambda_.Function(self, "Function",
    code=lambda_.Code.from_asset("/path/to/handler",
        bundling=BundlingOptions(
            image=DockerImage.from_build("/path/to/dir/with/DockerFile",
                build_args={
                    "ARG1": "value1"
                }
            ),
            command=["my", "cool", "command"]
        )
    ),
    runtime=lambda_.Runtime.PYTHON_3_9,
    handler="index.handler"
)
```

## Language-specific APIs

Language-specific higher level constructs are provided in separate modules:

* `@aws-cdk/aws-lambda-nodejs`: [Github](https://github.com/aws/aws-cdk/tree/master/packages/%40aws-cdk/aws-lambda-nodejs) & [CDK Docs](https://docs.aws.amazon.com/cdk/api/latest/docs/aws-lambda-nodejs-readme.html)
* `@aws-cdk/aws-lambda-python`: [Github](https://github.com/aws/aws-cdk/tree/master/packages/%40aws-cdk/aws-lambda-python) & [CDK Docs](https://docs.aws.amazon.com/cdk/api/latest/docs/aws-lambda-python-readme.html)

## Code Signing

Code signing for AWS Lambda helps to ensure that only trusted code runs in your Lambda functions.
When enabled, AWS Lambda checks every code deployment and verifies that the code package is signed by a trusted source.
For more information, see [Configuring code signing for AWS Lambda](https://docs.aws.amazon.com/lambda/latest/dg/configuration-codesigning.html).
The following code configures a function with code signing.

```python
import aws_cdk.aws_signer as signer


signing_profile = signer.SigningProfile(self, "SigningProfile",
    platform=signer.Platform.AWS_LAMBDA_SHA384_ECDSA
)

code_signing_config = lambda_.CodeSigningConfig(self, "CodeSigningConfig",
    signing_profiles=[signing_profile]
)

lambda_.Function(self, "Function",
    code_signing_config=code_signing_config,
    runtime=lambda_.Runtime.NODEJS_16_X,
    handler="index.handler",
    code=lambda_.Code.from_asset(path.join(__dirname, "lambda-handler"))
)
```
'''
import abc
import builtins
import datetime
import enum
import typing

import jsii
import publication
import typing_extensions

from typeguard import check_type

from ._jsii import *

import aws_cdk.assets as _aws_cdk_assets_b1c45fb6
import aws_cdk.aws_applicationautoscaling as _aws_cdk_aws_applicationautoscaling_a31e8c21
import aws_cdk.aws_cloudwatch as _aws_cdk_aws_cloudwatch_9b88bb94
import aws_cdk.aws_codeguruprofiler as _aws_cdk_aws_codeguruprofiler_5a603484
import aws_cdk.aws_ec2 as _aws_cdk_aws_ec2_67de8e8d
import aws_cdk.aws_ecr as _aws_cdk_aws_ecr_093ed842
import aws_cdk.aws_ecr_assets as _aws_cdk_aws_ecr_assets_ef09b699
import aws_cdk.aws_efs as _aws_cdk_aws_efs_b9f7a603
import aws_cdk.aws_iam as _aws_cdk_aws_iam_940a1ce0
import aws_cdk.aws_kms as _aws_cdk_aws_kms_e491a92b
import aws_cdk.aws_logs as _aws_cdk_aws_logs_6c4320fb
import aws_cdk.aws_s3 as _aws_cdk_aws_s3_55f001a5
import aws_cdk.aws_s3_assets as _aws_cdk_aws_s3_assets_525817d7
import aws_cdk.aws_signer as _aws_cdk_aws_signer_a6bf19cf
import aws_cdk.aws_sns as _aws_cdk_aws_sns_889c7272
import aws_cdk.aws_sqs as _aws_cdk_aws_sqs_48bffef9
import aws_cdk.core as _aws_cdk_core_f4b25747
import constructs as _constructs_77d1e7e8


@jsii.data_type(
    jsii_type="@aws-cdk/aws-lambda.AliasAttributes",
    jsii_struct_bases=[],
    name_mapping={"alias_name": "aliasName", "alias_version": "aliasVersion"},
)
class AliasAttributes:
    def __init__(self, *, alias_name: builtins.str, alias_version: "IVersion") -> None:
        '''
        :param alias_name: 
        :param alias_version: 

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_lambda as lambda_
            
            # version: lambda.Version
            
            alias_attributes = lambda.AliasAttributes(
                alias_name="aliasName",
                alias_version=version
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__12ef5cf15912b9c8f9ddef5e350fae5cf38d09b1dddcf67c85e2156ad4ef53b3)
            check_type(argname="argument alias_name", value=alias_name, expected_type=type_hints["alias_name"])
            check_type(argname="argument alias_version", value=alias_version, expected_type=type_hints["alias_version"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "alias_name": alias_name,
            "alias_version": alias_version,
        }

    @builtins.property
    def alias_name(self) -> builtins.str:
        result = self._values.get("alias_name")
        assert result is not None, "Required property 'alias_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def alias_version(self) -> "IVersion":
        result = self._values.get("alias_version")
        assert result is not None, "Required property 'alias_version' is missing"
        return typing.cast("IVersion", result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AliasAttributes(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class Architecture(
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-lambda.Architecture",
):
    '''Architectures supported by AWS Lambda.

    :exampleMetadata: infused

    Example::

        lambda_.Function(self, "MyFunction",
            runtime=lambda_.Runtime.NODEJS_16_X,
            handler="index.handler",
            architecture=lambda_.Architecture.ARM_64,
            code=lambda_.Code.from_asset(path.join(__dirname, "lambda-handler")),
            insights_version=lambda_.LambdaInsightsVersion.VERSION_1_0_119_0
        )
    '''

    @jsii.member(jsii_name="custom")
    @builtins.classmethod
    def custom(
        cls,
        name: builtins.str,
        docker_platform: typing.Optional[builtins.str] = None,
    ) -> "Architecture":
        '''Used to specify a custom architecture name.

        Use this if the architecture name is not yet supported by the CDK.

        :param name: the architecture name as recognized by AWS Lambda.
        :param docker_platform: the platform to use for this architecture when building with Docker.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d0d47a89b5e4eab587919c482f42426f76e66b4936d36aaaca11ebfe71f5af4a)
            check_type(argname="argument name", value=name, expected_type=type_hints["name"])
            check_type(argname="argument docker_platform", value=docker_platform, expected_type=type_hints["docker_platform"])
        return typing.cast("Architecture", jsii.sinvoke(cls, "custom", [name, docker_platform]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ARM_64")
    def ARM_64(cls) -> "Architecture":
        '''64 bit architecture with the ARM instruction set.'''
        return typing.cast("Architecture", jsii.sget(cls, "ARM_64"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="X86_64")
    def X86_64(cls) -> "Architecture":
        '''64 bit architecture with x86 instruction set.'''
        return typing.cast("Architecture", jsii.sget(cls, "X86_64"))

    @builtins.property
    @jsii.member(jsii_name="dockerPlatform")
    def docker_platform(self) -> builtins.str:
        '''The platform to use for this architecture when building with Docker.'''
        return typing.cast(builtins.str, jsii.get(self, "dockerPlatform"))

    @builtins.property
    @jsii.member(jsii_name="name")
    def name(self) -> builtins.str:
        '''The name of the architecture as recognized by the AWS Lambda service APIs.'''
        return typing.cast(builtins.str, jsii.get(self, "name"))


@jsii.data_type(
    jsii_type="@aws-cdk/aws-lambda.AssetImageCodeProps",
    jsii_struct_bases=[_aws_cdk_aws_ecr_assets_ef09b699.DockerImageAssetOptions],
    name_mapping={
        "exclude": "exclude",
        "follow": "follow",
        "ignore_mode": "ignoreMode",
        "extra_hash": "extraHash",
        "follow_symlinks": "followSymlinks",
        "build_args": "buildArgs",
        "file": "file",
        "invalidation": "invalidation",
        "network_mode": "networkMode",
        "platform": "platform",
        "repository_name": "repositoryName",
        "target": "target",
        "cmd": "cmd",
        "entrypoint": "entrypoint",
        "working_directory": "workingDirectory",
    },
)
class AssetImageCodeProps(_aws_cdk_aws_ecr_assets_ef09b699.DockerImageAssetOptions):
    def __init__(
        self,
        *,
        exclude: typing.Optional[typing.Sequence[builtins.str]] = None,
        follow: typing.Optional[_aws_cdk_assets_b1c45fb6.FollowMode] = None,
        ignore_mode: typing.Optional[_aws_cdk_core_f4b25747.IgnoreMode] = None,
        extra_hash: typing.Optional[builtins.str] = None,
        follow_symlinks: typing.Optional[_aws_cdk_core_f4b25747.SymlinkFollowMode] = None,
        build_args: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        file: typing.Optional[builtins.str] = None,
        invalidation: typing.Optional[typing.Union[_aws_cdk_aws_ecr_assets_ef09b699.DockerImageAssetInvalidationOptions, typing.Dict[builtins.str, typing.Any]]] = None,
        network_mode: typing.Optional[_aws_cdk_aws_ecr_assets_ef09b699.NetworkMode] = None,
        platform: typing.Optional[_aws_cdk_aws_ecr_assets_ef09b699.Platform] = None,
        repository_name: typing.Optional[builtins.str] = None,
        target: typing.Optional[builtins.str] = None,
        cmd: typing.Optional[typing.Sequence[builtins.str]] = None,
        entrypoint: typing.Optional[typing.Sequence[builtins.str]] = None,
        working_directory: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties to initialize a new AssetImage.

        :param exclude: Glob patterns to exclude from the copy. Default: - nothing is excluded
        :param follow: (deprecated) A strategy for how to handle symlinks. Default: Never
        :param ignore_mode: The ignore behavior to use for exclude patterns. Default: IgnoreMode.GLOB
        :param extra_hash: Extra information to encode into the fingerprint (e.g. build instructions and other inputs). Default: - hash is only based on source content
        :param follow_symlinks: A strategy for how to handle symlinks. Default: SymlinkFollowMode.NEVER
        :param build_args: Build args to pass to the ``docker build`` command. Since Docker build arguments are resolved before deployment, keys and values cannot refer to unresolved tokens (such as ``lambda.functionArn`` or ``queue.queueUrl``). Default: - no build args are passed
        :param file: Path to the Dockerfile (relative to the directory). Default: 'Dockerfile'
        :param invalidation: Options to control which parameters are used to invalidate the asset hash. Default: - hash all parameters
        :param network_mode: Networking mode for the RUN commands during build. Support docker API 1.25+. Default: - no networking mode specified (the default networking mode ``NetworkMode.DEFAULT`` will be used)
        :param platform: Platform to build for. *Requires Docker Buildx*. Default: - no platform specified (the current machine architecture will be used)
        :param repository_name: (deprecated) ECR repository name. Specify this property if you need to statically address the image, e.g. from a Kubernetes Pod. Note, this is only the repository name, without the registry and the tag parts. Default: - the default ECR repository for CDK assets
        :param target: Docker target to build to. Default: - no target
        :param cmd: Specify or override the CMD on the specified Docker image or Dockerfile. This needs to be in the 'exec form', viz., ``[ 'executable', 'param1', 'param2' ]``. Default: - use the CMD specified in the docker image or Dockerfile.
        :param entrypoint: Specify or override the ENTRYPOINT on the specified Docker image or Dockerfile. An ENTRYPOINT allows you to configure a container that will run as an executable. This needs to be in the 'exec form', viz., ``[ 'executable', 'param1', 'param2' ]``. Default: - use the ENTRYPOINT in the docker image or Dockerfile.
        :param working_directory: Specify or override the WORKDIR on the specified Docker image or Dockerfile. A WORKDIR allows you to configure the working directory the container will use. Default: - use the WORKDIR in the docker image or Dockerfile.

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.assets as assets
            import aws_cdk.aws_ecr_assets as ecr_assets
            import aws_cdk.aws_lambda as lambda_
            import aws_cdk.core as cdk
            
            # network_mode: ecr_assets.NetworkMode
            # platform: ecr_assets.Platform
            
            asset_image_code_props = lambda.AssetImageCodeProps(
                build_args={
                    "build_args_key": "buildArgs"
                },
                cmd=["cmd"],
                entrypoint=["entrypoint"],
                exclude=["exclude"],
                extra_hash="extraHash",
                file="file",
                follow=assets.FollowMode.NEVER,
                follow_symlinks=cdk.SymlinkFollowMode.NEVER,
                ignore_mode=cdk.IgnoreMode.GLOB,
                invalidation=ecr_assets.DockerImageAssetInvalidationOptions(
                    build_args=False,
                    extra_hash=False,
                    file=False,
                    network_mode=False,
                    platform=False,
                    repository_name=False,
                    target=False
                ),
                network_mode=network_mode,
                platform=platform,
                repository_name="repositoryName",
                target="target",
                working_directory="workingDirectory"
            )
        '''
        if isinstance(invalidation, dict):
            invalidation = _aws_cdk_aws_ecr_assets_ef09b699.DockerImageAssetInvalidationOptions(**invalidation)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__405c28c1bb8fba0cb601eabbfe48dce08c27e3b07cde0bfebaaa89426e63dcdf)
            check_type(argname="argument exclude", value=exclude, expected_type=type_hints["exclude"])
            check_type(argname="argument follow", value=follow, expected_type=type_hints["follow"])
            check_type(argname="argument ignore_mode", value=ignore_mode, expected_type=type_hints["ignore_mode"])
            check_type(argname="argument extra_hash", value=extra_hash, expected_type=type_hints["extra_hash"])
            check_type(argname="argument follow_symlinks", value=follow_symlinks, expected_type=type_hints["follow_symlinks"])
            check_type(argname="argument build_args", value=build_args, expected_type=type_hints["build_args"])
            check_type(argname="argument file", value=file, expected_type=type_hints["file"])
            check_type(argname="argument invalidation", value=invalidation, expected_type=type_hints["invalidation"])
            check_type(argname="argument network_mode", value=network_mode, expected_type=type_hints["network_mode"])
            check_type(argname="argument platform", value=platform, expected_type=type_hints["platform"])
            check_type(argname="argument repository_name", value=repository_name, expected_type=type_hints["repository_name"])
            check_type(argname="argument target", value=target, expected_type=type_hints["target"])
            check_type(argname="argument cmd", value=cmd, expected_type=type_hints["cmd"])
            check_type(argname="argument entrypoint", value=entrypoint, expected_type=type_hints["entrypoint"])
            check_type(argname="argument working_directory", value=working_directory, expected_type=type_hints["working_directory"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if exclude is not None:
            self._values["exclude"] = exclude
        if follow is not None:
            self._values["follow"] = follow
        if ignore_mode is not None:
            self._values["ignore_mode"] = ignore_mode
        if extra_hash is not None:
            self._values["extra_hash"] = extra_hash
        if follow_symlinks is not None:
            self._values["follow_symlinks"] = follow_symlinks
        if build_args is not None:
            self._values["build_args"] = build_args
        if file is not None:
            self._values["file"] = file
        if invalidation is not None:
            self._values["invalidation"] = invalidation
        if network_mode is not None:
            self._values["network_mode"] = network_mode
        if platform is not None:
            self._values["platform"] = platform
        if repository_name is not None:
            self._values["repository_name"] = repository_name
        if target is not None:
            self._values["target"] = target
        if cmd is not None:
            self._values["cmd"] = cmd
        if entrypoint is not None:
            self._values["entrypoint"] = entrypoint
        if working_directory is not None:
            self._values["working_directory"] = working_directory

    @builtins.property
    def exclude(self) -> typing.Optional[typing.List[builtins.str]]:
        '''Glob patterns to exclude from the copy.

        :default: - nothing is excluded
        '''
        result = self._values.get("exclude")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def follow(self) -> typing.Optional[_aws_cdk_assets_b1c45fb6.FollowMode]:
        '''(deprecated) A strategy for how to handle symlinks.

        :default: Never

        :deprecated: use ``followSymlinks`` instead

        :stability: deprecated
        '''
        result = self._values.get("follow")
        return typing.cast(typing.Optional[_aws_cdk_assets_b1c45fb6.FollowMode], result)

    @builtins.property
    def ignore_mode(self) -> typing.Optional[_aws_cdk_core_f4b25747.IgnoreMode]:
        '''The ignore behavior to use for exclude patterns.

        :default: IgnoreMode.GLOB
        '''
        result = self._values.get("ignore_mode")
        return typing.cast(typing.Optional[_aws_cdk_core_f4b25747.IgnoreMode], result)

    @builtins.property
    def extra_hash(self) -> typing.Optional[builtins.str]:
        '''Extra information to encode into the fingerprint (e.g. build instructions and other inputs).

        :default: - hash is only based on source content
        '''
        result = self._values.get("extra_hash")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def follow_symlinks(
        self,
    ) -> typing.Optional[_aws_cdk_core_f4b25747.SymlinkFollowMode]:
        '''A strategy for how to handle symlinks.

        :default: SymlinkFollowMode.NEVER
        '''
        result = self._values.get("follow_symlinks")
        return typing.cast(typing.Optional[_aws_cdk_core_f4b25747.SymlinkFollowMode], result)

    @builtins.property
    def build_args(self) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''Build args to pass to the ``docker build`` command.

        Since Docker build arguments are resolved before deployment, keys and
        values cannot refer to unresolved tokens (such as ``lambda.functionArn`` or
        ``queue.queueUrl``).

        :default: - no build args are passed
        '''
        result = self._values.get("build_args")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    @builtins.property
    def file(self) -> typing.Optional[builtins.str]:
        '''Path to the Dockerfile (relative to the directory).

        :default: 'Dockerfile'
        '''
        result = self._values.get("file")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def invalidation(
        self,
    ) -> typing.Optional[_aws_cdk_aws_ecr_assets_ef09b699.DockerImageAssetInvalidationOptions]:
        '''Options to control which parameters are used to invalidate the asset hash.

        :default: - hash all parameters
        '''
        result = self._values.get("invalidation")
        return typing.cast(typing.Optional[_aws_cdk_aws_ecr_assets_ef09b699.DockerImageAssetInvalidationOptions], result)

    @builtins.property
    def network_mode(
        self,
    ) -> typing.Optional[_aws_cdk_aws_ecr_assets_ef09b699.NetworkMode]:
        '''Networking mode for the RUN commands during build.

        Support docker API 1.25+.

        :default: - no networking mode specified (the default networking mode ``NetworkMode.DEFAULT`` will be used)
        '''
        result = self._values.get("network_mode")
        return typing.cast(typing.Optional[_aws_cdk_aws_ecr_assets_ef09b699.NetworkMode], result)

    @builtins.property
    def platform(self) -> typing.Optional[_aws_cdk_aws_ecr_assets_ef09b699.Platform]:
        '''Platform to build for.

        *Requires Docker Buildx*.

        :default: - no platform specified (the current machine architecture will be used)
        '''
        result = self._values.get("platform")
        return typing.cast(typing.Optional[_aws_cdk_aws_ecr_assets_ef09b699.Platform], result)

    @builtins.property
    def repository_name(self) -> typing.Optional[builtins.str]:
        '''(deprecated) ECR repository name.

        Specify this property if you need to statically address the image, e.g.
        from a Kubernetes Pod. Note, this is only the repository name, without the
        registry and the tag parts.

        :default: - the default ECR repository for CDK assets

        :deprecated:

        to control the location of docker image assets, please override
        ``Stack.addDockerImageAsset``. this feature will be removed in future
        releases.

        :stability: deprecated
        '''
        result = self._values.get("repository_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def target(self) -> typing.Optional[builtins.str]:
        '''Docker target to build to.

        :default: - no target
        '''
        result = self._values.get("target")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def cmd(self) -> typing.Optional[typing.List[builtins.str]]:
        '''Specify or override the CMD on the specified Docker image or Dockerfile.

        This needs to be in the 'exec form', viz., ``[ 'executable', 'param1', 'param2' ]``.

        :default: - use the CMD specified in the docker image or Dockerfile.

        :see: https://docs.docker.com/engine/reference/builder/#cmd
        '''
        result = self._values.get("cmd")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def entrypoint(self) -> typing.Optional[typing.List[builtins.str]]:
        '''Specify or override the ENTRYPOINT on the specified Docker image or Dockerfile.

        An ENTRYPOINT allows you to configure a container that will run as an executable.
        This needs to be in the 'exec form', viz., ``[ 'executable', 'param1', 'param2' ]``.

        :default: - use the ENTRYPOINT in the docker image or Dockerfile.

        :see: https://docs.docker.com/engine/reference/builder/#entrypoint
        '''
        result = self._values.get("entrypoint")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def working_directory(self) -> typing.Optional[builtins.str]:
        '''Specify or override the WORKDIR on the specified Docker image or Dockerfile.

        A WORKDIR allows you to configure the working directory the container will use.

        :default: - use the WORKDIR in the docker image or Dockerfile.

        :see: https://docs.docker.com/engine/reference/builder/#workdir
        '''
        result = self._values.get("working_directory")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AssetImageCodeProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-lambda.AutoScalingOptions",
    jsii_struct_bases=[],
    name_mapping={"max_capacity": "maxCapacity", "min_capacity": "minCapacity"},
)
class AutoScalingOptions:
    def __init__(
        self,
        *,
        max_capacity: jsii.Number,
        min_capacity: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''Properties for enabling Lambda autoscaling.

        :param max_capacity: Maximum capacity to scale to.
        :param min_capacity: Minimum capacity to scale to. Default: 1

        :exampleMetadata: infused

        Example::

            import aws_cdk.aws_autoscaling as autoscaling
            
            # fn: lambda.Function
            
            alias = fn.add_alias("prod")
            
            # Create AutoScaling target
            as = alias.add_auto_scaling(max_capacity=50)
            
            # Configure Target Tracking
            as.scale_on_utilization(
                utilization_target=0.5
            )
            
            # Configure Scheduled Scaling
            as.scale_on_schedule("ScaleUpInTheMorning",
                schedule=autoscaling.Schedule.cron(hour="8", minute="0"),
                min_capacity=20
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__211ea11721a627991a0c19c46316d5a53cbaa3e9459a90f76ec99ec47f7b92c3)
            check_type(argname="argument max_capacity", value=max_capacity, expected_type=type_hints["max_capacity"])
            check_type(argname="argument min_capacity", value=min_capacity, expected_type=type_hints["min_capacity"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "max_capacity": max_capacity,
        }
        if min_capacity is not None:
            self._values["min_capacity"] = min_capacity

    @builtins.property
    def max_capacity(self) -> jsii.Number:
        '''Maximum capacity to scale to.'''
        result = self._values.get("max_capacity")
        assert result is not None, "Required property 'max_capacity' is missing"
        return typing.cast(jsii.Number, result)

    @builtins.property
    def min_capacity(self) -> typing.Optional[jsii.Number]:
        '''Minimum capacity to scale to.

        :default: 1
        '''
        result = self._values.get("min_capacity")
        return typing.cast(typing.Optional[jsii.Number], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AutoScalingOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_aws_cdk_core_f4b25747.IInspectable)
class CfnAlias(
    _aws_cdk_core_f4b25747.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-lambda.CfnAlias",
):
    '''A CloudFormation ``AWS::Lambda::Alias``.

    The ``AWS::Lambda::Alias`` resource creates an `alias <https://docs.aws.amazon.com/lambda/latest/dg/configuration-aliases.html>`_ for a Lambda function version. Use aliases to provide clients with a function identifier that you can update to invoke a different version.

    You can also map an alias to split invocation requests between two versions. Use the ``RoutingConfig`` parameter to specify a second version and the percentage of invocation requests that it receives.

    :cloudformationResource: AWS::Lambda::Alias
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-alias.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        import aws_cdk.aws_lambda as lambda_
        
        cfn_alias = lambda_.CfnAlias(self, "MyCfnAlias",
            function_name="functionName",
            function_version="functionVersion",
            name="name",
        
            # the properties below are optional
            description="description",
            provisioned_concurrency_config=lambda.CfnAlias.ProvisionedConcurrencyConfigurationProperty(
                provisioned_concurrent_executions=123
            ),
            routing_config=lambda.CfnAlias.AliasRoutingConfigurationProperty(
                additional_version_weights=[lambda.CfnAlias.VersionWeightProperty(
                    function_version="functionVersion",
                    function_weight=123
                )]
            )
        )
    '''

    def __init__(
        self,
        scope: _aws_cdk_core_f4b25747.Construct,
        id: builtins.str,
        *,
        function_name: builtins.str,
        function_version: builtins.str,
        name: builtins.str,
        description: typing.Optional[builtins.str] = None,
        provisioned_concurrency_config: typing.Optional[typing.Union[typing.Union["CfnAlias.ProvisionedConcurrencyConfigurationProperty", typing.Dict[builtins.str, typing.Any]], _aws_cdk_core_f4b25747.IResolvable]] = None,
        routing_config: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnAlias.AliasRoutingConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''Create a new ``AWS::Lambda::Alias``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param function_name: The name of the Lambda function. **Name formats** - *Function name* - ``MyFunction`` . - *Function ARN* - ``arn:aws:lambda:us-west-2:123456789012:function:MyFunction`` . - *Partial ARN* - ``123456789012:function:MyFunction`` . The length constraint applies only to the full ARN. If you specify only the function name, it is limited to 64 characters in length.
        :param function_version: The function version that the alias invokes.
        :param name: The name of the alias.
        :param description: A description of the alias.
        :param provisioned_concurrency_config: Specifies a `provisioned concurrency <https://docs.aws.amazon.com/lambda/latest/dg/configuration-concurrency.html>`_ configuration for a function's alias.
        :param routing_config: The `routing configuration <https://docs.aws.amazon.com/lambda/latest/dg/lambda-traffic-shifting-using-aliases.html>`_ of the alias.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d63f0edf46b39a7ac64e2bcb9331752f23dd2c605146507bc6c48af68746a4a9)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnAliasProps(
            function_name=function_name,
            function_version=function_version,
            name=name,
            description=description,
            provisioned_concurrency_config=provisioned_concurrency_config,
            routing_config=routing_config,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _aws_cdk_core_f4b25747.TreeInspector) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__fa9902b9500fc4660ab452d5cc4cc2e8031325aeed72c4206929a734ddffa3b6)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__7f15fb16f5608fb9c4d3463e3d65ce234010b31b02f7e7accdc8c73462bbf5a4)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="functionName")
    def function_name(self) -> builtins.str:
        '''The name of the Lambda function.

        **Name formats** - *Function name* - ``MyFunction`` .

        - *Function ARN* - ``arn:aws:lambda:us-west-2:123456789012:function:MyFunction`` .
        - *Partial ARN* - ``123456789012:function:MyFunction`` .

        The length constraint applies only to the full ARN. If you specify only the function name, it is limited to 64 characters in length.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-alias.html#cfn-lambda-alias-functionname
        '''
        return typing.cast(builtins.str, jsii.get(self, "functionName"))

    @function_name.setter
    def function_name(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f1e9d0502557d975c0695c3ca61088b7e37d54eae655817fcb390d633f47ac9a)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "functionName", value)

    @builtins.property
    @jsii.member(jsii_name="functionVersion")
    def function_version(self) -> builtins.str:
        '''The function version that the alias invokes.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-alias.html#cfn-lambda-alias-functionversion
        '''
        return typing.cast(builtins.str, jsii.get(self, "functionVersion"))

    @function_version.setter
    def function_version(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__00ce32e61e154419d23dc5a9486f9ea679618fa58e0c5db58122d63d5bc39025)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "functionVersion", value)

    @builtins.property
    @jsii.member(jsii_name="name")
    def name(self) -> builtins.str:
        '''The name of the alias.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-alias.html#cfn-lambda-alias-name
        '''
        return typing.cast(builtins.str, jsii.get(self, "name"))

    @name.setter
    def name(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__bab16e82ef097084fad79882975d207edc7797aa7e4faeb609d496c46060fe91)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "name", value)

    @builtins.property
    @jsii.member(jsii_name="description")
    def description(self) -> typing.Optional[builtins.str]:
        '''A description of the alias.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-alias.html#cfn-lambda-alias-description
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "description"))

    @description.setter
    def description(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ac3da53ea2baf4e93df1307ccea8a53dc8cb585eadecf0755c025259b1531d9d)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "description", value)

    @builtins.property
    @jsii.member(jsii_name="provisionedConcurrencyConfig")
    def provisioned_concurrency_config(
        self,
    ) -> typing.Optional[typing.Union["CfnAlias.ProvisionedConcurrencyConfigurationProperty", _aws_cdk_core_f4b25747.IResolvable]]:
        '''Specifies a `provisioned concurrency <https://docs.aws.amazon.com/lambda/latest/dg/configuration-concurrency.html>`_ configuration for a function's alias.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-alias.html#cfn-lambda-alias-provisionedconcurrencyconfig
        '''
        return typing.cast(typing.Optional[typing.Union["CfnAlias.ProvisionedConcurrencyConfigurationProperty", _aws_cdk_core_f4b25747.IResolvable]], jsii.get(self, "provisionedConcurrencyConfig"))

    @provisioned_concurrency_config.setter
    def provisioned_concurrency_config(
        self,
        value: typing.Optional[typing.Union["CfnAlias.ProvisionedConcurrencyConfigurationProperty", _aws_cdk_core_f4b25747.IResolvable]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2d2e13c572252d8b5953a12cebc527f452b9907eac147e14bbb614bbbff2a180)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "provisionedConcurrencyConfig", value)

    @builtins.property
    @jsii.member(jsii_name="routingConfig")
    def routing_config(
        self,
    ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnAlias.AliasRoutingConfigurationProperty"]]:
        '''The `routing configuration <https://docs.aws.amazon.com/lambda/latest/dg/lambda-traffic-shifting-using-aliases.html>`_ of the alias.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-alias.html#cfn-lambda-alias-routingconfig
        '''
        return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnAlias.AliasRoutingConfigurationProperty"]], jsii.get(self, "routingConfig"))

    @routing_config.setter
    def routing_config(
        self,
        value: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnAlias.AliasRoutingConfigurationProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4f27ce5114a865ce6f8ffae025e3305dba4e094afd0226d4db27c01ecc2697d9)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "routingConfig", value)

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-lambda.CfnAlias.AliasRoutingConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={"additional_version_weights": "additionalVersionWeights"},
    )
    class AliasRoutingConfigurationProperty:
        def __init__(
            self,
            *,
            additional_version_weights: typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Sequence[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnAlias.VersionWeightProperty", typing.Dict[builtins.str, typing.Any]]]]],
        ) -> None:
            '''The `traffic-shifting <https://docs.aws.amazon.com/lambda/latest/dg/lambda-traffic-shifting-using-aliases.html>`_ configuration of a Lambda function alias.

            :param additional_version_weights: The second version, and the percentage of traffic that's routed to it.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-alias-aliasroutingconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_lambda as lambda_
                
                alias_routing_configuration_property = lambda.CfnAlias.AliasRoutingConfigurationProperty(
                    additional_version_weights=[lambda.CfnAlias.VersionWeightProperty(
                        function_version="functionVersion",
                        function_weight=123
                    )]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__79514b056b53fbe5df6414433012ead8b3c59cb341e0c09ce8f5757094d6a717)
                check_type(argname="argument additional_version_weights", value=additional_version_weights, expected_type=type_hints["additional_version_weights"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "additional_version_weights": additional_version_weights,
            }

        @builtins.property
        def additional_version_weights(
            self,
        ) -> typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.List[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnAlias.VersionWeightProperty"]]]:
            '''The second version, and the percentage of traffic that's routed to it.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-alias-aliasroutingconfiguration.html#cfn-lambda-alias-aliasroutingconfiguration-additionalversionweights
            '''
            result = self._values.get("additional_version_weights")
            assert result is not None, "Required property 'additional_version_weights' is missing"
            return typing.cast(typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.List[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnAlias.VersionWeightProperty"]]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "AliasRoutingConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-lambda.CfnAlias.ProvisionedConcurrencyConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "provisioned_concurrent_executions": "provisionedConcurrentExecutions",
        },
    )
    class ProvisionedConcurrencyConfigurationProperty:
        def __init__(self, *, provisioned_concurrent_executions: jsii.Number) -> None:
            '''A provisioned concurrency configuration for a function's alias.

            :param provisioned_concurrent_executions: The amount of provisioned concurrency to allocate for the alias.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-alias-provisionedconcurrencyconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_lambda as lambda_
                
                provisioned_concurrency_configuration_property = lambda.CfnAlias.ProvisionedConcurrencyConfigurationProperty(
                    provisioned_concurrent_executions=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__c3d177bcd513df88bbe63db0b8dd5e563b0931b71e67c7069ea03b91ec0a536e)
                check_type(argname="argument provisioned_concurrent_executions", value=provisioned_concurrent_executions, expected_type=type_hints["provisioned_concurrent_executions"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "provisioned_concurrent_executions": provisioned_concurrent_executions,
            }

        @builtins.property
        def provisioned_concurrent_executions(self) -> jsii.Number:
            '''The amount of provisioned concurrency to allocate for the alias.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-alias-provisionedconcurrencyconfiguration.html#cfn-lambda-alias-provisionedconcurrencyconfiguration-provisionedconcurrentexecutions
            '''
            result = self._values.get("provisioned_concurrent_executions")
            assert result is not None, "Required property 'provisioned_concurrent_executions' is missing"
            return typing.cast(jsii.Number, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ProvisionedConcurrencyConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-lambda.CfnAlias.VersionWeightProperty",
        jsii_struct_bases=[],
        name_mapping={
            "function_version": "functionVersion",
            "function_weight": "functionWeight",
        },
    )
    class VersionWeightProperty:
        def __init__(
            self,
            *,
            function_version: builtins.str,
            function_weight: jsii.Number,
        ) -> None:
            '''The `traffic-shifting <https://docs.aws.amazon.com/lambda/latest/dg/lambda-traffic-shifting-using-aliases.html>`_ configuration of a Lambda function alias.

            :param function_version: The qualifier of the second version.
            :param function_weight: The percentage of traffic that the alias routes to the second version.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-alias-versionweight.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_lambda as lambda_
                
                version_weight_property = lambda.CfnAlias.VersionWeightProperty(
                    function_version="functionVersion",
                    function_weight=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__de837e0288554f38ac1efaaebead177b701d202dee16a64c82229003dc8a5ac9)
                check_type(argname="argument function_version", value=function_version, expected_type=type_hints["function_version"])
                check_type(argname="argument function_weight", value=function_weight, expected_type=type_hints["function_weight"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "function_version": function_version,
                "function_weight": function_weight,
            }

        @builtins.property
        def function_version(self) -> builtins.str:
            '''The qualifier of the second version.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-alias-versionweight.html#cfn-lambda-alias-versionweight-functionversion
            '''
            result = self._values.get("function_version")
            assert result is not None, "Required property 'function_version' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def function_weight(self) -> jsii.Number:
            '''The percentage of traffic that the alias routes to the second version.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-alias-versionweight.html#cfn-lambda-alias-versionweight-functionweight
            '''
            result = self._values.get("function_weight")
            assert result is not None, "Required property 'function_weight' is missing"
            return typing.cast(jsii.Number, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VersionWeightProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-lambda.CfnAliasProps",
    jsii_struct_bases=[],
    name_mapping={
        "function_name": "functionName",
        "function_version": "functionVersion",
        "name": "name",
        "description": "description",
        "provisioned_concurrency_config": "provisionedConcurrencyConfig",
        "routing_config": "routingConfig",
    },
)
class CfnAliasProps:
    def __init__(
        self,
        *,
        function_name: builtins.str,
        function_version: builtins.str,
        name: builtins.str,
        description: typing.Optional[builtins.str] = None,
        provisioned_concurrency_config: typing.Optional[typing.Union[typing.Union[CfnAlias.ProvisionedConcurrencyConfigurationProperty, typing.Dict[builtins.str, typing.Any]], _aws_cdk_core_f4b25747.IResolvable]] = None,
        routing_config: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union[CfnAlias.AliasRoutingConfigurationProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''Properties for defining a ``CfnAlias``.

        :param function_name: The name of the Lambda function. **Name formats** - *Function name* - ``MyFunction`` . - *Function ARN* - ``arn:aws:lambda:us-west-2:123456789012:function:MyFunction`` . - *Partial ARN* - ``123456789012:function:MyFunction`` . The length constraint applies only to the full ARN. If you specify only the function name, it is limited to 64 characters in length.
        :param function_version: The function version that the alias invokes.
        :param name: The name of the alias.
        :param description: A description of the alias.
        :param provisioned_concurrency_config: Specifies a `provisioned concurrency <https://docs.aws.amazon.com/lambda/latest/dg/configuration-concurrency.html>`_ configuration for a function's alias.
        :param routing_config: The `routing configuration <https://docs.aws.amazon.com/lambda/latest/dg/lambda-traffic-shifting-using-aliases.html>`_ of the alias.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-alias.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_lambda as lambda_
            
            cfn_alias_props = lambda.CfnAliasProps(
                function_name="functionName",
                function_version="functionVersion",
                name="name",
            
                # the properties below are optional
                description="description",
                provisioned_concurrency_config=lambda.CfnAlias.ProvisionedConcurrencyConfigurationProperty(
                    provisioned_concurrent_executions=123
                ),
                routing_config=lambda.CfnAlias.AliasRoutingConfigurationProperty(
                    additional_version_weights=[lambda.CfnAlias.VersionWeightProperty(
                        function_version="functionVersion",
                        function_weight=123
                    )]
                )
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5250ac24ceadc7aed63345829ef77d848dc67b5dac1652ad6fa0c7a94f491a9e)
            check_type(argname="argument function_name", value=function_name, expected_type=type_hints["function_name"])
            check_type(argname="argument function_version", value=function_version, expected_type=type_hints["function_version"])
            check_type(argname="argument name", value=name, expected_type=type_hints["name"])
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument provisioned_concurrency_config", value=provisioned_concurrency_config, expected_type=type_hints["provisioned_concurrency_config"])
            check_type(argname="argument routing_config", value=routing_config, expected_type=type_hints["routing_config"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "function_name": function_name,
            "function_version": function_version,
            "name": name,
        }
        if description is not None:
            self._values["description"] = description
        if provisioned_concurrency_config is not None:
            self._values["provisioned_concurrency_config"] = provisioned_concurrency_config
        if routing_config is not None:
            self._values["routing_config"] = routing_config

    @builtins.property
    def function_name(self) -> builtins.str:
        '''The name of the Lambda function.

        **Name formats** - *Function name* - ``MyFunction`` .

        - *Function ARN* - ``arn:aws:lambda:us-west-2:123456789012:function:MyFunction`` .
        - *Partial ARN* - ``123456789012:function:MyFunction`` .

        The length constraint applies only to the full ARN. If you specify only the function name, it is limited to 64 characters in length.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-alias.html#cfn-lambda-alias-functionname
        '''
        result = self._values.get("function_name")
        assert result is not None, "Required property 'function_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def function_version(self) -> builtins.str:
        '''The function version that the alias invokes.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-alias.html#cfn-lambda-alias-functionversion
        '''
        result = self._values.get("function_version")
        assert result is not None, "Required property 'function_version' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def name(self) -> builtins.str:
        '''The name of the alias.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-alias.html#cfn-lambda-alias-name
        '''
        result = self._values.get("name")
        assert result is not None, "Required property 'name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''A description of the alias.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-alias.html#cfn-lambda-alias-description
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def provisioned_concurrency_config(
        self,
    ) -> typing.Optional[typing.Union[CfnAlias.ProvisionedConcurrencyConfigurationProperty, _aws_cdk_core_f4b25747.IResolvable]]:
        '''Specifies a `provisioned concurrency <https://docs.aws.amazon.com/lambda/latest/dg/configuration-concurrency.html>`_ configuration for a function's alias.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-alias.html#cfn-lambda-alias-provisionedconcurrencyconfig
        '''
        result = self._values.get("provisioned_concurrency_config")
        return typing.cast(typing.Optional[typing.Union[CfnAlias.ProvisionedConcurrencyConfigurationProperty, _aws_cdk_core_f4b25747.IResolvable]], result)

    @builtins.property
    def routing_config(
        self,
    ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, CfnAlias.AliasRoutingConfigurationProperty]]:
        '''The `routing configuration <https://docs.aws.amazon.com/lambda/latest/dg/lambda-traffic-shifting-using-aliases.html>`_ of the alias.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-alias.html#cfn-lambda-alias-routingconfig
        '''
        result = self._values.get("routing_config")
        return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, CfnAlias.AliasRoutingConfigurationProperty]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnAliasProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_aws_cdk_core_f4b25747.IInspectable)
class CfnCodeSigningConfig(
    _aws_cdk_core_f4b25747.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-lambda.CfnCodeSigningConfig",
):
    '''A CloudFormation ``AWS::Lambda::CodeSigningConfig``.

    Details about a `Code signing configuration <https://docs.aws.amazon.com/lambda/latest/dg/configuration-codesigning.html>`_ .

    :cloudformationResource: AWS::Lambda::CodeSigningConfig
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-codesigningconfig.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        import aws_cdk.aws_lambda as lambda_
        
        cfn_code_signing_config = lambda_.CfnCodeSigningConfig(self, "MyCfnCodeSigningConfig",
            allowed_publishers=lambda.CfnCodeSigningConfig.AllowedPublishersProperty(
                signing_profile_version_arns=["signingProfileVersionArns"]
            ),
        
            # the properties below are optional
            code_signing_policies=lambda.CfnCodeSigningConfig.CodeSigningPoliciesProperty(
                untrusted_artifact_on_deployment="untrustedArtifactOnDeployment"
            ),
            description="description"
        )
    '''

    def __init__(
        self,
        scope: _aws_cdk_core_f4b25747.Construct,
        id: builtins.str,
        *,
        allowed_publishers: typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnCodeSigningConfig.AllowedPublishersProperty", typing.Dict[builtins.str, typing.Any]]],
        code_signing_policies: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnCodeSigningConfig.CodeSigningPoliciesProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        description: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Create a new ``AWS::Lambda::CodeSigningConfig``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param allowed_publishers: List of allowed publishers.
        :param code_signing_policies: The code signing policy controls the validation failure action for signature mismatch or expiry.
        :param description: Code signing configuration description.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e38903a9e49a92a3e4bb534366d488403ec27484371aa740d583e79d80d2c0bf)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnCodeSigningConfigProps(
            allowed_publishers=allowed_publishers,
            code_signing_policies=code_signing_policies,
            description=description,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _aws_cdk_core_f4b25747.TreeInspector) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__cc056992bec2b53d843729189f1d72e2e0e0b8ae20ce826aac5b1fa4ea894e87)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__bb850035a3a180d2b9ab5d5c9a2175b65332e62ddb202d68dd546a14448584af)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrCodeSigningConfigArn")
    def attr_code_signing_config_arn(self) -> builtins.str:
        '''The Amazon Resource Name (ARN) of the code signing configuration.

        :cloudformationAttribute: CodeSigningConfigArn
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrCodeSigningConfigArn"))

    @builtins.property
    @jsii.member(jsii_name="attrCodeSigningConfigId")
    def attr_code_signing_config_id(self) -> builtins.str:
        '''The code signing configuration ID.

        :cloudformationAttribute: CodeSigningConfigId
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrCodeSigningConfigId"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="allowedPublishers")
    def allowed_publishers(
        self,
    ) -> typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnCodeSigningConfig.AllowedPublishersProperty"]:
        '''List of allowed publishers.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-codesigningconfig.html#cfn-lambda-codesigningconfig-allowedpublishers
        '''
        return typing.cast(typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnCodeSigningConfig.AllowedPublishersProperty"], jsii.get(self, "allowedPublishers"))

    @allowed_publishers.setter
    def allowed_publishers(
        self,
        value: typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnCodeSigningConfig.AllowedPublishersProperty"],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3afd76b21d1162b2776754269f8becf75af350c0e5837981b317bab3f9c00ac6)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "allowedPublishers", value)

    @builtins.property
    @jsii.member(jsii_name="codeSigningPolicies")
    def code_signing_policies(
        self,
    ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnCodeSigningConfig.CodeSigningPoliciesProperty"]]:
        '''The code signing policy controls the validation failure action for signature mismatch or expiry.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-codesigningconfig.html#cfn-lambda-codesigningconfig-codesigningpolicies
        '''
        return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnCodeSigningConfig.CodeSigningPoliciesProperty"]], jsii.get(self, "codeSigningPolicies"))

    @code_signing_policies.setter
    def code_signing_policies(
        self,
        value: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnCodeSigningConfig.CodeSigningPoliciesProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2be15c9cb8043785934b9ea5f7dcd629c3c3aa3bc0f2ec4e0bbfc5ddaa60ea56)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "codeSigningPolicies", value)

    @builtins.property
    @jsii.member(jsii_name="description")
    def description(self) -> typing.Optional[builtins.str]:
        '''Code signing configuration description.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-codesigningconfig.html#cfn-lambda-codesigningconfig-description
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "description"))

    @description.setter
    def description(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__167ddfb4e867097dfb08bb5f82dc5f526f5d874b4b0ed14c02eddcbae0447132)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "description", value)

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-lambda.CfnCodeSigningConfig.AllowedPublishersProperty",
        jsii_struct_bases=[],
        name_mapping={"signing_profile_version_arns": "signingProfileVersionArns"},
    )
    class AllowedPublishersProperty:
        def __init__(
            self,
            *,
            signing_profile_version_arns: typing.Sequence[builtins.str],
        ) -> None:
            '''List of signing profiles that can sign a code package.

            :param signing_profile_version_arns: The Amazon Resource Name (ARN) for each of the signing profiles. A signing profile defines a trusted user who can sign a code package.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-codesigningconfig-allowedpublishers.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_lambda as lambda_
                
                allowed_publishers_property = lambda.CfnCodeSigningConfig.AllowedPublishersProperty(
                    signing_profile_version_arns=["signingProfileVersionArns"]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__a84e44f6eb419f856c3156a534de0585201d4af1ad54816d30a18c1276199027)
                check_type(argname="argument signing_profile_version_arns", value=signing_profile_version_arns, expected_type=type_hints["signing_profile_version_arns"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "signing_profile_version_arns": signing_profile_version_arns,
            }

        @builtins.property
        def signing_profile_version_arns(self) -> typing.List[builtins.str]:
            '''The Amazon Resource Name (ARN) for each of the signing profiles.

            A signing profile defines a trusted user who can sign a code package.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-codesigningconfig-allowedpublishers.html#cfn-lambda-codesigningconfig-allowedpublishers-signingprofileversionarns
            '''
            result = self._values.get("signing_profile_version_arns")
            assert result is not None, "Required property 'signing_profile_version_arns' is missing"
            return typing.cast(typing.List[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "AllowedPublishersProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-lambda.CfnCodeSigningConfig.CodeSigningPoliciesProperty",
        jsii_struct_bases=[],
        name_mapping={
            "untrusted_artifact_on_deployment": "untrustedArtifactOnDeployment",
        },
    )
    class CodeSigningPoliciesProperty:
        def __init__(self, *, untrusted_artifact_on_deployment: builtins.str) -> None:
            '''Code signing configuration `policies <https://docs.aws.amazon.com/lambda/latest/dg/configuration-codesigning.html#config-codesigning-policies>`_ specify the validation failure action for signature mismatch or expiry.

            :param untrusted_artifact_on_deployment: Code signing configuration policy for deployment validation failure. If you set the policy to ``Enforce`` , Lambda blocks the deployment request if signature validation checks fail. If you set the policy to ``Warn`` , Lambda allows the deployment and creates a CloudWatch log. Default value: ``Warn``

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-codesigningconfig-codesigningpolicies.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_lambda as lambda_
                
                code_signing_policies_property = lambda.CfnCodeSigningConfig.CodeSigningPoliciesProperty(
                    untrusted_artifact_on_deployment="untrustedArtifactOnDeployment"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__63543e8d5559f3f69e248639dba6041bf467bc4e2a06f40f926db02d0dabc482)
                check_type(argname="argument untrusted_artifact_on_deployment", value=untrusted_artifact_on_deployment, expected_type=type_hints["untrusted_artifact_on_deployment"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "untrusted_artifact_on_deployment": untrusted_artifact_on_deployment,
            }

        @builtins.property
        def untrusted_artifact_on_deployment(self) -> builtins.str:
            '''Code signing configuration policy for deployment validation failure.

            If you set the policy to ``Enforce`` , Lambda blocks the deployment request if signature validation checks fail. If you set the policy to ``Warn`` , Lambda allows the deployment and creates a CloudWatch log.

            Default value: ``Warn``

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-codesigningconfig-codesigningpolicies.html#cfn-lambda-codesigningconfig-codesigningpolicies-untrustedartifactondeployment
            '''
            result = self._values.get("untrusted_artifact_on_deployment")
            assert result is not None, "Required property 'untrusted_artifact_on_deployment' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "CodeSigningPoliciesProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-lambda.CfnCodeSigningConfigProps",
    jsii_struct_bases=[],
    name_mapping={
        "allowed_publishers": "allowedPublishers",
        "code_signing_policies": "codeSigningPolicies",
        "description": "description",
    },
)
class CfnCodeSigningConfigProps:
    def __init__(
        self,
        *,
        allowed_publishers: typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union[CfnCodeSigningConfig.AllowedPublishersProperty, typing.Dict[builtins.str, typing.Any]]],
        code_signing_policies: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union[CfnCodeSigningConfig.CodeSigningPoliciesProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        description: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for defining a ``CfnCodeSigningConfig``.

        :param allowed_publishers: List of allowed publishers.
        :param code_signing_policies: The code signing policy controls the validation failure action for signature mismatch or expiry.
        :param description: Code signing configuration description.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-codesigningconfig.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_lambda as lambda_
            
            cfn_code_signing_config_props = lambda.CfnCodeSigningConfigProps(
                allowed_publishers=lambda.CfnCodeSigningConfig.AllowedPublishersProperty(
                    signing_profile_version_arns=["signingProfileVersionArns"]
                ),
            
                # the properties below are optional
                code_signing_policies=lambda.CfnCodeSigningConfig.CodeSigningPoliciesProperty(
                    untrusted_artifact_on_deployment="untrustedArtifactOnDeployment"
                ),
                description="description"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__1e83ef645ee63a4af3a91f5b7dc94265c97061f76b407a3efa2a46e08b1aa11d)
            check_type(argname="argument allowed_publishers", value=allowed_publishers, expected_type=type_hints["allowed_publishers"])
            check_type(argname="argument code_signing_policies", value=code_signing_policies, expected_type=type_hints["code_signing_policies"])
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "allowed_publishers": allowed_publishers,
        }
        if code_signing_policies is not None:
            self._values["code_signing_policies"] = code_signing_policies
        if description is not None:
            self._values["description"] = description

    @builtins.property
    def allowed_publishers(
        self,
    ) -> typing.Union[_aws_cdk_core_f4b25747.IResolvable, CfnCodeSigningConfig.AllowedPublishersProperty]:
        '''List of allowed publishers.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-codesigningconfig.html#cfn-lambda-codesigningconfig-allowedpublishers
        '''
        result = self._values.get("allowed_publishers")
        assert result is not None, "Required property 'allowed_publishers' is missing"
        return typing.cast(typing.Union[_aws_cdk_core_f4b25747.IResolvable, CfnCodeSigningConfig.AllowedPublishersProperty], result)

    @builtins.property
    def code_signing_policies(
        self,
    ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, CfnCodeSigningConfig.CodeSigningPoliciesProperty]]:
        '''The code signing policy controls the validation failure action for signature mismatch or expiry.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-codesigningconfig.html#cfn-lambda-codesigningconfig-codesigningpolicies
        '''
        result = self._values.get("code_signing_policies")
        return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, CfnCodeSigningConfig.CodeSigningPoliciesProperty]], result)

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''Code signing configuration description.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-codesigningconfig.html#cfn-lambda-codesigningconfig-description
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnCodeSigningConfigProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_aws_cdk_core_f4b25747.IInspectable)
class CfnEventInvokeConfig(
    _aws_cdk_core_f4b25747.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-lambda.CfnEventInvokeConfig",
):
    '''A CloudFormation ``AWS::Lambda::EventInvokeConfig``.

    The ``AWS::Lambda::EventInvokeConfig`` resource configures options for `asynchronous invocation <https://docs.aws.amazon.com/lambda/latest/dg/invocation-async.html>`_ on a version or an alias.

    By default, Lambda retries an asynchronous invocation twice if the function returns an error. It retains events in a queue for up to six hours. When an event fails all processing attempts or stays in the asynchronous invocation queue for too long, Lambda discards it.

    :cloudformationResource: AWS::Lambda::EventInvokeConfig
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-eventinvokeconfig.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        import aws_cdk.aws_lambda as lambda_
        
        cfn_event_invoke_config = lambda_.CfnEventInvokeConfig(self, "MyCfnEventInvokeConfig",
            function_name="functionName",
            qualifier="qualifier",
        
            # the properties below are optional
            destination_config=lambda.CfnEventInvokeConfig.DestinationConfigProperty(
                on_failure=lambda.CfnEventInvokeConfig.OnFailureProperty(
                    destination="destination"
                ),
                on_success=lambda.CfnEventInvokeConfig.OnSuccessProperty(
                    destination="destination"
                )
            ),
            maximum_event_age_in_seconds=123,
            maximum_retry_attempts=123
        )
    '''

    def __init__(
        self,
        scope: _aws_cdk_core_f4b25747.Construct,
        id: builtins.str,
        *,
        function_name: builtins.str,
        qualifier: builtins.str,
        destination_config: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnEventInvokeConfig.DestinationConfigProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        maximum_event_age_in_seconds: typing.Optional[jsii.Number] = None,
        maximum_retry_attempts: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''Create a new ``AWS::Lambda::EventInvokeConfig``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param function_name: The name of the Lambda function. *Minimum* : ``1`` *Maximum* : ``64`` *Pattern* : ``([a-zA-Z0-9-_]+)``
        :param qualifier: The identifier of a version or alias. - *Version* - A version number. - *Alias* - An alias name. - *Latest* - To specify the unpublished version, use ``$LATEST`` .
        :param destination_config: A destination for events after they have been sent to a function for processing. **Destinations** - *Function* - The Amazon Resource Name (ARN) of a Lambda function. - *Queue* - The ARN of an SQS queue. - *Topic* - The ARN of an SNS topic. - *Event Bus* - The ARN of an Amazon EventBridge event bus.
        :param maximum_event_age_in_seconds: The maximum age of a request that Lambda sends to a function for processing.
        :param maximum_retry_attempts: The maximum number of times to retry when the function returns an error.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d129186e5976e8697b954035fa745f7225f974830762eb6882532d3f77058afb)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnEventInvokeConfigProps(
            function_name=function_name,
            qualifier=qualifier,
            destination_config=destination_config,
            maximum_event_age_in_seconds=maximum_event_age_in_seconds,
            maximum_retry_attempts=maximum_retry_attempts,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _aws_cdk_core_f4b25747.TreeInspector) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e0f17cd3d49527a534e1184de7afa0d82487a5dd272de790d172e7c8ece7a62e)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f8bf19f667e5925874578757c0c343171c3a7c8f1100ff140c676afa42c777cb)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="functionName")
    def function_name(self) -> builtins.str:
        '''The name of the Lambda function.

        *Minimum* : ``1``

        *Maximum* : ``64``

        *Pattern* : ``([a-zA-Z0-9-_]+)``

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-eventinvokeconfig.html#cfn-lambda-eventinvokeconfig-functionname
        '''
        return typing.cast(builtins.str, jsii.get(self, "functionName"))

    @function_name.setter
    def function_name(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__14f01a2c21a02a5b19ebdfa50b2d8a48992c9204ea15a2369ae490818092bd6c)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "functionName", value)

    @builtins.property
    @jsii.member(jsii_name="qualifier")
    def qualifier(self) -> builtins.str:
        '''The identifier of a version or alias.

        - *Version* - A version number.
        - *Alias* - An alias name.
        - *Latest* - To specify the unpublished version, use ``$LATEST`` .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-eventinvokeconfig.html#cfn-lambda-eventinvokeconfig-qualifier
        '''
        return typing.cast(builtins.str, jsii.get(self, "qualifier"))

    @qualifier.setter
    def qualifier(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d0add05d86f792359343adbbe00ce006565f44873a4be92c00f326c90039c430)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "qualifier", value)

    @builtins.property
    @jsii.member(jsii_name="destinationConfig")
    def destination_config(
        self,
    ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnEventInvokeConfig.DestinationConfigProperty"]]:
        '''A destination for events after they have been sent to a function for processing.

        **Destinations** - *Function* - The Amazon Resource Name (ARN) of a Lambda function.

        - *Queue* - The ARN of an SQS queue.
        - *Topic* - The ARN of an SNS topic.
        - *Event Bus* - The ARN of an Amazon EventBridge event bus.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-eventinvokeconfig.html#cfn-lambda-eventinvokeconfig-destinationconfig
        '''
        return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnEventInvokeConfig.DestinationConfigProperty"]], jsii.get(self, "destinationConfig"))

    @destination_config.setter
    def destination_config(
        self,
        value: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnEventInvokeConfig.DestinationConfigProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c5c0f36a624a1a33f6c46a7069d5b8c7d46cb60cba73d282d33075c534459e79)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "destinationConfig", value)

    @builtins.property
    @jsii.member(jsii_name="maximumEventAgeInSeconds")
    def maximum_event_age_in_seconds(self) -> typing.Optional[jsii.Number]:
        '''The maximum age of a request that Lambda sends to a function for processing.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-eventinvokeconfig.html#cfn-lambda-eventinvokeconfig-maximumeventageinseconds
        '''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "maximumEventAgeInSeconds"))

    @maximum_event_age_in_seconds.setter
    def maximum_event_age_in_seconds(self, value: typing.Optional[jsii.Number]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3ae4ce2279d04bba0ce75c501b63b11bb5641919ddd64a20b5d2a33b06f3f638)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "maximumEventAgeInSeconds", value)

    @builtins.property
    @jsii.member(jsii_name="maximumRetryAttempts")
    def maximum_retry_attempts(self) -> typing.Optional[jsii.Number]:
        '''The maximum number of times to retry when the function returns an error.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-eventinvokeconfig.html#cfn-lambda-eventinvokeconfig-maximumretryattempts
        '''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "maximumRetryAttempts"))

    @maximum_retry_attempts.setter
    def maximum_retry_attempts(self, value: typing.Optional[jsii.Number]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__419a476a6b2a4c0ac5cb579d5ab4984429d1d219197e19acfd4e803bed6c563b)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "maximumRetryAttempts", value)

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-lambda.CfnEventInvokeConfig.DestinationConfigProperty",
        jsii_struct_bases=[],
        name_mapping={"on_failure": "onFailure", "on_success": "onSuccess"},
    )
    class DestinationConfigProperty:
        def __init__(
            self,
            *,
            on_failure: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnEventInvokeConfig.OnFailureProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            on_success: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnEventInvokeConfig.OnSuccessProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''A configuration object that specifies the destination of an event after Lambda processes it.

            :param on_failure: The destination configuration for failed invocations.
            :param on_success: The destination configuration for successful invocations.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-eventinvokeconfig-destinationconfig.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_lambda as lambda_
                
                destination_config_property = lambda.CfnEventInvokeConfig.DestinationConfigProperty(
                    on_failure=lambda.CfnEventInvokeConfig.OnFailureProperty(
                        destination="destination"
                    ),
                    on_success=lambda.CfnEventInvokeConfig.OnSuccessProperty(
                        destination="destination"
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__e6e1d960543b982fae5283aea60893459173ba7a2140dce0c73bfeaa6804f0e9)
                check_type(argname="argument on_failure", value=on_failure, expected_type=type_hints["on_failure"])
                check_type(argname="argument on_success", value=on_success, expected_type=type_hints["on_success"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if on_failure is not None:
                self._values["on_failure"] = on_failure
            if on_success is not None:
                self._values["on_success"] = on_success

        @builtins.property
        def on_failure(
            self,
        ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnEventInvokeConfig.OnFailureProperty"]]:
            '''The destination configuration for failed invocations.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-eventinvokeconfig-destinationconfig.html#cfn-lambda-eventinvokeconfig-destinationconfig-onfailure
            '''
            result = self._values.get("on_failure")
            return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnEventInvokeConfig.OnFailureProperty"]], result)

        @builtins.property
        def on_success(
            self,
        ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnEventInvokeConfig.OnSuccessProperty"]]:
            '''The destination configuration for successful invocations.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-eventinvokeconfig-destinationconfig.html#cfn-lambda-eventinvokeconfig-destinationconfig-onsuccess
            '''
            result = self._values.get("on_success")
            return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnEventInvokeConfig.OnSuccessProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "DestinationConfigProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-lambda.CfnEventInvokeConfig.OnFailureProperty",
        jsii_struct_bases=[],
        name_mapping={"destination": "destination"},
    )
    class OnFailureProperty:
        def __init__(self, *, destination: builtins.str) -> None:
            '''A destination for events that failed processing.

            :param destination: The Amazon Resource Name (ARN) of the destination resource.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-eventinvokeconfig-destinationconfig-onfailure.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_lambda as lambda_
                
                on_failure_property = lambda.CfnEventInvokeConfig.OnFailureProperty(
                    destination="destination"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__ee993b85e1ec1364d779a8af74fea4b33dbc0da7cd3a23375c9af6a4f8305a01)
                check_type(argname="argument destination", value=destination, expected_type=type_hints["destination"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "destination": destination,
            }

        @builtins.property
        def destination(self) -> builtins.str:
            '''The Amazon Resource Name (ARN) of the destination resource.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-eventinvokeconfig-destinationconfig-onfailure.html#cfn-lambda-eventinvokeconfig-destinationconfig-onfailure-destination
            '''
            result = self._values.get("destination")
            assert result is not None, "Required property 'destination' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "OnFailureProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-lambda.CfnEventInvokeConfig.OnSuccessProperty",
        jsii_struct_bases=[],
        name_mapping={"destination": "destination"},
    )
    class OnSuccessProperty:
        def __init__(self, *, destination: builtins.str) -> None:
            '''A destination for events that were processed successfully.

            :param destination: The Amazon Resource Name (ARN) of the destination resource.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-eventinvokeconfig-destinationconfig-onsuccess.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_lambda as lambda_
                
                on_success_property = lambda.CfnEventInvokeConfig.OnSuccessProperty(
                    destination="destination"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__e46497782cd1f860f38755169fd9aa975eb6a91df546d89e8ab94a3c729d9939)
                check_type(argname="argument destination", value=destination, expected_type=type_hints["destination"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "destination": destination,
            }

        @builtins.property
        def destination(self) -> builtins.str:
            '''The Amazon Resource Name (ARN) of the destination resource.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-eventinvokeconfig-destinationconfig-onsuccess.html#cfn-lambda-eventinvokeconfig-destinationconfig-onsuccess-destination
            '''
            result = self._values.get("destination")
            assert result is not None, "Required property 'destination' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "OnSuccessProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-lambda.CfnEventInvokeConfigProps",
    jsii_struct_bases=[],
    name_mapping={
        "function_name": "functionName",
        "qualifier": "qualifier",
        "destination_config": "destinationConfig",
        "maximum_event_age_in_seconds": "maximumEventAgeInSeconds",
        "maximum_retry_attempts": "maximumRetryAttempts",
    },
)
class CfnEventInvokeConfigProps:
    def __init__(
        self,
        *,
        function_name: builtins.str,
        qualifier: builtins.str,
        destination_config: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union[CfnEventInvokeConfig.DestinationConfigProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        maximum_event_age_in_seconds: typing.Optional[jsii.Number] = None,
        maximum_retry_attempts: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''Properties for defining a ``CfnEventInvokeConfig``.

        :param function_name: The name of the Lambda function. *Minimum* : ``1`` *Maximum* : ``64`` *Pattern* : ``([a-zA-Z0-9-_]+)``
        :param qualifier: The identifier of a version or alias. - *Version* - A version number. - *Alias* - An alias name. - *Latest* - To specify the unpublished version, use ``$LATEST`` .
        :param destination_config: A destination for events after they have been sent to a function for processing. **Destinations** - *Function* - The Amazon Resource Name (ARN) of a Lambda function. - *Queue* - The ARN of an SQS queue. - *Topic* - The ARN of an SNS topic. - *Event Bus* - The ARN of an Amazon EventBridge event bus.
        :param maximum_event_age_in_seconds: The maximum age of a request that Lambda sends to a function for processing.
        :param maximum_retry_attempts: The maximum number of times to retry when the function returns an error.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-eventinvokeconfig.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_lambda as lambda_
            
            cfn_event_invoke_config_props = lambda.CfnEventInvokeConfigProps(
                function_name="functionName",
                qualifier="qualifier",
            
                # the properties below are optional
                destination_config=lambda.CfnEventInvokeConfig.DestinationConfigProperty(
                    on_failure=lambda.CfnEventInvokeConfig.OnFailureProperty(
                        destination="destination"
                    ),
                    on_success=lambda.CfnEventInvokeConfig.OnSuccessProperty(
                        destination="destination"
                    )
                ),
                maximum_event_age_in_seconds=123,
                maximum_retry_attempts=123
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__70e04fc45e9b2a3beaa772b13f6db04ac16dd42b36de1b0d8aa5c4930b00791e)
            check_type(argname="argument function_name", value=function_name, expected_type=type_hints["function_name"])
            check_type(argname="argument qualifier", value=qualifier, expected_type=type_hints["qualifier"])
            check_type(argname="argument destination_config", value=destination_config, expected_type=type_hints["destination_config"])
            check_type(argname="argument maximum_event_age_in_seconds", value=maximum_event_age_in_seconds, expected_type=type_hints["maximum_event_age_in_seconds"])
            check_type(argname="argument maximum_retry_attempts", value=maximum_retry_attempts, expected_type=type_hints["maximum_retry_attempts"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "function_name": function_name,
            "qualifier": qualifier,
        }
        if destination_config is not None:
            self._values["destination_config"] = destination_config
        if maximum_event_age_in_seconds is not None:
            self._values["maximum_event_age_in_seconds"] = maximum_event_age_in_seconds
        if maximum_retry_attempts is not None:
            self._values["maximum_retry_attempts"] = maximum_retry_attempts

    @builtins.property
    def function_name(self) -> builtins.str:
        '''The name of the Lambda function.

        *Minimum* : ``1``

        *Maximum* : ``64``

        *Pattern* : ``([a-zA-Z0-9-_]+)``

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-eventinvokeconfig.html#cfn-lambda-eventinvokeconfig-functionname
        '''
        result = self._values.get("function_name")
        assert result is not None, "Required property 'function_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def qualifier(self) -> builtins.str:
        '''The identifier of a version or alias.

        - *Version* - A version number.
        - *Alias* - An alias name.
        - *Latest* - To specify the unpublished version, use ``$LATEST`` .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-eventinvokeconfig.html#cfn-lambda-eventinvokeconfig-qualifier
        '''
        result = self._values.get("qualifier")
        assert result is not None, "Required property 'qualifier' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def destination_config(
        self,
    ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, CfnEventInvokeConfig.DestinationConfigProperty]]:
        '''A destination for events after they have been sent to a function for processing.

        **Destinations** - *Function* - The Amazon Resource Name (ARN) of a Lambda function.

        - *Queue* - The ARN of an SQS queue.
        - *Topic* - The ARN of an SNS topic.
        - *Event Bus* - The ARN of an Amazon EventBridge event bus.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-eventinvokeconfig.html#cfn-lambda-eventinvokeconfig-destinationconfig
        '''
        result = self._values.get("destination_config")
        return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, CfnEventInvokeConfig.DestinationConfigProperty]], result)

    @builtins.property
    def maximum_event_age_in_seconds(self) -> typing.Optional[jsii.Number]:
        '''The maximum age of a request that Lambda sends to a function for processing.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-eventinvokeconfig.html#cfn-lambda-eventinvokeconfig-maximumeventageinseconds
        '''
        result = self._values.get("maximum_event_age_in_seconds")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def maximum_retry_attempts(self) -> typing.Optional[jsii.Number]:
        '''The maximum number of times to retry when the function returns an error.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-eventinvokeconfig.html#cfn-lambda-eventinvokeconfig-maximumretryattempts
        '''
        result = self._values.get("maximum_retry_attempts")
        return typing.cast(typing.Optional[jsii.Number], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnEventInvokeConfigProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_aws_cdk_core_f4b25747.IInspectable)
class CfnEventSourceMapping(
    _aws_cdk_core_f4b25747.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-lambda.CfnEventSourceMapping",
):
    '''A CloudFormation ``AWS::Lambda::EventSourceMapping``.

    The ``AWS::Lambda::EventSourceMapping`` resource creates a mapping between an event source and an AWS Lambda function. Lambda reads items from the event source and triggers the function.

    For details about each event source type, see the following topics. In particular, each of the topics describes the required and optional parameters for the specific event source.

    - `Configuring a Dynamo DB stream as an event source <https://docs.aws.amazon.com/lambda/latest/dg/with-ddb.html#services-dynamodb-eventsourcemapping>`_
    - `Configuring a Kinesis stream as an event source <https://docs.aws.amazon.com/lambda/latest/dg/with-kinesis.html#services-kinesis-eventsourcemapping>`_
    - `Configuring an SQS queue as an event source <https://docs.aws.amazon.com/lambda/latest/dg/with-sqs.html#events-sqs-eventsource>`_
    - `Configuring an MQ broker as an event source <https://docs.aws.amazon.com/lambda/latest/dg/with-mq.html#services-mq-eventsourcemapping>`_
    - `Configuring MSK as an event source <https://docs.aws.amazon.com/lambda/latest/dg/with-msk.html>`_
    - `Configuring Self-Managed Apache Kafka as an event source <https://docs.aws.amazon.com/lambda/latest/dg/kafka-smaa.html>`_

    :cloudformationResource: AWS::Lambda::EventSourceMapping
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-eventsourcemapping.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        import aws_cdk.aws_lambda as lambda_
        
        cfn_event_source_mapping = lambda_.CfnEventSourceMapping(self, "MyCfnEventSourceMapping",
            function_name="functionName",
        
            # the properties below are optional
            amazon_managed_kafka_event_source_config=lambda.CfnEventSourceMapping.AmazonManagedKafkaEventSourceConfigProperty(
                consumer_group_id="consumerGroupId"
            ),
            batch_size=123,
            bisect_batch_on_function_error=False,
            destination_config=lambda.CfnEventSourceMapping.DestinationConfigProperty(
                on_failure=lambda.CfnEventSourceMapping.OnFailureProperty(
                    destination="destination"
                )
            ),
            enabled=False,
            event_source_arn="eventSourceArn",
            filter_criteria=lambda.CfnEventSourceMapping.FilterCriteriaProperty(
                filters=[lambda.CfnEventSourceMapping.FilterProperty(
                    pattern="pattern"
                )]
            ),
            function_response_types=["functionResponseTypes"],
            maximum_batching_window_in_seconds=123,
            maximum_record_age_in_seconds=123,
            maximum_retry_attempts=123,
            parallelization_factor=123,
            queues=["queues"],
            scaling_config=lambda.CfnEventSourceMapping.ScalingConfigProperty(
                maximum_concurrency=123
            ),
            self_managed_event_source=lambda.CfnEventSourceMapping.SelfManagedEventSourceProperty(
                endpoints=lambda.CfnEventSourceMapping.EndpointsProperty(
                    kafka_bootstrap_servers=["kafkaBootstrapServers"]
                )
            ),
            self_managed_kafka_event_source_config=lambda.CfnEventSourceMapping.SelfManagedKafkaEventSourceConfigProperty(
                consumer_group_id="consumerGroupId"
            ),
            source_access_configurations=[lambda.CfnEventSourceMapping.SourceAccessConfigurationProperty(
                type="type",
                uri="uri"
            )],
            starting_position="startingPosition",
            starting_position_timestamp=123,
            topics=["topics"],
            tumbling_window_in_seconds=123
        )
    '''

    def __init__(
        self,
        scope: _aws_cdk_core_f4b25747.Construct,
        id: builtins.str,
        *,
        function_name: builtins.str,
        amazon_managed_kafka_event_source_config: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnEventSourceMapping.AmazonManagedKafkaEventSourceConfigProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        batch_size: typing.Optional[jsii.Number] = None,
        bisect_batch_on_function_error: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]] = None,
        destination_config: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnEventSourceMapping.DestinationConfigProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        enabled: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]] = None,
        event_source_arn: typing.Optional[builtins.str] = None,
        filter_criteria: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnEventSourceMapping.FilterCriteriaProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        function_response_types: typing.Optional[typing.Sequence[builtins.str]] = None,
        maximum_batching_window_in_seconds: typing.Optional[jsii.Number] = None,
        maximum_record_age_in_seconds: typing.Optional[jsii.Number] = None,
        maximum_retry_attempts: typing.Optional[jsii.Number] = None,
        parallelization_factor: typing.Optional[jsii.Number] = None,
        queues: typing.Optional[typing.Sequence[builtins.str]] = None,
        scaling_config: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnEventSourceMapping.ScalingConfigProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        self_managed_event_source: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnEventSourceMapping.SelfManagedEventSourceProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        self_managed_kafka_event_source_config: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnEventSourceMapping.SelfManagedKafkaEventSourceConfigProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        source_access_configurations: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Sequence[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnEventSourceMapping.SourceAccessConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
        starting_position: typing.Optional[builtins.str] = None,
        starting_position_timestamp: typing.Optional[jsii.Number] = None,
        topics: typing.Optional[typing.Sequence[builtins.str]] = None,
        tumbling_window_in_seconds: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''Create a new ``AWS::Lambda::EventSourceMapping``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param function_name: The name of the Lambda function. **Name formats** - *Function name* - ``MyFunction`` . - *Function ARN* - ``arn:aws:lambda:us-west-2:123456789012:function:MyFunction`` . - *Version or Alias ARN* - ``arn:aws:lambda:us-west-2:123456789012:function:MyFunction:PROD`` . - *Partial ARN* - ``123456789012:function:MyFunction`` . The length constraint applies only to the full ARN. If you specify only the function name, it's limited to 64 characters in length.
        :param amazon_managed_kafka_event_source_config: Specific configuration settings for an Amazon Managed Streaming for Apache Kafka (Amazon MSK) event source.
        :param batch_size: The maximum number of records in each batch that Lambda pulls from your stream or queue and sends to your function. Lambda passes all of the records in the batch to the function in a single call, up to the payload limit for synchronous invocation (6 MB). - *Amazon Kinesis* - Default 100. Max 10,000. - *Amazon DynamoDB Streams* - Default 100. Max 10,000. - *Amazon Simple Queue Service* - Default 10. For standard queues the max is 10,000. For FIFO queues the max is 10. - *Amazon Managed Streaming for Apache Kafka* - Default 100. Max 10,000. - *Self-managed Apache Kafka* - Default 100. Max 10,000. - *Amazon MQ (ActiveMQ and RabbitMQ)* - Default 100. Max 10,000.
        :param bisect_batch_on_function_error: (Streams only) If the function returns an error, split the batch in two and retry. The default value is false.
        :param destination_config: (Streams only) An Amazon SQS queue or Amazon SNS topic destination for discarded records.
        :param enabled: When true, the event source mapping is active. When false, Lambda pauses polling and invocation. Default: True
        :param event_source_arn: The Amazon Resource Name (ARN) of the event source. - *Amazon Kinesis* - The ARN of the data stream or a stream consumer. - *Amazon DynamoDB Streams* - The ARN of the stream. - *Amazon Simple Queue Service* - The ARN of the queue. - *Amazon Managed Streaming for Apache Kafka* - The ARN of the cluster. - *Amazon MQ* - The ARN of the broker.
        :param filter_criteria: An object that defines the filter criteria that determine whether Lambda should process an event. For more information, see `Lambda event filtering <https://docs.aws.amazon.com/lambda/latest/dg/invocation-eventfiltering.html>`_ .
        :param function_response_types: (Streams and SQS) A list of current response type enums applied to the event source mapping. Valid Values: ``ReportBatchItemFailures``
        :param maximum_batching_window_in_seconds: The maximum amount of time, in seconds, that Lambda spends gathering records before invoking the function. *Default ( Kinesis , DynamoDB , Amazon SQS event sources)* : 0 *Default ( Amazon MSK , Kafka, Amazon MQ event sources)* : 500 ms *Related setting:* When you set ``BatchSize`` to a value greater than 10, you must set ``MaximumBatchingWindowInSeconds`` to at least 1.
        :param maximum_record_age_in_seconds: (Streams only) Discard records older than the specified age. The default value is -1, which sets the maximum age to infinite. When the value is set to infinite, Lambda never discards old records.
        :param maximum_retry_attempts: (Streams only) Discard records after the specified number of retries. The default value is -1, which sets the maximum number of retries to infinite. When MaximumRetryAttempts is infinite, Lambda retries failed records until the record expires in the event source.
        :param parallelization_factor: (Streams only) The number of batches to process concurrently from each shard. The default value is 1.
        :param queues: (Amazon MQ) The name of the Amazon MQ broker destination queue to consume.
        :param scaling_config: ``AWS::Lambda::EventSourceMapping.ScalingConfig``.
        :param self_managed_event_source: The self-managed Apache Kafka cluster for your event source.
        :param self_managed_kafka_event_source_config: Specific configuration settings for a self-managed Apache Kafka event source.
        :param source_access_configurations: An array of the authentication protocol, VPC components, or virtual host to secure and define your event source.
        :param starting_position: The position in a stream from which to start reading. Required for Amazon Kinesis and Amazon DynamoDB. - *LATEST* - Read only new records. - *TRIM_HORIZON* - Process all available records. - *AT_TIMESTAMP* - Specify a time from which to start reading records.
        :param starting_position_timestamp: With ``StartingPosition`` set to ``AT_TIMESTAMP`` , the time from which to start reading, in Unix time seconds.
        :param topics: The name of the Kafka topic.
        :param tumbling_window_in_seconds: (Streams only) The duration in seconds of a processing window. The range is between 1 second and 900 seconds.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__bc51e76aa3f826f4f2a1badfca6f79c185ce90b773c6e4e6d60462c15f89095f)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnEventSourceMappingProps(
            function_name=function_name,
            amazon_managed_kafka_event_source_config=amazon_managed_kafka_event_source_config,
            batch_size=batch_size,
            bisect_batch_on_function_error=bisect_batch_on_function_error,
            destination_config=destination_config,
            enabled=enabled,
            event_source_arn=event_source_arn,
            filter_criteria=filter_criteria,
            function_response_types=function_response_types,
            maximum_batching_window_in_seconds=maximum_batching_window_in_seconds,
            maximum_record_age_in_seconds=maximum_record_age_in_seconds,
            maximum_retry_attempts=maximum_retry_attempts,
            parallelization_factor=parallelization_factor,
            queues=queues,
            scaling_config=scaling_config,
            self_managed_event_source=self_managed_event_source,
            self_managed_kafka_event_source_config=self_managed_kafka_event_source_config,
            source_access_configurations=source_access_configurations,
            starting_position=starting_position,
            starting_position_timestamp=starting_position_timestamp,
            topics=topics,
            tumbling_window_in_seconds=tumbling_window_in_seconds,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _aws_cdk_core_f4b25747.TreeInspector) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5c7fd79fe740cce31f88fdf7168df03b1fcd7b73056b78f1be208652c092d29a)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e1a854639f1238fb76d6612e626aa2579ec1fd486ca20c22c0734b7771ef15f4)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrId")
    def attr_id(self) -> builtins.str:
        '''The event source mapping's ID.

        :cloudformationAttribute: Id
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrId"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="functionName")
    def function_name(self) -> builtins.str:
        '''The name of the Lambda function.

        **Name formats** - *Function name* - ``MyFunction`` .

        - *Function ARN* - ``arn:aws:lambda:us-west-2:123456789012:function:MyFunction`` .
        - *Version or Alias ARN* - ``arn:aws:lambda:us-west-2:123456789012:function:MyFunction:PROD`` .
        - *Partial ARN* - ``123456789012:function:MyFunction`` .

        The length constraint applies only to the full ARN. If you specify only the function name, it's limited to 64 characters in length.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-eventsourcemapping.html#cfn-lambda-eventsourcemapping-functionname
        '''
        return typing.cast(builtins.str, jsii.get(self, "functionName"))

    @function_name.setter
    def function_name(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__54354783301d7cc505d3584395d6b6b6f8d7f3aa1c9ecbcce2a68eb86c7bd327)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "functionName", value)

    @builtins.property
    @jsii.member(jsii_name="amazonManagedKafkaEventSourceConfig")
    def amazon_managed_kafka_event_source_config(
        self,
    ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnEventSourceMapping.AmazonManagedKafkaEventSourceConfigProperty"]]:
        '''Specific configuration settings for an Amazon Managed Streaming for Apache Kafka (Amazon MSK) event source.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-eventsourcemapping.html#cfn-lambda-eventsourcemapping-amazonmanagedkafkaeventsourceconfig
        '''
        return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnEventSourceMapping.AmazonManagedKafkaEventSourceConfigProperty"]], jsii.get(self, "amazonManagedKafkaEventSourceConfig"))

    @amazon_managed_kafka_event_source_config.setter
    def amazon_managed_kafka_event_source_config(
        self,
        value: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnEventSourceMapping.AmazonManagedKafkaEventSourceConfigProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f543ec58a5801c466d029351d866006c0f8e0483aa02bb5c33041d00f244b4db)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "amazonManagedKafkaEventSourceConfig", value)

    @builtins.property
    @jsii.member(jsii_name="batchSize")
    def batch_size(self) -> typing.Optional[jsii.Number]:
        '''The maximum number of records in each batch that Lambda pulls from your stream or queue and sends to your function.

        Lambda passes all of the records in the batch to the function in a single call, up to the payload limit for synchronous invocation (6 MB).

        - *Amazon Kinesis* - Default 100. Max 10,000.
        - *Amazon DynamoDB Streams* - Default 100. Max 10,000.
        - *Amazon Simple Queue Service* - Default 10. For standard queues the max is 10,000. For FIFO queues the max is 10.
        - *Amazon Managed Streaming for Apache Kafka* - Default 100. Max 10,000.
        - *Self-managed Apache Kafka* - Default 100. Max 10,000.
        - *Amazon MQ (ActiveMQ and RabbitMQ)* - Default 100. Max 10,000.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-eventsourcemapping.html#cfn-lambda-eventsourcemapping-batchsize
        '''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "batchSize"))

    @batch_size.setter
    def batch_size(self, value: typing.Optional[jsii.Number]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5ab3ed6d98dbe5642581b16c5b4e24f29827281bfbe5ed4840abf252011f69ed)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "batchSize", value)

    @builtins.property
    @jsii.member(jsii_name="bisectBatchOnFunctionError")
    def bisect_batch_on_function_error(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]]:
        '''(Streams only) If the function returns an error, split the batch in two and retry.

        The default value is false.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-eventsourcemapping.html#cfn-lambda-eventsourcemapping-bisectbatchonfunctionerror
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]], jsii.get(self, "bisectBatchOnFunctionError"))

    @bisect_batch_on_function_error.setter
    def bisect_batch_on_function_error(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ece00c6aba32140edec6db87dfdd762346aec4398ab1108e0ba4e3a0d52052b2)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "bisectBatchOnFunctionError", value)

    @builtins.property
    @jsii.member(jsii_name="destinationConfig")
    def destination_config(
        self,
    ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnEventSourceMapping.DestinationConfigProperty"]]:
        '''(Streams only) An Amazon SQS queue or Amazon SNS topic destination for discarded records.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-eventsourcemapping.html#cfn-lambda-eventsourcemapping-destinationconfig
        '''
        return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnEventSourceMapping.DestinationConfigProperty"]], jsii.get(self, "destinationConfig"))

    @destination_config.setter
    def destination_config(
        self,
        value: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnEventSourceMapping.DestinationConfigProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b8bd92740ae363491f2b332a5f4b8930d13a4d60d6c74d6052d0340167bf06da)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "destinationConfig", value)

    @builtins.property
    @jsii.member(jsii_name="enabled")
    def enabled(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]]:
        '''When true, the event source mapping is active. When false, Lambda pauses polling and invocation.

        Default: True

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-eventsourcemapping.html#cfn-lambda-eventsourcemapping-enabled
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]], jsii.get(self, "enabled"))

    @enabled.setter
    def enabled(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e16a8c512ffd1d0255c5ae9be3508065f209040f0bf53032d3c8877182bb3eca)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "enabled", value)

    @builtins.property
    @jsii.member(jsii_name="eventSourceArn")
    def event_source_arn(self) -> typing.Optional[builtins.str]:
        '''The Amazon Resource Name (ARN) of the event source.

        - *Amazon Kinesis* - The ARN of the data stream or a stream consumer.
        - *Amazon DynamoDB Streams* - The ARN of the stream.
        - *Amazon Simple Queue Service* - The ARN of the queue.
        - *Amazon Managed Streaming for Apache Kafka* - The ARN of the cluster.
        - *Amazon MQ* - The ARN of the broker.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-eventsourcemapping.html#cfn-lambda-eventsourcemapping-eventsourcearn
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "eventSourceArn"))

    @event_source_arn.setter
    def event_source_arn(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a3a0b924993ac3f8adacb2eb606f79f3dd14b6aff3f5d214e5219943766e1721)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "eventSourceArn", value)

    @builtins.property
    @jsii.member(jsii_name="filterCriteria")
    def filter_criteria(
        self,
    ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnEventSourceMapping.FilterCriteriaProperty"]]:
        '''An object that defines the filter criteria that determine whether Lambda should process an event.

        For more information, see `Lambda event filtering <https://docs.aws.amazon.com/lambda/latest/dg/invocation-eventfiltering.html>`_ .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-eventsourcemapping.html#cfn-lambda-eventsourcemapping-filtercriteria
        '''
        return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnEventSourceMapping.FilterCriteriaProperty"]], jsii.get(self, "filterCriteria"))

    @filter_criteria.setter
    def filter_criteria(
        self,
        value: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnEventSourceMapping.FilterCriteriaProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__05192938476939f3e97429f01e37cca2a6be5c3d1c2fa715b8829db3876f402b)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "filterCriteria", value)

    @builtins.property
    @jsii.member(jsii_name="functionResponseTypes")
    def function_response_types(self) -> typing.Optional[typing.List[builtins.str]]:
        '''(Streams and SQS) A list of current response type enums applied to the event source mapping.

        Valid Values: ``ReportBatchItemFailures``

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-eventsourcemapping.html#cfn-lambda-eventsourcemapping-functionresponsetypes
        '''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "functionResponseTypes"))

    @function_response_types.setter
    def function_response_types(
        self,
        value: typing.Optional[typing.List[builtins.str]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__883d032729003e0a8df926ba9811e3b8bc3923fca6adac033cdcaaee9672f94d)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "functionResponseTypes", value)

    @builtins.property
    @jsii.member(jsii_name="maximumBatchingWindowInSeconds")
    def maximum_batching_window_in_seconds(self) -> typing.Optional[jsii.Number]:
        '''The maximum amount of time, in seconds, that Lambda spends gathering records before invoking the function.

        *Default ( Kinesis , DynamoDB , Amazon SQS event sources)* : 0

        *Default ( Amazon MSK , Kafka, Amazon MQ event sources)* : 500 ms

        *Related setting:* When you set ``BatchSize`` to a value greater than 10, you must set ``MaximumBatchingWindowInSeconds`` to at least 1.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-eventsourcemapping.html#cfn-lambda-eventsourcemapping-maximumbatchingwindowinseconds
        '''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "maximumBatchingWindowInSeconds"))

    @maximum_batching_window_in_seconds.setter
    def maximum_batching_window_in_seconds(
        self,
        value: typing.Optional[jsii.Number],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2122237078faf08538debc167f65a1723c5c5a536305d67f9f569657ffec3b59)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "maximumBatchingWindowInSeconds", value)

    @builtins.property
    @jsii.member(jsii_name="maximumRecordAgeInSeconds")
    def maximum_record_age_in_seconds(self) -> typing.Optional[jsii.Number]:
        '''(Streams only) Discard records older than the specified age.

        The default value is -1,
        which sets the maximum age to infinite. When the value is set to infinite, Lambda never discards old records.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-eventsourcemapping.html#cfn-lambda-eventsourcemapping-maximumrecordageinseconds
        '''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "maximumRecordAgeInSeconds"))

    @maximum_record_age_in_seconds.setter
    def maximum_record_age_in_seconds(
        self,
        value: typing.Optional[jsii.Number],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2e764275415de01b0380a8467c70eecf479094cc165019d4fb296d4ef4035bd9)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "maximumRecordAgeInSeconds", value)

    @builtins.property
    @jsii.member(jsii_name="maximumRetryAttempts")
    def maximum_retry_attempts(self) -> typing.Optional[jsii.Number]:
        '''(Streams only) Discard records after the specified number of retries.

        The default value is -1,
        which sets the maximum number of retries to infinite. When MaximumRetryAttempts is infinite, Lambda retries failed records until the record expires in the event source.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-eventsourcemapping.html#cfn-lambda-eventsourcemapping-maximumretryattempts
        '''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "maximumRetryAttempts"))

    @maximum_retry_attempts.setter
    def maximum_retry_attempts(self, value: typing.Optional[jsii.Number]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__fa22436b9d5cac73d3bed88d0d997d113b13193c9f229788ae05578b24d3d130)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "maximumRetryAttempts", value)

    @builtins.property
    @jsii.member(jsii_name="parallelizationFactor")
    def parallelization_factor(self) -> typing.Optional[jsii.Number]:
        '''(Streams only) The number of batches to process concurrently from each shard.

        The default value is 1.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-eventsourcemapping.html#cfn-lambda-eventsourcemapping-parallelizationfactor
        '''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "parallelizationFactor"))

    @parallelization_factor.setter
    def parallelization_factor(self, value: typing.Optional[jsii.Number]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e0181e2cb21156edc251b3ac39bda78914b7f4949a2dace42b0be42d98042831)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "parallelizationFactor", value)

    @builtins.property
    @jsii.member(jsii_name="queues")
    def queues(self) -> typing.Optional[typing.List[builtins.str]]:
        '''(Amazon MQ) The name of the Amazon MQ broker destination queue to consume.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-eventsourcemapping.html#cfn-lambda-eventsourcemapping-queues
        '''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "queues"))

    @queues.setter
    def queues(self, value: typing.Optional[typing.List[builtins.str]]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9b436f1aa865b0d892b109bd8e111ebd5bfd09468d5ce454d823a5a751972bb1)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "queues", value)

    @builtins.property
    @jsii.member(jsii_name="scalingConfig")
    def scaling_config(
        self,
    ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnEventSourceMapping.ScalingConfigProperty"]]:
        '''``AWS::Lambda::EventSourceMapping.ScalingConfig``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-eventsourcemapping.html#cfn-lambda-eventsourcemapping-scalingconfig
        '''
        return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnEventSourceMapping.ScalingConfigProperty"]], jsii.get(self, "scalingConfig"))

    @scaling_config.setter
    def scaling_config(
        self,
        value: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnEventSourceMapping.ScalingConfigProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0e2da0cd34b2d7c0997e7ee2dc6630b1a83e11318b1472add78070482eade46f)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "scalingConfig", value)

    @builtins.property
    @jsii.member(jsii_name="selfManagedEventSource")
    def self_managed_event_source(
        self,
    ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnEventSourceMapping.SelfManagedEventSourceProperty"]]:
        '''The self-managed Apache Kafka cluster for your event source.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-eventsourcemapping.html#cfn-lambda-eventsourcemapping-selfmanagedeventsource
        '''
        return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnEventSourceMapping.SelfManagedEventSourceProperty"]], jsii.get(self, "selfManagedEventSource"))

    @self_managed_event_source.setter
    def self_managed_event_source(
        self,
        value: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnEventSourceMapping.SelfManagedEventSourceProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ba4a027b0fe756f86d0a0fce9e7b2d3efd82425587344d43d440330a87d07d1e)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "selfManagedEventSource", value)

    @builtins.property
    @jsii.member(jsii_name="selfManagedKafkaEventSourceConfig")
    def self_managed_kafka_event_source_config(
        self,
    ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnEventSourceMapping.SelfManagedKafkaEventSourceConfigProperty"]]:
        '''Specific configuration settings for a self-managed Apache Kafka event source.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-eventsourcemapping.html#cfn-lambda-eventsourcemapping-selfmanagedkafkaeventsourceconfig
        '''
        return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnEventSourceMapping.SelfManagedKafkaEventSourceConfigProperty"]], jsii.get(self, "selfManagedKafkaEventSourceConfig"))

    @self_managed_kafka_event_source_config.setter
    def self_managed_kafka_event_source_config(
        self,
        value: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnEventSourceMapping.SelfManagedKafkaEventSourceConfigProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c4e281717a38b86ec397a4e03e280f4d639785b8839065bfbb13b71877aae761)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "selfManagedKafkaEventSourceConfig", value)

    @builtins.property
    @jsii.member(jsii_name="sourceAccessConfigurations")
    def source_access_configurations(
        self,
    ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.List[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnEventSourceMapping.SourceAccessConfigurationProperty"]]]]:
        '''An array of the authentication protocol, VPC components, or virtual host to secure and define your event source.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-eventsourcemapping.html#cfn-lambda-eventsourcemapping-sourceaccessconfigurations
        '''
        return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.List[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnEventSourceMapping.SourceAccessConfigurationProperty"]]]], jsii.get(self, "sourceAccessConfigurations"))

    @source_access_configurations.setter
    def source_access_configurations(
        self,
        value: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.List[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnEventSourceMapping.SourceAccessConfigurationProperty"]]]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e033a472e25a285a08ded3f7dbabbe66df4179a4097cab5b7a9632eb857f9a54)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "sourceAccessConfigurations", value)

    @builtins.property
    @jsii.member(jsii_name="startingPosition")
    def starting_position(self) -> typing.Optional[builtins.str]:
        '''The position in a stream from which to start reading. Required for Amazon Kinesis and Amazon DynamoDB.

        - *LATEST* - Read only new records.
        - *TRIM_HORIZON* - Process all available records.
        - *AT_TIMESTAMP* - Specify a time from which to start reading records.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-eventsourcemapping.html#cfn-lambda-eventsourcemapping-startingposition
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "startingPosition"))

    @starting_position.setter
    def starting_position(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__16baa1d5627ef0f84228cd21a6a42c33f56a42d0b63dcf00c13974c76e92d2fb)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "startingPosition", value)

    @builtins.property
    @jsii.member(jsii_name="startingPositionTimestamp")
    def starting_position_timestamp(self) -> typing.Optional[jsii.Number]:
        '''With ``StartingPosition`` set to ``AT_TIMESTAMP`` , the time from which to start reading, in Unix time seconds.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-eventsourcemapping.html#cfn-lambda-eventsourcemapping-startingpositiontimestamp
        '''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "startingPositionTimestamp"))

    @starting_position_timestamp.setter
    def starting_position_timestamp(self, value: typing.Optional[jsii.Number]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9a473ebfe96e86f045a6147de22428bef9c2e582e7706b098ff3621fb62eb6e7)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "startingPositionTimestamp", value)

    @builtins.property
    @jsii.member(jsii_name="topics")
    def topics(self) -> typing.Optional[typing.List[builtins.str]]:
        '''The name of the Kafka topic.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-eventsourcemapping.html#cfn-lambda-eventsourcemapping-topics
        '''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "topics"))

    @topics.setter
    def topics(self, value: typing.Optional[typing.List[builtins.str]]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__cad5a3e0f85893d512c79014062c45b1bf4f49e34040cf95545c8c603f8c7bf4)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "topics", value)

    @builtins.property
    @jsii.member(jsii_name="tumblingWindowInSeconds")
    def tumbling_window_in_seconds(self) -> typing.Optional[jsii.Number]:
        '''(Streams only) The duration in seconds of a processing window.

        The range is between 1 second and 900 seconds.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-eventsourcemapping.html#cfn-lambda-eventsourcemapping-tumblingwindowinseconds
        '''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "tumblingWindowInSeconds"))

    @tumbling_window_in_seconds.setter
    def tumbling_window_in_seconds(self, value: typing.Optional[jsii.Number]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3beed7035233c2b2e7a323534d5fc94427c63b3c946390c293f6284d3744a21a)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "tumblingWindowInSeconds", value)

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-lambda.CfnEventSourceMapping.AmazonManagedKafkaEventSourceConfigProperty",
        jsii_struct_bases=[],
        name_mapping={"consumer_group_id": "consumerGroupId"},
    )
    class AmazonManagedKafkaEventSourceConfigProperty:
        def __init__(
            self,
            *,
            consumer_group_id: typing.Optional[builtins.str] = None,
        ) -> None:
            '''Specific configuration settings for an Amazon Managed Streaming for Apache Kafka (Amazon MSK) event source.

            :param consumer_group_id: The identifier for the Kafka consumer group to join. The consumer group ID must be unique among all your Kafka event sources. After creating a Kafka event source mapping with the consumer group ID specified, you cannot update this value. For more information, see `Customizable consumer group ID <https://docs.aws.amazon.com/lambda/latest/dg/with-msk.html#services-msk-consumer-group-id>`_ .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-eventsourcemapping-amazonmanagedkafkaeventsourceconfig.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_lambda as lambda_
                
                amazon_managed_kafka_event_source_config_property = lambda.CfnEventSourceMapping.AmazonManagedKafkaEventSourceConfigProperty(
                    consumer_group_id="consumerGroupId"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__f4ba03c85ae6e57e3a5e949da494dbc59733310ab132261d479743fc20c2aee0)
                check_type(argname="argument consumer_group_id", value=consumer_group_id, expected_type=type_hints["consumer_group_id"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if consumer_group_id is not None:
                self._values["consumer_group_id"] = consumer_group_id

        @builtins.property
        def consumer_group_id(self) -> typing.Optional[builtins.str]:
            '''The identifier for the Kafka consumer group to join.

            The consumer group ID must be unique among all your Kafka event sources. After creating a Kafka event source mapping with the consumer group ID specified, you cannot update this value. For more information, see `Customizable consumer group ID <https://docs.aws.amazon.com/lambda/latest/dg/with-msk.html#services-msk-consumer-group-id>`_ .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-eventsourcemapping-amazonmanagedkafkaeventsourceconfig.html#cfn-lambda-eventsourcemapping-amazonmanagedkafkaeventsourceconfig-consumergroupid
            '''
            result = self._values.get("consumer_group_id")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "AmazonManagedKafkaEventSourceConfigProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-lambda.CfnEventSourceMapping.DestinationConfigProperty",
        jsii_struct_bases=[],
        name_mapping={"on_failure": "onFailure"},
    )
    class DestinationConfigProperty:
        def __init__(
            self,
            *,
            on_failure: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnEventSourceMapping.OnFailureProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''A configuration object that specifies the destination of an event after Lambda processes it.

            :param on_failure: The destination configuration for failed invocations.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-eventsourcemapping-destinationconfig.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_lambda as lambda_
                
                destination_config_property = lambda.CfnEventSourceMapping.DestinationConfigProperty(
                    on_failure=lambda.CfnEventSourceMapping.OnFailureProperty(
                        destination="destination"
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__6557b3836bfb3b691f2cd7323c2258fa92d4fa65ed56e4cade3cd0d1249285ca)
                check_type(argname="argument on_failure", value=on_failure, expected_type=type_hints["on_failure"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if on_failure is not None:
                self._values["on_failure"] = on_failure

        @builtins.property
        def on_failure(
            self,
        ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnEventSourceMapping.OnFailureProperty"]]:
            '''The destination configuration for failed invocations.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-eventsourcemapping-destinationconfig.html#cfn-lambda-eventsourcemapping-destinationconfig-onfailure
            '''
            result = self._values.get("on_failure")
            return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnEventSourceMapping.OnFailureProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "DestinationConfigProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-lambda.CfnEventSourceMapping.EndpointsProperty",
        jsii_struct_bases=[],
        name_mapping={"kafka_bootstrap_servers": "kafkaBootstrapServers"},
    )
    class EndpointsProperty:
        def __init__(
            self,
            *,
            kafka_bootstrap_servers: typing.Optional[typing.Sequence[builtins.str]] = None,
        ) -> None:
            '''The list of bootstrap servers for your Kafka brokers in the following format: ``"KafkaBootstrapServers": ["abc.xyz.com:xxxx","abc2.xyz.com:xxxx"]`` .

            :param kafka_bootstrap_servers: The list of bootstrap servers for your Kafka brokers in the following format: ``"KafkaBootstrapServers": ["abc.xyz.com:xxxx","abc2.xyz.com:xxxx"]`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-eventsourcemapping-endpoints.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_lambda as lambda_
                
                endpoints_property = lambda.CfnEventSourceMapping.EndpointsProperty(
                    kafka_bootstrap_servers=["kafkaBootstrapServers"]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__0c473c1c3f65b8a86fb1d9f5c57ed756962347173be06dcaf312fbbaa8045228)
                check_type(argname="argument kafka_bootstrap_servers", value=kafka_bootstrap_servers, expected_type=type_hints["kafka_bootstrap_servers"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if kafka_bootstrap_servers is not None:
                self._values["kafka_bootstrap_servers"] = kafka_bootstrap_servers

        @builtins.property
        def kafka_bootstrap_servers(self) -> typing.Optional[typing.List[builtins.str]]:
            '''The list of bootstrap servers for your Kafka brokers in the following format: ``"KafkaBootstrapServers": ["abc.xyz.com:xxxx","abc2.xyz.com:xxxx"]`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-eventsourcemapping-endpoints.html#cfn-lambda-eventsourcemapping-endpoints-kafkabootstrapservers
            '''
            result = self._values.get("kafka_bootstrap_servers")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "EndpointsProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-lambda.CfnEventSourceMapping.FilterCriteriaProperty",
        jsii_struct_bases=[],
        name_mapping={"filters": "filters"},
    )
    class FilterCriteriaProperty:
        def __init__(
            self,
            *,
            filters: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Sequence[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnEventSourceMapping.FilterProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
        ) -> None:
            '''An object that contains the filters for an event source.

            :param filters: A list of filters.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-eventsourcemapping-filtercriteria.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_lambda as lambda_
                
                filter_criteria_property = lambda.CfnEventSourceMapping.FilterCriteriaProperty(
                    filters=[lambda.CfnEventSourceMapping.FilterProperty(
                        pattern="pattern"
                    )]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__d422f268dca4f17ad386647c6efa5005621a180cab41b86cfff1f2d8f9cb4575)
                check_type(argname="argument filters", value=filters, expected_type=type_hints["filters"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if filters is not None:
                self._values["filters"] = filters

        @builtins.property
        def filters(
            self,
        ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.List[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnEventSourceMapping.FilterProperty"]]]]:
            '''A list of filters.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-eventsourcemapping-filtercriteria.html#cfn-lambda-eventsourcemapping-filtercriteria-filters
            '''
            result = self._values.get("filters")
            return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.List[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnEventSourceMapping.FilterProperty"]]]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "FilterCriteriaProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-lambda.CfnEventSourceMapping.FilterProperty",
        jsii_struct_bases=[],
        name_mapping={"pattern": "pattern"},
    )
    class FilterProperty:
        def __init__(self, *, pattern: typing.Optional[builtins.str] = None) -> None:
            '''A structure within a ``FilterCriteria`` object that defines an event filtering pattern.

            :param pattern: A filter pattern. For more information on the syntax of a filter pattern, see `Filter rule syntax <https://docs.aws.amazon.com/lambda/latest/dg/invocation-eventfiltering.html#filtering-syntax>`_ .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-eventsourcemapping-filter.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_lambda as lambda_
                
                filter_property = lambda.CfnEventSourceMapping.FilterProperty(
                    pattern="pattern"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__16465f5a00e31a1a5014cd5b351d9b178516026248f5bbeb85e29617a9fbb614)
                check_type(argname="argument pattern", value=pattern, expected_type=type_hints["pattern"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if pattern is not None:
                self._values["pattern"] = pattern

        @builtins.property
        def pattern(self) -> typing.Optional[builtins.str]:
            '''A filter pattern.

            For more information on the syntax of a filter pattern, see `Filter rule syntax <https://docs.aws.amazon.com/lambda/latest/dg/invocation-eventfiltering.html#filtering-syntax>`_ .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-eventsourcemapping-filter.html#cfn-lambda-eventsourcemapping-filter-pattern
            '''
            result = self._values.get("pattern")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "FilterProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-lambda.CfnEventSourceMapping.OnFailureProperty",
        jsii_struct_bases=[],
        name_mapping={"destination": "destination"},
    )
    class OnFailureProperty:
        def __init__(
            self,
            *,
            destination: typing.Optional[builtins.str] = None,
        ) -> None:
            '''A destination for events that failed processing.

            :param destination: The Amazon Resource Name (ARN) of the destination resource.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-eventsourcemapping-onfailure.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_lambda as lambda_
                
                on_failure_property = lambda.CfnEventSourceMapping.OnFailureProperty(
                    destination="destination"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__ad946b051c139cdefdbbd228178e9d7bf58bfc7f1f022e79ad81d5ea5f165b93)
                check_type(argname="argument destination", value=destination, expected_type=type_hints["destination"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if destination is not None:
                self._values["destination"] = destination

        @builtins.property
        def destination(self) -> typing.Optional[builtins.str]:
            '''The Amazon Resource Name (ARN) of the destination resource.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-eventsourcemapping-onfailure.html#cfn-lambda-eventsourcemapping-onfailure-destination
            '''
            result = self._values.get("destination")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "OnFailureProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-lambda.CfnEventSourceMapping.ScalingConfigProperty",
        jsii_struct_bases=[],
        name_mapping={"maximum_concurrency": "maximumConcurrency"},
    )
    class ScalingConfigProperty:
        def __init__(
            self,
            *,
            maximum_concurrency: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''
            :param maximum_concurrency: ``CfnEventSourceMapping.ScalingConfigProperty.MaximumConcurrency``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-eventsourcemapping-scalingconfig.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_lambda as lambda_
                
                scaling_config_property = lambda.CfnEventSourceMapping.ScalingConfigProperty(
                    maximum_concurrency=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__7b38e8c17fd49ebe5faf365df04a671a4cf8e7b13b4c7f7c2d68f10e2bfdcbb9)
                check_type(argname="argument maximum_concurrency", value=maximum_concurrency, expected_type=type_hints["maximum_concurrency"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if maximum_concurrency is not None:
                self._values["maximum_concurrency"] = maximum_concurrency

        @builtins.property
        def maximum_concurrency(self) -> typing.Optional[jsii.Number]:
            '''``CfnEventSourceMapping.ScalingConfigProperty.MaximumConcurrency``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-eventsourcemapping-scalingconfig.html#cfn-lambda-eventsourcemapping-scalingconfig-maximumconcurrency
            '''
            result = self._values.get("maximum_concurrency")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ScalingConfigProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-lambda.CfnEventSourceMapping.SelfManagedEventSourceProperty",
        jsii_struct_bases=[],
        name_mapping={"endpoints": "endpoints"},
    )
    class SelfManagedEventSourceProperty:
        def __init__(
            self,
            *,
            endpoints: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnEventSourceMapping.EndpointsProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''The self-managed Apache Kafka cluster for your event source.

            :param endpoints: The list of bootstrap servers for your Kafka brokers in the following format: ``"KafkaBootstrapServers": ["abc.xyz.com:xxxx","abc2.xyz.com:xxxx"]`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-eventsourcemapping-selfmanagedeventsource.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_lambda as lambda_
                
                self_managed_event_source_property = lambda.CfnEventSourceMapping.SelfManagedEventSourceProperty(
                    endpoints=lambda.CfnEventSourceMapping.EndpointsProperty(
                        kafka_bootstrap_servers=["kafkaBootstrapServers"]
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__5151afc9ab275d50e06a69a800359d9acb3694d830478489a9c9ca0d19b81deb)
                check_type(argname="argument endpoints", value=endpoints, expected_type=type_hints["endpoints"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if endpoints is not None:
                self._values["endpoints"] = endpoints

        @builtins.property
        def endpoints(
            self,
        ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnEventSourceMapping.EndpointsProperty"]]:
            '''The list of bootstrap servers for your Kafka brokers in the following format: ``"KafkaBootstrapServers": ["abc.xyz.com:xxxx","abc2.xyz.com:xxxx"]`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-eventsourcemapping-selfmanagedeventsource.html#cfn-lambda-eventsourcemapping-selfmanagedeventsource-endpoints
            '''
            result = self._values.get("endpoints")
            return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnEventSourceMapping.EndpointsProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "SelfManagedEventSourceProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-lambda.CfnEventSourceMapping.SelfManagedKafkaEventSourceConfigProperty",
        jsii_struct_bases=[],
        name_mapping={"consumer_group_id": "consumerGroupId"},
    )
    class SelfManagedKafkaEventSourceConfigProperty:
        def __init__(
            self,
            *,
            consumer_group_id: typing.Optional[builtins.str] = None,
        ) -> None:
            '''Specific configuration settings for a self-managed Apache Kafka event source.

            :param consumer_group_id: The identifier for the Kafka consumer group to join. The consumer group ID must be unique among all your Kafka event sources. After creating a Kafka event source mapping with the consumer group ID specified, you cannot update this value. For more information, see `Customizable consumer group ID <https://docs.aws.amazon.com/lambda/latest/dg/with-msk.html#services-msk-consumer-group-id>`_ .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-eventsourcemapping-selfmanagedkafkaeventsourceconfig.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_lambda as lambda_
                
                self_managed_kafka_event_source_config_property = lambda.CfnEventSourceMapping.SelfManagedKafkaEventSourceConfigProperty(
                    consumer_group_id="consumerGroupId"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__29a9378f62686b4054c1048c570e5dbcc7e5d5def73225c23008097d529a1157)
                check_type(argname="argument consumer_group_id", value=consumer_group_id, expected_type=type_hints["consumer_group_id"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if consumer_group_id is not None:
                self._values["consumer_group_id"] = consumer_group_id

        @builtins.property
        def consumer_group_id(self) -> typing.Optional[builtins.str]:
            '''The identifier for the Kafka consumer group to join.

            The consumer group ID must be unique among all your Kafka event sources. After creating a Kafka event source mapping with the consumer group ID specified, you cannot update this value. For more information, see `Customizable consumer group ID <https://docs.aws.amazon.com/lambda/latest/dg/with-msk.html#services-msk-consumer-group-id>`_ .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-eventsourcemapping-selfmanagedkafkaeventsourceconfig.html#cfn-lambda-eventsourcemapping-selfmanagedkafkaeventsourceconfig-consumergroupid
            '''
            result = self._values.get("consumer_group_id")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "SelfManagedKafkaEventSourceConfigProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-lambda.CfnEventSourceMapping.SourceAccessConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={"type": "type", "uri": "uri"},
    )
    class SourceAccessConfigurationProperty:
        def __init__(
            self,
            *,
            type: typing.Optional[builtins.str] = None,
            uri: typing.Optional[builtins.str] = None,
        ) -> None:
            '''An array of the authentication protocol, VPC components, or virtual host to secure and define your event source.

            :param type: The type of authentication protocol, VPC components, or virtual host for your event source. For example: ``"Type":"SASL_SCRAM_512_AUTH"`` . - ``BASIC_AUTH`` - (Amazon MQ) The AWS Secrets Manager secret that stores your broker credentials. - ``BASIC_AUTH`` - (Self-managed Apache Kafka) The Secrets Manager ARN of your secret key used for SASL/PLAIN authentication of your Apache Kafka brokers. - ``VPC_SUBNET`` - (Self-managed Apache Kafka) The subnets associated with your VPC. Lambda connects to these subnets to fetch data from your self-managed Apache Kafka cluster. - ``VPC_SECURITY_GROUP`` - (Self-managed Apache Kafka) The VPC security group used to manage access to your self-managed Apache Kafka brokers. - ``SASL_SCRAM_256_AUTH`` - (Self-managed Apache Kafka) The Secrets Manager ARN of your secret key used for SASL SCRAM-256 authentication of your self-managed Apache Kafka brokers. - ``SASL_SCRAM_512_AUTH`` - (Amazon MSK, Self-managed Apache Kafka) The Secrets Manager ARN of your secret key used for SASL SCRAM-512 authentication of your self-managed Apache Kafka brokers. - ``VIRTUAL_HOST`` - (RabbitMQ) The name of the virtual host in your RabbitMQ broker. Lambda uses this RabbitMQ host as the event source. This property cannot be specified in an UpdateEventSourceMapping API call. - ``CLIENT_CERTIFICATE_TLS_AUTH`` - (Amazon MSK, self-managed Apache Kafka) The Secrets Manager ARN of your secret key containing the certificate chain (X.509 PEM), private key (PKCS#8 PEM), and private key password (optional) used for mutual TLS authentication of your MSK/Apache Kafka brokers. - ``SERVER_ROOT_CA_CERTIFICATE`` - (Self-managed Apache Kafka) The Secrets Manager ARN of your secret key containing the root CA certificate (X.509 PEM) used for TLS encryption of your Apache Kafka brokers.
            :param uri: The value for your chosen configuration in ``Type`` . For example: ``"URI": "arn:aws:secretsmanager:us-east-1:01234567890:secret:MyBrokerSecretName"`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-eventsourcemapping-sourceaccessconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_lambda as lambda_
                
                source_access_configuration_property = lambda.CfnEventSourceMapping.SourceAccessConfigurationProperty(
                    type="type",
                    uri="uri"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__5e19ddbf7d0bd0b9863bd5d255657a751f6ee0e4d340199dc521ee6565e3eefd)
                check_type(argname="argument type", value=type, expected_type=type_hints["type"])
                check_type(argname="argument uri", value=uri, expected_type=type_hints["uri"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if type is not None:
                self._values["type"] = type
            if uri is not None:
                self._values["uri"] = uri

        @builtins.property
        def type(self) -> typing.Optional[builtins.str]:
            '''The type of authentication protocol, VPC components, or virtual host for your event source. For example: ``"Type":"SASL_SCRAM_512_AUTH"`` .

            - ``BASIC_AUTH`` - (Amazon MQ) The AWS Secrets Manager secret that stores your broker credentials.
            - ``BASIC_AUTH`` - (Self-managed Apache Kafka) The Secrets Manager ARN of your secret key used for SASL/PLAIN authentication of your Apache Kafka brokers.
            - ``VPC_SUBNET`` - (Self-managed Apache Kafka) The subnets associated with your VPC. Lambda connects to these subnets to fetch data from your self-managed Apache Kafka cluster.
            - ``VPC_SECURITY_GROUP`` - (Self-managed Apache Kafka) The VPC security group used to manage access to your self-managed Apache Kafka brokers.
            - ``SASL_SCRAM_256_AUTH`` - (Self-managed Apache Kafka) The Secrets Manager ARN of your secret key used for SASL SCRAM-256 authentication of your self-managed Apache Kafka brokers.
            - ``SASL_SCRAM_512_AUTH`` - (Amazon MSK, Self-managed Apache Kafka) The Secrets Manager ARN of your secret key used for SASL SCRAM-512 authentication of your self-managed Apache Kafka brokers.
            - ``VIRTUAL_HOST`` - (RabbitMQ) The name of the virtual host in your RabbitMQ broker. Lambda uses this RabbitMQ host as the event source. This property cannot be specified in an UpdateEventSourceMapping API call.
            - ``CLIENT_CERTIFICATE_TLS_AUTH`` - (Amazon MSK, self-managed Apache Kafka) The Secrets Manager ARN of your secret key containing the certificate chain (X.509 PEM), private key (PKCS#8 PEM), and private key password (optional) used for mutual TLS authentication of your MSK/Apache Kafka brokers.
            - ``SERVER_ROOT_CA_CERTIFICATE`` - (Self-managed Apache Kafka) The Secrets Manager ARN of your secret key containing the root CA certificate (X.509 PEM) used for TLS encryption of your Apache Kafka brokers.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-eventsourcemapping-sourceaccessconfiguration.html#cfn-lambda-eventsourcemapping-sourceaccessconfiguration-type
            '''
            result = self._values.get("type")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def uri(self) -> typing.Optional[builtins.str]:
            '''The value for your chosen configuration in ``Type`` .

            For example: ``"URI": "arn:aws:secretsmanager:us-east-1:01234567890:secret:MyBrokerSecretName"`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-eventsourcemapping-sourceaccessconfiguration.html#cfn-lambda-eventsourcemapping-sourceaccessconfiguration-uri
            '''
            result = self._values.get("uri")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "SourceAccessConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-lambda.CfnEventSourceMappingProps",
    jsii_struct_bases=[],
    name_mapping={
        "function_name": "functionName",
        "amazon_managed_kafka_event_source_config": "amazonManagedKafkaEventSourceConfig",
        "batch_size": "batchSize",
        "bisect_batch_on_function_error": "bisectBatchOnFunctionError",
        "destination_config": "destinationConfig",
        "enabled": "enabled",
        "event_source_arn": "eventSourceArn",
        "filter_criteria": "filterCriteria",
        "function_response_types": "functionResponseTypes",
        "maximum_batching_window_in_seconds": "maximumBatchingWindowInSeconds",
        "maximum_record_age_in_seconds": "maximumRecordAgeInSeconds",
        "maximum_retry_attempts": "maximumRetryAttempts",
        "parallelization_factor": "parallelizationFactor",
        "queues": "queues",
        "scaling_config": "scalingConfig",
        "self_managed_event_source": "selfManagedEventSource",
        "self_managed_kafka_event_source_config": "selfManagedKafkaEventSourceConfig",
        "source_access_configurations": "sourceAccessConfigurations",
        "starting_position": "startingPosition",
        "starting_position_timestamp": "startingPositionTimestamp",
        "topics": "topics",
        "tumbling_window_in_seconds": "tumblingWindowInSeconds",
    },
)
class CfnEventSourceMappingProps:
    def __init__(
        self,
        *,
        function_name: builtins.str,
        amazon_managed_kafka_event_source_config: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union[CfnEventSourceMapping.AmazonManagedKafkaEventSourceConfigProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        batch_size: typing.Optional[jsii.Number] = None,
        bisect_batch_on_function_error: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]] = None,
        destination_config: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union[CfnEventSourceMapping.DestinationConfigProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        enabled: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]] = None,
        event_source_arn: typing.Optional[builtins.str] = None,
        filter_criteria: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union[CfnEventSourceMapping.FilterCriteriaProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        function_response_types: typing.Optional[typing.Sequence[builtins.str]] = None,
        maximum_batching_window_in_seconds: typing.Optional[jsii.Number] = None,
        maximum_record_age_in_seconds: typing.Optional[jsii.Number] = None,
        maximum_retry_attempts: typing.Optional[jsii.Number] = None,
        parallelization_factor: typing.Optional[jsii.Number] = None,
        queues: typing.Optional[typing.Sequence[builtins.str]] = None,
        scaling_config: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union[CfnEventSourceMapping.ScalingConfigProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        self_managed_event_source: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union[CfnEventSourceMapping.SelfManagedEventSourceProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        self_managed_kafka_event_source_config: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union[CfnEventSourceMapping.SelfManagedKafkaEventSourceConfigProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        source_access_configurations: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Sequence[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union[CfnEventSourceMapping.SourceAccessConfigurationProperty, typing.Dict[builtins.str, typing.Any]]]]]] = None,
        starting_position: typing.Optional[builtins.str] = None,
        starting_position_timestamp: typing.Optional[jsii.Number] = None,
        topics: typing.Optional[typing.Sequence[builtins.str]] = None,
        tumbling_window_in_seconds: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''Properties for defining a ``CfnEventSourceMapping``.

        :param function_name: The name of the Lambda function. **Name formats** - *Function name* - ``MyFunction`` . - *Function ARN* - ``arn:aws:lambda:us-west-2:123456789012:function:MyFunction`` . - *Version or Alias ARN* - ``arn:aws:lambda:us-west-2:123456789012:function:MyFunction:PROD`` . - *Partial ARN* - ``123456789012:function:MyFunction`` . The length constraint applies only to the full ARN. If you specify only the function name, it's limited to 64 characters in length.
        :param amazon_managed_kafka_event_source_config: Specific configuration settings for an Amazon Managed Streaming for Apache Kafka (Amazon MSK) event source.
        :param batch_size: The maximum number of records in each batch that Lambda pulls from your stream or queue and sends to your function. Lambda passes all of the records in the batch to the function in a single call, up to the payload limit for synchronous invocation (6 MB). - *Amazon Kinesis* - Default 100. Max 10,000. - *Amazon DynamoDB Streams* - Default 100. Max 10,000. - *Amazon Simple Queue Service* - Default 10. For standard queues the max is 10,000. For FIFO queues the max is 10. - *Amazon Managed Streaming for Apache Kafka* - Default 100. Max 10,000. - *Self-managed Apache Kafka* - Default 100. Max 10,000. - *Amazon MQ (ActiveMQ and RabbitMQ)* - Default 100. Max 10,000.
        :param bisect_batch_on_function_error: (Streams only) If the function returns an error, split the batch in two and retry. The default value is false.
        :param destination_config: (Streams only) An Amazon SQS queue or Amazon SNS topic destination for discarded records.
        :param enabled: When true, the event source mapping is active. When false, Lambda pauses polling and invocation. Default: True
        :param event_source_arn: The Amazon Resource Name (ARN) of the event source. - *Amazon Kinesis* - The ARN of the data stream or a stream consumer. - *Amazon DynamoDB Streams* - The ARN of the stream. - *Amazon Simple Queue Service* - The ARN of the queue. - *Amazon Managed Streaming for Apache Kafka* - The ARN of the cluster. - *Amazon MQ* - The ARN of the broker.
        :param filter_criteria: An object that defines the filter criteria that determine whether Lambda should process an event. For more information, see `Lambda event filtering <https://docs.aws.amazon.com/lambda/latest/dg/invocation-eventfiltering.html>`_ .
        :param function_response_types: (Streams and SQS) A list of current response type enums applied to the event source mapping. Valid Values: ``ReportBatchItemFailures``
        :param maximum_batching_window_in_seconds: The maximum amount of time, in seconds, that Lambda spends gathering records before invoking the function. *Default ( Kinesis , DynamoDB , Amazon SQS event sources)* : 0 *Default ( Amazon MSK , Kafka, Amazon MQ event sources)* : 500 ms *Related setting:* When you set ``BatchSize`` to a value greater than 10, you must set ``MaximumBatchingWindowInSeconds`` to at least 1.
        :param maximum_record_age_in_seconds: (Streams only) Discard records older than the specified age. The default value is -1, which sets the maximum age to infinite. When the value is set to infinite, Lambda never discards old records.
        :param maximum_retry_attempts: (Streams only) Discard records after the specified number of retries. The default value is -1, which sets the maximum number of retries to infinite. When MaximumRetryAttempts is infinite, Lambda retries failed records until the record expires in the event source.
        :param parallelization_factor: (Streams only) The number of batches to process concurrently from each shard. The default value is 1.
        :param queues: (Amazon MQ) The name of the Amazon MQ broker destination queue to consume.
        :param scaling_config: ``AWS::Lambda::EventSourceMapping.ScalingConfig``.
        :param self_managed_event_source: The self-managed Apache Kafka cluster for your event source.
        :param self_managed_kafka_event_source_config: Specific configuration settings for a self-managed Apache Kafka event source.
        :param source_access_configurations: An array of the authentication protocol, VPC components, or virtual host to secure and define your event source.
        :param starting_position: The position in a stream from which to start reading. Required for Amazon Kinesis and Amazon DynamoDB. - *LATEST* - Read only new records. - *TRIM_HORIZON* - Process all available records. - *AT_TIMESTAMP* - Specify a time from which to start reading records.
        :param starting_position_timestamp: With ``StartingPosition`` set to ``AT_TIMESTAMP`` , the time from which to start reading, in Unix time seconds.
        :param topics: The name of the Kafka topic.
        :param tumbling_window_in_seconds: (Streams only) The duration in seconds of a processing window. The range is between 1 second and 900 seconds.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-eventsourcemapping.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_lambda as lambda_
            
            cfn_event_source_mapping_props = lambda.CfnEventSourceMappingProps(
                function_name="functionName",
            
                # the properties below are optional
                amazon_managed_kafka_event_source_config=lambda.CfnEventSourceMapping.AmazonManagedKafkaEventSourceConfigProperty(
                    consumer_group_id="consumerGroupId"
                ),
                batch_size=123,
                bisect_batch_on_function_error=False,
                destination_config=lambda.CfnEventSourceMapping.DestinationConfigProperty(
                    on_failure=lambda.CfnEventSourceMapping.OnFailureProperty(
                        destination="destination"
                    )
                ),
                enabled=False,
                event_source_arn="eventSourceArn",
                filter_criteria=lambda.CfnEventSourceMapping.FilterCriteriaProperty(
                    filters=[lambda.CfnEventSourceMapping.FilterProperty(
                        pattern="pattern"
                    )]
                ),
                function_response_types=["functionResponseTypes"],
                maximum_batching_window_in_seconds=123,
                maximum_record_age_in_seconds=123,
                maximum_retry_attempts=123,
                parallelization_factor=123,
                queues=["queues"],
                scaling_config=lambda.CfnEventSourceMapping.ScalingConfigProperty(
                    maximum_concurrency=123
                ),
                self_managed_event_source=lambda.CfnEventSourceMapping.SelfManagedEventSourceProperty(
                    endpoints=lambda.CfnEventSourceMapping.EndpointsProperty(
                        kafka_bootstrap_servers=["kafkaBootstrapServers"]
                    )
                ),
                self_managed_kafka_event_source_config=lambda.CfnEventSourceMapping.SelfManagedKafkaEventSourceConfigProperty(
                    consumer_group_id="consumerGroupId"
                ),
                source_access_configurations=[lambda.CfnEventSourceMapping.SourceAccessConfigurationProperty(
                    type="type",
                    uri="uri"
                )],
                starting_position="startingPosition",
                starting_position_timestamp=123,
                topics=["topics"],
                tumbling_window_in_seconds=123
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f6be2190fb728b46389157d6fa529d7fd20a0f40ab99a4386603cd6383e67268)
            check_type(argname="argument function_name", value=function_name, expected_type=type_hints["function_name"])
            check_type(argname="argument amazon_managed_kafka_event_source_config", value=amazon_managed_kafka_event_source_config, expected_type=type_hints["amazon_managed_kafka_event_source_config"])
            check_type(argname="argument batch_size", value=batch_size, expected_type=type_hints["batch_size"])
            check_type(argname="argument bisect_batch_on_function_error", value=bisect_batch_on_function_error, expected_type=type_hints["bisect_batch_on_function_error"])
            check_type(argname="argument destination_config", value=destination_config, expected_type=type_hints["destination_config"])
            check_type(argname="argument enabled", value=enabled, expected_type=type_hints["enabled"])
            check_type(argname="argument event_source_arn", value=event_source_arn, expected_type=type_hints["event_source_arn"])
            check_type(argname="argument filter_criteria", value=filter_criteria, expected_type=type_hints["filter_criteria"])
            check_type(argname="argument function_response_types", value=function_response_types, expected_type=type_hints["function_response_types"])
            check_type(argname="argument maximum_batching_window_in_seconds", value=maximum_batching_window_in_seconds, expected_type=type_hints["maximum_batching_window_in_seconds"])
            check_type(argname="argument maximum_record_age_in_seconds", value=maximum_record_age_in_seconds, expected_type=type_hints["maximum_record_age_in_seconds"])
            check_type(argname="argument maximum_retry_attempts", value=maximum_retry_attempts, expected_type=type_hints["maximum_retry_attempts"])
            check_type(argname="argument parallelization_factor", value=parallelization_factor, expected_type=type_hints["parallelization_factor"])
            check_type(argname="argument queues", value=queues, expected_type=type_hints["queues"])
            check_type(argname="argument scaling_config", value=scaling_config, expected_type=type_hints["scaling_config"])
            check_type(argname="argument self_managed_event_source", value=self_managed_event_source, expected_type=type_hints["self_managed_event_source"])
            check_type(argname="argument self_managed_kafka_event_source_config", value=self_managed_kafka_event_source_config, expected_type=type_hints["self_managed_kafka_event_source_config"])
            check_type(argname="argument source_access_configurations", value=source_access_configurations, expected_type=type_hints["source_access_configurations"])
            check_type(argname="argument starting_position", value=starting_position, expected_type=type_hints["starting_position"])
            check_type(argname="argument starting_position_timestamp", value=starting_position_timestamp, expected_type=type_hints["starting_position_timestamp"])
            check_type(argname="argument topics", value=topics, expected_type=type_hints["topics"])
            check_type(argname="argument tumbling_window_in_seconds", value=tumbling_window_in_seconds, expected_type=type_hints["tumbling_window_in_seconds"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "function_name": function_name,
        }
        if amazon_managed_kafka_event_source_config is not None:
            self._values["amazon_managed_kafka_event_source_config"] = amazon_managed_kafka_event_source_config
        if batch_size is not None:
            self._values["batch_size"] = batch_size
        if bisect_batch_on_function_error is not None:
            self._values["bisect_batch_on_function_error"] = bisect_batch_on_function_error
        if destination_config is not None:
            self._values["destination_config"] = destination_config
        if enabled is not None:
            self._values["enabled"] = enabled
        if event_source_arn is not None:
            self._values["event_source_arn"] = event_source_arn
        if filter_criteria is not None:
            self._values["filter_criteria"] = filter_criteria
        if function_response_types is not None:
            self._values["function_response_types"] = function_response_types
        if maximum_batching_window_in_seconds is not None:
            self._values["maximum_batching_window_in_seconds"] = maximum_batching_window_in_seconds
        if maximum_record_age_in_seconds is not None:
            self._values["maximum_record_age_in_seconds"] = maximum_record_age_in_seconds
        if maximum_retry_attempts is not None:
            self._values["maximum_retry_attempts"] = maximum_retry_attempts
        if parallelization_factor is not None:
            self._values["parallelization_factor"] = parallelization_factor
        if queues is not None:
            self._values["queues"] = queues
        if scaling_config is not None:
            self._values["scaling_config"] = scaling_config
        if self_managed_event_source is not None:
            self._values["self_managed_event_source"] = self_managed_event_source
        if self_managed_kafka_event_source_config is not None:
            self._values["self_managed_kafka_event_source_config"] = self_managed_kafka_event_source_config
        if source_access_configurations is not None:
            self._values["source_access_configurations"] = source_access_configurations
        if starting_position is not None:
            self._values["starting_position"] = starting_position
        if starting_position_timestamp is not None:
            self._values["starting_position_timestamp"] = starting_position_timestamp
        if topics is not None:
            self._values["topics"] = topics
        if tumbling_window_in_seconds is not None:
            self._values["tumbling_window_in_seconds"] = tumbling_window_in_seconds

    @builtins.property
    def function_name(self) -> builtins.str:
        '''The name of the Lambda function.

        **Name formats** - *Function name* - ``MyFunction`` .

        - *Function ARN* - ``arn:aws:lambda:us-west-2:123456789012:function:MyFunction`` .
        - *Version or Alias ARN* - ``arn:aws:lambda:us-west-2:123456789012:function:MyFunction:PROD`` .
        - *Partial ARN* - ``123456789012:function:MyFunction`` .

        The length constraint applies only to the full ARN. If you specify only the function name, it's limited to 64 characters in length.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-eventsourcemapping.html#cfn-lambda-eventsourcemapping-functionname
        '''
        result = self._values.get("function_name")
        assert result is not None, "Required property 'function_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def amazon_managed_kafka_event_source_config(
        self,
    ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, CfnEventSourceMapping.AmazonManagedKafkaEventSourceConfigProperty]]:
        '''Specific configuration settings for an Amazon Managed Streaming for Apache Kafka (Amazon MSK) event source.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-eventsourcemapping.html#cfn-lambda-eventsourcemapping-amazonmanagedkafkaeventsourceconfig
        '''
        result = self._values.get("amazon_managed_kafka_event_source_config")
        return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, CfnEventSourceMapping.AmazonManagedKafkaEventSourceConfigProperty]], result)

    @builtins.property
    def batch_size(self) -> typing.Optional[jsii.Number]:
        '''The maximum number of records in each batch that Lambda pulls from your stream or queue and sends to your function.

        Lambda passes all of the records in the batch to the function in a single call, up to the payload limit for synchronous invocation (6 MB).

        - *Amazon Kinesis* - Default 100. Max 10,000.
        - *Amazon DynamoDB Streams* - Default 100. Max 10,000.
        - *Amazon Simple Queue Service* - Default 10. For standard queues the max is 10,000. For FIFO queues the max is 10.
        - *Amazon Managed Streaming for Apache Kafka* - Default 100. Max 10,000.
        - *Self-managed Apache Kafka* - Default 100. Max 10,000.
        - *Amazon MQ (ActiveMQ and RabbitMQ)* - Default 100. Max 10,000.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-eventsourcemapping.html#cfn-lambda-eventsourcemapping-batchsize
        '''
        result = self._values.get("batch_size")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def bisect_batch_on_function_error(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]]:
        '''(Streams only) If the function returns an error, split the batch in two and retry.

        The default value is false.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-eventsourcemapping.html#cfn-lambda-eventsourcemapping-bisectbatchonfunctionerror
        '''
        result = self._values.get("bisect_batch_on_function_error")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]], result)

    @builtins.property
    def destination_config(
        self,
    ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, CfnEventSourceMapping.DestinationConfigProperty]]:
        '''(Streams only) An Amazon SQS queue or Amazon SNS topic destination for discarded records.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-eventsourcemapping.html#cfn-lambda-eventsourcemapping-destinationconfig
        '''
        result = self._values.get("destination_config")
        return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, CfnEventSourceMapping.DestinationConfigProperty]], result)

    @builtins.property
    def enabled(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]]:
        '''When true, the event source mapping is active. When false, Lambda pauses polling and invocation.

        Default: True

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-eventsourcemapping.html#cfn-lambda-eventsourcemapping-enabled
        '''
        result = self._values.get("enabled")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]], result)

    @builtins.property
    def event_source_arn(self) -> typing.Optional[builtins.str]:
        '''The Amazon Resource Name (ARN) of the event source.

        - *Amazon Kinesis* - The ARN of the data stream or a stream consumer.
        - *Amazon DynamoDB Streams* - The ARN of the stream.
        - *Amazon Simple Queue Service* - The ARN of the queue.
        - *Amazon Managed Streaming for Apache Kafka* - The ARN of the cluster.
        - *Amazon MQ* - The ARN of the broker.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-eventsourcemapping.html#cfn-lambda-eventsourcemapping-eventsourcearn
        '''
        result = self._values.get("event_source_arn")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def filter_criteria(
        self,
    ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, CfnEventSourceMapping.FilterCriteriaProperty]]:
        '''An object that defines the filter criteria that determine whether Lambda should process an event.

        For more information, see `Lambda event filtering <https://docs.aws.amazon.com/lambda/latest/dg/invocation-eventfiltering.html>`_ .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-eventsourcemapping.html#cfn-lambda-eventsourcemapping-filtercriteria
        '''
        result = self._values.get("filter_criteria")
        return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, CfnEventSourceMapping.FilterCriteriaProperty]], result)

    @builtins.property
    def function_response_types(self) -> typing.Optional[typing.List[builtins.str]]:
        '''(Streams and SQS) A list of current response type enums applied to the event source mapping.

        Valid Values: ``ReportBatchItemFailures``

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-eventsourcemapping.html#cfn-lambda-eventsourcemapping-functionresponsetypes
        '''
        result = self._values.get("function_response_types")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def maximum_batching_window_in_seconds(self) -> typing.Optional[jsii.Number]:
        '''The maximum amount of time, in seconds, that Lambda spends gathering records before invoking the function.

        *Default ( Kinesis , DynamoDB , Amazon SQS event sources)* : 0

        *Default ( Amazon MSK , Kafka, Amazon MQ event sources)* : 500 ms

        *Related setting:* When you set ``BatchSize`` to a value greater than 10, you must set ``MaximumBatchingWindowInSeconds`` to at least 1.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-eventsourcemapping.html#cfn-lambda-eventsourcemapping-maximumbatchingwindowinseconds
        '''
        result = self._values.get("maximum_batching_window_in_seconds")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def maximum_record_age_in_seconds(self) -> typing.Optional[jsii.Number]:
        '''(Streams only) Discard records older than the specified age.

        The default value is -1,
        which sets the maximum age to infinite. When the value is set to infinite, Lambda never discards old records.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-eventsourcemapping.html#cfn-lambda-eventsourcemapping-maximumrecordageinseconds
        '''
        result = self._values.get("maximum_record_age_in_seconds")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def maximum_retry_attempts(self) -> typing.Optional[jsii.Number]:
        '''(Streams only) Discard records after the specified number of retries.

        The default value is -1,
        which sets the maximum number of retries to infinite. When MaximumRetryAttempts is infinite, Lambda retries failed records until the record expires in the event source.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-eventsourcemapping.html#cfn-lambda-eventsourcemapping-maximumretryattempts
        '''
        result = self._values.get("maximum_retry_attempts")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def parallelization_factor(self) -> typing.Optional[jsii.Number]:
        '''(Streams only) The number of batches to process concurrently from each shard.

        The default value is 1.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-eventsourcemapping.html#cfn-lambda-eventsourcemapping-parallelizationfactor
        '''
        result = self._values.get("parallelization_factor")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def queues(self) -> typing.Optional[typing.List[builtins.str]]:
        '''(Amazon MQ) The name of the Amazon MQ broker destination queue to consume.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-eventsourcemapping.html#cfn-lambda-eventsourcemapping-queues
        '''
        result = self._values.get("queues")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def scaling_config(
        self,
    ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, CfnEventSourceMapping.ScalingConfigProperty]]:
        '''``AWS::Lambda::EventSourceMapping.ScalingConfig``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-eventsourcemapping.html#cfn-lambda-eventsourcemapping-scalingconfig
        '''
        result = self._values.get("scaling_config")
        return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, CfnEventSourceMapping.ScalingConfigProperty]], result)

    @builtins.property
    def self_managed_event_source(
        self,
    ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, CfnEventSourceMapping.SelfManagedEventSourceProperty]]:
        '''The self-managed Apache Kafka cluster for your event source.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-eventsourcemapping.html#cfn-lambda-eventsourcemapping-selfmanagedeventsource
        '''
        result = self._values.get("self_managed_event_source")
        return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, CfnEventSourceMapping.SelfManagedEventSourceProperty]], result)

    @builtins.property
    def self_managed_kafka_event_source_config(
        self,
    ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, CfnEventSourceMapping.SelfManagedKafkaEventSourceConfigProperty]]:
        '''Specific configuration settings for a self-managed Apache Kafka event source.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-eventsourcemapping.html#cfn-lambda-eventsourcemapping-selfmanagedkafkaeventsourceconfig
        '''
        result = self._values.get("self_managed_kafka_event_source_config")
        return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, CfnEventSourceMapping.SelfManagedKafkaEventSourceConfigProperty]], result)

    @builtins.property
    def source_access_configurations(
        self,
    ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.List[typing.Union[_aws_cdk_core_f4b25747.IResolvable, CfnEventSourceMapping.SourceAccessConfigurationProperty]]]]:
        '''An array of the authentication protocol, VPC components, or virtual host to secure and define your event source.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-eventsourcemapping.html#cfn-lambda-eventsourcemapping-sourceaccessconfigurations
        '''
        result = self._values.get("source_access_configurations")
        return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.List[typing.Union[_aws_cdk_core_f4b25747.IResolvable, CfnEventSourceMapping.SourceAccessConfigurationProperty]]]], result)

    @builtins.property
    def starting_position(self) -> typing.Optional[builtins.str]:
        '''The position in a stream from which to start reading. Required for Amazon Kinesis and Amazon DynamoDB.

        - *LATEST* - Read only new records.
        - *TRIM_HORIZON* - Process all available records.
        - *AT_TIMESTAMP* - Specify a time from which to start reading records.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-eventsourcemapping.html#cfn-lambda-eventsourcemapping-startingposition
        '''
        result = self._values.get("starting_position")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def starting_position_timestamp(self) -> typing.Optional[jsii.Number]:
        '''With ``StartingPosition`` set to ``AT_TIMESTAMP`` , the time from which to start reading, in Unix time seconds.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-eventsourcemapping.html#cfn-lambda-eventsourcemapping-startingpositiontimestamp
        '''
        result = self._values.get("starting_position_timestamp")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def topics(self) -> typing.Optional[typing.List[builtins.str]]:
        '''The name of the Kafka topic.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-eventsourcemapping.html#cfn-lambda-eventsourcemapping-topics
        '''
        result = self._values.get("topics")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def tumbling_window_in_seconds(self) -> typing.Optional[jsii.Number]:
        '''(Streams only) The duration in seconds of a processing window.

        The range is between 1 second and 900 seconds.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-eventsourcemapping.html#cfn-lambda-eventsourcemapping-tumblingwindowinseconds
        '''
        result = self._values.get("tumbling_window_in_seconds")
        return typing.cast(typing.Optional[jsii.Number], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnEventSourceMappingProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_aws_cdk_core_f4b25747.IInspectable)
class CfnFunction(
    _aws_cdk_core_f4b25747.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-lambda.CfnFunction",
):
    '''A CloudFormation ``AWS::Lambda::Function``.

    The ``AWS::Lambda::Function`` resource creates a Lambda function. To create a function, you need a `deployment package <https://docs.aws.amazon.com/lambda/latest/dg/gettingstarted-package.html>`_ and an `execution role <https://docs.aws.amazon.com/lambda/latest/dg/lambda-intro-execution-role.html>`_ . The deployment package is a .zip file archive or container image that contains your function code. The execution role grants the function permission to use AWS services, such as Amazon CloudWatch Logs for log streaming and AWS X-Ray for request tracing.

    You set the package type to ``Image`` if the deployment package is a `container image <https://docs.aws.amazon.com/lambda/latest/dg/lambda-images.html>`_ . For a container image, the code property must include the URI of a container image in the Amazon ECR registry. You do not need to specify the handler and runtime properties.

    You set the package type to ``Zip`` if the deployment package is a `.zip file archive <https://docs.aws.amazon.com/lambda/latest/dg/gettingstarted-package.html#gettingstarted-package-zip>`_ . For a .zip file archive, the code property specifies the location of the .zip file. You must also specify the handler and runtime properties. For a Python example, see `Deploy Python Lambda functions with .zip file archives <https://docs.aws.amazon.com/lambda/latest/dg/python-package.html>`_ .

    You can use `code signing <https://docs.aws.amazon.com/lambda/latest/dg/configuration-codesigning.html>`_ if your deployment package is a .zip file archive. To enable code signing for this function, specify the ARN of a code-signing configuration. When a user attempts to deploy a code package with ``UpdateFunctionCode`` , Lambda checks that the code package has a valid signature from a trusted publisher. The code-signing configuration includes a set of signing profiles, which define the trusted publishers for this function.

    Note that you configure `provisioned concurrency <https://docs.aws.amazon.com/lambda/latest/dg/configuration-concurrency.html>`_ on a ``AWS::Lambda::Version`` or a ``AWS::Lambda::Alias`` .

    For a complete introduction to Lambda functions, see `What is Lambda? <https://docs.aws.amazon.com/lambda/latest/dg/lambda-welcome.html>`_ in the *Lambda developer guide.*

    :cloudformationResource: AWS::Lambda::Function
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-function.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        import aws_cdk.aws_lambda as lambda_
        
        cfn_function = lambda_.CfnFunction(self, "MyCfnFunction",
            code=lambda.CfnFunction.CodeProperty(
                image_uri="imageUri",
                s3_bucket="s3Bucket",
                s3_key="s3Key",
                s3_object_version="s3ObjectVersion",
                zip_file="zipFile"
            ),
            role="role",
        
            # the properties below are optional
            architectures=["architectures"],
            code_signing_config_arn="codeSigningConfigArn",
            dead_letter_config=lambda.CfnFunction.DeadLetterConfigProperty(
                target_arn="targetArn"
            ),
            description="description",
            environment=lambda.CfnFunction.EnvironmentProperty(
                variables={
                    "variables_key": "variables"
                }
            ),
            ephemeral_storage=lambda.CfnFunction.EphemeralStorageProperty(
                size=123
            ),
            file_system_configs=[lambda.CfnFunction.FileSystemConfigProperty(
                arn="arn",
                local_mount_path="localMountPath"
            )],
            function_name="functionName",
            handler="handler",
            image_config=lambda.CfnFunction.ImageConfigProperty(
                command=["command"],
                entry_point=["entryPoint"],
                working_directory="workingDirectory"
            ),
            kms_key_arn="kmsKeyArn",
            layers=["layers"],
            memory_size=123,
            package_type="packageType",
            reserved_concurrent_executions=123,
            runtime="runtime",
            snap_start=lambda.CfnFunction.SnapStartProperty(
                apply_on="applyOn"
            ),
            tags=[CfnTag(
                key="key",
                value="value"
            )],
            timeout=123,
            tracing_config=lambda.CfnFunction.TracingConfigProperty(
                mode="mode"
            ),
            vpc_config=lambda.CfnFunction.VpcConfigProperty(
                security_group_ids=["securityGroupIds"],
                subnet_ids=["subnetIds"]
            )
        )
    '''

    def __init__(
        self,
        scope: _aws_cdk_core_f4b25747.Construct,
        id: builtins.str,
        *,
        code: typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnFunction.CodeProperty", typing.Dict[builtins.str, typing.Any]]],
        role: builtins.str,
        architectures: typing.Optional[typing.Sequence[builtins.str]] = None,
        code_signing_config_arn: typing.Optional[builtins.str] = None,
        dead_letter_config: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnFunction.DeadLetterConfigProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        description: typing.Optional[builtins.str] = None,
        environment: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnFunction.EnvironmentProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ephemeral_storage: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnFunction.EphemeralStorageProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        file_system_configs: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Sequence[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnFunction.FileSystemConfigProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
        function_name: typing.Optional[builtins.str] = None,
        handler: typing.Optional[builtins.str] = None,
        image_config: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnFunction.ImageConfigProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        kms_key_arn: typing.Optional[builtins.str] = None,
        layers: typing.Optional[typing.Sequence[builtins.str]] = None,
        memory_size: typing.Optional[jsii.Number] = None,
        package_type: typing.Optional[builtins.str] = None,
        reserved_concurrent_executions: typing.Optional[jsii.Number] = None,
        runtime: typing.Optional[builtins.str] = None,
        snap_start: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnFunction.SnapStartProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_aws_cdk_core_f4b25747.CfnTag, typing.Dict[builtins.str, typing.Any]]]] = None,
        timeout: typing.Optional[jsii.Number] = None,
        tracing_config: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnFunction.TracingConfigProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        vpc_config: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnFunction.VpcConfigProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''Create a new ``AWS::Lambda::Function``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param code: The code for the function.
        :param role: The Amazon Resource Name (ARN) of the function's execution role.
        :param architectures: The instruction set architecture that the function supports. Enter a string array with one of the valid values (arm64 or x86_64). The default value is ``x86_64`` .
        :param code_signing_config_arn: To enable code signing for this function, specify the ARN of a code-signing configuration. A code-signing configuration includes a set of signing profiles, which define the trusted publishers for this function.
        :param dead_letter_config: A dead-letter queue configuration that specifies the queue or topic where Lambda sends asynchronous events when they fail processing. For more information, see `Dead-letter queues <https://docs.aws.amazon.com/lambda/latest/dg/invocation-async.html#invocation-dlq>`_ .
        :param description: A description of the function.
        :param environment: Environment variables that are accessible from function code during execution.
        :param ephemeral_storage: The size of the function's ``/tmp`` directory in MB. The default value is 512, but it can be any whole number between 512 and 10,240 MB.
        :param file_system_configs: Connection settings for an Amazon EFS file system. To connect a function to a file system, a mount target must be available in every Availability Zone that your function connects to. If your template contains an `AWS::EFS::MountTarget <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-efs-mounttarget.html>`_ resource, you must also specify a ``DependsOn`` attribute to ensure that the mount target is created or updated before the function. For more information about using the ``DependsOn`` attribute, see `DependsOn Attribute <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-attribute-dependson.html>`_ .
        :param function_name: The name of the Lambda function, up to 64 characters in length. If you don't specify a name, AWS CloudFormation generates one. If you specify a name, you cannot perform updates that require replacement of this resource. You can perform updates that require no or some interruption. If you must replace the resource, specify a new name.
        :param handler: The name of the method within your code that Lambda calls to run your function. Handler is required if the deployment package is a .zip file archive. The format includes the file name. It can also include namespaces and other qualifiers, depending on the runtime. For more information, see `Lambda programming model <https://docs.aws.amazon.com/lambda/latest/dg/foundation-progmodel.html>`_ .
        :param image_config: Configuration values that override the container image Dockerfile settings. For more information, see `Container image settings <https://docs.aws.amazon.com/lambda/latest/dg/images-create.html#images-parms>`_ .
        :param kms_key_arn: The ARN of the AWS Key Management Service ( AWS KMS ) key that's used to encrypt your function's environment variables. If it's not provided, Lambda uses a default service key.
        :param layers: A list of `function layers <https://docs.aws.amazon.com/lambda/latest/dg/configuration-layers.html>`_ to add to the function's execution environment. Specify each layer by its ARN, including the version.
        :param memory_size: The amount of `memory available to the function <https://docs.aws.amazon.com/lambda/latest/dg/configuration-function-common.html#configuration-memory-console>`_ at runtime. Increasing the function memory also increases its CPU allocation. The default value is 128 MB. The value can be any multiple of 1 MB.
        :param package_type: The type of deployment package. Set to ``Image`` for container image and set ``Zip`` for .zip file archive.
        :param reserved_concurrent_executions: The number of simultaneous executions to reserve for the function.
        :param runtime: The identifier of the function's `runtime <https://docs.aws.amazon.com/lambda/latest/dg/lambda-runtimes.html>`_ . Runtime is required if the deployment package is a .zip file archive.
        :param snap_start: The function's `AWS Lambda SnapStart <https://docs.aws.amazon.com/lambda/latest/dg/snapstart.html>`_ setting.
        :param tags: A list of `tags <https://docs.aws.amazon.com/lambda/latest/dg/tagging.html>`_ to apply to the function.
        :param timeout: The amount of time (in seconds) that Lambda allows a function to run before stopping it. The default is 3 seconds. The maximum allowed value is 900 seconds. For more information, see `Lambda execution environment <https://docs.aws.amazon.com/lambda/latest/dg/runtimes-context.html>`_ .
        :param tracing_config: Set ``Mode`` to ``Active`` to sample and trace a subset of incoming requests with `X-Ray <https://docs.aws.amazon.com/lambda/latest/dg/services-xray.html>`_ .
        :param vpc_config: For network connectivity to AWS resources in a `VPC <https://docs.aws.amazon.com/lambda/latest/dg/configuration-network.html>`_ , specify a list of security groups and subnets in the VPC.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b8f7b7b3eb301b53f546ca38199ef9cd99663213af4312da4c097b7503d77087)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnFunctionProps(
            code=code,
            role=role,
            architectures=architectures,
            code_signing_config_arn=code_signing_config_arn,
            dead_letter_config=dead_letter_config,
            description=description,
            environment=environment,
            ephemeral_storage=ephemeral_storage,
            file_system_configs=file_system_configs,
            function_name=function_name,
            handler=handler,
            image_config=image_config,
            kms_key_arn=kms_key_arn,
            layers=layers,
            memory_size=memory_size,
            package_type=package_type,
            reserved_concurrent_executions=reserved_concurrent_executions,
            runtime=runtime,
            snap_start=snap_start,
            tags=tags,
            timeout=timeout,
            tracing_config=tracing_config,
            vpc_config=vpc_config,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _aws_cdk_core_f4b25747.TreeInspector) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__da1af6604c8a11b416ad5ce0a6a9098fc2846aa9b48b0f44a78bae791d76d926)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9af9baf17e249aca999459eb3ce7f13985dcefa2ec3b7fbbd10b984d65791080)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrArn")
    def attr_arn(self) -> builtins.str:
        '''The Amazon Resource Name (ARN) of the function.

        :cloudformationAttribute: Arn
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrArn"))

    @builtins.property
    @jsii.member(jsii_name="attrSnapStartResponseApplyOn")
    def attr_snap_start_response_apply_on(self) -> builtins.str:
        '''
        :cloudformationAttribute: SnapStartResponse.ApplyOn
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrSnapStartResponseApplyOn"))

    @builtins.property
    @jsii.member(jsii_name="attrSnapStartResponseOptimizationStatus")
    def attr_snap_start_response_optimization_status(self) -> builtins.str:
        '''
        :cloudformationAttribute: SnapStartResponse.OptimizationStatus
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrSnapStartResponseOptimizationStatus"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="tags")
    def tags(self) -> _aws_cdk_core_f4b25747.TagManager:
        '''A list of `tags <https://docs.aws.amazon.com/lambda/latest/dg/tagging.html>`_ to apply to the function.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-function.html#cfn-lambda-function-tags
        '''
        return typing.cast(_aws_cdk_core_f4b25747.TagManager, jsii.get(self, "tags"))

    @builtins.property
    @jsii.member(jsii_name="code")
    def code(
        self,
    ) -> typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnFunction.CodeProperty"]:
        '''The code for the function.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-function.html#cfn-lambda-function-code
        '''
        return typing.cast(typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnFunction.CodeProperty"], jsii.get(self, "code"))

    @code.setter
    def code(
        self,
        value: typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnFunction.CodeProperty"],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__7708d11566820e58b28ca37d3d8f6b00d61d12bb7de9383a44eefa2191f9fdef)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "code", value)

    @builtins.property
    @jsii.member(jsii_name="role")
    def role(self) -> builtins.str:
        '''The Amazon Resource Name (ARN) of the function's execution role.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-function.html#cfn-lambda-function-role
        '''
        return typing.cast(builtins.str, jsii.get(self, "role"))

    @role.setter
    def role(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__04de5aed12d09b3a52f7223c14f130340d58933b5d83580fce3fe9b08b81a40a)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "role", value)

    @builtins.property
    @jsii.member(jsii_name="architectures")
    def architectures(self) -> typing.Optional[typing.List[builtins.str]]:
        '''The instruction set architecture that the function supports.

        Enter a string array with one of the valid values (arm64 or x86_64). The default value is ``x86_64`` .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-function.html#cfn-lambda-function-architectures
        '''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "architectures"))

    @architectures.setter
    def architectures(self, value: typing.Optional[typing.List[builtins.str]]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9ffe8dae995a02fbd912a7fa86a06c5a0cc0b0fccc2d42f936a6acbedede7069)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "architectures", value)

    @builtins.property
    @jsii.member(jsii_name="codeSigningConfigArn")
    def code_signing_config_arn(self) -> typing.Optional[builtins.str]:
        '''To enable code signing for this function, specify the ARN of a code-signing configuration.

        A code-signing configuration
        includes a set of signing profiles, which define the trusted publishers for this function.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-function.html#cfn-lambda-function-codesigningconfigarn
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "codeSigningConfigArn"))

    @code_signing_config_arn.setter
    def code_signing_config_arn(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__7a798f7bad5671d23b59909e51c44e25678b9af4e25877fb1df632feba23a33f)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "codeSigningConfigArn", value)

    @builtins.property
    @jsii.member(jsii_name="deadLetterConfig")
    def dead_letter_config(
        self,
    ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnFunction.DeadLetterConfigProperty"]]:
        '''A dead-letter queue configuration that specifies the queue or topic where Lambda sends asynchronous events when they fail processing.

        For more information, see `Dead-letter queues <https://docs.aws.amazon.com/lambda/latest/dg/invocation-async.html#invocation-dlq>`_ .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-function.html#cfn-lambda-function-deadletterconfig
        '''
        return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnFunction.DeadLetterConfigProperty"]], jsii.get(self, "deadLetterConfig"))

    @dead_letter_config.setter
    def dead_letter_config(
        self,
        value: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnFunction.DeadLetterConfigProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__54b6cbb01be24c5f037c8730dbdc99147faa02862296151ea1e816deafbcc220)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "deadLetterConfig", value)

    @builtins.property
    @jsii.member(jsii_name="description")
    def description(self) -> typing.Optional[builtins.str]:
        '''A description of the function.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-function.html#cfn-lambda-function-description
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "description"))

    @description.setter
    def description(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c586a2eaea459545bf148085eb4c92926e851245871a85664efb8e183dae6faa)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "description", value)

    @builtins.property
    @jsii.member(jsii_name="environment")
    def environment(
        self,
    ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnFunction.EnvironmentProperty"]]:
        '''Environment variables that are accessible from function code during execution.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-function.html#cfn-lambda-function-environment
        '''
        return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnFunction.EnvironmentProperty"]], jsii.get(self, "environment"))

    @environment.setter
    def environment(
        self,
        value: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnFunction.EnvironmentProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__64a804ae78c8a97d153202759527afa0132701f9bcb8ee050ddf4e2b97004243)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "environment", value)

    @builtins.property
    @jsii.member(jsii_name="ephemeralStorage")
    def ephemeral_storage(
        self,
    ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnFunction.EphemeralStorageProperty"]]:
        '''The size of the function's ``/tmp`` directory in MB.

        The default value is 512, but it can be any whole number between 512 and 10,240 MB.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-function.html#cfn-lambda-function-ephemeralstorage
        '''
        return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnFunction.EphemeralStorageProperty"]], jsii.get(self, "ephemeralStorage"))

    @ephemeral_storage.setter
    def ephemeral_storage(
        self,
        value: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnFunction.EphemeralStorageProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0f9e2e37cc67f4fb42e3161463f15ec499b5e375945348353fee9b83be233064)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "ephemeralStorage", value)

    @builtins.property
    @jsii.member(jsii_name="fileSystemConfigs")
    def file_system_configs(
        self,
    ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.List[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnFunction.FileSystemConfigProperty"]]]]:
        '''Connection settings for an Amazon EFS file system.

        To connect a function to a file system, a mount target must be available in every Availability Zone that your function connects to. If your template contains an `AWS::EFS::MountTarget <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-efs-mounttarget.html>`_ resource, you must also specify a ``DependsOn`` attribute to ensure that the mount target is created or updated before the function.

        For more information about using the ``DependsOn`` attribute, see `DependsOn Attribute <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-attribute-dependson.html>`_ .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-function.html#cfn-lambda-function-filesystemconfigs
        '''
        return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.List[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnFunction.FileSystemConfigProperty"]]]], jsii.get(self, "fileSystemConfigs"))

    @file_system_configs.setter
    def file_system_configs(
        self,
        value: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.List[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnFunction.FileSystemConfigProperty"]]]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ad56ae1f17d3697885fc329c11fdc030c390a5f35c25874510d65b1bf14d523a)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "fileSystemConfigs", value)

    @builtins.property
    @jsii.member(jsii_name="functionName")
    def function_name(self) -> typing.Optional[builtins.str]:
        '''The name of the Lambda function, up to 64 characters in length.

        If you don't specify a name, AWS CloudFormation generates one.

        If you specify a name, you cannot perform updates that require replacement of this resource. You can perform updates that require no or some interruption. If you must replace the resource, specify a new name.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-function.html#cfn-lambda-function-functionname
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "functionName"))

    @function_name.setter
    def function_name(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__93ed916abfa90b5ac389bfe500cd11225d8fa74f045808ae8b012d18d79bc657)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "functionName", value)

    @builtins.property
    @jsii.member(jsii_name="handler")
    def handler(self) -> typing.Optional[builtins.str]:
        '''The name of the method within your code that Lambda calls to run your function.

        Handler is required if the deployment package is a .zip file archive. The format includes the file name. It can also include namespaces and other qualifiers, depending on the runtime. For more information, see `Lambda programming model <https://docs.aws.amazon.com/lambda/latest/dg/foundation-progmodel.html>`_ .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-function.html#cfn-lambda-function-handler
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "handler"))

    @handler.setter
    def handler(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2bcf0d84f0318483745961f0ec3fd8d5ff0827a822159a3b6442923e4619db45)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "handler", value)

    @builtins.property
    @jsii.member(jsii_name="imageConfig")
    def image_config(
        self,
    ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnFunction.ImageConfigProperty"]]:
        '''Configuration values that override the container image Dockerfile settings.

        For more information, see `Container image settings <https://docs.aws.amazon.com/lambda/latest/dg/images-create.html#images-parms>`_ .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-function.html#cfn-lambda-function-imageconfig
        '''
        return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnFunction.ImageConfigProperty"]], jsii.get(self, "imageConfig"))

    @image_config.setter
    def image_config(
        self,
        value: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnFunction.ImageConfigProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8e9a4ccc263a47ed2541bb7a499676bec32060a7d11bed50961a10b7e466394e)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "imageConfig", value)

    @builtins.property
    @jsii.member(jsii_name="kmsKeyArn")
    def kms_key_arn(self) -> typing.Optional[builtins.str]:
        '''The ARN of the AWS Key Management Service ( AWS KMS ) key that's used to encrypt your function's environment variables.

        If it's not provided, Lambda uses a default service key.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-function.html#cfn-lambda-function-kmskeyarn
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "kmsKeyArn"))

    @kms_key_arn.setter
    def kms_key_arn(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__276f491bd6689780ec65352fa2b97628dadc00cd0b8e00681f196e1c14998e78)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "kmsKeyArn", value)

    @builtins.property
    @jsii.member(jsii_name="layers")
    def layers(self) -> typing.Optional[typing.List[builtins.str]]:
        '''A list of `function layers <https://docs.aws.amazon.com/lambda/latest/dg/configuration-layers.html>`_ to add to the function's execution environment. Specify each layer by its ARN, including the version.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-function.html#cfn-lambda-function-layers
        '''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "layers"))

    @layers.setter
    def layers(self, value: typing.Optional[typing.List[builtins.str]]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__feae41feb8915652f92c70bf3d21369821a5cb276c0fcb127e2b3b88c99e2cc0)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "layers", value)

    @builtins.property
    @jsii.member(jsii_name="memorySize")
    def memory_size(self) -> typing.Optional[jsii.Number]:
        '''The amount of `memory available to the function <https://docs.aws.amazon.com/lambda/latest/dg/configuration-function-common.html#configuration-memory-console>`_ at runtime. Increasing the function memory also increases its CPU allocation. The default value is 128 MB. The value can be any multiple of 1 MB.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-function.html#cfn-lambda-function-memorysize
        '''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "memorySize"))

    @memory_size.setter
    def memory_size(self, value: typing.Optional[jsii.Number]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__75da6e8994f743009946f78191bd0521efc8866d7fdec8490d56027d88ad09d6)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "memorySize", value)

    @builtins.property
    @jsii.member(jsii_name="packageType")
    def package_type(self) -> typing.Optional[builtins.str]:
        '''The type of deployment package.

        Set to ``Image`` for container image and set ``Zip`` for .zip file archive.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-function.html#cfn-lambda-function-packagetype
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "packageType"))

    @package_type.setter
    def package_type(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ac9d470e75061728bad7eb4c9224c9f93cad8d56ae4a0b5d3e892667d97b6f4d)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "packageType", value)

    @builtins.property
    @jsii.member(jsii_name="reservedConcurrentExecutions")
    def reserved_concurrent_executions(self) -> typing.Optional[jsii.Number]:
        '''The number of simultaneous executions to reserve for the function.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-function.html#cfn-lambda-function-reservedconcurrentexecutions
        '''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "reservedConcurrentExecutions"))

    @reserved_concurrent_executions.setter
    def reserved_concurrent_executions(
        self,
        value: typing.Optional[jsii.Number],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a78a951636fa87f6d576ab77119ea78e189c7b2dd6ac3386103fa9c386b2bb99)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "reservedConcurrentExecutions", value)

    @builtins.property
    @jsii.member(jsii_name="runtime")
    def runtime(self) -> typing.Optional[builtins.str]:
        '''The identifier of the function's `runtime <https://docs.aws.amazon.com/lambda/latest/dg/lambda-runtimes.html>`_ . Runtime is required if the deployment package is a .zip file archive.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-function.html#cfn-lambda-function-runtime
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "runtime"))

    @runtime.setter
    def runtime(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__6a5184a80443a91591ffe830e5e5fa03af993db0d823dea5cf42950b8cd58e0b)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "runtime", value)

    @builtins.property
    @jsii.member(jsii_name="snapStart")
    def snap_start(
        self,
    ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnFunction.SnapStartProperty"]]:
        '''The function's `AWS Lambda SnapStart <https://docs.aws.amazon.com/lambda/latest/dg/snapstart.html>`_ setting.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-function.html#cfn-lambda-function-snapstart
        '''
        return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnFunction.SnapStartProperty"]], jsii.get(self, "snapStart"))

    @snap_start.setter
    def snap_start(
        self,
        value: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnFunction.SnapStartProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__bf5c31b2b862805ce3ef8776a14f2170964b0c4a21908798a423b72b37d858ac)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "snapStart", value)

    @builtins.property
    @jsii.member(jsii_name="timeout")
    def timeout(self) -> typing.Optional[jsii.Number]:
        '''The amount of time (in seconds) that Lambda allows a function to run before stopping it.

        The default is 3 seconds. The maximum allowed value is 900 seconds. For more information, see `Lambda execution environment <https://docs.aws.amazon.com/lambda/latest/dg/runtimes-context.html>`_ .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-function.html#cfn-lambda-function-timeout
        '''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "timeout"))

    @timeout.setter
    def timeout(self, value: typing.Optional[jsii.Number]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__6105f877706a4b5397fc5385d8f77bed968c08ad7f5887f4c2e515f279d70977)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "timeout", value)

    @builtins.property
    @jsii.member(jsii_name="tracingConfig")
    def tracing_config(
        self,
    ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnFunction.TracingConfigProperty"]]:
        '''Set ``Mode`` to ``Active`` to sample and trace a subset of incoming requests with `X-Ray <https://docs.aws.amazon.com/lambda/latest/dg/services-xray.html>`_ .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-function.html#cfn-lambda-function-tracingconfig
        '''
        return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnFunction.TracingConfigProperty"]], jsii.get(self, "tracingConfig"))

    @tracing_config.setter
    def tracing_config(
        self,
        value: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnFunction.TracingConfigProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c3fe7c6fe29f9b1368634494b52710fa3853cef5a5f51544ceece717894c17a6)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "tracingConfig", value)

    @builtins.property
    @jsii.member(jsii_name="vpcConfig")
    def vpc_config(
        self,
    ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnFunction.VpcConfigProperty"]]:
        '''For network connectivity to AWS resources in a `VPC <https://docs.aws.amazon.com/lambda/latest/dg/configuration-network.html>`_ , specify a list of security groups and subnets in the VPC.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-function.html#cfn-lambda-function-vpcconfig
        '''
        return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnFunction.VpcConfigProperty"]], jsii.get(self, "vpcConfig"))

    @vpc_config.setter
    def vpc_config(
        self,
        value: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnFunction.VpcConfigProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__155f1268f21c38dd86a358d0b23eff87bd79fe7fcc8b246ee64aae143b42a5ea)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "vpcConfig", value)

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-lambda.CfnFunction.CodeProperty",
        jsii_struct_bases=[],
        name_mapping={
            "image_uri": "imageUri",
            "s3_bucket": "s3Bucket",
            "s3_key": "s3Key",
            "s3_object_version": "s3ObjectVersion",
            "zip_file": "zipFile",
        },
    )
    class CodeProperty:
        def __init__(
            self,
            *,
            image_uri: typing.Optional[builtins.str] = None,
            s3_bucket: typing.Optional[builtins.str] = None,
            s3_key: typing.Optional[builtins.str] = None,
            s3_object_version: typing.Optional[builtins.str] = None,
            zip_file: typing.Optional[builtins.str] = None,
        ) -> None:
            '''The `deployment package <https://docs.aws.amazon.com/lambda/latest/dg/gettingstarted-package.html>`_ for a Lambda function. To deploy a function defined as a container image, you specify the location of a container image in the Amazon ECR registry. For a .zip file deployment package, you can specify the location of an object in Amazon S3. For Node.js and Python functions, you can specify the function code inline in the template.

            Changes to a deployment package in Amazon S3 are not detected automatically during stack updates. To update the function code, change the object key or version in the template.

            :param image_uri: URI of a `container image <https://docs.aws.amazon.com/lambda/latest/dg/lambda-images.html>`_ in the Amazon ECR registry.
            :param s3_bucket: An Amazon S3 bucket in the same AWS Region as your function. The bucket can be in a different AWS account .
            :param s3_key: The Amazon S3 key of the deployment package.
            :param s3_object_version: For versioned objects, the version of the deployment package object to use.
            :param zip_file: (Node.js and Python) The source code of your Lambda function. If you include your function source inline with this parameter, AWS CloudFormation places it in a file named ``index`` and zips it to create a `deployment package <https://docs.aws.amazon.com/lambda/latest/dg/gettingstarted-package.html>`_ . This zip file cannot exceed 4MB. For the ``Handler`` property, the first part of the handler identifier must be ``index`` . For example, ``index.handler`` . For JSON, you must escape quotes and special characters such as newline ( ``\\n`` ) with a backslash. If you specify a function that interacts with an AWS CloudFormation custom resource, you don't have to write your own functions to send responses to the custom resource that invoked the function. AWS CloudFormation provides a response module ( `cfn-response <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cfn-lambda-function-code-cfnresponsemodule.html>`_ ) that simplifies sending responses. See `Using AWS Lambda with AWS CloudFormation <https://docs.aws.amazon.com/lambda/latest/dg/services-cloudformation.html>`_ for details.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-function-code.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_lambda as lambda_
                
                code_property = lambda.CfnFunction.CodeProperty(
                    image_uri="imageUri",
                    s3_bucket="s3Bucket",
                    s3_key="s3Key",
                    s3_object_version="s3ObjectVersion",
                    zip_file="zipFile"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__3898d7f7e9c082641c04fb784a66eeaa7b2acac8a61dddbb94a5127c4a34c4a2)
                check_type(argname="argument image_uri", value=image_uri, expected_type=type_hints["image_uri"])
                check_type(argname="argument s3_bucket", value=s3_bucket, expected_type=type_hints["s3_bucket"])
                check_type(argname="argument s3_key", value=s3_key, expected_type=type_hints["s3_key"])
                check_type(argname="argument s3_object_version", value=s3_object_version, expected_type=type_hints["s3_object_version"])
                check_type(argname="argument zip_file", value=zip_file, expected_type=type_hints["zip_file"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if image_uri is not None:
                self._values["image_uri"] = image_uri
            if s3_bucket is not None:
                self._values["s3_bucket"] = s3_bucket
            if s3_key is not None:
                self._values["s3_key"] = s3_key
            if s3_object_version is not None:
                self._values["s3_object_version"] = s3_object_version
            if zip_file is not None:
                self._values["zip_file"] = zip_file

        @builtins.property
        def image_uri(self) -> typing.Optional[builtins.str]:
            '''URI of a `container image <https://docs.aws.amazon.com/lambda/latest/dg/lambda-images.html>`_ in the Amazon ECR registry.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-function-code.html#cfn-lambda-function-code-imageuri
            '''
            result = self._values.get("image_uri")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def s3_bucket(self) -> typing.Optional[builtins.str]:
            '''An Amazon S3 bucket in the same AWS Region as your function.

            The bucket can be in a different AWS account .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-function-code.html#cfn-lambda-function-code-s3bucket
            '''
            result = self._values.get("s3_bucket")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def s3_key(self) -> typing.Optional[builtins.str]:
            '''The Amazon S3 key of the deployment package.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-function-code.html#cfn-lambda-function-code-s3key
            '''
            result = self._values.get("s3_key")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def s3_object_version(self) -> typing.Optional[builtins.str]:
            '''For versioned objects, the version of the deployment package object to use.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-function-code.html#cfn-lambda-function-code-s3objectversion
            '''
            result = self._values.get("s3_object_version")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def zip_file(self) -> typing.Optional[builtins.str]:
            '''(Node.js and Python) The source code of your Lambda function. If you include your function source inline with this parameter, AWS CloudFormation places it in a file named ``index`` and zips it to create a `deployment package <https://docs.aws.amazon.com/lambda/latest/dg/gettingstarted-package.html>`_ . This zip file cannot exceed 4MB. For the ``Handler`` property, the first part of the handler identifier must be ``index`` . For example, ``index.handler`` .

            For JSON, you must escape quotes and special characters such as newline ( ``\\n`` ) with a backslash.

            If you specify a function that interacts with an AWS CloudFormation custom resource, you don't have to write your own functions to send responses to the custom resource that invoked the function. AWS CloudFormation provides a response module ( `cfn-response <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cfn-lambda-function-code-cfnresponsemodule.html>`_ ) that simplifies sending responses. See `Using AWS Lambda with AWS CloudFormation <https://docs.aws.amazon.com/lambda/latest/dg/services-cloudformation.html>`_ for details.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-function-code.html#cfn-lambda-function-code-zipfile
            '''
            result = self._values.get("zip_file")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "CodeProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-lambda.CfnFunction.DeadLetterConfigProperty",
        jsii_struct_bases=[],
        name_mapping={"target_arn": "targetArn"},
    )
    class DeadLetterConfigProperty:
        def __init__(self, *, target_arn: typing.Optional[builtins.str] = None) -> None:
            '''The `dead-letter queue <https://docs.aws.amazon.com/lambda/latest/dg/invocation-async.html#dlq>`_ for failed asynchronous invocations.

            :param target_arn: The Amazon Resource Name (ARN) of an Amazon SQS queue or Amazon SNS topic.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-function-deadletterconfig.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_lambda as lambda_
                
                dead_letter_config_property = lambda.CfnFunction.DeadLetterConfigProperty(
                    target_arn="targetArn"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__7dc6302e7d643f0131328e3ea6e6a352fa0900ed6374413239904147d7383676)
                check_type(argname="argument target_arn", value=target_arn, expected_type=type_hints["target_arn"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if target_arn is not None:
                self._values["target_arn"] = target_arn

        @builtins.property
        def target_arn(self) -> typing.Optional[builtins.str]:
            '''The Amazon Resource Name (ARN) of an Amazon SQS queue or Amazon SNS topic.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-function-deadletterconfig.html#cfn-lambda-function-deadletterconfig-targetarn
            '''
            result = self._values.get("target_arn")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "DeadLetterConfigProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-lambda.CfnFunction.EnvironmentProperty",
        jsii_struct_bases=[],
        name_mapping={"variables": "variables"},
    )
    class EnvironmentProperty:
        def __init__(
            self,
            *,
            variables: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Mapping[builtins.str, builtins.str]]] = None,
        ) -> None:
            '''A function's environment variable settings.

            You can use environment variables to adjust your function's behavior without updating code. An environment variable is a pair of strings that are stored in a function's version-specific configuration.

            :param variables: Environment variable key-value pairs. For more information, see `Using Lambda environment variables <https://docs.aws.amazon.com/lambda/latest/dg/configuration-envvars.html>`_ .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-function-environment.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_lambda as lambda_
                
                environment_property = lambda.CfnFunction.EnvironmentProperty(
                    variables={
                        "variables_key": "variables"
                    }
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__c7e7028a72cba9773b41134347d890b82d8d41587f68e31ba1247335a8e00c50)
                check_type(argname="argument variables", value=variables, expected_type=type_hints["variables"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if variables is not None:
                self._values["variables"] = variables

        @builtins.property
        def variables(
            self,
        ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Mapping[builtins.str, builtins.str]]]:
            '''Environment variable key-value pairs.

            For more information, see `Using Lambda environment variables <https://docs.aws.amazon.com/lambda/latest/dg/configuration-envvars.html>`_ .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-function-environment.html#cfn-lambda-function-environment-variables
            '''
            result = self._values.get("variables")
            return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Mapping[builtins.str, builtins.str]]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "EnvironmentProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-lambda.CfnFunction.EphemeralStorageProperty",
        jsii_struct_bases=[],
        name_mapping={"size": "size"},
    )
    class EphemeralStorageProperty:
        def __init__(self, *, size: jsii.Number) -> None:
            '''The size of the function's ``/tmp`` directory in MB.

            The default value is 512, but it can be any whole number between 512 and 10,240 MB.

            :param size: The size of the function's ``/tmp`` directory.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-function-ephemeralstorage.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_lambda as lambda_
                
                ephemeral_storage_property = lambda.CfnFunction.EphemeralStorageProperty(
                    size=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__6f0e1692fa2d57cd0bc7c4ba7ecfea1929c5c74c7c034c6d251ee3da5e3725e8)
                check_type(argname="argument size", value=size, expected_type=type_hints["size"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "size": size,
            }

        @builtins.property
        def size(self) -> jsii.Number:
            '''The size of the function's ``/tmp`` directory.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-function-ephemeralstorage.html#cfn-lambda-function-ephemeralstorage-size
            '''
            result = self._values.get("size")
            assert result is not None, "Required property 'size' is missing"
            return typing.cast(jsii.Number, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "EphemeralStorageProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-lambda.CfnFunction.FileSystemConfigProperty",
        jsii_struct_bases=[],
        name_mapping={"arn": "arn", "local_mount_path": "localMountPath"},
    )
    class FileSystemConfigProperty:
        def __init__(
            self,
            *,
            arn: builtins.str,
            local_mount_path: builtins.str,
        ) -> None:
            '''Details about the connection between a Lambda function and an `Amazon EFS file system <https://docs.aws.amazon.com/lambda/latest/dg/configuration-filesystem.html>`_ .

            :param arn: The Amazon Resource Name (ARN) of the Amazon EFS access point that provides access to the file system.
            :param local_mount_path: The path where the function can access the file system, starting with ``/mnt/`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-function-filesystemconfig.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_lambda as lambda_
                
                file_system_config_property = lambda.CfnFunction.FileSystemConfigProperty(
                    arn="arn",
                    local_mount_path="localMountPath"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__adc36851daef14bdc03afeaf05a9791d3478e731abe1386d6d35a96ccda8bcef)
                check_type(argname="argument arn", value=arn, expected_type=type_hints["arn"])
                check_type(argname="argument local_mount_path", value=local_mount_path, expected_type=type_hints["local_mount_path"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "arn": arn,
                "local_mount_path": local_mount_path,
            }

        @builtins.property
        def arn(self) -> builtins.str:
            '''The Amazon Resource Name (ARN) of the Amazon EFS access point that provides access to the file system.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-function-filesystemconfig.html#cfn-lambda-function-filesystemconfig-arn
            '''
            result = self._values.get("arn")
            assert result is not None, "Required property 'arn' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def local_mount_path(self) -> builtins.str:
            '''The path where the function can access the file system, starting with ``/mnt/`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-function-filesystemconfig.html#cfn-lambda-function-filesystemconfig-localmountpath
            '''
            result = self._values.get("local_mount_path")
            assert result is not None, "Required property 'local_mount_path' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "FileSystemConfigProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-lambda.CfnFunction.ImageConfigProperty",
        jsii_struct_bases=[],
        name_mapping={
            "command": "command",
            "entry_point": "entryPoint",
            "working_directory": "workingDirectory",
        },
    )
    class ImageConfigProperty:
        def __init__(
            self,
            *,
            command: typing.Optional[typing.Sequence[builtins.str]] = None,
            entry_point: typing.Optional[typing.Sequence[builtins.str]] = None,
            working_directory: typing.Optional[builtins.str] = None,
        ) -> None:
            '''Configuration values that override the container image Dockerfile settings.

            For more information, see `Container image settings <https://docs.aws.amazon.com/lambda/latest/dg/images-create.html#images-parms>`_ .

            :param command: Specifies parameters that you want to pass in with ENTRYPOINT.
            :param entry_point: Specifies the entry point to their application, which is typically the location of the runtime executable.
            :param working_directory: Specifies the working directory.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-function-imageconfig.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_lambda as lambda_
                
                image_config_property = lambda.CfnFunction.ImageConfigProperty(
                    command=["command"],
                    entry_point=["entryPoint"],
                    working_directory="workingDirectory"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__7b3dc1d895525748ebca8a2349c2c087e7d3274cf329009f5beae9a7dc92d27d)
                check_type(argname="argument command", value=command, expected_type=type_hints["command"])
                check_type(argname="argument entry_point", value=entry_point, expected_type=type_hints["entry_point"])
                check_type(argname="argument working_directory", value=working_directory, expected_type=type_hints["working_directory"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if command is not None:
                self._values["command"] = command
            if entry_point is not None:
                self._values["entry_point"] = entry_point
            if working_directory is not None:
                self._values["working_directory"] = working_directory

        @builtins.property
        def command(self) -> typing.Optional[typing.List[builtins.str]]:
            '''Specifies parameters that you want to pass in with ENTRYPOINT.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-function-imageconfig.html#cfn-lambda-function-imageconfig-command
            '''
            result = self._values.get("command")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def entry_point(self) -> typing.Optional[typing.List[builtins.str]]:
            '''Specifies the entry point to their application, which is typically the location of the runtime executable.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-function-imageconfig.html#cfn-lambda-function-imageconfig-entrypoint
            '''
            result = self._values.get("entry_point")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def working_directory(self) -> typing.Optional[builtins.str]:
            '''Specifies the working directory.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-function-imageconfig.html#cfn-lambda-function-imageconfig-workingdirectory
            '''
            result = self._values.get("working_directory")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ImageConfigProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-lambda.CfnFunction.SnapStartProperty",
        jsii_struct_bases=[],
        name_mapping={"apply_on": "applyOn"},
    )
    class SnapStartProperty:
        def __init__(self, *, apply_on: builtins.str) -> None:
            '''The function's `AWS Lambda SnapStart <https://docs.aws.amazon.com/lambda/latest/dg/snapstart.html>`_ setting.

            :param apply_on: Set ``ApplyOn`` to ``PublishedVersions`` to create a snapshot of the initialized execution environment when you publish a function version.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-function-snapstart.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_lambda as lambda_
                
                snap_start_property = lambda.CfnFunction.SnapStartProperty(
                    apply_on="applyOn"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__9af3422a367ec203eac7fd8c29fb18778ecf325f04955373ec9c47b8aab7e086)
                check_type(argname="argument apply_on", value=apply_on, expected_type=type_hints["apply_on"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "apply_on": apply_on,
            }

        @builtins.property
        def apply_on(self) -> builtins.str:
            '''Set ``ApplyOn`` to ``PublishedVersions`` to create a snapshot of the initialized execution environment when you publish a function version.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-function-snapstart.html#cfn-lambda-function-snapstart-applyon
            '''
            result = self._values.get("apply_on")
            assert result is not None, "Required property 'apply_on' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "SnapStartProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-lambda.CfnFunction.SnapStartResponseProperty",
        jsii_struct_bases=[],
        name_mapping={
            "apply_on": "applyOn",
            "optimization_status": "optimizationStatus",
        },
    )
    class SnapStartResponseProperty:
        def __init__(
            self,
            *,
            apply_on: typing.Optional[builtins.str] = None,
            optimization_status: typing.Optional[builtins.str] = None,
        ) -> None:
            '''The function's `SnapStart <https://docs.aws.amazon.com/lambda/latest/dg/snapstart.html>`_ setting.

            :param apply_on: When set to ``PublishedVersions`` , Lambda creates a snapshot of the execution environment when you publish a function version.
            :param optimization_status: When you provide a `qualified Amazon Resource Name (ARN) <https://docs.aws.amazon.com/lambda/latest/dg/configuration-versions.html#versioning-versions-using>`_ , this response element indicates whether SnapStart is activated for the specified function version.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-function-snapstartresponse.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_lambda as lambda_
                
                snap_start_response_property = lambda.CfnFunction.SnapStartResponseProperty(
                    apply_on="applyOn",
                    optimization_status="optimizationStatus"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__2067523b542ecbb506e48c133069f4ff84ca51229ae7820bfee17ab71025c834)
                check_type(argname="argument apply_on", value=apply_on, expected_type=type_hints["apply_on"])
                check_type(argname="argument optimization_status", value=optimization_status, expected_type=type_hints["optimization_status"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if apply_on is not None:
                self._values["apply_on"] = apply_on
            if optimization_status is not None:
                self._values["optimization_status"] = optimization_status

        @builtins.property
        def apply_on(self) -> typing.Optional[builtins.str]:
            '''When set to ``PublishedVersions`` , Lambda creates a snapshot of the execution environment when you publish a function version.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-function-snapstartresponse.html#cfn-lambda-function-snapstartresponse-applyon
            '''
            result = self._values.get("apply_on")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def optimization_status(self) -> typing.Optional[builtins.str]:
            '''When you provide a `qualified Amazon Resource Name (ARN) <https://docs.aws.amazon.com/lambda/latest/dg/configuration-versions.html#versioning-versions-using>`_ , this response element indicates whether SnapStart is activated for the specified function version.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-function-snapstartresponse.html#cfn-lambda-function-snapstartresponse-optimizationstatus
            '''
            result = self._values.get("optimization_status")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "SnapStartResponseProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-lambda.CfnFunction.TracingConfigProperty",
        jsii_struct_bases=[],
        name_mapping={"mode": "mode"},
    )
    class TracingConfigProperty:
        def __init__(self, *, mode: typing.Optional[builtins.str] = None) -> None:
            '''The function's `AWS X-Ray <https://docs.aws.amazon.com/lambda/latest/dg/services-xray.html>`_ tracing configuration. To sample and record incoming requests, set ``Mode`` to ``Active`` .

            :param mode: The tracing mode.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-function-tracingconfig.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_lambda as lambda_
                
                tracing_config_property = lambda.CfnFunction.TracingConfigProperty(
                    mode="mode"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__8ee4486c06fcad78ef62afd98f19b90d5bf548aed7f3ade776c80e717f16623e)
                check_type(argname="argument mode", value=mode, expected_type=type_hints["mode"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if mode is not None:
                self._values["mode"] = mode

        @builtins.property
        def mode(self) -> typing.Optional[builtins.str]:
            '''The tracing mode.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-function-tracingconfig.html#cfn-lambda-function-tracingconfig-mode
            '''
            result = self._values.get("mode")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "TracingConfigProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-lambda.CfnFunction.VpcConfigProperty",
        jsii_struct_bases=[],
        name_mapping={
            "security_group_ids": "securityGroupIds",
            "subnet_ids": "subnetIds",
        },
    )
    class VpcConfigProperty:
        def __init__(
            self,
            *,
            security_group_ids: typing.Optional[typing.Sequence[builtins.str]] = None,
            subnet_ids: typing.Optional[typing.Sequence[builtins.str]] = None,
        ) -> None:
            '''The VPC security groups and subnets that are attached to a Lambda function.

            When you connect a function to a VPC, Lambda creates an elastic network interface for each combination of security group and subnet in the function's VPC configuration. The function can only access resources and the internet through that VPC. For more information, see `VPC Settings <https://docs.aws.amazon.com/lambda/latest/dg/configuration-vpc.html>`_ .
            .. epigraph::

               When you delete a function, AWS CloudFormation monitors the state of its network interfaces and waits for Lambda to delete them before proceeding. If the VPC is defined in the same stack, the network interfaces need to be deleted by Lambda before AWS CloudFormation can delete the VPC's resources.

               To monitor network interfaces, AWS CloudFormation needs the ``ec2:DescribeNetworkInterfaces`` permission. It obtains this from the user or role that modifies the stack. If you don't provide this permission, AWS CloudFormation does not wait for network interfaces to be deleted.

            :param security_group_ids: A list of VPC security group IDs.
            :param subnet_ids: A list of VPC subnet IDs.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-function-vpcconfig.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_lambda as lambda_
                
                vpc_config_property = lambda.CfnFunction.VpcConfigProperty(
                    security_group_ids=["securityGroupIds"],
                    subnet_ids=["subnetIds"]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__06f87e83c8536df437dc426f54dc7f733b6d64c14a740c813db0b8f96ec55d00)
                check_type(argname="argument security_group_ids", value=security_group_ids, expected_type=type_hints["security_group_ids"])
                check_type(argname="argument subnet_ids", value=subnet_ids, expected_type=type_hints["subnet_ids"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if security_group_ids is not None:
                self._values["security_group_ids"] = security_group_ids
            if subnet_ids is not None:
                self._values["subnet_ids"] = subnet_ids

        @builtins.property
        def security_group_ids(self) -> typing.Optional[typing.List[builtins.str]]:
            '''A list of VPC security group IDs.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-function-vpcconfig.html#cfn-lambda-function-vpcconfig-securitygroupids
            '''
            result = self._values.get("security_group_ids")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def subnet_ids(self) -> typing.Optional[typing.List[builtins.str]]:
            '''A list of VPC subnet IDs.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-function-vpcconfig.html#cfn-lambda-function-vpcconfig-subnetids
            '''
            result = self._values.get("subnet_ids")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VpcConfigProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-lambda.CfnFunctionProps",
    jsii_struct_bases=[],
    name_mapping={
        "code": "code",
        "role": "role",
        "architectures": "architectures",
        "code_signing_config_arn": "codeSigningConfigArn",
        "dead_letter_config": "deadLetterConfig",
        "description": "description",
        "environment": "environment",
        "ephemeral_storage": "ephemeralStorage",
        "file_system_configs": "fileSystemConfigs",
        "function_name": "functionName",
        "handler": "handler",
        "image_config": "imageConfig",
        "kms_key_arn": "kmsKeyArn",
        "layers": "layers",
        "memory_size": "memorySize",
        "package_type": "packageType",
        "reserved_concurrent_executions": "reservedConcurrentExecutions",
        "runtime": "runtime",
        "snap_start": "snapStart",
        "tags": "tags",
        "timeout": "timeout",
        "tracing_config": "tracingConfig",
        "vpc_config": "vpcConfig",
    },
)
class CfnFunctionProps:
    def __init__(
        self,
        *,
        code: typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union[CfnFunction.CodeProperty, typing.Dict[builtins.str, typing.Any]]],
        role: builtins.str,
        architectures: typing.Optional[typing.Sequence[builtins.str]] = None,
        code_signing_config_arn: typing.Optional[builtins.str] = None,
        dead_letter_config: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union[CfnFunction.DeadLetterConfigProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        description: typing.Optional[builtins.str] = None,
        environment: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union[CfnFunction.EnvironmentProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        ephemeral_storage: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union[CfnFunction.EphemeralStorageProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        file_system_configs: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Sequence[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union[CfnFunction.FileSystemConfigProperty, typing.Dict[builtins.str, typing.Any]]]]]] = None,
        function_name: typing.Optional[builtins.str] = None,
        handler: typing.Optional[builtins.str] = None,
        image_config: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union[CfnFunction.ImageConfigProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        kms_key_arn: typing.Optional[builtins.str] = None,
        layers: typing.Optional[typing.Sequence[builtins.str]] = None,
        memory_size: typing.Optional[jsii.Number] = None,
        package_type: typing.Optional[builtins.str] = None,
        reserved_concurrent_executions: typing.Optional[jsii.Number] = None,
        runtime: typing.Optional[builtins.str] = None,
        snap_start: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union[CfnFunction.SnapStartProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_aws_cdk_core_f4b25747.CfnTag, typing.Dict[builtins.str, typing.Any]]]] = None,
        timeout: typing.Optional[jsii.Number] = None,
        tracing_config: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union[CfnFunction.TracingConfigProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        vpc_config: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union[CfnFunction.VpcConfigProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''Properties for defining a ``CfnFunction``.

        :param code: The code for the function.
        :param role: The Amazon Resource Name (ARN) of the function's execution role.
        :param architectures: The instruction set architecture that the function supports. Enter a string array with one of the valid values (arm64 or x86_64). The default value is ``x86_64`` .
        :param code_signing_config_arn: To enable code signing for this function, specify the ARN of a code-signing configuration. A code-signing configuration includes a set of signing profiles, which define the trusted publishers for this function.
        :param dead_letter_config: A dead-letter queue configuration that specifies the queue or topic where Lambda sends asynchronous events when they fail processing. For more information, see `Dead-letter queues <https://docs.aws.amazon.com/lambda/latest/dg/invocation-async.html#invocation-dlq>`_ .
        :param description: A description of the function.
        :param environment: Environment variables that are accessible from function code during execution.
        :param ephemeral_storage: The size of the function's ``/tmp`` directory in MB. The default value is 512, but it can be any whole number between 512 and 10,240 MB.
        :param file_system_configs: Connection settings for an Amazon EFS file system. To connect a function to a file system, a mount target must be available in every Availability Zone that your function connects to. If your template contains an `AWS::EFS::MountTarget <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-efs-mounttarget.html>`_ resource, you must also specify a ``DependsOn`` attribute to ensure that the mount target is created or updated before the function. For more information about using the ``DependsOn`` attribute, see `DependsOn Attribute <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-attribute-dependson.html>`_ .
        :param function_name: The name of the Lambda function, up to 64 characters in length. If you don't specify a name, AWS CloudFormation generates one. If you specify a name, you cannot perform updates that require replacement of this resource. You can perform updates that require no or some interruption. If you must replace the resource, specify a new name.
        :param handler: The name of the method within your code that Lambda calls to run your function. Handler is required if the deployment package is a .zip file archive. The format includes the file name. It can also include namespaces and other qualifiers, depending on the runtime. For more information, see `Lambda programming model <https://docs.aws.amazon.com/lambda/latest/dg/foundation-progmodel.html>`_ .
        :param image_config: Configuration values that override the container image Dockerfile settings. For more information, see `Container image settings <https://docs.aws.amazon.com/lambda/latest/dg/images-create.html#images-parms>`_ .
        :param kms_key_arn: The ARN of the AWS Key Management Service ( AWS KMS ) key that's used to encrypt your function's environment variables. If it's not provided, Lambda uses a default service key.
        :param layers: A list of `function layers <https://docs.aws.amazon.com/lambda/latest/dg/configuration-layers.html>`_ to add to the function's execution environment. Specify each layer by its ARN, including the version.
        :param memory_size: The amount of `memory available to the function <https://docs.aws.amazon.com/lambda/latest/dg/configuration-function-common.html#configuration-memory-console>`_ at runtime. Increasing the function memory also increases its CPU allocation. The default value is 128 MB. The value can be any multiple of 1 MB.
        :param package_type: The type of deployment package. Set to ``Image`` for container image and set ``Zip`` for .zip file archive.
        :param reserved_concurrent_executions: The number of simultaneous executions to reserve for the function.
        :param runtime: The identifier of the function's `runtime <https://docs.aws.amazon.com/lambda/latest/dg/lambda-runtimes.html>`_ . Runtime is required if the deployment package is a .zip file archive.
        :param snap_start: The function's `AWS Lambda SnapStart <https://docs.aws.amazon.com/lambda/latest/dg/snapstart.html>`_ setting.
        :param tags: A list of `tags <https://docs.aws.amazon.com/lambda/latest/dg/tagging.html>`_ to apply to the function.
        :param timeout: The amount of time (in seconds) that Lambda allows a function to run before stopping it. The default is 3 seconds. The maximum allowed value is 900 seconds. For more information, see `Lambda execution environment <https://docs.aws.amazon.com/lambda/latest/dg/runtimes-context.html>`_ .
        :param tracing_config: Set ``Mode`` to ``Active`` to sample and trace a subset of incoming requests with `X-Ray <https://docs.aws.amazon.com/lambda/latest/dg/services-xray.html>`_ .
        :param vpc_config: For network connectivity to AWS resources in a `VPC <https://docs.aws.amazon.com/lambda/latest/dg/configuration-network.html>`_ , specify a list of security groups and subnets in the VPC.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-function.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_lambda as lambda_
            
            cfn_function_props = lambda.CfnFunctionProps(
                code=lambda.CfnFunction.CodeProperty(
                    image_uri="imageUri",
                    s3_bucket="s3Bucket",
                    s3_key="s3Key",
                    s3_object_version="s3ObjectVersion",
                    zip_file="zipFile"
                ),
                role="role",
            
                # the properties below are optional
                architectures=["architectures"],
                code_signing_config_arn="codeSigningConfigArn",
                dead_letter_config=lambda.CfnFunction.DeadLetterConfigProperty(
                    target_arn="targetArn"
                ),
                description="description",
                environment=lambda.CfnFunction.EnvironmentProperty(
                    variables={
                        "variables_key": "variables"
                    }
                ),
                ephemeral_storage=lambda.CfnFunction.EphemeralStorageProperty(
                    size=123
                ),
                file_system_configs=[lambda.CfnFunction.FileSystemConfigProperty(
                    arn="arn",
                    local_mount_path="localMountPath"
                )],
                function_name="functionName",
                handler="handler",
                image_config=lambda.CfnFunction.ImageConfigProperty(
                    command=["command"],
                    entry_point=["entryPoint"],
                    working_directory="workingDirectory"
                ),
                kms_key_arn="kmsKeyArn",
                layers=["layers"],
                memory_size=123,
                package_type="packageType",
                reserved_concurrent_executions=123,
                runtime="runtime",
                snap_start=lambda.CfnFunction.SnapStartProperty(
                    apply_on="applyOn"
                ),
                tags=[CfnTag(
                    key="key",
                    value="value"
                )],
                timeout=123,
                tracing_config=lambda.CfnFunction.TracingConfigProperty(
                    mode="mode"
                ),
                vpc_config=lambda.CfnFunction.VpcConfigProperty(
                    security_group_ids=["securityGroupIds"],
                    subnet_ids=["subnetIds"]
                )
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__83cf589d9d7de858ff0fbc2e5f9312b2bad9ceff29e192e7872bb7368eed3107)
            check_type(argname="argument code", value=code, expected_type=type_hints["code"])
            check_type(argname="argument role", value=role, expected_type=type_hints["role"])
            check_type(argname="argument architectures", value=architectures, expected_type=type_hints["architectures"])
            check_type(argname="argument code_signing_config_arn", value=code_signing_config_arn, expected_type=type_hints["code_signing_config_arn"])
            check_type(argname="argument dead_letter_config", value=dead_letter_config, expected_type=type_hints["dead_letter_config"])
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument environment", value=environment, expected_type=type_hints["environment"])
            check_type(argname="argument ephemeral_storage", value=ephemeral_storage, expected_type=type_hints["ephemeral_storage"])
            check_type(argname="argument file_system_configs", value=file_system_configs, expected_type=type_hints["file_system_configs"])
            check_type(argname="argument function_name", value=function_name, expected_type=type_hints["function_name"])
            check_type(argname="argument handler", value=handler, expected_type=type_hints["handler"])
            check_type(argname="argument image_config", value=image_config, expected_type=type_hints["image_config"])
            check_type(argname="argument kms_key_arn", value=kms_key_arn, expected_type=type_hints["kms_key_arn"])
            check_type(argname="argument layers", value=layers, expected_type=type_hints["layers"])
            check_type(argname="argument memory_size", value=memory_size, expected_type=type_hints["memory_size"])
            check_type(argname="argument package_type", value=package_type, expected_type=type_hints["package_type"])
            check_type(argname="argument reserved_concurrent_executions", value=reserved_concurrent_executions, expected_type=type_hints["reserved_concurrent_executions"])
            check_type(argname="argument runtime", value=runtime, expected_type=type_hints["runtime"])
            check_type(argname="argument snap_start", value=snap_start, expected_type=type_hints["snap_start"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
            check_type(argname="argument timeout", value=timeout, expected_type=type_hints["timeout"])
            check_type(argname="argument tracing_config", value=tracing_config, expected_type=type_hints["tracing_config"])
            check_type(argname="argument vpc_config", value=vpc_config, expected_type=type_hints["vpc_config"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "code": code,
            "role": role,
        }
        if architectures is not None:
            self._values["architectures"] = architectures
        if code_signing_config_arn is not None:
            self._values["code_signing_config_arn"] = code_signing_config_arn
        if dead_letter_config is not None:
            self._values["dead_letter_config"] = dead_letter_config
        if description is not None:
            self._values["description"] = description
        if environment is not None:
            self._values["environment"] = environment
        if ephemeral_storage is not None:
            self._values["ephemeral_storage"] = ephemeral_storage
        if file_system_configs is not None:
            self._values["file_system_configs"] = file_system_configs
        if function_name is not None:
            self._values["function_name"] = function_name
        if handler is not None:
            self._values["handler"] = handler
        if image_config is not None:
            self._values["image_config"] = image_config
        if kms_key_arn is not None:
            self._values["kms_key_arn"] = kms_key_arn
        if layers is not None:
            self._values["layers"] = layers
        if memory_size is not None:
            self._values["memory_size"] = memory_size
        if package_type is not None:
            self._values["package_type"] = package_type
        if reserved_concurrent_executions is not None:
            self._values["reserved_concurrent_executions"] = reserved_concurrent_executions
        if runtime is not None:
            self._values["runtime"] = runtime
        if snap_start is not None:
            self._values["snap_start"] = snap_start
        if tags is not None:
            self._values["tags"] = tags
        if timeout is not None:
            self._values["timeout"] = timeout
        if tracing_config is not None:
            self._values["tracing_config"] = tracing_config
        if vpc_config is not None:
            self._values["vpc_config"] = vpc_config

    @builtins.property
    def code(
        self,
    ) -> typing.Union[_aws_cdk_core_f4b25747.IResolvable, CfnFunction.CodeProperty]:
        '''The code for the function.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-function.html#cfn-lambda-function-code
        '''
        result = self._values.get("code")
        assert result is not None, "Required property 'code' is missing"
        return typing.cast(typing.Union[_aws_cdk_core_f4b25747.IResolvable, CfnFunction.CodeProperty], result)

    @builtins.property
    def role(self) -> builtins.str:
        '''The Amazon Resource Name (ARN) of the function's execution role.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-function.html#cfn-lambda-function-role
        '''
        result = self._values.get("role")
        assert result is not None, "Required property 'role' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def architectures(self) -> typing.Optional[typing.List[builtins.str]]:
        '''The instruction set architecture that the function supports.

        Enter a string array with one of the valid values (arm64 or x86_64). The default value is ``x86_64`` .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-function.html#cfn-lambda-function-architectures
        '''
        result = self._values.get("architectures")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def code_signing_config_arn(self) -> typing.Optional[builtins.str]:
        '''To enable code signing for this function, specify the ARN of a code-signing configuration.

        A code-signing configuration
        includes a set of signing profiles, which define the trusted publishers for this function.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-function.html#cfn-lambda-function-codesigningconfigarn
        '''
        result = self._values.get("code_signing_config_arn")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def dead_letter_config(
        self,
    ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, CfnFunction.DeadLetterConfigProperty]]:
        '''A dead-letter queue configuration that specifies the queue or topic where Lambda sends asynchronous events when they fail processing.

        For more information, see `Dead-letter queues <https://docs.aws.amazon.com/lambda/latest/dg/invocation-async.html#invocation-dlq>`_ .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-function.html#cfn-lambda-function-deadletterconfig
        '''
        result = self._values.get("dead_letter_config")
        return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, CfnFunction.DeadLetterConfigProperty]], result)

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''A description of the function.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-function.html#cfn-lambda-function-description
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def environment(
        self,
    ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, CfnFunction.EnvironmentProperty]]:
        '''Environment variables that are accessible from function code during execution.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-function.html#cfn-lambda-function-environment
        '''
        result = self._values.get("environment")
        return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, CfnFunction.EnvironmentProperty]], result)

    @builtins.property
    def ephemeral_storage(
        self,
    ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, CfnFunction.EphemeralStorageProperty]]:
        '''The size of the function's ``/tmp`` directory in MB.

        The default value is 512, but it can be any whole number between 512 and 10,240 MB.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-function.html#cfn-lambda-function-ephemeralstorage
        '''
        result = self._values.get("ephemeral_storage")
        return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, CfnFunction.EphemeralStorageProperty]], result)

    @builtins.property
    def file_system_configs(
        self,
    ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.List[typing.Union[_aws_cdk_core_f4b25747.IResolvable, CfnFunction.FileSystemConfigProperty]]]]:
        '''Connection settings for an Amazon EFS file system.

        To connect a function to a file system, a mount target must be available in every Availability Zone that your function connects to. If your template contains an `AWS::EFS::MountTarget <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-efs-mounttarget.html>`_ resource, you must also specify a ``DependsOn`` attribute to ensure that the mount target is created or updated before the function.

        For more information about using the ``DependsOn`` attribute, see `DependsOn Attribute <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-attribute-dependson.html>`_ .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-function.html#cfn-lambda-function-filesystemconfigs
        '''
        result = self._values.get("file_system_configs")
        return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.List[typing.Union[_aws_cdk_core_f4b25747.IResolvable, CfnFunction.FileSystemConfigProperty]]]], result)

    @builtins.property
    def function_name(self) -> typing.Optional[builtins.str]:
        '''The name of the Lambda function, up to 64 characters in length.

        If you don't specify a name, AWS CloudFormation generates one.

        If you specify a name, you cannot perform updates that require replacement of this resource. You can perform updates that require no or some interruption. If you must replace the resource, specify a new name.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-function.html#cfn-lambda-function-functionname
        '''
        result = self._values.get("function_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def handler(self) -> typing.Optional[builtins.str]:
        '''The name of the method within your code that Lambda calls to run your function.

        Handler is required if the deployment package is a .zip file archive. The format includes the file name. It can also include namespaces and other qualifiers, depending on the runtime. For more information, see `Lambda programming model <https://docs.aws.amazon.com/lambda/latest/dg/foundation-progmodel.html>`_ .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-function.html#cfn-lambda-function-handler
        '''
        result = self._values.get("handler")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def image_config(
        self,
    ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, CfnFunction.ImageConfigProperty]]:
        '''Configuration values that override the container image Dockerfile settings.

        For more information, see `Container image settings <https://docs.aws.amazon.com/lambda/latest/dg/images-create.html#images-parms>`_ .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-function.html#cfn-lambda-function-imageconfig
        '''
        result = self._values.get("image_config")
        return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, CfnFunction.ImageConfigProperty]], result)

    @builtins.property
    def kms_key_arn(self) -> typing.Optional[builtins.str]:
        '''The ARN of the AWS Key Management Service ( AWS KMS ) key that's used to encrypt your function's environment variables.

        If it's not provided, Lambda uses a default service key.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-function.html#cfn-lambda-function-kmskeyarn
        '''
        result = self._values.get("kms_key_arn")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def layers(self) -> typing.Optional[typing.List[builtins.str]]:
        '''A list of `function layers <https://docs.aws.amazon.com/lambda/latest/dg/configuration-layers.html>`_ to add to the function's execution environment. Specify each layer by its ARN, including the version.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-function.html#cfn-lambda-function-layers
        '''
        result = self._values.get("layers")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def memory_size(self) -> typing.Optional[jsii.Number]:
        '''The amount of `memory available to the function <https://docs.aws.amazon.com/lambda/latest/dg/configuration-function-common.html#configuration-memory-console>`_ at runtime. Increasing the function memory also increases its CPU allocation. The default value is 128 MB. The value can be any multiple of 1 MB.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-function.html#cfn-lambda-function-memorysize
        '''
        result = self._values.get("memory_size")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def package_type(self) -> typing.Optional[builtins.str]:
        '''The type of deployment package.

        Set to ``Image`` for container image and set ``Zip`` for .zip file archive.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-function.html#cfn-lambda-function-packagetype
        '''
        result = self._values.get("package_type")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def reserved_concurrent_executions(self) -> typing.Optional[jsii.Number]:
        '''The number of simultaneous executions to reserve for the function.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-function.html#cfn-lambda-function-reservedconcurrentexecutions
        '''
        result = self._values.get("reserved_concurrent_executions")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def runtime(self) -> typing.Optional[builtins.str]:
        '''The identifier of the function's `runtime <https://docs.aws.amazon.com/lambda/latest/dg/lambda-runtimes.html>`_ . Runtime is required if the deployment package is a .zip file archive.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-function.html#cfn-lambda-function-runtime
        '''
        result = self._values.get("runtime")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def snap_start(
        self,
    ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, CfnFunction.SnapStartProperty]]:
        '''The function's `AWS Lambda SnapStart <https://docs.aws.amazon.com/lambda/latest/dg/snapstart.html>`_ setting.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-function.html#cfn-lambda-function-snapstart
        '''
        result = self._values.get("snap_start")
        return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, CfnFunction.SnapStartProperty]], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[_aws_cdk_core_f4b25747.CfnTag]]:
        '''A list of `tags <https://docs.aws.amazon.com/lambda/latest/dg/tagging.html>`_ to apply to the function.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-function.html#cfn-lambda-function-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[_aws_cdk_core_f4b25747.CfnTag]], result)

    @builtins.property
    def timeout(self) -> typing.Optional[jsii.Number]:
        '''The amount of time (in seconds) that Lambda allows a function to run before stopping it.

        The default is 3 seconds. The maximum allowed value is 900 seconds. For more information, see `Lambda execution environment <https://docs.aws.amazon.com/lambda/latest/dg/runtimes-context.html>`_ .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-function.html#cfn-lambda-function-timeout
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def tracing_config(
        self,
    ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, CfnFunction.TracingConfigProperty]]:
        '''Set ``Mode`` to ``Active`` to sample and trace a subset of incoming requests with `X-Ray <https://docs.aws.amazon.com/lambda/latest/dg/services-xray.html>`_ .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-function.html#cfn-lambda-function-tracingconfig
        '''
        result = self._values.get("tracing_config")
        return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, CfnFunction.TracingConfigProperty]], result)

    @builtins.property
    def vpc_config(
        self,
    ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, CfnFunction.VpcConfigProperty]]:
        '''For network connectivity to AWS resources in a `VPC <https://docs.aws.amazon.com/lambda/latest/dg/configuration-network.html>`_ , specify a list of security groups and subnets in the VPC.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-function.html#cfn-lambda-function-vpcconfig
        '''
        result = self._values.get("vpc_config")
        return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, CfnFunction.VpcConfigProperty]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnFunctionProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_aws_cdk_core_f4b25747.IInspectable)
class CfnLayerVersion(
    _aws_cdk_core_f4b25747.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-lambda.CfnLayerVersion",
):
    '''A CloudFormation ``AWS::Lambda::LayerVersion``.

    The ``AWS::Lambda::LayerVersion`` resource creates a `Lambda layer <https://docs.aws.amazon.com/lambda/latest/dg/configuration-layers.html>`_ from a ZIP archive.

    :cloudformationResource: AWS::Lambda::LayerVersion
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-layerversion.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        import aws_cdk.aws_lambda as lambda_
        
        cfn_layer_version = lambda_.CfnLayerVersion(self, "MyCfnLayerVersion",
            content=lambda.CfnLayerVersion.ContentProperty(
                s3_bucket="s3Bucket",
                s3_key="s3Key",
        
                # the properties below are optional
                s3_object_version="s3ObjectVersion"
            ),
        
            # the properties below are optional
            compatible_architectures=["compatibleArchitectures"],
            compatible_runtimes=["compatibleRuntimes"],
            description="description",
            layer_name="layerName",
            license_info="licenseInfo"
        )
    '''

    def __init__(
        self,
        scope: _aws_cdk_core_f4b25747.Construct,
        id: builtins.str,
        *,
        content: typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnLayerVersion.ContentProperty", typing.Dict[builtins.str, typing.Any]]],
        compatible_architectures: typing.Optional[typing.Sequence[builtins.str]] = None,
        compatible_runtimes: typing.Optional[typing.Sequence[builtins.str]] = None,
        description: typing.Optional[builtins.str] = None,
        layer_name: typing.Optional[builtins.str] = None,
        license_info: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Create a new ``AWS::Lambda::LayerVersion``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param content: The function layer archive.
        :param compatible_architectures: A list of compatible `instruction set architectures <https://docs.aws.amazon.com/lambda/latest/dg/foundation-arch.html>`_ .
        :param compatible_runtimes: A list of compatible `function runtimes <https://docs.aws.amazon.com/lambda/latest/dg/lambda-runtimes.html>`_ . Used for filtering with `ListLayers <https://docs.aws.amazon.com/lambda/latest/dg/API_ListLayers.html>`_ and `ListLayerVersions <https://docs.aws.amazon.com/lambda/latest/dg/API_ListLayerVersions.html>`_ .
        :param description: The description of the version.
        :param layer_name: The name or Amazon Resource Name (ARN) of the layer.
        :param license_info: The layer's software license. It can be any of the following:. - An `SPDX license identifier <https://docs.aws.amazon.com/https://spdx.org/licenses/>`_ . For example, ``MIT`` . - The URL of a license hosted on the internet. For example, ``https://opensource.org/licenses/MIT`` . - The full text of the license.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d57472a88ea7967a726756e87318c5527a030c48e644cbe8591ef7677268094c)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnLayerVersionProps(
            content=content,
            compatible_architectures=compatible_architectures,
            compatible_runtimes=compatible_runtimes,
            description=description,
            layer_name=layer_name,
            license_info=license_info,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _aws_cdk_core_f4b25747.TreeInspector) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9143627a3942eac0a7d60f56720b13371630306163d0cee9abd7b8b648ecd2a3)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__95b563f04a43d07f967ef0f790740ca3680b81a5a4ecb6385d72d923d0b0008d)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="content")
    def content(
        self,
    ) -> typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnLayerVersion.ContentProperty"]:
        '''The function layer archive.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-layerversion.html#cfn-lambda-layerversion-content
        '''
        return typing.cast(typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnLayerVersion.ContentProperty"], jsii.get(self, "content"))

    @content.setter
    def content(
        self,
        value: typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnLayerVersion.ContentProperty"],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a3e91c66425ae3099008d814bb920f6967b9a52294f9f899aa1f53140a55340e)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "content", value)

    @builtins.property
    @jsii.member(jsii_name="compatibleArchitectures")
    def compatible_architectures(self) -> typing.Optional[typing.List[builtins.str]]:
        '''A list of compatible `instruction set architectures <https://docs.aws.amazon.com/lambda/latest/dg/foundation-arch.html>`_ .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-layerversion.html#cfn-lambda-layerversion-compatiblearchitectures
        '''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "compatibleArchitectures"))

    @compatible_architectures.setter
    def compatible_architectures(
        self,
        value: typing.Optional[typing.List[builtins.str]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c522f5b85c85529c9e62dea6b9daf25f2cca4a6a3e1e4ee20b76167411ff53a5)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "compatibleArchitectures", value)

    @builtins.property
    @jsii.member(jsii_name="compatibleRuntimes")
    def compatible_runtimes(self) -> typing.Optional[typing.List[builtins.str]]:
        '''A list of compatible `function runtimes <https://docs.aws.amazon.com/lambda/latest/dg/lambda-runtimes.html>`_ . Used for filtering with `ListLayers <https://docs.aws.amazon.com/lambda/latest/dg/API_ListLayers.html>`_ and `ListLayerVersions <https://docs.aws.amazon.com/lambda/latest/dg/API_ListLayerVersions.html>`_ .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-layerversion.html#cfn-lambda-layerversion-compatibleruntimes
        '''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "compatibleRuntimes"))

    @compatible_runtimes.setter
    def compatible_runtimes(
        self,
        value: typing.Optional[typing.List[builtins.str]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8456fb176cc7af0b37aefc06f35a4f06e5cbc1edc82181be8142dad2c11cc492)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "compatibleRuntimes", value)

    @builtins.property
    @jsii.member(jsii_name="description")
    def description(self) -> typing.Optional[builtins.str]:
        '''The description of the version.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-layerversion.html#cfn-lambda-layerversion-description
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "description"))

    @description.setter
    def description(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__215f20c823b3b123197a92195eef8cbab1e1b09741cae5bd3bb35c4ce2e5c46e)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "description", value)

    @builtins.property
    @jsii.member(jsii_name="layerName")
    def layer_name(self) -> typing.Optional[builtins.str]:
        '''The name or Amazon Resource Name (ARN) of the layer.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-layerversion.html#cfn-lambda-layerversion-layername
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "layerName"))

    @layer_name.setter
    def layer_name(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d77eb70c4f81c32fcf4d1ed69e1ace57cbe94629a7d2381c97c40583eb1dcc43)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "layerName", value)

    @builtins.property
    @jsii.member(jsii_name="licenseInfo")
    def license_info(self) -> typing.Optional[builtins.str]:
        '''The layer's software license. It can be any of the following:.

        - An `SPDX license identifier <https://docs.aws.amazon.com/https://spdx.org/licenses/>`_ . For example, ``MIT`` .
        - The URL of a license hosted on the internet. For example, ``https://opensource.org/licenses/MIT`` .
        - The full text of the license.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-layerversion.html#cfn-lambda-layerversion-licenseinfo
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "licenseInfo"))

    @license_info.setter
    def license_info(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__58e9f77657a3f1838af45b411a01820a8e0201ac0c6c0ce52f76c85214d9055a)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "licenseInfo", value)

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-lambda.CfnLayerVersion.ContentProperty",
        jsii_struct_bases=[],
        name_mapping={
            "s3_bucket": "s3Bucket",
            "s3_key": "s3Key",
            "s3_object_version": "s3ObjectVersion",
        },
    )
    class ContentProperty:
        def __init__(
            self,
            *,
            s3_bucket: builtins.str,
            s3_key: builtins.str,
            s3_object_version: typing.Optional[builtins.str] = None,
        ) -> None:
            '''A ZIP archive that contains the contents of an `Lambda layer <https://docs.aws.amazon.com/lambda/latest/dg/configuration-layers.html>`_ .

            :param s3_bucket: The Amazon S3 bucket of the layer archive.
            :param s3_key: The Amazon S3 key of the layer archive.
            :param s3_object_version: For versioned objects, the version of the layer archive object to use.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-layerversion-content.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_lambda as lambda_
                
                content_property = lambda.CfnLayerVersion.ContentProperty(
                    s3_bucket="s3Bucket",
                    s3_key="s3Key",
                
                    # the properties below are optional
                    s3_object_version="s3ObjectVersion"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__07d7e08ad8c5514ca4beccb6d8a56c1e98d9cda7050cadabfac276302f047f88)
                check_type(argname="argument s3_bucket", value=s3_bucket, expected_type=type_hints["s3_bucket"])
                check_type(argname="argument s3_key", value=s3_key, expected_type=type_hints["s3_key"])
                check_type(argname="argument s3_object_version", value=s3_object_version, expected_type=type_hints["s3_object_version"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "s3_bucket": s3_bucket,
                "s3_key": s3_key,
            }
            if s3_object_version is not None:
                self._values["s3_object_version"] = s3_object_version

        @builtins.property
        def s3_bucket(self) -> builtins.str:
            '''The Amazon S3 bucket of the layer archive.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-layerversion-content.html#cfn-lambda-layerversion-content-s3bucket
            '''
            result = self._values.get("s3_bucket")
            assert result is not None, "Required property 's3_bucket' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def s3_key(self) -> builtins.str:
            '''The Amazon S3 key of the layer archive.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-layerversion-content.html#cfn-lambda-layerversion-content-s3key
            '''
            result = self._values.get("s3_key")
            assert result is not None, "Required property 's3_key' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def s3_object_version(self) -> typing.Optional[builtins.str]:
            '''For versioned objects, the version of the layer archive object to use.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-layerversion-content.html#cfn-lambda-layerversion-content-s3objectversion
            '''
            result = self._values.get("s3_object_version")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ContentProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.implements(_aws_cdk_core_f4b25747.IInspectable)
class CfnLayerVersionPermission(
    _aws_cdk_core_f4b25747.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-lambda.CfnLayerVersionPermission",
):
    '''A CloudFormation ``AWS::Lambda::LayerVersionPermission``.

    The ``AWS::Lambda::LayerVersionPermission`` resource adds permissions to the resource-based policy of a version of an `Lambda layer <https://docs.aws.amazon.com/lambda/latest/dg/configuration-layers.html>`_ . Use this action to grant layer usage permission to other accounts. You can grant permission to a single account, all AWS accounts, or all accounts in an organization.
    .. epigraph::

       Since the release of the `UpdateReplacePolicy <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-attribute-updatereplacepolicy.html>`_ both ``UpdateReplacePolicy`` and ``DeletionPolicy`` are required to protect your Resources/LayerPermissions from deletion.

    :cloudformationResource: AWS::Lambda::LayerVersionPermission
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-layerversionpermission.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        import aws_cdk.aws_lambda as lambda_
        
        cfn_layer_version_permission = lambda_.CfnLayerVersionPermission(self, "MyCfnLayerVersionPermission",
            action="action",
            layer_version_arn="layerVersionArn",
            principal="principal",
        
            # the properties below are optional
            organization_id="organizationId"
        )
    '''

    def __init__(
        self,
        scope: _aws_cdk_core_f4b25747.Construct,
        id: builtins.str,
        *,
        action: builtins.str,
        layer_version_arn: builtins.str,
        principal: builtins.str,
        organization_id: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Create a new ``AWS::Lambda::LayerVersionPermission``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param action: The API action that grants access to the layer. For example, ``lambda:GetLayerVersion`` .
        :param layer_version_arn: The name or Amazon Resource Name (ARN) of the layer.
        :param principal: An account ID, or ``*`` to grant layer usage permission to all accounts in an organization, or all AWS accounts (if ``organizationId`` is not specified). For the last case, make sure that you really do want all AWS accounts to have usage permission to this layer.
        :param organization_id: With the principal set to ``*`` , grant permission to all accounts in the specified organization.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__73b6eda709b7e0821006db18cd90def476b7a294d30a97436a45b0603eca7838)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnLayerVersionPermissionProps(
            action=action,
            layer_version_arn=layer_version_arn,
            principal=principal,
            organization_id=organization_id,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _aws_cdk_core_f4b25747.TreeInspector) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b2d3745fa7340092314bd5b4f6362b396eeccd94c2492f49c43db9577b02c6db)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b0675d9057473b63d262899cecb2b8a5dc92a355d1bbab73b909daea2cad0eba)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="action")
    def action(self) -> builtins.str:
        '''The API action that grants access to the layer.

        For example, ``lambda:GetLayerVersion`` .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-layerversionpermission.html#cfn-lambda-layerversionpermission-action
        '''
        return typing.cast(builtins.str, jsii.get(self, "action"))

    @action.setter
    def action(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a2fdcb553cdea6724bdc7999522c9a110aafdf0b650fd6e66ea2a279626f398d)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "action", value)

    @builtins.property
    @jsii.member(jsii_name="layerVersionArn")
    def layer_version_arn(self) -> builtins.str:
        '''The name or Amazon Resource Name (ARN) of the layer.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-layerversionpermission.html#cfn-lambda-layerversionpermission-layerversionarn
        '''
        return typing.cast(builtins.str, jsii.get(self, "layerVersionArn"))

    @layer_version_arn.setter
    def layer_version_arn(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c9a4453fb08b728d936a9ab944ba82ce7813ae01c6e03db28568509d70f1bbdb)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "layerVersionArn", value)

    @builtins.property
    @jsii.member(jsii_name="principal")
    def principal(self) -> builtins.str:
        '''An account ID, or ``*`` to grant layer usage permission to all accounts in an organization, or all AWS accounts (if ``organizationId`` is not specified).

        For the last case, make sure that you really do want all AWS accounts to have usage permission to this layer.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-layerversionpermission.html#cfn-lambda-layerversionpermission-principal
        '''
        return typing.cast(builtins.str, jsii.get(self, "principal"))

    @principal.setter
    def principal(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__7de0f701478afcc5ea97055fb072feb45358d056b1c07d2a34b082e5175f5117)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "principal", value)

    @builtins.property
    @jsii.member(jsii_name="organizationId")
    def organization_id(self) -> typing.Optional[builtins.str]:
        '''With the principal set to ``*`` , grant permission to all accounts in the specified organization.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-layerversionpermission.html#cfn-lambda-layerversionpermission-organizationid
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "organizationId"))

    @organization_id.setter
    def organization_id(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a43aa2d97b44b777d0cbcd1cc89970e8c7fedae6d6adfa936eec6e81b84dc517)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "organizationId", value)


@jsii.data_type(
    jsii_type="@aws-cdk/aws-lambda.CfnLayerVersionPermissionProps",
    jsii_struct_bases=[],
    name_mapping={
        "action": "action",
        "layer_version_arn": "layerVersionArn",
        "principal": "principal",
        "organization_id": "organizationId",
    },
)
class CfnLayerVersionPermissionProps:
    def __init__(
        self,
        *,
        action: builtins.str,
        layer_version_arn: builtins.str,
        principal: builtins.str,
        organization_id: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for defining a ``CfnLayerVersionPermission``.

        :param action: The API action that grants access to the layer. For example, ``lambda:GetLayerVersion`` .
        :param layer_version_arn: The name or Amazon Resource Name (ARN) of the layer.
        :param principal: An account ID, or ``*`` to grant layer usage permission to all accounts in an organization, or all AWS accounts (if ``organizationId`` is not specified). For the last case, make sure that you really do want all AWS accounts to have usage permission to this layer.
        :param organization_id: With the principal set to ``*`` , grant permission to all accounts in the specified organization.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-layerversionpermission.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_lambda as lambda_
            
            cfn_layer_version_permission_props = lambda.CfnLayerVersionPermissionProps(
                action="action",
                layer_version_arn="layerVersionArn",
                principal="principal",
            
                # the properties below are optional
                organization_id="organizationId"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__6f07ce4e31ac6486f7bfcd5f9a26ab7ccc3eea5d975d46a802552860445404b4)
            check_type(argname="argument action", value=action, expected_type=type_hints["action"])
            check_type(argname="argument layer_version_arn", value=layer_version_arn, expected_type=type_hints["layer_version_arn"])
            check_type(argname="argument principal", value=principal, expected_type=type_hints["principal"])
            check_type(argname="argument organization_id", value=organization_id, expected_type=type_hints["organization_id"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "action": action,
            "layer_version_arn": layer_version_arn,
            "principal": principal,
        }
        if organization_id is not None:
            self._values["organization_id"] = organization_id

    @builtins.property
    def action(self) -> builtins.str:
        '''The API action that grants access to the layer.

        For example, ``lambda:GetLayerVersion`` .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-layerversionpermission.html#cfn-lambda-layerversionpermission-action
        '''
        result = self._values.get("action")
        assert result is not None, "Required property 'action' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def layer_version_arn(self) -> builtins.str:
        '''The name or Amazon Resource Name (ARN) of the layer.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-layerversionpermission.html#cfn-lambda-layerversionpermission-layerversionarn
        '''
        result = self._values.get("layer_version_arn")
        assert result is not None, "Required property 'layer_version_arn' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def principal(self) -> builtins.str:
        '''An account ID, or ``*`` to grant layer usage permission to all accounts in an organization, or all AWS accounts (if ``organizationId`` is not specified).

        For the last case, make sure that you really do want all AWS accounts to have usage permission to this layer.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-layerversionpermission.html#cfn-lambda-layerversionpermission-principal
        '''
        result = self._values.get("principal")
        assert result is not None, "Required property 'principal' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def organization_id(self) -> typing.Optional[builtins.str]:
        '''With the principal set to ``*`` , grant permission to all accounts in the specified organization.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-layerversionpermission.html#cfn-lambda-layerversionpermission-organizationid
        '''
        result = self._values.get("organization_id")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnLayerVersionPermissionProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-lambda.CfnLayerVersionProps",
    jsii_struct_bases=[],
    name_mapping={
        "content": "content",
        "compatible_architectures": "compatibleArchitectures",
        "compatible_runtimes": "compatibleRuntimes",
        "description": "description",
        "layer_name": "layerName",
        "license_info": "licenseInfo",
    },
)
class CfnLayerVersionProps:
    def __init__(
        self,
        *,
        content: typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union[CfnLayerVersion.ContentProperty, typing.Dict[builtins.str, typing.Any]]],
        compatible_architectures: typing.Optional[typing.Sequence[builtins.str]] = None,
        compatible_runtimes: typing.Optional[typing.Sequence[builtins.str]] = None,
        description: typing.Optional[builtins.str] = None,
        layer_name: typing.Optional[builtins.str] = None,
        license_info: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for defining a ``CfnLayerVersion``.

        :param content: The function layer archive.
        :param compatible_architectures: A list of compatible `instruction set architectures <https://docs.aws.amazon.com/lambda/latest/dg/foundation-arch.html>`_ .
        :param compatible_runtimes: A list of compatible `function runtimes <https://docs.aws.amazon.com/lambda/latest/dg/lambda-runtimes.html>`_ . Used for filtering with `ListLayers <https://docs.aws.amazon.com/lambda/latest/dg/API_ListLayers.html>`_ and `ListLayerVersions <https://docs.aws.amazon.com/lambda/latest/dg/API_ListLayerVersions.html>`_ .
        :param description: The description of the version.
        :param layer_name: The name or Amazon Resource Name (ARN) of the layer.
        :param license_info: The layer's software license. It can be any of the following:. - An `SPDX license identifier <https://docs.aws.amazon.com/https://spdx.org/licenses/>`_ . For example, ``MIT`` . - The URL of a license hosted on the internet. For example, ``https://opensource.org/licenses/MIT`` . - The full text of the license.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-layerversion.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_lambda as lambda_
            
            cfn_layer_version_props = lambda.CfnLayerVersionProps(
                content=lambda.CfnLayerVersion.ContentProperty(
                    s3_bucket="s3Bucket",
                    s3_key="s3Key",
            
                    # the properties below are optional
                    s3_object_version="s3ObjectVersion"
                ),
            
                # the properties below are optional
                compatible_architectures=["compatibleArchitectures"],
                compatible_runtimes=["compatibleRuntimes"],
                description="description",
                layer_name="layerName",
                license_info="licenseInfo"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__bf791a2ff602bc823e0f52226e60cb6a761bf193cb8f0582778f73b0d5767940)
            check_type(argname="argument content", value=content, expected_type=type_hints["content"])
            check_type(argname="argument compatible_architectures", value=compatible_architectures, expected_type=type_hints["compatible_architectures"])
            check_type(argname="argument compatible_runtimes", value=compatible_runtimes, expected_type=type_hints["compatible_runtimes"])
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument layer_name", value=layer_name, expected_type=type_hints["layer_name"])
            check_type(argname="argument license_info", value=license_info, expected_type=type_hints["license_info"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "content": content,
        }
        if compatible_architectures is not None:
            self._values["compatible_architectures"] = compatible_architectures
        if compatible_runtimes is not None:
            self._values["compatible_runtimes"] = compatible_runtimes
        if description is not None:
            self._values["description"] = description
        if layer_name is not None:
            self._values["layer_name"] = layer_name
        if license_info is not None:
            self._values["license_info"] = license_info

    @builtins.property
    def content(
        self,
    ) -> typing.Union[_aws_cdk_core_f4b25747.IResolvable, CfnLayerVersion.ContentProperty]:
        '''The function layer archive.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-layerversion.html#cfn-lambda-layerversion-content
        '''
        result = self._values.get("content")
        assert result is not None, "Required property 'content' is missing"
        return typing.cast(typing.Union[_aws_cdk_core_f4b25747.IResolvable, CfnLayerVersion.ContentProperty], result)

    @builtins.property
    def compatible_architectures(self) -> typing.Optional[typing.List[builtins.str]]:
        '''A list of compatible `instruction set architectures <https://docs.aws.amazon.com/lambda/latest/dg/foundation-arch.html>`_ .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-layerversion.html#cfn-lambda-layerversion-compatiblearchitectures
        '''
        result = self._values.get("compatible_architectures")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def compatible_runtimes(self) -> typing.Optional[typing.List[builtins.str]]:
        '''A list of compatible `function runtimes <https://docs.aws.amazon.com/lambda/latest/dg/lambda-runtimes.html>`_ . Used for filtering with `ListLayers <https://docs.aws.amazon.com/lambda/latest/dg/API_ListLayers.html>`_ and `ListLayerVersions <https://docs.aws.amazon.com/lambda/latest/dg/API_ListLayerVersions.html>`_ .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-layerversion.html#cfn-lambda-layerversion-compatibleruntimes
        '''
        result = self._values.get("compatible_runtimes")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''The description of the version.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-layerversion.html#cfn-lambda-layerversion-description
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def layer_name(self) -> typing.Optional[builtins.str]:
        '''The name or Amazon Resource Name (ARN) of the layer.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-layerversion.html#cfn-lambda-layerversion-layername
        '''
        result = self._values.get("layer_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def license_info(self) -> typing.Optional[builtins.str]:
        '''The layer's software license. It can be any of the following:.

        - An `SPDX license identifier <https://docs.aws.amazon.com/https://spdx.org/licenses/>`_ . For example, ``MIT`` .
        - The URL of a license hosted on the internet. For example, ``https://opensource.org/licenses/MIT`` .
        - The full text of the license.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-layerversion.html#cfn-lambda-layerversion-licenseinfo
        '''
        result = self._values.get("license_info")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnLayerVersionProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-lambda.CfnParametersCodeProps",
    jsii_struct_bases=[],
    name_mapping={
        "bucket_name_param": "bucketNameParam",
        "object_key_param": "objectKeyParam",
    },
)
class CfnParametersCodeProps:
    def __init__(
        self,
        *,
        bucket_name_param: typing.Optional[_aws_cdk_core_f4b25747.CfnParameter] = None,
        object_key_param: typing.Optional[_aws_cdk_core_f4b25747.CfnParameter] = None,
    ) -> None:
        '''Construction properties for {@link CfnParametersCode}.

        :param bucket_name_param: The CloudFormation parameter that represents the name of the S3 Bucket where the Lambda code will be located in. Must be of type 'String'. Default: a new parameter will be created
        :param object_key_param: The CloudFormation parameter that represents the path inside the S3 Bucket where the Lambda code will be located at. Must be of type 'String'. Default: a new parameter will be created

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_lambda as lambda_
            import aws_cdk.core as cdk
            
            # cfn_parameter: cdk.CfnParameter
            
            cfn_parameters_code_props = lambda.CfnParametersCodeProps(
                bucket_name_param=cfn_parameter,
                object_key_param=cfn_parameter
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__617ecaad98ccabcef1b33490b7a146e3e839c89368a9caa8ae1be43098d5ac5e)
            check_type(argname="argument bucket_name_param", value=bucket_name_param, expected_type=type_hints["bucket_name_param"])
            check_type(argname="argument object_key_param", value=object_key_param, expected_type=type_hints["object_key_param"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if bucket_name_param is not None:
            self._values["bucket_name_param"] = bucket_name_param
        if object_key_param is not None:
            self._values["object_key_param"] = object_key_param

    @builtins.property
    def bucket_name_param(self) -> typing.Optional[_aws_cdk_core_f4b25747.CfnParameter]:
        '''The CloudFormation parameter that represents the name of the S3 Bucket where the Lambda code will be located in.

        Must be of type 'String'.

        :default: a new parameter will be created
        '''
        result = self._values.get("bucket_name_param")
        return typing.cast(typing.Optional[_aws_cdk_core_f4b25747.CfnParameter], result)

    @builtins.property
    def object_key_param(self) -> typing.Optional[_aws_cdk_core_f4b25747.CfnParameter]:
        '''The CloudFormation parameter that represents the path inside the S3 Bucket where the Lambda code will be located at.

        Must be of type 'String'.

        :default: a new parameter will be created
        '''
        result = self._values.get("object_key_param")
        return typing.cast(typing.Optional[_aws_cdk_core_f4b25747.CfnParameter], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnParametersCodeProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_aws_cdk_core_f4b25747.IInspectable)
class CfnPermission(
    _aws_cdk_core_f4b25747.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-lambda.CfnPermission",
):
    '''A CloudFormation ``AWS::Lambda::Permission``.

    The ``AWS::Lambda::Permission`` resource grants an AWS service or another account permission to use a function. You can apply the policy at the function level, or specify a qualifier to restrict access to a single version or alias. If you use a qualifier, the invoker must use the full Amazon Resource Name (ARN) of that version or alias to invoke the function.

    To grant permission to another account, specify the account ID as the ``Principal`` . To grant permission to an organization defined in AWS Organizations , specify the organization ID as the ``PrincipalOrgID`` . For AWS services, the principal is a domain-style identifier defined by the service, like ``s3.amazonaws.com`` or ``sns.amazonaws.com`` . For AWS services, you can also specify the ARN of the associated resource as the ``SourceArn`` . If you grant permission to a service principal without specifying the source, other accounts could potentially configure resources in their account to invoke your Lambda function.

    If your function has a function URL, you can specify the ``FunctionUrlAuthType`` parameter. This adds a condition to your permission that only applies when your function URL's ``AuthType`` matches the specified ``FunctionUrlAuthType`` . For more information about the ``AuthType`` parameter, see `Security and auth model for Lambda function URLs <https://docs.aws.amazon.com/lambda/latest/dg/urls-auth.html>`_ .

    This resource adds a statement to a resource-based permission policy for the function. For more information about function policies, see `Lambda Function Policies <https://docs.aws.amazon.com/lambda/latest/dg/access-control-resource-based.html>`_ .

    :cloudformationResource: AWS::Lambda::Permission
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-permission.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        import aws_cdk.aws_lambda as lambda_
        
        cfn_permission = lambda_.CfnPermission(self, "MyCfnPermission",
            action="action",
            function_name="functionName",
            principal="principal",
        
            # the properties below are optional
            event_source_token="eventSourceToken",
            function_url_auth_type="functionUrlAuthType",
            principal_org_id="principalOrgId",
            source_account="sourceAccount",
            source_arn="sourceArn"
        )
    '''

    def __init__(
        self,
        scope: _aws_cdk_core_f4b25747.Construct,
        id: builtins.str,
        *,
        action: builtins.str,
        function_name: builtins.str,
        principal: builtins.str,
        event_source_token: typing.Optional[builtins.str] = None,
        function_url_auth_type: typing.Optional[builtins.str] = None,
        principal_org_id: typing.Optional[builtins.str] = None,
        source_account: typing.Optional[builtins.str] = None,
        source_arn: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Create a new ``AWS::Lambda::Permission``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param action: The action that the principal can use on the function. For example, ``lambda:InvokeFunction`` or ``lambda:GetFunction`` .
        :param function_name: The name of the Lambda function, version, or alias. **Name formats** - *Function name* – ``my-function`` (name-only), ``my-function:v1`` (with alias). - *Function ARN* – ``arn:aws:lambda:us-west-2:123456789012:function:my-function`` . - *Partial ARN* – ``123456789012:function:my-function`` . You can append a version number or alias to any of the formats. The length constraint applies only to the full ARN. If you specify only the function name, it is limited to 64 characters in length.
        :param principal: The AWS service or AWS account that invokes the function. If you specify a service, use ``SourceArn`` or ``SourceAccount`` to limit who can invoke the function through that service.
        :param event_source_token: For Alexa Smart Home functions, a token that the invoker must supply.
        :param function_url_auth_type: The type of authentication that your function URL uses. Set to ``AWS_IAM`` if you want to restrict access to authenticated IAM users only. Set to ``NONE`` if you want to bypass IAM authentication to create a public endpoint. For more information, see `Security and auth model for Lambda function URLs <https://docs.aws.amazon.com/lambda/latest/dg/urls-auth.html>`_ .
        :param principal_org_id: The identifier for your organization in AWS Organizations . Use this to grant permissions to all the AWS accounts under this organization.
        :param source_account: For AWS service , the ID of the AWS account that owns the resource. Use this together with ``SourceArn`` to ensure that the specified account owns the resource. It is possible for an Amazon S3 bucket to be deleted by its owner and recreated by another account.
        :param source_arn: For AWS services , the ARN of the AWS resource that invokes the function. For example, an Amazon S3 bucket or Amazon SNS topic. Note that Lambda configures the comparison using the ``StringLike`` operator.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e0b1f9d7cf87100e8857119ccb5bb23f79adc090c98e613da4546338f9feddbf)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnPermissionProps(
            action=action,
            function_name=function_name,
            principal=principal,
            event_source_token=event_source_token,
            function_url_auth_type=function_url_auth_type,
            principal_org_id=principal_org_id,
            source_account=source_account,
            source_arn=source_arn,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _aws_cdk_core_f4b25747.TreeInspector) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f40c5584c74e4176c495fd94b11162c256aad90ae158b608aedc6e108c262bd2)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0eb194de298dc48ee6813ff4d9314c103727fe4e4a3d1bb22cd37612f183e7b3)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="action")
    def action(self) -> builtins.str:
        '''The action that the principal can use on the function.

        For example, ``lambda:InvokeFunction`` or ``lambda:GetFunction`` .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-permission.html#cfn-lambda-permission-action
        '''
        return typing.cast(builtins.str, jsii.get(self, "action"))

    @action.setter
    def action(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b13759d9570323328e14086af6c1d27470c5f1f7e5e2c3acfa62d68bfd838583)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "action", value)

    @builtins.property
    @jsii.member(jsii_name="functionName")
    def function_name(self) -> builtins.str:
        '''The name of the Lambda function, version, or alias.

        **Name formats** - *Function name* – ``my-function`` (name-only), ``my-function:v1`` (with alias).

        - *Function ARN* – ``arn:aws:lambda:us-west-2:123456789012:function:my-function`` .
        - *Partial ARN* – ``123456789012:function:my-function`` .

        You can append a version number or alias to any of the formats. The length constraint applies only to the full ARN. If you specify only the function name, it is limited to 64 characters in length.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-permission.html#cfn-lambda-permission-functionname
        '''
        return typing.cast(builtins.str, jsii.get(self, "functionName"))

    @function_name.setter
    def function_name(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__aa2c1a6637f21b0bb00cb4aeb7357314ab83c7c341dd02760dcb3058f5683efb)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "functionName", value)

    @builtins.property
    @jsii.member(jsii_name="principal")
    def principal(self) -> builtins.str:
        '''The AWS service or AWS account that invokes the function.

        If you specify a service, use ``SourceArn`` or ``SourceAccount`` to limit who can invoke the function through that service.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-permission.html#cfn-lambda-permission-principal
        '''
        return typing.cast(builtins.str, jsii.get(self, "principal"))

    @principal.setter
    def principal(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__70865e5217101a9d2dd4a8905db858296491037f8a97f65c6a7dc9f03343d142)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "principal", value)

    @builtins.property
    @jsii.member(jsii_name="eventSourceToken")
    def event_source_token(self) -> typing.Optional[builtins.str]:
        '''For Alexa Smart Home functions, a token that the invoker must supply.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-permission.html#cfn-lambda-permission-eventsourcetoken
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "eventSourceToken"))

    @event_source_token.setter
    def event_source_token(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0c66101ec7ad650dd8a1ed82b3ee19fb78aef165059abeb7b6a6d950939e7279)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "eventSourceToken", value)

    @builtins.property
    @jsii.member(jsii_name="functionUrlAuthType")
    def function_url_auth_type(self) -> typing.Optional[builtins.str]:
        '''The type of authentication that your function URL uses.

        Set to ``AWS_IAM`` if you want to restrict access to authenticated IAM users only. Set to ``NONE`` if you want to bypass IAM authentication to create a public endpoint. For more information, see `Security and auth model for Lambda function URLs <https://docs.aws.amazon.com/lambda/latest/dg/urls-auth.html>`_ .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-permission.html#cfn-lambda-permission-functionurlauthtype
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "functionUrlAuthType"))

    @function_url_auth_type.setter
    def function_url_auth_type(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3af2c412bfec44a4d11a6840e689da120e00b2c81bbfd48cf33b779126299342)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "functionUrlAuthType", value)

    @builtins.property
    @jsii.member(jsii_name="principalOrgId")
    def principal_org_id(self) -> typing.Optional[builtins.str]:
        '''The identifier for your organization in AWS Organizations .

        Use this to grant permissions to all the AWS accounts under this organization.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-permission.html#cfn-lambda-permission-principalorgid
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "principalOrgId"))

    @principal_org_id.setter
    def principal_org_id(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__cd4a9b329b618cf6f2686eb05f869cf7a93a6c49f5d9a4a7cc55c3e4d4c6f2fb)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "principalOrgId", value)

    @builtins.property
    @jsii.member(jsii_name="sourceAccount")
    def source_account(self) -> typing.Optional[builtins.str]:
        '''For AWS service , the ID of the AWS account that owns the resource.

        Use this together with ``SourceArn`` to ensure that the specified account owns the resource. It is possible for an Amazon S3 bucket to be deleted by its owner and recreated by another account.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-permission.html#cfn-lambda-permission-sourceaccount
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "sourceAccount"))

    @source_account.setter
    def source_account(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f5ff6bbdf6e915094501b28b47440c0f493353ea64f1dedf0873c19dab44979b)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "sourceAccount", value)

    @builtins.property
    @jsii.member(jsii_name="sourceArn")
    def source_arn(self) -> typing.Optional[builtins.str]:
        '''For AWS services , the ARN of the AWS resource that invokes the function.

        For example, an Amazon S3 bucket or Amazon SNS topic.

        Note that Lambda configures the comparison using the ``StringLike`` operator.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-permission.html#cfn-lambda-permission-sourcearn
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "sourceArn"))

    @source_arn.setter
    def source_arn(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__14b033a67fe953325a3070e595a49338a7bfe4258e5a9bea2b237c8d11a7f589)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "sourceArn", value)


@jsii.data_type(
    jsii_type="@aws-cdk/aws-lambda.CfnPermissionProps",
    jsii_struct_bases=[],
    name_mapping={
        "action": "action",
        "function_name": "functionName",
        "principal": "principal",
        "event_source_token": "eventSourceToken",
        "function_url_auth_type": "functionUrlAuthType",
        "principal_org_id": "principalOrgId",
        "source_account": "sourceAccount",
        "source_arn": "sourceArn",
    },
)
class CfnPermissionProps:
    def __init__(
        self,
        *,
        action: builtins.str,
        function_name: builtins.str,
        principal: builtins.str,
        event_source_token: typing.Optional[builtins.str] = None,
        function_url_auth_type: typing.Optional[builtins.str] = None,
        principal_org_id: typing.Optional[builtins.str] = None,
        source_account: typing.Optional[builtins.str] = None,
        source_arn: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for defining a ``CfnPermission``.

        :param action: The action that the principal can use on the function. For example, ``lambda:InvokeFunction`` or ``lambda:GetFunction`` .
        :param function_name: The name of the Lambda function, version, or alias. **Name formats** - *Function name* – ``my-function`` (name-only), ``my-function:v1`` (with alias). - *Function ARN* – ``arn:aws:lambda:us-west-2:123456789012:function:my-function`` . - *Partial ARN* – ``123456789012:function:my-function`` . You can append a version number or alias to any of the formats. The length constraint applies only to the full ARN. If you specify only the function name, it is limited to 64 characters in length.
        :param principal: The AWS service or AWS account that invokes the function. If you specify a service, use ``SourceArn`` or ``SourceAccount`` to limit who can invoke the function through that service.
        :param event_source_token: For Alexa Smart Home functions, a token that the invoker must supply.
        :param function_url_auth_type: The type of authentication that your function URL uses. Set to ``AWS_IAM`` if you want to restrict access to authenticated IAM users only. Set to ``NONE`` if you want to bypass IAM authentication to create a public endpoint. For more information, see `Security and auth model for Lambda function URLs <https://docs.aws.amazon.com/lambda/latest/dg/urls-auth.html>`_ .
        :param principal_org_id: The identifier for your organization in AWS Organizations . Use this to grant permissions to all the AWS accounts under this organization.
        :param source_account: For AWS service , the ID of the AWS account that owns the resource. Use this together with ``SourceArn`` to ensure that the specified account owns the resource. It is possible for an Amazon S3 bucket to be deleted by its owner and recreated by another account.
        :param source_arn: For AWS services , the ARN of the AWS resource that invokes the function. For example, an Amazon S3 bucket or Amazon SNS topic. Note that Lambda configures the comparison using the ``StringLike`` operator.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-permission.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_lambda as lambda_
            
            cfn_permission_props = lambda.CfnPermissionProps(
                action="action",
                function_name="functionName",
                principal="principal",
            
                # the properties below are optional
                event_source_token="eventSourceToken",
                function_url_auth_type="functionUrlAuthType",
                principal_org_id="principalOrgId",
                source_account="sourceAccount",
                source_arn="sourceArn"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__cde021fbeb4847c414d5abf15de40f052ee8480b4a224effc5634e139b2c474b)
            check_type(argname="argument action", value=action, expected_type=type_hints["action"])
            check_type(argname="argument function_name", value=function_name, expected_type=type_hints["function_name"])
            check_type(argname="argument principal", value=principal, expected_type=type_hints["principal"])
            check_type(argname="argument event_source_token", value=event_source_token, expected_type=type_hints["event_source_token"])
            check_type(argname="argument function_url_auth_type", value=function_url_auth_type, expected_type=type_hints["function_url_auth_type"])
            check_type(argname="argument principal_org_id", value=principal_org_id, expected_type=type_hints["principal_org_id"])
            check_type(argname="argument source_account", value=source_account, expected_type=type_hints["source_account"])
            check_type(argname="argument source_arn", value=source_arn, expected_type=type_hints["source_arn"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "action": action,
            "function_name": function_name,
            "principal": principal,
        }
        if event_source_token is not None:
            self._values["event_source_token"] = event_source_token
        if function_url_auth_type is not None:
            self._values["function_url_auth_type"] = function_url_auth_type
        if principal_org_id is not None:
            self._values["principal_org_id"] = principal_org_id
        if source_account is not None:
            self._values["source_account"] = source_account
        if source_arn is not None:
            self._values["source_arn"] = source_arn

    @builtins.property
    def action(self) -> builtins.str:
        '''The action that the principal can use on the function.

        For example, ``lambda:InvokeFunction`` or ``lambda:GetFunction`` .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-permission.html#cfn-lambda-permission-action
        '''
        result = self._values.get("action")
        assert result is not None, "Required property 'action' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def function_name(self) -> builtins.str:
        '''The name of the Lambda function, version, or alias.

        **Name formats** - *Function name* – ``my-function`` (name-only), ``my-function:v1`` (with alias).

        - *Function ARN* – ``arn:aws:lambda:us-west-2:123456789012:function:my-function`` .
        - *Partial ARN* – ``123456789012:function:my-function`` .

        You can append a version number or alias to any of the formats. The length constraint applies only to the full ARN. If you specify only the function name, it is limited to 64 characters in length.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-permission.html#cfn-lambda-permission-functionname
        '''
        result = self._values.get("function_name")
        assert result is not None, "Required property 'function_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def principal(self) -> builtins.str:
        '''The AWS service or AWS account that invokes the function.

        If you specify a service, use ``SourceArn`` or ``SourceAccount`` to limit who can invoke the function through that service.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-permission.html#cfn-lambda-permission-principal
        '''
        result = self._values.get("principal")
        assert result is not None, "Required property 'principal' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def event_source_token(self) -> typing.Optional[builtins.str]:
        '''For Alexa Smart Home functions, a token that the invoker must supply.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-permission.html#cfn-lambda-permission-eventsourcetoken
        '''
        result = self._values.get("event_source_token")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def function_url_auth_type(self) -> typing.Optional[builtins.str]:
        '''The type of authentication that your function URL uses.

        Set to ``AWS_IAM`` if you want to restrict access to authenticated IAM users only. Set to ``NONE`` if you want to bypass IAM authentication to create a public endpoint. For more information, see `Security and auth model for Lambda function URLs <https://docs.aws.amazon.com/lambda/latest/dg/urls-auth.html>`_ .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-permission.html#cfn-lambda-permission-functionurlauthtype
        '''
        result = self._values.get("function_url_auth_type")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def principal_org_id(self) -> typing.Optional[builtins.str]:
        '''The identifier for your organization in AWS Organizations .

        Use this to grant permissions to all the AWS accounts under this organization.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-permission.html#cfn-lambda-permission-principalorgid
        '''
        result = self._values.get("principal_org_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def source_account(self) -> typing.Optional[builtins.str]:
        '''For AWS service , the ID of the AWS account that owns the resource.

        Use this together with ``SourceArn`` to ensure that the specified account owns the resource. It is possible for an Amazon S3 bucket to be deleted by its owner and recreated by another account.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-permission.html#cfn-lambda-permission-sourceaccount
        '''
        result = self._values.get("source_account")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def source_arn(self) -> typing.Optional[builtins.str]:
        '''For AWS services , the ARN of the AWS resource that invokes the function.

        For example, an Amazon S3 bucket or Amazon SNS topic.

        Note that Lambda configures the comparison using the ``StringLike`` operator.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-permission.html#cfn-lambda-permission-sourcearn
        '''
        result = self._values.get("source_arn")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnPermissionProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_aws_cdk_core_f4b25747.IInspectable)
class CfnUrl(
    _aws_cdk_core_f4b25747.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-lambda.CfnUrl",
):
    '''A CloudFormation ``AWS::Lambda::Url``.

    The ``AWS::Lambda::Url`` resource creates a function URL with the specified configuration parameters. A `function URL <https://docs.aws.amazon.com/lambda/latest/dg/lambda-urls.html>`_ is a dedicated HTTP(S) endpoint that you can use to invoke your function.

    :cloudformationResource: AWS::Lambda::Url
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-url.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        import aws_cdk.aws_lambda as lambda_
        
        cfn_url = lambda_.CfnUrl(self, "MyCfnUrl",
            auth_type="authType",
            target_function_arn="targetFunctionArn",
        
            # the properties below are optional
            cors=lambda.CfnUrl.CorsProperty(
                allow_credentials=False,
                allow_headers=["allowHeaders"],
                allow_methods=["allowMethods"],
                allow_origins=["allowOrigins"],
                expose_headers=["exposeHeaders"],
                max_age=123
            ),
            invoke_mode="invokeMode",
            qualifier="qualifier"
        )
    '''

    def __init__(
        self,
        scope: _aws_cdk_core_f4b25747.Construct,
        id: builtins.str,
        *,
        auth_type: builtins.str,
        target_function_arn: builtins.str,
        cors: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnUrl.CorsProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        invoke_mode: typing.Optional[builtins.str] = None,
        qualifier: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Create a new ``AWS::Lambda::Url``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param auth_type: The type of authentication that your function URL uses. Set to ``AWS_IAM`` if you want to restrict access to authenticated IAM users only. Set to ``NONE`` if you want to bypass IAM authentication to create a public endpoint. For more information, see `Security and auth model for Lambda function URLs <https://docs.aws.amazon.com/lambda/latest/dg/urls-auth.html>`_ .
        :param target_function_arn: The name of the Lambda function. **Name formats** - *Function name* - ``my-function`` . - *Function ARN* - ``arn:aws:lambda:us-west-2:123456789012:function:my-function`` . - *Partial ARN* - ``123456789012:function:my-function`` . The length constraint applies only to the full ARN. If you specify only the function name, it is limited to 64 characters in length.
        :param cors: The `Cross-Origin Resource Sharing (CORS) <https://docs.aws.amazon.com/https://developer.mozilla.org/en-US/docs/Web/HTTP/CORS>`_ settings for your function URL.
        :param invoke_mode: ``AWS::Lambda::Url.InvokeMode``.
        :param qualifier: The alias name.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__44d5d083cecf12aa3a06a5e19750778daef4514abd11ab0aec1a7586de749255)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnUrlProps(
            auth_type=auth_type,
            target_function_arn=target_function_arn,
            cors=cors,
            invoke_mode=invoke_mode,
            qualifier=qualifier,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _aws_cdk_core_f4b25747.TreeInspector) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__918637872f9b022749c9dad36a7dad1e5e040d7c86966e1f8cc81c341565981e)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f63866c87dab90a5132994bdec9c55683188b629d58ee16f7dfd704d69ca5b32)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrFunctionArn")
    def attr_function_arn(self) -> builtins.str:
        '''The Amazon Resource Name (ARN) of the function.

        :cloudformationAttribute: FunctionArn
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrFunctionArn"))

    @builtins.property
    @jsii.member(jsii_name="attrFunctionUrl")
    def attr_function_url(self) -> builtins.str:
        '''The HTTP URL endpoint for your function.

        :cloudformationAttribute: FunctionUrl
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrFunctionUrl"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="authType")
    def auth_type(self) -> builtins.str:
        '''The type of authentication that your function URL uses.

        Set to ``AWS_IAM`` if you want to restrict access to authenticated IAM users only. Set to ``NONE`` if you want to bypass IAM authentication to create a public endpoint. For more information, see `Security and auth model for Lambda function URLs <https://docs.aws.amazon.com/lambda/latest/dg/urls-auth.html>`_ .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-url.html#cfn-lambda-url-authtype
        '''
        return typing.cast(builtins.str, jsii.get(self, "authType"))

    @auth_type.setter
    def auth_type(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0cc475a620df21644a68b79e6c8e24270fddf1825d726fcbeca23d650c68440b)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "authType", value)

    @builtins.property
    @jsii.member(jsii_name="targetFunctionArn")
    def target_function_arn(self) -> builtins.str:
        '''The name of the Lambda function.

        **Name formats** - *Function name* - ``my-function`` .

        - *Function ARN* - ``arn:aws:lambda:us-west-2:123456789012:function:my-function`` .
        - *Partial ARN* - ``123456789012:function:my-function`` .

        The length constraint applies only to the full ARN. If you specify only the function name, it is limited to 64 characters in length.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-url.html#cfn-lambda-url-targetfunctionarn
        '''
        return typing.cast(builtins.str, jsii.get(self, "targetFunctionArn"))

    @target_function_arn.setter
    def target_function_arn(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__277b3b61f6633b9f89e91d053833bd11654a2f0291dddea9fb2c264cee384020)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "targetFunctionArn", value)

    @builtins.property
    @jsii.member(jsii_name="cors")
    def cors(
        self,
    ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnUrl.CorsProperty"]]:
        '''The `Cross-Origin Resource Sharing (CORS) <https://docs.aws.amazon.com/https://developer.mozilla.org/en-US/docs/Web/HTTP/CORS>`_ settings for your function URL.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-url.html#cfn-lambda-url-cors
        '''
        return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnUrl.CorsProperty"]], jsii.get(self, "cors"))

    @cors.setter
    def cors(
        self,
        value: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnUrl.CorsProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f47707a18cf7ddb5b3c9016da4e09b789e5629d0bef620b5eea65ce249f98074)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "cors", value)

    @builtins.property
    @jsii.member(jsii_name="invokeMode")
    def invoke_mode(self) -> typing.Optional[builtins.str]:
        '''``AWS::Lambda::Url.InvokeMode``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-url.html#cfn-lambda-url-invokemode
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "invokeMode"))

    @invoke_mode.setter
    def invoke_mode(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f0615f235c2ed6a261df5243e02860b658448b0552f85e60051a90c997ceea90)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "invokeMode", value)

    @builtins.property
    @jsii.member(jsii_name="qualifier")
    def qualifier(self) -> typing.Optional[builtins.str]:
        '''The alias name.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-url.html#cfn-lambda-url-qualifier
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "qualifier"))

    @qualifier.setter
    def qualifier(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__dc1e30506e819fe79f03dc1538dc80764431c7e312f04ed8a6cc8dc2810abe4e)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "qualifier", value)

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-lambda.CfnUrl.CorsProperty",
        jsii_struct_bases=[],
        name_mapping={
            "allow_credentials": "allowCredentials",
            "allow_headers": "allowHeaders",
            "allow_methods": "allowMethods",
            "allow_origins": "allowOrigins",
            "expose_headers": "exposeHeaders",
            "max_age": "maxAge",
        },
    )
    class CorsProperty:
        def __init__(
            self,
            *,
            allow_credentials: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]] = None,
            allow_headers: typing.Optional[typing.Sequence[builtins.str]] = None,
            allow_methods: typing.Optional[typing.Sequence[builtins.str]] = None,
            allow_origins: typing.Optional[typing.Sequence[builtins.str]] = None,
            expose_headers: typing.Optional[typing.Sequence[builtins.str]] = None,
            max_age: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''The `Cross-Origin Resource Sharing (CORS) <https://docs.aws.amazon.com/https://developer.mozilla.org/en-US/docs/Web/HTTP/CORS>`_ settings for your function URL. Use CORS to grant access to your function URL from any origin. You can also use CORS to control access for specific HTTP headers and methods in requests to your function URL.

            :param allow_credentials: Whether you want to allow cookies or other credentials in requests to your function URL. The default is ``false`` .
            :param allow_headers: The HTTP headers that origins can include in requests to your function URL. For example: ``Date`` , ``Keep-Alive`` , ``X-Custom-Header`` .
            :param allow_methods: The HTTP methods that are allowed when calling your function URL. For example: ``GET`` , ``POST`` , ``DELETE`` , or the wildcard character ( ``*`` ).
            :param allow_origins: The origins that can access your function URL. You can list any number of specific origins, separated by a comma. For example: ``https://www.example.com`` , ``http://localhost:60905`` . Alternatively, you can grant access to all origins with the wildcard character ( ``*`` ).
            :param expose_headers: The HTTP headers in your function response that you want to expose to origins that call your function URL. For example: ``Date`` , ``Keep-Alive`` , ``X-Custom-Header`` .
            :param max_age: The maximum amount of time, in seconds, that browsers can cache results of a preflight request. By default, this is set to ``0`` , which means the browser will not cache results.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-url-cors.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_lambda as lambda_
                
                cors_property = lambda.CfnUrl.CorsProperty(
                    allow_credentials=False,
                    allow_headers=["allowHeaders"],
                    allow_methods=["allowMethods"],
                    allow_origins=["allowOrigins"],
                    expose_headers=["exposeHeaders"],
                    max_age=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__abe5034bccf3639df21ab3b0022da3d0ebcfae7d981c464753f63e359a444819)
                check_type(argname="argument allow_credentials", value=allow_credentials, expected_type=type_hints["allow_credentials"])
                check_type(argname="argument allow_headers", value=allow_headers, expected_type=type_hints["allow_headers"])
                check_type(argname="argument allow_methods", value=allow_methods, expected_type=type_hints["allow_methods"])
                check_type(argname="argument allow_origins", value=allow_origins, expected_type=type_hints["allow_origins"])
                check_type(argname="argument expose_headers", value=expose_headers, expected_type=type_hints["expose_headers"])
                check_type(argname="argument max_age", value=max_age, expected_type=type_hints["max_age"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if allow_credentials is not None:
                self._values["allow_credentials"] = allow_credentials
            if allow_headers is not None:
                self._values["allow_headers"] = allow_headers
            if allow_methods is not None:
                self._values["allow_methods"] = allow_methods
            if allow_origins is not None:
                self._values["allow_origins"] = allow_origins
            if expose_headers is not None:
                self._values["expose_headers"] = expose_headers
            if max_age is not None:
                self._values["max_age"] = max_age

        @builtins.property
        def allow_credentials(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]]:
            '''Whether you want to allow cookies or other credentials in requests to your function URL.

            The default is ``false`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-url-cors.html#cfn-lambda-url-cors-allowcredentials
            '''
            result = self._values.get("allow_credentials")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]], result)

        @builtins.property
        def allow_headers(self) -> typing.Optional[typing.List[builtins.str]]:
            '''The HTTP headers that origins can include in requests to your function URL.

            For example: ``Date`` , ``Keep-Alive`` , ``X-Custom-Header`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-url-cors.html#cfn-lambda-url-cors-allowheaders
            '''
            result = self._values.get("allow_headers")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def allow_methods(self) -> typing.Optional[typing.List[builtins.str]]:
            '''The HTTP methods that are allowed when calling your function URL.

            For example: ``GET`` , ``POST`` , ``DELETE`` , or the wildcard character ( ``*`` ).

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-url-cors.html#cfn-lambda-url-cors-allowmethods
            '''
            result = self._values.get("allow_methods")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def allow_origins(self) -> typing.Optional[typing.List[builtins.str]]:
            '''The origins that can access your function URL.

            You can list any number of specific origins, separated by a comma. For example: ``https://www.example.com`` , ``http://localhost:60905`` .

            Alternatively, you can grant access to all origins with the wildcard character ( ``*`` ).

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-url-cors.html#cfn-lambda-url-cors-alloworigins
            '''
            result = self._values.get("allow_origins")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def expose_headers(self) -> typing.Optional[typing.List[builtins.str]]:
            '''The HTTP headers in your function response that you want to expose to origins that call your function URL.

            For example: ``Date`` , ``Keep-Alive`` , ``X-Custom-Header`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-url-cors.html#cfn-lambda-url-cors-exposeheaders
            '''
            result = self._values.get("expose_headers")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def max_age(self) -> typing.Optional[jsii.Number]:
            '''The maximum amount of time, in seconds, that browsers can cache results of a preflight request.

            By default, this is set to ``0`` , which means the browser will not cache results.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-url-cors.html#cfn-lambda-url-cors-maxage
            '''
            result = self._values.get("max_age")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "CorsProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-lambda.CfnUrlProps",
    jsii_struct_bases=[],
    name_mapping={
        "auth_type": "authType",
        "target_function_arn": "targetFunctionArn",
        "cors": "cors",
        "invoke_mode": "invokeMode",
        "qualifier": "qualifier",
    },
)
class CfnUrlProps:
    def __init__(
        self,
        *,
        auth_type: builtins.str,
        target_function_arn: builtins.str,
        cors: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union[CfnUrl.CorsProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        invoke_mode: typing.Optional[builtins.str] = None,
        qualifier: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for defining a ``CfnUrl``.

        :param auth_type: The type of authentication that your function URL uses. Set to ``AWS_IAM`` if you want to restrict access to authenticated IAM users only. Set to ``NONE`` if you want to bypass IAM authentication to create a public endpoint. For more information, see `Security and auth model for Lambda function URLs <https://docs.aws.amazon.com/lambda/latest/dg/urls-auth.html>`_ .
        :param target_function_arn: The name of the Lambda function. **Name formats** - *Function name* - ``my-function`` . - *Function ARN* - ``arn:aws:lambda:us-west-2:123456789012:function:my-function`` . - *Partial ARN* - ``123456789012:function:my-function`` . The length constraint applies only to the full ARN. If you specify only the function name, it is limited to 64 characters in length.
        :param cors: The `Cross-Origin Resource Sharing (CORS) <https://docs.aws.amazon.com/https://developer.mozilla.org/en-US/docs/Web/HTTP/CORS>`_ settings for your function URL.
        :param invoke_mode: ``AWS::Lambda::Url.InvokeMode``.
        :param qualifier: The alias name.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-url.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_lambda as lambda_
            
            cfn_url_props = lambda.CfnUrlProps(
                auth_type="authType",
                target_function_arn="targetFunctionArn",
            
                # the properties below are optional
                cors=lambda.CfnUrl.CorsProperty(
                    allow_credentials=False,
                    allow_headers=["allowHeaders"],
                    allow_methods=["allowMethods"],
                    allow_origins=["allowOrigins"],
                    expose_headers=["exposeHeaders"],
                    max_age=123
                ),
                invoke_mode="invokeMode",
                qualifier="qualifier"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9fdf0660b7c26fafee550e00fdb26a39bfc3012137bfc645c1e22273523d1e36)
            check_type(argname="argument auth_type", value=auth_type, expected_type=type_hints["auth_type"])
            check_type(argname="argument target_function_arn", value=target_function_arn, expected_type=type_hints["target_function_arn"])
            check_type(argname="argument cors", value=cors, expected_type=type_hints["cors"])
            check_type(argname="argument invoke_mode", value=invoke_mode, expected_type=type_hints["invoke_mode"])
            check_type(argname="argument qualifier", value=qualifier, expected_type=type_hints["qualifier"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "auth_type": auth_type,
            "target_function_arn": target_function_arn,
        }
        if cors is not None:
            self._values["cors"] = cors
        if invoke_mode is not None:
            self._values["invoke_mode"] = invoke_mode
        if qualifier is not None:
            self._values["qualifier"] = qualifier

    @builtins.property
    def auth_type(self) -> builtins.str:
        '''The type of authentication that your function URL uses.

        Set to ``AWS_IAM`` if you want to restrict access to authenticated IAM users only. Set to ``NONE`` if you want to bypass IAM authentication to create a public endpoint. For more information, see `Security and auth model for Lambda function URLs <https://docs.aws.amazon.com/lambda/latest/dg/urls-auth.html>`_ .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-url.html#cfn-lambda-url-authtype
        '''
        result = self._values.get("auth_type")
        assert result is not None, "Required property 'auth_type' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def target_function_arn(self) -> builtins.str:
        '''The name of the Lambda function.

        **Name formats** - *Function name* - ``my-function`` .

        - *Function ARN* - ``arn:aws:lambda:us-west-2:123456789012:function:my-function`` .
        - *Partial ARN* - ``123456789012:function:my-function`` .

        The length constraint applies only to the full ARN. If you specify only the function name, it is limited to 64 characters in length.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-url.html#cfn-lambda-url-targetfunctionarn
        '''
        result = self._values.get("target_function_arn")
        assert result is not None, "Required property 'target_function_arn' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def cors(
        self,
    ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, CfnUrl.CorsProperty]]:
        '''The `Cross-Origin Resource Sharing (CORS) <https://docs.aws.amazon.com/https://developer.mozilla.org/en-US/docs/Web/HTTP/CORS>`_ settings for your function URL.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-url.html#cfn-lambda-url-cors
        '''
        result = self._values.get("cors")
        return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, CfnUrl.CorsProperty]], result)

    @builtins.property
    def invoke_mode(self) -> typing.Optional[builtins.str]:
        '''``AWS::Lambda::Url.InvokeMode``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-url.html#cfn-lambda-url-invokemode
        '''
        result = self._values.get("invoke_mode")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def qualifier(self) -> typing.Optional[builtins.str]:
        '''The alias name.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-url.html#cfn-lambda-url-qualifier
        '''
        result = self._values.get("qualifier")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnUrlProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_aws_cdk_core_f4b25747.IInspectable)
class CfnVersion(
    _aws_cdk_core_f4b25747.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-lambda.CfnVersion",
):
    '''A CloudFormation ``AWS::Lambda::Version``.

    The ``AWS::Lambda::Version`` resource creates a `version <https://docs.aws.amazon.com/lambda/latest/dg/versioning-aliases.html>`_ from the current code and configuration of a function. Use versions to create a snapshot of your function code and configuration that doesn't change.

    :cloudformationResource: AWS::Lambda::Version
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-version.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        import aws_cdk.aws_lambda as lambda_
        
        cfn_version = lambda_.CfnVersion(self, "MyCfnVersion",
            function_name="functionName",
        
            # the properties below are optional
            code_sha256="codeSha256",
            description="description",
            provisioned_concurrency_config=lambda.CfnVersion.ProvisionedConcurrencyConfigurationProperty(
                provisioned_concurrent_executions=123
            )
        )
    '''

    def __init__(
        self,
        scope: _aws_cdk_core_f4b25747.Construct,
        id: builtins.str,
        *,
        function_name: builtins.str,
        code_sha256: typing.Optional[builtins.str] = None,
        description: typing.Optional[builtins.str] = None,
        provisioned_concurrency_config: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnVersion.ProvisionedConcurrencyConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''Create a new ``AWS::Lambda::Version``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param function_name: The name of the Lambda function. **Name formats** - *Function name* - ``MyFunction`` . - *Function ARN* - ``arn:aws:lambda:us-west-2:123456789012:function:MyFunction`` . - *Partial ARN* - ``123456789012:function:MyFunction`` . The length constraint applies only to the full ARN. If you specify only the function name, it is limited to 64 characters in length.
        :param code_sha256: Only publish a version if the hash value matches the value that's specified. Use this option to avoid publishing a version if the function code has changed since you last updated it. Updates are not supported for this property.
        :param description: A description for the version to override the description in the function configuration. Updates are not supported for this property.
        :param provisioned_concurrency_config: Specifies a provisioned concurrency configuration for a function's version. Updates are not supported for this property.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__60a793c5ba90abf2afa42b55301a59c388e6d5915e9ee24883aed88d6e8ab70c)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnVersionProps(
            function_name=function_name,
            code_sha256=code_sha256,
            description=description,
            provisioned_concurrency_config=provisioned_concurrency_config,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _aws_cdk_core_f4b25747.TreeInspector) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__66da5ffbd39beb01832a37d905f080772d25d566c8968f1f48cd3e210145ed11)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a7f2e28794bad5912d3e4ff3b6d1f52c80333253c0cf700d9b8b67c78fc0a1dc)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrVersion")
    def attr_version(self) -> builtins.str:
        '''The version number.

        :cloudformationAttribute: Version
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrVersion"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="functionName")
    def function_name(self) -> builtins.str:
        '''The name of the Lambda function.

        **Name formats** - *Function name* - ``MyFunction`` .

        - *Function ARN* - ``arn:aws:lambda:us-west-2:123456789012:function:MyFunction`` .
        - *Partial ARN* - ``123456789012:function:MyFunction`` .

        The length constraint applies only to the full ARN. If you specify only the function name, it is limited to 64 characters in length.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-version.html#cfn-lambda-version-functionname
        '''
        return typing.cast(builtins.str, jsii.get(self, "functionName"))

    @function_name.setter
    def function_name(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5ce086e9a4038598e1ea9b440f0fa927865aaeea8383ca0240ba451d75aaf600)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "functionName", value)

    @builtins.property
    @jsii.member(jsii_name="codeSha256")
    def code_sha256(self) -> typing.Optional[builtins.str]:
        '''Only publish a version if the hash value matches the value that's specified.

        Use this option to avoid publishing a version if the function code has changed since you last updated it. Updates are not supported for this property.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-version.html#cfn-lambda-version-codesha256
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "codeSha256"))

    @code_sha256.setter
    def code_sha256(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__bedad95038df8e34790c6a5ba5e39d9dffc0a5f659b6605b9aea21564fd29eb1)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "codeSha256", value)

    @builtins.property
    @jsii.member(jsii_name="description")
    def description(self) -> typing.Optional[builtins.str]:
        '''A description for the version to override the description in the function configuration.

        Updates are not supported for this property.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-version.html#cfn-lambda-version-description
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "description"))

    @description.setter
    def description(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ea3cab432efe4c04fc4c102c7aa259e84544969448fe6a3b1be543c7602bd956)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "description", value)

    @builtins.property
    @jsii.member(jsii_name="provisionedConcurrencyConfig")
    def provisioned_concurrency_config(
        self,
    ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnVersion.ProvisionedConcurrencyConfigurationProperty"]]:
        '''Specifies a provisioned concurrency configuration for a function's version.

        Updates are not supported for this property.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-version.html#cfn-lambda-version-provisionedconcurrencyconfig
        '''
        return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnVersion.ProvisionedConcurrencyConfigurationProperty"]], jsii.get(self, "provisionedConcurrencyConfig"))

    @provisioned_concurrency_config.setter
    def provisioned_concurrency_config(
        self,
        value: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnVersion.ProvisionedConcurrencyConfigurationProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__54ea1205c523ca0607ee49212e64d9b93aaa1031613245e61270bcfcee89e3ec)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "provisionedConcurrencyConfig", value)

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-lambda.CfnVersion.ProvisionedConcurrencyConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "provisioned_concurrent_executions": "provisionedConcurrentExecutions",
        },
    )
    class ProvisionedConcurrencyConfigurationProperty:
        def __init__(self, *, provisioned_concurrent_executions: jsii.Number) -> None:
            '''A `provisioned concurrency <https://docs.aws.amazon.com/lambda/latest/dg/configuration-concurrency.html>`_ configuration for a function's version.

            :param provisioned_concurrent_executions: The amount of provisioned concurrency to allocate for the version.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-version-provisionedconcurrencyconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_lambda as lambda_
                
                provisioned_concurrency_configuration_property = lambda.CfnVersion.ProvisionedConcurrencyConfigurationProperty(
                    provisioned_concurrent_executions=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__7db38bcc3c227a71f814890709d4c80663700f63dd2eb619cf32d2bf6718d995)
                check_type(argname="argument provisioned_concurrent_executions", value=provisioned_concurrent_executions, expected_type=type_hints["provisioned_concurrent_executions"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "provisioned_concurrent_executions": provisioned_concurrent_executions,
            }

        @builtins.property
        def provisioned_concurrent_executions(self) -> jsii.Number:
            '''The amount of provisioned concurrency to allocate for the version.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-version-provisionedconcurrencyconfiguration.html#cfn-lambda-version-provisionedconcurrencyconfiguration-provisionedconcurrentexecutions
            '''
            result = self._values.get("provisioned_concurrent_executions")
            assert result is not None, "Required property 'provisioned_concurrent_executions' is missing"
            return typing.cast(jsii.Number, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ProvisionedConcurrencyConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-lambda.CfnVersionProps",
    jsii_struct_bases=[],
    name_mapping={
        "function_name": "functionName",
        "code_sha256": "codeSha256",
        "description": "description",
        "provisioned_concurrency_config": "provisionedConcurrencyConfig",
    },
)
class CfnVersionProps:
    def __init__(
        self,
        *,
        function_name: builtins.str,
        code_sha256: typing.Optional[builtins.str] = None,
        description: typing.Optional[builtins.str] = None,
        provisioned_concurrency_config: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union[CfnVersion.ProvisionedConcurrencyConfigurationProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''Properties for defining a ``CfnVersion``.

        :param function_name: The name of the Lambda function. **Name formats** - *Function name* - ``MyFunction`` . - *Function ARN* - ``arn:aws:lambda:us-west-2:123456789012:function:MyFunction`` . - *Partial ARN* - ``123456789012:function:MyFunction`` . The length constraint applies only to the full ARN. If you specify only the function name, it is limited to 64 characters in length.
        :param code_sha256: Only publish a version if the hash value matches the value that's specified. Use this option to avoid publishing a version if the function code has changed since you last updated it. Updates are not supported for this property.
        :param description: A description for the version to override the description in the function configuration. Updates are not supported for this property.
        :param provisioned_concurrency_config: Specifies a provisioned concurrency configuration for a function's version. Updates are not supported for this property.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-version.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_lambda as lambda_
            
            cfn_version_props = lambda.CfnVersionProps(
                function_name="functionName",
            
                # the properties below are optional
                code_sha256="codeSha256",
                description="description",
                provisioned_concurrency_config=lambda.CfnVersion.ProvisionedConcurrencyConfigurationProperty(
                    provisioned_concurrent_executions=123
                )
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__866abc6bc31dc200546818d9bc047b59cb47527f9c0bae75c36cc2b273abf164)
            check_type(argname="argument function_name", value=function_name, expected_type=type_hints["function_name"])
            check_type(argname="argument code_sha256", value=code_sha256, expected_type=type_hints["code_sha256"])
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument provisioned_concurrency_config", value=provisioned_concurrency_config, expected_type=type_hints["provisioned_concurrency_config"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "function_name": function_name,
        }
        if code_sha256 is not None:
            self._values["code_sha256"] = code_sha256
        if description is not None:
            self._values["description"] = description
        if provisioned_concurrency_config is not None:
            self._values["provisioned_concurrency_config"] = provisioned_concurrency_config

    @builtins.property
    def function_name(self) -> builtins.str:
        '''The name of the Lambda function.

        **Name formats** - *Function name* - ``MyFunction`` .

        - *Function ARN* - ``arn:aws:lambda:us-west-2:123456789012:function:MyFunction`` .
        - *Partial ARN* - ``123456789012:function:MyFunction`` .

        The length constraint applies only to the full ARN. If you specify only the function name, it is limited to 64 characters in length.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-version.html#cfn-lambda-version-functionname
        '''
        result = self._values.get("function_name")
        assert result is not None, "Required property 'function_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def code_sha256(self) -> typing.Optional[builtins.str]:
        '''Only publish a version if the hash value matches the value that's specified.

        Use this option to avoid publishing a version if the function code has changed since you last updated it. Updates are not supported for this property.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-version.html#cfn-lambda-version-codesha256
        '''
        result = self._values.get("code_sha256")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''A description for the version to override the description in the function configuration.

        Updates are not supported for this property.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-version.html#cfn-lambda-version-description
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def provisioned_concurrency_config(
        self,
    ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, CfnVersion.ProvisionedConcurrencyConfigurationProperty]]:
        '''Specifies a provisioned concurrency configuration for a function's version.

        Updates are not supported for this property.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-version.html#cfn-lambda-version-provisionedconcurrencyconfig
        '''
        result = self._values.get("provisioned_concurrency_config")
        return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, CfnVersion.ProvisionedConcurrencyConfigurationProperty]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnVersionProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class Code(metaclass=jsii.JSIIAbstractClass, jsii_type="@aws-cdk/aws-lambda.Code"):
    '''Represents the Lambda Handler Code.

    :exampleMetadata: infused

    Example::

        import aws_cdk.aws_signer as signer
        
        
        signing_profile = signer.SigningProfile(self, "SigningProfile",
            platform=signer.Platform.AWS_LAMBDA_SHA384_ECDSA
        )
        
        code_signing_config = lambda_.CodeSigningConfig(self, "CodeSigningConfig",
            signing_profiles=[signing_profile]
        )
        
        lambda_.Function(self, "Function",
            code_signing_config=code_signing_config,
            runtime=lambda_.Runtime.NODEJS_16_X,
            handler="index.handler",
            code=lambda_.Code.from_asset(path.join(__dirname, "lambda-handler"))
        )
    '''

    def __init__(self) -> None:
        jsii.create(self.__class__, self, [])

    @jsii.member(jsii_name="asset")
    @builtins.classmethod
    def asset(cls, path: builtins.str) -> "AssetCode":
        '''(deprecated) DEPRECATED.

        :param path: -

        :deprecated: use ``fromAsset``

        :stability: deprecated
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__98377cc3012adac89d28df9d499ef16b9cbe27d1e93cfa9271ec605df0b25908)
            check_type(argname="argument path", value=path, expected_type=type_hints["path"])
        return typing.cast("AssetCode", jsii.sinvoke(cls, "asset", [path]))

    @jsii.member(jsii_name="bucket")
    @builtins.classmethod
    def bucket(
        cls,
        bucket: _aws_cdk_aws_s3_55f001a5.IBucket,
        key: builtins.str,
        object_version: typing.Optional[builtins.str] = None,
    ) -> "S3Code":
        '''(deprecated) DEPRECATED.

        :param bucket: -
        :param key: -
        :param object_version: -

        :deprecated: use ``fromBucket``

        :stability: deprecated
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3940fa5f67d3b3e0099c6b0265061fbccb07f1ef0c33bcafd6813a40b1a7f676)
            check_type(argname="argument bucket", value=bucket, expected_type=type_hints["bucket"])
            check_type(argname="argument key", value=key, expected_type=type_hints["key"])
            check_type(argname="argument object_version", value=object_version, expected_type=type_hints["object_version"])
        return typing.cast("S3Code", jsii.sinvoke(cls, "bucket", [bucket, key, object_version]))

    @jsii.member(jsii_name="cfnParameters")
    @builtins.classmethod
    def cfn_parameters(
        cls,
        *,
        bucket_name_param: typing.Optional[_aws_cdk_core_f4b25747.CfnParameter] = None,
        object_key_param: typing.Optional[_aws_cdk_core_f4b25747.CfnParameter] = None,
    ) -> "CfnParametersCode":
        '''(deprecated) DEPRECATED.

        :param bucket_name_param: The CloudFormation parameter that represents the name of the S3 Bucket where the Lambda code will be located in. Must be of type 'String'. Default: a new parameter will be created
        :param object_key_param: The CloudFormation parameter that represents the path inside the S3 Bucket where the Lambda code will be located at. Must be of type 'String'. Default: a new parameter will be created

        :deprecated: use ``fromCfnParameters``

        :stability: deprecated
        '''
        props = CfnParametersCodeProps(
            bucket_name_param=bucket_name_param, object_key_param=object_key_param
        )

        return typing.cast("CfnParametersCode", jsii.sinvoke(cls, "cfnParameters", [props]))

    @jsii.member(jsii_name="fromAsset")
    @builtins.classmethod
    def from_asset(
        cls,
        path: builtins.str,
        *,
        readers: typing.Optional[typing.Sequence[_aws_cdk_aws_iam_940a1ce0.IGrantable]] = None,
        source_hash: typing.Optional[builtins.str] = None,
        exclude: typing.Optional[typing.Sequence[builtins.str]] = None,
        follow: typing.Optional[_aws_cdk_assets_b1c45fb6.FollowMode] = None,
        ignore_mode: typing.Optional[_aws_cdk_core_f4b25747.IgnoreMode] = None,
        follow_symlinks: typing.Optional[_aws_cdk_core_f4b25747.SymlinkFollowMode] = None,
        asset_hash: typing.Optional[builtins.str] = None,
        asset_hash_type: typing.Optional[_aws_cdk_core_f4b25747.AssetHashType] = None,
        bundling: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.BundlingOptions, typing.Dict[builtins.str, typing.Any]]] = None,
    ) -> "AssetCode":
        '''Loads the function code from a local disk path.

        :param path: Either a directory with the Lambda code bundle or a .zip file.
        :param readers: A list of principals that should be able to read this asset from S3. You can use ``asset.grantRead(principal)`` to grant read permissions later. Default: - No principals that can read file asset.
        :param source_hash: (deprecated) Custom hash to use when identifying the specific version of the asset. For consistency, this custom hash will be SHA256 hashed and encoded as hex. The resulting hash will be the asset hash. NOTE: the source hash is used in order to identify a specific revision of the asset, and used for optimizing and caching deployment activities related to this asset such as packaging, uploading to Amazon S3, etc. If you chose to customize the source hash, you will need to make sure it is updated every time the source changes, or otherwise it is possible that some deployments will not be invalidated. Default: - automatically calculate source hash based on the contents of the source file or directory.
        :param exclude: (deprecated) Glob patterns to exclude from the copy. Default: nothing is excluded
        :param follow: (deprecated) A strategy for how to handle symlinks. Default: Never
        :param ignore_mode: (deprecated) The ignore behavior to use for exclude patterns. Default: - GLOB for file assets, DOCKER or GLOB for docker assets depending on whether the '
        :param follow_symlinks: A strategy for how to handle symlinks. Default: SymlinkFollowMode.NEVER
        :param asset_hash: Specify a custom hash for this asset. If ``assetHashType`` is set it must be set to ``AssetHashType.CUSTOM``. For consistency, this custom hash will be SHA256 hashed and encoded as hex. The resulting hash will be the asset hash. NOTE: the hash is used in order to identify a specific revision of the asset, and used for optimizing and caching deployment activities related to this asset such as packaging, uploading to Amazon S3, etc. If you chose to customize the hash, you will need to make sure it is updated every time the asset changes, or otherwise it is possible that some deployments will not be invalidated. Default: - based on ``assetHashType``
        :param asset_hash_type: Specifies the type of hash to calculate for this asset. If ``assetHash`` is configured, this option must be ``undefined`` or ``AssetHashType.CUSTOM``. Default: - the default is ``AssetHashType.SOURCE``, but if ``assetHash`` is explicitly specified this value defaults to ``AssetHashType.CUSTOM``.
        :param bundling: Bundle the asset by executing a command in a Docker container or a custom bundling provider. The asset path will be mounted at ``/asset-input``. The Docker container is responsible for putting content at ``/asset-output``. The content at ``/asset-output`` will be zipped and used as the final asset. Default: - uploaded as-is to S3 if the asset is a regular file or a .zip file, archived into a .zip file and uploaded to S3 otherwise
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__292fb9a5d171550c9f040b782e8416d587d80592fb28408987581762baab7e16)
            check_type(argname="argument path", value=path, expected_type=type_hints["path"])
        options = _aws_cdk_aws_s3_assets_525817d7.AssetOptions(
            readers=readers,
            source_hash=source_hash,
            exclude=exclude,
            follow=follow,
            ignore_mode=ignore_mode,
            follow_symlinks=follow_symlinks,
            asset_hash=asset_hash,
            asset_hash_type=asset_hash_type,
            bundling=bundling,
        )

        return typing.cast("AssetCode", jsii.sinvoke(cls, "fromAsset", [path, options]))

    @jsii.member(jsii_name="fromAssetImage")
    @builtins.classmethod
    def from_asset_image(
        cls,
        directory: builtins.str,
        *,
        cmd: typing.Optional[typing.Sequence[builtins.str]] = None,
        entrypoint: typing.Optional[typing.Sequence[builtins.str]] = None,
        working_directory: typing.Optional[builtins.str] = None,
        build_args: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        file: typing.Optional[builtins.str] = None,
        invalidation: typing.Optional[typing.Union[_aws_cdk_aws_ecr_assets_ef09b699.DockerImageAssetInvalidationOptions, typing.Dict[builtins.str, typing.Any]]] = None,
        network_mode: typing.Optional[_aws_cdk_aws_ecr_assets_ef09b699.NetworkMode] = None,
        platform: typing.Optional[_aws_cdk_aws_ecr_assets_ef09b699.Platform] = None,
        repository_name: typing.Optional[builtins.str] = None,
        target: typing.Optional[builtins.str] = None,
        extra_hash: typing.Optional[builtins.str] = None,
        exclude: typing.Optional[typing.Sequence[builtins.str]] = None,
        follow: typing.Optional[_aws_cdk_assets_b1c45fb6.FollowMode] = None,
        ignore_mode: typing.Optional[_aws_cdk_core_f4b25747.IgnoreMode] = None,
        follow_symlinks: typing.Optional[_aws_cdk_core_f4b25747.SymlinkFollowMode] = None,
    ) -> "AssetImageCode":
        '''Create an ECR image from the specified asset and bind it as the Lambda code.

        :param directory: the directory from which the asset must be created.
        :param cmd: Specify or override the CMD on the specified Docker image or Dockerfile. This needs to be in the 'exec form', viz., ``[ 'executable', 'param1', 'param2' ]``. Default: - use the CMD specified in the docker image or Dockerfile.
        :param entrypoint: Specify or override the ENTRYPOINT on the specified Docker image or Dockerfile. An ENTRYPOINT allows you to configure a container that will run as an executable. This needs to be in the 'exec form', viz., ``[ 'executable', 'param1', 'param2' ]``. Default: - use the ENTRYPOINT in the docker image or Dockerfile.
        :param working_directory: Specify or override the WORKDIR on the specified Docker image or Dockerfile. A WORKDIR allows you to configure the working directory the container will use. Default: - use the WORKDIR in the docker image or Dockerfile.
        :param build_args: Build args to pass to the ``docker build`` command. Since Docker build arguments are resolved before deployment, keys and values cannot refer to unresolved tokens (such as ``lambda.functionArn`` or ``queue.queueUrl``). Default: - no build args are passed
        :param file: Path to the Dockerfile (relative to the directory). Default: 'Dockerfile'
        :param invalidation: Options to control which parameters are used to invalidate the asset hash. Default: - hash all parameters
        :param network_mode: Networking mode for the RUN commands during build. Support docker API 1.25+. Default: - no networking mode specified (the default networking mode ``NetworkMode.DEFAULT`` will be used)
        :param platform: Platform to build for. *Requires Docker Buildx*. Default: - no platform specified (the current machine architecture will be used)
        :param repository_name: (deprecated) ECR repository name. Specify this property if you need to statically address the image, e.g. from a Kubernetes Pod. Note, this is only the repository name, without the registry and the tag parts. Default: - the default ECR repository for CDK assets
        :param target: Docker target to build to. Default: - no target
        :param extra_hash: (deprecated) Extra information to encode into the fingerprint (e.g. build instructions and other inputs). Default: - hash is only based on source content
        :param exclude: (deprecated) Glob patterns to exclude from the copy. Default: nothing is excluded
        :param follow: (deprecated) A strategy for how to handle symlinks. Default: Never
        :param ignore_mode: (deprecated) The ignore behavior to use for exclude patterns. Default: - GLOB for file assets, DOCKER or GLOB for docker assets depending on whether the '
        :param follow_symlinks: A strategy for how to handle symlinks. Default: SymlinkFollowMode.NEVER
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e02c8e68313a4702896dc818218ad2ae4b442b3def0111922a5e9b80d23a2bd8)
            check_type(argname="argument directory", value=directory, expected_type=type_hints["directory"])
        props = AssetImageCodeProps(
            cmd=cmd,
            entrypoint=entrypoint,
            working_directory=working_directory,
            build_args=build_args,
            file=file,
            invalidation=invalidation,
            network_mode=network_mode,
            platform=platform,
            repository_name=repository_name,
            target=target,
            extra_hash=extra_hash,
            exclude=exclude,
            follow=follow,
            ignore_mode=ignore_mode,
            follow_symlinks=follow_symlinks,
        )

        return typing.cast("AssetImageCode", jsii.sinvoke(cls, "fromAssetImage", [directory, props]))

    @jsii.member(jsii_name="fromBucket")
    @builtins.classmethod
    def from_bucket(
        cls,
        bucket: _aws_cdk_aws_s3_55f001a5.IBucket,
        key: builtins.str,
        object_version: typing.Optional[builtins.str] = None,
    ) -> "S3Code":
        '''Lambda handler code as an S3 object.

        :param bucket: The S3 bucket.
        :param key: The object key.
        :param object_version: Optional S3 object version.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f37fa6396ee93bb22f1927bed5d4d9080dd3a0b6fd1c9767491601ab89d2ca7e)
            check_type(argname="argument bucket", value=bucket, expected_type=type_hints["bucket"])
            check_type(argname="argument key", value=key, expected_type=type_hints["key"])
            check_type(argname="argument object_version", value=object_version, expected_type=type_hints["object_version"])
        return typing.cast("S3Code", jsii.sinvoke(cls, "fromBucket", [bucket, key, object_version]))

    @jsii.member(jsii_name="fromCfnParameters")
    @builtins.classmethod
    def from_cfn_parameters(
        cls,
        *,
        bucket_name_param: typing.Optional[_aws_cdk_core_f4b25747.CfnParameter] = None,
        object_key_param: typing.Optional[_aws_cdk_core_f4b25747.CfnParameter] = None,
    ) -> "CfnParametersCode":
        '''Creates a new Lambda source defined using CloudFormation parameters.

        :param bucket_name_param: The CloudFormation parameter that represents the name of the S3 Bucket where the Lambda code will be located in. Must be of type 'String'. Default: a new parameter will be created
        :param object_key_param: The CloudFormation parameter that represents the path inside the S3 Bucket where the Lambda code will be located at. Must be of type 'String'. Default: a new parameter will be created

        :return: a new instance of ``CfnParametersCode``
        '''
        props = CfnParametersCodeProps(
            bucket_name_param=bucket_name_param, object_key_param=object_key_param
        )

        return typing.cast("CfnParametersCode", jsii.sinvoke(cls, "fromCfnParameters", [props]))

    @jsii.member(jsii_name="fromDockerBuild")
    @builtins.classmethod
    def from_docker_build(
        cls,
        path: builtins.str,
        *,
        image_path: typing.Optional[builtins.str] = None,
        output_path: typing.Optional[builtins.str] = None,
        build_args: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        file: typing.Optional[builtins.str] = None,
        platform: typing.Optional[builtins.str] = None,
    ) -> "AssetCode":
        '''Loads the function code from an asset created by a Docker build.

        By default, the asset is expected to be located at ``/asset`` in the
        image.

        :param path: The path to the directory containing the Docker file.
        :param image_path: The path in the Docker image where the asset is located after the build operation. Default: /asset
        :param output_path: The path on the local filesystem where the asset will be copied using ``docker cp``. Default: - a unique temporary directory in the system temp directory
        :param build_args: Build args. Default: - no build args
        :param file: Name of the Dockerfile, must relative to the docker build path. Default: ``Dockerfile``
        :param platform: Set platform if server is multi-platform capable. *Requires Docker Engine API v1.38+*. Example value: ``linux/amd64`` Default: - no platform specified
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__830f30f198ce9ba715d68d1a8e28208fc998215f50d1b1a214822c0d3e595d7b)
            check_type(argname="argument path", value=path, expected_type=type_hints["path"])
        options = DockerBuildAssetOptions(
            image_path=image_path,
            output_path=output_path,
            build_args=build_args,
            file=file,
            platform=platform,
        )

        return typing.cast("AssetCode", jsii.sinvoke(cls, "fromDockerBuild", [path, options]))

    @jsii.member(jsii_name="fromEcrImage")
    @builtins.classmethod
    def from_ecr_image(
        cls,
        repository: _aws_cdk_aws_ecr_093ed842.IRepository,
        *,
        cmd: typing.Optional[typing.Sequence[builtins.str]] = None,
        entrypoint: typing.Optional[typing.Sequence[builtins.str]] = None,
        tag: typing.Optional[builtins.str] = None,
        tag_or_digest: typing.Optional[builtins.str] = None,
        working_directory: typing.Optional[builtins.str] = None,
    ) -> "EcrImageCode":
        '''Use an existing ECR image as the Lambda code.

        :param repository: the ECR repository that the image is in.
        :param cmd: Specify or override the CMD on the specified Docker image or Dockerfile. This needs to be in the 'exec form', viz., ``[ 'executable', 'param1', 'param2' ]``. Default: - use the CMD specified in the docker image or Dockerfile.
        :param entrypoint: Specify or override the ENTRYPOINT on the specified Docker image or Dockerfile. An ENTRYPOINT allows you to configure a container that will run as an executable. This needs to be in the 'exec form', viz., ``[ 'executable', 'param1', 'param2' ]``. Default: - use the ENTRYPOINT in the docker image or Dockerfile.
        :param tag: (deprecated) The image tag to use when pulling the image from ECR. Default: 'latest'
        :param tag_or_digest: The image tag or digest to use when pulling the image from ECR (digests must start with ``sha256:``). Default: 'latest'
        :param working_directory: Specify or override the WORKDIR on the specified Docker image or Dockerfile. A WORKDIR allows you to configure the working directory the container will use. Default: - use the WORKDIR in the docker image or Dockerfile.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__29ade1265d8a00f67b2c62411bb184583e6e7c7cd8f4a72f40088526f06d8467)
            check_type(argname="argument repository", value=repository, expected_type=type_hints["repository"])
        props = EcrImageCodeProps(
            cmd=cmd,
            entrypoint=entrypoint,
            tag=tag,
            tag_or_digest=tag_or_digest,
            working_directory=working_directory,
        )

        return typing.cast("EcrImageCode", jsii.sinvoke(cls, "fromEcrImage", [repository, props]))

    @jsii.member(jsii_name="fromInline")
    @builtins.classmethod
    def from_inline(cls, code: builtins.str) -> "InlineCode":
        '''Inline code for Lambda handler.

        :param code: The actual handler code (limited to 4KiB).

        :return: ``LambdaInlineCode`` with inline code.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__51ba70f0680b564133f1ff59c804656eea40da5bbc733bdec874162352497e4e)
            check_type(argname="argument code", value=code, expected_type=type_hints["code"])
        return typing.cast("InlineCode", jsii.sinvoke(cls, "fromInline", [code]))

    @jsii.member(jsii_name="inline")
    @builtins.classmethod
    def inline(cls, code: builtins.str) -> "InlineCode":
        '''(deprecated) DEPRECATED.

        :param code: -

        :deprecated: use ``fromInline``

        :stability: deprecated
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__051b94233a30cd86fc3b332f5c4e7d40a887bc2e09982e3323cfdf26fa3424e6)
            check_type(argname="argument code", value=code, expected_type=type_hints["code"])
        return typing.cast("InlineCode", jsii.sinvoke(cls, "inline", [code]))

    @jsii.member(jsii_name="bind")
    @abc.abstractmethod
    def bind(self, scope: _aws_cdk_core_f4b25747.Construct) -> "CodeConfig":
        '''Called when the lambda or layer is initialized to allow this object to bind to the stack, add resources and have fun.

        :param scope: The binding scope. Don't be smart about trying to down-cast or assume it's initialized. You may just use it as a construct scope.
        '''
        ...

    @jsii.member(jsii_name="bindToResource")
    def bind_to_resource(
        self,
        _resource: _aws_cdk_core_f4b25747.CfnResource,
        *,
        resource_property: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Called after the CFN function resource has been created to allow the code class to bind to it.

        Specifically it's required to allow assets to add
        metadata for tooling like SAM CLI to be able to find their origins.

        :param _resource: -
        :param resource_property: The name of the CloudFormation property to annotate with asset metadata. Default: Code
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a4642662a851596d1739409a53abe0cc707ff06b3a40395e55943677bcfc660c)
            check_type(argname="argument _resource", value=_resource, expected_type=type_hints["_resource"])
        _options = ResourceBindOptions(resource_property=resource_property)

        return typing.cast(None, jsii.invoke(self, "bindToResource", [_resource, _options]))

    @builtins.property
    @jsii.member(jsii_name="isInline")
    @abc.abstractmethod
    def is_inline(self) -> builtins.bool:
        '''(deprecated) Determines whether this Code is inline code or not.

        :deprecated:

        this value is ignored since inline is now determined based on the
        the ``inlineCode`` field of ``CodeConfig`` returned from ``bind()``.

        :stability: deprecated
        '''
        ...


class _CodeProxy(Code):
    @jsii.member(jsii_name="bind")
    def bind(self, scope: _aws_cdk_core_f4b25747.Construct) -> "CodeConfig":
        '''Called when the lambda or layer is initialized to allow this object to bind to the stack, add resources and have fun.

        :param scope: The binding scope. Don't be smart about trying to down-cast or assume it's initialized. You may just use it as a construct scope.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a4001d9534b5ee2f208056f26005fe2215a473fbe6ff2f3655feb48d4f1b2dc0)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
        return typing.cast("CodeConfig", jsii.invoke(self, "bind", [scope]))

    @builtins.property
    @jsii.member(jsii_name="isInline")
    def is_inline(self) -> builtins.bool:
        '''(deprecated) Determines whether this Code is inline code or not.

        :deprecated:

        this value is ignored since inline is now determined based on the
        the ``inlineCode`` field of ``CodeConfig`` returned from ``bind()``.

        :stability: deprecated
        '''
        return typing.cast(builtins.bool, jsii.get(self, "isInline"))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the abstract class
typing.cast(typing.Any, Code).__jsii_proxy_class__ = lambda : _CodeProxy


@jsii.data_type(
    jsii_type="@aws-cdk/aws-lambda.CodeConfig",
    jsii_struct_bases=[],
    name_mapping={
        "image": "image",
        "inline_code": "inlineCode",
        "s3_location": "s3Location",
    },
)
class CodeConfig:
    def __init__(
        self,
        *,
        image: typing.Optional[typing.Union["CodeImageConfig", typing.Dict[builtins.str, typing.Any]]] = None,
        inline_code: typing.Optional[builtins.str] = None,
        s3_location: typing.Optional[typing.Union[_aws_cdk_aws_s3_55f001a5.Location, typing.Dict[builtins.str, typing.Any]]] = None,
    ) -> None:
        '''Result of binding ``Code`` into a ``Function``.

        :param image: Docker image configuration (mutually exclusive with ``s3Location`` and ``inlineCode``). Default: - code is not an ECR container image
        :param inline_code: Inline code (mutually exclusive with ``s3Location`` and ``image``). Default: - code is not inline code
        :param s3_location: The location of the code in S3 (mutually exclusive with ``inlineCode`` and ``image``). Default: - code is not an s3 location

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_lambda as lambda_
            
            code_config = lambda.CodeConfig(
                image=lambda.CodeImageConfig(
                    image_uri="imageUri",
            
                    # the properties below are optional
                    cmd=["cmd"],
                    entrypoint=["entrypoint"],
                    working_directory="workingDirectory"
                ),
                inline_code="inlineCode",
                s3_location=Location(
                    bucket_name="bucketName",
                    object_key="objectKey",
            
                    # the properties below are optional
                    object_version="objectVersion"
                )
            )
        '''
        if isinstance(image, dict):
            image = CodeImageConfig(**image)
        if isinstance(s3_location, dict):
            s3_location = _aws_cdk_aws_s3_55f001a5.Location(**s3_location)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__1d312039dc0380cbf0ea8a1f86e3bc3d6c4630b18a1a14cfb43c47785e80cf76)
            check_type(argname="argument image", value=image, expected_type=type_hints["image"])
            check_type(argname="argument inline_code", value=inline_code, expected_type=type_hints["inline_code"])
            check_type(argname="argument s3_location", value=s3_location, expected_type=type_hints["s3_location"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if image is not None:
            self._values["image"] = image
        if inline_code is not None:
            self._values["inline_code"] = inline_code
        if s3_location is not None:
            self._values["s3_location"] = s3_location

    @builtins.property
    def image(self) -> typing.Optional["CodeImageConfig"]:
        '''Docker image configuration (mutually exclusive with ``s3Location`` and ``inlineCode``).

        :default: - code is not an ECR container image
        '''
        result = self._values.get("image")
        return typing.cast(typing.Optional["CodeImageConfig"], result)

    @builtins.property
    def inline_code(self) -> typing.Optional[builtins.str]:
        '''Inline code (mutually exclusive with ``s3Location`` and ``image``).

        :default: - code is not inline code
        '''
        result = self._values.get("inline_code")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def s3_location(self) -> typing.Optional[_aws_cdk_aws_s3_55f001a5.Location]:
        '''The location of the code in S3 (mutually exclusive with ``inlineCode`` and ``image``).

        :default: - code is not an s3 location
        '''
        result = self._values.get("s3_location")
        return typing.cast(typing.Optional[_aws_cdk_aws_s3_55f001a5.Location], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CodeConfig(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-lambda.CodeImageConfig",
    jsii_struct_bases=[],
    name_mapping={
        "image_uri": "imageUri",
        "cmd": "cmd",
        "entrypoint": "entrypoint",
        "working_directory": "workingDirectory",
    },
)
class CodeImageConfig:
    def __init__(
        self,
        *,
        image_uri: builtins.str,
        cmd: typing.Optional[typing.Sequence[builtins.str]] = None,
        entrypoint: typing.Optional[typing.Sequence[builtins.str]] = None,
        working_directory: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Result of the bind when an ECR image is used.

        :param image_uri: URI to the Docker image.
        :param cmd: Specify or override the CMD on the specified Docker image or Dockerfile. This needs to be in the 'exec form', viz., ``[ 'executable', 'param1', 'param2' ]``. Default: - use the CMD specified in the docker image or Dockerfile.
        :param entrypoint: Specify or override the ENTRYPOINT on the specified Docker image or Dockerfile. An ENTRYPOINT allows you to configure a container that will run as an executable. This needs to be in the 'exec form', viz., ``[ 'executable', 'param1', 'param2' ]``. Default: - use the ENTRYPOINT in the docker image or Dockerfile.
        :param working_directory: Specify or override the WORKDIR on the specified Docker image or Dockerfile. A WORKDIR allows you to configure the working directory the container will use. Default: - use the WORKDIR in the docker image or Dockerfile.

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_lambda as lambda_
            
            code_image_config = lambda.CodeImageConfig(
                image_uri="imageUri",
            
                # the properties below are optional
                cmd=["cmd"],
                entrypoint=["entrypoint"],
                working_directory="workingDirectory"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c7f786d81f42cb75b17e6ae2c05244433deb0e68d01f9e44299a2255e4d1c7bd)
            check_type(argname="argument image_uri", value=image_uri, expected_type=type_hints["image_uri"])
            check_type(argname="argument cmd", value=cmd, expected_type=type_hints["cmd"])
            check_type(argname="argument entrypoint", value=entrypoint, expected_type=type_hints["entrypoint"])
            check_type(argname="argument working_directory", value=working_directory, expected_type=type_hints["working_directory"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "image_uri": image_uri,
        }
        if cmd is not None:
            self._values["cmd"] = cmd
        if entrypoint is not None:
            self._values["entrypoint"] = entrypoint
        if working_directory is not None:
            self._values["working_directory"] = working_directory

    @builtins.property
    def image_uri(self) -> builtins.str:
        '''URI to the Docker image.'''
        result = self._values.get("image_uri")
        assert result is not None, "Required property 'image_uri' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def cmd(self) -> typing.Optional[typing.List[builtins.str]]:
        '''Specify or override the CMD on the specified Docker image or Dockerfile.

        This needs to be in the 'exec form', viz., ``[ 'executable', 'param1', 'param2' ]``.

        :default: - use the CMD specified in the docker image or Dockerfile.

        :see: https://docs.docker.com/engine/reference/builder/#cmd
        '''
        result = self._values.get("cmd")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def entrypoint(self) -> typing.Optional[typing.List[builtins.str]]:
        '''Specify or override the ENTRYPOINT on the specified Docker image or Dockerfile.

        An ENTRYPOINT allows you to configure a container that will run as an executable.
        This needs to be in the 'exec form', viz., ``[ 'executable', 'param1', 'param2' ]``.

        :default: - use the ENTRYPOINT in the docker image or Dockerfile.

        :see: https://docs.docker.com/engine/reference/builder/#entrypoint
        '''
        result = self._values.get("entrypoint")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def working_directory(self) -> typing.Optional[builtins.str]:
        '''Specify or override the WORKDIR on the specified Docker image or Dockerfile.

        A WORKDIR allows you to configure the working directory the container will use.

        :default: - use the WORKDIR in the docker image or Dockerfile.

        :see: https://docs.docker.com/engine/reference/builder/#workdir
        '''
        result = self._values.get("working_directory")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CodeImageConfig(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-lambda.CodeSigningConfigProps",
    jsii_struct_bases=[],
    name_mapping={
        "signing_profiles": "signingProfiles",
        "description": "description",
        "untrusted_artifact_on_deployment": "untrustedArtifactOnDeployment",
    },
)
class CodeSigningConfigProps:
    def __init__(
        self,
        *,
        signing_profiles: typing.Sequence[_aws_cdk_aws_signer_a6bf19cf.ISigningProfile],
        description: typing.Optional[builtins.str] = None,
        untrusted_artifact_on_deployment: typing.Optional["UntrustedArtifactOnDeployment"] = None,
    ) -> None:
        '''Construction properties for a Code Signing Config object.

        :param signing_profiles: List of signing profiles that defines a trusted user who can sign a code package.
        :param description: Code signing configuration description. Default: - No description.
        :param untrusted_artifact_on_deployment: Code signing configuration policy for deployment validation failure. If you set the policy to Enforce, Lambda blocks the deployment request if signature validation checks fail. If you set the policy to Warn, Lambda allows the deployment and creates a CloudWatch log. Default: UntrustedArtifactOnDeployment.WARN

        :exampleMetadata: infused

        Example::

            import aws_cdk.aws_signer as signer
            
            
            signing_profile = signer.SigningProfile(self, "SigningProfile",
                platform=signer.Platform.AWS_LAMBDA_SHA384_ECDSA
            )
            
            code_signing_config = lambda_.CodeSigningConfig(self, "CodeSigningConfig",
                signing_profiles=[signing_profile]
            )
            
            lambda_.Function(self, "Function",
                code_signing_config=code_signing_config,
                runtime=lambda_.Runtime.NODEJS_16_X,
                handler="index.handler",
                code=lambda_.Code.from_asset(path.join(__dirname, "lambda-handler"))
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3afc101f80278c21783c9bb5919cc06529f04144a607a10882b08795b3942c3b)
            check_type(argname="argument signing_profiles", value=signing_profiles, expected_type=type_hints["signing_profiles"])
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument untrusted_artifact_on_deployment", value=untrusted_artifact_on_deployment, expected_type=type_hints["untrusted_artifact_on_deployment"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "signing_profiles": signing_profiles,
        }
        if description is not None:
            self._values["description"] = description
        if untrusted_artifact_on_deployment is not None:
            self._values["untrusted_artifact_on_deployment"] = untrusted_artifact_on_deployment

    @builtins.property
    def signing_profiles(
        self,
    ) -> typing.List[_aws_cdk_aws_signer_a6bf19cf.ISigningProfile]:
        '''List of signing profiles that defines a trusted user who can sign a code package.'''
        result = self._values.get("signing_profiles")
        assert result is not None, "Required property 'signing_profiles' is missing"
        return typing.cast(typing.List[_aws_cdk_aws_signer_a6bf19cf.ISigningProfile], result)

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''Code signing configuration description.

        :default: - No description.
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def untrusted_artifact_on_deployment(
        self,
    ) -> typing.Optional["UntrustedArtifactOnDeployment"]:
        '''Code signing configuration policy for deployment validation failure.

        If you set the policy to Enforce, Lambda blocks the deployment request
        if signature validation checks fail.
        If you set the policy to Warn, Lambda allows the deployment and
        creates a CloudWatch log.

        :default: UntrustedArtifactOnDeployment.WARN
        '''
        result = self._values.get("untrusted_artifact_on_deployment")
        return typing.cast(typing.Optional["UntrustedArtifactOnDeployment"], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CodeSigningConfigProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-lambda.DestinationConfig",
    jsii_struct_bases=[],
    name_mapping={"destination": "destination"},
)
class DestinationConfig:
    def __init__(self, *, destination: builtins.str) -> None:
        '''A destination configuration.

        :param destination: The Amazon Resource Name (ARN) of the destination resource.

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_lambda as lambda_
            
            destination_config = lambda.DestinationConfig(
                destination="destination"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__361970191555daa10880f2db6b7f72fbc1c72cc3e2a59c6e6a7197ecfa05619a)
            check_type(argname="argument destination", value=destination, expected_type=type_hints["destination"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "destination": destination,
        }

    @builtins.property
    def destination(self) -> builtins.str:
        '''The Amazon Resource Name (ARN) of the destination resource.'''
        result = self._values.get("destination")
        assert result is not None, "Required property 'destination' is missing"
        return typing.cast(builtins.str, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "DestinationConfig(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-lambda.DestinationOptions",
    jsii_struct_bases=[],
    name_mapping={"type": "type"},
)
class DestinationOptions:
    def __init__(self, *, type: "DestinationType") -> None:
        '''Options when binding a destination to a function.

        :param type: The destination type.

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_lambda as lambda_
            
            destination_options = lambda.DestinationOptions(
                type=lambda_.DestinationType.FAILURE
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__67355057d7700f3583da5cd957dd948144b3bf554d1de1f539064873ca5c78c0)
            check_type(argname="argument type", value=type, expected_type=type_hints["type"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "type": type,
        }

    @builtins.property
    def type(self) -> "DestinationType":
        '''The destination type.'''
        result = self._values.get("type")
        assert result is not None, "Required property 'type' is missing"
        return typing.cast("DestinationType", result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "DestinationOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="@aws-cdk/aws-lambda.DestinationType")
class DestinationType(enum.Enum):
    '''The type of destination.'''

    FAILURE = "FAILURE"
    '''Failure.'''
    SUCCESS = "SUCCESS"
    '''Success.'''


@jsii.data_type(
    jsii_type="@aws-cdk/aws-lambda.DlqDestinationConfig",
    jsii_struct_bases=[],
    name_mapping={"destination": "destination"},
)
class DlqDestinationConfig:
    def __init__(self, *, destination: builtins.str) -> None:
        '''A destination configuration.

        :param destination: The Amazon Resource Name (ARN) of the destination resource.

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_lambda as lambda_
            
            dlq_destination_config = lambda.DlqDestinationConfig(
                destination="destination"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9a5b6a9fb4fdfba2cd08f93328fdc05ec67bd5e43db59c1401c177c45de0ff12)
            check_type(argname="argument destination", value=destination, expected_type=type_hints["destination"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "destination": destination,
        }

    @builtins.property
    def destination(self) -> builtins.str:
        '''The Amazon Resource Name (ARN) of the destination resource.'''
        result = self._values.get("destination")
        assert result is not None, "Required property 'destination' is missing"
        return typing.cast(builtins.str, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "DlqDestinationConfig(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-lambda.DockerBuildAssetOptions",
    jsii_struct_bases=[_aws_cdk_core_f4b25747.DockerBuildOptions],
    name_mapping={
        "build_args": "buildArgs",
        "file": "file",
        "platform": "platform",
        "image_path": "imagePath",
        "output_path": "outputPath",
    },
)
class DockerBuildAssetOptions(_aws_cdk_core_f4b25747.DockerBuildOptions):
    def __init__(
        self,
        *,
        build_args: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        file: typing.Optional[builtins.str] = None,
        platform: typing.Optional[builtins.str] = None,
        image_path: typing.Optional[builtins.str] = None,
        output_path: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Options when creating an asset from a Docker build.

        :param build_args: Build args. Default: - no build args
        :param file: Name of the Dockerfile, must relative to the docker build path. Default: ``Dockerfile``
        :param platform: Set platform if server is multi-platform capable. *Requires Docker Engine API v1.38+*. Example value: ``linux/amd64`` Default: - no platform specified
        :param image_path: The path in the Docker image where the asset is located after the build operation. Default: /asset
        :param output_path: The path on the local filesystem where the asset will be copied using ``docker cp``. Default: - a unique temporary directory in the system temp directory

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_lambda as lambda_
            
            docker_build_asset_options = lambda.DockerBuildAssetOptions(
                build_args={
                    "build_args_key": "buildArgs"
                },
                file="file",
                image_path="imagePath",
                output_path="outputPath",
                platform="platform"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d7f02dc145ca600f1af867a55cd0506f7869b1140e022dad6864d63b5559701f)
            check_type(argname="argument build_args", value=build_args, expected_type=type_hints["build_args"])
            check_type(argname="argument file", value=file, expected_type=type_hints["file"])
            check_type(argname="argument platform", value=platform, expected_type=type_hints["platform"])
            check_type(argname="argument image_path", value=image_path, expected_type=type_hints["image_path"])
            check_type(argname="argument output_path", value=output_path, expected_type=type_hints["output_path"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if build_args is not None:
            self._values["build_args"] = build_args
        if file is not None:
            self._values["file"] = file
        if platform is not None:
            self._values["platform"] = platform
        if image_path is not None:
            self._values["image_path"] = image_path
        if output_path is not None:
            self._values["output_path"] = output_path

    @builtins.property
    def build_args(self) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''Build args.

        :default: - no build args
        '''
        result = self._values.get("build_args")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    @builtins.property
    def file(self) -> typing.Optional[builtins.str]:
        '''Name of the Dockerfile, must relative to the docker build path.

        :default: ``Dockerfile``
        '''
        result = self._values.get("file")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def platform(self) -> typing.Optional[builtins.str]:
        '''Set platform if server is multi-platform capable. *Requires Docker Engine API v1.38+*.

        Example value: ``linux/amd64``

        :default: - no platform specified
        '''
        result = self._values.get("platform")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def image_path(self) -> typing.Optional[builtins.str]:
        '''The path in the Docker image where the asset is located after the build operation.

        :default: /asset
        '''
        result = self._values.get("image_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def output_path(self) -> typing.Optional[builtins.str]:
        '''The path on the local filesystem where the asset will be copied using ``docker cp``.

        :default: - a unique temporary directory in the system temp directory
        '''
        result = self._values.get("output_path")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "DockerBuildAssetOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class DockerImageCode(
    metaclass=jsii.JSIIAbstractClass,
    jsii_type="@aws-cdk/aws-lambda.DockerImageCode",
):
    '''Code property for the DockerImageFunction construct.

    :exampleMetadata: infused

    Example::

        lambda_.DockerImageFunction(self, "AssetFunction",
            code=lambda_.DockerImageCode.from_image_asset(path.join(__dirname, "docker-handler"))
        )
    '''

    def __init__(self) -> None:
        jsii.create(self.__class__, self, [])

    @jsii.member(jsii_name="fromEcr")
    @builtins.classmethod
    def from_ecr(
        cls,
        repository: _aws_cdk_aws_ecr_093ed842.IRepository,
        *,
        cmd: typing.Optional[typing.Sequence[builtins.str]] = None,
        entrypoint: typing.Optional[typing.Sequence[builtins.str]] = None,
        tag: typing.Optional[builtins.str] = None,
        tag_or_digest: typing.Optional[builtins.str] = None,
        working_directory: typing.Optional[builtins.str] = None,
    ) -> "DockerImageCode":
        '''Use an existing ECR image as the Lambda code.

        :param repository: the ECR repository that the image is in.
        :param cmd: Specify or override the CMD on the specified Docker image or Dockerfile. This needs to be in the 'exec form', viz., ``[ 'executable', 'param1', 'param2' ]``. Default: - use the CMD specified in the docker image or Dockerfile.
        :param entrypoint: Specify or override the ENTRYPOINT on the specified Docker image or Dockerfile. An ENTRYPOINT allows you to configure a container that will run as an executable. This needs to be in the 'exec form', viz., ``[ 'executable', 'param1', 'param2' ]``. Default: - use the ENTRYPOINT in the docker image or Dockerfile.
        :param tag: (deprecated) The image tag to use when pulling the image from ECR. Default: 'latest'
        :param tag_or_digest: The image tag or digest to use when pulling the image from ECR (digests must start with ``sha256:``). Default: 'latest'
        :param working_directory: Specify or override the WORKDIR on the specified Docker image or Dockerfile. A WORKDIR allows you to configure the working directory the container will use. Default: - use the WORKDIR in the docker image or Dockerfile.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b934a9a91224633634952b03605ff257a2bbedf8ff2f4be5088c896724b3bb41)
            check_type(argname="argument repository", value=repository, expected_type=type_hints["repository"])
        props = EcrImageCodeProps(
            cmd=cmd,
            entrypoint=entrypoint,
            tag=tag,
            tag_or_digest=tag_or_digest,
            working_directory=working_directory,
        )

        return typing.cast("DockerImageCode", jsii.sinvoke(cls, "fromEcr", [repository, props]))

    @jsii.member(jsii_name="fromImageAsset")
    @builtins.classmethod
    def from_image_asset(
        cls,
        directory: builtins.str,
        *,
        cmd: typing.Optional[typing.Sequence[builtins.str]] = None,
        entrypoint: typing.Optional[typing.Sequence[builtins.str]] = None,
        working_directory: typing.Optional[builtins.str] = None,
        build_args: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        file: typing.Optional[builtins.str] = None,
        invalidation: typing.Optional[typing.Union[_aws_cdk_aws_ecr_assets_ef09b699.DockerImageAssetInvalidationOptions, typing.Dict[builtins.str, typing.Any]]] = None,
        network_mode: typing.Optional[_aws_cdk_aws_ecr_assets_ef09b699.NetworkMode] = None,
        platform: typing.Optional[_aws_cdk_aws_ecr_assets_ef09b699.Platform] = None,
        repository_name: typing.Optional[builtins.str] = None,
        target: typing.Optional[builtins.str] = None,
        extra_hash: typing.Optional[builtins.str] = None,
        exclude: typing.Optional[typing.Sequence[builtins.str]] = None,
        follow: typing.Optional[_aws_cdk_assets_b1c45fb6.FollowMode] = None,
        ignore_mode: typing.Optional[_aws_cdk_core_f4b25747.IgnoreMode] = None,
        follow_symlinks: typing.Optional[_aws_cdk_core_f4b25747.SymlinkFollowMode] = None,
    ) -> "DockerImageCode":
        '''Create an ECR image from the specified asset and bind it as the Lambda code.

        :param directory: the directory from which the asset must be created.
        :param cmd: Specify or override the CMD on the specified Docker image or Dockerfile. This needs to be in the 'exec form', viz., ``[ 'executable', 'param1', 'param2' ]``. Default: - use the CMD specified in the docker image or Dockerfile.
        :param entrypoint: Specify or override the ENTRYPOINT on the specified Docker image or Dockerfile. An ENTRYPOINT allows you to configure a container that will run as an executable. This needs to be in the 'exec form', viz., ``[ 'executable', 'param1', 'param2' ]``. Default: - use the ENTRYPOINT in the docker image or Dockerfile.
        :param working_directory: Specify or override the WORKDIR on the specified Docker image or Dockerfile. A WORKDIR allows you to configure the working directory the container will use. Default: - use the WORKDIR in the docker image or Dockerfile.
        :param build_args: Build args to pass to the ``docker build`` command. Since Docker build arguments are resolved before deployment, keys and values cannot refer to unresolved tokens (such as ``lambda.functionArn`` or ``queue.queueUrl``). Default: - no build args are passed
        :param file: Path to the Dockerfile (relative to the directory). Default: 'Dockerfile'
        :param invalidation: Options to control which parameters are used to invalidate the asset hash. Default: - hash all parameters
        :param network_mode: Networking mode for the RUN commands during build. Support docker API 1.25+. Default: - no networking mode specified (the default networking mode ``NetworkMode.DEFAULT`` will be used)
        :param platform: Platform to build for. *Requires Docker Buildx*. Default: - no platform specified (the current machine architecture will be used)
        :param repository_name: (deprecated) ECR repository name. Specify this property if you need to statically address the image, e.g. from a Kubernetes Pod. Note, this is only the repository name, without the registry and the tag parts. Default: - the default ECR repository for CDK assets
        :param target: Docker target to build to. Default: - no target
        :param extra_hash: (deprecated) Extra information to encode into the fingerprint (e.g. build instructions and other inputs). Default: - hash is only based on source content
        :param exclude: (deprecated) Glob patterns to exclude from the copy. Default: nothing is excluded
        :param follow: (deprecated) A strategy for how to handle symlinks. Default: Never
        :param ignore_mode: (deprecated) The ignore behavior to use for exclude patterns. Default: - GLOB for file assets, DOCKER or GLOB for docker assets depending on whether the '
        :param follow_symlinks: A strategy for how to handle symlinks. Default: SymlinkFollowMode.NEVER
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__32c23524a98c68e71a0a3a02379348fca13dbdc61fa1f74a00148a590accb1f1)
            check_type(argname="argument directory", value=directory, expected_type=type_hints["directory"])
        props = AssetImageCodeProps(
            cmd=cmd,
            entrypoint=entrypoint,
            working_directory=working_directory,
            build_args=build_args,
            file=file,
            invalidation=invalidation,
            network_mode=network_mode,
            platform=platform,
            repository_name=repository_name,
            target=target,
            extra_hash=extra_hash,
            exclude=exclude,
            follow=follow,
            ignore_mode=ignore_mode,
            follow_symlinks=follow_symlinks,
        )

        return typing.cast("DockerImageCode", jsii.sinvoke(cls, "fromImageAsset", [directory, props]))


class _DockerImageCodeProxy(DockerImageCode):
    pass

# Adding a "__jsii_proxy_class__(): typing.Type" function to the abstract class
typing.cast(typing.Any, DockerImageCode).__jsii_proxy_class__ = lambda : _DockerImageCodeProxy


class EcrImageCode(
    Code,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-lambda.EcrImageCode",
):
    '''Represents a Docker image in ECR that can be bound as Lambda Code.

    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        import aws_cdk.aws_ecr as ecr
        import aws_cdk.aws_lambda as lambda_
        
        # repository: ecr.Repository
        
        ecr_image_code = lambda_.EcrImageCode(repository,
            cmd=["cmd"],
            entrypoint=["entrypoint"],
            tag="tag",
            tag_or_digest="tagOrDigest",
            working_directory="workingDirectory"
        )
    '''

    def __init__(
        self,
        repository: _aws_cdk_aws_ecr_093ed842.IRepository,
        *,
        cmd: typing.Optional[typing.Sequence[builtins.str]] = None,
        entrypoint: typing.Optional[typing.Sequence[builtins.str]] = None,
        tag: typing.Optional[builtins.str] = None,
        tag_or_digest: typing.Optional[builtins.str] = None,
        working_directory: typing.Optional[builtins.str] = None,
    ) -> None:
        '''
        :param repository: -
        :param cmd: Specify or override the CMD on the specified Docker image or Dockerfile. This needs to be in the 'exec form', viz., ``[ 'executable', 'param1', 'param2' ]``. Default: - use the CMD specified in the docker image or Dockerfile.
        :param entrypoint: Specify or override the ENTRYPOINT on the specified Docker image or Dockerfile. An ENTRYPOINT allows you to configure a container that will run as an executable. This needs to be in the 'exec form', viz., ``[ 'executable', 'param1', 'param2' ]``. Default: - use the ENTRYPOINT in the docker image or Dockerfile.
        :param tag: (deprecated) The image tag to use when pulling the image from ECR. Default: 'latest'
        :param tag_or_digest: The image tag or digest to use when pulling the image from ECR (digests must start with ``sha256:``). Default: 'latest'
        :param working_directory: Specify or override the WORKDIR on the specified Docker image or Dockerfile. A WORKDIR allows you to configure the working directory the container will use. Default: - use the WORKDIR in the docker image or Dockerfile.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__cd0cf7e2febb1bab142292bd9345ac8c82c1acb08444f249fbf8e818e4c3963c)
            check_type(argname="argument repository", value=repository, expected_type=type_hints["repository"])
        props = EcrImageCodeProps(
            cmd=cmd,
            entrypoint=entrypoint,
            tag=tag,
            tag_or_digest=tag_or_digest,
            working_directory=working_directory,
        )

        jsii.create(self.__class__, self, [repository, props])

    @jsii.member(jsii_name="bind")
    def bind(self, _: _aws_cdk_core_f4b25747.Construct) -> CodeConfig:
        '''Called when the lambda or layer is initialized to allow this object to bind to the stack, add resources and have fun.

        :param _: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__05bcdf06a8df5283e5c6b7a5d247e4ac79fe732311e0760316d1665a50274c2f)
            check_type(argname="argument _", value=_, expected_type=type_hints["_"])
        return typing.cast(CodeConfig, jsii.invoke(self, "bind", [_]))

    @builtins.property
    @jsii.member(jsii_name="isInline")
    def is_inline(self) -> builtins.bool:
        '''Determines whether this Code is inline code or not.'''
        return typing.cast(builtins.bool, jsii.get(self, "isInline"))


@jsii.data_type(
    jsii_type="@aws-cdk/aws-lambda.EcrImageCodeProps",
    jsii_struct_bases=[],
    name_mapping={
        "cmd": "cmd",
        "entrypoint": "entrypoint",
        "tag": "tag",
        "tag_or_digest": "tagOrDigest",
        "working_directory": "workingDirectory",
    },
)
class EcrImageCodeProps:
    def __init__(
        self,
        *,
        cmd: typing.Optional[typing.Sequence[builtins.str]] = None,
        entrypoint: typing.Optional[typing.Sequence[builtins.str]] = None,
        tag: typing.Optional[builtins.str] = None,
        tag_or_digest: typing.Optional[builtins.str] = None,
        working_directory: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties to initialize a new EcrImageCode.

        :param cmd: Specify or override the CMD on the specified Docker image or Dockerfile. This needs to be in the 'exec form', viz., ``[ 'executable', 'param1', 'param2' ]``. Default: - use the CMD specified in the docker image or Dockerfile.
        :param entrypoint: Specify or override the ENTRYPOINT on the specified Docker image or Dockerfile. An ENTRYPOINT allows you to configure a container that will run as an executable. This needs to be in the 'exec form', viz., ``[ 'executable', 'param1', 'param2' ]``. Default: - use the ENTRYPOINT in the docker image or Dockerfile.
        :param tag: (deprecated) The image tag to use when pulling the image from ECR. Default: 'latest'
        :param tag_or_digest: The image tag or digest to use when pulling the image from ECR (digests must start with ``sha256:``). Default: 'latest'
        :param working_directory: Specify or override the WORKDIR on the specified Docker image or Dockerfile. A WORKDIR allows you to configure the working directory the container will use. Default: - use the WORKDIR in the docker image or Dockerfile.

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_lambda as lambda_
            
            ecr_image_code_props = lambda.EcrImageCodeProps(
                cmd=["cmd"],
                entrypoint=["entrypoint"],
                tag="tag",
                tag_or_digest="tagOrDigest",
                working_directory="workingDirectory"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3c7bd27a9d27a9103b54933a26bf9593a3dbe43b3e51ff3814d2e2cd12a6d78f)
            check_type(argname="argument cmd", value=cmd, expected_type=type_hints["cmd"])
            check_type(argname="argument entrypoint", value=entrypoint, expected_type=type_hints["entrypoint"])
            check_type(argname="argument tag", value=tag, expected_type=type_hints["tag"])
            check_type(argname="argument tag_or_digest", value=tag_or_digest, expected_type=type_hints["tag_or_digest"])
            check_type(argname="argument working_directory", value=working_directory, expected_type=type_hints["working_directory"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if cmd is not None:
            self._values["cmd"] = cmd
        if entrypoint is not None:
            self._values["entrypoint"] = entrypoint
        if tag is not None:
            self._values["tag"] = tag
        if tag_or_digest is not None:
            self._values["tag_or_digest"] = tag_or_digest
        if working_directory is not None:
            self._values["working_directory"] = working_directory

    @builtins.property
    def cmd(self) -> typing.Optional[typing.List[builtins.str]]:
        '''Specify or override the CMD on the specified Docker image or Dockerfile.

        This needs to be in the 'exec form', viz., ``[ 'executable', 'param1', 'param2' ]``.

        :default: - use the CMD specified in the docker image or Dockerfile.

        :see: https://docs.docker.com/engine/reference/builder/#cmd
        '''
        result = self._values.get("cmd")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def entrypoint(self) -> typing.Optional[typing.List[builtins.str]]:
        '''Specify or override the ENTRYPOINT on the specified Docker image or Dockerfile.

        An ENTRYPOINT allows you to configure a container that will run as an executable.
        This needs to be in the 'exec form', viz., ``[ 'executable', 'param1', 'param2' ]``.

        :default: - use the ENTRYPOINT in the docker image or Dockerfile.

        :see: https://docs.docker.com/engine/reference/builder/#entrypoint
        '''
        result = self._values.get("entrypoint")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def tag(self) -> typing.Optional[builtins.str]:
        '''(deprecated) The image tag to use when pulling the image from ECR.

        :default: 'latest'

        :deprecated: use ``tagOrDigest``

        :stability: deprecated
        '''
        result = self._values.get("tag")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def tag_or_digest(self) -> typing.Optional[builtins.str]:
        '''The image tag or digest to use when pulling the image from ECR (digests must start with ``sha256:``).

        :default: 'latest'
        '''
        result = self._values.get("tag_or_digest")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def working_directory(self) -> typing.Optional[builtins.str]:
        '''Specify or override the WORKDIR on the specified Docker image or Dockerfile.

        A WORKDIR allows you to configure the working directory the container will use.

        :default: - use the WORKDIR in the docker image or Dockerfile.

        :see: https://docs.docker.com/engine/reference/builder/#workdir
        '''
        result = self._values.get("working_directory")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "EcrImageCodeProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-lambda.EnvironmentOptions",
    jsii_struct_bases=[],
    name_mapping={"remove_in_edge": "removeInEdge"},
)
class EnvironmentOptions:
    def __init__(
        self,
        *,
        remove_in_edge: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''Environment variables options.

        :param remove_in_edge: When used in Lambda@Edge via edgeArn() API, these environment variables will be removed. If not set, an error will be thrown. Default: false - using the function in Lambda

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_lambda as lambda_
            
            environment_options = lambda.EnvironmentOptions(
                remove_in_edge=False
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d5e9a848ef2c0608a317e4bc1de7e0058efcd0ee573787a813aee68f3097b69e)
            check_type(argname="argument remove_in_edge", value=remove_in_edge, expected_type=type_hints["remove_in_edge"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if remove_in_edge is not None:
            self._values["remove_in_edge"] = remove_in_edge

    @builtins.property
    def remove_in_edge(self) -> typing.Optional[builtins.bool]:
        '''When used in Lambda@Edge via edgeArn() API, these environment variables will be removed.

        If not set, an error will be thrown.

        :default: false - using the function in Lambda

        :see: https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/lambda-requirements-limits.html#lambda-requirements-lambda-function-configuration
        :Edge: will throw
        '''
        result = self._values.get("remove_in_edge")
        return typing.cast(typing.Optional[builtins.bool], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "EnvironmentOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class EventInvokeConfig(
    _aws_cdk_core_f4b25747.Resource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-lambda.EventInvokeConfig",
):
    '''Configure options for asynchronous invocation on a version or an alias.

    By default, Lambda retries an asynchronous invocation twice if the function
    returns an error. It retains events in a queue for up to six hours. When an
    event fails all processing attempts or stays in the asynchronous invocation
    queue for too long, Lambda discards it.

    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        import aws_cdk.aws_lambda as lambda_
        import aws_cdk.core as cdk
        
        # destination: lambda.IDestination
        # function_: lambda.Function
        
        event_invoke_config = lambda_.EventInvokeConfig(self, "MyEventInvokeConfig",
            function=function_,
        
            # the properties below are optional
            max_event_age=cdk.Duration.minutes(30),
            on_failure=destination,
            on_success=destination,
            qualifier="qualifier",
            retry_attempts=123
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        function: "IFunction",
        qualifier: typing.Optional[builtins.str] = None,
        max_event_age: typing.Optional[_aws_cdk_core_f4b25747.Duration] = None,
        on_failure: typing.Optional["IDestination"] = None,
        on_success: typing.Optional["IDestination"] = None,
        retry_attempts: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param function: The Lambda function.
        :param qualifier: The qualifier. Default: - latest version
        :param max_event_age: The maximum age of a request that Lambda sends to a function for processing. Minimum: 60 seconds Maximum: 6 hours Default: Duration.hours(6)
        :param on_failure: The destination for failed invocations. Default: - no destination
        :param on_success: The destination for successful invocations. Default: - no destination
        :param retry_attempts: The maximum number of times to retry when the function returns an error. Minimum: 0 Maximum: 2 Default: 2
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c906f99927c57f80476b0667bd67632d75ea33950c7b27c1b9b1ca1e9a637351)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = EventInvokeConfigProps(
            function=function,
            qualifier=qualifier,
            max_event_age=max_event_age,
            on_failure=on_failure,
            on_success=on_success,
            retry_attempts=retry_attempts,
        )

        jsii.create(self.__class__, self, [scope, id, props])


@jsii.data_type(
    jsii_type="@aws-cdk/aws-lambda.EventInvokeConfigOptions",
    jsii_struct_bases=[],
    name_mapping={
        "max_event_age": "maxEventAge",
        "on_failure": "onFailure",
        "on_success": "onSuccess",
        "retry_attempts": "retryAttempts",
    },
)
class EventInvokeConfigOptions:
    def __init__(
        self,
        *,
        max_event_age: typing.Optional[_aws_cdk_core_f4b25747.Duration] = None,
        on_failure: typing.Optional["IDestination"] = None,
        on_success: typing.Optional["IDestination"] = None,
        retry_attempts: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''Options to add an EventInvokeConfig to a function.

        :param max_event_age: The maximum age of a request that Lambda sends to a function for processing. Minimum: 60 seconds Maximum: 6 hours Default: Duration.hours(6)
        :param on_failure: The destination for failed invocations. Default: - no destination
        :param on_success: The destination for successful invocations. Default: - no destination
        :param retry_attempts: The maximum number of times to retry when the function returns an error. Minimum: 0 Maximum: 2 Default: 2

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_lambda as lambda_
            import aws_cdk.core as cdk
            
            # destination: lambda.IDestination
            
            event_invoke_config_options = lambda.EventInvokeConfigOptions(
                max_event_age=cdk.Duration.minutes(30),
                on_failure=destination,
                on_success=destination,
                retry_attempts=123
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ab317263c8c8fd21cd4764123af29bf82b0d2ae0717d3f57b2035ff01ee98e80)
            check_type(argname="argument max_event_age", value=max_event_age, expected_type=type_hints["max_event_age"])
            check_type(argname="argument on_failure", value=on_failure, expected_type=type_hints["on_failure"])
            check_type(argname="argument on_success", value=on_success, expected_type=type_hints["on_success"])
            check_type(argname="argument retry_attempts", value=retry_attempts, expected_type=type_hints["retry_attempts"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if max_event_age is not None:
            self._values["max_event_age"] = max_event_age
        if on_failure is not None:
            self._values["on_failure"] = on_failure
        if on_success is not None:
            self._values["on_success"] = on_success
        if retry_attempts is not None:
            self._values["retry_attempts"] = retry_attempts

    @builtins.property
    def max_event_age(self) -> typing.Optional[_aws_cdk_core_f4b25747.Duration]:
        '''The maximum age of a request that Lambda sends to a function for processing.

        Minimum: 60 seconds
        Maximum: 6 hours

        :default: Duration.hours(6)
        '''
        result = self._values.get("max_event_age")
        return typing.cast(typing.Optional[_aws_cdk_core_f4b25747.Duration], result)

    @builtins.property
    def on_failure(self) -> typing.Optional["IDestination"]:
        '''The destination for failed invocations.

        :default: - no destination
        '''
        result = self._values.get("on_failure")
        return typing.cast(typing.Optional["IDestination"], result)

    @builtins.property
    def on_success(self) -> typing.Optional["IDestination"]:
        '''The destination for successful invocations.

        :default: - no destination
        '''
        result = self._values.get("on_success")
        return typing.cast(typing.Optional["IDestination"], result)

    @builtins.property
    def retry_attempts(self) -> typing.Optional[jsii.Number]:
        '''The maximum number of times to retry when the function returns an error.

        Minimum: 0
        Maximum: 2

        :default: 2
        '''
        result = self._values.get("retry_attempts")
        return typing.cast(typing.Optional[jsii.Number], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "EventInvokeConfigOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-lambda.EventInvokeConfigProps",
    jsii_struct_bases=[EventInvokeConfigOptions],
    name_mapping={
        "max_event_age": "maxEventAge",
        "on_failure": "onFailure",
        "on_success": "onSuccess",
        "retry_attempts": "retryAttempts",
        "function": "function",
        "qualifier": "qualifier",
    },
)
class EventInvokeConfigProps(EventInvokeConfigOptions):
    def __init__(
        self,
        *,
        max_event_age: typing.Optional[_aws_cdk_core_f4b25747.Duration] = None,
        on_failure: typing.Optional["IDestination"] = None,
        on_success: typing.Optional["IDestination"] = None,
        retry_attempts: typing.Optional[jsii.Number] = None,
        function: "IFunction",
        qualifier: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for an EventInvokeConfig.

        :param max_event_age: The maximum age of a request that Lambda sends to a function for processing. Minimum: 60 seconds Maximum: 6 hours Default: Duration.hours(6)
        :param on_failure: The destination for failed invocations. Default: - no destination
        :param on_success: The destination for successful invocations. Default: - no destination
        :param retry_attempts: The maximum number of times to retry when the function returns an error. Minimum: 0 Maximum: 2 Default: 2
        :param function: The Lambda function.
        :param qualifier: The qualifier. Default: - latest version

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_lambda as lambda_
            import aws_cdk.core as cdk
            
            # destination: lambda.IDestination
            # function_: lambda.Function
            
            event_invoke_config_props = lambda.EventInvokeConfigProps(
                function=function_,
            
                # the properties below are optional
                max_event_age=cdk.Duration.minutes(30),
                on_failure=destination,
                on_success=destination,
                qualifier="qualifier",
                retry_attempts=123
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f7516aeb19f27588bfd8689a8eafb7df0c80df748a38c85d0f7f3f92b70dc442)
            check_type(argname="argument max_event_age", value=max_event_age, expected_type=type_hints["max_event_age"])
            check_type(argname="argument on_failure", value=on_failure, expected_type=type_hints["on_failure"])
            check_type(argname="argument on_success", value=on_success, expected_type=type_hints["on_success"])
            check_type(argname="argument retry_attempts", value=retry_attempts, expected_type=type_hints["retry_attempts"])
            check_type(argname="argument function", value=function, expected_type=type_hints["function"])
            check_type(argname="argument qualifier", value=qualifier, expected_type=type_hints["qualifier"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "function": function,
        }
        if max_event_age is not None:
            self._values["max_event_age"] = max_event_age
        if on_failure is not None:
            self._values["on_failure"] = on_failure
        if on_success is not None:
            self._values["on_success"] = on_success
        if retry_attempts is not None:
            self._values["retry_attempts"] = retry_attempts
        if qualifier is not None:
            self._values["qualifier"] = qualifier

    @builtins.property
    def max_event_age(self) -> typing.Optional[_aws_cdk_core_f4b25747.Duration]:
        '''The maximum age of a request that Lambda sends to a function for processing.

        Minimum: 60 seconds
        Maximum: 6 hours

        :default: Duration.hours(6)
        '''
        result = self._values.get("max_event_age")
        return typing.cast(typing.Optional[_aws_cdk_core_f4b25747.Duration], result)

    @builtins.property
    def on_failure(self) -> typing.Optional["IDestination"]:
        '''The destination for failed invocations.

        :default: - no destination
        '''
        result = self._values.get("on_failure")
        return typing.cast(typing.Optional["IDestination"], result)

    @builtins.property
    def on_success(self) -> typing.Optional["IDestination"]:
        '''The destination for successful invocations.

        :default: - no destination
        '''
        result = self._values.get("on_success")
        return typing.cast(typing.Optional["IDestination"], result)

    @builtins.property
    def retry_attempts(self) -> typing.Optional[jsii.Number]:
        '''The maximum number of times to retry when the function returns an error.

        Minimum: 0
        Maximum: 2

        :default: 2
        '''
        result = self._values.get("retry_attempts")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def function(self) -> "IFunction":
        '''The Lambda function.'''
        result = self._values.get("function")
        assert result is not None, "Required property 'function' is missing"
        return typing.cast("IFunction", result)

    @builtins.property
    def qualifier(self) -> typing.Optional[builtins.str]:
        '''The qualifier.

        :default: - latest version
        '''
        result = self._values.get("qualifier")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "EventInvokeConfigProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-lambda.EventSourceMappingOptions",
    jsii_struct_bases=[],
    name_mapping={
        "batch_size": "batchSize",
        "bisect_batch_on_error": "bisectBatchOnError",
        "enabled": "enabled",
        "event_source_arn": "eventSourceArn",
        "kafka_bootstrap_servers": "kafkaBootstrapServers",
        "kafka_topic": "kafkaTopic",
        "max_batching_window": "maxBatchingWindow",
        "max_record_age": "maxRecordAge",
        "on_failure": "onFailure",
        "parallelization_factor": "parallelizationFactor",
        "report_batch_item_failures": "reportBatchItemFailures",
        "retry_attempts": "retryAttempts",
        "source_access_configurations": "sourceAccessConfigurations",
        "starting_position": "startingPosition",
        "tumbling_window": "tumblingWindow",
    },
)
class EventSourceMappingOptions:
    def __init__(
        self,
        *,
        batch_size: typing.Optional[jsii.Number] = None,
        bisect_batch_on_error: typing.Optional[builtins.bool] = None,
        enabled: typing.Optional[builtins.bool] = None,
        event_source_arn: typing.Optional[builtins.str] = None,
        kafka_bootstrap_servers: typing.Optional[typing.Sequence[builtins.str]] = None,
        kafka_topic: typing.Optional[builtins.str] = None,
        max_batching_window: typing.Optional[_aws_cdk_core_f4b25747.Duration] = None,
        max_record_age: typing.Optional[_aws_cdk_core_f4b25747.Duration] = None,
        on_failure: typing.Optional["IEventSourceDlq"] = None,
        parallelization_factor: typing.Optional[jsii.Number] = None,
        report_batch_item_failures: typing.Optional[builtins.bool] = None,
        retry_attempts: typing.Optional[jsii.Number] = None,
        source_access_configurations: typing.Optional[typing.Sequence[typing.Union["SourceAccessConfiguration", typing.Dict[builtins.str, typing.Any]]]] = None,
        starting_position: typing.Optional["StartingPosition"] = None,
        tumbling_window: typing.Optional[_aws_cdk_core_f4b25747.Duration] = None,
    ) -> None:
        '''
        :param batch_size: The largest number of records that AWS Lambda will retrieve from your event source at the time of invoking your function. Your function receives an event with all the retrieved records. Valid Range: Minimum value of 1. Maximum value of 10000. Default: - Amazon Kinesis, Amazon DynamoDB, and Amazon MSK is 100 records. The default for Amazon SQS is 10 messages. For standard SQS queues, the maximum is 10,000. For FIFO SQS queues, the maximum is 10.
        :param bisect_batch_on_error: If the function returns an error, split the batch in two and retry. Default: false
        :param enabled: Set to false to disable the event source upon creation. Default: true
        :param event_source_arn: The Amazon Resource Name (ARN) of the event source. Any record added to this stream can invoke the Lambda function. Default: - not set if using a self managed Kafka cluster, throws an error otherwise
        :param kafka_bootstrap_servers: A list of host and port pairs that are the addresses of the Kafka brokers in a self managed "bootstrap" Kafka cluster that a Kafka client connects to initially to bootstrap itself. They are in the format ``abc.example.com:9096``. Default: - none
        :param kafka_topic: The name of the Kafka topic. Default: - no topic
        :param max_batching_window: The maximum amount of time to gather records before invoking the function. Maximum of Duration.minutes(5) Default: Duration.seconds(0)
        :param max_record_age: The maximum age of a record that Lambda sends to a function for processing. Valid Range: - Minimum value of 60 seconds - Maximum value of 7 days Default: - infinite or until the record expires.
        :param on_failure: An Amazon SQS queue or Amazon SNS topic destination for discarded records. Default: discarded records are ignored
        :param parallelization_factor: The number of batches to process from each shard concurrently. Valid Range: - Minimum value of 1 - Maximum value of 10 Default: 1
        :param report_batch_item_failures: Allow functions to return partially successful responses for a batch of records. Default: false
        :param retry_attempts: The maximum number of times to retry when the function returns an error. Set to ``undefined`` if you want lambda to keep retrying infinitely or until the record expires. Valid Range: - Minimum value of 0 - Maximum value of 10000 Default: - infinite or until the record expires.
        :param source_access_configurations: Specific settings like the authentication protocol or the VPC components to secure access to your event source. Default: - none
        :param starting_position: The position in the DynamoDB, Kinesis or MSK stream where AWS Lambda should start reading. Default: - Required for Amazon Kinesis, Amazon DynamoDB, and Amazon MSK Streams sources.
        :param tumbling_window: The size of the tumbling windows to group records sent to DynamoDB or Kinesis. Default: - None

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_lambda as lambda_
            import aws_cdk.core as cdk
            
            # event_source_dlq: lambda.IEventSourceDlq
            # source_access_configuration_type: lambda.SourceAccessConfigurationType
            
            event_source_mapping_options = lambda.EventSourceMappingOptions(
                batch_size=123,
                bisect_batch_on_error=False,
                enabled=False,
                event_source_arn="eventSourceArn",
                kafka_bootstrap_servers=["kafkaBootstrapServers"],
                kafka_topic="kafkaTopic",
                max_batching_window=cdk.Duration.minutes(30),
                max_record_age=cdk.Duration.minutes(30),
                on_failure=event_source_dlq,
                parallelization_factor=123,
                report_batch_item_failures=False,
                retry_attempts=123,
                source_access_configurations=[lambda.SourceAccessConfiguration(
                    type=source_access_configuration_type,
                    uri="uri"
                )],
                starting_position=lambda_.StartingPosition.TRIM_HORIZON,
                tumbling_window=cdk.Duration.minutes(30)
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__1c29244a27a40996beb9fbb3181e4cd277dd2ff1cfda15e5b4c3662116980c77)
            check_type(argname="argument batch_size", value=batch_size, expected_type=type_hints["batch_size"])
            check_type(argname="argument bisect_batch_on_error", value=bisect_batch_on_error, expected_type=type_hints["bisect_batch_on_error"])
            check_type(argname="argument enabled", value=enabled, expected_type=type_hints["enabled"])
            check_type(argname="argument event_source_arn", value=event_source_arn, expected_type=type_hints["event_source_arn"])
            check_type(argname="argument kafka_bootstrap_servers", value=kafka_bootstrap_servers, expected_type=type_hints["kafka_bootstrap_servers"])
            check_type(argname="argument kafka_topic", value=kafka_topic, expected_type=type_hints["kafka_topic"])
            check_type(argname="argument max_batching_window", value=max_batching_window, expected_type=type_hints["max_batching_window"])
            check_type(argname="argument max_record_age", value=max_record_age, expected_type=type_hints["max_record_age"])
            check_type(argname="argument on_failure", value=on_failure, expected_type=type_hints["on_failure"])
            check_type(argname="argument parallelization_factor", value=parallelization_factor, expected_type=type_hints["parallelization_factor"])
            check_type(argname="argument report_batch_item_failures", value=report_batch_item_failures, expected_type=type_hints["report_batch_item_failures"])
            check_type(argname="argument retry_attempts", value=retry_attempts, expected_type=type_hints["retry_attempts"])
            check_type(argname="argument source_access_configurations", value=source_access_configurations, expected_type=type_hints["source_access_configurations"])
            check_type(argname="argument starting_position", value=starting_position, expected_type=type_hints["starting_position"])
            check_type(argname="argument tumbling_window", value=tumbling_window, expected_type=type_hints["tumbling_window"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if batch_size is not None:
            self._values["batch_size"] = batch_size
        if bisect_batch_on_error is not None:
            self._values["bisect_batch_on_error"] = bisect_batch_on_error
        if enabled is not None:
            self._values["enabled"] = enabled
        if event_source_arn is not None:
            self._values["event_source_arn"] = event_source_arn
        if kafka_bootstrap_servers is not None:
            self._values["kafka_bootstrap_servers"] = kafka_bootstrap_servers
        if kafka_topic is not None:
            self._values["kafka_topic"] = kafka_topic
        if max_batching_window is not None:
            self._values["max_batching_window"] = max_batching_window
        if max_record_age is not None:
            self._values["max_record_age"] = max_record_age
        if on_failure is not None:
            self._values["on_failure"] = on_failure
        if parallelization_factor is not None:
            self._values["parallelization_factor"] = parallelization_factor
        if report_batch_item_failures is not None:
            self._values["report_batch_item_failures"] = report_batch_item_failures
        if retry_attempts is not None:
            self._values["retry_attempts"] = retry_attempts
        if source_access_configurations is not None:
            self._values["source_access_configurations"] = source_access_configurations
        if starting_position is not None:
            self._values["starting_position"] = starting_position
        if tumbling_window is not None:
            self._values["tumbling_window"] = tumbling_window

    @builtins.property
    def batch_size(self) -> typing.Optional[jsii.Number]:
        '''The largest number of records that AWS Lambda will retrieve from your event source at the time of invoking your function.

        Your function receives an
        event with all the retrieved records.

        Valid Range: Minimum value of 1. Maximum value of 10000.

        :default:

        - Amazon Kinesis, Amazon DynamoDB, and Amazon MSK is 100 records.
        The default for Amazon SQS is 10 messages. For standard SQS queues, the maximum is 10,000. For FIFO SQS queues, the maximum is 10.
        '''
        result = self._values.get("batch_size")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def bisect_batch_on_error(self) -> typing.Optional[builtins.bool]:
        '''If the function returns an error, split the batch in two and retry.

        :default: false
        '''
        result = self._values.get("bisect_batch_on_error")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def enabled(self) -> typing.Optional[builtins.bool]:
        '''Set to false to disable the event source upon creation.

        :default: true
        '''
        result = self._values.get("enabled")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def event_source_arn(self) -> typing.Optional[builtins.str]:
        '''The Amazon Resource Name (ARN) of the event source.

        Any record added to
        this stream can invoke the Lambda function.

        :default: - not set if using a self managed Kafka cluster, throws an error otherwise
        '''
        result = self._values.get("event_source_arn")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def kafka_bootstrap_servers(self) -> typing.Optional[typing.List[builtins.str]]:
        '''A list of host and port pairs that are the addresses of the Kafka brokers in a self managed "bootstrap" Kafka cluster that a Kafka client connects to initially to bootstrap itself.

        They are in the format ``abc.example.com:9096``.

        :default: - none
        '''
        result = self._values.get("kafka_bootstrap_servers")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def kafka_topic(self) -> typing.Optional[builtins.str]:
        '''The name of the Kafka topic.

        :default: - no topic
        '''
        result = self._values.get("kafka_topic")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def max_batching_window(self) -> typing.Optional[_aws_cdk_core_f4b25747.Duration]:
        '''The maximum amount of time to gather records before invoking the function.

        Maximum of Duration.minutes(5)

        :default: Duration.seconds(0)
        '''
        result = self._values.get("max_batching_window")
        return typing.cast(typing.Optional[_aws_cdk_core_f4b25747.Duration], result)

    @builtins.property
    def max_record_age(self) -> typing.Optional[_aws_cdk_core_f4b25747.Duration]:
        '''The maximum age of a record that Lambda sends to a function for processing.

        Valid Range:

        - Minimum value of 60 seconds
        - Maximum value of 7 days

        :default: - infinite or until the record expires.
        '''
        result = self._values.get("max_record_age")
        return typing.cast(typing.Optional[_aws_cdk_core_f4b25747.Duration], result)

    @builtins.property
    def on_failure(self) -> typing.Optional["IEventSourceDlq"]:
        '''An Amazon SQS queue or Amazon SNS topic destination for discarded records.

        :default: discarded records are ignored
        '''
        result = self._values.get("on_failure")
        return typing.cast(typing.Optional["IEventSourceDlq"], result)

    @builtins.property
    def parallelization_factor(self) -> typing.Optional[jsii.Number]:
        '''The number of batches to process from each shard concurrently.

        Valid Range:

        - Minimum value of 1
        - Maximum value of 10

        :default: 1
        '''
        result = self._values.get("parallelization_factor")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def report_batch_item_failures(self) -> typing.Optional[builtins.bool]:
        '''Allow functions to return partially successful responses for a batch of records.

        :default: false

        :see: https://docs.aws.amazon.com/lambda/latest/dg/with-ddb.html#services-ddb-batchfailurereporting
        '''
        result = self._values.get("report_batch_item_failures")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def retry_attempts(self) -> typing.Optional[jsii.Number]:
        '''The maximum number of times to retry when the function returns an error.

        Set to ``undefined`` if you want lambda to keep retrying infinitely or until
        the record expires.

        Valid Range:

        - Minimum value of 0
        - Maximum value of 10000

        :default: - infinite or until the record expires.
        '''
        result = self._values.get("retry_attempts")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def source_access_configurations(
        self,
    ) -> typing.Optional[typing.List["SourceAccessConfiguration"]]:
        '''Specific settings like the authentication protocol or the VPC components to secure access to your event source.

        :default: - none

        :see: https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-eventsourcemapping-sourceaccessconfiguration.html
        '''
        result = self._values.get("source_access_configurations")
        return typing.cast(typing.Optional[typing.List["SourceAccessConfiguration"]], result)

    @builtins.property
    def starting_position(self) -> typing.Optional["StartingPosition"]:
        '''The position in the DynamoDB, Kinesis or MSK stream where AWS Lambda should start reading.

        :default: - Required for Amazon Kinesis, Amazon DynamoDB, and Amazon MSK Streams sources.

        :see: https://docs.aws.amazon.com/kinesis/latest/APIReference/API_GetShardIterator.html#Kinesis-GetShardIterator-request-ShardIteratorType
        '''
        result = self._values.get("starting_position")
        return typing.cast(typing.Optional["StartingPosition"], result)

    @builtins.property
    def tumbling_window(self) -> typing.Optional[_aws_cdk_core_f4b25747.Duration]:
        '''The size of the tumbling windows to group records sent to DynamoDB or Kinesis.

        :default: - None

        :see:

        https://docs.aws.amazon.com/lambda/latest/dg/with-ddb.html#services-ddb-windows

        Valid Range: 0 - 15 minutes
        '''
        result = self._values.get("tumbling_window")
        return typing.cast(typing.Optional[_aws_cdk_core_f4b25747.Duration], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "EventSourceMappingOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-lambda.EventSourceMappingProps",
    jsii_struct_bases=[EventSourceMappingOptions],
    name_mapping={
        "batch_size": "batchSize",
        "bisect_batch_on_error": "bisectBatchOnError",
        "enabled": "enabled",
        "event_source_arn": "eventSourceArn",
        "kafka_bootstrap_servers": "kafkaBootstrapServers",
        "kafka_topic": "kafkaTopic",
        "max_batching_window": "maxBatchingWindow",
        "max_record_age": "maxRecordAge",
        "on_failure": "onFailure",
        "parallelization_factor": "parallelizationFactor",
        "report_batch_item_failures": "reportBatchItemFailures",
        "retry_attempts": "retryAttempts",
        "source_access_configurations": "sourceAccessConfigurations",
        "starting_position": "startingPosition",
        "tumbling_window": "tumblingWindow",
        "target": "target",
    },
)
class EventSourceMappingProps(EventSourceMappingOptions):
    def __init__(
        self,
        *,
        batch_size: typing.Optional[jsii.Number] = None,
        bisect_batch_on_error: typing.Optional[builtins.bool] = None,
        enabled: typing.Optional[builtins.bool] = None,
        event_source_arn: typing.Optional[builtins.str] = None,
        kafka_bootstrap_servers: typing.Optional[typing.Sequence[builtins.str]] = None,
        kafka_topic: typing.Optional[builtins.str] = None,
        max_batching_window: typing.Optional[_aws_cdk_core_f4b25747.Duration] = None,
        max_record_age: typing.Optional[_aws_cdk_core_f4b25747.Duration] = None,
        on_failure: typing.Optional["IEventSourceDlq"] = None,
        parallelization_factor: typing.Optional[jsii.Number] = None,
        report_batch_item_failures: typing.Optional[builtins.bool] = None,
        retry_attempts: typing.Optional[jsii.Number] = None,
        source_access_configurations: typing.Optional[typing.Sequence[typing.Union["SourceAccessConfiguration", typing.Dict[builtins.str, typing.Any]]]] = None,
        starting_position: typing.Optional["StartingPosition"] = None,
        tumbling_window: typing.Optional[_aws_cdk_core_f4b25747.Duration] = None,
        target: "IFunction",
    ) -> None:
        '''Properties for declaring a new event source mapping.

        :param batch_size: The largest number of records that AWS Lambda will retrieve from your event source at the time of invoking your function. Your function receives an event with all the retrieved records. Valid Range: Minimum value of 1. Maximum value of 10000. Default: - Amazon Kinesis, Amazon DynamoDB, and Amazon MSK is 100 records. The default for Amazon SQS is 10 messages. For standard SQS queues, the maximum is 10,000. For FIFO SQS queues, the maximum is 10.
        :param bisect_batch_on_error: If the function returns an error, split the batch in two and retry. Default: false
        :param enabled: Set to false to disable the event source upon creation. Default: true
        :param event_source_arn: The Amazon Resource Name (ARN) of the event source. Any record added to this stream can invoke the Lambda function. Default: - not set if using a self managed Kafka cluster, throws an error otherwise
        :param kafka_bootstrap_servers: A list of host and port pairs that are the addresses of the Kafka brokers in a self managed "bootstrap" Kafka cluster that a Kafka client connects to initially to bootstrap itself. They are in the format ``abc.example.com:9096``. Default: - none
        :param kafka_topic: The name of the Kafka topic. Default: - no topic
        :param max_batching_window: The maximum amount of time to gather records before invoking the function. Maximum of Duration.minutes(5) Default: Duration.seconds(0)
        :param max_record_age: The maximum age of a record that Lambda sends to a function for processing. Valid Range: - Minimum value of 60 seconds - Maximum value of 7 days Default: - infinite or until the record expires.
        :param on_failure: An Amazon SQS queue or Amazon SNS topic destination for discarded records. Default: discarded records are ignored
        :param parallelization_factor: The number of batches to process from each shard concurrently. Valid Range: - Minimum value of 1 - Maximum value of 10 Default: 1
        :param report_batch_item_failures: Allow functions to return partially successful responses for a batch of records. Default: false
        :param retry_attempts: The maximum number of times to retry when the function returns an error. Set to ``undefined`` if you want lambda to keep retrying infinitely or until the record expires. Valid Range: - Minimum value of 0 - Maximum value of 10000 Default: - infinite or until the record expires.
        :param source_access_configurations: Specific settings like the authentication protocol or the VPC components to secure access to your event source. Default: - none
        :param starting_position: The position in the DynamoDB, Kinesis or MSK stream where AWS Lambda should start reading. Default: - Required for Amazon Kinesis, Amazon DynamoDB, and Amazon MSK Streams sources.
        :param tumbling_window: The size of the tumbling windows to group records sent to DynamoDB or Kinesis. Default: - None
        :param target: The target AWS Lambda function.

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_lambda as lambda_
            import aws_cdk.core as cdk
            
            # event_source_dlq: lambda.IEventSourceDlq
            # function_: lambda.Function
            # source_access_configuration_type: lambda.SourceAccessConfigurationType
            
            event_source_mapping_props = lambda.EventSourceMappingProps(
                target=function_,
            
                # the properties below are optional
                batch_size=123,
                bisect_batch_on_error=False,
                enabled=False,
                event_source_arn="eventSourceArn",
                kafka_bootstrap_servers=["kafkaBootstrapServers"],
                kafka_topic="kafkaTopic",
                max_batching_window=cdk.Duration.minutes(30),
                max_record_age=cdk.Duration.minutes(30),
                on_failure=event_source_dlq,
                parallelization_factor=123,
                report_batch_item_failures=False,
                retry_attempts=123,
                source_access_configurations=[lambda.SourceAccessConfiguration(
                    type=source_access_configuration_type,
                    uri="uri"
                )],
                starting_position=lambda_.StartingPosition.TRIM_HORIZON,
                tumbling_window=cdk.Duration.minutes(30)
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__95afe96451541703ecd97b0b2592955661fedfeb22f01bb223c224f6fa6d2c10)
            check_type(argname="argument batch_size", value=batch_size, expected_type=type_hints["batch_size"])
            check_type(argname="argument bisect_batch_on_error", value=bisect_batch_on_error, expected_type=type_hints["bisect_batch_on_error"])
            check_type(argname="argument enabled", value=enabled, expected_type=type_hints["enabled"])
            check_type(argname="argument event_source_arn", value=event_source_arn, expected_type=type_hints["event_source_arn"])
            check_type(argname="argument kafka_bootstrap_servers", value=kafka_bootstrap_servers, expected_type=type_hints["kafka_bootstrap_servers"])
            check_type(argname="argument kafka_topic", value=kafka_topic, expected_type=type_hints["kafka_topic"])
            check_type(argname="argument max_batching_window", value=max_batching_window, expected_type=type_hints["max_batching_window"])
            check_type(argname="argument max_record_age", value=max_record_age, expected_type=type_hints["max_record_age"])
            check_type(argname="argument on_failure", value=on_failure, expected_type=type_hints["on_failure"])
            check_type(argname="argument parallelization_factor", value=parallelization_factor, expected_type=type_hints["parallelization_factor"])
            check_type(argname="argument report_batch_item_failures", value=report_batch_item_failures, expected_type=type_hints["report_batch_item_failures"])
            check_type(argname="argument retry_attempts", value=retry_attempts, expected_type=type_hints["retry_attempts"])
            check_type(argname="argument source_access_configurations", value=source_access_configurations, expected_type=type_hints["source_access_configurations"])
            check_type(argname="argument starting_position", value=starting_position, expected_type=type_hints["starting_position"])
            check_type(argname="argument tumbling_window", value=tumbling_window, expected_type=type_hints["tumbling_window"])
            check_type(argname="argument target", value=target, expected_type=type_hints["target"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "target": target,
        }
        if batch_size is not None:
            self._values["batch_size"] = batch_size
        if bisect_batch_on_error is not None:
            self._values["bisect_batch_on_error"] = bisect_batch_on_error
        if enabled is not None:
            self._values["enabled"] = enabled
        if event_source_arn is not None:
            self._values["event_source_arn"] = event_source_arn
        if kafka_bootstrap_servers is not None:
            self._values["kafka_bootstrap_servers"] = kafka_bootstrap_servers
        if kafka_topic is not None:
            self._values["kafka_topic"] = kafka_topic
        if max_batching_window is not None:
            self._values["max_batching_window"] = max_batching_window
        if max_record_age is not None:
            self._values["max_record_age"] = max_record_age
        if on_failure is not None:
            self._values["on_failure"] = on_failure
        if parallelization_factor is not None:
            self._values["parallelization_factor"] = parallelization_factor
        if report_batch_item_failures is not None:
            self._values["report_batch_item_failures"] = report_batch_item_failures
        if retry_attempts is not None:
            self._values["retry_attempts"] = retry_attempts
        if source_access_configurations is not None:
            self._values["source_access_configurations"] = source_access_configurations
        if starting_position is not None:
            self._values["starting_position"] = starting_position
        if tumbling_window is not None:
            self._values["tumbling_window"] = tumbling_window

    @builtins.property
    def batch_size(self) -> typing.Optional[jsii.Number]:
        '''The largest number of records that AWS Lambda will retrieve from your event source at the time of invoking your function.

        Your function receives an
        event with all the retrieved records.

        Valid Range: Minimum value of 1. Maximum value of 10000.

        :default:

        - Amazon Kinesis, Amazon DynamoDB, and Amazon MSK is 100 records.
        The default for Amazon SQS is 10 messages. For standard SQS queues, the maximum is 10,000. For FIFO SQS queues, the maximum is 10.
        '''
        result = self._values.get("batch_size")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def bisect_batch_on_error(self) -> typing.Optional[builtins.bool]:
        '''If the function returns an error, split the batch in two and retry.

        :default: false
        '''
        result = self._values.get("bisect_batch_on_error")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def enabled(self) -> typing.Optional[builtins.bool]:
        '''Set to false to disable the event source upon creation.

        :default: true
        '''
        result = self._values.get("enabled")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def event_source_arn(self) -> typing.Optional[builtins.str]:
        '''The Amazon Resource Name (ARN) of the event source.

        Any record added to
        this stream can invoke the Lambda function.

        :default: - not set if using a self managed Kafka cluster, throws an error otherwise
        '''
        result = self._values.get("event_source_arn")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def kafka_bootstrap_servers(self) -> typing.Optional[typing.List[builtins.str]]:
        '''A list of host and port pairs that are the addresses of the Kafka brokers in a self managed "bootstrap" Kafka cluster that a Kafka client connects to initially to bootstrap itself.

        They are in the format ``abc.example.com:9096``.

        :default: - none
        '''
        result = self._values.get("kafka_bootstrap_servers")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def kafka_topic(self) -> typing.Optional[builtins.str]:
        '''The name of the Kafka topic.

        :default: - no topic
        '''
        result = self._values.get("kafka_topic")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def max_batching_window(self) -> typing.Optional[_aws_cdk_core_f4b25747.Duration]:
        '''The maximum amount of time to gather records before invoking the function.

        Maximum of Duration.minutes(5)

        :default: Duration.seconds(0)
        '''
        result = self._values.get("max_batching_window")
        return typing.cast(typing.Optional[_aws_cdk_core_f4b25747.Duration], result)

    @builtins.property
    def max_record_age(self) -> typing.Optional[_aws_cdk_core_f4b25747.Duration]:
        '''The maximum age of a record that Lambda sends to a function for processing.

        Valid Range:

        - Minimum value of 60 seconds
        - Maximum value of 7 days

        :default: - infinite or until the record expires.
        '''
        result = self._values.get("max_record_age")
        return typing.cast(typing.Optional[_aws_cdk_core_f4b25747.Duration], result)

    @builtins.property
    def on_failure(self) -> typing.Optional["IEventSourceDlq"]:
        '''An Amazon SQS queue or Amazon SNS topic destination for discarded records.

        :default: discarded records are ignored
        '''
        result = self._values.get("on_failure")
        return typing.cast(typing.Optional["IEventSourceDlq"], result)

    @builtins.property
    def parallelization_factor(self) -> typing.Optional[jsii.Number]:
        '''The number of batches to process from each shard concurrently.

        Valid Range:

        - Minimum value of 1
        - Maximum value of 10

        :default: 1
        '''
        result = self._values.get("parallelization_factor")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def report_batch_item_failures(self) -> typing.Optional[builtins.bool]:
        '''Allow functions to return partially successful responses for a batch of records.

        :default: false

        :see: https://docs.aws.amazon.com/lambda/latest/dg/with-ddb.html#services-ddb-batchfailurereporting
        '''
        result = self._values.get("report_batch_item_failures")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def retry_attempts(self) -> typing.Optional[jsii.Number]:
        '''The maximum number of times to retry when the function returns an error.

        Set to ``undefined`` if you want lambda to keep retrying infinitely or until
        the record expires.

        Valid Range:

        - Minimum value of 0
        - Maximum value of 10000

        :default: - infinite or until the record expires.
        '''
        result = self._values.get("retry_attempts")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def source_access_configurations(
        self,
    ) -> typing.Optional[typing.List["SourceAccessConfiguration"]]:
        '''Specific settings like the authentication protocol or the VPC components to secure access to your event source.

        :default: - none

        :see: https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-eventsourcemapping-sourceaccessconfiguration.html
        '''
        result = self._values.get("source_access_configurations")
        return typing.cast(typing.Optional[typing.List["SourceAccessConfiguration"]], result)

    @builtins.property
    def starting_position(self) -> typing.Optional["StartingPosition"]:
        '''The position in the DynamoDB, Kinesis or MSK stream where AWS Lambda should start reading.

        :default: - Required for Amazon Kinesis, Amazon DynamoDB, and Amazon MSK Streams sources.

        :see: https://docs.aws.amazon.com/kinesis/latest/APIReference/API_GetShardIterator.html#Kinesis-GetShardIterator-request-ShardIteratorType
        '''
        result = self._values.get("starting_position")
        return typing.cast(typing.Optional["StartingPosition"], result)

    @builtins.property
    def tumbling_window(self) -> typing.Optional[_aws_cdk_core_f4b25747.Duration]:
        '''The size of the tumbling windows to group records sent to DynamoDB or Kinesis.

        :default: - None

        :see:

        https://docs.aws.amazon.com/lambda/latest/dg/with-ddb.html#services-ddb-windows

        Valid Range: 0 - 15 minutes
        '''
        result = self._values.get("tumbling_window")
        return typing.cast(typing.Optional[_aws_cdk_core_f4b25747.Duration], result)

    @builtins.property
    def target(self) -> "IFunction":
        '''The target AWS Lambda function.'''
        result = self._values.get("target")
        assert result is not None, "Required property 'target' is missing"
        return typing.cast("IFunction", result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "EventSourceMappingProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class FileSystem(metaclass=jsii.JSIIMeta, jsii_type="@aws-cdk/aws-lambda.FileSystem"):
    '''Represents the filesystem for the Lambda function.

    :exampleMetadata: infused

    Example::

        import aws_cdk.aws_ec2 as ec2
        import aws_cdk.aws_efs as efs
        
        
        # create a new VPC
        vpc = ec2.Vpc(self, "VPC")
        
        # create a new Amazon EFS filesystem
        file_system = efs.FileSystem(self, "Efs", vpc=vpc)
        
        # create a new access point from the filesystem
        access_point = file_system.add_access_point("AccessPoint",
            # set /export/lambda as the root of the access point
            path="/export/lambda",
            # as /export/lambda does not exist in a new efs filesystem, the efs will create the directory with the following createAcl
            create_acl=efs.Acl(
                owner_uid="1001",
                owner_gid="1001",
                permissions="750"
            ),
            # enforce the POSIX identity so lambda function will access with this identity
            posix_user=efs.PosixUser(
                uid="1001",
                gid="1001"
            )
        )
        
        fn = lambda_.Function(self, "MyLambda",
            # mount the access point to /mnt/msg in the lambda runtime environment
            filesystem=lambda_.FileSystem.from_efs_access_point(access_point, "/mnt/msg"),
            runtime=lambda_.Runtime.NODEJS_16_X,
            handler="index.handler",
            code=lambda_.Code.from_asset(path.join(__dirname, "lambda-handler")),
            vpc=vpc
        )
    '''

    def __init__(
        self,
        *,
        arn: builtins.str,
        local_mount_path: builtins.str,
        connections: typing.Optional[_aws_cdk_aws_ec2_67de8e8d.Connections] = None,
        dependency: typing.Optional[typing.Sequence[_aws_cdk_core_f4b25747.IDependable]] = None,
        policies: typing.Optional[typing.Sequence[_aws_cdk_aws_iam_940a1ce0.PolicyStatement]] = None,
    ) -> None:
        '''
        :param arn: ARN of the access point.
        :param local_mount_path: mount path in the lambda runtime environment.
        :param connections: connections object used to allow ingress traffic from lambda function. Default: - no connections required to add extra ingress rules for Lambda function
        :param dependency: array of IDependable that lambda function depends on. Default: - no dependency
        :param policies: additional IAM policies required for the lambda function. Default: - no additional policies required
        '''
        config = FileSystemConfig(
            arn=arn,
            local_mount_path=local_mount_path,
            connections=connections,
            dependency=dependency,
            policies=policies,
        )

        jsii.create(self.__class__, self, [config])

    @jsii.member(jsii_name="fromEfsAccessPoint")
    @builtins.classmethod
    def from_efs_access_point(
        cls,
        ap: _aws_cdk_aws_efs_b9f7a603.IAccessPoint,
        mount_path: builtins.str,
    ) -> "FileSystem":
        '''mount the filesystem from Amazon EFS.

        :param ap: the Amazon EFS access point.
        :param mount_path: the target path in the lambda runtime environment.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8ab12cecee808278ed6a8ff85a264454d7361d89746d3e7be029eaecddfa3696)
            check_type(argname="argument ap", value=ap, expected_type=type_hints["ap"])
            check_type(argname="argument mount_path", value=mount_path, expected_type=type_hints["mount_path"])
        return typing.cast("FileSystem", jsii.sinvoke(cls, "fromEfsAccessPoint", [ap, mount_path]))

    @builtins.property
    @jsii.member(jsii_name="config")
    def config(self) -> "FileSystemConfig":
        '''the FileSystem configurations for the Lambda function.'''
        return typing.cast("FileSystemConfig", jsii.get(self, "config"))


@jsii.data_type(
    jsii_type="@aws-cdk/aws-lambda.FileSystemConfig",
    jsii_struct_bases=[],
    name_mapping={
        "arn": "arn",
        "local_mount_path": "localMountPath",
        "connections": "connections",
        "dependency": "dependency",
        "policies": "policies",
    },
)
class FileSystemConfig:
    def __init__(
        self,
        *,
        arn: builtins.str,
        local_mount_path: builtins.str,
        connections: typing.Optional[_aws_cdk_aws_ec2_67de8e8d.Connections] = None,
        dependency: typing.Optional[typing.Sequence[_aws_cdk_core_f4b25747.IDependable]] = None,
        policies: typing.Optional[typing.Sequence[_aws_cdk_aws_iam_940a1ce0.PolicyStatement]] = None,
    ) -> None:
        '''FileSystem configurations for the Lambda function.

        :param arn: ARN of the access point.
        :param local_mount_path: mount path in the lambda runtime environment.
        :param connections: connections object used to allow ingress traffic from lambda function. Default: - no connections required to add extra ingress rules for Lambda function
        :param dependency: array of IDependable that lambda function depends on. Default: - no dependency
        :param policies: additional IAM policies required for the lambda function. Default: - no additional policies required

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_ec2 as ec2
            import aws_cdk.aws_iam as iam
            import aws_cdk.aws_lambda as lambda_
            import aws_cdk.core as cdk
            
            # connections: ec2.Connections
            # dependable: cdk.IDependable
            # policy_statement: iam.PolicyStatement
            
            file_system_config = lambda.FileSystemConfig(
                arn="arn",
                local_mount_path="localMountPath",
            
                # the properties below are optional
                connections=connections,
                dependency=[dependable],
                policies=[policy_statement]
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__070ba46afc1a2d0b909fde160cb25e3359667c77dbe0c837147fc3ccd4212acb)
            check_type(argname="argument arn", value=arn, expected_type=type_hints["arn"])
            check_type(argname="argument local_mount_path", value=local_mount_path, expected_type=type_hints["local_mount_path"])
            check_type(argname="argument connections", value=connections, expected_type=type_hints["connections"])
            check_type(argname="argument dependency", value=dependency, expected_type=type_hints["dependency"])
            check_type(argname="argument policies", value=policies, expected_type=type_hints["policies"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "arn": arn,
            "local_mount_path": local_mount_path,
        }
        if connections is not None:
            self._values["connections"] = connections
        if dependency is not None:
            self._values["dependency"] = dependency
        if policies is not None:
            self._values["policies"] = policies

    @builtins.property
    def arn(self) -> builtins.str:
        '''ARN of the access point.'''
        result = self._values.get("arn")
        assert result is not None, "Required property 'arn' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def local_mount_path(self) -> builtins.str:
        '''mount path in the lambda runtime environment.'''
        result = self._values.get("local_mount_path")
        assert result is not None, "Required property 'local_mount_path' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def connections(self) -> typing.Optional[_aws_cdk_aws_ec2_67de8e8d.Connections]:
        '''connections object used to allow ingress traffic from lambda function.

        :default: - no connections required to add extra ingress rules for Lambda function
        '''
        result = self._values.get("connections")
        return typing.cast(typing.Optional[_aws_cdk_aws_ec2_67de8e8d.Connections], result)

    @builtins.property
    def dependency(
        self,
    ) -> typing.Optional[typing.List[_aws_cdk_core_f4b25747.IDependable]]:
        '''array of IDependable that lambda function depends on.

        :default: - no dependency
        '''
        result = self._values.get("dependency")
        return typing.cast(typing.Optional[typing.List[_aws_cdk_core_f4b25747.IDependable]], result)

    @builtins.property
    def policies(
        self,
    ) -> typing.Optional[typing.List[_aws_cdk_aws_iam_940a1ce0.PolicyStatement]]:
        '''additional IAM policies required for the lambda function.

        :default: - no additional policies required
        '''
        result = self._values.get("policies")
        return typing.cast(typing.Optional[typing.List[_aws_cdk_aws_iam_940a1ce0.PolicyStatement]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "FileSystemConfig(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-lambda.FunctionAttributes",
    jsii_struct_bases=[],
    name_mapping={
        "function_arn": "functionArn",
        "architecture": "architecture",
        "role": "role",
        "same_environment": "sameEnvironment",
        "security_group": "securityGroup",
        "security_group_id": "securityGroupId",
        "skip_permissions": "skipPermissions",
    },
)
class FunctionAttributes:
    def __init__(
        self,
        *,
        function_arn: builtins.str,
        architecture: typing.Optional[Architecture] = None,
        role: typing.Optional[_aws_cdk_aws_iam_940a1ce0.IRole] = None,
        same_environment: typing.Optional[builtins.bool] = None,
        security_group: typing.Optional[_aws_cdk_aws_ec2_67de8e8d.ISecurityGroup] = None,
        security_group_id: typing.Optional[builtins.str] = None,
        skip_permissions: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''Represents a Lambda function defined outside of this stack.

        :param function_arn: The ARN of the Lambda function. Format: arn::lambda:::function:
        :param architecture: The architecture of this Lambda Function (this is an optional attribute and defaults to X86_64). Default: - Architecture.X86_64
        :param role: The IAM execution role associated with this function. If the role is not specified, any role-related operations will no-op.
        :param same_environment: Setting this property informs the CDK that the imported function is in the same environment as the stack. This affects certain behaviours such as, whether this function's permission can be modified. When not configured, the CDK attempts to auto-determine this. For environment agnostic stacks, i.e., stacks where the account is not specified with the ``env`` property, this is determined to be false. Set this to property *ONLY IF* the imported function is in the same account as the stack it's imported in. Default: - depends: true, if the Stack is configured with an explicit ``env`` (account and region) and the account is the same as this function. For environment-agnostic stacks this will default to ``false``.
        :param security_group: The security group of this Lambda, if in a VPC. This needs to be given in order to support allowing connections to this Lambda.
        :param security_group_id: (deprecated) Id of the security group of this Lambda, if in a VPC. This needs to be given in order to support allowing connections to this Lambda.
        :param skip_permissions: Setting this property informs the CDK that the imported function ALREADY HAS the necessary permissions for what you are trying to do. When not configured, the CDK attempts to auto-determine whether or not additional permissions are necessary on the function when grant APIs are used. If the CDK tried to add permissions on an imported lambda, it will fail. Set this property *ONLY IF* you are committing to manage the imported function's permissions outside of CDK. You are acknowledging that your CDK code alone will have insufficient permissions to access the imported function. Default: false

        :exampleMetadata: infused

        Example::

            fn = lambda_.Function.from_function_attributes(self, "Function",
                function_arn="arn:aws:lambda:us-east-1:123456789012:function:MyFn",
                # The following are optional properties for specific use cases and should be used with caution:
            
                # Use Case: imported function is in the same account as the stack. This tells the CDK that it
                # can modify the function's permissions.
                same_environment=True,
            
                # Use Case: imported function is in a different account and user commits to ensuring that the
                # imported function has the correct permissions outside the CDK.
                skip_permissions=True
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e70081ebe6e699cd267d3d0278482739e025eb69c750fbaef0a1697404142eae)
            check_type(argname="argument function_arn", value=function_arn, expected_type=type_hints["function_arn"])
            check_type(argname="argument architecture", value=architecture, expected_type=type_hints["architecture"])
            check_type(argname="argument role", value=role, expected_type=type_hints["role"])
            check_type(argname="argument same_environment", value=same_environment, expected_type=type_hints["same_environment"])
            check_type(argname="argument security_group", value=security_group, expected_type=type_hints["security_group"])
            check_type(argname="argument security_group_id", value=security_group_id, expected_type=type_hints["security_group_id"])
            check_type(argname="argument skip_permissions", value=skip_permissions, expected_type=type_hints["skip_permissions"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "function_arn": function_arn,
        }
        if architecture is not None:
            self._values["architecture"] = architecture
        if role is not None:
            self._values["role"] = role
        if same_environment is not None:
            self._values["same_environment"] = same_environment
        if security_group is not None:
            self._values["security_group"] = security_group
        if security_group_id is not None:
            self._values["security_group_id"] = security_group_id
        if skip_permissions is not None:
            self._values["skip_permissions"] = skip_permissions

    @builtins.property
    def function_arn(self) -> builtins.str:
        '''The ARN of the Lambda function.

        Format: arn::lambda:::function:
        '''
        result = self._values.get("function_arn")
        assert result is not None, "Required property 'function_arn' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def architecture(self) -> typing.Optional[Architecture]:
        '''The architecture of this Lambda Function (this is an optional attribute and defaults to X86_64).

        :default: - Architecture.X86_64
        '''
        result = self._values.get("architecture")
        return typing.cast(typing.Optional[Architecture], result)

    @builtins.property
    def role(self) -> typing.Optional[_aws_cdk_aws_iam_940a1ce0.IRole]:
        '''The IAM execution role associated with this function.

        If the role is not specified, any role-related operations will no-op.
        '''
        result = self._values.get("role")
        return typing.cast(typing.Optional[_aws_cdk_aws_iam_940a1ce0.IRole], result)

    @builtins.property
    def same_environment(self) -> typing.Optional[builtins.bool]:
        '''Setting this property informs the CDK that the imported function is in the same environment as the stack.

        This affects certain behaviours such as, whether this function's permission can be modified.
        When not configured, the CDK attempts to auto-determine this. For environment agnostic stacks, i.e., stacks
        where the account is not specified with the ``env`` property, this is determined to be false.

        Set this to property *ONLY IF* the imported function is in the same account as the stack
        it's imported in.

        :default:

        - depends: true, if the Stack is configured with an explicit ``env`` (account and region) and the account is the same as this function.
        For environment-agnostic stacks this will default to ``false``.
        '''
        result = self._values.get("same_environment")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def security_group(
        self,
    ) -> typing.Optional[_aws_cdk_aws_ec2_67de8e8d.ISecurityGroup]:
        '''The security group of this Lambda, if in a VPC.

        This needs to be given in order to support allowing connections
        to this Lambda.
        '''
        result = self._values.get("security_group")
        return typing.cast(typing.Optional[_aws_cdk_aws_ec2_67de8e8d.ISecurityGroup], result)

    @builtins.property
    def security_group_id(self) -> typing.Optional[builtins.str]:
        '''(deprecated) Id of the security group of this Lambda, if in a VPC.

        This needs to be given in order to support allowing connections
        to this Lambda.

        :deprecated: use ``securityGroup`` instead

        :stability: deprecated
        '''
        result = self._values.get("security_group_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def skip_permissions(self) -> typing.Optional[builtins.bool]:
        '''Setting this property informs the CDK that the imported function ALREADY HAS the necessary permissions for what you are trying to do.

        When not configured, the CDK attempts to auto-determine whether or not
        additional permissions are necessary on the function when grant APIs are used. If the CDK tried to add
        permissions on an imported lambda, it will fail.

        Set this property *ONLY IF* you are committing to manage the imported function's permissions outside of
        CDK. You are acknowledging that your CDK code alone will have insufficient permissions to access the
        imported function.

        :default: false
        '''
        result = self._values.get("skip_permissions")
        return typing.cast(typing.Optional[builtins.bool], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "FunctionAttributes(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-lambda.FunctionOptions",
    jsii_struct_bases=[EventInvokeConfigOptions],
    name_mapping={
        "max_event_age": "maxEventAge",
        "on_failure": "onFailure",
        "on_success": "onSuccess",
        "retry_attempts": "retryAttempts",
        "allow_all_outbound": "allowAllOutbound",
        "allow_public_subnet": "allowPublicSubnet",
        "architecture": "architecture",
        "architectures": "architectures",
        "code_signing_config": "codeSigningConfig",
        "current_version_options": "currentVersionOptions",
        "dead_letter_queue": "deadLetterQueue",
        "dead_letter_queue_enabled": "deadLetterQueueEnabled",
        "dead_letter_topic": "deadLetterTopic",
        "description": "description",
        "environment": "environment",
        "environment_encryption": "environmentEncryption",
        "ephemeral_storage_size": "ephemeralStorageSize",
        "events": "events",
        "filesystem": "filesystem",
        "function_name": "functionName",
        "initial_policy": "initialPolicy",
        "insights_version": "insightsVersion",
        "layers": "layers",
        "log_retention": "logRetention",
        "log_retention_retry_options": "logRetentionRetryOptions",
        "log_retention_role": "logRetentionRole",
        "memory_size": "memorySize",
        "profiling": "profiling",
        "profiling_group": "profilingGroup",
        "reserved_concurrent_executions": "reservedConcurrentExecutions",
        "role": "role",
        "security_group": "securityGroup",
        "security_groups": "securityGroups",
        "timeout": "timeout",
        "tracing": "tracing",
        "vpc": "vpc",
        "vpc_subnets": "vpcSubnets",
    },
)
class FunctionOptions(EventInvokeConfigOptions):
    def __init__(
        self,
        *,
        max_event_age: typing.Optional[_aws_cdk_core_f4b25747.Duration] = None,
        on_failure: typing.Optional["IDestination"] = None,
        on_success: typing.Optional["IDestination"] = None,
        retry_attempts: typing.Optional[jsii.Number] = None,
        allow_all_outbound: typing.Optional[builtins.bool] = None,
        allow_public_subnet: typing.Optional[builtins.bool] = None,
        architecture: typing.Optional[Architecture] = None,
        architectures: typing.Optional[typing.Sequence[Architecture]] = None,
        code_signing_config: typing.Optional["ICodeSigningConfig"] = None,
        current_version_options: typing.Optional[typing.Union["VersionOptions", typing.Dict[builtins.str, typing.Any]]] = None,
        dead_letter_queue: typing.Optional[_aws_cdk_aws_sqs_48bffef9.IQueue] = None,
        dead_letter_queue_enabled: typing.Optional[builtins.bool] = None,
        dead_letter_topic: typing.Optional[_aws_cdk_aws_sns_889c7272.ITopic] = None,
        description: typing.Optional[builtins.str] = None,
        environment: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        environment_encryption: typing.Optional[_aws_cdk_aws_kms_e491a92b.IKey] = None,
        ephemeral_storage_size: typing.Optional[_aws_cdk_core_f4b25747.Size] = None,
        events: typing.Optional[typing.Sequence["IEventSource"]] = None,
        filesystem: typing.Optional[FileSystem] = None,
        function_name: typing.Optional[builtins.str] = None,
        initial_policy: typing.Optional[typing.Sequence[_aws_cdk_aws_iam_940a1ce0.PolicyStatement]] = None,
        insights_version: typing.Optional["LambdaInsightsVersion"] = None,
        layers: typing.Optional[typing.Sequence["ILayerVersion"]] = None,
        log_retention: typing.Optional[_aws_cdk_aws_logs_6c4320fb.RetentionDays] = None,
        log_retention_retry_options: typing.Optional[typing.Union["LogRetentionRetryOptions", typing.Dict[builtins.str, typing.Any]]] = None,
        log_retention_role: typing.Optional[_aws_cdk_aws_iam_940a1ce0.IRole] = None,
        memory_size: typing.Optional[jsii.Number] = None,
        profiling: typing.Optional[builtins.bool] = None,
        profiling_group: typing.Optional[_aws_cdk_aws_codeguruprofiler_5a603484.IProfilingGroup] = None,
        reserved_concurrent_executions: typing.Optional[jsii.Number] = None,
        role: typing.Optional[_aws_cdk_aws_iam_940a1ce0.IRole] = None,
        security_group: typing.Optional[_aws_cdk_aws_ec2_67de8e8d.ISecurityGroup] = None,
        security_groups: typing.Optional[typing.Sequence[_aws_cdk_aws_ec2_67de8e8d.ISecurityGroup]] = None,
        timeout: typing.Optional[_aws_cdk_core_f4b25747.Duration] = None,
        tracing: typing.Optional["Tracing"] = None,
        vpc: typing.Optional[_aws_cdk_aws_ec2_67de8e8d.IVpc] = None,
        vpc_subnets: typing.Optional[typing.Union[_aws_cdk_aws_ec2_67de8e8d.SubnetSelection, typing.Dict[builtins.str, typing.Any]]] = None,
    ) -> None:
        '''Non runtime options.

        :param max_event_age: The maximum age of a request that Lambda sends to a function for processing. Minimum: 60 seconds Maximum: 6 hours Default: Duration.hours(6)
        :param on_failure: The destination for failed invocations. Default: - no destination
        :param on_success: The destination for successful invocations. Default: - no destination
        :param retry_attempts: The maximum number of times to retry when the function returns an error. Minimum: 0 Maximum: 2 Default: 2
        :param allow_all_outbound: Whether to allow the Lambda to send all network traffic. If set to false, you must individually add traffic rules to allow the Lambda to connect to network targets. Default: true
        :param allow_public_subnet: Lambda Functions in a public subnet can NOT access the internet. Use this property to acknowledge this limitation and still place the function in a public subnet. Default: false
        :param architecture: The system architectures compatible with this lambda function. Default: Architecture.X86_64
        :param architectures: (deprecated) DEPRECATED. Default: [Architecture.X86_64]
        :param code_signing_config: Code signing config associated with this function. Default: - Not Sign the Code
        :param current_version_options: Options for the ``lambda.Version`` resource automatically created by the ``fn.currentVersion`` method. Default: - default options as described in ``VersionOptions``
        :param dead_letter_queue: The SQS queue to use if DLQ is enabled. If SNS topic is desired, specify ``deadLetterTopic`` property instead. Default: - SQS queue with 14 day retention period if ``deadLetterQueueEnabled`` is ``true``
        :param dead_letter_queue_enabled: Enabled DLQ. If ``deadLetterQueue`` is undefined, an SQS queue with default options will be defined for your Function. Default: - false unless ``deadLetterQueue`` is set, which implies DLQ is enabled.
        :param dead_letter_topic: The SNS topic to use as a DLQ. Note that if ``deadLetterQueueEnabled`` is set to ``true``, an SQS queue will be created rather than an SNS topic. Using an SNS topic as a DLQ requires this property to be set explicitly. Default: - no SNS topic
        :param description: A description of the function. Default: - No description.
        :param environment: Key-value pairs that Lambda caches and makes available for your Lambda functions. Use environment variables to apply configuration changes, such as test and production environment configurations, without changing your Lambda function source code. Default: - No environment variables.
        :param environment_encryption: The AWS KMS key that's used to encrypt your function's environment variables. Default: - AWS Lambda creates and uses an AWS managed customer master key (CMK).
        :param ephemeral_storage_size: The size of the function’s /tmp directory in MiB. Default: 512 MiB
        :param events: Event sources for this function. You can also add event sources using ``addEventSource``. Default: - No event sources.
        :param filesystem: The filesystem configuration for the lambda function. Default: - will not mount any filesystem
        :param function_name: A name for the function. Default: - AWS CloudFormation generates a unique physical ID and uses that ID for the function's name. For more information, see Name Type.
        :param initial_policy: Initial policy statements to add to the created Lambda Role. You can call ``addToRolePolicy`` to the created lambda to add statements post creation. Default: - No policy statements are added to the created Lambda role.
        :param insights_version: Specify the version of CloudWatch Lambda insights to use for monitoring. Default: - No Lambda Insights
        :param layers: A list of layers to add to the function's execution environment. You can configure your Lambda function to pull in additional code during initialization in the form of layers. Layers are packages of libraries or other dependencies that can be used by multiple functions. Default: - No layers.
        :param log_retention: The number of days log events are kept in CloudWatch Logs. When updating this property, unsetting it doesn't remove the log retention policy. To remove the retention policy, set the value to ``INFINITE``. Default: logs.RetentionDays.INFINITE
        :param log_retention_retry_options: When log retention is specified, a custom resource attempts to create the CloudWatch log group. These options control the retry policy when interacting with CloudWatch APIs. Default: - Default AWS SDK retry options.
        :param log_retention_role: The IAM role for the Lambda function associated with the custom resource that sets the retention policy. Default: - A new role is created.
        :param memory_size: The amount of memory, in MB, that is allocated to your Lambda function. Lambda uses this value to proportionally allocate the amount of CPU power. For more information, see Resource Model in the AWS Lambda Developer Guide. Default: 128
        :param profiling: Enable profiling. Default: - No profiling.
        :param profiling_group: Profiling Group. Default: - A new profiling group will be created if ``profiling`` is set.
        :param reserved_concurrent_executions: The maximum of concurrent executions you want to reserve for the function. Default: - No specific limit - account limit.
        :param role: Lambda execution role. This is the role that will be assumed by the function upon execution. It controls the permissions that the function will have. The Role must be assumable by the 'lambda.amazonaws.com' service principal. The default Role automatically has permissions granted for Lambda execution. If you provide a Role, you must add the relevant AWS managed policies yourself. The relevant managed policies are "service-role/AWSLambdaBasicExecutionRole" and "service-role/AWSLambdaVPCAccessExecutionRole". Default: - A unique role will be generated for this lambda function. Both supplied and generated roles can always be changed by calling ``addToRolePolicy``.
        :param security_group: (deprecated) What security group to associate with the Lambda's network interfaces. This property is being deprecated, consider using securityGroups instead. Only used if 'vpc' is supplied. Use securityGroups property instead. Function constructor will throw an error if both are specified. Default: - If the function is placed within a VPC and a security group is not specified, either by this or securityGroups prop, a dedicated security group will be created for this function.
        :param security_groups: The list of security groups to associate with the Lambda's network interfaces. Only used if 'vpc' is supplied. Default: - If the function is placed within a VPC and a security group is not specified, either by this or securityGroup prop, a dedicated security group will be created for this function.
        :param timeout: The function execution time (in seconds) after which Lambda terminates the function. Because the execution time affects cost, set this value based on the function's expected execution time. Default: Duration.seconds(3)
        :param tracing: Enable AWS X-Ray Tracing for Lambda Function. Default: Tracing.Disabled
        :param vpc: VPC network to place Lambda network interfaces. Specify this if the Lambda function needs to access resources in a VPC. Default: - Function is not placed within a VPC.
        :param vpc_subnets: Where to place the network interfaces within the VPC. Only used if 'vpc' is supplied. Note: internet access for Lambdas requires a NAT gateway, so picking Public subnets is not allowed. Default: - the Vpc default strategy if not specified

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_codeguruprofiler as codeguruprofiler
            import aws_cdk.aws_ec2 as ec2
            import aws_cdk.aws_iam as iam
            import aws_cdk.aws_kms as kms
            import aws_cdk.aws_lambda as lambda_
            import aws_cdk.aws_logs as logs
            import aws_cdk.aws_sns as sns
            import aws_cdk.aws_sqs as sqs
            import aws_cdk.core as cdk
            
            # architecture: lambda.Architecture
            # code_signing_config: lambda.CodeSigningConfig
            # destination: lambda.IDestination
            # event_source: lambda.IEventSource
            # file_system: lambda.FileSystem
            # key: kms.Key
            # lambda_insights_version: lambda.LambdaInsightsVersion
            # layer_version: lambda.LayerVersion
            # policy_statement: iam.PolicyStatement
            # profiling_group: codeguruprofiler.ProfilingGroup
            # queue: sqs.Queue
            # role: iam.Role
            # security_group: ec2.SecurityGroup
            # size: cdk.Size
            # subnet: ec2.Subnet
            # subnet_filter: ec2.SubnetFilter
            # topic: sns.Topic
            # vpc: ec2.Vpc
            
            function_options = lambda.FunctionOptions(
                allow_all_outbound=False,
                allow_public_subnet=False,
                architecture=architecture,
                architectures=[architecture],
                code_signing_config=code_signing_config,
                current_version_options=lambda.VersionOptions(
                    code_sha256="codeSha256",
                    description="description",
                    max_event_age=cdk.Duration.minutes(30),
                    on_failure=destination,
                    on_success=destination,
                    provisioned_concurrent_executions=123,
                    removal_policy=cdk.RemovalPolicy.DESTROY,
                    retry_attempts=123
                ),
                dead_letter_queue=queue,
                dead_letter_queue_enabled=False,
                dead_letter_topic=topic,
                description="description",
                environment={
                    "environment_key": "environment"
                },
                environment_encryption=key,
                ephemeral_storage_size=size,
                events=[event_source],
                filesystem=file_system,
                function_name="functionName",
                initial_policy=[policy_statement],
                insights_version=lambda_insights_version,
                layers=[layer_version],
                log_retention=logs.RetentionDays.ONE_DAY,
                log_retention_retry_options=lambda.LogRetentionRetryOptions(
                    base=cdk.Duration.minutes(30),
                    max_retries=123
                ),
                log_retention_role=role,
                max_event_age=cdk.Duration.minutes(30),
                memory_size=123,
                on_failure=destination,
                on_success=destination,
                profiling=False,
                profiling_group=profiling_group,
                reserved_concurrent_executions=123,
                retry_attempts=123,
                role=role,
                security_group=security_group,
                security_groups=[security_group],
                timeout=cdk.Duration.minutes(30),
                tracing=lambda_.Tracing.ACTIVE,
                vpc=vpc,
                vpc_subnets=ec2.SubnetSelection(
                    availability_zones=["availabilityZones"],
                    one_per_az=False,
                    subnet_filters=[subnet_filter],
                    subnet_group_name="subnetGroupName",
                    subnet_name="subnetName",
                    subnets=[subnet],
                    subnet_type=ec2.SubnetType.ISOLATED
                )
            )
        '''
        if isinstance(current_version_options, dict):
            current_version_options = VersionOptions(**current_version_options)
        if isinstance(log_retention_retry_options, dict):
            log_retention_retry_options = LogRetentionRetryOptions(**log_retention_retry_options)
        if isinstance(vpc_subnets, dict):
            vpc_subnets = _aws_cdk_aws_ec2_67de8e8d.SubnetSelection(**vpc_subnets)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__dc1476624dbb25f8138b8601afa61d345a2e2848e553b0b942f9dbc100396048)
            check_type(argname="argument max_event_age", value=max_event_age, expected_type=type_hints["max_event_age"])
            check_type(argname="argument on_failure", value=on_failure, expected_type=type_hints["on_failure"])
            check_type(argname="argument on_success", value=on_success, expected_type=type_hints["on_success"])
            check_type(argname="argument retry_attempts", value=retry_attempts, expected_type=type_hints["retry_attempts"])
            check_type(argname="argument allow_all_outbound", value=allow_all_outbound, expected_type=type_hints["allow_all_outbound"])
            check_type(argname="argument allow_public_subnet", value=allow_public_subnet, expected_type=type_hints["allow_public_subnet"])
            check_type(argname="argument architecture", value=architecture, expected_type=type_hints["architecture"])
            check_type(argname="argument architectures", value=architectures, expected_type=type_hints["architectures"])
            check_type(argname="argument code_signing_config", value=code_signing_config, expected_type=type_hints["code_signing_config"])
            check_type(argname="argument current_version_options", value=current_version_options, expected_type=type_hints["current_version_options"])
            check_type(argname="argument dead_letter_queue", value=dead_letter_queue, expected_type=type_hints["dead_letter_queue"])
            check_type(argname="argument dead_letter_queue_enabled", value=dead_letter_queue_enabled, expected_type=type_hints["dead_letter_queue_enabled"])
            check_type(argname="argument dead_letter_topic", value=dead_letter_topic, expected_type=type_hints["dead_letter_topic"])
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument environment", value=environment, expected_type=type_hints["environment"])
            check_type(argname="argument environment_encryption", value=environment_encryption, expected_type=type_hints["environment_encryption"])
            check_type(argname="argument ephemeral_storage_size", value=ephemeral_storage_size, expected_type=type_hints["ephemeral_storage_size"])
            check_type(argname="argument events", value=events, expected_type=type_hints["events"])
            check_type(argname="argument filesystem", value=filesystem, expected_type=type_hints["filesystem"])
            check_type(argname="argument function_name", value=function_name, expected_type=type_hints["function_name"])
            check_type(argname="argument initial_policy", value=initial_policy, expected_type=type_hints["initial_policy"])
            check_type(argname="argument insights_version", value=insights_version, expected_type=type_hints["insights_version"])
            check_type(argname="argument layers", value=layers, expected_type=type_hints["layers"])
            check_type(argname="argument log_retention", value=log_retention, expected_type=type_hints["log_retention"])
            check_type(argname="argument log_retention_retry_options", value=log_retention_retry_options, expected_type=type_hints["log_retention_retry_options"])
            check_type(argname="argument log_retention_role", value=log_retention_role, expected_type=type_hints["log_retention_role"])
            check_type(argname="argument memory_size", value=memory_size, expected_type=type_hints["memory_size"])
            check_type(argname="argument profiling", value=profiling, expected_type=type_hints["profiling"])
            check_type(argname="argument profiling_group", value=profiling_group, expected_type=type_hints["profiling_group"])
            check_type(argname="argument reserved_concurrent_executions", value=reserved_concurrent_executions, expected_type=type_hints["reserved_concurrent_executions"])
            check_type(argname="argument role", value=role, expected_type=type_hints["role"])
            check_type(argname="argument security_group", value=security_group, expected_type=type_hints["security_group"])
            check_type(argname="argument security_groups", value=security_groups, expected_type=type_hints["security_groups"])
            check_type(argname="argument timeout", value=timeout, expected_type=type_hints["timeout"])
            check_type(argname="argument tracing", value=tracing, expected_type=type_hints["tracing"])
            check_type(argname="argument vpc", value=vpc, expected_type=type_hints["vpc"])
            check_type(argname="argument vpc_subnets", value=vpc_subnets, expected_type=type_hints["vpc_subnets"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if max_event_age is not None:
            self._values["max_event_age"] = max_event_age
        if on_failure is not None:
            self._values["on_failure"] = on_failure
        if on_success is not None:
            self._values["on_success"] = on_success
        if retry_attempts is not None:
            self._values["retry_attempts"] = retry_attempts
        if allow_all_outbound is not None:
            self._values["allow_all_outbound"] = allow_all_outbound
        if allow_public_subnet is not None:
            self._values["allow_public_subnet"] = allow_public_subnet
        if architecture is not None:
            self._values["architecture"] = architecture
        if architectures is not None:
            self._values["architectures"] = architectures
        if code_signing_config is not None:
            self._values["code_signing_config"] = code_signing_config
        if current_version_options is not None:
            self._values["current_version_options"] = current_version_options
        if dead_letter_queue is not None:
            self._values["dead_letter_queue"] = dead_letter_queue
        if dead_letter_queue_enabled is not None:
            self._values["dead_letter_queue_enabled"] = dead_letter_queue_enabled
        if dead_letter_topic is not None:
            self._values["dead_letter_topic"] = dead_letter_topic
        if description is not None:
            self._values["description"] = description
        if environment is not None:
            self._values["environment"] = environment
        if environment_encryption is not None:
            self._values["environment_encryption"] = environment_encryption
        if ephemeral_storage_size is not None:
            self._values["ephemeral_storage_size"] = ephemeral_storage_size
        if events is not None:
            self._values["events"] = events
        if filesystem is not None:
            self._values["filesystem"] = filesystem
        if function_name is not None:
            self._values["function_name"] = function_name
        if initial_policy is not None:
            self._values["initial_policy"] = initial_policy
        if insights_version is not None:
            self._values["insights_version"] = insights_version
        if layers is not None:
            self._values["layers"] = layers
        if log_retention is not None:
            self._values["log_retention"] = log_retention
        if log_retention_retry_options is not None:
            self._values["log_retention_retry_options"] = log_retention_retry_options
        if log_retention_role is not None:
            self._values["log_retention_role"] = log_retention_role
        if memory_size is not None:
            self._values["memory_size"] = memory_size
        if profiling is not None:
            self._values["profiling"] = profiling
        if profiling_group is not None:
            self._values["profiling_group"] = profiling_group
        if reserved_concurrent_executions is not None:
            self._values["reserved_concurrent_executions"] = reserved_concurrent_executions
        if role is not None:
            self._values["role"] = role
        if security_group is not None:
            self._values["security_group"] = security_group
        if security_groups is not None:
            self._values["security_groups"] = security_groups
        if timeout is not None:
            self._values["timeout"] = timeout
        if tracing is not None:
            self._values["tracing"] = tracing
        if vpc is not None:
            self._values["vpc"] = vpc
        if vpc_subnets is not None:
            self._values["vpc_subnets"] = vpc_subnets

    @builtins.property
    def max_event_age(self) -> typing.Optional[_aws_cdk_core_f4b25747.Duration]:
        '''The maximum age of a request that Lambda sends to a function for processing.

        Minimum: 60 seconds
        Maximum: 6 hours

        :default: Duration.hours(6)
        '''
        result = self._values.get("max_event_age")
        return typing.cast(typing.Optional[_aws_cdk_core_f4b25747.Duration], result)

    @builtins.property
    def on_failure(self) -> typing.Optional["IDestination"]:
        '''The destination for failed invocations.

        :default: - no destination
        '''
        result = self._values.get("on_failure")
        return typing.cast(typing.Optional["IDestination"], result)

    @builtins.property
    def on_success(self) -> typing.Optional["IDestination"]:
        '''The destination for successful invocations.

        :default: - no destination
        '''
        result = self._values.get("on_success")
        return typing.cast(typing.Optional["IDestination"], result)

    @builtins.property
    def retry_attempts(self) -> typing.Optional[jsii.Number]:
        '''The maximum number of times to retry when the function returns an error.

        Minimum: 0
        Maximum: 2

        :default: 2
        '''
        result = self._values.get("retry_attempts")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def allow_all_outbound(self) -> typing.Optional[builtins.bool]:
        '''Whether to allow the Lambda to send all network traffic.

        If set to false, you must individually add traffic rules to allow the
        Lambda to connect to network targets.

        :default: true
        '''
        result = self._values.get("allow_all_outbound")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def allow_public_subnet(self) -> typing.Optional[builtins.bool]:
        '''Lambda Functions in a public subnet can NOT access the internet.

        Use this property to acknowledge this limitation and still place the function in a public subnet.

        :default: false

        :see: https://stackoverflow.com/questions/52992085/why-cant-an-aws-lambda-function-inside-a-public-subnet-in-a-vpc-connect-to-the/52994841#52994841
        '''
        result = self._values.get("allow_public_subnet")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def architecture(self) -> typing.Optional[Architecture]:
        '''The system architectures compatible with this lambda function.

        :default: Architecture.X86_64
        '''
        result = self._values.get("architecture")
        return typing.cast(typing.Optional[Architecture], result)

    @builtins.property
    def architectures(self) -> typing.Optional[typing.List[Architecture]]:
        '''(deprecated) DEPRECATED.

        :default: [Architecture.X86_64]

        :deprecated: use ``architecture``

        :stability: deprecated
        '''
        result = self._values.get("architectures")
        return typing.cast(typing.Optional[typing.List[Architecture]], result)

    @builtins.property
    def code_signing_config(self) -> typing.Optional["ICodeSigningConfig"]:
        '''Code signing config associated with this function.

        :default: - Not Sign the Code
        '''
        result = self._values.get("code_signing_config")
        return typing.cast(typing.Optional["ICodeSigningConfig"], result)

    @builtins.property
    def current_version_options(self) -> typing.Optional["VersionOptions"]:
        '''Options for the ``lambda.Version`` resource automatically created by the ``fn.currentVersion`` method.

        :default: - default options as described in ``VersionOptions``
        '''
        result = self._values.get("current_version_options")
        return typing.cast(typing.Optional["VersionOptions"], result)

    @builtins.property
    def dead_letter_queue(self) -> typing.Optional[_aws_cdk_aws_sqs_48bffef9.IQueue]:
        '''The SQS queue to use if DLQ is enabled.

        If SNS topic is desired, specify ``deadLetterTopic`` property instead.

        :default: - SQS queue with 14 day retention period if ``deadLetterQueueEnabled`` is ``true``
        '''
        result = self._values.get("dead_letter_queue")
        return typing.cast(typing.Optional[_aws_cdk_aws_sqs_48bffef9.IQueue], result)

    @builtins.property
    def dead_letter_queue_enabled(self) -> typing.Optional[builtins.bool]:
        '''Enabled DLQ.

        If ``deadLetterQueue`` is undefined,
        an SQS queue with default options will be defined for your Function.

        :default: - false unless ``deadLetterQueue`` is set, which implies DLQ is enabled.
        '''
        result = self._values.get("dead_letter_queue_enabled")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def dead_letter_topic(self) -> typing.Optional[_aws_cdk_aws_sns_889c7272.ITopic]:
        '''The SNS topic to use as a DLQ.

        Note that if ``deadLetterQueueEnabled`` is set to ``true``, an SQS queue will be created
        rather than an SNS topic. Using an SNS topic as a DLQ requires this property to be set explicitly.

        :default: - no SNS topic
        '''
        result = self._values.get("dead_letter_topic")
        return typing.cast(typing.Optional[_aws_cdk_aws_sns_889c7272.ITopic], result)

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''A description of the function.

        :default: - No description.
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def environment(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''Key-value pairs that Lambda caches and makes available for your Lambda functions.

        Use environment variables to apply configuration changes, such
        as test and production environment configurations, without changing your
        Lambda function source code.

        :default: - No environment variables.
        '''
        result = self._values.get("environment")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    @builtins.property
    def environment_encryption(self) -> typing.Optional[_aws_cdk_aws_kms_e491a92b.IKey]:
        '''The AWS KMS key that's used to encrypt your function's environment variables.

        :default: - AWS Lambda creates and uses an AWS managed customer master key (CMK).
        '''
        result = self._values.get("environment_encryption")
        return typing.cast(typing.Optional[_aws_cdk_aws_kms_e491a92b.IKey], result)

    @builtins.property
    def ephemeral_storage_size(self) -> typing.Optional[_aws_cdk_core_f4b25747.Size]:
        '''The size of the function’s /tmp directory in MiB.

        :default: 512 MiB
        '''
        result = self._values.get("ephemeral_storage_size")
        return typing.cast(typing.Optional[_aws_cdk_core_f4b25747.Size], result)

    @builtins.property
    def events(self) -> typing.Optional[typing.List["IEventSource"]]:
        '''Event sources for this function.

        You can also add event sources using ``addEventSource``.

        :default: - No event sources.
        '''
        result = self._values.get("events")
        return typing.cast(typing.Optional[typing.List["IEventSource"]], result)

    @builtins.property
    def filesystem(self) -> typing.Optional[FileSystem]:
        '''The filesystem configuration for the lambda function.

        :default: - will not mount any filesystem
        '''
        result = self._values.get("filesystem")
        return typing.cast(typing.Optional[FileSystem], result)

    @builtins.property
    def function_name(self) -> typing.Optional[builtins.str]:
        '''A name for the function.

        :default:

        - AWS CloudFormation generates a unique physical ID and uses that
        ID for the function's name. For more information, see Name Type.
        '''
        result = self._values.get("function_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def initial_policy(
        self,
    ) -> typing.Optional[typing.List[_aws_cdk_aws_iam_940a1ce0.PolicyStatement]]:
        '''Initial policy statements to add to the created Lambda Role.

        You can call ``addToRolePolicy`` to the created lambda to add statements post creation.

        :default: - No policy statements are added to the created Lambda role.
        '''
        result = self._values.get("initial_policy")
        return typing.cast(typing.Optional[typing.List[_aws_cdk_aws_iam_940a1ce0.PolicyStatement]], result)

    @builtins.property
    def insights_version(self) -> typing.Optional["LambdaInsightsVersion"]:
        '''Specify the version of CloudWatch Lambda insights to use for monitoring.

        :default: - No Lambda Insights

        :see: https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/Lambda-Insights-Getting-Started-docker.html
        '''
        result = self._values.get("insights_version")
        return typing.cast(typing.Optional["LambdaInsightsVersion"], result)

    @builtins.property
    def layers(self) -> typing.Optional[typing.List["ILayerVersion"]]:
        '''A list of layers to add to the function's execution environment.

        You can configure your Lambda function to pull in
        additional code during initialization in the form of layers. Layers are packages of libraries or other dependencies
        that can be used by multiple functions.

        :default: - No layers.
        '''
        result = self._values.get("layers")
        return typing.cast(typing.Optional[typing.List["ILayerVersion"]], result)

    @builtins.property
    def log_retention(
        self,
    ) -> typing.Optional[_aws_cdk_aws_logs_6c4320fb.RetentionDays]:
        '''The number of days log events are kept in CloudWatch Logs.

        When updating
        this property, unsetting it doesn't remove the log retention policy. To
        remove the retention policy, set the value to ``INFINITE``.

        :default: logs.RetentionDays.INFINITE
        '''
        result = self._values.get("log_retention")
        return typing.cast(typing.Optional[_aws_cdk_aws_logs_6c4320fb.RetentionDays], result)

    @builtins.property
    def log_retention_retry_options(
        self,
    ) -> typing.Optional["LogRetentionRetryOptions"]:
        '''When log retention is specified, a custom resource attempts to create the CloudWatch log group.

        These options control the retry policy when interacting with CloudWatch APIs.

        :default: - Default AWS SDK retry options.
        '''
        result = self._values.get("log_retention_retry_options")
        return typing.cast(typing.Optional["LogRetentionRetryOptions"], result)

    @builtins.property
    def log_retention_role(self) -> typing.Optional[_aws_cdk_aws_iam_940a1ce0.IRole]:
        '''The IAM role for the Lambda function associated with the custom resource that sets the retention policy.

        :default: - A new role is created.
        '''
        result = self._values.get("log_retention_role")
        return typing.cast(typing.Optional[_aws_cdk_aws_iam_940a1ce0.IRole], result)

    @builtins.property
    def memory_size(self) -> typing.Optional[jsii.Number]:
        '''The amount of memory, in MB, that is allocated to your Lambda function.

        Lambda uses this value to proportionally allocate the amount of CPU
        power. For more information, see Resource Model in the AWS Lambda
        Developer Guide.

        :default: 128
        '''
        result = self._values.get("memory_size")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def profiling(self) -> typing.Optional[builtins.bool]:
        '''Enable profiling.

        :default: - No profiling.

        :see: https://docs.aws.amazon.com/codeguru/latest/profiler-ug/setting-up-lambda.html
        '''
        result = self._values.get("profiling")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def profiling_group(
        self,
    ) -> typing.Optional[_aws_cdk_aws_codeguruprofiler_5a603484.IProfilingGroup]:
        '''Profiling Group.

        :default: - A new profiling group will be created if ``profiling`` is set.

        :see: https://docs.aws.amazon.com/codeguru/latest/profiler-ug/setting-up-lambda.html
        '''
        result = self._values.get("profiling_group")
        return typing.cast(typing.Optional[_aws_cdk_aws_codeguruprofiler_5a603484.IProfilingGroup], result)

    @builtins.property
    def reserved_concurrent_executions(self) -> typing.Optional[jsii.Number]:
        '''The maximum of concurrent executions you want to reserve for the function.

        :default: - No specific limit - account limit.

        :see: https://docs.aws.amazon.com/lambda/latest/dg/concurrent-executions.html
        '''
        result = self._values.get("reserved_concurrent_executions")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def role(self) -> typing.Optional[_aws_cdk_aws_iam_940a1ce0.IRole]:
        '''Lambda execution role.

        This is the role that will be assumed by the function upon execution.
        It controls the permissions that the function will have. The Role must
        be assumable by the 'lambda.amazonaws.com' service principal.

        The default Role automatically has permissions granted for Lambda execution. If you
        provide a Role, you must add the relevant AWS managed policies yourself.

        The relevant managed policies are "service-role/AWSLambdaBasicExecutionRole" and
        "service-role/AWSLambdaVPCAccessExecutionRole".

        :default:

        - A unique role will be generated for this lambda function.
        Both supplied and generated roles can always be changed by calling ``addToRolePolicy``.
        '''
        result = self._values.get("role")
        return typing.cast(typing.Optional[_aws_cdk_aws_iam_940a1ce0.IRole], result)

    @builtins.property
    def security_group(
        self,
    ) -> typing.Optional[_aws_cdk_aws_ec2_67de8e8d.ISecurityGroup]:
        '''(deprecated) What security group to associate with the Lambda's network interfaces. This property is being deprecated, consider using securityGroups instead.

        Only used if 'vpc' is supplied.

        Use securityGroups property instead.
        Function constructor will throw an error if both are specified.

        :default:

        - If the function is placed within a VPC and a security group is
        not specified, either by this or securityGroups prop, a dedicated security
        group will be created for this function.

        :deprecated: - This property is deprecated, use securityGroups instead

        :stability: deprecated
        '''
        result = self._values.get("security_group")
        return typing.cast(typing.Optional[_aws_cdk_aws_ec2_67de8e8d.ISecurityGroup], result)

    @builtins.property
    def security_groups(
        self,
    ) -> typing.Optional[typing.List[_aws_cdk_aws_ec2_67de8e8d.ISecurityGroup]]:
        '''The list of security groups to associate with the Lambda's network interfaces.

        Only used if 'vpc' is supplied.

        :default:

        - If the function is placed within a VPC and a security group is
        not specified, either by this or securityGroup prop, a dedicated security
        group will be created for this function.
        '''
        result = self._values.get("security_groups")
        return typing.cast(typing.Optional[typing.List[_aws_cdk_aws_ec2_67de8e8d.ISecurityGroup]], result)

    @builtins.property
    def timeout(self) -> typing.Optional[_aws_cdk_core_f4b25747.Duration]:
        '''The function execution time (in seconds) after which Lambda terminates the function.

        Because the execution time affects cost, set this value
        based on the function's expected execution time.

        :default: Duration.seconds(3)
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[_aws_cdk_core_f4b25747.Duration], result)

    @builtins.property
    def tracing(self) -> typing.Optional["Tracing"]:
        '''Enable AWS X-Ray Tracing for Lambda Function.

        :default: Tracing.Disabled
        '''
        result = self._values.get("tracing")
        return typing.cast(typing.Optional["Tracing"], result)

    @builtins.property
    def vpc(self) -> typing.Optional[_aws_cdk_aws_ec2_67de8e8d.IVpc]:
        '''VPC network to place Lambda network interfaces.

        Specify this if the Lambda function needs to access resources in a VPC.

        :default: - Function is not placed within a VPC.
        '''
        result = self._values.get("vpc")
        return typing.cast(typing.Optional[_aws_cdk_aws_ec2_67de8e8d.IVpc], result)

    @builtins.property
    def vpc_subnets(self) -> typing.Optional[_aws_cdk_aws_ec2_67de8e8d.SubnetSelection]:
        '''Where to place the network interfaces within the VPC.

        Only used if 'vpc' is supplied. Note: internet access for Lambdas
        requires a NAT gateway, so picking Public subnets is not allowed.

        :default: - the Vpc default strategy if not specified
        '''
        result = self._values.get("vpc_subnets")
        return typing.cast(typing.Optional[_aws_cdk_aws_ec2_67de8e8d.SubnetSelection], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "FunctionOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-lambda.FunctionProps",
    jsii_struct_bases=[FunctionOptions],
    name_mapping={
        "max_event_age": "maxEventAge",
        "on_failure": "onFailure",
        "on_success": "onSuccess",
        "retry_attempts": "retryAttempts",
        "allow_all_outbound": "allowAllOutbound",
        "allow_public_subnet": "allowPublicSubnet",
        "architecture": "architecture",
        "architectures": "architectures",
        "code_signing_config": "codeSigningConfig",
        "current_version_options": "currentVersionOptions",
        "dead_letter_queue": "deadLetterQueue",
        "dead_letter_queue_enabled": "deadLetterQueueEnabled",
        "dead_letter_topic": "deadLetterTopic",
        "description": "description",
        "environment": "environment",
        "environment_encryption": "environmentEncryption",
        "ephemeral_storage_size": "ephemeralStorageSize",
        "events": "events",
        "filesystem": "filesystem",
        "function_name": "functionName",
        "initial_policy": "initialPolicy",
        "insights_version": "insightsVersion",
        "layers": "layers",
        "log_retention": "logRetention",
        "log_retention_retry_options": "logRetentionRetryOptions",
        "log_retention_role": "logRetentionRole",
        "memory_size": "memorySize",
        "profiling": "profiling",
        "profiling_group": "profilingGroup",
        "reserved_concurrent_executions": "reservedConcurrentExecutions",
        "role": "role",
        "security_group": "securityGroup",
        "security_groups": "securityGroups",
        "timeout": "timeout",
        "tracing": "tracing",
        "vpc": "vpc",
        "vpc_subnets": "vpcSubnets",
        "code": "code",
        "handler": "handler",
        "runtime": "runtime",
    },
)
class FunctionProps(FunctionOptions):
    def __init__(
        self,
        *,
        max_event_age: typing.Optional[_aws_cdk_core_f4b25747.Duration] = None,
        on_failure: typing.Optional["IDestination"] = None,
        on_success: typing.Optional["IDestination"] = None,
        retry_attempts: typing.Optional[jsii.Number] = None,
        allow_all_outbound: typing.Optional[builtins.bool] = None,
        allow_public_subnet: typing.Optional[builtins.bool] = None,
        architecture: typing.Optional[Architecture] = None,
        architectures: typing.Optional[typing.Sequence[Architecture]] = None,
        code_signing_config: typing.Optional["ICodeSigningConfig"] = None,
        current_version_options: typing.Optional[typing.Union["VersionOptions", typing.Dict[builtins.str, typing.Any]]] = None,
        dead_letter_queue: typing.Optional[_aws_cdk_aws_sqs_48bffef9.IQueue] = None,
        dead_letter_queue_enabled: typing.Optional[builtins.bool] = None,
        dead_letter_topic: typing.Optional[_aws_cdk_aws_sns_889c7272.ITopic] = None,
        description: typing.Optional[builtins.str] = None,
        environment: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        environment_encryption: typing.Optional[_aws_cdk_aws_kms_e491a92b.IKey] = None,
        ephemeral_storage_size: typing.Optional[_aws_cdk_core_f4b25747.Size] = None,
        events: typing.Optional[typing.Sequence["IEventSource"]] = None,
        filesystem: typing.Optional[FileSystem] = None,
        function_name: typing.Optional[builtins.str] = None,
        initial_policy: typing.Optional[typing.Sequence[_aws_cdk_aws_iam_940a1ce0.PolicyStatement]] = None,
        insights_version: typing.Optional["LambdaInsightsVersion"] = None,
        layers: typing.Optional[typing.Sequence["ILayerVersion"]] = None,
        log_retention: typing.Optional[_aws_cdk_aws_logs_6c4320fb.RetentionDays] = None,
        log_retention_retry_options: typing.Optional[typing.Union["LogRetentionRetryOptions", typing.Dict[builtins.str, typing.Any]]] = None,
        log_retention_role: typing.Optional[_aws_cdk_aws_iam_940a1ce0.IRole] = None,
        memory_size: typing.Optional[jsii.Number] = None,
        profiling: typing.Optional[builtins.bool] = None,
        profiling_group: typing.Optional[_aws_cdk_aws_codeguruprofiler_5a603484.IProfilingGroup] = None,
        reserved_concurrent_executions: typing.Optional[jsii.Number] = None,
        role: typing.Optional[_aws_cdk_aws_iam_940a1ce0.IRole] = None,
        security_group: typing.Optional[_aws_cdk_aws_ec2_67de8e8d.ISecurityGroup] = None,
        security_groups: typing.Optional[typing.Sequence[_aws_cdk_aws_ec2_67de8e8d.ISecurityGroup]] = None,
        timeout: typing.Optional[_aws_cdk_core_f4b25747.Duration] = None,
        tracing: typing.Optional["Tracing"] = None,
        vpc: typing.Optional[_aws_cdk_aws_ec2_67de8e8d.IVpc] = None,
        vpc_subnets: typing.Optional[typing.Union[_aws_cdk_aws_ec2_67de8e8d.SubnetSelection, typing.Dict[builtins.str, typing.Any]]] = None,
        code: Code,
        handler: builtins.str,
        runtime: "Runtime",
    ) -> None:
        '''
        :param max_event_age: The maximum age of a request that Lambda sends to a function for processing. Minimum: 60 seconds Maximum: 6 hours Default: Duration.hours(6)
        :param on_failure: The destination for failed invocations. Default: - no destination
        :param on_success: The destination for successful invocations. Default: - no destination
        :param retry_attempts: The maximum number of times to retry when the function returns an error. Minimum: 0 Maximum: 2 Default: 2
        :param allow_all_outbound: Whether to allow the Lambda to send all network traffic. If set to false, you must individually add traffic rules to allow the Lambda to connect to network targets. Default: true
        :param allow_public_subnet: Lambda Functions in a public subnet can NOT access the internet. Use this property to acknowledge this limitation and still place the function in a public subnet. Default: false
        :param architecture: The system architectures compatible with this lambda function. Default: Architecture.X86_64
        :param architectures: (deprecated) DEPRECATED. Default: [Architecture.X86_64]
        :param code_signing_config: Code signing config associated with this function. Default: - Not Sign the Code
        :param current_version_options: Options for the ``lambda.Version`` resource automatically created by the ``fn.currentVersion`` method. Default: - default options as described in ``VersionOptions``
        :param dead_letter_queue: The SQS queue to use if DLQ is enabled. If SNS topic is desired, specify ``deadLetterTopic`` property instead. Default: - SQS queue with 14 day retention period if ``deadLetterQueueEnabled`` is ``true``
        :param dead_letter_queue_enabled: Enabled DLQ. If ``deadLetterQueue`` is undefined, an SQS queue with default options will be defined for your Function. Default: - false unless ``deadLetterQueue`` is set, which implies DLQ is enabled.
        :param dead_letter_topic: The SNS topic to use as a DLQ. Note that if ``deadLetterQueueEnabled`` is set to ``true``, an SQS queue will be created rather than an SNS topic. Using an SNS topic as a DLQ requires this property to be set explicitly. Default: - no SNS topic
        :param description: A description of the function. Default: - No description.
        :param environment: Key-value pairs that Lambda caches and makes available for your Lambda functions. Use environment variables to apply configuration changes, such as test and production environment configurations, without changing your Lambda function source code. Default: - No environment variables.
        :param environment_encryption: The AWS KMS key that's used to encrypt your function's environment variables. Default: - AWS Lambda creates and uses an AWS managed customer master key (CMK).
        :param ephemeral_storage_size: The size of the function’s /tmp directory in MiB. Default: 512 MiB
        :param events: Event sources for this function. You can also add event sources using ``addEventSource``. Default: - No event sources.
        :param filesystem: The filesystem configuration for the lambda function. Default: - will not mount any filesystem
        :param function_name: A name for the function. Default: - AWS CloudFormation generates a unique physical ID and uses that ID for the function's name. For more information, see Name Type.
        :param initial_policy: Initial policy statements to add to the created Lambda Role. You can call ``addToRolePolicy`` to the created lambda to add statements post creation. Default: - No policy statements are added to the created Lambda role.
        :param insights_version: Specify the version of CloudWatch Lambda insights to use for monitoring. Default: - No Lambda Insights
        :param layers: A list of layers to add to the function's execution environment. You can configure your Lambda function to pull in additional code during initialization in the form of layers. Layers are packages of libraries or other dependencies that can be used by multiple functions. Default: - No layers.
        :param log_retention: The number of days log events are kept in CloudWatch Logs. When updating this property, unsetting it doesn't remove the log retention policy. To remove the retention policy, set the value to ``INFINITE``. Default: logs.RetentionDays.INFINITE
        :param log_retention_retry_options: When log retention is specified, a custom resource attempts to create the CloudWatch log group. These options control the retry policy when interacting with CloudWatch APIs. Default: - Default AWS SDK retry options.
        :param log_retention_role: The IAM role for the Lambda function associated with the custom resource that sets the retention policy. Default: - A new role is created.
        :param memory_size: The amount of memory, in MB, that is allocated to your Lambda function. Lambda uses this value to proportionally allocate the amount of CPU power. For more information, see Resource Model in the AWS Lambda Developer Guide. Default: 128
        :param profiling: Enable profiling. Default: - No profiling.
        :param profiling_group: Profiling Group. Default: - A new profiling group will be created if ``profiling`` is set.
        :param reserved_concurrent_executions: The maximum of concurrent executions you want to reserve for the function. Default: - No specific limit - account limit.
        :param role: Lambda execution role. This is the role that will be assumed by the function upon execution. It controls the permissions that the function will have. The Role must be assumable by the 'lambda.amazonaws.com' service principal. The default Role automatically has permissions granted for Lambda execution. If you provide a Role, you must add the relevant AWS managed policies yourself. The relevant managed policies are "service-role/AWSLambdaBasicExecutionRole" and "service-role/AWSLambdaVPCAccessExecutionRole". Default: - A unique role will be generated for this lambda function. Both supplied and generated roles can always be changed by calling ``addToRolePolicy``.
        :param security_group: (deprecated) What security group to associate with the Lambda's network interfaces. This property is being deprecated, consider using securityGroups instead. Only used if 'vpc' is supplied. Use securityGroups property instead. Function constructor will throw an error if both are specified. Default: - If the function is placed within a VPC and a security group is not specified, either by this or securityGroups prop, a dedicated security group will be created for this function.
        :param security_groups: The list of security groups to associate with the Lambda's network interfaces. Only used if 'vpc' is supplied. Default: - If the function is placed within a VPC and a security group is not specified, either by this or securityGroup prop, a dedicated security group will be created for this function.
        :param timeout: The function execution time (in seconds) after which Lambda terminates the function. Because the execution time affects cost, set this value based on the function's expected execution time. Default: Duration.seconds(3)
        :param tracing: Enable AWS X-Ray Tracing for Lambda Function. Default: Tracing.Disabled
        :param vpc: VPC network to place Lambda network interfaces. Specify this if the Lambda function needs to access resources in a VPC. Default: - Function is not placed within a VPC.
        :param vpc_subnets: Where to place the network interfaces within the VPC. Only used if 'vpc' is supplied. Note: internet access for Lambdas requires a NAT gateway, so picking Public subnets is not allowed. Default: - the Vpc default strategy if not specified
        :param code: The source code of your Lambda function. You can point to a file in an Amazon Simple Storage Service (Amazon S3) bucket or specify your source code as inline text.
        :param handler: The name of the method within your code that Lambda calls to execute your function. The format includes the file name. It can also include namespaces and other qualifiers, depending on the runtime. For more information, see https://docs.aws.amazon.com/lambda/latest/dg/gettingstarted-features.html#gettingstarted-features-programmingmodel. Use ``Handler.FROM_IMAGE`` when defining a function from a Docker image. NOTE: If you specify your source code as inline text by specifying the ZipFile property within the Code property, specify index.function_name as the handler.
        :param runtime: The runtime environment for the Lambda function that you are uploading. For valid values, see the Runtime property in the AWS Lambda Developer Guide. Use ``Runtime.FROM_IMAGE`` when when defining a function from a Docker image.

        :exampleMetadata: infused

        Example::

            import aws_cdk.aws_lambda as lambda_
            
            
            fn = lambda_.Function(self, "MyFunc",
                runtime=lambda_.Runtime.NODEJS_14_X,
                handler="index.handler",
                code=lambda_.Code.from_inline("exports.handler = handler.toString()")
            )
            
            rule = events.Rule(self, "rule",
                event_pattern=events.EventPattern(
                    source=["aws.ec2"]
                )
            )
            
            queue = sqs.Queue(self, "Queue")
            
            rule.add_target(targets.LambdaFunction(fn,
                dead_letter_queue=queue,  # Optional: add a dead letter queue
                max_event_age=cdk.Duration.hours(2),  # Optional: set the maxEventAge retry policy
                retry_attempts=2
            ))
        '''
        if isinstance(current_version_options, dict):
            current_version_options = VersionOptions(**current_version_options)
        if isinstance(log_retention_retry_options, dict):
            log_retention_retry_options = LogRetentionRetryOptions(**log_retention_retry_options)
        if isinstance(vpc_subnets, dict):
            vpc_subnets = _aws_cdk_aws_ec2_67de8e8d.SubnetSelection(**vpc_subnets)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__58580562bfac81d2b5782bf7318055ebe4e796a40771c488bdc5437dfd37de73)
            check_type(argname="argument max_event_age", value=max_event_age, expected_type=type_hints["max_event_age"])
            check_type(argname="argument on_failure", value=on_failure, expected_type=type_hints["on_failure"])
            check_type(argname="argument on_success", value=on_success, expected_type=type_hints["on_success"])
            check_type(argname="argument retry_attempts", value=retry_attempts, expected_type=type_hints["retry_attempts"])
            check_type(argname="argument allow_all_outbound", value=allow_all_outbound, expected_type=type_hints["allow_all_outbound"])
            check_type(argname="argument allow_public_subnet", value=allow_public_subnet, expected_type=type_hints["allow_public_subnet"])
            check_type(argname="argument architecture", value=architecture, expected_type=type_hints["architecture"])
            check_type(argname="argument architectures", value=architectures, expected_type=type_hints["architectures"])
            check_type(argname="argument code_signing_config", value=code_signing_config, expected_type=type_hints["code_signing_config"])
            check_type(argname="argument current_version_options", value=current_version_options, expected_type=type_hints["current_version_options"])
            check_type(argname="argument dead_letter_queue", value=dead_letter_queue, expected_type=type_hints["dead_letter_queue"])
            check_type(argname="argument dead_letter_queue_enabled", value=dead_letter_queue_enabled, expected_type=type_hints["dead_letter_queue_enabled"])
            check_type(argname="argument dead_letter_topic", value=dead_letter_topic, expected_type=type_hints["dead_letter_topic"])
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument environment", value=environment, expected_type=type_hints["environment"])
            check_type(argname="argument environment_encryption", value=environment_encryption, expected_type=type_hints["environment_encryption"])
            check_type(argname="argument ephemeral_storage_size", value=ephemeral_storage_size, expected_type=type_hints["ephemeral_storage_size"])
            check_type(argname="argument events", value=events, expected_type=type_hints["events"])
            check_type(argname="argument filesystem", value=filesystem, expected_type=type_hints["filesystem"])
            check_type(argname="argument function_name", value=function_name, expected_type=type_hints["function_name"])
            check_type(argname="argument initial_policy", value=initial_policy, expected_type=type_hints["initial_policy"])
            check_type(argname="argument insights_version", value=insights_version, expected_type=type_hints["insights_version"])
            check_type(argname="argument layers", value=layers, expected_type=type_hints["layers"])
            check_type(argname="argument log_retention", value=log_retention, expected_type=type_hints["log_retention"])
            check_type(argname="argument log_retention_retry_options", value=log_retention_retry_options, expected_type=type_hints["log_retention_retry_options"])
            check_type(argname="argument log_retention_role", value=log_retention_role, expected_type=type_hints["log_retention_role"])
            check_type(argname="argument memory_size", value=memory_size, expected_type=type_hints["memory_size"])
            check_type(argname="argument profiling", value=profiling, expected_type=type_hints["profiling"])
            check_type(argname="argument profiling_group", value=profiling_group, expected_type=type_hints["profiling_group"])
            check_type(argname="argument reserved_concurrent_executions", value=reserved_concurrent_executions, expected_type=type_hints["reserved_concurrent_executions"])
            check_type(argname="argument role", value=role, expected_type=type_hints["role"])
            check_type(argname="argument security_group", value=security_group, expected_type=type_hints["security_group"])
            check_type(argname="argument security_groups", value=security_groups, expected_type=type_hints["security_groups"])
            check_type(argname="argument timeout", value=timeout, expected_type=type_hints["timeout"])
            check_type(argname="argument tracing", value=tracing, expected_type=type_hints["tracing"])
            check_type(argname="argument vpc", value=vpc, expected_type=type_hints["vpc"])
            check_type(argname="argument vpc_subnets", value=vpc_subnets, expected_type=type_hints["vpc_subnets"])
            check_type(argname="argument code", value=code, expected_type=type_hints["code"])
            check_type(argname="argument handler", value=handler, expected_type=type_hints["handler"])
            check_type(argname="argument runtime", value=runtime, expected_type=type_hints["runtime"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "code": code,
            "handler": handler,
            "runtime": runtime,
        }
        if max_event_age is not None:
            self._values["max_event_age"] = max_event_age
        if on_failure is not None:
            self._values["on_failure"] = on_failure
        if on_success is not None:
            self._values["on_success"] = on_success
        if retry_attempts is not None:
            self._values["retry_attempts"] = retry_attempts
        if allow_all_outbound is not None:
            self._values["allow_all_outbound"] = allow_all_outbound
        if allow_public_subnet is not None:
            self._values["allow_public_subnet"] = allow_public_subnet
        if architecture is not None:
            self._values["architecture"] = architecture
        if architectures is not None:
            self._values["architectures"] = architectures
        if code_signing_config is not None:
            self._values["code_signing_config"] = code_signing_config
        if current_version_options is not None:
            self._values["current_version_options"] = current_version_options
        if dead_letter_queue is not None:
            self._values["dead_letter_queue"] = dead_letter_queue
        if dead_letter_queue_enabled is not None:
            self._values["dead_letter_queue_enabled"] = dead_letter_queue_enabled
        if dead_letter_topic is not None:
            self._values["dead_letter_topic"] = dead_letter_topic
        if description is not None:
            self._values["description"] = description
        if environment is not None:
            self._values["environment"] = environment
        if environment_encryption is not None:
            self._values["environment_encryption"] = environment_encryption
        if ephemeral_storage_size is not None:
            self._values["ephemeral_storage_size"] = ephemeral_storage_size
        if events is not None:
            self._values["events"] = events
        if filesystem is not None:
            self._values["filesystem"] = filesystem
        if function_name is not None:
            self._values["function_name"] = function_name
        if initial_policy is not None:
            self._values["initial_policy"] = initial_policy
        if insights_version is not None:
            self._values["insights_version"] = insights_version
        if layers is not None:
            self._values["layers"] = layers
        if log_retention is not None:
            self._values["log_retention"] = log_retention
        if log_retention_retry_options is not None:
            self._values["log_retention_retry_options"] = log_retention_retry_options
        if log_retention_role is not None:
            self._values["log_retention_role"] = log_retention_role
        if memory_size is not None:
            self._values["memory_size"] = memory_size
        if profiling is not None:
            self._values["profiling"] = profiling
        if profiling_group is not None:
            self._values["profiling_group"] = profiling_group
        if reserved_concurrent_executions is not None:
            self._values["reserved_concurrent_executions"] = reserved_concurrent_executions
        if role is not None:
            self._values["role"] = role
        if security_group is not None:
            self._values["security_group"] = security_group
        if security_groups is not None:
            self._values["security_groups"] = security_groups
        if timeout is not None:
            self._values["timeout"] = timeout
        if tracing is not None:
            self._values["tracing"] = tracing
        if vpc is not None:
            self._values["vpc"] = vpc
        if vpc_subnets is not None:
            self._values["vpc_subnets"] = vpc_subnets

    @builtins.property
    def max_event_age(self) -> typing.Optional[_aws_cdk_core_f4b25747.Duration]:
        '''The maximum age of a request that Lambda sends to a function for processing.

        Minimum: 60 seconds
        Maximum: 6 hours

        :default: Duration.hours(6)
        '''
        result = self._values.get("max_event_age")
        return typing.cast(typing.Optional[_aws_cdk_core_f4b25747.Duration], result)

    @builtins.property
    def on_failure(self) -> typing.Optional["IDestination"]:
        '''The destination for failed invocations.

        :default: - no destination
        '''
        result = self._values.get("on_failure")
        return typing.cast(typing.Optional["IDestination"], result)

    @builtins.property
    def on_success(self) -> typing.Optional["IDestination"]:
        '''The destination for successful invocations.

        :default: - no destination
        '''
        result = self._values.get("on_success")
        return typing.cast(typing.Optional["IDestination"], result)

    @builtins.property
    def retry_attempts(self) -> typing.Optional[jsii.Number]:
        '''The maximum number of times to retry when the function returns an error.

        Minimum: 0
        Maximum: 2

        :default: 2
        '''
        result = self._values.get("retry_attempts")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def allow_all_outbound(self) -> typing.Optional[builtins.bool]:
        '''Whether to allow the Lambda to send all network traffic.

        If set to false, you must individually add traffic rules to allow the
        Lambda to connect to network targets.

        :default: true
        '''
        result = self._values.get("allow_all_outbound")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def allow_public_subnet(self) -> typing.Optional[builtins.bool]:
        '''Lambda Functions in a public subnet can NOT access the internet.

        Use this property to acknowledge this limitation and still place the function in a public subnet.

        :default: false

        :see: https://stackoverflow.com/questions/52992085/why-cant-an-aws-lambda-function-inside-a-public-subnet-in-a-vpc-connect-to-the/52994841#52994841
        '''
        result = self._values.get("allow_public_subnet")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def architecture(self) -> typing.Optional[Architecture]:
        '''The system architectures compatible with this lambda function.

        :default: Architecture.X86_64
        '''
        result = self._values.get("architecture")
        return typing.cast(typing.Optional[Architecture], result)

    @builtins.property
    def architectures(self) -> typing.Optional[typing.List[Architecture]]:
        '''(deprecated) DEPRECATED.

        :default: [Architecture.X86_64]

        :deprecated: use ``architecture``

        :stability: deprecated
        '''
        result = self._values.get("architectures")
        return typing.cast(typing.Optional[typing.List[Architecture]], result)

    @builtins.property
    def code_signing_config(self) -> typing.Optional["ICodeSigningConfig"]:
        '''Code signing config associated with this function.

        :default: - Not Sign the Code
        '''
        result = self._values.get("code_signing_config")
        return typing.cast(typing.Optional["ICodeSigningConfig"], result)

    @builtins.property
    def current_version_options(self) -> typing.Optional["VersionOptions"]:
        '''Options for the ``lambda.Version`` resource automatically created by the ``fn.currentVersion`` method.

        :default: - default options as described in ``VersionOptions``
        '''
        result = self._values.get("current_version_options")
        return typing.cast(typing.Optional["VersionOptions"], result)

    @builtins.property
    def dead_letter_queue(self) -> typing.Optional[_aws_cdk_aws_sqs_48bffef9.IQueue]:
        '''The SQS queue to use if DLQ is enabled.

        If SNS topic is desired, specify ``deadLetterTopic`` property instead.

        :default: - SQS queue with 14 day retention period if ``deadLetterQueueEnabled`` is ``true``
        '''
        result = self._values.get("dead_letter_queue")
        return typing.cast(typing.Optional[_aws_cdk_aws_sqs_48bffef9.IQueue], result)

    @builtins.property
    def dead_letter_queue_enabled(self) -> typing.Optional[builtins.bool]:
        '''Enabled DLQ.

        If ``deadLetterQueue`` is undefined,
        an SQS queue with default options will be defined for your Function.

        :default: - false unless ``deadLetterQueue`` is set, which implies DLQ is enabled.
        '''
        result = self._values.get("dead_letter_queue_enabled")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def dead_letter_topic(self) -> typing.Optional[_aws_cdk_aws_sns_889c7272.ITopic]:
        '''The SNS topic to use as a DLQ.

        Note that if ``deadLetterQueueEnabled`` is set to ``true``, an SQS queue will be created
        rather than an SNS topic. Using an SNS topic as a DLQ requires this property to be set explicitly.

        :default: - no SNS topic
        '''
        result = self._values.get("dead_letter_topic")
        return typing.cast(typing.Optional[_aws_cdk_aws_sns_889c7272.ITopic], result)

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''A description of the function.

        :default: - No description.
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def environment(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''Key-value pairs that Lambda caches and makes available for your Lambda functions.

        Use environment variables to apply configuration changes, such
        as test and production environment configurations, without changing your
        Lambda function source code.

        :default: - No environment variables.
        '''
        result = self._values.get("environment")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    @builtins.property
    def environment_encryption(self) -> typing.Optional[_aws_cdk_aws_kms_e491a92b.IKey]:
        '''The AWS KMS key that's used to encrypt your function's environment variables.

        :default: - AWS Lambda creates and uses an AWS managed customer master key (CMK).
        '''
        result = self._values.get("environment_encryption")
        return typing.cast(typing.Optional[_aws_cdk_aws_kms_e491a92b.IKey], result)

    @builtins.property
    def ephemeral_storage_size(self) -> typing.Optional[_aws_cdk_core_f4b25747.Size]:
        '''The size of the function’s /tmp directory in MiB.

        :default: 512 MiB
        '''
        result = self._values.get("ephemeral_storage_size")
        return typing.cast(typing.Optional[_aws_cdk_core_f4b25747.Size], result)

    @builtins.property
    def events(self) -> typing.Optional[typing.List["IEventSource"]]:
        '''Event sources for this function.

        You can also add event sources using ``addEventSource``.

        :default: - No event sources.
        '''
        result = self._values.get("events")
        return typing.cast(typing.Optional[typing.List["IEventSource"]], result)

    @builtins.property
    def filesystem(self) -> typing.Optional[FileSystem]:
        '''The filesystem configuration for the lambda function.

        :default: - will not mount any filesystem
        '''
        result = self._values.get("filesystem")
        return typing.cast(typing.Optional[FileSystem], result)

    @builtins.property
    def function_name(self) -> typing.Optional[builtins.str]:
        '''A name for the function.

        :default:

        - AWS CloudFormation generates a unique physical ID and uses that
        ID for the function's name. For more information, see Name Type.
        '''
        result = self._values.get("function_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def initial_policy(
        self,
    ) -> typing.Optional[typing.List[_aws_cdk_aws_iam_940a1ce0.PolicyStatement]]:
        '''Initial policy statements to add to the created Lambda Role.

        You can call ``addToRolePolicy`` to the created lambda to add statements post creation.

        :default: - No policy statements are added to the created Lambda role.
        '''
        result = self._values.get("initial_policy")
        return typing.cast(typing.Optional[typing.List[_aws_cdk_aws_iam_940a1ce0.PolicyStatement]], result)

    @builtins.property
    def insights_version(self) -> typing.Optional["LambdaInsightsVersion"]:
        '''Specify the version of CloudWatch Lambda insights to use for monitoring.

        :default: - No Lambda Insights

        :see: https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/Lambda-Insights-Getting-Started-docker.html
        '''
        result = self._values.get("insights_version")
        return typing.cast(typing.Optional["LambdaInsightsVersion"], result)

    @builtins.property
    def layers(self) -> typing.Optional[typing.List["ILayerVersion"]]:
        '''A list of layers to add to the function's execution environment.

        You can configure your Lambda function to pull in
        additional code during initialization in the form of layers. Layers are packages of libraries or other dependencies
        that can be used by multiple functions.

        :default: - No layers.
        '''
        result = self._values.get("layers")
        return typing.cast(typing.Optional[typing.List["ILayerVersion"]], result)

    @builtins.property
    def log_retention(
        self,
    ) -> typing.Optional[_aws_cdk_aws_logs_6c4320fb.RetentionDays]:
        '''The number of days log events are kept in CloudWatch Logs.

        When updating
        this property, unsetting it doesn't remove the log retention policy. To
        remove the retention policy, set the value to ``INFINITE``.

        :default: logs.RetentionDays.INFINITE
        '''
        result = self._values.get("log_retention")
        return typing.cast(typing.Optional[_aws_cdk_aws_logs_6c4320fb.RetentionDays], result)

    @builtins.property
    def log_retention_retry_options(
        self,
    ) -> typing.Optional["LogRetentionRetryOptions"]:
        '''When log retention is specified, a custom resource attempts to create the CloudWatch log group.

        These options control the retry policy when interacting with CloudWatch APIs.

        :default: - Default AWS SDK retry options.
        '''
        result = self._values.get("log_retention_retry_options")
        return typing.cast(typing.Optional["LogRetentionRetryOptions"], result)

    @builtins.property
    def log_retention_role(self) -> typing.Optional[_aws_cdk_aws_iam_940a1ce0.IRole]:
        '''The IAM role for the Lambda function associated with the custom resource that sets the retention policy.

        :default: - A new role is created.
        '''
        result = self._values.get("log_retention_role")
        return typing.cast(typing.Optional[_aws_cdk_aws_iam_940a1ce0.IRole], result)

    @builtins.property
    def memory_size(self) -> typing.Optional[jsii.Number]:
        '''The amount of memory, in MB, that is allocated to your Lambda function.

        Lambda uses this value to proportionally allocate the amount of CPU
        power. For more information, see Resource Model in the AWS Lambda
        Developer Guide.

        :default: 128
        '''
        result = self._values.get("memory_size")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def profiling(self) -> typing.Optional[builtins.bool]:
        '''Enable profiling.

        :default: - No profiling.

        :see: https://docs.aws.amazon.com/codeguru/latest/profiler-ug/setting-up-lambda.html
        '''
        result = self._values.get("profiling")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def profiling_group(
        self,
    ) -> typing.Optional[_aws_cdk_aws_codeguruprofiler_5a603484.IProfilingGroup]:
        '''Profiling Group.

        :default: - A new profiling group will be created if ``profiling`` is set.

        :see: https://docs.aws.amazon.com/codeguru/latest/profiler-ug/setting-up-lambda.html
        '''
        result = self._values.get("profiling_group")
        return typing.cast(typing.Optional[_aws_cdk_aws_codeguruprofiler_5a603484.IProfilingGroup], result)

    @builtins.property
    def reserved_concurrent_executions(self) -> typing.Optional[jsii.Number]:
        '''The maximum of concurrent executions you want to reserve for the function.

        :default: - No specific limit - account limit.

        :see: https://docs.aws.amazon.com/lambda/latest/dg/concurrent-executions.html
        '''
        result = self._values.get("reserved_concurrent_executions")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def role(self) -> typing.Optional[_aws_cdk_aws_iam_940a1ce0.IRole]:
        '''Lambda execution role.

        This is the role that will be assumed by the function upon execution.
        It controls the permissions that the function will have. The Role must
        be assumable by the 'lambda.amazonaws.com' service principal.

        The default Role automatically has permissions granted for Lambda execution. If you
        provide a Role, you must add the relevant AWS managed policies yourself.

        The relevant managed policies are "service-role/AWSLambdaBasicExecutionRole" and
        "service-role/AWSLambdaVPCAccessExecutionRole".

        :default:

        - A unique role will be generated for this lambda function.
        Both supplied and generated roles can always be changed by calling ``addToRolePolicy``.
        '''
        result = self._values.get("role")
        return typing.cast(typing.Optional[_aws_cdk_aws_iam_940a1ce0.IRole], result)

    @builtins.property
    def security_group(
        self,
    ) -> typing.Optional[_aws_cdk_aws_ec2_67de8e8d.ISecurityGroup]:
        '''(deprecated) What security group to associate with the Lambda's network interfaces. This property is being deprecated, consider using securityGroups instead.

        Only used if 'vpc' is supplied.

        Use securityGroups property instead.
        Function constructor will throw an error if both are specified.

        :default:

        - If the function is placed within a VPC and a security group is
        not specified, either by this or securityGroups prop, a dedicated security
        group will be created for this function.

        :deprecated: - This property is deprecated, use securityGroups instead

        :stability: deprecated
        '''
        result = self._values.get("security_group")
        return typing.cast(typing.Optional[_aws_cdk_aws_ec2_67de8e8d.ISecurityGroup], result)

    @builtins.property
    def security_groups(
        self,
    ) -> typing.Optional[typing.List[_aws_cdk_aws_ec2_67de8e8d.ISecurityGroup]]:
        '''The list of security groups to associate with the Lambda's network interfaces.

        Only used if 'vpc' is supplied.

        :default:

        - If the function is placed within a VPC and a security group is
        not specified, either by this or securityGroup prop, a dedicated security
        group will be created for this function.
        '''
        result = self._values.get("security_groups")
        return typing.cast(typing.Optional[typing.List[_aws_cdk_aws_ec2_67de8e8d.ISecurityGroup]], result)

    @builtins.property
    def timeout(self) -> typing.Optional[_aws_cdk_core_f4b25747.Duration]:
        '''The function execution time (in seconds) after which Lambda terminates the function.

        Because the execution time affects cost, set this value
        based on the function's expected execution time.

        :default: Duration.seconds(3)
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[_aws_cdk_core_f4b25747.Duration], result)

    @builtins.property
    def tracing(self) -> typing.Optional["Tracing"]:
        '''Enable AWS X-Ray Tracing for Lambda Function.

        :default: Tracing.Disabled
        '''
        result = self._values.get("tracing")
        return typing.cast(typing.Optional["Tracing"], result)

    @builtins.property
    def vpc(self) -> typing.Optional[_aws_cdk_aws_ec2_67de8e8d.IVpc]:
        '''VPC network to place Lambda network interfaces.

        Specify this if the Lambda function needs to access resources in a VPC.

        :default: - Function is not placed within a VPC.
        '''
        result = self._values.get("vpc")
        return typing.cast(typing.Optional[_aws_cdk_aws_ec2_67de8e8d.IVpc], result)

    @builtins.property
    def vpc_subnets(self) -> typing.Optional[_aws_cdk_aws_ec2_67de8e8d.SubnetSelection]:
        '''Where to place the network interfaces within the VPC.

        Only used if 'vpc' is supplied. Note: internet access for Lambdas
        requires a NAT gateway, so picking Public subnets is not allowed.

        :default: - the Vpc default strategy if not specified
        '''
        result = self._values.get("vpc_subnets")
        return typing.cast(typing.Optional[_aws_cdk_aws_ec2_67de8e8d.SubnetSelection], result)

    @builtins.property
    def code(self) -> Code:
        '''The source code of your Lambda function.

        You can point to a file in an
        Amazon Simple Storage Service (Amazon S3) bucket or specify your source
        code as inline text.
        '''
        result = self._values.get("code")
        assert result is not None, "Required property 'code' is missing"
        return typing.cast(Code, result)

    @builtins.property
    def handler(self) -> builtins.str:
        '''The name of the method within your code that Lambda calls to execute your function.

        The format includes the file name. It can also include
        namespaces and other qualifiers, depending on the runtime.
        For more information, see https://docs.aws.amazon.com/lambda/latest/dg/gettingstarted-features.html#gettingstarted-features-programmingmodel.

        Use ``Handler.FROM_IMAGE`` when defining a function from a Docker image.

        NOTE: If you specify your source code as inline text by specifying the
        ZipFile property within the Code property, specify index.function_name as
        the handler.
        '''
        result = self._values.get("handler")
        assert result is not None, "Required property 'handler' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def runtime(self) -> "Runtime":
        '''The runtime environment for the Lambda function that you are uploading.

        For valid values, see the Runtime property in the AWS Lambda Developer
        Guide.

        Use ``Runtime.FROM_IMAGE`` when when defining a function from a Docker image.
        '''
        result = self._values.get("runtime")
        assert result is not None, "Required property 'runtime' is missing"
        return typing.cast("Runtime", result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "FunctionProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="@aws-cdk/aws-lambda.FunctionUrlAuthType")
class FunctionUrlAuthType(enum.Enum):
    '''The auth types for a function url.

    :exampleMetadata: infused

    Example::

        # Can be a Function or an Alias
        # fn: lambda.Function
        
        
        fn_url = fn.add_function_url(
            auth_type=lambda_.FunctionUrlAuthType.NONE
        )
        
        CfnOutput(self, "TheUrl",
            value=fn_url.url
        )
    '''

    AWS_IAM = "AWS_IAM"
    '''Restrict access to authenticated IAM users only.'''
    NONE = "NONE"
    '''Bypass IAM authentication to create a public endpoint.'''


@jsii.data_type(
    jsii_type="@aws-cdk/aws-lambda.FunctionUrlCorsOptions",
    jsii_struct_bases=[],
    name_mapping={
        "allow_credentials": "allowCredentials",
        "allowed_headers": "allowedHeaders",
        "allowed_methods": "allowedMethods",
        "allowed_origins": "allowedOrigins",
        "exposed_headers": "exposedHeaders",
        "max_age": "maxAge",
    },
)
class FunctionUrlCorsOptions:
    def __init__(
        self,
        *,
        allow_credentials: typing.Optional[builtins.bool] = None,
        allowed_headers: typing.Optional[typing.Sequence[builtins.str]] = None,
        allowed_methods: typing.Optional[typing.Sequence["HttpMethod"]] = None,
        allowed_origins: typing.Optional[typing.Sequence[builtins.str]] = None,
        exposed_headers: typing.Optional[typing.Sequence[builtins.str]] = None,
        max_age: typing.Optional[_aws_cdk_core_f4b25747.Duration] = None,
    ) -> None:
        '''Specifies a cross-origin access property for a function URL.

        :param allow_credentials: Whether to allow cookies or other credentials in requests to your function URL. Default: false
        :param allowed_headers: Headers that are specified in the Access-Control-Request-Headers header. Default: - No headers allowed.
        :param allowed_methods: An HTTP method that you allow the origin to execute. Default: - [HttpMethod.ALL]
        :param allowed_origins: One or more origins you want customers to be able to access the bucket from. Default: - No origins allowed.
        :param exposed_headers: One or more headers in the response that you want customers to be able to access from their applications. Default: - No headers exposed.
        :param max_age: The time in seconds that your browser is to cache the preflight response for the specified resource. Default: - Browser default of 5 seconds.

        :exampleMetadata: infused

        Example::

            # fn: lambda.Function
            
            
            fn.add_function_url(
                auth_type=lambda_.FunctionUrlAuthType.NONE,
                cors=lambda.FunctionUrlCorsOptions(
                    # Allow this to be called from websites on https://example.com.
                    # Can also be ['*'] to allow all domain.
                    allowed_origins=["https://example.com"]
                )
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d992ec42ea412231bd29c33cc1d51abde48b0f004974c55abd7dfe90104eca13)
            check_type(argname="argument allow_credentials", value=allow_credentials, expected_type=type_hints["allow_credentials"])
            check_type(argname="argument allowed_headers", value=allowed_headers, expected_type=type_hints["allowed_headers"])
            check_type(argname="argument allowed_methods", value=allowed_methods, expected_type=type_hints["allowed_methods"])
            check_type(argname="argument allowed_origins", value=allowed_origins, expected_type=type_hints["allowed_origins"])
            check_type(argname="argument exposed_headers", value=exposed_headers, expected_type=type_hints["exposed_headers"])
            check_type(argname="argument max_age", value=max_age, expected_type=type_hints["max_age"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if allow_credentials is not None:
            self._values["allow_credentials"] = allow_credentials
        if allowed_headers is not None:
            self._values["allowed_headers"] = allowed_headers
        if allowed_methods is not None:
            self._values["allowed_methods"] = allowed_methods
        if allowed_origins is not None:
            self._values["allowed_origins"] = allowed_origins
        if exposed_headers is not None:
            self._values["exposed_headers"] = exposed_headers
        if max_age is not None:
            self._values["max_age"] = max_age

    @builtins.property
    def allow_credentials(self) -> typing.Optional[builtins.bool]:
        '''Whether to allow cookies or other credentials in requests to your function URL.

        :default: false
        '''
        result = self._values.get("allow_credentials")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def allowed_headers(self) -> typing.Optional[typing.List[builtins.str]]:
        '''Headers that are specified in the Access-Control-Request-Headers header.

        :default: - No headers allowed.
        '''
        result = self._values.get("allowed_headers")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def allowed_methods(self) -> typing.Optional[typing.List["HttpMethod"]]:
        '''An HTTP method that you allow the origin to execute.

        :default: - [HttpMethod.ALL]
        '''
        result = self._values.get("allowed_methods")
        return typing.cast(typing.Optional[typing.List["HttpMethod"]], result)

    @builtins.property
    def allowed_origins(self) -> typing.Optional[typing.List[builtins.str]]:
        '''One or more origins you want customers to be able to access the bucket from.

        :default: - No origins allowed.
        '''
        result = self._values.get("allowed_origins")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def exposed_headers(self) -> typing.Optional[typing.List[builtins.str]]:
        '''One or more headers in the response that you want customers to be able to access from their applications.

        :default: - No headers exposed.
        '''
        result = self._values.get("exposed_headers")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def max_age(self) -> typing.Optional[_aws_cdk_core_f4b25747.Duration]:
        '''The time in seconds that your browser is to cache the preflight response for the specified resource.

        :default: - Browser default of 5 seconds.
        '''
        result = self._values.get("max_age")
        return typing.cast(typing.Optional[_aws_cdk_core_f4b25747.Duration], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "FunctionUrlCorsOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-lambda.FunctionUrlOptions",
    jsii_struct_bases=[],
    name_mapping={"auth_type": "authType", "cors": "cors"},
)
class FunctionUrlOptions:
    def __init__(
        self,
        *,
        auth_type: typing.Optional[FunctionUrlAuthType] = None,
        cors: typing.Optional[typing.Union[FunctionUrlCorsOptions, typing.Dict[builtins.str, typing.Any]]] = None,
    ) -> None:
        '''Options to add a url to a Lambda function.

        :param auth_type: The type of authentication that your function URL uses. Default: FunctionUrlAuthType.AWS_IAM
        :param cors: The cross-origin resource sharing (CORS) settings for your function URL. Default: - No CORS configuration.

        :exampleMetadata: infused

        Example::

            # Can be a Function or an Alias
            # fn: lambda.Function
            
            
            fn_url = fn.add_function_url(
                auth_type=lambda_.FunctionUrlAuthType.NONE
            )
            
            CfnOutput(self, "TheUrl",
                value=fn_url.url
            )
        '''
        if isinstance(cors, dict):
            cors = FunctionUrlCorsOptions(**cors)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__62a7d9a7dbab8eec8cfc02433fa9f6ba06d7e81dcadbf9a853afb3f39f055112)
            check_type(argname="argument auth_type", value=auth_type, expected_type=type_hints["auth_type"])
            check_type(argname="argument cors", value=cors, expected_type=type_hints["cors"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if auth_type is not None:
            self._values["auth_type"] = auth_type
        if cors is not None:
            self._values["cors"] = cors

    @builtins.property
    def auth_type(self) -> typing.Optional[FunctionUrlAuthType]:
        '''The type of authentication that your function URL uses.

        :default: FunctionUrlAuthType.AWS_IAM
        '''
        result = self._values.get("auth_type")
        return typing.cast(typing.Optional[FunctionUrlAuthType], result)

    @builtins.property
    def cors(self) -> typing.Optional[FunctionUrlCorsOptions]:
        '''The cross-origin resource sharing (CORS) settings for your function URL.

        :default: - No CORS configuration.
        '''
        result = self._values.get("cors")
        return typing.cast(typing.Optional[FunctionUrlCorsOptions], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "FunctionUrlOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-lambda.FunctionUrlProps",
    jsii_struct_bases=[FunctionUrlOptions],
    name_mapping={"auth_type": "authType", "cors": "cors", "function": "function"},
)
class FunctionUrlProps(FunctionUrlOptions):
    def __init__(
        self,
        *,
        auth_type: typing.Optional[FunctionUrlAuthType] = None,
        cors: typing.Optional[typing.Union[FunctionUrlCorsOptions, typing.Dict[builtins.str, typing.Any]]] = None,
        function: "IFunction",
    ) -> None:
        '''Properties for a FunctionUrl.

        :param auth_type: The type of authentication that your function URL uses. Default: FunctionUrlAuthType.AWS_IAM
        :param cors: The cross-origin resource sharing (CORS) settings for your function URL. Default: - No CORS configuration.
        :param function: The function to which this url refers. It can also be an ``Alias`` but not a ``Version``.

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_lambda as lambda_
            import aws_cdk.core as cdk
            
            # function_: lambda.Function
            
            function_url_props = lambda.FunctionUrlProps(
                function=function_,
            
                # the properties below are optional
                auth_type=lambda_.FunctionUrlAuthType.AWS_IAM,
                cors=lambda.FunctionUrlCorsOptions(
                    allow_credentials=False,
                    allowed_headers=["allowedHeaders"],
                    allowed_methods=[lambda_.HttpMethod.GET],
                    allowed_origins=["allowedOrigins"],
                    exposed_headers=["exposedHeaders"],
                    max_age=cdk.Duration.minutes(30)
                )
            )
        '''
        if isinstance(cors, dict):
            cors = FunctionUrlCorsOptions(**cors)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__11de3565e6dcf1233ddc5a63b146b7dc8600426b7c79759f3b8ff01844a0710d)
            check_type(argname="argument auth_type", value=auth_type, expected_type=type_hints["auth_type"])
            check_type(argname="argument cors", value=cors, expected_type=type_hints["cors"])
            check_type(argname="argument function", value=function, expected_type=type_hints["function"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "function": function,
        }
        if auth_type is not None:
            self._values["auth_type"] = auth_type
        if cors is not None:
            self._values["cors"] = cors

    @builtins.property
    def auth_type(self) -> typing.Optional[FunctionUrlAuthType]:
        '''The type of authentication that your function URL uses.

        :default: FunctionUrlAuthType.AWS_IAM
        '''
        result = self._values.get("auth_type")
        return typing.cast(typing.Optional[FunctionUrlAuthType], result)

    @builtins.property
    def cors(self) -> typing.Optional[FunctionUrlCorsOptions]:
        '''The cross-origin resource sharing (CORS) settings for your function URL.

        :default: - No CORS configuration.
        '''
        result = self._values.get("cors")
        return typing.cast(typing.Optional[FunctionUrlCorsOptions], result)

    @builtins.property
    def function(self) -> "IFunction":
        '''The function to which this url refers.

        It can also be an ``Alias`` but not a ``Version``.
        '''
        result = self._values.get("function")
        assert result is not None, "Required property 'function' is missing"
        return typing.cast("IFunction", result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "FunctionUrlProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_aws_cdk_core_f4b25747.IAspect)
class FunctionVersionUpgrade(
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-lambda.FunctionVersionUpgrade",
):
    '''Aspect for upgrading function versions when the feature flag provided feature flag present.

    This can be necessary when the feature flag
    changes the function hash, as such changes must be associated with a new
    version. This aspect will change the function description in these cases,
    which "validates" the new function hash.

    :exampleMetadata: infused

    Example::

        stack = Stack()
        Aspects.of(stack).add(lambda_.FunctionVersionUpgrade(LAMBDA_RECOGNIZE_VERSION_PROPS))
    '''

    def __init__(
        self,
        feature_flag: builtins.str,
        enabled: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''
        :param feature_flag: -
        :param enabled: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__6c732efc01c30109a52349251ad65d0649eaf07f8564b751cbdee5df99db44b8)
            check_type(argname="argument feature_flag", value=feature_flag, expected_type=type_hints["feature_flag"])
            check_type(argname="argument enabled", value=enabled, expected_type=type_hints["enabled"])
        jsii.create(self.__class__, self, [feature_flag, enabled])

    @jsii.member(jsii_name="visit")
    def visit(self, node: _aws_cdk_core_f4b25747.IConstruct) -> None:
        '''All aspects can visit an IConstruct.

        :param node: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5dca966966037daca1e1b2e8c79b197a3e66b898b5ab1ce92da4733c5131d667)
            check_type(argname="argument node", value=node, expected_type=type_hints["node"])
        return typing.cast(None, jsii.invoke(self, "visit", [node]))


class Handler(metaclass=jsii.JSIIMeta, jsii_type="@aws-cdk/aws-lambda.Handler"):
    '''Lambda function handler.'''

    @jsii.python.classproperty
    @jsii.member(jsii_name="FROM_IMAGE")
    def FROM_IMAGE(cls) -> builtins.str:
        '''A special handler when the function handler is part of a Docker image.'''
        return typing.cast(builtins.str, jsii.sget(cls, "FROM_IMAGE"))


@jsii.enum(jsii_type="@aws-cdk/aws-lambda.HttpMethod")
class HttpMethod(enum.Enum):
    '''All http request methods.'''

    GET = "GET"
    '''The GET method requests a representation of the specified resource.'''
    PUT = "PUT"
    '''The PUT method replaces all current representations of the target resource with the request payload.'''
    HEAD = "HEAD"
    '''The HEAD method asks for a response identical to that of a GET request, but without the response body.'''
    POST = "POST"
    '''The POST method is used to submit an entity to the specified resource, often causing a change in state or side effects on the server.'''
    DELETE = "DELETE"
    '''The DELETE method deletes the specified resource.'''
    PATCH = "PATCH"
    '''The PATCH method applies partial modifications to a resource.'''
    OPTIONS = "OPTIONS"
    '''The OPTIONS method describes the communication options for the target resource.'''
    ALL = "ALL"
    '''The wildcard entry to allow all methods.'''


@jsii.interface(jsii_type="@aws-cdk/aws-lambda.ICodeSigningConfig")
class ICodeSigningConfig(_aws_cdk_core_f4b25747.IResource, typing_extensions.Protocol):
    '''A Code Signing Config.'''

    @builtins.property
    @jsii.member(jsii_name="codeSigningConfigArn")
    def code_signing_config_arn(self) -> builtins.str:
        '''The ARN of Code Signing Config.

        :attribute: true
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="codeSigningConfigId")
    def code_signing_config_id(self) -> builtins.str:
        '''The id of Code Signing Config.

        :attribute: true
        '''
        ...


class _ICodeSigningConfigProxy(
    jsii.proxy_for(_aws_cdk_core_f4b25747.IResource), # type: ignore[misc]
):
    '''A Code Signing Config.'''

    __jsii_type__: typing.ClassVar[str] = "@aws-cdk/aws-lambda.ICodeSigningConfig"

    @builtins.property
    @jsii.member(jsii_name="codeSigningConfigArn")
    def code_signing_config_arn(self) -> builtins.str:
        '''The ARN of Code Signing Config.

        :attribute: true
        '''
        return typing.cast(builtins.str, jsii.get(self, "codeSigningConfigArn"))

    @builtins.property
    @jsii.member(jsii_name="codeSigningConfigId")
    def code_signing_config_id(self) -> builtins.str:
        '''The id of Code Signing Config.

        :attribute: true
        '''
        return typing.cast(builtins.str, jsii.get(self, "codeSigningConfigId"))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, ICodeSigningConfig).__jsii_proxy_class__ = lambda : _ICodeSigningConfigProxy


@jsii.interface(jsii_type="@aws-cdk/aws-lambda.IDestination")
class IDestination(typing_extensions.Protocol):
    '''A Lambda destination.'''

    @jsii.member(jsii_name="bind")
    def bind(
        self,
        scope: _aws_cdk_core_f4b25747.Construct,
        fn: "IFunction",
        *,
        type: DestinationType,
    ) -> DestinationConfig:
        '''Binds this destination to the Lambda function.

        :param scope: -
        :param fn: -
        :param type: The destination type.
        '''
        ...


class _IDestinationProxy:
    '''A Lambda destination.'''

    __jsii_type__: typing.ClassVar[str] = "@aws-cdk/aws-lambda.IDestination"

    @jsii.member(jsii_name="bind")
    def bind(
        self,
        scope: _aws_cdk_core_f4b25747.Construct,
        fn: "IFunction",
        *,
        type: DestinationType,
    ) -> DestinationConfig:
        '''Binds this destination to the Lambda function.

        :param scope: -
        :param fn: -
        :param type: The destination type.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d558d6e89e348734a75a7de7fb98ec499b3308daf0475f9df7d7a7487089f470)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument fn", value=fn, expected_type=type_hints["fn"])
        options = DestinationOptions(type=type)

        return typing.cast(DestinationConfig, jsii.invoke(self, "bind", [scope, fn, options]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IDestination).__jsii_proxy_class__ = lambda : _IDestinationProxy


@jsii.interface(jsii_type="@aws-cdk/aws-lambda.IEventSource")
class IEventSource(typing_extensions.Protocol):
    '''An abstract class which represents an AWS Lambda event source.'''

    @jsii.member(jsii_name="bind")
    def bind(self, target: "IFunction") -> None:
        '''Called by ``lambda.addEventSource`` to allow the event source to bind to this function.

        :param target: That lambda function to bind to.
        '''
        ...


class _IEventSourceProxy:
    '''An abstract class which represents an AWS Lambda event source.'''

    __jsii_type__: typing.ClassVar[str] = "@aws-cdk/aws-lambda.IEventSource"

    @jsii.member(jsii_name="bind")
    def bind(self, target: "IFunction") -> None:
        '''Called by ``lambda.addEventSource`` to allow the event source to bind to this function.

        :param target: That lambda function to bind to.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3016713e25b3564e37baee81e7c1c5af5481c850bf1c506571a264ef0c76dc4e)
            check_type(argname="argument target", value=target, expected_type=type_hints["target"])
        return typing.cast(None, jsii.invoke(self, "bind", [target]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IEventSource).__jsii_proxy_class__ = lambda : _IEventSourceProxy


@jsii.interface(jsii_type="@aws-cdk/aws-lambda.IEventSourceDlq")
class IEventSourceDlq(typing_extensions.Protocol):
    '''A DLQ for an event source.'''

    @jsii.member(jsii_name="bind")
    def bind(
        self,
        target: "IEventSourceMapping",
        target_handler: "IFunction",
    ) -> DlqDestinationConfig:
        '''Returns the DLQ destination config of the DLQ.

        :param target: -
        :param target_handler: -
        '''
        ...


class _IEventSourceDlqProxy:
    '''A DLQ for an event source.'''

    __jsii_type__: typing.ClassVar[str] = "@aws-cdk/aws-lambda.IEventSourceDlq"

    @jsii.member(jsii_name="bind")
    def bind(
        self,
        target: "IEventSourceMapping",
        target_handler: "IFunction",
    ) -> DlqDestinationConfig:
        '''Returns the DLQ destination config of the DLQ.

        :param target: -
        :param target_handler: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__7b1080d290a80dd3dc70269778d1500dd874e0c13b6100356da7629ef5cd4517)
            check_type(argname="argument target", value=target, expected_type=type_hints["target"])
            check_type(argname="argument target_handler", value=target_handler, expected_type=type_hints["target_handler"])
        return typing.cast(DlqDestinationConfig, jsii.invoke(self, "bind", [target, target_handler]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IEventSourceDlq).__jsii_proxy_class__ = lambda : _IEventSourceDlqProxy


@jsii.interface(jsii_type="@aws-cdk/aws-lambda.IEventSourceMapping")
class IEventSourceMapping(_aws_cdk_core_f4b25747.IResource, typing_extensions.Protocol):
    '''Represents an event source mapping for a lambda function.

    :see: https://docs.aws.amazon.com/lambda/latest/dg/invocation-eventsourcemapping.html
    '''

    @builtins.property
    @jsii.member(jsii_name="eventSourceMappingId")
    def event_source_mapping_id(self) -> builtins.str:
        '''The identifier for this EventSourceMapping.

        :attribute: true
        '''
        ...


class _IEventSourceMappingProxy(
    jsii.proxy_for(_aws_cdk_core_f4b25747.IResource), # type: ignore[misc]
):
    '''Represents an event source mapping for a lambda function.

    :see: https://docs.aws.amazon.com/lambda/latest/dg/invocation-eventsourcemapping.html
    '''

    __jsii_type__: typing.ClassVar[str] = "@aws-cdk/aws-lambda.IEventSourceMapping"

    @builtins.property
    @jsii.member(jsii_name="eventSourceMappingId")
    def event_source_mapping_id(self) -> builtins.str:
        '''The identifier for this EventSourceMapping.

        :attribute: true
        '''
        return typing.cast(builtins.str, jsii.get(self, "eventSourceMappingId"))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IEventSourceMapping).__jsii_proxy_class__ = lambda : _IEventSourceMappingProxy


@jsii.interface(jsii_type="@aws-cdk/aws-lambda.IFunction")
class IFunction(
    _aws_cdk_core_f4b25747.IResource,
    _aws_cdk_aws_ec2_67de8e8d.IConnectable,
    _aws_cdk_aws_iam_940a1ce0.IGrantable,
    typing_extensions.Protocol,
):
    @builtins.property
    @jsii.member(jsii_name="architecture")
    def architecture(self) -> Architecture:
        '''The system architectures compatible with this lambda function.'''
        ...

    @builtins.property
    @jsii.member(jsii_name="functionArn")
    def function_arn(self) -> builtins.str:
        '''The ARN of the function.

        :attribute: true
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="functionName")
    def function_name(self) -> builtins.str:
        '''The name of the function.

        :attribute: true
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="isBoundToVpc")
    def is_bound_to_vpc(self) -> builtins.bool:
        '''Whether or not this Lambda function was bound to a VPC.

        If this is is ``false``, trying to access the ``connections`` object will fail.
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="latestVersion")
    def latest_version(self) -> "IVersion":
        '''The ``$LATEST`` version of this function.

        Note that this is reference to a non-specific AWS Lambda version, which
        means the function this version refers to can return different results in
        different invocations.

        To obtain a reference to an explicit version which references the current
        function configuration, use ``lambdaFunction.currentVersion`` instead.
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="permissionsNode")
    def permissions_node(self) -> _aws_cdk_core_f4b25747.ConstructNode:
        '''The construct node where permissions are attached.'''
        ...

    @builtins.property
    @jsii.member(jsii_name="resourceArnsForGrantInvoke")
    def resource_arns_for_grant_invoke(self) -> typing.List[builtins.str]:
        '''The ARN(s) to put into the resource field of the generated IAM policy for grantInvoke().

        This property is for cdk modules to consume only. You should not need to use this property.
        Instead, use grantInvoke() directly.
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="role")
    def role(self) -> typing.Optional[_aws_cdk_aws_iam_940a1ce0.IRole]:
        '''The IAM role associated with this function.'''
        ...

    @jsii.member(jsii_name="addEventSource")
    def add_event_source(self, source: IEventSource) -> None:
        '''Adds an event source to this function.

        Event sources are implemented in the @aws-cdk/aws-lambda-event-sources module.

        The following example adds an SQS Queue as an event source::

           import { SqsEventSource } from '@aws-cdk/aws-lambda-event-sources';
           myFunction.addEventSource(new SqsEventSource(myQueue));

        :param source: -
        '''
        ...

    @jsii.member(jsii_name="addEventSourceMapping")
    def add_event_source_mapping(
        self,
        id: builtins.str,
        *,
        batch_size: typing.Optional[jsii.Number] = None,
        bisect_batch_on_error: typing.Optional[builtins.bool] = None,
        enabled: typing.Optional[builtins.bool] = None,
        event_source_arn: typing.Optional[builtins.str] = None,
        kafka_bootstrap_servers: typing.Optional[typing.Sequence[builtins.str]] = None,
        kafka_topic: typing.Optional[builtins.str] = None,
        max_batching_window: typing.Optional[_aws_cdk_core_f4b25747.Duration] = None,
        max_record_age: typing.Optional[_aws_cdk_core_f4b25747.Duration] = None,
        on_failure: typing.Optional[IEventSourceDlq] = None,
        parallelization_factor: typing.Optional[jsii.Number] = None,
        report_batch_item_failures: typing.Optional[builtins.bool] = None,
        retry_attempts: typing.Optional[jsii.Number] = None,
        source_access_configurations: typing.Optional[typing.Sequence[typing.Union["SourceAccessConfiguration", typing.Dict[builtins.str, typing.Any]]]] = None,
        starting_position: typing.Optional["StartingPosition"] = None,
        tumbling_window: typing.Optional[_aws_cdk_core_f4b25747.Duration] = None,
    ) -> "EventSourceMapping":
        '''Adds an event source that maps to this AWS Lambda function.

        :param id: construct ID.
        :param batch_size: The largest number of records that AWS Lambda will retrieve from your event source at the time of invoking your function. Your function receives an event with all the retrieved records. Valid Range: Minimum value of 1. Maximum value of 10000. Default: - Amazon Kinesis, Amazon DynamoDB, and Amazon MSK is 100 records. The default for Amazon SQS is 10 messages. For standard SQS queues, the maximum is 10,000. For FIFO SQS queues, the maximum is 10.
        :param bisect_batch_on_error: If the function returns an error, split the batch in two and retry. Default: false
        :param enabled: Set to false to disable the event source upon creation. Default: true
        :param event_source_arn: The Amazon Resource Name (ARN) of the event source. Any record added to this stream can invoke the Lambda function. Default: - not set if using a self managed Kafka cluster, throws an error otherwise
        :param kafka_bootstrap_servers: A list of host and port pairs that are the addresses of the Kafka brokers in a self managed "bootstrap" Kafka cluster that a Kafka client connects to initially to bootstrap itself. They are in the format ``abc.example.com:9096``. Default: - none
        :param kafka_topic: The name of the Kafka topic. Default: - no topic
        :param max_batching_window: The maximum amount of time to gather records before invoking the function. Maximum of Duration.minutes(5) Default: Duration.seconds(0)
        :param max_record_age: The maximum age of a record that Lambda sends to a function for processing. Valid Range: - Minimum value of 60 seconds - Maximum value of 7 days Default: - infinite or until the record expires.
        :param on_failure: An Amazon SQS queue or Amazon SNS topic destination for discarded records. Default: discarded records are ignored
        :param parallelization_factor: The number of batches to process from each shard concurrently. Valid Range: - Minimum value of 1 - Maximum value of 10 Default: 1
        :param report_batch_item_failures: Allow functions to return partially successful responses for a batch of records. Default: false
        :param retry_attempts: The maximum number of times to retry when the function returns an error. Set to ``undefined`` if you want lambda to keep retrying infinitely or until the record expires. Valid Range: - Minimum value of 0 - Maximum value of 10000 Default: - infinite or until the record expires.
        :param source_access_configurations: Specific settings like the authentication protocol or the VPC components to secure access to your event source. Default: - none
        :param starting_position: The position in the DynamoDB, Kinesis or MSK stream where AWS Lambda should start reading. Default: - Required for Amazon Kinesis, Amazon DynamoDB, and Amazon MSK Streams sources.
        :param tumbling_window: The size of the tumbling windows to group records sent to DynamoDB or Kinesis. Default: - None
        '''
        ...

    @jsii.member(jsii_name="addFunctionUrl")
    def add_function_url(
        self,
        *,
        auth_type: typing.Optional[FunctionUrlAuthType] = None,
        cors: typing.Optional[typing.Union[FunctionUrlCorsOptions, typing.Dict[builtins.str, typing.Any]]] = None,
    ) -> "FunctionUrl":
        '''Adds a url to this lambda function.

        :param auth_type: The type of authentication that your function URL uses. Default: FunctionUrlAuthType.AWS_IAM
        :param cors: The cross-origin resource sharing (CORS) settings for your function URL. Default: - No CORS configuration.
        '''
        ...

    @jsii.member(jsii_name="addPermission")
    def add_permission(
        self,
        id: builtins.str,
        *,
        principal: _aws_cdk_aws_iam_940a1ce0.IPrincipal,
        action: typing.Optional[builtins.str] = None,
        event_source_token: typing.Optional[builtins.str] = None,
        function_url_auth_type: typing.Optional[FunctionUrlAuthType] = None,
        scope: typing.Optional[_aws_cdk_core_f4b25747.Construct] = None,
        source_account: typing.Optional[builtins.str] = None,
        source_arn: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Adds a permission to the Lambda resource policy.

        :param id: The id for the permission construct.
        :param principal: The entity for which you are granting permission to invoke the Lambda function. This entity can be any valid AWS service principal, such as s3.amazonaws.com or sns.amazonaws.com, or, if you are granting cross-account permission, an AWS account ID. For example, you might want to allow a custom application in another AWS account to push events to Lambda by invoking your function. The principal can be either an AccountPrincipal or a ServicePrincipal.
        :param action: The Lambda actions that you want to allow in this statement. For example, you can specify lambda:CreateFunction to specify a certain action, or use a wildcard (``lambda:*``) to grant permission to all Lambda actions. For a list of actions, see Actions and Condition Context Keys for AWS Lambda in the IAM User Guide. Default: 'lambda:InvokeFunction'
        :param event_source_token: A unique token that must be supplied by the principal invoking the function. Default: The caller would not need to present a token.
        :param function_url_auth_type: The authType for the function URL that you are granting permissions for. Default: - No functionUrlAuthType
        :param scope: The scope to which the permission constructs be attached. The default is the Lambda function construct itself, but this would need to be different in cases such as cross-stack references where the Permissions would need to sit closer to the consumer of this permission (i.e., the caller). Default: - The instance of lambda.IFunction
        :param source_account: The AWS account ID (without hyphens) of the source owner. For example, if you specify an S3 bucket in the SourceArn property, this value is the bucket owner's account ID. You can use this property to ensure that all source principals are owned by a specific account.
        :param source_arn: The ARN of a resource that is invoking your function. When granting Amazon Simple Storage Service (Amazon S3) permission to invoke your function, specify this property with the bucket ARN as its value. This ensures that events generated only from the specified bucket, not just any bucket from any AWS account that creates a mapping to your function, can invoke the function.

        :see: Permission for details.
        '''
        ...

    @jsii.member(jsii_name="addToRolePolicy")
    def add_to_role_policy(
        self,
        statement: _aws_cdk_aws_iam_940a1ce0.PolicyStatement,
    ) -> None:
        '''Adds a statement to the IAM role assumed by the instance.

        :param statement: -
        '''
        ...

    @jsii.member(jsii_name="configureAsyncInvoke")
    def configure_async_invoke(
        self,
        *,
        max_event_age: typing.Optional[_aws_cdk_core_f4b25747.Duration] = None,
        on_failure: typing.Optional[IDestination] = None,
        on_success: typing.Optional[IDestination] = None,
        retry_attempts: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''Configures options for asynchronous invocation.

        :param max_event_age: The maximum age of a request that Lambda sends to a function for processing. Minimum: 60 seconds Maximum: 6 hours Default: Duration.hours(6)
        :param on_failure: The destination for failed invocations. Default: - no destination
        :param on_success: The destination for successful invocations. Default: - no destination
        :param retry_attempts: The maximum number of times to retry when the function returns an error. Minimum: 0 Maximum: 2 Default: 2
        '''
        ...

    @jsii.member(jsii_name="grantInvoke")
    def grant_invoke(
        self,
        identity: _aws_cdk_aws_iam_940a1ce0.IGrantable,
    ) -> _aws_cdk_aws_iam_940a1ce0.Grant:
        '''Grant the given identity permissions to invoke this Lambda.

        :param identity: -
        '''
        ...

    @jsii.member(jsii_name="grantInvokeUrl")
    def grant_invoke_url(
        self,
        identity: _aws_cdk_aws_iam_940a1ce0.IGrantable,
    ) -> _aws_cdk_aws_iam_940a1ce0.Grant:
        '''Grant the given identity permissions to invoke this Lambda Function URL.

        :param identity: -
        '''
        ...

    @jsii.member(jsii_name="metric")
    def metric(
        self,
        metric_name: builtins.str,
        *,
        account: typing.Optional[builtins.str] = None,
        color: typing.Optional[builtins.str] = None,
        dimensions: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        dimensions_map: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        label: typing.Optional[builtins.str] = None,
        period: typing.Optional[_aws_cdk_core_f4b25747.Duration] = None,
        region: typing.Optional[builtins.str] = None,
        statistic: typing.Optional[builtins.str] = None,
        unit: typing.Optional[_aws_cdk_aws_cloudwatch_9b88bb94.Unit] = None,
    ) -> _aws_cdk_aws_cloudwatch_9b88bb94.Metric:
        '''Return the given named metric for this Lambda Return the given named metric for this Function.

        :param metric_name: -
        :param account: Account which this metric comes from. Default: - Deployment account.
        :param color: The hex color code, prefixed with '#' (e.g. '#00ff00'), to use when this metric is rendered on a graph. The ``Color`` class has a set of standard colors that can be used here. Default: - Automatic color
        :param dimensions: (deprecated) Dimensions of the metric. Default: - No dimensions.
        :param dimensions_map: Dimensions of the metric. Default: - No dimensions.
        :param label: Label for this metric when added to a Graph in a Dashboard. You can use `dynamic labels <https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/graph-dynamic-labels.html>`_ to show summary information about the entire displayed time series in the legend. For example, if you use:: [max: ${MAX}] MyMetric As the metric label, the maximum value in the visible range will be shown next to the time series name in the graph's legend. Default: - No label
        :param period: The period over which the specified statistic is applied. Default: Duration.minutes(5)
        :param region: Region which this metric comes from. Default: - Deployment region.
        :param statistic: What function to use for aggregating. Can be one of the following: - "Minimum" | "min" - "Maximum" | "max" - "Average" | "avg" - "Sum" | "sum" - "SampleCount | "n" - "pNN.NN" Default: Average
        :param unit: Unit used to filter the metric stream. Only refer to datums emitted to the metric stream with the given unit and ignore all others. Only useful when datums are being emitted to the same metric stream under different units. The default is to use all matric datums in the stream, regardless of unit, which is recommended in nearly all cases. CloudWatch does not honor this property for graphs. Default: - All metric datums in the given metric stream
        '''
        ...

    @jsii.member(jsii_name="metricDuration")
    def metric_duration(
        self,
        *,
        account: typing.Optional[builtins.str] = None,
        color: typing.Optional[builtins.str] = None,
        dimensions: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        dimensions_map: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        label: typing.Optional[builtins.str] = None,
        period: typing.Optional[_aws_cdk_core_f4b25747.Duration] = None,
        region: typing.Optional[builtins.str] = None,
        statistic: typing.Optional[builtins.str] = None,
        unit: typing.Optional[_aws_cdk_aws_cloudwatch_9b88bb94.Unit] = None,
    ) -> _aws_cdk_aws_cloudwatch_9b88bb94.Metric:
        '''Metric for the Duration of this Lambda How long execution of this Lambda takes.

        Average over 5 minutes

        :param account: Account which this metric comes from. Default: - Deployment account.
        :param color: The hex color code, prefixed with '#' (e.g. '#00ff00'), to use when this metric is rendered on a graph. The ``Color`` class has a set of standard colors that can be used here. Default: - Automatic color
        :param dimensions: (deprecated) Dimensions of the metric. Default: - No dimensions.
        :param dimensions_map: Dimensions of the metric. Default: - No dimensions.
        :param label: Label for this metric when added to a Graph in a Dashboard. You can use `dynamic labels <https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/graph-dynamic-labels.html>`_ to show summary information about the entire displayed time series in the legend. For example, if you use:: [max: ${MAX}] MyMetric As the metric label, the maximum value in the visible range will be shown next to the time series name in the graph's legend. Default: - No label
        :param period: The period over which the specified statistic is applied. Default: Duration.minutes(5)
        :param region: Region which this metric comes from. Default: - Deployment region.
        :param statistic: What function to use for aggregating. Can be one of the following: - "Minimum" | "min" - "Maximum" | "max" - "Average" | "avg" - "Sum" | "sum" - "SampleCount | "n" - "pNN.NN" Default: Average
        :param unit: Unit used to filter the metric stream. Only refer to datums emitted to the metric stream with the given unit and ignore all others. Only useful when datums are being emitted to the same metric stream under different units. The default is to use all matric datums in the stream, regardless of unit, which is recommended in nearly all cases. CloudWatch does not honor this property for graphs. Default: - All metric datums in the given metric stream

        :default: average over 5 minutes
        '''
        ...

    @jsii.member(jsii_name="metricErrors")
    def metric_errors(
        self,
        *,
        account: typing.Optional[builtins.str] = None,
        color: typing.Optional[builtins.str] = None,
        dimensions: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        dimensions_map: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        label: typing.Optional[builtins.str] = None,
        period: typing.Optional[_aws_cdk_core_f4b25747.Duration] = None,
        region: typing.Optional[builtins.str] = None,
        statistic: typing.Optional[builtins.str] = None,
        unit: typing.Optional[_aws_cdk_aws_cloudwatch_9b88bb94.Unit] = None,
    ) -> _aws_cdk_aws_cloudwatch_9b88bb94.Metric:
        '''How many invocations of this Lambda fail.

        Sum over 5 minutes

        :param account: Account which this metric comes from. Default: - Deployment account.
        :param color: The hex color code, prefixed with '#' (e.g. '#00ff00'), to use when this metric is rendered on a graph. The ``Color`` class has a set of standard colors that can be used here. Default: - Automatic color
        :param dimensions: (deprecated) Dimensions of the metric. Default: - No dimensions.
        :param dimensions_map: Dimensions of the metric. Default: - No dimensions.
        :param label: Label for this metric when added to a Graph in a Dashboard. You can use `dynamic labels <https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/graph-dynamic-labels.html>`_ to show summary information about the entire displayed time series in the legend. For example, if you use:: [max: ${MAX}] MyMetric As the metric label, the maximum value in the visible range will be shown next to the time series name in the graph's legend. Default: - No label
        :param period: The period over which the specified statistic is applied. Default: Duration.minutes(5)
        :param region: Region which this metric comes from. Default: - Deployment region.
        :param statistic: What function to use for aggregating. Can be one of the following: - "Minimum" | "min" - "Maximum" | "max" - "Average" | "avg" - "Sum" | "sum" - "SampleCount | "n" - "pNN.NN" Default: Average
        :param unit: Unit used to filter the metric stream. Only refer to datums emitted to the metric stream with the given unit and ignore all others. Only useful when datums are being emitted to the same metric stream under different units. The default is to use all matric datums in the stream, regardless of unit, which is recommended in nearly all cases. CloudWatch does not honor this property for graphs. Default: - All metric datums in the given metric stream
        '''
        ...

    @jsii.member(jsii_name="metricInvocations")
    def metric_invocations(
        self,
        *,
        account: typing.Optional[builtins.str] = None,
        color: typing.Optional[builtins.str] = None,
        dimensions: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        dimensions_map: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        label: typing.Optional[builtins.str] = None,
        period: typing.Optional[_aws_cdk_core_f4b25747.Duration] = None,
        region: typing.Optional[builtins.str] = None,
        statistic: typing.Optional[builtins.str] = None,
        unit: typing.Optional[_aws_cdk_aws_cloudwatch_9b88bb94.Unit] = None,
    ) -> _aws_cdk_aws_cloudwatch_9b88bb94.Metric:
        '''Metric for the number of invocations of this Lambda How often this Lambda is invoked.

        Sum over 5 minutes

        :param account: Account which this metric comes from. Default: - Deployment account.
        :param color: The hex color code, prefixed with '#' (e.g. '#00ff00'), to use when this metric is rendered on a graph. The ``Color`` class has a set of standard colors that can be used here. Default: - Automatic color
        :param dimensions: (deprecated) Dimensions of the metric. Default: - No dimensions.
        :param dimensions_map: Dimensions of the metric. Default: - No dimensions.
        :param label: Label for this metric when added to a Graph in a Dashboard. You can use `dynamic labels <https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/graph-dynamic-labels.html>`_ to show summary information about the entire displayed time series in the legend. For example, if you use:: [max: ${MAX}] MyMetric As the metric label, the maximum value in the visible range will be shown next to the time series name in the graph's legend. Default: - No label
        :param period: The period over which the specified statistic is applied. Default: Duration.minutes(5)
        :param region: Region which this metric comes from. Default: - Deployment region.
        :param statistic: What function to use for aggregating. Can be one of the following: - "Minimum" | "min" - "Maximum" | "max" - "Average" | "avg" - "Sum" | "sum" - "SampleCount | "n" - "pNN.NN" Default: Average
        :param unit: Unit used to filter the metric stream. Only refer to datums emitted to the metric stream with the given unit and ignore all others. Only useful when datums are being emitted to the same metric stream under different units. The default is to use all matric datums in the stream, regardless of unit, which is recommended in nearly all cases. CloudWatch does not honor this property for graphs. Default: - All metric datums in the given metric stream

        :default: sum over 5 minutes
        '''
        ...

    @jsii.member(jsii_name="metricThrottles")
    def metric_throttles(
        self,
        *,
        account: typing.Optional[builtins.str] = None,
        color: typing.Optional[builtins.str] = None,
        dimensions: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        dimensions_map: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        label: typing.Optional[builtins.str] = None,
        period: typing.Optional[_aws_cdk_core_f4b25747.Duration] = None,
        region: typing.Optional[builtins.str] = None,
        statistic: typing.Optional[builtins.str] = None,
        unit: typing.Optional[_aws_cdk_aws_cloudwatch_9b88bb94.Unit] = None,
    ) -> _aws_cdk_aws_cloudwatch_9b88bb94.Metric:
        '''Metric for the number of throttled invocations of this Lambda How often this Lambda is throttled.

        Sum over 5 minutes

        :param account: Account which this metric comes from. Default: - Deployment account.
        :param color: The hex color code, prefixed with '#' (e.g. '#00ff00'), to use when this metric is rendered on a graph. The ``Color`` class has a set of standard colors that can be used here. Default: - Automatic color
        :param dimensions: (deprecated) Dimensions of the metric. Default: - No dimensions.
        :param dimensions_map: Dimensions of the metric. Default: - No dimensions.
        :param label: Label for this metric when added to a Graph in a Dashboard. You can use `dynamic labels <https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/graph-dynamic-labels.html>`_ to show summary information about the entire displayed time series in the legend. For example, if you use:: [max: ${MAX}] MyMetric As the metric label, the maximum value in the visible range will be shown next to the time series name in the graph's legend. Default: - No label
        :param period: The period over which the specified statistic is applied. Default: Duration.minutes(5)
        :param region: Region which this metric comes from. Default: - Deployment region.
        :param statistic: What function to use for aggregating. Can be one of the following: - "Minimum" | "min" - "Maximum" | "max" - "Average" | "avg" - "Sum" | "sum" - "SampleCount | "n" - "pNN.NN" Default: Average
        :param unit: Unit used to filter the metric stream. Only refer to datums emitted to the metric stream with the given unit and ignore all others. Only useful when datums are being emitted to the same metric stream under different units. The default is to use all matric datums in the stream, regardless of unit, which is recommended in nearly all cases. CloudWatch does not honor this property for graphs. Default: - All metric datums in the given metric stream

        :default: sum over 5 minutes
        '''
        ...


class _IFunctionProxy(
    jsii.proxy_for(_aws_cdk_core_f4b25747.IResource), # type: ignore[misc]
    jsii.proxy_for(_aws_cdk_aws_ec2_67de8e8d.IConnectable), # type: ignore[misc]
    jsii.proxy_for(_aws_cdk_aws_iam_940a1ce0.IGrantable), # type: ignore[misc]
):
    __jsii_type__: typing.ClassVar[str] = "@aws-cdk/aws-lambda.IFunction"

    @builtins.property
    @jsii.member(jsii_name="architecture")
    def architecture(self) -> Architecture:
        '''The system architectures compatible with this lambda function.'''
        return typing.cast(Architecture, jsii.get(self, "architecture"))

    @builtins.property
    @jsii.member(jsii_name="functionArn")
    def function_arn(self) -> builtins.str:
        '''The ARN of the function.

        :attribute: true
        '''
        return typing.cast(builtins.str, jsii.get(self, "functionArn"))

    @builtins.property
    @jsii.member(jsii_name="functionName")
    def function_name(self) -> builtins.str:
        '''The name of the function.

        :attribute: true
        '''
        return typing.cast(builtins.str, jsii.get(self, "functionName"))

    @builtins.property
    @jsii.member(jsii_name="isBoundToVpc")
    def is_bound_to_vpc(self) -> builtins.bool:
        '''Whether or not this Lambda function was bound to a VPC.

        If this is is ``false``, trying to access the ``connections`` object will fail.
        '''
        return typing.cast(builtins.bool, jsii.get(self, "isBoundToVpc"))

    @builtins.property
    @jsii.member(jsii_name="latestVersion")
    def latest_version(self) -> "IVersion":
        '''The ``$LATEST`` version of this function.

        Note that this is reference to a non-specific AWS Lambda version, which
        means the function this version refers to can return different results in
        different invocations.

        To obtain a reference to an explicit version which references the current
        function configuration, use ``lambdaFunction.currentVersion`` instead.
        '''
        return typing.cast("IVersion", jsii.get(self, "latestVersion"))

    @builtins.property
    @jsii.member(jsii_name="permissionsNode")
    def permissions_node(self) -> _aws_cdk_core_f4b25747.ConstructNode:
        '''The construct node where permissions are attached.'''
        return typing.cast(_aws_cdk_core_f4b25747.ConstructNode, jsii.get(self, "permissionsNode"))

    @builtins.property
    @jsii.member(jsii_name="resourceArnsForGrantInvoke")
    def resource_arns_for_grant_invoke(self) -> typing.List[builtins.str]:
        '''The ARN(s) to put into the resource field of the generated IAM policy for grantInvoke().

        This property is for cdk modules to consume only. You should not need to use this property.
        Instead, use grantInvoke() directly.
        '''
        return typing.cast(typing.List[builtins.str], jsii.get(self, "resourceArnsForGrantInvoke"))

    @builtins.property
    @jsii.member(jsii_name="role")
    def role(self) -> typing.Optional[_aws_cdk_aws_iam_940a1ce0.IRole]:
        '''The IAM role associated with this function.'''
        return typing.cast(typing.Optional[_aws_cdk_aws_iam_940a1ce0.IRole], jsii.get(self, "role"))

    @jsii.member(jsii_name="addEventSource")
    def add_event_source(self, source: IEventSource) -> None:
        '''Adds an event source to this function.

        Event sources are implemented in the @aws-cdk/aws-lambda-event-sources module.

        The following example adds an SQS Queue as an event source::

           import { SqsEventSource } from '@aws-cdk/aws-lambda-event-sources';
           myFunction.addEventSource(new SqsEventSource(myQueue));

        :param source: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f536b7e16a869615c618ed06282682ee627830dce89614a6e15b9011a9b3d90a)
            check_type(argname="argument source", value=source, expected_type=type_hints["source"])
        return typing.cast(None, jsii.invoke(self, "addEventSource", [source]))

    @jsii.member(jsii_name="addEventSourceMapping")
    def add_event_source_mapping(
        self,
        id: builtins.str,
        *,
        batch_size: typing.Optional[jsii.Number] = None,
        bisect_batch_on_error: typing.Optional[builtins.bool] = None,
        enabled: typing.Optional[builtins.bool] = None,
        event_source_arn: typing.Optional[builtins.str] = None,
        kafka_bootstrap_servers: typing.Optional[typing.Sequence[builtins.str]] = None,
        kafka_topic: typing.Optional[builtins.str] = None,
        max_batching_window: typing.Optional[_aws_cdk_core_f4b25747.Duration] = None,
        max_record_age: typing.Optional[_aws_cdk_core_f4b25747.Duration] = None,
        on_failure: typing.Optional[IEventSourceDlq] = None,
        parallelization_factor: typing.Optional[jsii.Number] = None,
        report_batch_item_failures: typing.Optional[builtins.bool] = None,
        retry_attempts: typing.Optional[jsii.Number] = None,
        source_access_configurations: typing.Optional[typing.Sequence[typing.Union["SourceAccessConfiguration", typing.Dict[builtins.str, typing.Any]]]] = None,
        starting_position: typing.Optional["StartingPosition"] = None,
        tumbling_window: typing.Optional[_aws_cdk_core_f4b25747.Duration] = None,
    ) -> "EventSourceMapping":
        '''Adds an event source that maps to this AWS Lambda function.

        :param id: construct ID.
        :param batch_size: The largest number of records that AWS Lambda will retrieve from your event source at the time of invoking your function. Your function receives an event with all the retrieved records. Valid Range: Minimum value of 1. Maximum value of 10000. Default: - Amazon Kinesis, Amazon DynamoDB, and Amazon MSK is 100 records. The default for Amazon SQS is 10 messages. For standard SQS queues, the maximum is 10,000. For FIFO SQS queues, the maximum is 10.
        :param bisect_batch_on_error: If the function returns an error, split the batch in two and retry. Default: false
        :param enabled: Set to false to disable the event source upon creation. Default: true
        :param event_source_arn: The Amazon Resource Name (ARN) of the event source. Any record added to this stream can invoke the Lambda function. Default: - not set if using a self managed Kafka cluster, throws an error otherwise
        :param kafka_bootstrap_servers: A list of host and port pairs that are the addresses of the Kafka brokers in a self managed "bootstrap" Kafka cluster that a Kafka client connects to initially to bootstrap itself. They are in the format ``abc.example.com:9096``. Default: - none
        :param kafka_topic: The name of the Kafka topic. Default: - no topic
        :param max_batching_window: The maximum amount of time to gather records before invoking the function. Maximum of Duration.minutes(5) Default: Duration.seconds(0)
        :param max_record_age: The maximum age of a record that Lambda sends to a function for processing. Valid Range: - Minimum value of 60 seconds - Maximum value of 7 days Default: - infinite or until the record expires.
        :param on_failure: An Amazon SQS queue or Amazon SNS topic destination for discarded records. Default: discarded records are ignored
        :param parallelization_factor: The number of batches to process from each shard concurrently. Valid Range: - Minimum value of 1 - Maximum value of 10 Default: 1
        :param report_batch_item_failures: Allow functions to return partially successful responses for a batch of records. Default: false
        :param retry_attempts: The maximum number of times to retry when the function returns an error. Set to ``undefined`` if you want lambda to keep retrying infinitely or until the record expires. Valid Range: - Minimum value of 0 - Maximum value of 10000 Default: - infinite or until the record expires.
        :param source_access_configurations: Specific settings like the authentication protocol or the VPC components to secure access to your event source. Default: - none
        :param starting_position: The position in the DynamoDB, Kinesis or MSK stream where AWS Lambda should start reading. Default: - Required for Amazon Kinesis, Amazon DynamoDB, and Amazon MSK Streams sources.
        :param tumbling_window: The size of the tumbling windows to group records sent to DynamoDB or Kinesis. Default: - None
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__95146039d51a25572f19e042d0dbdf837ecbf515c180ff166184e3147993405b)
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        options = EventSourceMappingOptions(
            batch_size=batch_size,
            bisect_batch_on_error=bisect_batch_on_error,
            enabled=enabled,
            event_source_arn=event_source_arn,
            kafka_bootstrap_servers=kafka_bootstrap_servers,
            kafka_topic=kafka_topic,
            max_batching_window=max_batching_window,
            max_record_age=max_record_age,
            on_failure=on_failure,
            parallelization_factor=parallelization_factor,
            report_batch_item_failures=report_batch_item_failures,
            retry_attempts=retry_attempts,
            source_access_configurations=source_access_configurations,
            starting_position=starting_position,
            tumbling_window=tumbling_window,
        )

        return typing.cast("EventSourceMapping", jsii.invoke(self, "addEventSourceMapping", [id, options]))

    @jsii.member(jsii_name="addFunctionUrl")
    def add_function_url(
        self,
        *,
        auth_type: typing.Optional[FunctionUrlAuthType] = None,
        cors: typing.Optional[typing.Union[FunctionUrlCorsOptions, typing.Dict[builtins.str, typing.Any]]] = None,
    ) -> "FunctionUrl":
        '''Adds a url to this lambda function.

        :param auth_type: The type of authentication that your function URL uses. Default: FunctionUrlAuthType.AWS_IAM
        :param cors: The cross-origin resource sharing (CORS) settings for your function URL. Default: - No CORS configuration.
        '''
        options = FunctionUrlOptions(auth_type=auth_type, cors=cors)

        return typing.cast("FunctionUrl", jsii.invoke(self, "addFunctionUrl", [options]))

    @jsii.member(jsii_name="addPermission")
    def add_permission(
        self,
        id: builtins.str,
        *,
        principal: _aws_cdk_aws_iam_940a1ce0.IPrincipal,
        action: typing.Optional[builtins.str] = None,
        event_source_token: typing.Optional[builtins.str] = None,
        function_url_auth_type: typing.Optional[FunctionUrlAuthType] = None,
        scope: typing.Optional[_aws_cdk_core_f4b25747.Construct] = None,
        source_account: typing.Optional[builtins.str] = None,
        source_arn: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Adds a permission to the Lambda resource policy.

        :param id: The id for the permission construct.
        :param principal: The entity for which you are granting permission to invoke the Lambda function. This entity can be any valid AWS service principal, such as s3.amazonaws.com or sns.amazonaws.com, or, if you are granting cross-account permission, an AWS account ID. For example, you might want to allow a custom application in another AWS account to push events to Lambda by invoking your function. The principal can be either an AccountPrincipal or a ServicePrincipal.
        :param action: The Lambda actions that you want to allow in this statement. For example, you can specify lambda:CreateFunction to specify a certain action, or use a wildcard (``lambda:*``) to grant permission to all Lambda actions. For a list of actions, see Actions and Condition Context Keys for AWS Lambda in the IAM User Guide. Default: 'lambda:InvokeFunction'
        :param event_source_token: A unique token that must be supplied by the principal invoking the function. Default: The caller would not need to present a token.
        :param function_url_auth_type: The authType for the function URL that you are granting permissions for. Default: - No functionUrlAuthType
        :param scope: The scope to which the permission constructs be attached. The default is the Lambda function construct itself, but this would need to be different in cases such as cross-stack references where the Permissions would need to sit closer to the consumer of this permission (i.e., the caller). Default: - The instance of lambda.IFunction
        :param source_account: The AWS account ID (without hyphens) of the source owner. For example, if you specify an S3 bucket in the SourceArn property, this value is the bucket owner's account ID. You can use this property to ensure that all source principals are owned by a specific account.
        :param source_arn: The ARN of a resource that is invoking your function. When granting Amazon Simple Storage Service (Amazon S3) permission to invoke your function, specify this property with the bucket ARN as its value. This ensures that events generated only from the specified bucket, not just any bucket from any AWS account that creates a mapping to your function, can invoke the function.

        :see: Permission for details.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5b3384e8e440546c27c2aab3f7f54e9fc47a2b057f5ec715828b3c8d4d615818)
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        permission = Permission(
            principal=principal,
            action=action,
            event_source_token=event_source_token,
            function_url_auth_type=function_url_auth_type,
            scope=scope,
            source_account=source_account,
            source_arn=source_arn,
        )

        return typing.cast(None, jsii.invoke(self, "addPermission", [id, permission]))

    @jsii.member(jsii_name="addToRolePolicy")
    def add_to_role_policy(
        self,
        statement: _aws_cdk_aws_iam_940a1ce0.PolicyStatement,
    ) -> None:
        '''Adds a statement to the IAM role assumed by the instance.

        :param statement: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0a48f9726cfb7e97142cec929e2d66d9a57147bd3c7a6f387d9923c7c35e460d)
            check_type(argname="argument statement", value=statement, expected_type=type_hints["statement"])
        return typing.cast(None, jsii.invoke(self, "addToRolePolicy", [statement]))

    @jsii.member(jsii_name="configureAsyncInvoke")
    def configure_async_invoke(
        self,
        *,
        max_event_age: typing.Optional[_aws_cdk_core_f4b25747.Duration] = None,
        on_failure: typing.Optional[IDestination] = None,
        on_success: typing.Optional[IDestination] = None,
        retry_attempts: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''Configures options for asynchronous invocation.

        :param max_event_age: The maximum age of a request that Lambda sends to a function for processing. Minimum: 60 seconds Maximum: 6 hours Default: Duration.hours(6)
        :param on_failure: The destination for failed invocations. Default: - no destination
        :param on_success: The destination for successful invocations. Default: - no destination
        :param retry_attempts: The maximum number of times to retry when the function returns an error. Minimum: 0 Maximum: 2 Default: 2
        '''
        options = EventInvokeConfigOptions(
            max_event_age=max_event_age,
            on_failure=on_failure,
            on_success=on_success,
            retry_attempts=retry_attempts,
        )

        return typing.cast(None, jsii.invoke(self, "configureAsyncInvoke", [options]))

    @jsii.member(jsii_name="grantInvoke")
    def grant_invoke(
        self,
        identity: _aws_cdk_aws_iam_940a1ce0.IGrantable,
    ) -> _aws_cdk_aws_iam_940a1ce0.Grant:
        '''Grant the given identity permissions to invoke this Lambda.

        :param identity: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__88faf54af08f8c11f0fbfa8545a53f57dcd9551f0634fa2ad685fb4e2a25557c)
            check_type(argname="argument identity", value=identity, expected_type=type_hints["identity"])
        return typing.cast(_aws_cdk_aws_iam_940a1ce0.Grant, jsii.invoke(self, "grantInvoke", [identity]))

    @jsii.member(jsii_name="grantInvokeUrl")
    def grant_invoke_url(
        self,
        identity: _aws_cdk_aws_iam_940a1ce0.IGrantable,
    ) -> _aws_cdk_aws_iam_940a1ce0.Grant:
        '''Grant the given identity permissions to invoke this Lambda Function URL.

        :param identity: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e88550edcf72cdfa3cc90862d1ba43b05c7832a5a626d5ea3bb1d77d68409820)
            check_type(argname="argument identity", value=identity, expected_type=type_hints["identity"])
        return typing.cast(_aws_cdk_aws_iam_940a1ce0.Grant, jsii.invoke(self, "grantInvokeUrl", [identity]))

    @jsii.member(jsii_name="metric")
    def metric(
        self,
        metric_name: builtins.str,
        *,
        account: typing.Optional[builtins.str] = None,
        color: typing.Optional[builtins.str] = None,
        dimensions: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        dimensions_map: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        label: typing.Optional[builtins.str] = None,
        period: typing.Optional[_aws_cdk_core_f4b25747.Duration] = None,
        region: typing.Optional[builtins.str] = None,
        statistic: typing.Optional[builtins.str] = None,
        unit: typing.Optional[_aws_cdk_aws_cloudwatch_9b88bb94.Unit] = None,
    ) -> _aws_cdk_aws_cloudwatch_9b88bb94.Metric:
        '''Return the given named metric for this Lambda Return the given named metric for this Function.

        :param metric_name: -
        :param account: Account which this metric comes from. Default: - Deployment account.
        :param color: The hex color code, prefixed with '#' (e.g. '#00ff00'), to use when this metric is rendered on a graph. The ``Color`` class has a set of standard colors that can be used here. Default: - Automatic color
        :param dimensions: (deprecated) Dimensions of the metric. Default: - No dimensions.
        :param dimensions_map: Dimensions of the metric. Default: - No dimensions.
        :param label: Label for this metric when added to a Graph in a Dashboard. You can use `dynamic labels <https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/graph-dynamic-labels.html>`_ to show summary information about the entire displayed time series in the legend. For example, if you use:: [max: ${MAX}] MyMetric As the metric label, the maximum value in the visible range will be shown next to the time series name in the graph's legend. Default: - No label
        :param period: The period over which the specified statistic is applied. Default: Duration.minutes(5)
        :param region: Region which this metric comes from. Default: - Deployment region.
        :param statistic: What function to use for aggregating. Can be one of the following: - "Minimum" | "min" - "Maximum" | "max" - "Average" | "avg" - "Sum" | "sum" - "SampleCount | "n" - "pNN.NN" Default: Average
        :param unit: Unit used to filter the metric stream. Only refer to datums emitted to the metric stream with the given unit and ignore all others. Only useful when datums are being emitted to the same metric stream under different units. The default is to use all matric datums in the stream, regardless of unit, which is recommended in nearly all cases. CloudWatch does not honor this property for graphs. Default: - All metric datums in the given metric stream
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__41d7cc9bc6393ebe737c57e016b447969bbe837affa41cdfb39f8ef2b50824ad)
            check_type(argname="argument metric_name", value=metric_name, expected_type=type_hints["metric_name"])
        props = _aws_cdk_aws_cloudwatch_9b88bb94.MetricOptions(
            account=account,
            color=color,
            dimensions=dimensions,
            dimensions_map=dimensions_map,
            label=label,
            period=period,
            region=region,
            statistic=statistic,
            unit=unit,
        )

        return typing.cast(_aws_cdk_aws_cloudwatch_9b88bb94.Metric, jsii.invoke(self, "metric", [metric_name, props]))

    @jsii.member(jsii_name="metricDuration")
    def metric_duration(
        self,
        *,
        account: typing.Optional[builtins.str] = None,
        color: typing.Optional[builtins.str] = None,
        dimensions: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        dimensions_map: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        label: typing.Optional[builtins.str] = None,
        period: typing.Optional[_aws_cdk_core_f4b25747.Duration] = None,
        region: typing.Optional[builtins.str] = None,
        statistic: typing.Optional[builtins.str] = None,
        unit: typing.Optional[_aws_cdk_aws_cloudwatch_9b88bb94.Unit] = None,
    ) -> _aws_cdk_aws_cloudwatch_9b88bb94.Metric:
        '''Metric for the Duration of this Lambda How long execution of this Lambda takes.

        Average over 5 minutes

        :param account: Account which this metric comes from. Default: - Deployment account.
        :param color: The hex color code, prefixed with '#' (e.g. '#00ff00'), to use when this metric is rendered on a graph. The ``Color`` class has a set of standard colors that can be used here. Default: - Automatic color
        :param dimensions: (deprecated) Dimensions of the metric. Default: - No dimensions.
        :param dimensions_map: Dimensions of the metric. Default: - No dimensions.
        :param label: Label for this metric when added to a Graph in a Dashboard. You can use `dynamic labels <https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/graph-dynamic-labels.html>`_ to show summary information about the entire displayed time series in the legend. For example, if you use:: [max: ${MAX}] MyMetric As the metric label, the maximum value in the visible range will be shown next to the time series name in the graph's legend. Default: - No label
        :param period: The period over which the specified statistic is applied. Default: Duration.minutes(5)
        :param region: Region which this metric comes from. Default: - Deployment region.
        :param statistic: What function to use for aggregating. Can be one of the following: - "Minimum" | "min" - "Maximum" | "max" - "Average" | "avg" - "Sum" | "sum" - "SampleCount | "n" - "pNN.NN" Default: Average
        :param unit: Unit used to filter the metric stream. Only refer to datums emitted to the metric stream with the given unit and ignore all others. Only useful when datums are being emitted to the same metric stream under different units. The default is to use all matric datums in the stream, regardless of unit, which is recommended in nearly all cases. CloudWatch does not honor this property for graphs. Default: - All metric datums in the given metric stream

        :default: average over 5 minutes
        '''
        props = _aws_cdk_aws_cloudwatch_9b88bb94.MetricOptions(
            account=account,
            color=color,
            dimensions=dimensions,
            dimensions_map=dimensions_map,
            label=label,
            period=period,
            region=region,
            statistic=statistic,
            unit=unit,
        )

        return typing.cast(_aws_cdk_aws_cloudwatch_9b88bb94.Metric, jsii.invoke(self, "metricDuration", [props]))

    @jsii.member(jsii_name="metricErrors")
    def metric_errors(
        self,
        *,
        account: typing.Optional[builtins.str] = None,
        color: typing.Optional[builtins.str] = None,
        dimensions: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        dimensions_map: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        label: typing.Optional[builtins.str] = None,
        period: typing.Optional[_aws_cdk_core_f4b25747.Duration] = None,
        region: typing.Optional[builtins.str] = None,
        statistic: typing.Optional[builtins.str] = None,
        unit: typing.Optional[_aws_cdk_aws_cloudwatch_9b88bb94.Unit] = None,
    ) -> _aws_cdk_aws_cloudwatch_9b88bb94.Metric:
        '''How many invocations of this Lambda fail.

        Sum over 5 minutes

        :param account: Account which this metric comes from. Default: - Deployment account.
        :param color: The hex color code, prefixed with '#' (e.g. '#00ff00'), to use when this metric is rendered on a graph. The ``Color`` class has a set of standard colors that can be used here. Default: - Automatic color
        :param dimensions: (deprecated) Dimensions of the metric. Default: - No dimensions.
        :param dimensions_map: Dimensions of the metric. Default: - No dimensions.
        :param label: Label for this metric when added to a Graph in a Dashboard. You can use `dynamic labels <https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/graph-dynamic-labels.html>`_ to show summary information about the entire displayed time series in the legend. For example, if you use:: [max: ${MAX}] MyMetric As the metric label, the maximum value in the visible range will be shown next to the time series name in the graph's legend. Default: - No label
        :param period: The period over which the specified statistic is applied. Default: Duration.minutes(5)
        :param region: Region which this metric comes from. Default: - Deployment region.
        :param statistic: What function to use for aggregating. Can be one of the following: - "Minimum" | "min" - "Maximum" | "max" - "Average" | "avg" - "Sum" | "sum" - "SampleCount | "n" - "pNN.NN" Default: Average
        :param unit: Unit used to filter the metric stream. Only refer to datums emitted to the metric stream with the given unit and ignore all others. Only useful when datums are being emitted to the same metric stream under different units. The default is to use all matric datums in the stream, regardless of unit, which is recommended in nearly all cases. CloudWatch does not honor this property for graphs. Default: - All metric datums in the given metric stream
        '''
        props = _aws_cdk_aws_cloudwatch_9b88bb94.MetricOptions(
            account=account,
            color=color,
            dimensions=dimensions,
            dimensions_map=dimensions_map,
            label=label,
            period=period,
            region=region,
            statistic=statistic,
            unit=unit,
        )

        return typing.cast(_aws_cdk_aws_cloudwatch_9b88bb94.Metric, jsii.invoke(self, "metricErrors", [props]))

    @jsii.member(jsii_name="metricInvocations")
    def metric_invocations(
        self,
        *,
        account: typing.Optional[builtins.str] = None,
        color: typing.Optional[builtins.str] = None,
        dimensions: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        dimensions_map: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        label: typing.Optional[builtins.str] = None,
        period: typing.Optional[_aws_cdk_core_f4b25747.Duration] = None,
        region: typing.Optional[builtins.str] = None,
        statistic: typing.Optional[builtins.str] = None,
        unit: typing.Optional[_aws_cdk_aws_cloudwatch_9b88bb94.Unit] = None,
    ) -> _aws_cdk_aws_cloudwatch_9b88bb94.Metric:
        '''Metric for the number of invocations of this Lambda How often this Lambda is invoked.

        Sum over 5 minutes

        :param account: Account which this metric comes from. Default: - Deployment account.
        :param color: The hex color code, prefixed with '#' (e.g. '#00ff00'), to use when this metric is rendered on a graph. The ``Color`` class has a set of standard colors that can be used here. Default: - Automatic color
        :param dimensions: (deprecated) Dimensions of the metric. Default: - No dimensions.
        :param dimensions_map: Dimensions of the metric. Default: - No dimensions.
        :param label: Label for this metric when added to a Graph in a Dashboard. You can use `dynamic labels <https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/graph-dynamic-labels.html>`_ to show summary information about the entire displayed time series in the legend. For example, if you use:: [max: ${MAX}] MyMetric As the metric label, the maximum value in the visible range will be shown next to the time series name in the graph's legend. Default: - No label
        :param period: The period over which the specified statistic is applied. Default: Duration.minutes(5)
        :param region: Region which this metric comes from. Default: - Deployment region.
        :param statistic: What function to use for aggregating. Can be one of the following: - "Minimum" | "min" - "Maximum" | "max" - "Average" | "avg" - "Sum" | "sum" - "SampleCount | "n" - "pNN.NN" Default: Average
        :param unit: Unit used to filter the metric stream. Only refer to datums emitted to the metric stream with the given unit and ignore all others. Only useful when datums are being emitted to the same metric stream under different units. The default is to use all matric datums in the stream, regardless of unit, which is recommended in nearly all cases. CloudWatch does not honor this property for graphs. Default: - All metric datums in the given metric stream

        :default: sum over 5 minutes
        '''
        props = _aws_cdk_aws_cloudwatch_9b88bb94.MetricOptions(
            account=account,
            color=color,
            dimensions=dimensions,
            dimensions_map=dimensions_map,
            label=label,
            period=period,
            region=region,
            statistic=statistic,
            unit=unit,
        )

        return typing.cast(_aws_cdk_aws_cloudwatch_9b88bb94.Metric, jsii.invoke(self, "metricInvocations", [props]))

    @jsii.member(jsii_name="metricThrottles")
    def metric_throttles(
        self,
        *,
        account: typing.Optional[builtins.str] = None,
        color: typing.Optional[builtins.str] = None,
        dimensions: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        dimensions_map: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        label: typing.Optional[builtins.str] = None,
        period: typing.Optional[_aws_cdk_core_f4b25747.Duration] = None,
        region: typing.Optional[builtins.str] = None,
        statistic: typing.Optional[builtins.str] = None,
        unit: typing.Optional[_aws_cdk_aws_cloudwatch_9b88bb94.Unit] = None,
    ) -> _aws_cdk_aws_cloudwatch_9b88bb94.Metric:
        '''Metric for the number of throttled invocations of this Lambda How often this Lambda is throttled.

        Sum over 5 minutes

        :param account: Account which this metric comes from. Default: - Deployment account.
        :param color: The hex color code, prefixed with '#' (e.g. '#00ff00'), to use when this metric is rendered on a graph. The ``Color`` class has a set of standard colors that can be used here. Default: - Automatic color
        :param dimensions: (deprecated) Dimensions of the metric. Default: - No dimensions.
        :param dimensions_map: Dimensions of the metric. Default: - No dimensions.
        :param label: Label for this metric when added to a Graph in a Dashboard. You can use `dynamic labels <https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/graph-dynamic-labels.html>`_ to show summary information about the entire displayed time series in the legend. For example, if you use:: [max: ${MAX}] MyMetric As the metric label, the maximum value in the visible range will be shown next to the time series name in the graph's legend. Default: - No label
        :param period: The period over which the specified statistic is applied. Default: Duration.minutes(5)
        :param region: Region which this metric comes from. Default: - Deployment region.
        :param statistic: What function to use for aggregating. Can be one of the following: - "Minimum" | "min" - "Maximum" | "max" - "Average" | "avg" - "Sum" | "sum" - "SampleCount | "n" - "pNN.NN" Default: Average
        :param unit: Unit used to filter the metric stream. Only refer to datums emitted to the metric stream with the given unit and ignore all others. Only useful when datums are being emitted to the same metric stream under different units. The default is to use all matric datums in the stream, regardless of unit, which is recommended in nearly all cases. CloudWatch does not honor this property for graphs. Default: - All metric datums in the given metric stream

        :default: sum over 5 minutes
        '''
        props = _aws_cdk_aws_cloudwatch_9b88bb94.MetricOptions(
            account=account,
            color=color,
            dimensions=dimensions,
            dimensions_map=dimensions_map,
            label=label,
            period=period,
            region=region,
            statistic=statistic,
            unit=unit,
        )

        return typing.cast(_aws_cdk_aws_cloudwatch_9b88bb94.Metric, jsii.invoke(self, "metricThrottles", [props]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IFunction).__jsii_proxy_class__ = lambda : _IFunctionProxy


@jsii.interface(jsii_type="@aws-cdk/aws-lambda.IFunctionUrl")
class IFunctionUrl(_aws_cdk_core_f4b25747.IResource, typing_extensions.Protocol):
    '''A Lambda function Url.'''

    @builtins.property
    @jsii.member(jsii_name="functionArn")
    def function_arn(self) -> builtins.str:
        '''The ARN of the function this URL refers to.

        :attribute: FunctionArn
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="url")
    def url(self) -> builtins.str:
        '''The url of the Lambda function.

        :attribute: FunctionUrl
        '''
        ...

    @jsii.member(jsii_name="grantInvokeUrl")
    def grant_invoke_url(
        self,
        identity: _aws_cdk_aws_iam_940a1ce0.IGrantable,
    ) -> _aws_cdk_aws_iam_940a1ce0.Grant:
        '''Grant the given identity permissions to invoke this Lambda Function URL.

        :param identity: -
        '''
        ...


class _IFunctionUrlProxy(
    jsii.proxy_for(_aws_cdk_core_f4b25747.IResource), # type: ignore[misc]
):
    '''A Lambda function Url.'''

    __jsii_type__: typing.ClassVar[str] = "@aws-cdk/aws-lambda.IFunctionUrl"

    @builtins.property
    @jsii.member(jsii_name="functionArn")
    def function_arn(self) -> builtins.str:
        '''The ARN of the function this URL refers to.

        :attribute: FunctionArn
        '''
        return typing.cast(builtins.str, jsii.get(self, "functionArn"))

    @builtins.property
    @jsii.member(jsii_name="url")
    def url(self) -> builtins.str:
        '''The url of the Lambda function.

        :attribute: FunctionUrl
        '''
        return typing.cast(builtins.str, jsii.get(self, "url"))

    @jsii.member(jsii_name="grantInvokeUrl")
    def grant_invoke_url(
        self,
        identity: _aws_cdk_aws_iam_940a1ce0.IGrantable,
    ) -> _aws_cdk_aws_iam_940a1ce0.Grant:
        '''Grant the given identity permissions to invoke this Lambda Function URL.

        :param identity: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9823e16ddb784304516e2b6661e0e724a1081a81ecf01dc6c32d6e4db3b3e34a)
            check_type(argname="argument identity", value=identity, expected_type=type_hints["identity"])
        return typing.cast(_aws_cdk_aws_iam_940a1ce0.Grant, jsii.invoke(self, "grantInvokeUrl", [identity]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IFunctionUrl).__jsii_proxy_class__ = lambda : _IFunctionUrlProxy


@jsii.interface(jsii_type="@aws-cdk/aws-lambda.ILayerVersion")
class ILayerVersion(_aws_cdk_core_f4b25747.IResource, typing_extensions.Protocol):
    @builtins.property
    @jsii.member(jsii_name="layerVersionArn")
    def layer_version_arn(self) -> builtins.str:
        '''The ARN of the Lambda Layer version that this Layer defines.

        :attribute: true
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="compatibleRuntimes")
    def compatible_runtimes(self) -> typing.Optional[typing.List["Runtime"]]:
        '''The runtimes compatible with this Layer.

        :default: Runtime.All
        '''
        ...

    @jsii.member(jsii_name="addPermission")
    def add_permission(
        self,
        id: builtins.str,
        *,
        account_id: builtins.str,
        organization_id: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Add permission for this layer version to specific entities.

        Usage within
        the same account where the layer is defined is always allowed and does not
        require calling this method. Note that the principal that creates the
        Lambda function using the layer (for example, a CloudFormation changeset
        execution role) also needs to have the ``lambda:GetLayerVersion``
        permission on the layer version.

        :param id: the ID of the grant in the construct tree.
        :param account_id: The AWS Account id of the account that is authorized to use a Lambda Layer Version. The wild-card ``'*'`` can be used to grant access to "any" account (or any account in an organization when ``organizationId`` is specified).
        :param organization_id: The ID of the AWS Organization to which the grant is restricted. Can only be specified if ``accountId`` is ``'*'``
        '''
        ...


class _ILayerVersionProxy(
    jsii.proxy_for(_aws_cdk_core_f4b25747.IResource), # type: ignore[misc]
):
    __jsii_type__: typing.ClassVar[str] = "@aws-cdk/aws-lambda.ILayerVersion"

    @builtins.property
    @jsii.member(jsii_name="layerVersionArn")
    def layer_version_arn(self) -> builtins.str:
        '''The ARN of the Lambda Layer version that this Layer defines.

        :attribute: true
        '''
        return typing.cast(builtins.str, jsii.get(self, "layerVersionArn"))

    @builtins.property
    @jsii.member(jsii_name="compatibleRuntimes")
    def compatible_runtimes(self) -> typing.Optional[typing.List["Runtime"]]:
        '''The runtimes compatible with this Layer.

        :default: Runtime.All
        '''
        return typing.cast(typing.Optional[typing.List["Runtime"]], jsii.get(self, "compatibleRuntimes"))

    @jsii.member(jsii_name="addPermission")
    def add_permission(
        self,
        id: builtins.str,
        *,
        account_id: builtins.str,
        organization_id: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Add permission for this layer version to specific entities.

        Usage within
        the same account where the layer is defined is always allowed and does not
        require calling this method. Note that the principal that creates the
        Lambda function using the layer (for example, a CloudFormation changeset
        execution role) also needs to have the ``lambda:GetLayerVersion``
        permission on the layer version.

        :param id: the ID of the grant in the construct tree.
        :param account_id: The AWS Account id of the account that is authorized to use a Lambda Layer Version. The wild-card ``'*'`` can be used to grant access to "any" account (or any account in an organization when ``organizationId`` is specified).
        :param organization_id: The ID of the AWS Organization to which the grant is restricted. Can only be specified if ``accountId`` is ``'*'``
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__80d2bb9a5c359c53a58ffb7fcf9befdfb3392695b180346e7bc106d2e373ad9e)
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        permission = LayerVersionPermission(
            account_id=account_id, organization_id=organization_id
        )

        return typing.cast(None, jsii.invoke(self, "addPermission", [id, permission]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, ILayerVersion).__jsii_proxy_class__ = lambda : _ILayerVersionProxy


@jsii.interface(jsii_type="@aws-cdk/aws-lambda.IScalableFunctionAttribute")
class IScalableFunctionAttribute(
    _aws_cdk_core_f4b25747.IConstruct,
    typing_extensions.Protocol,
):
    '''Interface for scalable attributes.'''

    @jsii.member(jsii_name="scaleOnSchedule")
    def scale_on_schedule(
        self,
        id: builtins.str,
        *,
        schedule: _aws_cdk_aws_applicationautoscaling_a31e8c21.Schedule,
        end_time: typing.Optional[datetime.datetime] = None,
        max_capacity: typing.Optional[jsii.Number] = None,
        min_capacity: typing.Optional[jsii.Number] = None,
        start_time: typing.Optional[datetime.datetime] = None,
    ) -> None:
        '''Scale out or in based on schedule.

        :param id: -
        :param schedule: When to perform this action.
        :param end_time: When this scheduled action expires. Default: The rule never expires.
        :param max_capacity: The new maximum capacity. During the scheduled time, the current capacity is above the maximum capacity, Application Auto Scaling scales in to the maximum capacity. At least one of maxCapacity and minCapacity must be supplied. Default: No new maximum capacity
        :param min_capacity: The new minimum capacity. During the scheduled time, if the current capacity is below the minimum capacity, Application Auto Scaling scales out to the minimum capacity. At least one of maxCapacity and minCapacity must be supplied. Default: No new minimum capacity
        :param start_time: When this scheduled action becomes active. Default: The rule is activate immediately
        '''
        ...

    @jsii.member(jsii_name="scaleOnUtilization")
    def scale_on_utilization(
        self,
        *,
        utilization_target: jsii.Number,
        disable_scale_in: typing.Optional[builtins.bool] = None,
        policy_name: typing.Optional[builtins.str] = None,
        scale_in_cooldown: typing.Optional[_aws_cdk_core_f4b25747.Duration] = None,
        scale_out_cooldown: typing.Optional[_aws_cdk_core_f4b25747.Duration] = None,
    ) -> None:
        '''Scale out or in to keep utilization at a given level.

        The utilization is tracked by the
        LambdaProvisionedConcurrencyUtilization metric, emitted by lambda. See:
        https://docs.aws.amazon.com/lambda/latest/dg/monitoring-metrics.html#monitoring-metrics-concurrency

        :param utilization_target: Utilization target for the attribute. For example, .5 indicates that 50 percent of allocated provisioned concurrency is in use.
        :param disable_scale_in: Indicates whether scale in by the target tracking policy is disabled. If the value is true, scale in is disabled and the target tracking policy won't remove capacity from the scalable resource. Otherwise, scale in is enabled and the target tracking policy can remove capacity from the scalable resource. Default: false
        :param policy_name: A name for the scaling policy. Default: - Automatically generated name.
        :param scale_in_cooldown: Period after a scale in activity completes before another scale in activity can start. Default: Duration.seconds(300) for the following scalable targets: ECS services, Spot Fleet requests, EMR clusters, AppStream 2.0 fleets, Aurora DB clusters, Amazon SageMaker endpoint variants, Custom resources. For all other scalable targets, the default value is Duration.seconds(0): DynamoDB tables, DynamoDB global secondary indexes, Amazon Comprehend document classification endpoints, Lambda provisioned concurrency
        :param scale_out_cooldown: Period after a scale out activity completes before another scale out activity can start. Default: Duration.seconds(300) for the following scalable targets: ECS services, Spot Fleet requests, EMR clusters, AppStream 2.0 fleets, Aurora DB clusters, Amazon SageMaker endpoint variants, Custom resources. For all other scalable targets, the default value is Duration.seconds(0): DynamoDB tables, DynamoDB global secondary indexes, Amazon Comprehend document classification endpoints, Lambda provisioned concurrency
        '''
        ...


class _IScalableFunctionAttributeProxy(
    jsii.proxy_for(_aws_cdk_core_f4b25747.IConstruct), # type: ignore[misc]
):
    '''Interface for scalable attributes.'''

    __jsii_type__: typing.ClassVar[str] = "@aws-cdk/aws-lambda.IScalableFunctionAttribute"

    @jsii.member(jsii_name="scaleOnSchedule")
    def scale_on_schedule(
        self,
        id: builtins.str,
        *,
        schedule: _aws_cdk_aws_applicationautoscaling_a31e8c21.Schedule,
        end_time: typing.Optional[datetime.datetime] = None,
        max_capacity: typing.Optional[jsii.Number] = None,
        min_capacity: typing.Optional[jsii.Number] = None,
        start_time: typing.Optional[datetime.datetime] = None,
    ) -> None:
        '''Scale out or in based on schedule.

        :param id: -
        :param schedule: When to perform this action.
        :param end_time: When this scheduled action expires. Default: The rule never expires.
        :param max_capacity: The new maximum capacity. During the scheduled time, the current capacity is above the maximum capacity, Application Auto Scaling scales in to the maximum capacity. At least one of maxCapacity and minCapacity must be supplied. Default: No new maximum capacity
        :param min_capacity: The new minimum capacity. During the scheduled time, if the current capacity is below the minimum capacity, Application Auto Scaling scales out to the minimum capacity. At least one of maxCapacity and minCapacity must be supplied. Default: No new minimum capacity
        :param start_time: When this scheduled action becomes active. Default: The rule is activate immediately
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9ec97768e7e935058472dec18a83950b0ba7d92f551c6b35203cee207534d28e)
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        actions = _aws_cdk_aws_applicationautoscaling_a31e8c21.ScalingSchedule(
            schedule=schedule,
            end_time=end_time,
            max_capacity=max_capacity,
            min_capacity=min_capacity,
            start_time=start_time,
        )

        return typing.cast(None, jsii.invoke(self, "scaleOnSchedule", [id, actions]))

    @jsii.member(jsii_name="scaleOnUtilization")
    def scale_on_utilization(
        self,
        *,
        utilization_target: jsii.Number,
        disable_scale_in: typing.Optional[builtins.bool] = None,
        policy_name: typing.Optional[builtins.str] = None,
        scale_in_cooldown: typing.Optional[_aws_cdk_core_f4b25747.Duration] = None,
        scale_out_cooldown: typing.Optional[_aws_cdk_core_f4b25747.Duration] = None,
    ) -> None:
        '''Scale out or in to keep utilization at a given level.

        The utilization is tracked by the
        LambdaProvisionedConcurrencyUtilization metric, emitted by lambda. See:
        https://docs.aws.amazon.com/lambda/latest/dg/monitoring-metrics.html#monitoring-metrics-concurrency

        :param utilization_target: Utilization target for the attribute. For example, .5 indicates that 50 percent of allocated provisioned concurrency is in use.
        :param disable_scale_in: Indicates whether scale in by the target tracking policy is disabled. If the value is true, scale in is disabled and the target tracking policy won't remove capacity from the scalable resource. Otherwise, scale in is enabled and the target tracking policy can remove capacity from the scalable resource. Default: false
        :param policy_name: A name for the scaling policy. Default: - Automatically generated name.
        :param scale_in_cooldown: Period after a scale in activity completes before another scale in activity can start. Default: Duration.seconds(300) for the following scalable targets: ECS services, Spot Fleet requests, EMR clusters, AppStream 2.0 fleets, Aurora DB clusters, Amazon SageMaker endpoint variants, Custom resources. For all other scalable targets, the default value is Duration.seconds(0): DynamoDB tables, DynamoDB global secondary indexes, Amazon Comprehend document classification endpoints, Lambda provisioned concurrency
        :param scale_out_cooldown: Period after a scale out activity completes before another scale out activity can start. Default: Duration.seconds(300) for the following scalable targets: ECS services, Spot Fleet requests, EMR clusters, AppStream 2.0 fleets, Aurora DB clusters, Amazon SageMaker endpoint variants, Custom resources. For all other scalable targets, the default value is Duration.seconds(0): DynamoDB tables, DynamoDB global secondary indexes, Amazon Comprehend document classification endpoints, Lambda provisioned concurrency
        '''
        options = UtilizationScalingOptions(
            utilization_target=utilization_target,
            disable_scale_in=disable_scale_in,
            policy_name=policy_name,
            scale_in_cooldown=scale_in_cooldown,
            scale_out_cooldown=scale_out_cooldown,
        )

        return typing.cast(None, jsii.invoke(self, "scaleOnUtilization", [options]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IScalableFunctionAttribute).__jsii_proxy_class__ = lambda : _IScalableFunctionAttributeProxy


@jsii.interface(jsii_type="@aws-cdk/aws-lambda.IVersion")
class IVersion(IFunction, typing_extensions.Protocol):
    @builtins.property
    @jsii.member(jsii_name="edgeArn")
    def edge_arn(self) -> builtins.str:
        '''The ARN of the version for Lambda@Edge.'''
        ...

    @builtins.property
    @jsii.member(jsii_name="lambda")
    def lambda_(self) -> IFunction:
        '''The underlying AWS Lambda function.'''
        ...

    @builtins.property
    @jsii.member(jsii_name="version")
    def version(self) -> builtins.str:
        '''The most recently deployed version of this function.

        :attribute: true
        '''
        ...

    @jsii.member(jsii_name="addAlias")
    def add_alias(
        self,
        alias_name: builtins.str,
        *,
        additional_versions: typing.Optional[typing.Sequence[typing.Union["VersionWeight", typing.Dict[builtins.str, typing.Any]]]] = None,
        description: typing.Optional[builtins.str] = None,
        provisioned_concurrent_executions: typing.Optional[jsii.Number] = None,
        max_event_age: typing.Optional[_aws_cdk_core_f4b25747.Duration] = None,
        on_failure: typing.Optional[IDestination] = None,
        on_success: typing.Optional[IDestination] = None,
        retry_attempts: typing.Optional[jsii.Number] = None,
    ) -> "Alias":
        '''(deprecated) Defines an alias for this version.

        :param alias_name: The name of the alias.
        :param additional_versions: Additional versions with individual weights this alias points to. Individual additional version weights specified here should add up to (less than) one. All remaining weight is routed to the default version. For example, the config is Example:: version: "1" additionalVersions: [{ version: "2", weight: 0.05 }] Then 5% of traffic will be routed to function version 2, while the remaining 95% of traffic will be routed to function version 1. Default: No additional versions
        :param description: Description for the alias. Default: No description
        :param provisioned_concurrent_executions: Specifies a provisioned concurrency configuration for a function's alias. Default: No provisioned concurrency
        :param max_event_age: The maximum age of a request that Lambda sends to a function for processing. Minimum: 60 seconds Maximum: 6 hours Default: Duration.hours(6)
        :param on_failure: The destination for failed invocations. Default: - no destination
        :param on_success: The destination for successful invocations. Default: - no destination
        :param retry_attempts: The maximum number of times to retry when the function returns an error. Minimum: 0 Maximum: 2 Default: 2

        :deprecated: Calling ``addAlias`` on a ``Version`` object will cause the Alias to be replaced on every function update. Call ``function.addAlias()`` or ``new Alias()`` instead.

        :stability: deprecated
        '''
        ...


class _IVersionProxy(
    jsii.proxy_for(IFunction), # type: ignore[misc]
):
    __jsii_type__: typing.ClassVar[str] = "@aws-cdk/aws-lambda.IVersion"

    @builtins.property
    @jsii.member(jsii_name="edgeArn")
    def edge_arn(self) -> builtins.str:
        '''The ARN of the version for Lambda@Edge.'''
        return typing.cast(builtins.str, jsii.get(self, "edgeArn"))

    @builtins.property
    @jsii.member(jsii_name="lambda")
    def lambda_(self) -> IFunction:
        '''The underlying AWS Lambda function.'''
        return typing.cast(IFunction, jsii.get(self, "lambda"))

    @builtins.property
    @jsii.member(jsii_name="version")
    def version(self) -> builtins.str:
        '''The most recently deployed version of this function.

        :attribute: true
        '''
        return typing.cast(builtins.str, jsii.get(self, "version"))

    @jsii.member(jsii_name="addAlias")
    def add_alias(
        self,
        alias_name: builtins.str,
        *,
        additional_versions: typing.Optional[typing.Sequence[typing.Union["VersionWeight", typing.Dict[builtins.str, typing.Any]]]] = None,
        description: typing.Optional[builtins.str] = None,
        provisioned_concurrent_executions: typing.Optional[jsii.Number] = None,
        max_event_age: typing.Optional[_aws_cdk_core_f4b25747.Duration] = None,
        on_failure: typing.Optional[IDestination] = None,
        on_success: typing.Optional[IDestination] = None,
        retry_attempts: typing.Optional[jsii.Number] = None,
    ) -> "Alias":
        '''(deprecated) Defines an alias for this version.

        :param alias_name: The name of the alias.
        :param additional_versions: Additional versions with individual weights this alias points to. Individual additional version weights specified here should add up to (less than) one. All remaining weight is routed to the default version. For example, the config is Example:: version: "1" additionalVersions: [{ version: "2", weight: 0.05 }] Then 5% of traffic will be routed to function version 2, while the remaining 95% of traffic will be routed to function version 1. Default: No additional versions
        :param description: Description for the alias. Default: No description
        :param provisioned_concurrent_executions: Specifies a provisioned concurrency configuration for a function's alias. Default: No provisioned concurrency
        :param max_event_age: The maximum age of a request that Lambda sends to a function for processing. Minimum: 60 seconds Maximum: 6 hours Default: Duration.hours(6)
        :param on_failure: The destination for failed invocations. Default: - no destination
        :param on_success: The destination for successful invocations. Default: - no destination
        :param retry_attempts: The maximum number of times to retry when the function returns an error. Minimum: 0 Maximum: 2 Default: 2

        :deprecated: Calling ``addAlias`` on a ``Version`` object will cause the Alias to be replaced on every function update. Call ``function.addAlias()`` or ``new Alias()`` instead.

        :stability: deprecated
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__98b1c08d0517cee3c63c824acaafbf228143f8b9345028eaa91f955a15ecc794)
            check_type(argname="argument alias_name", value=alias_name, expected_type=type_hints["alias_name"])
        options = AliasOptions(
            additional_versions=additional_versions,
            description=description,
            provisioned_concurrent_executions=provisioned_concurrent_executions,
            max_event_age=max_event_age,
            on_failure=on_failure,
            on_success=on_success,
            retry_attempts=retry_attempts,
        )

        return typing.cast("Alias", jsii.invoke(self, "addAlias", [alias_name, options]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IVersion).__jsii_proxy_class__ = lambda : _IVersionProxy


class InlineCode(
    Code,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-lambda.InlineCode",
):
    '''Lambda code from an inline string (limited to 4KiB).

    :exampleMetadata: lit=test/integ.layer-version.lit.ts infused

    Example::

        layer = lambda_.LayerVersion(stack, "MyLayer",
            code=lambda_.Code.from_asset(path.join(__dirname, "layer-code")),
            compatible_runtimes=[lambda_.Runtime.NODEJS_14_X],
            license="Apache-2.0",
            description="A layer to test the L2 construct"
        )
        
        # To grant usage by other AWS accounts
        layer.add_permission("remote-account-grant", account_id=aws_account_id)
        
        # To grant usage to all accounts in some AWS Ogranization
        # layer.grantUsage({ accountId: '*', organizationId });
        
        lambda_.Function(stack, "MyLayeredLambda",
            code=lambda_.InlineCode("foo"),
            handler="index.handler",
            runtime=lambda_.Runtime.NODEJS_14_X,
            layers=[layer]
        )
    '''

    def __init__(self, code: builtins.str) -> None:
        '''
        :param code: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e6b14c57ccb8340d267f9766054d343878f444128eb0f5a90d91d62ebb979d32)
            check_type(argname="argument code", value=code, expected_type=type_hints["code"])
        jsii.create(self.__class__, self, [code])

    @jsii.member(jsii_name="bind")
    def bind(self, _scope: _aws_cdk_core_f4b25747.Construct) -> CodeConfig:
        '''Called when the lambda or layer is initialized to allow this object to bind to the stack, add resources and have fun.

        :param _scope: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2d6291b3785900af2df8b915fd795bf93dd8a1e080948b04b16ccd134c9e7511)
            check_type(argname="argument _scope", value=_scope, expected_type=type_hints["_scope"])
        return typing.cast(CodeConfig, jsii.invoke(self, "bind", [_scope]))

    @builtins.property
    @jsii.member(jsii_name="isInline")
    def is_inline(self) -> builtins.bool:
        '''Determines whether this Code is inline code or not.'''
        return typing.cast(builtins.bool, jsii.get(self, "isInline"))


class LambdaInsightsVersion(
    metaclass=jsii.JSIIAbstractClass,
    jsii_type="@aws-cdk/aws-lambda.LambdaInsightsVersion",
):
    '''Version of CloudWatch Lambda Insights.

    :exampleMetadata: infused

    Example::

        layer_arn = "arn:aws:lambda:us-east-1:580247275435:layer:LambdaInsightsExtension:14"
        lambda_.Function(self, "MyFunction",
            runtime=lambda_.Runtime.NODEJS_16_X,
            handler="index.handler",
            code=lambda_.Code.from_asset(path.join(__dirname, "lambda-handler")),
            insights_version=lambda_.LambdaInsightsVersion.from_insight_version_arn(layer_arn)
        )
    '''

    def __init__(self) -> None:
        jsii.create(self.__class__, self, [])

    @jsii.member(jsii_name="fromInsightVersionArn")
    @builtins.classmethod
    def from_insight_version_arn(cls, arn: builtins.str) -> "LambdaInsightsVersion":
        '''Use the insights extension associated with the provided ARN.

        Make sure the ARN is associated
        with same region as your function

        :param arn: -

        :see: https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/Lambda-Insights-extension-versions.html
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8b6effe77906e79ed88ae82a3b9930084041bebcdb270bf31834839de4bc1b17)
            check_type(argname="argument arn", value=arn, expected_type=type_hints["arn"])
        return typing.cast("LambdaInsightsVersion", jsii.sinvoke(cls, "fromInsightVersionArn", [arn]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VERSION_1_0_119_0")
    def VERSION_1_0_119_0(cls) -> "LambdaInsightsVersion":
        '''Version 1.0.119.0.'''
        return typing.cast("LambdaInsightsVersion", jsii.sget(cls, "VERSION_1_0_119_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VERSION_1_0_135_0")
    def VERSION_1_0_135_0(cls) -> "LambdaInsightsVersion":
        '''Version 1.0.135.0.'''
        return typing.cast("LambdaInsightsVersion", jsii.sget(cls, "VERSION_1_0_135_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VERSION_1_0_54_0")
    def VERSION_1_0_54_0(cls) -> "LambdaInsightsVersion":
        '''Version 1.0.54.0.'''
        return typing.cast("LambdaInsightsVersion", jsii.sget(cls, "VERSION_1_0_54_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VERSION_1_0_86_0")
    def VERSION_1_0_86_0(cls) -> "LambdaInsightsVersion":
        '''Version 1.0.86.0.'''
        return typing.cast("LambdaInsightsVersion", jsii.sget(cls, "VERSION_1_0_86_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VERSION_1_0_89_0")
    def VERSION_1_0_89_0(cls) -> "LambdaInsightsVersion":
        '''Version 1.0.89.0.'''
        return typing.cast("LambdaInsightsVersion", jsii.sget(cls, "VERSION_1_0_89_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VERSION_1_0_98_0")
    def VERSION_1_0_98_0(cls) -> "LambdaInsightsVersion":
        '''Version 1.0.98.0.'''
        return typing.cast("LambdaInsightsVersion", jsii.sget(cls, "VERSION_1_0_98_0"))

    @builtins.property
    @jsii.member(jsii_name="layerVersionArn")
    def layer_version_arn(self) -> builtins.str:
        '''The arn of the Lambda Insights extension.'''
        return typing.cast(builtins.str, jsii.get(self, "layerVersionArn"))


class _LambdaInsightsVersionProxy(LambdaInsightsVersion):
    pass

# Adding a "__jsii_proxy_class__(): typing.Type" function to the abstract class
typing.cast(typing.Any, LambdaInsightsVersion).__jsii_proxy_class__ = lambda : _LambdaInsightsVersionProxy


@jsii.data_type(
    jsii_type="@aws-cdk/aws-lambda.LambdaRuntimeProps",
    jsii_struct_bases=[],
    name_mapping={
        "bundling_docker_image": "bundlingDockerImage",
        "supports_code_guru_profiling": "supportsCodeGuruProfiling",
        "supports_inline_code": "supportsInlineCode",
    },
)
class LambdaRuntimeProps:
    def __init__(
        self,
        *,
        bundling_docker_image: typing.Optional[builtins.str] = None,
        supports_code_guru_profiling: typing.Optional[builtins.bool] = None,
        supports_inline_code: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''
        :param bundling_docker_image: The Docker image name to be used for bundling in this runtime. Default: - the latest docker image "amazon/public.ecr.aws/sam/build-" from https://gallery.ecr.aws
        :param supports_code_guru_profiling: Whether this runtime is integrated with and supported for profiling using Amazon CodeGuru Profiler. Default: false
        :param supports_inline_code: Whether the ``ZipFile`` (aka inline code) property can be used with this runtime. Default: false

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_lambda as lambda_
            
            lambda_runtime_props = lambda.LambdaRuntimeProps(
                bundling_docker_image="bundlingDockerImage",
                supports_code_guru_profiling=False,
                supports_inline_code=False
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__594d7e42d62c7c848c9ac90a72e2f5a7edef50c7aa7df44d30421b05142d34e3)
            check_type(argname="argument bundling_docker_image", value=bundling_docker_image, expected_type=type_hints["bundling_docker_image"])
            check_type(argname="argument supports_code_guru_profiling", value=supports_code_guru_profiling, expected_type=type_hints["supports_code_guru_profiling"])
            check_type(argname="argument supports_inline_code", value=supports_inline_code, expected_type=type_hints["supports_inline_code"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if bundling_docker_image is not None:
            self._values["bundling_docker_image"] = bundling_docker_image
        if supports_code_guru_profiling is not None:
            self._values["supports_code_guru_profiling"] = supports_code_guru_profiling
        if supports_inline_code is not None:
            self._values["supports_inline_code"] = supports_inline_code

    @builtins.property
    def bundling_docker_image(self) -> typing.Optional[builtins.str]:
        '''The Docker image name to be used for bundling in this runtime.

        :default: - the latest docker image "amazon/public.ecr.aws/sam/build-" from https://gallery.ecr.aws
        '''
        result = self._values.get("bundling_docker_image")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def supports_code_guru_profiling(self) -> typing.Optional[builtins.bool]:
        '''Whether this runtime is integrated with and supported for profiling using Amazon CodeGuru Profiler.

        :default: false
        '''
        result = self._values.get("supports_code_guru_profiling")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def supports_inline_code(self) -> typing.Optional[builtins.bool]:
        '''Whether the ``ZipFile`` (aka inline code) property can be used with this runtime.

        :default: false
        '''
        result = self._values.get("supports_inline_code")
        return typing.cast(typing.Optional[builtins.bool], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "LambdaRuntimeProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(ILayerVersion)
class LayerVersion(
    _aws_cdk_core_f4b25747.Resource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-lambda.LayerVersion",
):
    '''Defines a new Lambda Layer version.

    :exampleMetadata: infused

    Example::

        lambda_.LayerVersion(self, "MyLayer",
            removal_policy=RemovalPolicy.RETAIN,
            code=lambda_.Code.from_asset(path.join(__dirname, "lambda-handler")),
            compatible_architectures=[lambda_.Architecture.X86_64, lambda_.Architecture.ARM_64]
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        code: Code,
        compatible_architectures: typing.Optional[typing.Sequence[Architecture]] = None,
        compatible_runtimes: typing.Optional[typing.Sequence["Runtime"]] = None,
        description: typing.Optional[builtins.str] = None,
        layer_version_name: typing.Optional[builtins.str] = None,
        license: typing.Optional[builtins.str] = None,
        removal_policy: typing.Optional[_aws_cdk_core_f4b25747.RemovalPolicy] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param code: The content of this Layer. Using ``Code.fromInline`` is not supported.
        :param compatible_architectures: The system architectures compatible with this layer. Default: [Architecture.X86_64]
        :param compatible_runtimes: The runtimes compatible with this Layer. Default: - All runtimes are supported.
        :param description: The description the this Lambda Layer. Default: - No description.
        :param layer_version_name: The name of the layer. Default: - A name will be generated.
        :param license: The SPDX licence identifier or URL to the license file for this layer. Default: - No license information will be recorded.
        :param removal_policy: Whether to retain this version of the layer when a new version is added or when the stack is deleted. Default: RemovalPolicy.DESTROY
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__591f40ff0012f21f969c1d7c2243f9a509cdf890302dbfaa69b4a50d7567a7b2)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = LayerVersionProps(
            code=code,
            compatible_architectures=compatible_architectures,
            compatible_runtimes=compatible_runtimes,
            description=description,
            layer_version_name=layer_version_name,
            license=license,
            removal_policy=removal_policy,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="fromLayerVersionArn")
    @builtins.classmethod
    def from_layer_version_arn(
        cls,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        layer_version_arn: builtins.str,
    ) -> ILayerVersion:
        '''Imports a layer version by ARN.

        Assumes it is compatible with all Lambda runtimes.

        :param scope: -
        :param id: -
        :param layer_version_arn: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d37d76f069b258dd3a7cb5567a359e2cb9f8201523815f9091ea21cfa1a9ad91)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
            check_type(argname="argument layer_version_arn", value=layer_version_arn, expected_type=type_hints["layer_version_arn"])
        return typing.cast(ILayerVersion, jsii.sinvoke(cls, "fromLayerVersionArn", [scope, id, layer_version_arn]))

    @jsii.member(jsii_name="fromLayerVersionAttributes")
    @builtins.classmethod
    def from_layer_version_attributes(
        cls,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        layer_version_arn: builtins.str,
        compatible_runtimes: typing.Optional[typing.Sequence["Runtime"]] = None,
    ) -> ILayerVersion:
        '''Imports a Layer that has been defined externally.

        :param scope: the parent Construct that will use the imported layer.
        :param id: the id of the imported layer in the construct tree.
        :param layer_version_arn: The ARN of the LayerVersion.
        :param compatible_runtimes: The list of compatible runtimes with this Layer.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__47989f6ac5ab7e8b7dff4e39fd65b48a8836774af3bfe598b0d5be642db7443a)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        attrs = LayerVersionAttributes(
            layer_version_arn=layer_version_arn,
            compatible_runtimes=compatible_runtimes,
        )

        return typing.cast(ILayerVersion, jsii.sinvoke(cls, "fromLayerVersionAttributes", [scope, id, attrs]))

    @jsii.member(jsii_name="addPermission")
    def add_permission(
        self,
        id: builtins.str,
        *,
        account_id: builtins.str,
        organization_id: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Add permission for this layer version to specific entities.

        Usage within
        the same account where the layer is defined is always allowed and does not
        require calling this method. Note that the principal that creates the
        Lambda function using the layer (for example, a CloudFormation changeset
        execution role) also needs to have the ``lambda:GetLayerVersion``
        permission on the layer version.

        :param id: -
        :param account_id: The AWS Account id of the account that is authorized to use a Lambda Layer Version. The wild-card ``'*'`` can be used to grant access to "any" account (or any account in an organization when ``organizationId`` is specified).
        :param organization_id: The ID of the AWS Organization to which the grant is restricted. Can only be specified if ``accountId`` is ``'*'``
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3b90065c9fe46f0a9d4d9ec65ceefcc8a8a80b38ef2a899427c28133520d033b)
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        permission = LayerVersionPermission(
            account_id=account_id, organization_id=organization_id
        )

        return typing.cast(None, jsii.invoke(self, "addPermission", [id, permission]))

    @builtins.property
    @jsii.member(jsii_name="layerVersionArn")
    def layer_version_arn(self) -> builtins.str:
        '''The ARN of the Lambda Layer version that this Layer defines.'''
        return typing.cast(builtins.str, jsii.get(self, "layerVersionArn"))

    @builtins.property
    @jsii.member(jsii_name="compatibleRuntimes")
    def compatible_runtimes(self) -> typing.Optional[typing.List["Runtime"]]:
        '''The runtimes compatible with this Layer.'''
        return typing.cast(typing.Optional[typing.List["Runtime"]], jsii.get(self, "compatibleRuntimes"))


@jsii.data_type(
    jsii_type="@aws-cdk/aws-lambda.LayerVersionAttributes",
    jsii_struct_bases=[],
    name_mapping={
        "layer_version_arn": "layerVersionArn",
        "compatible_runtimes": "compatibleRuntimes",
    },
)
class LayerVersionAttributes:
    def __init__(
        self,
        *,
        layer_version_arn: builtins.str,
        compatible_runtimes: typing.Optional[typing.Sequence["Runtime"]] = None,
    ) -> None:
        '''Properties necessary to import a LayerVersion.

        :param layer_version_arn: The ARN of the LayerVersion.
        :param compatible_runtimes: The list of compatible runtimes with this Layer.

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_lambda as lambda_
            
            # runtime: lambda.Runtime
            
            layer_version_attributes = lambda.LayerVersionAttributes(
                layer_version_arn="layerVersionArn",
            
                # the properties below are optional
                compatible_runtimes=[runtime]
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a2f5e63ef963f78f0defbe840165ebf3a07b23d0b873dbf3d749c354128b6d4e)
            check_type(argname="argument layer_version_arn", value=layer_version_arn, expected_type=type_hints["layer_version_arn"])
            check_type(argname="argument compatible_runtimes", value=compatible_runtimes, expected_type=type_hints["compatible_runtimes"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "layer_version_arn": layer_version_arn,
        }
        if compatible_runtimes is not None:
            self._values["compatible_runtimes"] = compatible_runtimes

    @builtins.property
    def layer_version_arn(self) -> builtins.str:
        '''The ARN of the LayerVersion.'''
        result = self._values.get("layer_version_arn")
        assert result is not None, "Required property 'layer_version_arn' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def compatible_runtimes(self) -> typing.Optional[typing.List["Runtime"]]:
        '''The list of compatible runtimes with this Layer.'''
        result = self._values.get("compatible_runtimes")
        return typing.cast(typing.Optional[typing.List["Runtime"]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "LayerVersionAttributes(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-lambda.LayerVersionOptions",
    jsii_struct_bases=[],
    name_mapping={
        "description": "description",
        "layer_version_name": "layerVersionName",
        "license": "license",
        "removal_policy": "removalPolicy",
    },
)
class LayerVersionOptions:
    def __init__(
        self,
        *,
        description: typing.Optional[builtins.str] = None,
        layer_version_name: typing.Optional[builtins.str] = None,
        license: typing.Optional[builtins.str] = None,
        removal_policy: typing.Optional[_aws_cdk_core_f4b25747.RemovalPolicy] = None,
    ) -> None:
        '''Non runtime options.

        :param description: The description the this Lambda Layer. Default: - No description.
        :param layer_version_name: The name of the layer. Default: - A name will be generated.
        :param license: The SPDX licence identifier or URL to the license file for this layer. Default: - No license information will be recorded.
        :param removal_policy: Whether to retain this version of the layer when a new version is added or when the stack is deleted. Default: RemovalPolicy.DESTROY

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_lambda as lambda_
            import aws_cdk.core as cdk
            
            layer_version_options = lambda.LayerVersionOptions(
                description="description",
                layer_version_name="layerVersionName",
                license="license",
                removal_policy=cdk.RemovalPolicy.DESTROY
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3d5569d3d3265edbd62646a62bf2b77e7f5b78acaddd7abeaaf37e1830e3d64b)
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument layer_version_name", value=layer_version_name, expected_type=type_hints["layer_version_name"])
            check_type(argname="argument license", value=license, expected_type=type_hints["license"])
            check_type(argname="argument removal_policy", value=removal_policy, expected_type=type_hints["removal_policy"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if description is not None:
            self._values["description"] = description
        if layer_version_name is not None:
            self._values["layer_version_name"] = layer_version_name
        if license is not None:
            self._values["license"] = license
        if removal_policy is not None:
            self._values["removal_policy"] = removal_policy

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''The description the this Lambda Layer.

        :default: - No description.
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def layer_version_name(self) -> typing.Optional[builtins.str]:
        '''The name of the layer.

        :default: - A name will be generated.
        '''
        result = self._values.get("layer_version_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def license(self) -> typing.Optional[builtins.str]:
        '''The SPDX licence identifier or URL to the license file for this layer.

        :default: - No license information will be recorded.
        '''
        result = self._values.get("license")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def removal_policy(self) -> typing.Optional[_aws_cdk_core_f4b25747.RemovalPolicy]:
        '''Whether to retain this version of the layer when a new version is added or when the stack is deleted.

        :default: RemovalPolicy.DESTROY
        '''
        result = self._values.get("removal_policy")
        return typing.cast(typing.Optional[_aws_cdk_core_f4b25747.RemovalPolicy], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "LayerVersionOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-lambda.LayerVersionPermission",
    jsii_struct_bases=[],
    name_mapping={"account_id": "accountId", "organization_id": "organizationId"},
)
class LayerVersionPermission:
    def __init__(
        self,
        *,
        account_id: builtins.str,
        organization_id: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Identification of an account (or organization) that is allowed to access a Lambda Layer Version.

        :param account_id: The AWS Account id of the account that is authorized to use a Lambda Layer Version. The wild-card ``'*'`` can be used to grant access to "any" account (or any account in an organization when ``organizationId`` is specified).
        :param organization_id: The ID of the AWS Organization to which the grant is restricted. Can only be specified if ``accountId`` is ``'*'``

        :exampleMetadata: lit=test/integ.layer-version.lit.ts infused

        Example::

            layer = lambda_.LayerVersion(stack, "MyLayer",
                code=lambda_.Code.from_asset(path.join(__dirname, "layer-code")),
                compatible_runtimes=[lambda_.Runtime.NODEJS_14_X],
                license="Apache-2.0",
                description="A layer to test the L2 construct"
            )
            
            # To grant usage by other AWS accounts
            layer.add_permission("remote-account-grant", account_id=aws_account_id)
            
            # To grant usage to all accounts in some AWS Ogranization
            # layer.grantUsage({ accountId: '*', organizationId });
            
            lambda_.Function(stack, "MyLayeredLambda",
                code=lambda_.InlineCode("foo"),
                handler="index.handler",
                runtime=lambda_.Runtime.NODEJS_14_X,
                layers=[layer]
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5ab5903ba425acbe06102a84a7436030cd558a72a28bf9722f441703b85ae66b)
            check_type(argname="argument account_id", value=account_id, expected_type=type_hints["account_id"])
            check_type(argname="argument organization_id", value=organization_id, expected_type=type_hints["organization_id"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "account_id": account_id,
        }
        if organization_id is not None:
            self._values["organization_id"] = organization_id

    @builtins.property
    def account_id(self) -> builtins.str:
        '''The AWS Account id of the account that is authorized to use a Lambda Layer Version.

        The wild-card ``'*'`` can be
        used to grant access to "any" account (or any account in an organization when ``organizationId`` is specified).
        '''
        result = self._values.get("account_id")
        assert result is not None, "Required property 'account_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def organization_id(self) -> typing.Optional[builtins.str]:
        '''The ID of the AWS Organization to which the grant is restricted.

        Can only be specified if ``accountId`` is ``'*'``
        '''
        result = self._values.get("organization_id")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "LayerVersionPermission(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-lambda.LayerVersionProps",
    jsii_struct_bases=[LayerVersionOptions],
    name_mapping={
        "description": "description",
        "layer_version_name": "layerVersionName",
        "license": "license",
        "removal_policy": "removalPolicy",
        "code": "code",
        "compatible_architectures": "compatibleArchitectures",
        "compatible_runtimes": "compatibleRuntimes",
    },
)
class LayerVersionProps(LayerVersionOptions):
    def __init__(
        self,
        *,
        description: typing.Optional[builtins.str] = None,
        layer_version_name: typing.Optional[builtins.str] = None,
        license: typing.Optional[builtins.str] = None,
        removal_policy: typing.Optional[_aws_cdk_core_f4b25747.RemovalPolicy] = None,
        code: Code,
        compatible_architectures: typing.Optional[typing.Sequence[Architecture]] = None,
        compatible_runtimes: typing.Optional[typing.Sequence["Runtime"]] = None,
    ) -> None:
        '''
        :param description: The description the this Lambda Layer. Default: - No description.
        :param layer_version_name: The name of the layer. Default: - A name will be generated.
        :param license: The SPDX licence identifier or URL to the license file for this layer. Default: - No license information will be recorded.
        :param removal_policy: Whether to retain this version of the layer when a new version is added or when the stack is deleted. Default: RemovalPolicy.DESTROY
        :param code: The content of this Layer. Using ``Code.fromInline`` is not supported.
        :param compatible_architectures: The system architectures compatible with this layer. Default: [Architecture.X86_64]
        :param compatible_runtimes: The runtimes compatible with this Layer. Default: - All runtimes are supported.

        :exampleMetadata: infused

        Example::

            lambda_.LayerVersion(self, "MyLayer",
                removal_policy=RemovalPolicy.RETAIN,
                code=lambda_.Code.from_asset(path.join(__dirname, "lambda-handler")),
                compatible_architectures=[lambda_.Architecture.X86_64, lambda_.Architecture.ARM_64]
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__1b4ca230de9ef9e56450fd041915601bed3a344a83bd67ef649956fcf091515c)
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument layer_version_name", value=layer_version_name, expected_type=type_hints["layer_version_name"])
            check_type(argname="argument license", value=license, expected_type=type_hints["license"])
            check_type(argname="argument removal_policy", value=removal_policy, expected_type=type_hints["removal_policy"])
            check_type(argname="argument code", value=code, expected_type=type_hints["code"])
            check_type(argname="argument compatible_architectures", value=compatible_architectures, expected_type=type_hints["compatible_architectures"])
            check_type(argname="argument compatible_runtimes", value=compatible_runtimes, expected_type=type_hints["compatible_runtimes"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "code": code,
        }
        if description is not None:
            self._values["description"] = description
        if layer_version_name is not None:
            self._values["layer_version_name"] = layer_version_name
        if license is not None:
            self._values["license"] = license
        if removal_policy is not None:
            self._values["removal_policy"] = removal_policy
        if compatible_architectures is not None:
            self._values["compatible_architectures"] = compatible_architectures
        if compatible_runtimes is not None:
            self._values["compatible_runtimes"] = compatible_runtimes

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''The description the this Lambda Layer.

        :default: - No description.
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def layer_version_name(self) -> typing.Optional[builtins.str]:
        '''The name of the layer.

        :default: - A name will be generated.
        '''
        result = self._values.get("layer_version_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def license(self) -> typing.Optional[builtins.str]:
        '''The SPDX licence identifier or URL to the license file for this layer.

        :default: - No license information will be recorded.
        '''
        result = self._values.get("license")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def removal_policy(self) -> typing.Optional[_aws_cdk_core_f4b25747.RemovalPolicy]:
        '''Whether to retain this version of the layer when a new version is added or when the stack is deleted.

        :default: RemovalPolicy.DESTROY
        '''
        result = self._values.get("removal_policy")
        return typing.cast(typing.Optional[_aws_cdk_core_f4b25747.RemovalPolicy], result)

    @builtins.property
    def code(self) -> Code:
        '''The content of this Layer.

        Using ``Code.fromInline`` is not supported.
        '''
        result = self._values.get("code")
        assert result is not None, "Required property 'code' is missing"
        return typing.cast(Code, result)

    @builtins.property
    def compatible_architectures(self) -> typing.Optional[typing.List[Architecture]]:
        '''The system architectures compatible with this layer.

        :default: [Architecture.X86_64]
        '''
        result = self._values.get("compatible_architectures")
        return typing.cast(typing.Optional[typing.List[Architecture]], result)

    @builtins.property
    def compatible_runtimes(self) -> typing.Optional[typing.List["Runtime"]]:
        '''The runtimes compatible with this Layer.

        :default: - All runtimes are supported.
        '''
        result = self._values.get("compatible_runtimes")
        return typing.cast(typing.Optional[typing.List["Runtime"]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "LayerVersionProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class LogRetention(
    _aws_cdk_aws_logs_6c4320fb.LogRetention,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-lambda.LogRetention",
):
    '''(deprecated) Creates a custom resource to control the retention policy of a CloudWatch Logs log group.

    The log group is created if it doesn't already exist. The policy
    is removed when ``retentionDays`` is ``undefined`` or equal to ``Infinity``.

    :deprecated: use ``LogRetention`` from '

    :stability: deprecated
    :aws-cdk: /aws-logs' instead
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        import aws_cdk.aws_iam as iam
        import aws_cdk.aws_lambda as lambda_
        import aws_cdk.aws_logs as logs
        import aws_cdk.core as cdk
        
        # role: iam.Role
        
        log_retention = lambda_.LogRetention(self, "MyLogRetention",
            log_group_name="logGroupName",
            retention=logs.RetentionDays.ONE_DAY,
        
            # the properties below are optional
            log_group_region="logGroupRegion",
            log_retention_retry_options=logs.LogRetentionRetryOptions(
                base=cdk.Duration.minutes(30),
                max_retries=123
            ),
            role=role
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        log_group_name: builtins.str,
        retention: _aws_cdk_aws_logs_6c4320fb.RetentionDays,
        log_group_region: typing.Optional[builtins.str] = None,
        log_retention_retry_options: typing.Optional[typing.Union[_aws_cdk_aws_logs_6c4320fb.LogRetentionRetryOptions, typing.Dict[builtins.str, typing.Any]]] = None,
        role: typing.Optional[_aws_cdk_aws_iam_940a1ce0.IRole] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param log_group_name: The log group name.
        :param retention: The number of days log events are kept in CloudWatch Logs.
        :param log_group_region: The region where the log group should be created. Default: - same region as the stack
        :param log_retention_retry_options: Retry options for all AWS API calls. Default: - AWS SDK default retry options
        :param role: The IAM role for the Lambda function associated with the custom resource. Default: - A new role is created

        :stability: deprecated
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__924a06b5b61f8840316ceaa6066a9690000adae69338328ff51489c794465892)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = LogRetentionProps(
            log_group_name=log_group_name,
            retention=retention,
            log_group_region=log_group_region,
            log_retention_retry_options=log_retention_retry_options,
            role=role,
        )

        jsii.create(self.__class__, self, [scope, id, props])


@jsii.data_type(
    jsii_type="@aws-cdk/aws-lambda.LogRetentionProps",
    jsii_struct_bases=[_aws_cdk_aws_logs_6c4320fb.LogRetentionProps],
    name_mapping={
        "log_group_name": "logGroupName",
        "retention": "retention",
        "log_group_region": "logGroupRegion",
        "log_retention_retry_options": "logRetentionRetryOptions",
        "role": "role",
    },
)
class LogRetentionProps(_aws_cdk_aws_logs_6c4320fb.LogRetentionProps):
    def __init__(
        self,
        *,
        log_group_name: builtins.str,
        retention: _aws_cdk_aws_logs_6c4320fb.RetentionDays,
        log_group_region: typing.Optional[builtins.str] = None,
        log_retention_retry_options: typing.Optional[typing.Union[_aws_cdk_aws_logs_6c4320fb.LogRetentionRetryOptions, typing.Dict[builtins.str, typing.Any]]] = None,
        role: typing.Optional[_aws_cdk_aws_iam_940a1ce0.IRole] = None,
    ) -> None:
        '''(deprecated) Construction properties for a LogRetention.

        :param log_group_name: The log group name.
        :param retention: The number of days log events are kept in CloudWatch Logs.
        :param log_group_region: The region where the log group should be created. Default: - same region as the stack
        :param log_retention_retry_options: Retry options for all AWS API calls. Default: - AWS SDK default retry options
        :param role: The IAM role for the Lambda function associated with the custom resource. Default: - A new role is created

        :deprecated: use ``LogRetentionProps`` from '

        :stability: deprecated
        :aws-cdk: /aws-logs' instead
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_iam as iam
            import aws_cdk.aws_lambda as lambda_
            import aws_cdk.aws_logs as logs
            import aws_cdk.core as cdk
            
            # role: iam.Role
            
            log_retention_props = lambda.LogRetentionProps(
                log_group_name="logGroupName",
                retention=logs.RetentionDays.ONE_DAY,
            
                # the properties below are optional
                log_group_region="logGroupRegion",
                log_retention_retry_options=logs.LogRetentionRetryOptions(
                    base=cdk.Duration.minutes(30),
                    max_retries=123
                ),
                role=role
            )
        '''
        if isinstance(log_retention_retry_options, dict):
            log_retention_retry_options = _aws_cdk_aws_logs_6c4320fb.LogRetentionRetryOptions(**log_retention_retry_options)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d303956d9eea1581a58c747da2950656efae1d1fc1c1e576c24901d4c1ba9997)
            check_type(argname="argument log_group_name", value=log_group_name, expected_type=type_hints["log_group_name"])
            check_type(argname="argument retention", value=retention, expected_type=type_hints["retention"])
            check_type(argname="argument log_group_region", value=log_group_region, expected_type=type_hints["log_group_region"])
            check_type(argname="argument log_retention_retry_options", value=log_retention_retry_options, expected_type=type_hints["log_retention_retry_options"])
            check_type(argname="argument role", value=role, expected_type=type_hints["role"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "log_group_name": log_group_name,
            "retention": retention,
        }
        if log_group_region is not None:
            self._values["log_group_region"] = log_group_region
        if log_retention_retry_options is not None:
            self._values["log_retention_retry_options"] = log_retention_retry_options
        if role is not None:
            self._values["role"] = role

    @builtins.property
    def log_group_name(self) -> builtins.str:
        '''The log group name.'''
        result = self._values.get("log_group_name")
        assert result is not None, "Required property 'log_group_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def retention(self) -> _aws_cdk_aws_logs_6c4320fb.RetentionDays:
        '''The number of days log events are kept in CloudWatch Logs.'''
        result = self._values.get("retention")
        assert result is not None, "Required property 'retention' is missing"
        return typing.cast(_aws_cdk_aws_logs_6c4320fb.RetentionDays, result)

    @builtins.property
    def log_group_region(self) -> typing.Optional[builtins.str]:
        '''The region where the log group should be created.

        :default: - same region as the stack
        '''
        result = self._values.get("log_group_region")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def log_retention_retry_options(
        self,
    ) -> typing.Optional[_aws_cdk_aws_logs_6c4320fb.LogRetentionRetryOptions]:
        '''Retry options for all AWS API calls.

        :default: - AWS SDK default retry options
        '''
        result = self._values.get("log_retention_retry_options")
        return typing.cast(typing.Optional[_aws_cdk_aws_logs_6c4320fb.LogRetentionRetryOptions], result)

    @builtins.property
    def role(self) -> typing.Optional[_aws_cdk_aws_iam_940a1ce0.IRole]:
        '''The IAM role for the Lambda function associated with the custom resource.

        :default: - A new role is created
        '''
        result = self._values.get("role")
        return typing.cast(typing.Optional[_aws_cdk_aws_iam_940a1ce0.IRole], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "LogRetentionProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-lambda.LogRetentionRetryOptions",
    jsii_struct_bases=[_aws_cdk_aws_logs_6c4320fb.LogRetentionRetryOptions],
    name_mapping={"base": "base", "max_retries": "maxRetries"},
)
class LogRetentionRetryOptions(_aws_cdk_aws_logs_6c4320fb.LogRetentionRetryOptions):
    def __init__(
        self,
        *,
        base: typing.Optional[_aws_cdk_core_f4b25747.Duration] = None,
        max_retries: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''Retry options for all AWS API calls.

        :param base: The base duration to use in the exponential backoff for operation retries. Default: Duration.millis(100) (AWS SDK default)
        :param max_retries: The maximum amount of retries. Default: 3 (AWS SDK default)

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_lambda as lambda_
            import aws_cdk.core as cdk
            
            log_retention_retry_options = lambda.LogRetentionRetryOptions(
                base=cdk.Duration.minutes(30),
                max_retries=123
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__306cc2800c05d366484f02781d9767de1677361cd8662b4e54cf23dd5667e4b9)
            check_type(argname="argument base", value=base, expected_type=type_hints["base"])
            check_type(argname="argument max_retries", value=max_retries, expected_type=type_hints["max_retries"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if base is not None:
            self._values["base"] = base
        if max_retries is not None:
            self._values["max_retries"] = max_retries

    @builtins.property
    def base(self) -> typing.Optional[_aws_cdk_core_f4b25747.Duration]:
        '''The base duration to use in the exponential backoff for operation retries.

        :default: Duration.millis(100) (AWS SDK default)
        '''
        result = self._values.get("base")
        return typing.cast(typing.Optional[_aws_cdk_core_f4b25747.Duration], result)

    @builtins.property
    def max_retries(self) -> typing.Optional[jsii.Number]:
        '''The maximum amount of retries.

        :default: 3 (AWS SDK default)
        '''
        result = self._values.get("max_retries")
        return typing.cast(typing.Optional[jsii.Number], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "LogRetentionRetryOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-lambda.Permission",
    jsii_struct_bases=[],
    name_mapping={
        "principal": "principal",
        "action": "action",
        "event_source_token": "eventSourceToken",
        "function_url_auth_type": "functionUrlAuthType",
        "scope": "scope",
        "source_account": "sourceAccount",
        "source_arn": "sourceArn",
    },
)
class Permission:
    def __init__(
        self,
        *,
        principal: _aws_cdk_aws_iam_940a1ce0.IPrincipal,
        action: typing.Optional[builtins.str] = None,
        event_source_token: typing.Optional[builtins.str] = None,
        function_url_auth_type: typing.Optional[FunctionUrlAuthType] = None,
        scope: typing.Optional[_aws_cdk_core_f4b25747.Construct] = None,
        source_account: typing.Optional[builtins.str] = None,
        source_arn: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Represents a permission statement that can be added to a Lambda function's resource policy via the ``addPermission()`` method.

        :param principal: The entity for which you are granting permission to invoke the Lambda function. This entity can be any valid AWS service principal, such as s3.amazonaws.com or sns.amazonaws.com, or, if you are granting cross-account permission, an AWS account ID. For example, you might want to allow a custom application in another AWS account to push events to Lambda by invoking your function. The principal can be either an AccountPrincipal or a ServicePrincipal.
        :param action: The Lambda actions that you want to allow in this statement. For example, you can specify lambda:CreateFunction to specify a certain action, or use a wildcard (``lambda:*``) to grant permission to all Lambda actions. For a list of actions, see Actions and Condition Context Keys for AWS Lambda in the IAM User Guide. Default: 'lambda:InvokeFunction'
        :param event_source_token: A unique token that must be supplied by the principal invoking the function. Default: The caller would not need to present a token.
        :param function_url_auth_type: The authType for the function URL that you are granting permissions for. Default: - No functionUrlAuthType
        :param scope: The scope to which the permission constructs be attached. The default is the Lambda function construct itself, but this would need to be different in cases such as cross-stack references where the Permissions would need to sit closer to the consumer of this permission (i.e., the caller). Default: - The instance of lambda.IFunction
        :param source_account: The AWS account ID (without hyphens) of the source owner. For example, if you specify an S3 bucket in the SourceArn property, this value is the bucket owner's account ID. You can use this property to ensure that all source principals are owned by a specific account.
        :param source_arn: The ARN of a resource that is invoking your function. When granting Amazon Simple Storage Service (Amazon S3) permission to invoke your function, specify this property with the bucket ARN as its value. This ensures that events generated only from the specified bucket, not just any bucket from any AWS account that creates a mapping to your function, can invoke the function.

        :exampleMetadata: infused

        Example::

            # fn: lambda.Function
            
            principal = iam.ServicePrincipal("my-service")
            
            fn.grant_invoke(principal)
            
            # Equivalent to:
            fn.add_permission("my-service Invocation",
                principal=principal
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c2b481dffba82d6a246a3e58e3a7d31b991706ecc399e4301223afb9890b6598)
            check_type(argname="argument principal", value=principal, expected_type=type_hints["principal"])
            check_type(argname="argument action", value=action, expected_type=type_hints["action"])
            check_type(argname="argument event_source_token", value=event_source_token, expected_type=type_hints["event_source_token"])
            check_type(argname="argument function_url_auth_type", value=function_url_auth_type, expected_type=type_hints["function_url_auth_type"])
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument source_account", value=source_account, expected_type=type_hints["source_account"])
            check_type(argname="argument source_arn", value=source_arn, expected_type=type_hints["source_arn"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "principal": principal,
        }
        if action is not None:
            self._values["action"] = action
        if event_source_token is not None:
            self._values["event_source_token"] = event_source_token
        if function_url_auth_type is not None:
            self._values["function_url_auth_type"] = function_url_auth_type
        if scope is not None:
            self._values["scope"] = scope
        if source_account is not None:
            self._values["source_account"] = source_account
        if source_arn is not None:
            self._values["source_arn"] = source_arn

    @builtins.property
    def principal(self) -> _aws_cdk_aws_iam_940a1ce0.IPrincipal:
        '''The entity for which you are granting permission to invoke the Lambda function.

        This entity can be any valid AWS service principal, such as
        s3.amazonaws.com or sns.amazonaws.com, or, if you are granting
        cross-account permission, an AWS account ID. For example, you might want
        to allow a custom application in another AWS account to push events to
        Lambda by invoking your function.

        The principal can be either an AccountPrincipal or a ServicePrincipal.
        '''
        result = self._values.get("principal")
        assert result is not None, "Required property 'principal' is missing"
        return typing.cast(_aws_cdk_aws_iam_940a1ce0.IPrincipal, result)

    @builtins.property
    def action(self) -> typing.Optional[builtins.str]:
        '''The Lambda actions that you want to allow in this statement.

        For example,
        you can specify lambda:CreateFunction to specify a certain action, or use
        a wildcard (``lambda:*``) to grant permission to all Lambda actions. For a
        list of actions, see Actions and Condition Context Keys for AWS Lambda in
        the IAM User Guide.

        :default: 'lambda:InvokeFunction'
        '''
        result = self._values.get("action")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def event_source_token(self) -> typing.Optional[builtins.str]:
        '''A unique token that must be supplied by the principal invoking the function.

        :default: The caller would not need to present a token.
        '''
        result = self._values.get("event_source_token")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def function_url_auth_type(self) -> typing.Optional[FunctionUrlAuthType]:
        '''The authType for the function URL that you are granting permissions for.

        :default: - No functionUrlAuthType
        '''
        result = self._values.get("function_url_auth_type")
        return typing.cast(typing.Optional[FunctionUrlAuthType], result)

    @builtins.property
    def scope(self) -> typing.Optional[_aws_cdk_core_f4b25747.Construct]:
        '''The scope to which the permission constructs be attached.

        The default is
        the Lambda function construct itself, but this would need to be different
        in cases such as cross-stack references where the Permissions would need
        to sit closer to the consumer of this permission (i.e., the caller).

        :default: - The instance of lambda.IFunction
        '''
        result = self._values.get("scope")
        return typing.cast(typing.Optional[_aws_cdk_core_f4b25747.Construct], result)

    @builtins.property
    def source_account(self) -> typing.Optional[builtins.str]:
        '''The AWS account ID (without hyphens) of the source owner.

        For example, if
        you specify an S3 bucket in the SourceArn property, this value is the
        bucket owner's account ID. You can use this property to ensure that all
        source principals are owned by a specific account.
        '''
        result = self._values.get("source_account")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def source_arn(self) -> typing.Optional[builtins.str]:
        '''The ARN of a resource that is invoking your function.

        When granting
        Amazon Simple Storage Service (Amazon S3) permission to invoke your
        function, specify this property with the bucket ARN as its value. This
        ensures that events generated only from the specified bucket, not just
        any bucket from any AWS account that creates a mapping to your function,
        can invoke the function.
        '''
        result = self._values.get("source_arn")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "Permission(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-lambda.ResourceBindOptions",
    jsii_struct_bases=[],
    name_mapping={"resource_property": "resourceProperty"},
)
class ResourceBindOptions:
    def __init__(
        self,
        *,
        resource_property: typing.Optional[builtins.str] = None,
    ) -> None:
        '''
        :param resource_property: The name of the CloudFormation property to annotate with asset metadata. Default: Code

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_lambda as lambda_
            
            resource_bind_options = lambda.ResourceBindOptions(
                resource_property="resourceProperty"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__019aa1e5c7501a0f3e35e1370a01090d221e1a746c2972e2575bed926e951773)
            check_type(argname="argument resource_property", value=resource_property, expected_type=type_hints["resource_property"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if resource_property is not None:
            self._values["resource_property"] = resource_property

    @builtins.property
    def resource_property(self) -> typing.Optional[builtins.str]:
        '''The name of the CloudFormation property to annotate with asset metadata.

        :default: Code

        :see: https://github.com/aws/aws-cdk/issues/1432
        '''
        result = self._values.get("resource_property")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "ResourceBindOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class Runtime(metaclass=jsii.JSIIMeta, jsii_type="@aws-cdk/aws-lambda.Runtime"):
    '''Lambda function runtime environment.

    If you need to use a runtime name that doesn't exist as a static member, you
    can instantiate a ``Runtime`` object, e.g: ``new Runtime('nodejs99.99')``.

    :exampleMetadata: infused

    Example::

        import aws_cdk.aws_signer as signer
        
        
        signing_profile = signer.SigningProfile(self, "SigningProfile",
            platform=signer.Platform.AWS_LAMBDA_SHA384_ECDSA
        )
        
        code_signing_config = lambda_.CodeSigningConfig(self, "CodeSigningConfig",
            signing_profiles=[signing_profile]
        )
        
        lambda_.Function(self, "Function",
            code_signing_config=code_signing_config,
            runtime=lambda_.Runtime.NODEJS_16_X,
            handler="index.handler",
            code=lambda_.Code.from_asset(path.join(__dirname, "lambda-handler"))
        )
    '''

    def __init__(
        self,
        name: builtins.str,
        family: typing.Optional["RuntimeFamily"] = None,
        *,
        bundling_docker_image: typing.Optional[builtins.str] = None,
        supports_code_guru_profiling: typing.Optional[builtins.bool] = None,
        supports_inline_code: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''
        :param name: -
        :param family: -
        :param bundling_docker_image: The Docker image name to be used for bundling in this runtime. Default: - the latest docker image "amazon/public.ecr.aws/sam/build-" from https://gallery.ecr.aws
        :param supports_code_guru_profiling: Whether this runtime is integrated with and supported for profiling using Amazon CodeGuru Profiler. Default: false
        :param supports_inline_code: Whether the ``ZipFile`` (aka inline code) property can be used with this runtime. Default: false
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b54fa0efd42b09353e05a4ca50ce192d3e60ec313b6e263ae8af68bad760166b)
            check_type(argname="argument name", value=name, expected_type=type_hints["name"])
            check_type(argname="argument family", value=family, expected_type=type_hints["family"])
        props = LambdaRuntimeProps(
            bundling_docker_image=bundling_docker_image,
            supports_code_guru_profiling=supports_code_guru_profiling,
            supports_inline_code=supports_inline_code,
        )

        jsii.create(self.__class__, self, [name, family, props])

    @jsii.member(jsii_name="runtimeEquals")
    def runtime_equals(self, other: "Runtime") -> builtins.bool:
        '''
        :param other: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a7b21e14af9c2ae1e07cc8582ce0607fb643130dbe7f37d89bc7f3be198d129b)
            check_type(argname="argument other", value=other, expected_type=type_hints["other"])
        return typing.cast(builtins.bool, jsii.invoke(self, "runtimeEquals", [other]))

    @jsii.member(jsii_name="toString")
    def to_string(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.invoke(self, "toString", []))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ALL")
    def ALL(cls) -> typing.List["Runtime"]:
        '''A list of all known ``Runtime``'s.'''
        return typing.cast(typing.List["Runtime"], jsii.sget(cls, "ALL"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="DOTNET_6")
    def DOTNET_6(cls) -> "Runtime":
        '''The .NET 6 runtime (dotnet6).'''
        return typing.cast("Runtime", jsii.sget(cls, "DOTNET_6"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="DOTNET_CORE_1")
    def DOTNET_CORE_1(cls) -> "Runtime":
        '''The .NET Core 1.0 runtime (dotnetcore1.0) Legacy runtime no longer supported by AWS Lambda. Migrate to the latest .NET Core runtime.'''
        return typing.cast("Runtime", jsii.sget(cls, "DOTNET_CORE_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="DOTNET_CORE_2")
    def DOTNET_CORE_2(cls) -> "Runtime":
        '''The .NET Core 2.0 runtime (dotnetcore2.0) Legacy runtime no longer supported by AWS Lambda. Migrate to the latest .NET Core runtime.'''
        return typing.cast("Runtime", jsii.sget(cls, "DOTNET_CORE_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="DOTNET_CORE_2_1")
    def DOTNET_CORE_2_1(cls) -> "Runtime":
        '''The .NET Core 2.1 runtime (dotnetcore2.1) Legacy runtime no longer supported by AWS Lambda. Migrate to the latest .NET Core runtime.'''
        return typing.cast("Runtime", jsii.sget(cls, "DOTNET_CORE_2_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="DOTNET_CORE_3_1")
    def DOTNET_CORE_3_1(cls) -> "Runtime":
        '''The .NET Core 3.1 runtime (dotnetcore3.1).'''
        return typing.cast("Runtime", jsii.sget(cls, "DOTNET_CORE_3_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="FROM_IMAGE")
    def FROM_IMAGE(cls) -> "Runtime":
        '''A special runtime entry to be used when function is using a docker image.'''
        return typing.cast("Runtime", jsii.sget(cls, "FROM_IMAGE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="GO_1_X")
    def GO_1_X(cls) -> "Runtime":
        '''The Go 1.x runtime (go1.x).'''
        return typing.cast("Runtime", jsii.sget(cls, "GO_1_X"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="JAVA_11")
    def JAVA_11(cls) -> "Runtime":
        '''The Java 11 runtime (java11).'''
        return typing.cast("Runtime", jsii.sget(cls, "JAVA_11"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="JAVA_8")
    def JAVA_8(cls) -> "Runtime":
        '''The Java 8 runtime (java8).'''
        return typing.cast("Runtime", jsii.sget(cls, "JAVA_8"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="JAVA_8_CORRETTO")
    def JAVA_8_CORRETTO(cls) -> "Runtime":
        '''The Java 8 Corretto runtime (java8.al2).'''
        return typing.cast("Runtime", jsii.sget(cls, "JAVA_8_CORRETTO"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="NODEJS")
    def NODEJS(cls) -> "Runtime":
        '''(deprecated) The NodeJS runtime (nodejs).

        :deprecated: Legacy runtime no longer supported by AWS Lambda. Migrate to the latest NodeJS runtime.

        :stability: deprecated
        '''
        return typing.cast("Runtime", jsii.sget(cls, "NODEJS"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="NODEJS_10_X")
    def NODEJS_10_X(cls) -> "Runtime":
        '''(deprecated) The NodeJS 10.x runtime (nodejs10.x).

        :deprecated: Legacy runtime no longer supported by AWS Lambda. Migrate to the latest NodeJS runtime.

        :stability: deprecated
        '''
        return typing.cast("Runtime", jsii.sget(cls, "NODEJS_10_X"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="NODEJS_12_X")
    def NODEJS_12_X(cls) -> "Runtime":
        '''The NodeJS 12.x runtime (nodejs12.x).'''
        return typing.cast("Runtime", jsii.sget(cls, "NODEJS_12_X"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="NODEJS_14_X")
    def NODEJS_14_X(cls) -> "Runtime":
        '''The NodeJS 14.x runtime (nodejs14.x).'''
        return typing.cast("Runtime", jsii.sget(cls, "NODEJS_14_X"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="NODEJS_16_X")
    def NODEJS_16_X(cls) -> "Runtime":
        '''The NodeJS 16.x runtime (nodejs16.x).'''
        return typing.cast("Runtime", jsii.sget(cls, "NODEJS_16_X"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="NODEJS_4_3")
    def NODEJS_4_3(cls) -> "Runtime":
        '''(deprecated) The NodeJS 4.3 runtime (nodejs4.3).

        :deprecated: Legacy runtime no longer supported by AWS Lambda. Migrate to the latest NodeJS runtime.

        :stability: deprecated
        '''
        return typing.cast("Runtime", jsii.sget(cls, "NODEJS_4_3"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="NODEJS_6_10")
    def NODEJS_6_10(cls) -> "Runtime":
        '''(deprecated) The NodeJS 6.10 runtime (nodejs6.10).

        :deprecated: Legacy runtime no longer supported by AWS Lambda. Migrate to the latest NodeJS runtime.

        :stability: deprecated
        '''
        return typing.cast("Runtime", jsii.sget(cls, "NODEJS_6_10"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="NODEJS_8_10")
    def NODEJS_8_10(cls) -> "Runtime":
        '''(deprecated) The NodeJS 8.10 runtime (nodejs8.10).

        :deprecated: Legacy runtime no longer supported by AWS Lambda. Migrate to the latest NodeJS runtime.

        :stability: deprecated
        '''
        return typing.cast("Runtime", jsii.sget(cls, "NODEJS_8_10"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="PROVIDED")
    def PROVIDED(cls) -> "Runtime":
        '''The custom provided runtime (provided).'''
        return typing.cast("Runtime", jsii.sget(cls, "PROVIDED"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="PROVIDED_AL2")
    def PROVIDED_AL2(cls) -> "Runtime":
        '''The custom provided runtime (provided).'''
        return typing.cast("Runtime", jsii.sget(cls, "PROVIDED_AL2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="PYTHON_2_7")
    def PYTHON_2_7(cls) -> "Runtime":
        '''(deprecated) The Python 2.7 runtime (python2.7).

        :deprecated: Legacy runtime no longer supported by AWS Lambda. Migrate to the latest Python runtime.

        :stability: deprecated
        '''
        return typing.cast("Runtime", jsii.sget(cls, "PYTHON_2_7"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="PYTHON_3_6")
    def PYTHON_3_6(cls) -> "Runtime":
        '''(deprecated) The Python 3.6 runtime (python3.6) (not recommended).

        The Python 3.6 runtime is deprecated as of July 2022.

        :deprecated: Legacy runtime no longer supported by AWS Lambda. Migrate to the latest Python runtime.

        :stability: deprecated
        '''
        return typing.cast("Runtime", jsii.sget(cls, "PYTHON_3_6"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="PYTHON_3_7")
    def PYTHON_3_7(cls) -> "Runtime":
        '''The Python 3.7 runtime (python3.7).'''
        return typing.cast("Runtime", jsii.sget(cls, "PYTHON_3_7"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="PYTHON_3_8")
    def PYTHON_3_8(cls) -> "Runtime":
        '''The Python 3.8 runtime (python3.8).'''
        return typing.cast("Runtime", jsii.sget(cls, "PYTHON_3_8"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="PYTHON_3_9")
    def PYTHON_3_9(cls) -> "Runtime":
        '''The Python 3.9 runtime (python3.9).'''
        return typing.cast("Runtime", jsii.sget(cls, "PYTHON_3_9"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="RUBY_2_5")
    def RUBY_2_5(cls) -> "Runtime":
        '''The Ruby 2.5 runtime (ruby2.5) Legacy runtime no longer supported by AWS Lambda. Migrate to the latest Ruby runtime.'''
        return typing.cast("Runtime", jsii.sget(cls, "RUBY_2_5"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="RUBY_2_7")
    def RUBY_2_7(cls) -> "Runtime":
        '''The Ruby 2.7 runtime (ruby2.7).'''
        return typing.cast("Runtime", jsii.sget(cls, "RUBY_2_7"))

    @builtins.property
    @jsii.member(jsii_name="bundlingDockerImage")
    def bundling_docker_image(self) -> _aws_cdk_core_f4b25747.BundlingDockerImage:
        '''(deprecated) DEPRECATED.

        :deprecated: use ``bundlingImage``

        :stability: deprecated
        '''
        return typing.cast(_aws_cdk_core_f4b25747.BundlingDockerImage, jsii.get(self, "bundlingDockerImage"))

    @builtins.property
    @jsii.member(jsii_name="bundlingImage")
    def bundling_image(self) -> _aws_cdk_core_f4b25747.DockerImage:
        '''The bundling Docker image for this runtime.'''
        return typing.cast(_aws_cdk_core_f4b25747.DockerImage, jsii.get(self, "bundlingImage"))

    @builtins.property
    @jsii.member(jsii_name="name")
    def name(self) -> builtins.str:
        '''The name of this runtime, as expected by the Lambda resource.'''
        return typing.cast(builtins.str, jsii.get(self, "name"))

    @builtins.property
    @jsii.member(jsii_name="supportsCodeGuruProfiling")
    def supports_code_guru_profiling(self) -> builtins.bool:
        '''Whether this runtime is integrated with and supported for profiling using Amazon CodeGuru Profiler.'''
        return typing.cast(builtins.bool, jsii.get(self, "supportsCodeGuruProfiling"))

    @builtins.property
    @jsii.member(jsii_name="supportsInlineCode")
    def supports_inline_code(self) -> builtins.bool:
        '''Whether the ``ZipFile`` (aka inline code) property can be used with this runtime.'''
        return typing.cast(builtins.bool, jsii.get(self, "supportsInlineCode"))

    @builtins.property
    @jsii.member(jsii_name="family")
    def family(self) -> typing.Optional["RuntimeFamily"]:
        '''The runtime family.'''
        return typing.cast(typing.Optional["RuntimeFamily"], jsii.get(self, "family"))


@jsii.enum(jsii_type="@aws-cdk/aws-lambda.RuntimeFamily")
class RuntimeFamily(enum.Enum):
    NODEJS = "NODEJS"
    JAVA = "JAVA"
    PYTHON = "PYTHON"
    DOTNET_CORE = "DOTNET_CORE"
    GO = "GO"
    RUBY = "RUBY"
    OTHER = "OTHER"


class S3Code(Code, metaclass=jsii.JSIIMeta, jsii_type="@aws-cdk/aws-lambda.S3Code"):
    '''Lambda code from an S3 archive.

    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        import aws_cdk.aws_lambda as lambda_
        import aws_cdk.aws_s3 as s3
        
        # bucket: s3.Bucket
        
        s3_code = lambda_.S3Code(bucket, "key", "objectVersion")
    '''

    def __init__(
        self,
        bucket: _aws_cdk_aws_s3_55f001a5.IBucket,
        key: builtins.str,
        object_version: typing.Optional[builtins.str] = None,
    ) -> None:
        '''
        :param bucket: -
        :param key: -
        :param object_version: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0e0dc54e02bd972c31c1e50461a89301700312d03cb79728009f5a0936b9c80c)
            check_type(argname="argument bucket", value=bucket, expected_type=type_hints["bucket"])
            check_type(argname="argument key", value=key, expected_type=type_hints["key"])
            check_type(argname="argument object_version", value=object_version, expected_type=type_hints["object_version"])
        jsii.create(self.__class__, self, [bucket, key, object_version])

    @jsii.member(jsii_name="bind")
    def bind(self, _scope: _aws_cdk_core_f4b25747.Construct) -> CodeConfig:
        '''Called when the lambda or layer is initialized to allow this object to bind to the stack, add resources and have fun.

        :param _scope: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f87e71811405107202d4c02448e0a7c5847cb43cb3d7f332a508e394c2e7eaf3)
            check_type(argname="argument _scope", value=_scope, expected_type=type_hints["_scope"])
        return typing.cast(CodeConfig, jsii.invoke(self, "bind", [_scope]))

    @builtins.property
    @jsii.member(jsii_name="isInline")
    def is_inline(self) -> builtins.bool:
        '''Determines whether this Code is inline code or not.'''
        return typing.cast(builtins.bool, jsii.get(self, "isInline"))


@jsii.data_type(
    jsii_type="@aws-cdk/aws-lambda.SingletonFunctionProps",
    jsii_struct_bases=[FunctionProps],
    name_mapping={
        "max_event_age": "maxEventAge",
        "on_failure": "onFailure",
        "on_success": "onSuccess",
        "retry_attempts": "retryAttempts",
        "allow_all_outbound": "allowAllOutbound",
        "allow_public_subnet": "allowPublicSubnet",
        "architecture": "architecture",
        "architectures": "architectures",
        "code_signing_config": "codeSigningConfig",
        "current_version_options": "currentVersionOptions",
        "dead_letter_queue": "deadLetterQueue",
        "dead_letter_queue_enabled": "deadLetterQueueEnabled",
        "dead_letter_topic": "deadLetterTopic",
        "description": "description",
        "environment": "environment",
        "environment_encryption": "environmentEncryption",
        "ephemeral_storage_size": "ephemeralStorageSize",
        "events": "events",
        "filesystem": "filesystem",
        "function_name": "functionName",
        "initial_policy": "initialPolicy",
        "insights_version": "insightsVersion",
        "layers": "layers",
        "log_retention": "logRetention",
        "log_retention_retry_options": "logRetentionRetryOptions",
        "log_retention_role": "logRetentionRole",
        "memory_size": "memorySize",
        "profiling": "profiling",
        "profiling_group": "profilingGroup",
        "reserved_concurrent_executions": "reservedConcurrentExecutions",
        "role": "role",
        "security_group": "securityGroup",
        "security_groups": "securityGroups",
        "timeout": "timeout",
        "tracing": "tracing",
        "vpc": "vpc",
        "vpc_subnets": "vpcSubnets",
        "code": "code",
        "handler": "handler",
        "runtime": "runtime",
        "uuid": "uuid",
        "lambda_purpose": "lambdaPurpose",
    },
)
class SingletonFunctionProps(FunctionProps):
    def __init__(
        self,
        *,
        max_event_age: typing.Optional[_aws_cdk_core_f4b25747.Duration] = None,
        on_failure: typing.Optional[IDestination] = None,
        on_success: typing.Optional[IDestination] = None,
        retry_attempts: typing.Optional[jsii.Number] = None,
        allow_all_outbound: typing.Optional[builtins.bool] = None,
        allow_public_subnet: typing.Optional[builtins.bool] = None,
        architecture: typing.Optional[Architecture] = None,
        architectures: typing.Optional[typing.Sequence[Architecture]] = None,
        code_signing_config: typing.Optional[ICodeSigningConfig] = None,
        current_version_options: typing.Optional[typing.Union["VersionOptions", typing.Dict[builtins.str, typing.Any]]] = None,
        dead_letter_queue: typing.Optional[_aws_cdk_aws_sqs_48bffef9.IQueue] = None,
        dead_letter_queue_enabled: typing.Optional[builtins.bool] = None,
        dead_letter_topic: typing.Optional[_aws_cdk_aws_sns_889c7272.ITopic] = None,
        description: typing.Optional[builtins.str] = None,
        environment: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        environment_encryption: typing.Optional[_aws_cdk_aws_kms_e491a92b.IKey] = None,
        ephemeral_storage_size: typing.Optional[_aws_cdk_core_f4b25747.Size] = None,
        events: typing.Optional[typing.Sequence[IEventSource]] = None,
        filesystem: typing.Optional[FileSystem] = None,
        function_name: typing.Optional[builtins.str] = None,
        initial_policy: typing.Optional[typing.Sequence[_aws_cdk_aws_iam_940a1ce0.PolicyStatement]] = None,
        insights_version: typing.Optional[LambdaInsightsVersion] = None,
        layers: typing.Optional[typing.Sequence[ILayerVersion]] = None,
        log_retention: typing.Optional[_aws_cdk_aws_logs_6c4320fb.RetentionDays] = None,
        log_retention_retry_options: typing.Optional[typing.Union[LogRetentionRetryOptions, typing.Dict[builtins.str, typing.Any]]] = None,
        log_retention_role: typing.Optional[_aws_cdk_aws_iam_940a1ce0.IRole] = None,
        memory_size: typing.Optional[jsii.Number] = None,
        profiling: typing.Optional[builtins.bool] = None,
        profiling_group: typing.Optional[_aws_cdk_aws_codeguruprofiler_5a603484.IProfilingGroup] = None,
        reserved_concurrent_executions: typing.Optional[jsii.Number] = None,
        role: typing.Optional[_aws_cdk_aws_iam_940a1ce0.IRole] = None,
        security_group: typing.Optional[_aws_cdk_aws_ec2_67de8e8d.ISecurityGroup] = None,
        security_groups: typing.Optional[typing.Sequence[_aws_cdk_aws_ec2_67de8e8d.ISecurityGroup]] = None,
        timeout: typing.Optional[_aws_cdk_core_f4b25747.Duration] = None,
        tracing: typing.Optional["Tracing"] = None,
        vpc: typing.Optional[_aws_cdk_aws_ec2_67de8e8d.IVpc] = None,
        vpc_subnets: typing.Optional[typing.Union[_aws_cdk_aws_ec2_67de8e8d.SubnetSelection, typing.Dict[builtins.str, typing.Any]]] = None,
        code: Code,
        handler: builtins.str,
        runtime: Runtime,
        uuid: builtins.str,
        lambda_purpose: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for a newly created singleton Lambda.

        :param max_event_age: The maximum age of a request that Lambda sends to a function for processing. Minimum: 60 seconds Maximum: 6 hours Default: Duration.hours(6)
        :param on_failure: The destination for failed invocations. Default: - no destination
        :param on_success: The destination for successful invocations. Default: - no destination
        :param retry_attempts: The maximum number of times to retry when the function returns an error. Minimum: 0 Maximum: 2 Default: 2
        :param allow_all_outbound: Whether to allow the Lambda to send all network traffic. If set to false, you must individually add traffic rules to allow the Lambda to connect to network targets. Default: true
        :param allow_public_subnet: Lambda Functions in a public subnet can NOT access the internet. Use this property to acknowledge this limitation and still place the function in a public subnet. Default: false
        :param architecture: The system architectures compatible with this lambda function. Default: Architecture.X86_64
        :param architectures: (deprecated) DEPRECATED. Default: [Architecture.X86_64]
        :param code_signing_config: Code signing config associated with this function. Default: - Not Sign the Code
        :param current_version_options: Options for the ``lambda.Version`` resource automatically created by the ``fn.currentVersion`` method. Default: - default options as described in ``VersionOptions``
        :param dead_letter_queue: The SQS queue to use if DLQ is enabled. If SNS topic is desired, specify ``deadLetterTopic`` property instead. Default: - SQS queue with 14 day retention period if ``deadLetterQueueEnabled`` is ``true``
        :param dead_letter_queue_enabled: Enabled DLQ. If ``deadLetterQueue`` is undefined, an SQS queue with default options will be defined for your Function. Default: - false unless ``deadLetterQueue`` is set, which implies DLQ is enabled.
        :param dead_letter_topic: The SNS topic to use as a DLQ. Note that if ``deadLetterQueueEnabled`` is set to ``true``, an SQS queue will be created rather than an SNS topic. Using an SNS topic as a DLQ requires this property to be set explicitly. Default: - no SNS topic
        :param description: A description of the function. Default: - No description.
        :param environment: Key-value pairs that Lambda caches and makes available for your Lambda functions. Use environment variables to apply configuration changes, such as test and production environment configurations, without changing your Lambda function source code. Default: - No environment variables.
        :param environment_encryption: The AWS KMS key that's used to encrypt your function's environment variables. Default: - AWS Lambda creates and uses an AWS managed customer master key (CMK).
        :param ephemeral_storage_size: The size of the function’s /tmp directory in MiB. Default: 512 MiB
        :param events: Event sources for this function. You can also add event sources using ``addEventSource``. Default: - No event sources.
        :param filesystem: The filesystem configuration for the lambda function. Default: - will not mount any filesystem
        :param function_name: A name for the function. Default: - AWS CloudFormation generates a unique physical ID and uses that ID for the function's name. For more information, see Name Type.
        :param initial_policy: Initial policy statements to add to the created Lambda Role. You can call ``addToRolePolicy`` to the created lambda to add statements post creation. Default: - No policy statements are added to the created Lambda role.
        :param insights_version: Specify the version of CloudWatch Lambda insights to use for monitoring. Default: - No Lambda Insights
        :param layers: A list of layers to add to the function's execution environment. You can configure your Lambda function to pull in additional code during initialization in the form of layers. Layers are packages of libraries or other dependencies that can be used by multiple functions. Default: - No layers.
        :param log_retention: The number of days log events are kept in CloudWatch Logs. When updating this property, unsetting it doesn't remove the log retention policy. To remove the retention policy, set the value to ``INFINITE``. Default: logs.RetentionDays.INFINITE
        :param log_retention_retry_options: When log retention is specified, a custom resource attempts to create the CloudWatch log group. These options control the retry policy when interacting with CloudWatch APIs. Default: - Default AWS SDK retry options.
        :param log_retention_role: The IAM role for the Lambda function associated with the custom resource that sets the retention policy. Default: - A new role is created.
        :param memory_size: The amount of memory, in MB, that is allocated to your Lambda function. Lambda uses this value to proportionally allocate the amount of CPU power. For more information, see Resource Model in the AWS Lambda Developer Guide. Default: 128
        :param profiling: Enable profiling. Default: - No profiling.
        :param profiling_group: Profiling Group. Default: - A new profiling group will be created if ``profiling`` is set.
        :param reserved_concurrent_executions: The maximum of concurrent executions you want to reserve for the function. Default: - No specific limit - account limit.
        :param role: Lambda execution role. This is the role that will be assumed by the function upon execution. It controls the permissions that the function will have. The Role must be assumable by the 'lambda.amazonaws.com' service principal. The default Role automatically has permissions granted for Lambda execution. If you provide a Role, you must add the relevant AWS managed policies yourself. The relevant managed policies are "service-role/AWSLambdaBasicExecutionRole" and "service-role/AWSLambdaVPCAccessExecutionRole". Default: - A unique role will be generated for this lambda function. Both supplied and generated roles can always be changed by calling ``addToRolePolicy``.
        :param security_group: (deprecated) What security group to associate with the Lambda's network interfaces. This property is being deprecated, consider using securityGroups instead. Only used if 'vpc' is supplied. Use securityGroups property instead. Function constructor will throw an error if both are specified. Default: - If the function is placed within a VPC and a security group is not specified, either by this or securityGroups prop, a dedicated security group will be created for this function.
        :param security_groups: The list of security groups to associate with the Lambda's network interfaces. Only used if 'vpc' is supplied. Default: - If the function is placed within a VPC and a security group is not specified, either by this or securityGroup prop, a dedicated security group will be created for this function.
        :param timeout: The function execution time (in seconds) after which Lambda terminates the function. Because the execution time affects cost, set this value based on the function's expected execution time. Default: Duration.seconds(3)
        :param tracing: Enable AWS X-Ray Tracing for Lambda Function. Default: Tracing.Disabled
        :param vpc: VPC network to place Lambda network interfaces. Specify this if the Lambda function needs to access resources in a VPC. Default: - Function is not placed within a VPC.
        :param vpc_subnets: Where to place the network interfaces within the VPC. Only used if 'vpc' is supplied. Note: internet access for Lambdas requires a NAT gateway, so picking Public subnets is not allowed. Default: - the Vpc default strategy if not specified
        :param code: The source code of your Lambda function. You can point to a file in an Amazon Simple Storage Service (Amazon S3) bucket or specify your source code as inline text.
        :param handler: The name of the method within your code that Lambda calls to execute your function. The format includes the file name. It can also include namespaces and other qualifiers, depending on the runtime. For more information, see https://docs.aws.amazon.com/lambda/latest/dg/gettingstarted-features.html#gettingstarted-features-programmingmodel. Use ``Handler.FROM_IMAGE`` when defining a function from a Docker image. NOTE: If you specify your source code as inline text by specifying the ZipFile property within the Code property, specify index.function_name as the handler.
        :param runtime: The runtime environment for the Lambda function that you are uploading. For valid values, see the Runtime property in the AWS Lambda Developer Guide. Use ``Runtime.FROM_IMAGE`` when when defining a function from a Docker image.
        :param uuid: A unique identifier to identify this lambda. The identifier should be unique across all custom resource providers. We recommend generating a UUID per provider.
        :param lambda_purpose: A descriptive name for the purpose of this Lambda. If the Lambda does not have a physical name, this string will be reflected its generated name. The combination of lambdaPurpose and uuid must be unique. Default: SingletonLambda

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_codeguruprofiler as codeguruprofiler
            import aws_cdk.aws_ec2 as ec2
            import aws_cdk.aws_iam as iam
            import aws_cdk.aws_kms as kms
            import aws_cdk.aws_lambda as lambda_
            import aws_cdk.aws_logs as logs
            import aws_cdk.aws_sns as sns
            import aws_cdk.aws_sqs as sqs
            import aws_cdk.core as cdk
            
            # architecture: lambda.Architecture
            # code: lambda.Code
            # code_signing_config: lambda.CodeSigningConfig
            # destination: lambda.IDestination
            # event_source: lambda.IEventSource
            # file_system: lambda.FileSystem
            # key: kms.Key
            # lambda_insights_version: lambda.LambdaInsightsVersion
            # layer_version: lambda.LayerVersion
            # policy_statement: iam.PolicyStatement
            # profiling_group: codeguruprofiler.ProfilingGroup
            # queue: sqs.Queue
            # role: iam.Role
            # runtime: lambda.Runtime
            # security_group: ec2.SecurityGroup
            # size: cdk.Size
            # subnet: ec2.Subnet
            # subnet_filter: ec2.SubnetFilter
            # topic: sns.Topic
            # vpc: ec2.Vpc
            
            singleton_function_props = lambda.SingletonFunctionProps(
                code=code,
                handler="handler",
                runtime=runtime,
                uuid="uuid",
            
                # the properties below are optional
                allow_all_outbound=False,
                allow_public_subnet=False,
                architecture=architecture,
                architectures=[architecture],
                code_signing_config=code_signing_config,
                current_version_options=lambda.VersionOptions(
                    code_sha256="codeSha256",
                    description="description",
                    max_event_age=cdk.Duration.minutes(30),
                    on_failure=destination,
                    on_success=destination,
                    provisioned_concurrent_executions=123,
                    removal_policy=cdk.RemovalPolicy.DESTROY,
                    retry_attempts=123
                ),
                dead_letter_queue=queue,
                dead_letter_queue_enabled=False,
                dead_letter_topic=topic,
                description="description",
                environment={
                    "environment_key": "environment"
                },
                environment_encryption=key,
                ephemeral_storage_size=size,
                events=[event_source],
                filesystem=file_system,
                function_name="functionName",
                initial_policy=[policy_statement],
                insights_version=lambda_insights_version,
                lambda_purpose="lambdaPurpose",
                layers=[layer_version],
                log_retention=logs.RetentionDays.ONE_DAY,
                log_retention_retry_options=lambda.LogRetentionRetryOptions(
                    base=cdk.Duration.minutes(30),
                    max_retries=123
                ),
                log_retention_role=role,
                max_event_age=cdk.Duration.minutes(30),
                memory_size=123,
                on_failure=destination,
                on_success=destination,
                profiling=False,
                profiling_group=profiling_group,
                reserved_concurrent_executions=123,
                retry_attempts=123,
                role=role,
                security_group=security_group,
                security_groups=[security_group],
                timeout=cdk.Duration.minutes(30),
                tracing=lambda_.Tracing.ACTIVE,
                vpc=vpc,
                vpc_subnets=ec2.SubnetSelection(
                    availability_zones=["availabilityZones"],
                    one_per_az=False,
                    subnet_filters=[subnet_filter],
                    subnet_group_name="subnetGroupName",
                    subnet_name="subnetName",
                    subnets=[subnet],
                    subnet_type=ec2.SubnetType.ISOLATED
                )
            )
        '''
        if isinstance(current_version_options, dict):
            current_version_options = VersionOptions(**current_version_options)
        if isinstance(log_retention_retry_options, dict):
            log_retention_retry_options = LogRetentionRetryOptions(**log_retention_retry_options)
        if isinstance(vpc_subnets, dict):
            vpc_subnets = _aws_cdk_aws_ec2_67de8e8d.SubnetSelection(**vpc_subnets)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__216d7cc4e65666ed9a1bcb247a5fd3b13b9152a4161a80f50cfcab0ff381481e)
            check_type(argname="argument max_event_age", value=max_event_age, expected_type=type_hints["max_event_age"])
            check_type(argname="argument on_failure", value=on_failure, expected_type=type_hints["on_failure"])
            check_type(argname="argument on_success", value=on_success, expected_type=type_hints["on_success"])
            check_type(argname="argument retry_attempts", value=retry_attempts, expected_type=type_hints["retry_attempts"])
            check_type(argname="argument allow_all_outbound", value=allow_all_outbound, expected_type=type_hints["allow_all_outbound"])
            check_type(argname="argument allow_public_subnet", value=allow_public_subnet, expected_type=type_hints["allow_public_subnet"])
            check_type(argname="argument architecture", value=architecture, expected_type=type_hints["architecture"])
            check_type(argname="argument architectures", value=architectures, expected_type=type_hints["architectures"])
            check_type(argname="argument code_signing_config", value=code_signing_config, expected_type=type_hints["code_signing_config"])
            check_type(argname="argument current_version_options", value=current_version_options, expected_type=type_hints["current_version_options"])
            check_type(argname="argument dead_letter_queue", value=dead_letter_queue, expected_type=type_hints["dead_letter_queue"])
            check_type(argname="argument dead_letter_queue_enabled", value=dead_letter_queue_enabled, expected_type=type_hints["dead_letter_queue_enabled"])
            check_type(argname="argument dead_letter_topic", value=dead_letter_topic, expected_type=type_hints["dead_letter_topic"])
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument environment", value=environment, expected_type=type_hints["environment"])
            check_type(argname="argument environment_encryption", value=environment_encryption, expected_type=type_hints["environment_encryption"])
            check_type(argname="argument ephemeral_storage_size", value=ephemeral_storage_size, expected_type=type_hints["ephemeral_storage_size"])
            check_type(argname="argument events", value=events, expected_type=type_hints["events"])
            check_type(argname="argument filesystem", value=filesystem, expected_type=type_hints["filesystem"])
            check_type(argname="argument function_name", value=function_name, expected_type=type_hints["function_name"])
            check_type(argname="argument initial_policy", value=initial_policy, expected_type=type_hints["initial_policy"])
            check_type(argname="argument insights_version", value=insights_version, expected_type=type_hints["insights_version"])
            check_type(argname="argument layers", value=layers, expected_type=type_hints["layers"])
            check_type(argname="argument log_retention", value=log_retention, expected_type=type_hints["log_retention"])
            check_type(argname="argument log_retention_retry_options", value=log_retention_retry_options, expected_type=type_hints["log_retention_retry_options"])
            check_type(argname="argument log_retention_role", value=log_retention_role, expected_type=type_hints["log_retention_role"])
            check_type(argname="argument memory_size", value=memory_size, expected_type=type_hints["memory_size"])
            check_type(argname="argument profiling", value=profiling, expected_type=type_hints["profiling"])
            check_type(argname="argument profiling_group", value=profiling_group, expected_type=type_hints["profiling_group"])
            check_type(argname="argument reserved_concurrent_executions", value=reserved_concurrent_executions, expected_type=type_hints["reserved_concurrent_executions"])
            check_type(argname="argument role", value=role, expected_type=type_hints["role"])
            check_type(argname="argument security_group", value=security_group, expected_type=type_hints["security_group"])
            check_type(argname="argument security_groups", value=security_groups, expected_type=type_hints["security_groups"])
            check_type(argname="argument timeout", value=timeout, expected_type=type_hints["timeout"])
            check_type(argname="argument tracing", value=tracing, expected_type=type_hints["tracing"])
            check_type(argname="argument vpc", value=vpc, expected_type=type_hints["vpc"])
            check_type(argname="argument vpc_subnets", value=vpc_subnets, expected_type=type_hints["vpc_subnets"])
            check_type(argname="argument code", value=code, expected_type=type_hints["code"])
            check_type(argname="argument handler", value=handler, expected_type=type_hints["handler"])
            check_type(argname="argument runtime", value=runtime, expected_type=type_hints["runtime"])
            check_type(argname="argument uuid", value=uuid, expected_type=type_hints["uuid"])
            check_type(argname="argument lambda_purpose", value=lambda_purpose, expected_type=type_hints["lambda_purpose"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "code": code,
            "handler": handler,
            "runtime": runtime,
            "uuid": uuid,
        }
        if max_event_age is not None:
            self._values["max_event_age"] = max_event_age
        if on_failure is not None:
            self._values["on_failure"] = on_failure
        if on_success is not None:
            self._values["on_success"] = on_success
        if retry_attempts is not None:
            self._values["retry_attempts"] = retry_attempts
        if allow_all_outbound is not None:
            self._values["allow_all_outbound"] = allow_all_outbound
        if allow_public_subnet is not None:
            self._values["allow_public_subnet"] = allow_public_subnet
        if architecture is not None:
            self._values["architecture"] = architecture
        if architectures is not None:
            self._values["architectures"] = architectures
        if code_signing_config is not None:
            self._values["code_signing_config"] = code_signing_config
        if current_version_options is not None:
            self._values["current_version_options"] = current_version_options
        if dead_letter_queue is not None:
            self._values["dead_letter_queue"] = dead_letter_queue
        if dead_letter_queue_enabled is not None:
            self._values["dead_letter_queue_enabled"] = dead_letter_queue_enabled
        if dead_letter_topic is not None:
            self._values["dead_letter_topic"] = dead_letter_topic
        if description is not None:
            self._values["description"] = description
        if environment is not None:
            self._values["environment"] = environment
        if environment_encryption is not None:
            self._values["environment_encryption"] = environment_encryption
        if ephemeral_storage_size is not None:
            self._values["ephemeral_storage_size"] = ephemeral_storage_size
        if events is not None:
            self._values["events"] = events
        if filesystem is not None:
            self._values["filesystem"] = filesystem
        if function_name is not None:
            self._values["function_name"] = function_name
        if initial_policy is not None:
            self._values["initial_policy"] = initial_policy
        if insights_version is not None:
            self._values["insights_version"] = insights_version
        if layers is not None:
            self._values["layers"] = layers
        if log_retention is not None:
            self._values["log_retention"] = log_retention
        if log_retention_retry_options is not None:
            self._values["log_retention_retry_options"] = log_retention_retry_options
        if log_retention_role is not None:
            self._values["log_retention_role"] = log_retention_role
        if memory_size is not None:
            self._values["memory_size"] = memory_size
        if profiling is not None:
            self._values["profiling"] = profiling
        if profiling_group is not None:
            self._values["profiling_group"] = profiling_group
        if reserved_concurrent_executions is not None:
            self._values["reserved_concurrent_executions"] = reserved_concurrent_executions
        if role is not None:
            self._values["role"] = role
        if security_group is not None:
            self._values["security_group"] = security_group
        if security_groups is not None:
            self._values["security_groups"] = security_groups
        if timeout is not None:
            self._values["timeout"] = timeout
        if tracing is not None:
            self._values["tracing"] = tracing
        if vpc is not None:
            self._values["vpc"] = vpc
        if vpc_subnets is not None:
            self._values["vpc_subnets"] = vpc_subnets
        if lambda_purpose is not None:
            self._values["lambda_purpose"] = lambda_purpose

    @builtins.property
    def max_event_age(self) -> typing.Optional[_aws_cdk_core_f4b25747.Duration]:
        '''The maximum age of a request that Lambda sends to a function for processing.

        Minimum: 60 seconds
        Maximum: 6 hours

        :default: Duration.hours(6)
        '''
        result = self._values.get("max_event_age")
        return typing.cast(typing.Optional[_aws_cdk_core_f4b25747.Duration], result)

    @builtins.property
    def on_failure(self) -> typing.Optional[IDestination]:
        '''The destination for failed invocations.

        :default: - no destination
        '''
        result = self._values.get("on_failure")
        return typing.cast(typing.Optional[IDestination], result)

    @builtins.property
    def on_success(self) -> typing.Optional[IDestination]:
        '''The destination for successful invocations.

        :default: - no destination
        '''
        result = self._values.get("on_success")
        return typing.cast(typing.Optional[IDestination], result)

    @builtins.property
    def retry_attempts(self) -> typing.Optional[jsii.Number]:
        '''The maximum number of times to retry when the function returns an error.

        Minimum: 0
        Maximum: 2

        :default: 2
        '''
        result = self._values.get("retry_attempts")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def allow_all_outbound(self) -> typing.Optional[builtins.bool]:
        '''Whether to allow the Lambda to send all network traffic.

        If set to false, you must individually add traffic rules to allow the
        Lambda to connect to network targets.

        :default: true
        '''
        result = self._values.get("allow_all_outbound")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def allow_public_subnet(self) -> typing.Optional[builtins.bool]:
        '''Lambda Functions in a public subnet can NOT access the internet.

        Use this property to acknowledge this limitation and still place the function in a public subnet.

        :default: false

        :see: https://stackoverflow.com/questions/52992085/why-cant-an-aws-lambda-function-inside-a-public-subnet-in-a-vpc-connect-to-the/52994841#52994841
        '''
        result = self._values.get("allow_public_subnet")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def architecture(self) -> typing.Optional[Architecture]:
        '''The system architectures compatible with this lambda function.

        :default: Architecture.X86_64
        '''
        result = self._values.get("architecture")
        return typing.cast(typing.Optional[Architecture], result)

    @builtins.property
    def architectures(self) -> typing.Optional[typing.List[Architecture]]:
        '''(deprecated) DEPRECATED.

        :default: [Architecture.X86_64]

        :deprecated: use ``architecture``

        :stability: deprecated
        '''
        result = self._values.get("architectures")
        return typing.cast(typing.Optional[typing.List[Architecture]], result)

    @builtins.property
    def code_signing_config(self) -> typing.Optional[ICodeSigningConfig]:
        '''Code signing config associated with this function.

        :default: - Not Sign the Code
        '''
        result = self._values.get("code_signing_config")
        return typing.cast(typing.Optional[ICodeSigningConfig], result)

    @builtins.property
    def current_version_options(self) -> typing.Optional["VersionOptions"]:
        '''Options for the ``lambda.Version`` resource automatically created by the ``fn.currentVersion`` method.

        :default: - default options as described in ``VersionOptions``
        '''
        result = self._values.get("current_version_options")
        return typing.cast(typing.Optional["VersionOptions"], result)

    @builtins.property
    def dead_letter_queue(self) -> typing.Optional[_aws_cdk_aws_sqs_48bffef9.IQueue]:
        '''The SQS queue to use if DLQ is enabled.

        If SNS topic is desired, specify ``deadLetterTopic`` property instead.

        :default: - SQS queue with 14 day retention period if ``deadLetterQueueEnabled`` is ``true``
        '''
        result = self._values.get("dead_letter_queue")
        return typing.cast(typing.Optional[_aws_cdk_aws_sqs_48bffef9.IQueue], result)

    @builtins.property
    def dead_letter_queue_enabled(self) -> typing.Optional[builtins.bool]:
        '''Enabled DLQ.

        If ``deadLetterQueue`` is undefined,
        an SQS queue with default options will be defined for your Function.

        :default: - false unless ``deadLetterQueue`` is set, which implies DLQ is enabled.
        '''
        result = self._values.get("dead_letter_queue_enabled")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def dead_letter_topic(self) -> typing.Optional[_aws_cdk_aws_sns_889c7272.ITopic]:
        '''The SNS topic to use as a DLQ.

        Note that if ``deadLetterQueueEnabled`` is set to ``true``, an SQS queue will be created
        rather than an SNS topic. Using an SNS topic as a DLQ requires this property to be set explicitly.

        :default: - no SNS topic
        '''
        result = self._values.get("dead_letter_topic")
        return typing.cast(typing.Optional[_aws_cdk_aws_sns_889c7272.ITopic], result)

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''A description of the function.

        :default: - No description.
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def environment(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''Key-value pairs that Lambda caches and makes available for your Lambda functions.

        Use environment variables to apply configuration changes, such
        as test and production environment configurations, without changing your
        Lambda function source code.

        :default: - No environment variables.
        '''
        result = self._values.get("environment")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    @builtins.property
    def environment_encryption(self) -> typing.Optional[_aws_cdk_aws_kms_e491a92b.IKey]:
        '''The AWS KMS key that's used to encrypt your function's environment variables.

        :default: - AWS Lambda creates and uses an AWS managed customer master key (CMK).
        '''
        result = self._values.get("environment_encryption")
        return typing.cast(typing.Optional[_aws_cdk_aws_kms_e491a92b.IKey], result)

    @builtins.property
    def ephemeral_storage_size(self) -> typing.Optional[_aws_cdk_core_f4b25747.Size]:
        '''The size of the function’s /tmp directory in MiB.

        :default: 512 MiB
        '''
        result = self._values.get("ephemeral_storage_size")
        return typing.cast(typing.Optional[_aws_cdk_core_f4b25747.Size], result)

    @builtins.property
    def events(self) -> typing.Optional[typing.List[IEventSource]]:
        '''Event sources for this function.

        You can also add event sources using ``addEventSource``.

        :default: - No event sources.
        '''
        result = self._values.get("events")
        return typing.cast(typing.Optional[typing.List[IEventSource]], result)

    @builtins.property
    def filesystem(self) -> typing.Optional[FileSystem]:
        '''The filesystem configuration for the lambda function.

        :default: - will not mount any filesystem
        '''
        result = self._values.get("filesystem")
        return typing.cast(typing.Optional[FileSystem], result)

    @builtins.property
    def function_name(self) -> typing.Optional[builtins.str]:
        '''A name for the function.

        :default:

        - AWS CloudFormation generates a unique physical ID and uses that
        ID for the function's name. For more information, see Name Type.
        '''
        result = self._values.get("function_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def initial_policy(
        self,
    ) -> typing.Optional[typing.List[_aws_cdk_aws_iam_940a1ce0.PolicyStatement]]:
        '''Initial policy statements to add to the created Lambda Role.

        You can call ``addToRolePolicy`` to the created lambda to add statements post creation.

        :default: - No policy statements are added to the created Lambda role.
        '''
        result = self._values.get("initial_policy")
        return typing.cast(typing.Optional[typing.List[_aws_cdk_aws_iam_940a1ce0.PolicyStatement]], result)

    @builtins.property
    def insights_version(self) -> typing.Optional[LambdaInsightsVersion]:
        '''Specify the version of CloudWatch Lambda insights to use for monitoring.

        :default: - No Lambda Insights

        :see: https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/Lambda-Insights-Getting-Started-docker.html
        '''
        result = self._values.get("insights_version")
        return typing.cast(typing.Optional[LambdaInsightsVersion], result)

    @builtins.property
    def layers(self) -> typing.Optional[typing.List[ILayerVersion]]:
        '''A list of layers to add to the function's execution environment.

        You can configure your Lambda function to pull in
        additional code during initialization in the form of layers. Layers are packages of libraries or other dependencies
        that can be used by multiple functions.

        :default: - No layers.
        '''
        result = self._values.get("layers")
        return typing.cast(typing.Optional[typing.List[ILayerVersion]], result)

    @builtins.property
    def log_retention(
        self,
    ) -> typing.Optional[_aws_cdk_aws_logs_6c4320fb.RetentionDays]:
        '''The number of days log events are kept in CloudWatch Logs.

        When updating
        this property, unsetting it doesn't remove the log retention policy. To
        remove the retention policy, set the value to ``INFINITE``.

        :default: logs.RetentionDays.INFINITE
        '''
        result = self._values.get("log_retention")
        return typing.cast(typing.Optional[_aws_cdk_aws_logs_6c4320fb.RetentionDays], result)

    @builtins.property
    def log_retention_retry_options(self) -> typing.Optional[LogRetentionRetryOptions]:
        '''When log retention is specified, a custom resource attempts to create the CloudWatch log group.

        These options control the retry policy when interacting with CloudWatch APIs.

        :default: - Default AWS SDK retry options.
        '''
        result = self._values.get("log_retention_retry_options")
        return typing.cast(typing.Optional[LogRetentionRetryOptions], result)

    @builtins.property
    def log_retention_role(self) -> typing.Optional[_aws_cdk_aws_iam_940a1ce0.IRole]:
        '''The IAM role for the Lambda function associated with the custom resource that sets the retention policy.

        :default: - A new role is created.
        '''
        result = self._values.get("log_retention_role")
        return typing.cast(typing.Optional[_aws_cdk_aws_iam_940a1ce0.IRole], result)

    @builtins.property
    def memory_size(self) -> typing.Optional[jsii.Number]:
        '''The amount of memory, in MB, that is allocated to your Lambda function.

        Lambda uses this value to proportionally allocate the amount of CPU
        power. For more information, see Resource Model in the AWS Lambda
        Developer Guide.

        :default: 128
        '''
        result = self._values.get("memory_size")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def profiling(self) -> typing.Optional[builtins.bool]:
        '''Enable profiling.

        :default: - No profiling.

        :see: https://docs.aws.amazon.com/codeguru/latest/profiler-ug/setting-up-lambda.html
        '''
        result = self._values.get("profiling")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def profiling_group(
        self,
    ) -> typing.Optional[_aws_cdk_aws_codeguruprofiler_5a603484.IProfilingGroup]:
        '''Profiling Group.

        :default: - A new profiling group will be created if ``profiling`` is set.

        :see: https://docs.aws.amazon.com/codeguru/latest/profiler-ug/setting-up-lambda.html
        '''
        result = self._values.get("profiling_group")
        return typing.cast(typing.Optional[_aws_cdk_aws_codeguruprofiler_5a603484.IProfilingGroup], result)

    @builtins.property
    def reserved_concurrent_executions(self) -> typing.Optional[jsii.Number]:
        '''The maximum of concurrent executions you want to reserve for the function.

        :default: - No specific limit - account limit.

        :see: https://docs.aws.amazon.com/lambda/latest/dg/concurrent-executions.html
        '''
        result = self._values.get("reserved_concurrent_executions")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def role(self) -> typing.Optional[_aws_cdk_aws_iam_940a1ce0.IRole]:
        '''Lambda execution role.

        This is the role that will be assumed by the function upon execution.
        It controls the permissions that the function will have. The Role must
        be assumable by the 'lambda.amazonaws.com' service principal.

        The default Role automatically has permissions granted for Lambda execution. If you
        provide a Role, you must add the relevant AWS managed policies yourself.

        The relevant managed policies are "service-role/AWSLambdaBasicExecutionRole" and
        "service-role/AWSLambdaVPCAccessExecutionRole".

        :default:

        - A unique role will be generated for this lambda function.
        Both supplied and generated roles can always be changed by calling ``addToRolePolicy``.
        '''
        result = self._values.get("role")
        return typing.cast(typing.Optional[_aws_cdk_aws_iam_940a1ce0.IRole], result)

    @builtins.property
    def security_group(
        self,
    ) -> typing.Optional[_aws_cdk_aws_ec2_67de8e8d.ISecurityGroup]:
        '''(deprecated) What security group to associate with the Lambda's network interfaces. This property is being deprecated, consider using securityGroups instead.

        Only used if 'vpc' is supplied.

        Use securityGroups property instead.
        Function constructor will throw an error if both are specified.

        :default:

        - If the function is placed within a VPC and a security group is
        not specified, either by this or securityGroups prop, a dedicated security
        group will be created for this function.

        :deprecated: - This property is deprecated, use securityGroups instead

        :stability: deprecated
        '''
        result = self._values.get("security_group")
        return typing.cast(typing.Optional[_aws_cdk_aws_ec2_67de8e8d.ISecurityGroup], result)

    @builtins.property
    def security_groups(
        self,
    ) -> typing.Optional[typing.List[_aws_cdk_aws_ec2_67de8e8d.ISecurityGroup]]:
        '''The list of security groups to associate with the Lambda's network interfaces.

        Only used if 'vpc' is supplied.

        :default:

        - If the function is placed within a VPC and a security group is
        not specified, either by this or securityGroup prop, a dedicated security
        group will be created for this function.
        '''
        result = self._values.get("security_groups")
        return typing.cast(typing.Optional[typing.List[_aws_cdk_aws_ec2_67de8e8d.ISecurityGroup]], result)

    @builtins.property
    def timeout(self) -> typing.Optional[_aws_cdk_core_f4b25747.Duration]:
        '''The function execution time (in seconds) after which Lambda terminates the function.

        Because the execution time affects cost, set this value
        based on the function's expected execution time.

        :default: Duration.seconds(3)
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[_aws_cdk_core_f4b25747.Duration], result)

    @builtins.property
    def tracing(self) -> typing.Optional["Tracing"]:
        '''Enable AWS X-Ray Tracing for Lambda Function.

        :default: Tracing.Disabled
        '''
        result = self._values.get("tracing")
        return typing.cast(typing.Optional["Tracing"], result)

    @builtins.property
    def vpc(self) -> typing.Optional[_aws_cdk_aws_ec2_67de8e8d.IVpc]:
        '''VPC network to place Lambda network interfaces.

        Specify this if the Lambda function needs to access resources in a VPC.

        :default: - Function is not placed within a VPC.
        '''
        result = self._values.get("vpc")
        return typing.cast(typing.Optional[_aws_cdk_aws_ec2_67de8e8d.IVpc], result)

    @builtins.property
    def vpc_subnets(self) -> typing.Optional[_aws_cdk_aws_ec2_67de8e8d.SubnetSelection]:
        '''Where to place the network interfaces within the VPC.

        Only used if 'vpc' is supplied. Note: internet access for Lambdas
        requires a NAT gateway, so picking Public subnets is not allowed.

        :default: - the Vpc default strategy if not specified
        '''
        result = self._values.get("vpc_subnets")
        return typing.cast(typing.Optional[_aws_cdk_aws_ec2_67de8e8d.SubnetSelection], result)

    @builtins.property
    def code(self) -> Code:
        '''The source code of your Lambda function.

        You can point to a file in an
        Amazon Simple Storage Service (Amazon S3) bucket or specify your source
        code as inline text.
        '''
        result = self._values.get("code")
        assert result is not None, "Required property 'code' is missing"
        return typing.cast(Code, result)

    @builtins.property
    def handler(self) -> builtins.str:
        '''The name of the method within your code that Lambda calls to execute your function.

        The format includes the file name. It can also include
        namespaces and other qualifiers, depending on the runtime.
        For more information, see https://docs.aws.amazon.com/lambda/latest/dg/gettingstarted-features.html#gettingstarted-features-programmingmodel.

        Use ``Handler.FROM_IMAGE`` when defining a function from a Docker image.

        NOTE: If you specify your source code as inline text by specifying the
        ZipFile property within the Code property, specify index.function_name as
        the handler.
        '''
        result = self._values.get("handler")
        assert result is not None, "Required property 'handler' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def runtime(self) -> Runtime:
        '''The runtime environment for the Lambda function that you are uploading.

        For valid values, see the Runtime property in the AWS Lambda Developer
        Guide.

        Use ``Runtime.FROM_IMAGE`` when when defining a function from a Docker image.
        '''
        result = self._values.get("runtime")
        assert result is not None, "Required property 'runtime' is missing"
        return typing.cast(Runtime, result)

    @builtins.property
    def uuid(self) -> builtins.str:
        '''A unique identifier to identify this lambda.

        The identifier should be unique across all custom resource providers.
        We recommend generating a UUID per provider.
        '''
        result = self._values.get("uuid")
        assert result is not None, "Required property 'uuid' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def lambda_purpose(self) -> typing.Optional[builtins.str]:
        '''A descriptive name for the purpose of this Lambda.

        If the Lambda does not have a physical name, this string will be
        reflected its generated name. The combination of lambdaPurpose
        and uuid must be unique.

        :default: SingletonLambda
        '''
        result = self._values.get("lambda_purpose")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "SingletonFunctionProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-lambda.SourceAccessConfiguration",
    jsii_struct_bases=[],
    name_mapping={"type": "type", "uri": "uri"},
)
class SourceAccessConfiguration:
    def __init__(
        self,
        *,
        type: "SourceAccessConfigurationType",
        uri: builtins.str,
    ) -> None:
        '''Specific settings like the authentication protocol or the VPC components to secure access to your event source.

        :param type: The type of authentication protocol or the VPC components for your event source. For example: "SASL_SCRAM_512_AUTH".
        :param uri: The value for your chosen configuration in type. For example: "URI": "arn:aws:secretsmanager:us-east-1:01234567890:secret:MyBrokerSecretName". The exact string depends on the type.

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_lambda as lambda_
            
            # source_access_configuration_type: lambda.SourceAccessConfigurationType
            
            source_access_configuration = lambda.SourceAccessConfiguration(
                type=source_access_configuration_type,
                uri="uri"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__cb9695bbd46038a908189d97ad251a90077496b82eb9093c771711b8e5aa0667)
            check_type(argname="argument type", value=type, expected_type=type_hints["type"])
            check_type(argname="argument uri", value=uri, expected_type=type_hints["uri"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "type": type,
            "uri": uri,
        }

    @builtins.property
    def type(self) -> "SourceAccessConfigurationType":
        '''The type of authentication protocol or the VPC components for your event source.

        For example: "SASL_SCRAM_512_AUTH".
        '''
        result = self._values.get("type")
        assert result is not None, "Required property 'type' is missing"
        return typing.cast("SourceAccessConfigurationType", result)

    @builtins.property
    def uri(self) -> builtins.str:
        '''The value for your chosen configuration in type.

        For example: "URI": "arn:aws:secretsmanager:us-east-1:01234567890:secret:MyBrokerSecretName".
        The exact string depends on the type.

        :see: SourceAccessConfigurationType
        '''
        result = self._values.get("uri")
        assert result is not None, "Required property 'uri' is missing"
        return typing.cast(builtins.str, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "SourceAccessConfiguration(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class SourceAccessConfigurationType(
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-lambda.SourceAccessConfigurationType",
):
    '''The type of authentication protocol or the VPC components for your event source's SourceAccessConfiguration.

    :see: https://docs.aws.amazon.com/lambda/latest/dg/API_SourceAccessConfiguration.html#SSS-Type-SourceAccessConfiguration-Type
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        import aws_cdk.aws_lambda as lambda_
        
        source_access_configuration_type = lambda_.SourceAccessConfigurationType.BASIC_AUTH
    '''

    @jsii.member(jsii_name="of")
    @builtins.classmethod
    def of(cls, name: builtins.str) -> "SourceAccessConfigurationType":
        '''A custom source access configuration property.

        :param name: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__fa0a9d31e521c273269e42d135b335b85a1ac5f628ae15fbcbcee73ed34795c3)
            check_type(argname="argument name", value=name, expected_type=type_hints["name"])
        return typing.cast("SourceAccessConfigurationType", jsii.sinvoke(cls, "of", [name]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="BASIC_AUTH")
    def BASIC_AUTH(cls) -> "SourceAccessConfigurationType":
        '''(MQ) The Secrets Manager secret that stores your broker credentials.'''
        return typing.cast("SourceAccessConfigurationType", jsii.sget(cls, "BASIC_AUTH"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CLIENT_CERTIFICATE_TLS_AUTH")
    def CLIENT_CERTIFICATE_TLS_AUTH(cls) -> "SourceAccessConfigurationType":
        '''The Secrets Manager ARN of your secret key containing the certificate chain (X.509 PEM), private key (PKCS#8 PEM), and private key password (optional) used for mutual TLS authentication of your MSK/Apache Kafka brokers.'''
        return typing.cast("SourceAccessConfigurationType", jsii.sget(cls, "CLIENT_CERTIFICATE_TLS_AUTH"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="SASL_SCRAM_256_AUTH")
    def SASL_SCRAM_256_AUTH(cls) -> "SourceAccessConfigurationType":
        '''The Secrets Manager ARN of your secret key used for SASL SCRAM-256 authentication of your Self-Managed Apache Kafka brokers.'''
        return typing.cast("SourceAccessConfigurationType", jsii.sget(cls, "SASL_SCRAM_256_AUTH"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="SASL_SCRAM_512_AUTH")
    def SASL_SCRAM_512_AUTH(cls) -> "SourceAccessConfigurationType":
        '''The Secrets Manager ARN of your secret key used for SASL SCRAM-512 authentication of your Self-Managed Apache Kafka brokers.'''
        return typing.cast("SourceAccessConfigurationType", jsii.sget(cls, "SASL_SCRAM_512_AUTH"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VPC_SECURITY_GROUP")
    def VPC_SECURITY_GROUP(cls) -> "SourceAccessConfigurationType":
        '''The VPC security group used to manage access to your Self-Managed Apache Kafka brokers.'''
        return typing.cast("SourceAccessConfigurationType", jsii.sget(cls, "VPC_SECURITY_GROUP"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VPC_SUBNET")
    def VPC_SUBNET(cls) -> "SourceAccessConfigurationType":
        '''The subnets associated with your VPC.

        Lambda connects to these subnets to fetch data from your Self-Managed Apache Kafka cluster.
        '''
        return typing.cast("SourceAccessConfigurationType", jsii.sget(cls, "VPC_SUBNET"))

    @builtins.property
    @jsii.member(jsii_name="type")
    def type(self) -> builtins.str:
        '''The key to use in ``SourceAccessConfigurationProperty.Type`` property in CloudFormation.

        :see: https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-eventsourcemapping-sourceaccessconfiguration.html#cfn-lambda-eventsourcemapping-sourceaccessconfiguration-type
        '''
        return typing.cast(builtins.str, jsii.get(self, "type"))


@jsii.enum(jsii_type="@aws-cdk/aws-lambda.StartingPosition")
class StartingPosition(enum.Enum):
    '''The position in the DynamoDB, Kinesis or MSK stream where AWS Lambda should start reading.

    :exampleMetadata: infused

    Example::

        from aws_cdk.aws_secretsmanager import Secret
        from aws_cdk.aws_lambda_event_sources import SelfManagedKafkaEventSource
        
        # The secret that allows access to your self hosted Kafka cluster
        # secret: Secret
        
        # my_function: lambda.Function
        
        
        # The list of Kafka brokers
        bootstrap_servers = ["kafka-broker:9092"]
        
        # The Kafka topic you want to subscribe to
        topic = "some-cool-topic"
        my_function.add_event_source(SelfManagedKafkaEventSource(
            bootstrap_servers=bootstrap_servers,
            topic=topic,
            secret=secret,
            batch_size=100,  # default
            starting_position=lambda_.StartingPosition.TRIM_HORIZON
        ))
    '''

    TRIM_HORIZON = "TRIM_HORIZON"
    '''Start reading at the last untrimmed record in the shard in the system, which is the oldest data record in the shard.'''
    LATEST = "LATEST"
    '''Start reading just after the most recent record in the shard, so that you always read the most recent data in the shard.'''


@jsii.enum(jsii_type="@aws-cdk/aws-lambda.Tracing")
class Tracing(enum.Enum):
    '''X-Ray Tracing Modes (https://docs.aws.amazon.com/lambda/latest/dg/API_TracingConfig.html).

    :exampleMetadata: infused

    Example::

        fn = lambda_.Function(self, "MyFunction",
            runtime=lambda_.Runtime.NODEJS_16_X,
            handler="index.handler",
            code=lambda_.Code.from_inline("exports.handler = function(event, ctx, cb) { return cb(null, \"hi\"); }"),
            tracing=lambda_.Tracing.ACTIVE
        )
    '''

    ACTIVE = "ACTIVE"
    '''Lambda will respect any tracing header it receives from an upstream service.

    If no tracing header is received, Lambda will call X-Ray for a tracing decision.
    '''
    PASS_THROUGH = "PASS_THROUGH"
    '''Lambda will only trace the request from an upstream service if it contains a tracing header with "sampled=1".'''
    DISABLED = "DISABLED"
    '''Lambda will not trace any request.'''


@jsii.enum(jsii_type="@aws-cdk/aws-lambda.UntrustedArtifactOnDeployment")
class UntrustedArtifactOnDeployment(enum.Enum):
    '''Code signing configuration policy for deployment validation failure.'''

    ENFORCE = "ENFORCE"
    '''Lambda blocks the deployment request if signature validation checks fail.'''
    WARN = "WARN"
    '''Lambda allows the deployment of the code package, but issues a warning.

    Lambda issues a new Amazon CloudWatch metric, called a signature validation error and also stores the warning in CloudTrail.
    '''


@jsii.data_type(
    jsii_type="@aws-cdk/aws-lambda.UtilizationScalingOptions",
    jsii_struct_bases=[
        _aws_cdk_aws_applicationautoscaling_a31e8c21.BaseTargetTrackingProps
    ],
    name_mapping={
        "disable_scale_in": "disableScaleIn",
        "policy_name": "policyName",
        "scale_in_cooldown": "scaleInCooldown",
        "scale_out_cooldown": "scaleOutCooldown",
        "utilization_target": "utilizationTarget",
    },
)
class UtilizationScalingOptions(
    _aws_cdk_aws_applicationautoscaling_a31e8c21.BaseTargetTrackingProps,
):
    def __init__(
        self,
        *,
        disable_scale_in: typing.Optional[builtins.bool] = None,
        policy_name: typing.Optional[builtins.str] = None,
        scale_in_cooldown: typing.Optional[_aws_cdk_core_f4b25747.Duration] = None,
        scale_out_cooldown: typing.Optional[_aws_cdk_core_f4b25747.Duration] = None,
        utilization_target: jsii.Number,
    ) -> None:
        '''Options for enabling Lambda utilization tracking.

        :param disable_scale_in: Indicates whether scale in by the target tracking policy is disabled. If the value is true, scale in is disabled and the target tracking policy won't remove capacity from the scalable resource. Otherwise, scale in is enabled and the target tracking policy can remove capacity from the scalable resource. Default: false
        :param policy_name: A name for the scaling policy. Default: - Automatically generated name.
        :param scale_in_cooldown: Period after a scale in activity completes before another scale in activity can start. Default: Duration.seconds(300) for the following scalable targets: ECS services, Spot Fleet requests, EMR clusters, AppStream 2.0 fleets, Aurora DB clusters, Amazon SageMaker endpoint variants, Custom resources. For all other scalable targets, the default value is Duration.seconds(0): DynamoDB tables, DynamoDB global secondary indexes, Amazon Comprehend document classification endpoints, Lambda provisioned concurrency
        :param scale_out_cooldown: Period after a scale out activity completes before another scale out activity can start. Default: Duration.seconds(300) for the following scalable targets: ECS services, Spot Fleet requests, EMR clusters, AppStream 2.0 fleets, Aurora DB clusters, Amazon SageMaker endpoint variants, Custom resources. For all other scalable targets, the default value is Duration.seconds(0): DynamoDB tables, DynamoDB global secondary indexes, Amazon Comprehend document classification endpoints, Lambda provisioned concurrency
        :param utilization_target: Utilization target for the attribute. For example, .5 indicates that 50 percent of allocated provisioned concurrency is in use.

        :exampleMetadata: infused

        Example::

            import aws_cdk.aws_autoscaling as autoscaling
            
            # fn: lambda.Function
            
            alias = fn.add_alias("prod")
            
            # Create AutoScaling target
            as = alias.add_auto_scaling(max_capacity=50)
            
            # Configure Target Tracking
            as.scale_on_utilization(
                utilization_target=0.5
            )
            
            # Configure Scheduled Scaling
            as.scale_on_schedule("ScaleUpInTheMorning",
                schedule=autoscaling.Schedule.cron(hour="8", minute="0"),
                min_capacity=20
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9a3b6c0760a517cb88ca2dd72b8c8b217ac6a2882142dc96cd22981f2b569e67)
            check_type(argname="argument disable_scale_in", value=disable_scale_in, expected_type=type_hints["disable_scale_in"])
            check_type(argname="argument policy_name", value=policy_name, expected_type=type_hints["policy_name"])
            check_type(argname="argument scale_in_cooldown", value=scale_in_cooldown, expected_type=type_hints["scale_in_cooldown"])
            check_type(argname="argument scale_out_cooldown", value=scale_out_cooldown, expected_type=type_hints["scale_out_cooldown"])
            check_type(argname="argument utilization_target", value=utilization_target, expected_type=type_hints["utilization_target"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "utilization_target": utilization_target,
        }
        if disable_scale_in is not None:
            self._values["disable_scale_in"] = disable_scale_in
        if policy_name is not None:
            self._values["policy_name"] = policy_name
        if scale_in_cooldown is not None:
            self._values["scale_in_cooldown"] = scale_in_cooldown
        if scale_out_cooldown is not None:
            self._values["scale_out_cooldown"] = scale_out_cooldown

    @builtins.property
    def disable_scale_in(self) -> typing.Optional[builtins.bool]:
        '''Indicates whether scale in by the target tracking policy is disabled.

        If the value is true, scale in is disabled and the target tracking policy
        won't remove capacity from the scalable resource. Otherwise, scale in is
        enabled and the target tracking policy can remove capacity from the
        scalable resource.

        :default: false
        '''
        result = self._values.get("disable_scale_in")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def policy_name(self) -> typing.Optional[builtins.str]:
        '''A name for the scaling policy.

        :default: - Automatically generated name.
        '''
        result = self._values.get("policy_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def scale_in_cooldown(self) -> typing.Optional[_aws_cdk_core_f4b25747.Duration]:
        '''Period after a scale in activity completes before another scale in activity can start.

        :default:

        Duration.seconds(300) for the following scalable targets: ECS services,
        Spot Fleet requests, EMR clusters, AppStream 2.0 fleets, Aurora DB clusters,
        Amazon SageMaker endpoint variants, Custom resources. For all other scalable
        targets, the default value is Duration.seconds(0): DynamoDB tables, DynamoDB
        global secondary indexes, Amazon Comprehend document classification endpoints,
        Lambda provisioned concurrency
        '''
        result = self._values.get("scale_in_cooldown")
        return typing.cast(typing.Optional[_aws_cdk_core_f4b25747.Duration], result)

    @builtins.property
    def scale_out_cooldown(self) -> typing.Optional[_aws_cdk_core_f4b25747.Duration]:
        '''Period after a scale out activity completes before another scale out activity can start.

        :default:

        Duration.seconds(300) for the following scalable targets: ECS services,
        Spot Fleet requests, EMR clusters, AppStream 2.0 fleets, Aurora DB clusters,
        Amazon SageMaker endpoint variants, Custom resources. For all other scalable
        targets, the default value is Duration.seconds(0): DynamoDB tables, DynamoDB
        global secondary indexes, Amazon Comprehend document classification endpoints,
        Lambda provisioned concurrency
        '''
        result = self._values.get("scale_out_cooldown")
        return typing.cast(typing.Optional[_aws_cdk_core_f4b25747.Duration], result)

    @builtins.property
    def utilization_target(self) -> jsii.Number:
        '''Utilization target for the attribute.

        For example, .5 indicates that 50 percent of allocated provisioned concurrency is in use.
        '''
        result = self._values.get("utilization_target")
        assert result is not None, "Required property 'utilization_target' is missing"
        return typing.cast(jsii.Number, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "UtilizationScalingOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-lambda.VersionAttributes",
    jsii_struct_bases=[],
    name_mapping={"lambda_": "lambda", "version": "version"},
)
class VersionAttributes:
    def __init__(self, *, lambda_: IFunction, version: builtins.str) -> None:
        '''
        :param lambda_: The lambda function.
        :param version: The version.

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_lambda as lambda_
            
            # function_: lambda.Function
            
            version_attributes = lambda.VersionAttributes(
                lambda_=function_,
                version="version"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__bb7aac823d5fded5034439653ff7505512524368affb03897c7497e885f4a5ae)
            check_type(argname="argument lambda_", value=lambda_, expected_type=type_hints["lambda_"])
            check_type(argname="argument version", value=version, expected_type=type_hints["version"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "lambda_": lambda_,
            "version": version,
        }

    @builtins.property
    def lambda_(self) -> IFunction:
        '''The lambda function.'''
        result = self._values.get("lambda_")
        assert result is not None, "Required property 'lambda_' is missing"
        return typing.cast(IFunction, result)

    @builtins.property
    def version(self) -> builtins.str:
        '''The version.'''
        result = self._values.get("version")
        assert result is not None, "Required property 'version' is missing"
        return typing.cast(builtins.str, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "VersionAttributes(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-lambda.VersionOptions",
    jsii_struct_bases=[EventInvokeConfigOptions],
    name_mapping={
        "max_event_age": "maxEventAge",
        "on_failure": "onFailure",
        "on_success": "onSuccess",
        "retry_attempts": "retryAttempts",
        "code_sha256": "codeSha256",
        "description": "description",
        "provisioned_concurrent_executions": "provisionedConcurrentExecutions",
        "removal_policy": "removalPolicy",
    },
)
class VersionOptions(EventInvokeConfigOptions):
    def __init__(
        self,
        *,
        max_event_age: typing.Optional[_aws_cdk_core_f4b25747.Duration] = None,
        on_failure: typing.Optional[IDestination] = None,
        on_success: typing.Optional[IDestination] = None,
        retry_attempts: typing.Optional[jsii.Number] = None,
        code_sha256: typing.Optional[builtins.str] = None,
        description: typing.Optional[builtins.str] = None,
        provisioned_concurrent_executions: typing.Optional[jsii.Number] = None,
        removal_policy: typing.Optional[_aws_cdk_core_f4b25747.RemovalPolicy] = None,
    ) -> None:
        '''Options for ``lambda.Version``.

        :param max_event_age: The maximum age of a request that Lambda sends to a function for processing. Minimum: 60 seconds Maximum: 6 hours Default: Duration.hours(6)
        :param on_failure: The destination for failed invocations. Default: - no destination
        :param on_success: The destination for successful invocations. Default: - no destination
        :param retry_attempts: The maximum number of times to retry when the function returns an error. Minimum: 0 Maximum: 2 Default: 2
        :param code_sha256: SHA256 of the version of the Lambda source code. Specify to validate that you're deploying the right version. Default: No validation is performed
        :param description: Description of the version. Default: Description of the Lambda
        :param provisioned_concurrent_executions: Specifies a provisioned concurrency configuration for a function's version. Default: No provisioned concurrency
        :param removal_policy: Whether to retain old versions of this function when a new version is created. Default: RemovalPolicy.DESTROY

        :exampleMetadata: infused

        Example::

            fn = lambda_.Function(self, "MyFunction",
                current_version_options=lambda.VersionOptions(
                    removal_policy=RemovalPolicy.RETAIN,  # retain old versions
                    retry_attempts=1
                ),
                runtime=lambda_.Runtime.NODEJS_16_X,
                handler="index.handler",
                code=lambda_.Code.from_asset(path.join(__dirname, "lambda-handler"))
            )
            
            fn.add_alias("live")
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a38698ce8c3a014fd38037ae11d49a39c5c6409f2c48d6fbcd8a8d282c40f342)
            check_type(argname="argument max_event_age", value=max_event_age, expected_type=type_hints["max_event_age"])
            check_type(argname="argument on_failure", value=on_failure, expected_type=type_hints["on_failure"])
            check_type(argname="argument on_success", value=on_success, expected_type=type_hints["on_success"])
            check_type(argname="argument retry_attempts", value=retry_attempts, expected_type=type_hints["retry_attempts"])
            check_type(argname="argument code_sha256", value=code_sha256, expected_type=type_hints["code_sha256"])
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument provisioned_concurrent_executions", value=provisioned_concurrent_executions, expected_type=type_hints["provisioned_concurrent_executions"])
            check_type(argname="argument removal_policy", value=removal_policy, expected_type=type_hints["removal_policy"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if max_event_age is not None:
            self._values["max_event_age"] = max_event_age
        if on_failure is not None:
            self._values["on_failure"] = on_failure
        if on_success is not None:
            self._values["on_success"] = on_success
        if retry_attempts is not None:
            self._values["retry_attempts"] = retry_attempts
        if code_sha256 is not None:
            self._values["code_sha256"] = code_sha256
        if description is not None:
            self._values["description"] = description
        if provisioned_concurrent_executions is not None:
            self._values["provisioned_concurrent_executions"] = provisioned_concurrent_executions
        if removal_policy is not None:
            self._values["removal_policy"] = removal_policy

    @builtins.property
    def max_event_age(self) -> typing.Optional[_aws_cdk_core_f4b25747.Duration]:
        '''The maximum age of a request that Lambda sends to a function for processing.

        Minimum: 60 seconds
        Maximum: 6 hours

        :default: Duration.hours(6)
        '''
        result = self._values.get("max_event_age")
        return typing.cast(typing.Optional[_aws_cdk_core_f4b25747.Duration], result)

    @builtins.property
    def on_failure(self) -> typing.Optional[IDestination]:
        '''The destination for failed invocations.

        :default: - no destination
        '''
        result = self._values.get("on_failure")
        return typing.cast(typing.Optional[IDestination], result)

    @builtins.property
    def on_success(self) -> typing.Optional[IDestination]:
        '''The destination for successful invocations.

        :default: - no destination
        '''
        result = self._values.get("on_success")
        return typing.cast(typing.Optional[IDestination], result)

    @builtins.property
    def retry_attempts(self) -> typing.Optional[jsii.Number]:
        '''The maximum number of times to retry when the function returns an error.

        Minimum: 0
        Maximum: 2

        :default: 2
        '''
        result = self._values.get("retry_attempts")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def code_sha256(self) -> typing.Optional[builtins.str]:
        '''SHA256 of the version of the Lambda source code.

        Specify to validate that you're deploying the right version.

        :default: No validation is performed
        '''
        result = self._values.get("code_sha256")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''Description of the version.

        :default: Description of the Lambda
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def provisioned_concurrent_executions(self) -> typing.Optional[jsii.Number]:
        '''Specifies a provisioned concurrency configuration for a function's version.

        :default: No provisioned concurrency
        '''
        result = self._values.get("provisioned_concurrent_executions")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def removal_policy(self) -> typing.Optional[_aws_cdk_core_f4b25747.RemovalPolicy]:
        '''Whether to retain old versions of this function when a new version is created.

        :default: RemovalPolicy.DESTROY
        '''
        result = self._values.get("removal_policy")
        return typing.cast(typing.Optional[_aws_cdk_core_f4b25747.RemovalPolicy], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "VersionOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-lambda.VersionProps",
    jsii_struct_bases=[VersionOptions],
    name_mapping={
        "max_event_age": "maxEventAge",
        "on_failure": "onFailure",
        "on_success": "onSuccess",
        "retry_attempts": "retryAttempts",
        "code_sha256": "codeSha256",
        "description": "description",
        "provisioned_concurrent_executions": "provisionedConcurrentExecutions",
        "removal_policy": "removalPolicy",
        "lambda_": "lambda",
    },
)
class VersionProps(VersionOptions):
    def __init__(
        self,
        *,
        max_event_age: typing.Optional[_aws_cdk_core_f4b25747.Duration] = None,
        on_failure: typing.Optional[IDestination] = None,
        on_success: typing.Optional[IDestination] = None,
        retry_attempts: typing.Optional[jsii.Number] = None,
        code_sha256: typing.Optional[builtins.str] = None,
        description: typing.Optional[builtins.str] = None,
        provisioned_concurrent_executions: typing.Optional[jsii.Number] = None,
        removal_policy: typing.Optional[_aws_cdk_core_f4b25747.RemovalPolicy] = None,
        lambda_: IFunction,
    ) -> None:
        '''Properties for a new Lambda version.

        :param max_event_age: The maximum age of a request that Lambda sends to a function for processing. Minimum: 60 seconds Maximum: 6 hours Default: Duration.hours(6)
        :param on_failure: The destination for failed invocations. Default: - no destination
        :param on_success: The destination for successful invocations. Default: - no destination
        :param retry_attempts: The maximum number of times to retry when the function returns an error. Minimum: 0 Maximum: 2 Default: 2
        :param code_sha256: SHA256 of the version of the Lambda source code. Specify to validate that you're deploying the right version. Default: No validation is performed
        :param description: Description of the version. Default: Description of the Lambda
        :param provisioned_concurrent_executions: Specifies a provisioned concurrency configuration for a function's version. Default: No provisioned concurrency
        :param removal_policy: Whether to retain old versions of this function when a new version is created. Default: RemovalPolicy.DESTROY
        :param lambda_: Function to get the value of.

        :exampleMetadata: infused

        Example::

            # fn: lambda.Function
            
            version = lambda_.Version(self, "MyVersion",
                lambda_=fn
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a389e34f3c0b6f647f4fba378e3d247385144b8df8e24856b2a804bd453dd6d0)
            check_type(argname="argument max_event_age", value=max_event_age, expected_type=type_hints["max_event_age"])
            check_type(argname="argument on_failure", value=on_failure, expected_type=type_hints["on_failure"])
            check_type(argname="argument on_success", value=on_success, expected_type=type_hints["on_success"])
            check_type(argname="argument retry_attempts", value=retry_attempts, expected_type=type_hints["retry_attempts"])
            check_type(argname="argument code_sha256", value=code_sha256, expected_type=type_hints["code_sha256"])
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument provisioned_concurrent_executions", value=provisioned_concurrent_executions, expected_type=type_hints["provisioned_concurrent_executions"])
            check_type(argname="argument removal_policy", value=removal_policy, expected_type=type_hints["removal_policy"])
            check_type(argname="argument lambda_", value=lambda_, expected_type=type_hints["lambda_"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "lambda_": lambda_,
        }
        if max_event_age is not None:
            self._values["max_event_age"] = max_event_age
        if on_failure is not None:
            self._values["on_failure"] = on_failure
        if on_success is not None:
            self._values["on_success"] = on_success
        if retry_attempts is not None:
            self._values["retry_attempts"] = retry_attempts
        if code_sha256 is not None:
            self._values["code_sha256"] = code_sha256
        if description is not None:
            self._values["description"] = description
        if provisioned_concurrent_executions is not None:
            self._values["provisioned_concurrent_executions"] = provisioned_concurrent_executions
        if removal_policy is not None:
            self._values["removal_policy"] = removal_policy

    @builtins.property
    def max_event_age(self) -> typing.Optional[_aws_cdk_core_f4b25747.Duration]:
        '''The maximum age of a request that Lambda sends to a function for processing.

        Minimum: 60 seconds
        Maximum: 6 hours

        :default: Duration.hours(6)
        '''
        result = self._values.get("max_event_age")
        return typing.cast(typing.Optional[_aws_cdk_core_f4b25747.Duration], result)

    @builtins.property
    def on_failure(self) -> typing.Optional[IDestination]:
        '''The destination for failed invocations.

        :default: - no destination
        '''
        result = self._values.get("on_failure")
        return typing.cast(typing.Optional[IDestination], result)

    @builtins.property
    def on_success(self) -> typing.Optional[IDestination]:
        '''The destination for successful invocations.

        :default: - no destination
        '''
        result = self._values.get("on_success")
        return typing.cast(typing.Optional[IDestination], result)

    @builtins.property
    def retry_attempts(self) -> typing.Optional[jsii.Number]:
        '''The maximum number of times to retry when the function returns an error.

        Minimum: 0
        Maximum: 2

        :default: 2
        '''
        result = self._values.get("retry_attempts")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def code_sha256(self) -> typing.Optional[builtins.str]:
        '''SHA256 of the version of the Lambda source code.

        Specify to validate that you're deploying the right version.

        :default: No validation is performed
        '''
        result = self._values.get("code_sha256")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''Description of the version.

        :default: Description of the Lambda
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def provisioned_concurrent_executions(self) -> typing.Optional[jsii.Number]:
        '''Specifies a provisioned concurrency configuration for a function's version.

        :default: No provisioned concurrency
        '''
        result = self._values.get("provisioned_concurrent_executions")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def removal_policy(self) -> typing.Optional[_aws_cdk_core_f4b25747.RemovalPolicy]:
        '''Whether to retain old versions of this function when a new version is created.

        :default: RemovalPolicy.DESTROY
        '''
        result = self._values.get("removal_policy")
        return typing.cast(typing.Optional[_aws_cdk_core_f4b25747.RemovalPolicy], result)

    @builtins.property
    def lambda_(self) -> IFunction:
        '''Function to get the value of.'''
        result = self._values.get("lambda_")
        assert result is not None, "Required property 'lambda_' is missing"
        return typing.cast(IFunction, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "VersionProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-lambda.VersionWeight",
    jsii_struct_bases=[],
    name_mapping={"version": "version", "weight": "weight"},
)
class VersionWeight:
    def __init__(self, *, version: IVersion, weight: jsii.Number) -> None:
        '''A version/weight pair for routing traffic to Lambda functions.

        :param version: The version to route traffic to.
        :param weight: How much weight to assign to this version (0..1).

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_lambda as lambda_
            
            # version: lambda.Version
            
            version_weight = lambda.VersionWeight(
                version=version,
                weight=123
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8adada654af5165ab46b71afdfc05338045ec93443a7245f21951b73cf7ff9fd)
            check_type(argname="argument version", value=version, expected_type=type_hints["version"])
            check_type(argname="argument weight", value=weight, expected_type=type_hints["weight"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "version": version,
            "weight": weight,
        }

    @builtins.property
    def version(self) -> IVersion:
        '''The version to route traffic to.'''
        result = self._values.get("version")
        assert result is not None, "Required property 'version' is missing"
        return typing.cast(IVersion, result)

    @builtins.property
    def weight(self) -> jsii.Number:
        '''How much weight to assign to this version (0..1).'''
        result = self._values.get("weight")
        assert result is not None, "Required property 'weight' is missing"
        return typing.cast(jsii.Number, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "VersionWeight(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-lambda.AliasOptions",
    jsii_struct_bases=[EventInvokeConfigOptions],
    name_mapping={
        "max_event_age": "maxEventAge",
        "on_failure": "onFailure",
        "on_success": "onSuccess",
        "retry_attempts": "retryAttempts",
        "additional_versions": "additionalVersions",
        "description": "description",
        "provisioned_concurrent_executions": "provisionedConcurrentExecutions",
    },
)
class AliasOptions(EventInvokeConfigOptions):
    def __init__(
        self,
        *,
        max_event_age: typing.Optional[_aws_cdk_core_f4b25747.Duration] = None,
        on_failure: typing.Optional[IDestination] = None,
        on_success: typing.Optional[IDestination] = None,
        retry_attempts: typing.Optional[jsii.Number] = None,
        additional_versions: typing.Optional[typing.Sequence[typing.Union[VersionWeight, typing.Dict[builtins.str, typing.Any]]]] = None,
        description: typing.Optional[builtins.str] = None,
        provisioned_concurrent_executions: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''Options for ``lambda.Alias``.

        :param max_event_age: The maximum age of a request that Lambda sends to a function for processing. Minimum: 60 seconds Maximum: 6 hours Default: Duration.hours(6)
        :param on_failure: The destination for failed invocations. Default: - no destination
        :param on_success: The destination for successful invocations. Default: - no destination
        :param retry_attempts: The maximum number of times to retry when the function returns an error. Minimum: 0 Maximum: 2 Default: 2
        :param additional_versions: Additional versions with individual weights this alias points to. Individual additional version weights specified here should add up to (less than) one. All remaining weight is routed to the default version. For example, the config is Example:: version: "1" additionalVersions: [{ version: "2", weight: 0.05 }] Then 5% of traffic will be routed to function version 2, while the remaining 95% of traffic will be routed to function version 1. Default: No additional versions
        :param description: Description for the alias. Default: No description
        :param provisioned_concurrent_executions: Specifies a provisioned concurrency configuration for a function's alias. Default: No provisioned concurrency

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_lambda as lambda_
            import aws_cdk.core as cdk
            
            # destination: lambda.IDestination
            # version: lambda.Version
            
            alias_options = lambda.AliasOptions(
                additional_versions=[lambda.VersionWeight(
                    version=version,
                    weight=123
                )],
                description="description",
                max_event_age=cdk.Duration.minutes(30),
                on_failure=destination,
                on_success=destination,
                provisioned_concurrent_executions=123,
                retry_attempts=123
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4fffe1ebdf8d9d5b78d6d540d6d6d28b830f96f8555e2c46c4e156a91f02ddc2)
            check_type(argname="argument max_event_age", value=max_event_age, expected_type=type_hints["max_event_age"])
            check_type(argname="argument on_failure", value=on_failure, expected_type=type_hints["on_failure"])
            check_type(argname="argument on_success", value=on_success, expected_type=type_hints["on_success"])
            check_type(argname="argument retry_attempts", value=retry_attempts, expected_type=type_hints["retry_attempts"])
            check_type(argname="argument additional_versions", value=additional_versions, expected_type=type_hints["additional_versions"])
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument provisioned_concurrent_executions", value=provisioned_concurrent_executions, expected_type=type_hints["provisioned_concurrent_executions"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if max_event_age is not None:
            self._values["max_event_age"] = max_event_age
        if on_failure is not None:
            self._values["on_failure"] = on_failure
        if on_success is not None:
            self._values["on_success"] = on_success
        if retry_attempts is not None:
            self._values["retry_attempts"] = retry_attempts
        if additional_versions is not None:
            self._values["additional_versions"] = additional_versions
        if description is not None:
            self._values["description"] = description
        if provisioned_concurrent_executions is not None:
            self._values["provisioned_concurrent_executions"] = provisioned_concurrent_executions

    @builtins.property
    def max_event_age(self) -> typing.Optional[_aws_cdk_core_f4b25747.Duration]:
        '''The maximum age of a request that Lambda sends to a function for processing.

        Minimum: 60 seconds
        Maximum: 6 hours

        :default: Duration.hours(6)
        '''
        result = self._values.get("max_event_age")
        return typing.cast(typing.Optional[_aws_cdk_core_f4b25747.Duration], result)

    @builtins.property
    def on_failure(self) -> typing.Optional[IDestination]:
        '''The destination for failed invocations.

        :default: - no destination
        '''
        result = self._values.get("on_failure")
        return typing.cast(typing.Optional[IDestination], result)

    @builtins.property
    def on_success(self) -> typing.Optional[IDestination]:
        '''The destination for successful invocations.

        :default: - no destination
        '''
        result = self._values.get("on_success")
        return typing.cast(typing.Optional[IDestination], result)

    @builtins.property
    def retry_attempts(self) -> typing.Optional[jsii.Number]:
        '''The maximum number of times to retry when the function returns an error.

        Minimum: 0
        Maximum: 2

        :default: 2
        '''
        result = self._values.get("retry_attempts")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def additional_versions(self) -> typing.Optional[typing.List[VersionWeight]]:
        '''Additional versions with individual weights this alias points to.

        Individual additional version weights specified here should add up to
        (less than) one. All remaining weight is routed to the default
        version.

        For example, the config is Example::

           version: "1"
           additionalVersions: [{ version: "2", weight: 0.05 }]

        Then 5% of traffic will be routed to function version 2, while
        the remaining 95% of traffic will be routed to function version 1.

        :default: No additional versions
        '''
        result = self._values.get("additional_versions")
        return typing.cast(typing.Optional[typing.List[VersionWeight]], result)

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''Description for the alias.

        :default: No description
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def provisioned_concurrent_executions(self) -> typing.Optional[jsii.Number]:
        '''Specifies a provisioned concurrency configuration for a function's alias.

        :default: No provisioned concurrency
        '''
        result = self._values.get("provisioned_concurrent_executions")
        return typing.cast(typing.Optional[jsii.Number], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AliasOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-lambda.AliasProps",
    jsii_struct_bases=[AliasOptions],
    name_mapping={
        "max_event_age": "maxEventAge",
        "on_failure": "onFailure",
        "on_success": "onSuccess",
        "retry_attempts": "retryAttempts",
        "additional_versions": "additionalVersions",
        "description": "description",
        "provisioned_concurrent_executions": "provisionedConcurrentExecutions",
        "alias_name": "aliasName",
        "version": "version",
    },
)
class AliasProps(AliasOptions):
    def __init__(
        self,
        *,
        max_event_age: typing.Optional[_aws_cdk_core_f4b25747.Duration] = None,
        on_failure: typing.Optional[IDestination] = None,
        on_success: typing.Optional[IDestination] = None,
        retry_attempts: typing.Optional[jsii.Number] = None,
        additional_versions: typing.Optional[typing.Sequence[typing.Union[VersionWeight, typing.Dict[builtins.str, typing.Any]]]] = None,
        description: typing.Optional[builtins.str] = None,
        provisioned_concurrent_executions: typing.Optional[jsii.Number] = None,
        alias_name: builtins.str,
        version: IVersion,
    ) -> None:
        '''Properties for a new Lambda alias.

        :param max_event_age: The maximum age of a request that Lambda sends to a function for processing. Minimum: 60 seconds Maximum: 6 hours Default: Duration.hours(6)
        :param on_failure: The destination for failed invocations. Default: - no destination
        :param on_success: The destination for successful invocations. Default: - no destination
        :param retry_attempts: The maximum number of times to retry when the function returns an error. Minimum: 0 Maximum: 2 Default: 2
        :param additional_versions: Additional versions with individual weights this alias points to. Individual additional version weights specified here should add up to (less than) one. All remaining weight is routed to the default version. For example, the config is Example:: version: "1" additionalVersions: [{ version: "2", weight: 0.05 }] Then 5% of traffic will be routed to function version 2, while the remaining 95% of traffic will be routed to function version 1. Default: No additional versions
        :param description: Description for the alias. Default: No description
        :param provisioned_concurrent_executions: Specifies a provisioned concurrency configuration for a function's alias. Default: No provisioned concurrency
        :param alias_name: Name of this alias.
        :param version: Function version this alias refers to. Use lambda.currentVersion to reference a version with your latest changes.

        :exampleMetadata: infused

        Example::

            lambda_code = lambda_.Code.from_cfn_parameters()
            func = lambda_.Function(self, "Lambda",
                code=lambda_code,
                handler="index.handler",
                runtime=lambda_.Runtime.NODEJS_14_X
            )
            # used to make sure each CDK synthesis produces a different Version
            version = func.current_version
            alias = lambda_.Alias(self, "LambdaAlias",
                alias_name="Prod",
                version=version
            )
            
            codedeploy.LambdaDeploymentGroup(self, "DeploymentGroup",
                alias=alias,
                deployment_config=codedeploy.LambdaDeploymentConfig.LINEAR_10PERCENT_EVERY_1MINUTE
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b5dfe75ab8943215904aac269a672001c00a33a53baedbff99fd3e4672c728b1)
            check_type(argname="argument max_event_age", value=max_event_age, expected_type=type_hints["max_event_age"])
            check_type(argname="argument on_failure", value=on_failure, expected_type=type_hints["on_failure"])
            check_type(argname="argument on_success", value=on_success, expected_type=type_hints["on_success"])
            check_type(argname="argument retry_attempts", value=retry_attempts, expected_type=type_hints["retry_attempts"])
            check_type(argname="argument additional_versions", value=additional_versions, expected_type=type_hints["additional_versions"])
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument provisioned_concurrent_executions", value=provisioned_concurrent_executions, expected_type=type_hints["provisioned_concurrent_executions"])
            check_type(argname="argument alias_name", value=alias_name, expected_type=type_hints["alias_name"])
            check_type(argname="argument version", value=version, expected_type=type_hints["version"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "alias_name": alias_name,
            "version": version,
        }
        if max_event_age is not None:
            self._values["max_event_age"] = max_event_age
        if on_failure is not None:
            self._values["on_failure"] = on_failure
        if on_success is not None:
            self._values["on_success"] = on_success
        if retry_attempts is not None:
            self._values["retry_attempts"] = retry_attempts
        if additional_versions is not None:
            self._values["additional_versions"] = additional_versions
        if description is not None:
            self._values["description"] = description
        if provisioned_concurrent_executions is not None:
            self._values["provisioned_concurrent_executions"] = provisioned_concurrent_executions

    @builtins.property
    def max_event_age(self) -> typing.Optional[_aws_cdk_core_f4b25747.Duration]:
        '''The maximum age of a request that Lambda sends to a function for processing.

        Minimum: 60 seconds
        Maximum: 6 hours

        :default: Duration.hours(6)
        '''
        result = self._values.get("max_event_age")
        return typing.cast(typing.Optional[_aws_cdk_core_f4b25747.Duration], result)

    @builtins.property
    def on_failure(self) -> typing.Optional[IDestination]:
        '''The destination for failed invocations.

        :default: - no destination
        '''
        result = self._values.get("on_failure")
        return typing.cast(typing.Optional[IDestination], result)

    @builtins.property
    def on_success(self) -> typing.Optional[IDestination]:
        '''The destination for successful invocations.

        :default: - no destination
        '''
        result = self._values.get("on_success")
        return typing.cast(typing.Optional[IDestination], result)

    @builtins.property
    def retry_attempts(self) -> typing.Optional[jsii.Number]:
        '''The maximum number of times to retry when the function returns an error.

        Minimum: 0
        Maximum: 2

        :default: 2
        '''
        result = self._values.get("retry_attempts")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def additional_versions(self) -> typing.Optional[typing.List[VersionWeight]]:
        '''Additional versions with individual weights this alias points to.

        Individual additional version weights specified here should add up to
        (less than) one. All remaining weight is routed to the default
        version.

        For example, the config is Example::

           version: "1"
           additionalVersions: [{ version: "2", weight: 0.05 }]

        Then 5% of traffic will be routed to function version 2, while
        the remaining 95% of traffic will be routed to function version 1.

        :default: No additional versions
        '''
        result = self._values.get("additional_versions")
        return typing.cast(typing.Optional[typing.List[VersionWeight]], result)

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''Description for the alias.

        :default: No description
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def provisioned_concurrent_executions(self) -> typing.Optional[jsii.Number]:
        '''Specifies a provisioned concurrency configuration for a function's alias.

        :default: No provisioned concurrency
        '''
        result = self._values.get("provisioned_concurrent_executions")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def alias_name(self) -> builtins.str:
        '''Name of this alias.'''
        result = self._values.get("alias_name")
        assert result is not None, "Required property 'alias_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def version(self) -> IVersion:
        '''Function version this alias refers to.

        Use lambda.currentVersion to reference a version with your latest changes.
        '''
        result = self._values.get("version")
        assert result is not None, "Required property 'version' is mis