"use strict";
const assert_1 = require("@aws-cdk/assert");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
const testMetric = new lib_1.Metric({
    namespace: 'CDK/Test',
    metricName: 'Metric',
});
class TestAlarmAction {
    constructor(arn) {
        this.arn = arn;
    }
    bind(_scope, _alarm) {
        return { alarmActionArn: this.arn };
    }
}
module.exports = {
    'can make simple alarm'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.Alarm(stack, 'Alarm', {
            metric: testMetric,
            threshold: 1000,
            evaluationPeriods: 3,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::CloudWatch::Alarm', {
            ComparisonOperator: 'GreaterThanOrEqualToThreshold',
            EvaluationPeriods: 3,
            MetricName: 'Metric',
            Namespace: 'CDK/Test',
            Period: 300,
            Statistic: 'Average',
            Threshold: 1000,
        }));
        test.done();
    },
    'override metric period in Alarm'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.Alarm(stack, 'Alarm', {
            metric: testMetric,
            period: core_1.Duration.minutes(10),
            threshold: 1000,
            evaluationPeriods: 3,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::CloudWatch::Alarm', {
            ComparisonOperator: 'GreaterThanOrEqualToThreshold',
            EvaluationPeriods: 3,
            MetricName: 'Metric',
            Namespace: 'CDK/Test',
            Period: 600,
            Statistic: 'Average',
            Threshold: 1000,
        }));
        test.done();
    },
    'override statistic Alarm'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.Alarm(stack, 'Alarm', {
            metric: testMetric,
            statistic: 'max',
            threshold: 1000,
            evaluationPeriods: 3,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::CloudWatch::Alarm', {
            ComparisonOperator: 'GreaterThanOrEqualToThreshold',
            EvaluationPeriods: 3,
            MetricName: 'Metric',
            Namespace: 'CDK/Test',
            Period: 300,
            Statistic: 'Maximum',
            ExtendedStatistic: assert_1.ABSENT,
            Threshold: 1000,
        }));
        test.done();
    },
    'can use percentile in Alarm'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.Alarm(stack, 'Alarm', {
            metric: testMetric,
            statistic: 'P99',
            threshold: 1000,
            evaluationPeriods: 3,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::CloudWatch::Alarm', {
            ComparisonOperator: 'GreaterThanOrEqualToThreshold',
            EvaluationPeriods: 3,
            MetricName: 'Metric',
            Namespace: 'CDK/Test',
            Period: 300,
            Statistic: assert_1.ABSENT,
            ExtendedStatistic: 'p99',
            Threshold: 1000,
        }));
        test.done();
    },
    'can set DatapointsToAlarm'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.Alarm(stack, 'Alarm', {
            metric: testMetric,
            threshold: 1000,
            evaluationPeriods: 3,
            datapointsToAlarm: 2,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::CloudWatch::Alarm', {
            ComparisonOperator: 'GreaterThanOrEqualToThreshold',
            EvaluationPeriods: 3,
            DatapointsToAlarm: 2,
            MetricName: 'Metric',
            Namespace: 'CDK/Test',
            Period: 300,
            Statistic: 'Average',
            Threshold: 1000,
        }));
        test.done();
    },
    'can add actions to alarms'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const alarm = new lib_1.Alarm(stack, 'Alarm', {
            metric: testMetric,
            threshold: 1000,
            evaluationPeriods: 2,
        });
        alarm.addAlarmAction(new TestAlarmAction('A'));
        alarm.addInsufficientDataAction(new TestAlarmAction('B'));
        alarm.addOkAction(new TestAlarmAction('C'));
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::CloudWatch::Alarm', {
            AlarmActions: ['A'],
            InsufficientDataActions: ['B'],
            OKActions: ['C'],
        }));
        test.done();
    },
    'can make alarm directly from metric'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        testMetric.createAlarm(stack, 'Alarm', {
            threshold: 1000,
            evaluationPeriods: 2,
            statistic: 'min',
            period: core_1.Duration.seconds(10),
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::CloudWatch::Alarm', {
            ComparisonOperator: 'GreaterThanOrEqualToThreshold',
            EvaluationPeriods: 2,
            MetricName: 'Metric',
            Namespace: 'CDK/Test',
            Period: 10,
            Statistic: 'Minimum',
            Threshold: 1000,
        }));
        test.done();
    },
    'can use percentile string to make alarm'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        testMetric.createAlarm(stack, 'Alarm', {
            threshold: 1000,
            evaluationPeriods: 2,
            statistic: 'p99.9',
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::CloudWatch::Alarm', {
            ExtendedStatistic: 'p99.9',
        }));
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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