"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.KongDataPlane = void 0;
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
class KongDataPlane extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        props.nodegroup.role.addToPrincipalPolicy(new aws_cdk_lib_1.aws_iam.PolicyStatement({
            resources: ['*'],
            actions: [
                'secretsmanager:GetResourcePolicy',
                'secretsmanager:GetSecretValue',
                'secretsmanager:DescribeSecret',
                'secretsmanager:ListSecretVersionIds',
            ],
        }));
        const dp_external_secrets_helm = props.cluster.addHelmChart('DPExternalSecretsHandler', {
            repository: 'https://external-secrets.github.io/kubernetes-external-secrets/',
            chart: 'kubernetes-external-secrets',
            release: 'kubernetes-external-secrets',
            namespace: props.namespace,
            wait: true,
            values: {
                securityContext: {
                    fsGroup: 65534,
                },
                env: {
                    AWS_REGION: aws_cdk_lib_1.Stack.of(this).region,
                },
            },
        });
        const license_key_manifest = {
            apiVersion: 'kubernetes-client.io/v1',
            kind: 'ExternalSecret',
            metadata: {
                name: 'kong-license',
                namespace: props.namespace,
            },
            spec: {
                backendType: 'secretsManager',
                data: [
                    {
                        key: props.license_secret_name,
                        name: 'license',
                    },
                ],
            },
        };
        const kong_dp_secrets = props.cluster.addManifest('KongDpSecrets', license_key_manifest);
        kong_dp_secrets.node.addDependency(dp_external_secrets_helm);
        const kong_dp_helm = props.cluster.addHelmChart('KongDpHelm', {
            repository: 'https://charts.konghq.com',
            chart: 'kong',
            release: 'kong',
            namespace: props.namespace,
            timeout: aws_cdk_lib_1.Duration.minutes(15),
            wait: true,
            createNamespace: true,
            values: {
                resources: {
                    requests: {
                        cpu: '300m',
                        memory: '300Mi',
                    },
                    limits: {
                        cpu: '1200m',
                        memory: '800Mi',
                    },
                },
                autoscaling: {
                    enabled: true,
                    minReplicas: 1,
                    maxReplicas: 20,
                    metrics: [
                        {
                            type: 'Resource',
                            resource: {
                                name: 'cpu',
                                target: {
                                    type: 'Utilization',
                                    averageUtilization: 75,
                                },
                            },
                        },
                    ],
                },
                ingressController: {
                    enabled: false,
                },
                image: {
                    repository: 'kong/kong-gateway',
                    tag: '2.5.0.0-alpine',
                },
                env: {
                    database: 'off',
                    role: 'data_plane',
                    cluster_mtls: 'pki',
                    cluster_cert: '/etc/secrets/kong-dp-cluster-issuer-secret/tls.crt',
                    cluster_cert_key: '/etc/secrets/kong-dp-cluster-issuer-secret/tls.key',
                    cluster_ca_cert: '/etc/secrets/kong-dp-cluster-issuer-secret/ca.crt',
                    cluster_server_name: 'kong-cp.internal',
                    cluster_control_plane: `${props.cluster_dns}:8005`,
                    cluster_telemetry_endpoint: `${props.telemetry_dns}:8006`,
                },
                proxy: {
                    annotations: {
                        'service.beta.kubernetes.io/aws-load-balancer-type': 'nlb',
                        'service.beta.kubernetes.io/aws-load-balancer-additional-resource-tags': 'Type=proxy',
                        'external-dns.alpha.kubernetes.io/hostname': 'proxy.kong-dp.internal',
                    },
                },
                enterprise: {
                    enabled: true,
                    license_secret: 'kong-license',
                },
                manager: {
                    enabled: false,
                },
                portal: {
                    enabled: false,
                },
                portalapi: {
                    enabled: false,
                },
                secretVolumes: [
                    'kong-dp-cluster-issuer-secret',
                ],
            },
        });
        kong_dp_helm.node.addDependency(kong_dp_secrets);
    }
}
exports.KongDataPlane = KongDataPlane;
//# sourceMappingURL=data:application/json;base64,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