# Copyright (c) Microsoft Corporation. All rights reserved.
# Licensed under the MIT License. See License.txt in the project root for license information.

import os
from unittest import TestCase

from cdm.enums import CdmObjectType
from cdm.objectmodel import CdmEntityDefinition, CdmAttributeContext, CdmAttributeReference, CdmCorpusDefinition, \
    CdmArgumentDefinition


class AttributeContextUtil:
    """
    Multiple test classes in projections test the attribute context tree generated for various scenarios.
    This utility class helps generate the actual attribute context generated by the scenario, so that it can be compared with expected attribute context tree.
    This also handles the validation of the expected vs. actual attribute context.
    """
    def __init__(self):
        # This string is used to concatenate all the attribute contexts and traits of an entity into one string
        # so that we can compare it to the expected output txt file.
        self._bldr = ''

    def get_attribute_context_strings(self, resolved_entity: 'CdmEntityDefinition', attrib_context: 'CdmAttributeContext') -> str:
        """Function to get the attribute context string tree from a resolved entity"""

        # clear the string builder
        self._bldr = ''

        # get the corpus path for each attribute context in the tree
        self._get_content_declared_path(resolved_entity.attribute_context)

        # get the traits for all the attributes of a resolved entity
        self._get_traits(resolved_entity)

        return self._bldr

    def _get_content_declared_path(self, attrib_context: 'CdmAttributeContext') -> None:
        """Get the corpus path for each attribute context in the tree and build a string collection that we can
        compare with the expected attribute context corpus path collection."""
        if attrib_context and attrib_context.contents and len(attrib_context.contents) > 0:
            for i in range(len(attrib_context.contents)):
                content = attrib_context.contents[i]
                str = content.at_corpus_path
                self._bldr += str
                self._bldr += '\n'

                if not isinstance(attrib_context.contents[0], CdmAttributeReference):
                    self._get_content_declared_path(content)

    def _get_traits(self, resolved_entity: 'CdmEntityDefinition') -> None:
        """Get the traits for all the attributes of a resolved entity"""
        for attrib in resolved_entity.attributes:
            attrib_corpus_path = attrib.at_corpus_path
            self._bldr += attrib_corpus_path
            self._bldr += '\n'

            for trait in attrib.applied_traits:
                attrib_traits = trait.named_reference
                self._bldr += attrib_traits
                self._bldr += '\n'

                for args in trait.arguments:
                    self._get_argument_values_as_string(args)

    def _get_argument_values_as_string(self, args: 'CdmArgumentDefinition') -> None:
        param_name = args._resolved_parameter.name if args._resolved_parameter else None
        param_default_value = args._resolved_parameter.default_value if args._resolved_parameter else None

        if param_name or param_default_value:
            self._bldr += '  [Parameter (Name / DefaultValue): {} / {}]'.format(param_name if param_name else '', param_default_value if param_default_value else '')
            self._bldr += '\n'

        if isinstance(args.value, str):
            args_value = args.value
            if args_value:
                self._bldr += '  [Argument Value: {}]'.format(args_value)
                self._bldr += '\n'
        elif args.value.simple_named_reference == True if args.value else False:
            args_value = args.value.named_reference
            if args_value:
                self._bldr += '  [Argument Value: {}]'.format(args_value)
                self._bldr += '\n'
        elif args.value.explicit_reference.object_type == CdmObjectType.CONSTANT_ENTITY_DEF if args.value else False:
            const_ent = args.value.explicit_reference
            if const_ent:
                refs = []
                for val in const_ent.constant_values:
                    self._bldr += '  [Argument Value: {}]'.format(','.join(val))
                    self._bldr += '\n'

    @staticmethod
    async def validate_attribute_context(test: 'TestCase', corpus: 'CdmCorpusDefinition', expected_output_path: str, entity_name: str, resolved_entity: 'CdmEntityDefinition') -> None:
        """A function to validate if the attribute context tree & traits generated for a resolved entity is the same
        as the expected and saved attribute context tree & traits for a test case"""
        if resolved_entity.attribute_context:
            attr_ctx_util = AttributeContextUtil()

            # Expected
            expected_file_path = os.path.join(expected_output_path, 'AttrCtx_{}.txt'.format(entity_name))
            with open(expected_file_path) as expected_file:
                expected_text = expected_file.read()

            # Actual
            actual_file_path = os.path.join(expected_output_path.replace('ExpectedOutput', 'ActualOutput'), 'AttrCtx_{}.txt'.format(entity_name))
            actual_text = attr_ctx_util.get_attribute_context_strings(resolved_entity, resolved_entity.attribute_context)

            # Test if Actual is Equal to Expected
            test.assertEqual(expected_text, actual_text)

            # Save Actual AttrCtx_*.txt and Resolved_*.cdm.json
            actual_attr_ctx_file = open(actual_file_path, "w")
            actual_attr_ctx_file.write(actual_text)
            actual_attr_ctx_file.close()
            await resolved_entity.in_document.save_as_async('Resolved_{}.cdm.json'.format(entity_name), False)
