"use strict";
const core_1 = require("@aws-cdk/core");
const cxapi = require("@aws-cdk/cx-api");
const lib_1 = require("../lib");
function mockVpcContextProviderWith(test, response, paramValidator) {
    const previous = core_1.ContextProvider.getValue;
    core_1.ContextProvider.getValue = (_scope, options) => {
        // do some basic sanity checks
        test.equal(options.provider, cxapi.VPC_PROVIDER, `Expected provider to be: '${cxapi.VPC_PROVIDER}', got: '${options.provider}'`);
        test.equal((options.props || {}).returnAsymmetricSubnets, true, `Expected options.props.returnAsymmetricSubnets to be true, got: '${(options.props || {}).returnAsymmetricSubnets}'`);
        if (paramValidator) {
            paramValidator(options.props);
        }
        return {
            value: {
                availabilityZones: [],
                isolatedSubnetIds: undefined,
                isolatedSubnetNames: undefined,
                isolatedSubnetRouteTableIds: undefined,
                privateSubnetIds: undefined,
                privateSubnetNames: undefined,
                privateSubnetRouteTableIds: undefined,
                publicSubnetIds: undefined,
                publicSubnetNames: undefined,
                publicSubnetRouteTableIds: undefined,
                ...response,
            },
        };
    };
    return previous;
}
function restoreContextProvider(previous) {
    core_1.ContextProvider.getValue = previous;
}
module.exports = {
    'Vpc.fromLookup()': {
        'requires concrete values'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            test.throws(() => {
                lib_1.Vpc.fromLookup(stack, 'Vpc', {
                    vpcId: core_1.Lazy.stringValue({ produce: () => 'some-id' })
                });
            }, 'All arguments to Vpc.fromLookup() must be concrete');
            test.done();
        },
        'selecting subnets by name from a looked-up VPC does not throw'(test) {
            // GIVEN
            const stack = new core_1.Stack(undefined, undefined, { env: { region: 'us-east-1', account: '123456789012' } });
            const vpc = lib_1.Vpc.fromLookup(stack, 'VPC', {
                vpcId: 'vpc-1234'
            });
            // WHEN
            vpc.selectSubnets({ subnetName: 'Bleep' });
            // THEN: no exception
            test.done();
        },
        'accepts asymmetric subnets'(test) {
            const previous = mockVpcContextProviderWith(test, {
                vpcId: 'vpc-1234',
                subnetGroups: [
                    {
                        name: 'Public',
                        type: cxapi.VpcSubnetGroupType.PUBLIC,
                        subnets: [
                            {
                                subnetId: 'pub-sub-in-us-east-1a',
                                availabilityZone: 'us-east-1a',
                                routeTableId: 'rt-123',
                            },
                            {
                                subnetId: 'pub-sub-in-us-east-1b',
                                availabilityZone: 'us-east-1b',
                                routeTableId: 'rt-123',
                            },
                        ],
                    },
                    {
                        name: 'Private',
                        type: cxapi.VpcSubnetGroupType.PRIVATE,
                        subnets: [
                            {
                                subnetId: 'pri-sub-1-in-us-east-1c',
                                availabilityZone: 'us-east-1c',
                                routeTableId: 'rt-123',
                            },
                            {
                                subnetId: 'pri-sub-2-in-us-east-1c',
                                availabilityZone: 'us-east-1c',
                                routeTableId: 'rt-123',
                            },
                            {
                                subnetId: 'pri-sub-1-in-us-east-1d',
                                availabilityZone: 'us-east-1d',
                                routeTableId: 'rt-123',
                            },
                            {
                                subnetId: 'pri-sub-2-in-us-east-1d',
                                availabilityZone: 'us-east-1d',
                                routeTableId: 'rt-123',
                            },
                        ],
                    },
                ],
            }, options => {
                test.deepEqual(options.filter, {
                    isDefault: 'true',
                });
                test.equal(options.subnetGroupNameTag, undefined);
            });
            const stack = new core_1.Stack();
            const vpc = lib_1.Vpc.fromLookup(stack, 'Vpc', {
                isDefault: true,
            });
            test.deepEqual(vpc.availabilityZones, ['us-east-1a', 'us-east-1b', 'us-east-1c', 'us-east-1d']);
            test.equal(vpc.publicSubnets.length, 2);
            test.equal(vpc.privateSubnets.length, 4);
            test.equal(vpc.isolatedSubnets.length, 0);
            restoreContextProvider(previous);
            test.done();
        },
        'selectSubnets onePerAz works on imported VPC'(test) {
            const previous = mockVpcContextProviderWith(test, {
                vpcId: 'vpc-1234',
                subnetGroups: [
                    {
                        name: 'Public',
                        type: cxapi.VpcSubnetGroupType.PUBLIC,
                        subnets: [
                            {
                                subnetId: 'pub-sub-in-us-east-1a',
                                availabilityZone: 'us-east-1a',
                                routeTableId: 'rt-123',
                            },
                            {
                                subnetId: 'pub-sub-in-us-east-1b',
                                availabilityZone: 'us-east-1b',
                                routeTableId: 'rt-123',
                            },
                        ],
                    },
                    {
                        name: 'Private',
                        type: cxapi.VpcSubnetGroupType.PRIVATE,
                        subnets: [
                            {
                                subnetId: 'pri-sub-1-in-us-east-1c',
                                availabilityZone: 'us-east-1c',
                                routeTableId: 'rt-123',
                            },
                            {
                                subnetId: 'pri-sub-2-in-us-east-1c',
                                availabilityZone: 'us-east-1c',
                                routeTableId: 'rt-123',
                            },
                            {
                                subnetId: 'pri-sub-1-in-us-east-1d',
                                availabilityZone: 'us-east-1d',
                                routeTableId: 'rt-123',
                            },
                            {
                                subnetId: 'pri-sub-2-in-us-east-1d',
                                availabilityZone: 'us-east-1d',
                                routeTableId: 'rt-123',
                            },
                        ],
                    },
                ],
            }, options => {
                test.deepEqual(options.filter, {
                    isDefault: 'true',
                });
                test.equal(options.subnetGroupNameTag, undefined);
            });
            const stack = new core_1.Stack();
            const vpc = lib_1.Vpc.fromLookup(stack, 'Vpc', {
                isDefault: true,
            });
            // WHEN
            const subnets = vpc.selectSubnets({ subnetType: lib_1.SubnetType.PRIVATE, onePerAz: true });
            // THEN: we got 2 subnets and not 4
            test.deepEqual(subnets.subnets.map(s => s.availabilityZone), ['us-east-1c', 'us-east-1d']);
            restoreContextProvider(previous);
            test.done();
        },
    },
};
//# sourceMappingURL=data:application/json;base64,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