"use strict";
const assert_1 = require("@aws-cdk/assert");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
// tslint:disable-next-line:max-line-length
const lib_1 = require("../lib");
module.exports = {
    'gateway endpoint': {
        'add an endpoint to a vpc'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            // WHEN
            new lib_1.Vpc(stack, 'VpcNetwork', {
                gatewayEndpoints: {
                    S3: {
                        service: lib_1.GatewayVpcEndpointAwsService.S3
                    }
                }
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPCEndpoint', {
                ServiceName: {
                    'Fn::Join': [
                        '',
                        [
                            'com.amazonaws.',
                            {
                                Ref: 'AWS::Region'
                            },
                            '.s3'
                        ]
                    ]
                },
                VpcId: {
                    Ref: 'VpcNetworkB258E83A'
                },
                RouteTableIds: [
                    {
                        Ref: 'VpcNetworkPrivateSubnet1RouteTableCD085FF1'
                    },
                    {
                        Ref: 'VpcNetworkPrivateSubnet2RouteTableE97B328B'
                    },
                ],
                VpcEndpointType: 'Gateway'
            }));
            test.done();
        },
        'routing on private and public subnets'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            // WHEN
            new lib_1.Vpc(stack, 'VpcNetwork', {
                gatewayEndpoints: {
                    S3: {
                        service: lib_1.GatewayVpcEndpointAwsService.S3,
                        subnets: [
                            {
                                subnetType: lib_1.SubnetType.PUBLIC
                            },
                            {
                                subnetType: lib_1.SubnetType.PRIVATE
                            }
                        ]
                    }
                }
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPCEndpoint', {
                ServiceName: {
                    'Fn::Join': [
                        '',
                        [
                            'com.amazonaws.',
                            {
                                Ref: 'AWS::Region'
                            },
                            '.s3'
                        ]
                    ]
                },
                VpcId: {
                    Ref: 'VpcNetworkB258E83A'
                },
                RouteTableIds: [
                    {
                        Ref: 'VpcNetworkPublicSubnet1RouteTable25CCC53F'
                    },
                    {
                        Ref: 'VpcNetworkPublicSubnet2RouteTableE5F348DF'
                    },
                    {
                        Ref: 'VpcNetworkPrivateSubnet1RouteTableCD085FF1'
                    },
                    {
                        Ref: 'VpcNetworkPrivateSubnet2RouteTableE97B328B'
                    },
                ],
                VpcEndpointType: 'Gateway'
            }));
            test.done();
        },
        'add statements to policy'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const vpc = new lib_1.Vpc(stack, 'VpcNetwork');
            const endpoint = vpc.addGatewayEndpoint('S3', {
                service: lib_1.GatewayVpcEndpointAwsService.S3
            });
            // WHEN
            endpoint.addToPolicy(new aws_iam_1.PolicyStatement({
                principals: [new aws_iam_1.AnyPrincipal()],
                actions: ['s3:GetObject', 's3:ListBucket'],
                resources: ['*']
            }));
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPCEndpoint', {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: [
                                's3:GetObject',
                                's3:ListBucket'
                            ],
                            Effect: 'Allow',
                            Principal: '*',
                            Resource: '*'
                        }
                    ],
                    Version: '2012-10-17'
                }
            }));
            test.done();
        },
        'throws when adding a statement without a principal'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const vpc = new lib_1.Vpc(stack, 'VpcNetwork');
            const endpoint = vpc.addGatewayEndpoint('S3', {
                service: lib_1.GatewayVpcEndpointAwsService.S3
            });
            // THEN
            test.throws(() => endpoint.addToPolicy(new aws_iam_1.PolicyStatement({
                actions: ['s3:GetObject', 's3:ListBucket'],
                resources: ['*']
            })), /`Principal`/);
            test.done();
        },
        'import/export'(test) {
            // GIVEN
            const stack2 = new core_1.Stack();
            // WHEN
            const ep = lib_1.GatewayVpcEndpoint.fromGatewayVpcEndpointId(stack2, 'ImportedEndpoint', 'endpoint-id');
            // THEN
            test.deepEqual(ep.vpcEndpointId, 'endpoint-id');
            test.done();
        },
        'works with an imported vpc'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const vpc = lib_1.Vpc.fromVpcAttributes(stack, 'VPC', {
                vpcId: 'id',
                privateSubnetIds: ['1', '2', '3'],
                privateSubnetRouteTableIds: ['rt1', 'rt2', 'rt3'],
                availabilityZones: ['a', 'b', 'c']
            });
            // THEN
            vpc.addGatewayEndpoint('Gateway', { service: lib_1.GatewayVpcEndpointAwsService.S3 });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPCEndpoint', {
                ServiceName: { 'Fn::Join': ['', ['com.amazonaws.', { Ref: 'AWS::Region' }, '.s3']] },
                VpcId: 'id',
                RouteTableIds: ['rt1', 'rt2', 'rt3'],
                VpcEndpointType: 'Gateway',
            }));
            test.done();
        },
        'throws with an imported vpc without route tables ids'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const vpc = lib_1.Vpc.fromVpcAttributes(stack, 'VPC', {
                vpcId: 'id',
                privateSubnetIds: ['1', '2', '3'],
                availabilityZones: ['a', 'b', 'c']
            });
            test.throws(() => vpc.addGatewayEndpoint('Gateway', { service: lib_1.GatewayVpcEndpointAwsService.S3 }), /route table/);
            test.done();
        }
    },
    'interface endpoint': {
        'add an endpoint to a vpc'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const vpc = new lib_1.Vpc(stack, 'VpcNetwork');
            // WHEN
            vpc.addInterfaceEndpoint('EcrDocker', {
                service: lib_1.InterfaceVpcEndpointAwsService.ECR_DOCKER
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPCEndpoint', {
                ServiceName: {
                    'Fn::Join': [
                        '',
                        [
                            'com.amazonaws.',
                            {
                                Ref: 'AWS::Region'
                            },
                            '.ecr.dkr'
                        ]
                    ]
                },
                VpcId: {
                    Ref: 'VpcNetworkB258E83A'
                },
                PrivateDnsEnabled: true,
                SecurityGroupIds: [
                    {
                        'Fn::GetAtt': [
                            'VpcNetworkEcrDockerSecurityGroup7C91D347',
                            'GroupId'
                        ]
                    }
                ],
                SubnetIds: [
                    {
                        Ref: 'VpcNetworkPrivateSubnet1Subnet07BA143B'
                    },
                    {
                        Ref: 'VpcNetworkPrivateSubnet2Subnet5E4189D6'
                    },
                ],
                VpcEndpointType: 'Interface'
            }));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroup', {
                GroupDescription: 'VpcNetwork/EcrDocker/SecurityGroup',
                VpcId: {
                    Ref: 'VpcNetworkB258E83A'
                }
            }));
            test.done();
        },
        'import/export'(test) {
            // GIVEN
            const stack2 = new core_1.Stack();
            // WHEN
            const importedEndpoint = lib_1.InterfaceVpcEndpoint.fromInterfaceVpcEndpointAttributes(stack2, 'ImportedEndpoint', {
                securityGroups: [lib_1.SecurityGroup.fromSecurityGroupId(stack2, 'SG', 'security-group-id')],
                vpcEndpointId: 'vpc-endpoint-id',
                port: 80
            });
            importedEndpoint.connections.allowDefaultPortFromAnyIpv4();
            // THEN
            assert_1.expect(stack2).to(assert_1.haveResource('AWS::EC2::SecurityGroupIngress', {
                GroupId: 'security-group-id'
            }));
            test.deepEqual(importedEndpoint.vpcEndpointId, 'vpc-endpoint-id');
            test.done();
        },
        'with existing security groups'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const vpc = new lib_1.Vpc(stack, 'VpcNetwork');
            // WHEN
            vpc.addInterfaceEndpoint('EcrDocker', {
                service: lib_1.InterfaceVpcEndpointAwsService.ECR_DOCKER,
                securityGroups: [lib_1.SecurityGroup.fromSecurityGroupId(stack, 'SG', 'existing-id')]
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPCEndpoint', {
                SecurityGroupIds: ['existing-id'],
            }));
            test.done();
        },
        'security group has ingress by default'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const vpc = new lib_1.Vpc(stack, 'VpcNetwork');
            // WHEN
            vpc.addInterfaceEndpoint('SecretsManagerEndpoint', {
                service: lib_1.InterfaceVpcEndpointAwsService.SECRETS_MANAGER,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroup', {
                SecurityGroupIngress: [
                    {
                        CidrIp: { "Fn::GetAtt": ["VpcNetworkB258E83A", "CidrBlock"] },
                        FromPort: 443,
                        IpProtocol: "tcp",
                        ToPort: 443
                    }
                ]
            }));
            test.done();
        },
        'non-AWS service interface endpoint'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const vpc = new lib_1.Vpc(stack, 'VpcNetwork');
            // WHEN
            vpc.addInterfaceEndpoint('YourService', {
                service: new lib_1.InterfaceVpcEndpointService("com.amazonaws.vpce.us-east-1.vpce-svc-uuddlrlrbastrtsvc", 443)
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPCEndpoint', {
                ServiceName: "com.amazonaws.vpce.us-east-1.vpce-svc-uuddlrlrbastrtsvc",
                PrivateDnsEnabled: false
            }));
            test.done();
        },
        'marketplace partner service interface endpoint'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const vpc = new lib_1.Vpc(stack, 'VpcNetwork');
            // WHEN
            vpc.addInterfaceEndpoint('YourService', {
                service: { name: "com.amazonaws.vpce.us-east-1.vpce-svc-mktplacesvcwprdns",
                    port: 443,
                    privateDnsDefault: true }
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPCEndpoint', {
                ServiceName: "com.amazonaws.vpce.us-east-1.vpce-svc-mktplacesvcwprdns",
                PrivateDnsEnabled: true
            }));
            test.done();
        }
    }
};
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidGVzdC52cGMtZW5kcG9pbnQuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJ0ZXN0LnZwYy1lbmRwb2ludC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiO0FBQUEsNENBQXlFO0FBQ3pFLDhDQUFpRTtBQUNqRSx3Q0FBc0M7QUFFdEMsMkNBQTJDO0FBQzNDLGdDQUE2TDtBQUU3TCxpQkFBUztJQUNQLGtCQUFrQixFQUFFO1FBQ2xCLDBCQUEwQixDQUFDLElBQVU7WUFDbkMsUUFBUTtZQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksWUFBSyxFQUFFLENBQUM7WUFFMUIsT0FBTztZQUNQLElBQUksU0FBRyxDQUFDLEtBQUssRUFBRSxZQUFZLEVBQUU7Z0JBQzNCLGdCQUFnQixFQUFFO29CQUNoQixFQUFFLEVBQUU7d0JBQ0YsT0FBTyxFQUFFLGtDQUE0QixDQUFDLEVBQUU7cUJBQ3pDO2lCQUNGO2FBQ0YsQ0FBQyxDQUFDO1lBRUgsT0FBTztZQUNQLGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMscUJBQVksQ0FBQyx1QkFBdUIsRUFBRTtnQkFDckQsV0FBVyxFQUFFO29CQUNYLFVBQVUsRUFBRTt3QkFDVixFQUFFO3dCQUNGOzRCQUNFLGdCQUFnQjs0QkFDaEI7Z0NBQ0UsR0FBRyxFQUFFLGFBQWE7NkJBQ25COzRCQUNELEtBQUs7eUJBQ047cUJBQ0Y7aUJBQ0Y7Z0JBQ0QsS0FBSyxFQUFFO29CQUNMLEdBQUcsRUFBRSxvQkFBb0I7aUJBQzFCO2dCQUNELGFBQWEsRUFBRTtvQkFDYjt3QkFDRSxHQUFHLEVBQUUsNENBQTRDO3FCQUNsRDtvQkFDRDt3QkFDRSxHQUFHLEVBQUUsNENBQTRDO3FCQUNsRDtpQkFDRjtnQkFDRCxlQUFlLEVBQUUsU0FBUzthQUMzQixDQUFDLENBQUMsQ0FBQztZQUVKLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7UUFFRCx1Q0FBdUMsQ0FBQyxJQUFVO1lBQ2hELFFBQVE7WUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLFlBQUssRUFBRSxDQUFDO1lBRTFCLE9BQU87WUFDUCxJQUFJLFNBQUcsQ0FBQyxLQUFLLEVBQUUsWUFBWSxFQUFFO2dCQUMzQixnQkFBZ0IsRUFBRTtvQkFDaEIsRUFBRSxFQUFFO3dCQUNGLE9BQU8sRUFBRSxrQ0FBNEIsQ0FBQyxFQUFFO3dCQUN4QyxPQUFPLEVBQUU7NEJBQ1A7Z0NBQ0UsVUFBVSxFQUFFLGdCQUFVLENBQUMsTUFBTTs2QkFDOUI7NEJBQ0Q7Z0NBQ0UsVUFBVSxFQUFFLGdCQUFVLENBQUMsT0FBTzs2QkFDL0I7eUJBQ0Y7cUJBQ0Y7aUJBQ0Y7YUFDRixDQUFDLENBQUM7WUFFSCxPQUFPO1lBQ1AsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxxQkFBWSxDQUFDLHVCQUF1QixFQUFFO2dCQUNyRCxXQUFXLEVBQUU7b0JBQ1gsVUFBVSxFQUFFO3dCQUNWLEVBQUU7d0JBQ0Y7NEJBQ0UsZ0JBQWdCOzRCQUNoQjtnQ0FDRSxHQUFHLEVBQUUsYUFBYTs2QkFDbkI7NEJBQ0QsS0FBSzt5QkFDTjtxQkFDRjtpQkFDRjtnQkFDRCxLQUFLLEVBQUU7b0JBQ0wsR0FBRyxFQUFFLG9CQUFvQjtpQkFDMUI7Z0JBQ0QsYUFBYSxFQUFFO29CQUNiO3dCQUNFLEdBQUcsRUFBRSwyQ0FBMkM7cUJBQ2pEO29CQUNEO3dCQUNFLEdBQUcsRUFBRSwyQ0FBMkM7cUJBQ2pEO29CQUNEO3dCQUNFLEdBQUcsRUFBRSw0Q0FBNEM7cUJBQ2xEO29CQUNEO3dCQUNFLEdBQUcsRUFBRSw0Q0FBNEM7cUJBQ2xEO2lCQUNGO2dCQUNELGVBQWUsRUFBRSxTQUFTO2FBQzNCLENBQUMsQ0FBQyxDQUFDO1lBRUosSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ2QsQ0FBQztRQUVELDBCQUEwQixDQUFDLElBQVU7WUFDbkMsUUFBUTtZQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksWUFBSyxFQUFFLENBQUM7WUFDMUIsTUFBTSxHQUFHLEdBQUcsSUFBSSxTQUFHLENBQUMsS0FBSyxFQUFFLFlBQVksQ0FBQyxDQUFDO1lBQ3pDLE1BQU0sUUFBUSxHQUFHLEdBQUcsQ0FBQyxrQkFBa0IsQ0FBQyxJQUFJLEVBQUU7Z0JBQzVDLE9BQU8sRUFBRSxrQ0FBNEIsQ0FBQyxFQUFFO2FBQ3pDLENBQUMsQ0FBQztZQUVILE9BQU87WUFDUCxRQUFRLENBQUMsV0FBVyxDQUFDLElBQUkseUJBQWUsQ0FBQztnQkFDdkMsVUFBVSxFQUFFLENBQUMsSUFBSSxzQkFBWSxFQUFFLENBQUM7Z0JBQ2hDLE9BQU8sRUFBRSxDQUFDLGNBQWMsRUFBRSxlQUFlLENBQUM7Z0JBQzFDLFNBQVMsRUFBRSxDQUFDLEdBQUcsQ0FBQzthQUNqQixDQUFDLENBQUMsQ0FBQztZQUVKLE9BQU87WUFDUCxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsdUJBQXVCLEVBQUU7Z0JBQ3JELGNBQWMsRUFBRTtvQkFDZCxTQUFTLEVBQUU7d0JBQ1Q7NEJBQ0UsTUFBTSxFQUFFO2dDQUNOLGNBQWM7Z0NBQ2QsZUFBZTs2QkFDaEI7NEJBQ0QsTUFBTSxFQUFFLE9BQU87NEJBQ2YsU0FBUyxFQUFFLEdBQUc7NEJBQ2QsUUFBUSxFQUFFLEdBQUc7eUJBQ2Q7cUJBQ0Y7b0JBQ0QsT0FBTyxFQUFFLFlBQVk7aUJBQ3RCO2FBQ0YsQ0FBQyxDQUFDLENBQUM7WUFFSixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDZCxDQUFDO1FBRUQsb0RBQW9ELENBQUMsSUFBVTtZQUM3RCxRQUFRO1lBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxZQUFLLEVBQUUsQ0FBQztZQUMxQixNQUFNLEdBQUcsR0FBRyxJQUFJLFNBQUcsQ0FBQyxLQUFLLEVBQUUsWUFBWSxDQUFDLENBQUM7WUFDekMsTUFBTSxRQUFRLEdBQUcsR0FBRyxDQUFDLGtCQUFrQixDQUFDLElBQUksRUFBRTtnQkFDNUMsT0FBTyxFQUFFLGtDQUE0QixDQUFDLEVBQUU7YUFDekMsQ0FBQyxDQUFDO1lBRUgsT0FBTztZQUNQLElBQUksQ0FBQyxNQUFNLENBQUMsR0FBRyxFQUFFLENBQUMsUUFBUSxDQUFDLFdBQVcsQ0FBQyxJQUFJLHlCQUFlLENBQUM7Z0JBQ3pELE9BQU8sRUFBRSxDQUFDLGNBQWMsRUFBRSxlQUFlLENBQUM7Z0JBQzFDLFNBQVMsRUFBRSxDQUFDLEdBQUcsQ0FBQzthQUNqQixDQUFDLENBQUMsRUFBRSxhQUFhLENBQUMsQ0FBQztZQUVwQixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDZCxDQUFDO1FBRUQsZUFBZSxDQUFDLElBQVU7WUFDeEIsUUFBUTtZQUNSLE1BQU0sTUFBTSxHQUFHLElBQUksWUFBSyxFQUFFLENBQUM7WUFFM0IsT0FBTztZQUNQLE1BQU0sRUFBRSxHQUFHLHdCQUFrQixDQUFDLHdCQUF3QixDQUFDLE1BQU0sRUFBRSxrQkFBa0IsRUFBRSxhQUFhLENBQUMsQ0FBQztZQUVsRyxPQUFPO1lBQ1AsSUFBSSxDQUFDLFNBQVMsQ0FBQyxFQUFFLENBQUMsYUFBYSxFQUFFLGFBQWEsQ0FBQyxDQUFDO1lBQ2hELElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7UUFFRCw0QkFBNEIsQ0FBQyxJQUFVO1lBQ3JDLFFBQVE7WUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLFlBQUssRUFBRSxDQUFDO1lBQzFCLE1BQU0sR0FBRyxHQUFHLFNBQUcsQ0FBQyxpQkFBaUIsQ0FBQyxLQUFLLEVBQUUsS0FBSyxFQUFFO2dCQUM5QyxLQUFLLEVBQUUsSUFBSTtnQkFDWCxnQkFBZ0IsRUFBRSxDQUFDLEdBQUcsRUFBRSxHQUFHLEVBQUUsR0FBRyxDQUFDO2dCQUNqQywwQkFBMEIsRUFBRSxDQUFDLEtBQUssRUFBRSxLQUFLLEVBQUUsS0FBSyxDQUFDO2dCQUNqRCxpQkFBaUIsRUFBRSxDQUFDLEdBQUcsRUFBRSxHQUFHLEVBQUUsR0FBRyxDQUFDO2FBQ25DLENBQUMsQ0FBQztZQUVILE9BQU87WUFDUCxHQUFHLENBQUMsa0JBQWtCLENBQUMsU0FBUyxFQUFFLEVBQUUsT0FBTyxFQUFFLGtDQUE0QixDQUFDLEVBQUUsRUFBRSxDQUFDLENBQUM7WUFFaEYsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxxQkFBWSxDQUFDLHVCQUF1QixFQUFFO2dCQUNyRCxXQUFXLEVBQUUsRUFBRSxVQUFVLEVBQUUsQ0FBQyxFQUFFLEVBQUUsQ0FBQyxnQkFBZ0IsRUFBRSxFQUFFLEdBQUcsRUFBRSxhQUFhLEVBQUUsRUFBRSxLQUFLLENBQUMsQ0FBQyxFQUFFO2dCQUNwRixLQUFLLEVBQUUsSUFBSTtnQkFDWCxhQUFhLEVBQUUsQ0FBQyxLQUFLLEVBQUUsS0FBSyxFQUFFLEtBQUssQ0FBQztnQkFDcEMsZUFBZSxFQUFFLFNBQVM7YUFDM0IsQ0FBQyxDQUFDLENBQUM7WUFFSixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDZCxDQUFDO1FBRUQsc0RBQXNELENBQUMsSUFBVTtZQUMvRCxRQUFRO1lBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxZQUFLLEVBQUUsQ0FBQztZQUMxQixNQUFNLEdBQUcsR0FBRyxTQUFHLENBQUMsaUJBQWlCLENBQUMsS0FBSyxFQUFFLEtBQUssRUFBRTtnQkFDOUMsS0FBSyxFQUFFLElBQUk7Z0JBQ1gsZ0JBQWdCLEVBQUUsQ0FBQyxHQUFHLEVBQUUsR0FBRyxFQUFFLEdBQUcsQ0FBQztnQkFDakMsaUJBQWlCLEVBQUUsQ0FBQyxHQUFHLEVBQUUsR0FBRyxFQUFFLEdBQUcsQ0FBQzthQUNuQyxDQUFDLENBQUM7WUFFSCxJQUFJLENBQUMsTUFBTSxDQUFDLEdBQUcsRUFBRSxDQUFDLEdBQUcsQ0FBQyxrQkFBa0IsQ0FBQyxTQUFTLEVBQUUsRUFBRSxPQUFPLEVBQUUsa0NBQTRCLENBQUMsRUFBRSxFQUFFLENBQUMsRUFBRSxhQUFhLENBQUMsQ0FBQztZQUVsSCxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDZCxDQUFDO0tBQ0Y7SUFFRCxvQkFBb0IsRUFBRTtRQUNwQiwwQkFBMEIsQ0FBQyxJQUFVO1lBQ25DLFFBQVE7WUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLFlBQUssRUFBRSxDQUFDO1lBQzFCLE1BQU0sR0FBRyxHQUFHLElBQUksU0FBRyxDQUFDLEtBQUssRUFBRSxZQUFZLENBQUMsQ0FBQztZQUV6QyxPQUFPO1lBQ1AsR0FBRyxDQUFDLG9CQUFvQixDQUFDLFdBQVcsRUFBRTtnQkFDcEMsT0FBTyxFQUFFLG9DQUE4QixDQUFDLFVBQVU7YUFDbkQsQ0FBQyxDQUFDO1lBRUgsT0FBTztZQUNQLGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMscUJBQVksQ0FBQyx1QkFBdUIsRUFBRTtnQkFDckQsV0FBVyxFQUFFO29CQUNYLFVBQVUsRUFBRTt3QkFDVixFQUFFO3dCQUNGOzRCQUNFLGdCQUFnQjs0QkFDaEI7Z0NBQ0UsR0FBRyxFQUFFLGFBQWE7NkJBQ25COzRCQUNELFVBQVU7eUJBQ1g7cUJBQ0Y7aUJBQ0Y7Z0JBQ0QsS0FBSyxFQUFFO29CQUNMLEdBQUcsRUFBRSxvQkFBb0I7aUJBQzFCO2dCQUNELGlCQUFpQixFQUFFLElBQUk7Z0JBQ3ZCLGdCQUFnQixFQUFFO29CQUNoQjt3QkFDRSxZQUFZLEVBQUU7NEJBQ1osMENBQTBDOzRCQUMxQyxTQUFTO3lCQUNWO3FCQUNGO2lCQUNGO2dCQUNELFNBQVMsRUFBRTtvQkFDVDt3QkFDRSxHQUFHLEVBQUUsd0NBQXdDO3FCQUM5QztvQkFDRDt3QkFDRSxHQUFHLEVBQUUsd0NBQXdDO3FCQUM5QztpQkFDRjtnQkFDRCxlQUFlLEVBQUUsV0FBVzthQUM3QixDQUFDLENBQUMsQ0FBQztZQUVKLGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMscUJBQVksQ0FBQyx5QkFBeUIsRUFBRTtnQkFDdkQsZ0JBQWdCLEVBQUUsb0NBQW9DO2dCQUN0RCxLQUFLLEVBQUU7b0JBQ0wsR0FBRyxFQUFFLG9CQUFvQjtpQkFDMUI7YUFDRixDQUFDLENBQUMsQ0FBQztZQUVKLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7UUFFRCxlQUFlLENBQUMsSUFBVTtZQUN4QixRQUFRO1lBQ1IsTUFBTSxNQUFNLEdBQUcsSUFBSSxZQUFLLEVBQUUsQ0FBQztZQUUzQixPQUFPO1lBQ1AsTUFBTSxnQkFBZ0IsR0FBRywwQkFBb0IsQ0FBQyxrQ0FBa0MsQ0FBQyxNQUFNLEVBQUUsa0JBQWtCLEVBQUU7Z0JBQzNHLGNBQWMsRUFBRSxDQUFDLG1CQUFhLENBQUMsbUJBQW1CLENBQUMsTUFBTSxFQUFFLElBQUksRUFBRSxtQkFBbUIsQ0FBQyxDQUFDO2dCQUN0RixhQUFhLEVBQUUsaUJBQWlCO2dCQUNoQyxJQUFJLEVBQUUsRUFBRTthQUNULENBQUMsQ0FBQztZQUNILGdCQUFnQixDQUFDLFdBQVcsQ0FBQywyQkFBMkIsRUFBRSxDQUFDO1lBRTNELE9BQU87WUFDUCxlQUFNLENBQUMsTUFBTSxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsZ0NBQWdDLEVBQUU7Z0JBQy9ELE9BQU8sRUFBRSxtQkFBbUI7YUFDN0IsQ0FBQyxDQUFDLENBQUM7WUFDSixJQUFJLENBQUMsU0FBUyxDQUFDLGdCQUFnQixDQUFDLGFBQWEsRUFBRSxpQkFBaUIsQ0FBQyxDQUFDO1lBRWxFLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7UUFFRCwrQkFBK0IsQ0FBQyxJQUFVO1lBQ3hDLFFBQVE7WUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLFlBQUssRUFBRSxDQUFDO1lBQzFCLE1BQU0sR0FBRyxHQUFHLElBQUksU0FBRyxDQUFDLEtBQUssRUFBRSxZQUFZLENBQUMsQ0FBQztZQUV6QyxPQUFPO1lBQ1AsR0FBRyxDQUFDLG9CQUFvQixDQUFDLFdBQVcsRUFBRTtnQkFDcEMsT0FBTyxFQUFFLG9DQUE4QixDQUFDLFVBQVU7Z0JBQ2xELGNBQWMsRUFBRSxDQUFDLG1CQUFhLENBQUMsbUJBQW1CLENBQUMsS0FBSyxFQUFFLElBQUksRUFBRSxhQUFhLENBQUMsQ0FBQzthQUNoRixDQUFDLENBQUM7WUFFSCxPQUFPO1lBQ1AsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxxQkFBWSxDQUFDLHVCQUF1QixFQUFFO2dCQUNyRCxnQkFBZ0IsRUFBRSxDQUFDLGFBQWEsQ0FBQzthQUNsQyxDQUFDLENBQUMsQ0FBQztZQUVKLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7UUFDRCx1Q0FBdUMsQ0FBQyxJQUFVO1lBQ2hELFFBQVE7WUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLFlBQUssRUFBRSxDQUFDO1lBQzFCLE1BQU0sR0FBRyxHQUFHLElBQUksU0FBRyxDQUFDLEtBQUssRUFBRSxZQUFZLENBQUMsQ0FBQztZQUV6QyxPQUFPO1lBQ1AsR0FBRyxDQUFDLG9CQUFvQixDQUFDLHdCQUF3QixFQUFFO2dCQUNqRCxPQUFPLEVBQUUsb0NBQThCLENBQUMsZUFBZTthQUN4RCxDQUFDLENBQUM7WUFFSCxPQUFPO1lBQ1AsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyx5QkFBZ0IsQ0FBQyx5QkFBeUIsRUFBRTtnQkFDM0Qsb0JBQW9CLEVBQUU7b0JBQ3BCO3dCQUNFLE1BQU0sRUFBRSxFQUFFLFlBQVksRUFBRSxDQUFFLG9CQUFvQixFQUFFLFdBQVcsQ0FBRSxFQUFFO3dCQUMvRCxRQUFRLEVBQUUsR0FBRzt3QkFDYixVQUFVLEVBQUUsS0FBSzt3QkFDakIsTUFBTSxFQUFFLEdBQUc7cUJBQ1o7aUJBQ0Y7YUFDRixDQUFHLENBQUMsQ0FBQztZQUVOLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7UUFDRCxvQ0FBb0MsQ0FBQyxJQUFVO1lBQzdDLFFBQVE7WUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLFlBQUssRUFBRSxDQUFDO1lBQzFCLE1BQU0sR0FBRyxHQUFHLElBQUksU0FBRyxDQUFDLEtBQUssRUFBRSxZQUFZLENBQUMsQ0FBQztZQUV6QyxPQUFPO1lBQ1AsR0FBRyxDQUFDLG9CQUFvQixDQUFDLGFBQWEsRUFBRTtnQkFDdEMsT0FBTyxFQUFFLElBQUksaUNBQTJCLENBQUMseURBQXlELEVBQUUsR0FBRyxDQUFDO2FBQ3pHLENBQUMsQ0FBQztZQUVILE9BQU87WUFDUCxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsdUJBQXVCLEVBQUU7Z0JBQ3JELFdBQVcsRUFBRSx5REFBeUQ7Z0JBQ3RFLGlCQUFpQixFQUFFLEtBQUs7YUFDekIsQ0FBQyxDQUFDLENBQUM7WUFFSixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDZCxDQUFDO1FBQ0QsZ0RBQWdELENBQUMsSUFBVTtZQUN6RCxRQUFRO1lBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxZQUFLLEVBQUUsQ0FBQztZQUMxQixNQUFNLEdBQUcsR0FBRyxJQUFJLFNBQUcsQ0FBQyxLQUFLLEVBQUUsWUFBWSxDQUFDLENBQUM7WUFFekMsT0FBTztZQUNQLEdBQUcsQ0FBQyxvQkFBb0IsQ0FBQyxhQUFhLEVBQUU7Z0JBQ3RDLE9BQU8sRUFBRSxFQUFDLElBQUksRUFBRSx5REFBeUQ7b0JBQy9ELElBQUksRUFBRSxHQUFHO29CQUNULGlCQUFpQixFQUFFLElBQUksRUFBQzthQUNuQyxDQUFDLENBQUM7WUFFSCxPQUFPO1lBQ1AsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxxQkFBWSxDQUFDLHVCQUF1QixFQUFFO2dCQUNyRCxXQUFXLEVBQUUseURBQXlEO2dCQUN0RSxpQkFBaUIsRUFBRSxJQUFJO2FBQ3hCLENBQUMsQ0FBQyxDQUFDO1lBRUosSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ2QsQ0FBQztLQUNGO0NBQ0YsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IGV4cGVjdCwgaGF2ZVJlc291cmNlLCBoYXZlUmVzb3VyY2VMaWtlIH0gZnJvbSAnQGF3cy1jZGsvYXNzZXJ0JztcbmltcG9ydCB7IEFueVByaW5jaXBhbCwgUG9saWN5U3RhdGVtZW50IH0gZnJvbSAnQGF3cy1jZGsvYXdzLWlhbSc7XG5pbXBvcnQgeyBTdGFjayB9IGZyb20gJ0Bhd3MtY2RrL2NvcmUnO1xuaW1wb3J0IHsgVGVzdCB9IGZyb20gJ25vZGV1bml0Jztcbi8vIHRzbGludDpkaXNhYmxlLW5leHQtbGluZTptYXgtbGluZS1sZW5ndGhcbmltcG9ydCB7IEdhdGV3YXlWcGNFbmRwb2ludCwgR2F0ZXdheVZwY0VuZHBvaW50QXdzU2VydmljZSwgSW50ZXJmYWNlVnBjRW5kcG9pbnQsIEludGVyZmFjZVZwY0VuZHBvaW50QXdzU2VydmljZSwgSW50ZXJmYWNlVnBjRW5kcG9pbnRTZXJ2aWNlLCBTZWN1cml0eUdyb3VwLCBTdWJuZXRUeXBlLCBWcGMgfSBmcm9tICcuLi9saWInO1xuXG5leHBvcnQgPSB7XG4gICdnYXRld2F5IGVuZHBvaW50Jzoge1xuICAgICdhZGQgYW4gZW5kcG9pbnQgdG8gYSB2cGMnKHRlc3Q6IFRlc3QpIHtcbiAgICAgIC8vIEdJVkVOXG4gICAgICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuXG4gICAgICAvLyBXSEVOXG4gICAgICBuZXcgVnBjKHN0YWNrLCAnVnBjTmV0d29yaycsIHtcbiAgICAgICAgZ2F0ZXdheUVuZHBvaW50czoge1xuICAgICAgICAgIFMzOiB7XG4gICAgICAgICAgICBzZXJ2aWNlOiBHYXRld2F5VnBjRW5kcG9pbnRBd3NTZXJ2aWNlLlMzXG4gICAgICAgICAgfVxuICAgICAgICB9XG4gICAgICB9KTtcblxuICAgICAgLy8gVEhFTlxuICAgICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2UoJ0FXUzo6RUMyOjpWUENFbmRwb2ludCcsIHtcbiAgICAgICAgU2VydmljZU5hbWU6IHtcbiAgICAgICAgICAnRm46OkpvaW4nOiBbXG4gICAgICAgICAgICAnJyxcbiAgICAgICAgICAgIFtcbiAgICAgICAgICAgICAgJ2NvbS5hbWF6b25hd3MuJyxcbiAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgIFJlZjogJ0FXUzo6UmVnaW9uJ1xuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAnLnMzJ1xuICAgICAgICAgICAgXVxuICAgICAgICAgIF1cbiAgICAgICAgfSxcbiAgICAgICAgVnBjSWQ6IHtcbiAgICAgICAgICBSZWY6ICdWcGNOZXR3b3JrQjI1OEU4M0EnXG4gICAgICAgIH0sXG4gICAgICAgIFJvdXRlVGFibGVJZHM6IFtcbiAgICAgICAgICB7XG4gICAgICAgICAgICBSZWY6ICdWcGNOZXR3b3JrUHJpdmF0ZVN1Ym5ldDFSb3V0ZVRhYmxlQ0QwODVGRjEnXG4gICAgICAgICAgfSxcbiAgICAgICAgICB7XG4gICAgICAgICAgICBSZWY6ICdWcGNOZXR3b3JrUHJpdmF0ZVN1Ym5ldDJSb3V0ZVRhYmxlRTk3QjMyOEInXG4gICAgICAgICAgfSxcbiAgICAgICAgXSxcbiAgICAgICAgVnBjRW5kcG9pbnRUeXBlOiAnR2F0ZXdheSdcbiAgICAgIH0pKTtcblxuICAgICAgdGVzdC5kb25lKCk7XG4gICAgfSxcblxuICAgICdyb3V0aW5nIG9uIHByaXZhdGUgYW5kIHB1YmxpYyBzdWJuZXRzJyh0ZXN0OiBUZXN0KSB7XG4gICAgICAvLyBHSVZFTlxuICAgICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcblxuICAgICAgLy8gV0hFTlxuICAgICAgbmV3IFZwYyhzdGFjaywgJ1ZwY05ldHdvcmsnLCB7XG4gICAgICAgIGdhdGV3YXlFbmRwb2ludHM6IHtcbiAgICAgICAgICBTMzoge1xuICAgICAgICAgICAgc2VydmljZTogR2F0ZXdheVZwY0VuZHBvaW50QXdzU2VydmljZS5TMyxcbiAgICAgICAgICAgIHN1Ym5ldHM6IFtcbiAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgIHN1Ym5ldFR5cGU6IFN1Ym5ldFR5cGUuUFVCTElDXG4gICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICBzdWJuZXRUeXBlOiBTdWJuZXRUeXBlLlBSSVZBVEVcbiAgICAgICAgICAgICAgfVxuICAgICAgICAgICAgXVxuICAgICAgICAgIH1cbiAgICAgICAgfVxuICAgICAgfSk7XG5cbiAgICAgIC8vIFRIRU5cbiAgICAgIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlKCdBV1M6OkVDMjo6VlBDRW5kcG9pbnQnLCB7XG4gICAgICAgIFNlcnZpY2VOYW1lOiB7XG4gICAgICAgICAgJ0ZuOjpKb2luJzogW1xuICAgICAgICAgICAgJycsXG4gICAgICAgICAgICBbXG4gICAgICAgICAgICAgICdjb20uYW1hem9uYXdzLicsXG4gICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICBSZWY6ICdBV1M6OlJlZ2lvbidcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgJy5zMydcbiAgICAgICAgICAgIF1cbiAgICAgICAgICBdXG4gICAgICAgIH0sXG4gICAgICAgIFZwY0lkOiB7XG4gICAgICAgICAgUmVmOiAnVnBjTmV0d29ya0IyNThFODNBJ1xuICAgICAgICB9LFxuICAgICAgICBSb3V0ZVRhYmxlSWRzOiBbXG4gICAgICAgICAge1xuICAgICAgICAgICAgUmVmOiAnVnBjTmV0d29ya1B1YmxpY1N1Ym5ldDFSb3V0ZVRhYmxlMjVDQ0M1M0YnXG4gICAgICAgICAgfSxcbiAgICAgICAgICB7XG4gICAgICAgICAgICBSZWY6ICdWcGNOZXR3b3JrUHVibGljU3VibmV0MlJvdXRlVGFibGVFNUYzNDhERidcbiAgICAgICAgICB9LFxuICAgICAgICAgIHtcbiAgICAgICAgICAgIFJlZjogJ1ZwY05ldHdvcmtQcml2YXRlU3VibmV0MVJvdXRlVGFibGVDRDA4NUZGMSdcbiAgICAgICAgICB9LFxuICAgICAgICAgIHtcbiAgICAgICAgICAgIFJlZjogJ1ZwY05ldHdvcmtQcml2YXRlU3VibmV0MlJvdXRlVGFibGVFOTdCMzI4QidcbiAgICAgICAgICB9LFxuICAgICAgICBdLFxuICAgICAgICBWcGNFbmRwb2ludFR5cGU6ICdHYXRld2F5J1xuICAgICAgfSkpO1xuXG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9LFxuXG4gICAgJ2FkZCBzdGF0ZW1lbnRzIHRvIHBvbGljeScodGVzdDogVGVzdCkge1xuICAgICAgLy8gR0lWRU5cbiAgICAgIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG4gICAgICBjb25zdCB2cGMgPSBuZXcgVnBjKHN0YWNrLCAnVnBjTmV0d29yaycpO1xuICAgICAgY29uc3QgZW5kcG9pbnQgPSB2cGMuYWRkR2F0ZXdheUVuZHBvaW50KCdTMycsIHtcbiAgICAgICAgc2VydmljZTogR2F0ZXdheVZwY0VuZHBvaW50QXdzU2VydmljZS5TM1xuICAgICAgfSk7XG5cbiAgICAgIC8vIFdIRU5cbiAgICAgIGVuZHBvaW50LmFkZFRvUG9saWN5KG5ldyBQb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgICBwcmluY2lwYWxzOiBbbmV3IEFueVByaW5jaXBhbCgpXSxcbiAgICAgICAgYWN0aW9uczogWydzMzpHZXRPYmplY3QnLCAnczM6TGlzdEJ1Y2tldCddLFxuICAgICAgICByZXNvdXJjZXM6IFsnKiddXG4gICAgICB9KSk7XG5cbiAgICAgIC8vIFRIRU5cbiAgICAgIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlKCdBV1M6OkVDMjo6VlBDRW5kcG9pbnQnLCB7XG4gICAgICAgIFBvbGljeURvY3VtZW50OiB7XG4gICAgICAgICAgU3RhdGVtZW50OiBbXG4gICAgICAgICAgICB7XG4gICAgICAgICAgICAgIEFjdGlvbjogW1xuICAgICAgICAgICAgICAgICdzMzpHZXRPYmplY3QnLFxuICAgICAgICAgICAgICAgICdzMzpMaXN0QnVja2V0J1xuICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgICBFZmZlY3Q6ICdBbGxvdycsXG4gICAgICAgICAgICAgIFByaW5jaXBhbDogJyonLFxuICAgICAgICAgICAgICBSZXNvdXJjZTogJyonXG4gICAgICAgICAgICB9XG4gICAgICAgICAgXSxcbiAgICAgICAgICBWZXJzaW9uOiAnMjAxMi0xMC0xNydcbiAgICAgICAgfVxuICAgICAgfSkpO1xuXG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9LFxuXG4gICAgJ3Rocm93cyB3aGVuIGFkZGluZyBhIHN0YXRlbWVudCB3aXRob3V0IGEgcHJpbmNpcGFsJyh0ZXN0OiBUZXN0KSB7XG4gICAgICAvLyBHSVZFTlxuICAgICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcbiAgICAgIGNvbnN0IHZwYyA9IG5ldyBWcGMoc3RhY2ssICdWcGNOZXR3b3JrJyk7XG4gICAgICBjb25zdCBlbmRwb2ludCA9IHZwYy5hZGRHYXRld2F5RW5kcG9pbnQoJ1MzJywge1xuICAgICAgICBzZXJ2aWNlOiBHYXRld2F5VnBjRW5kcG9pbnRBd3NTZXJ2aWNlLlMzXG4gICAgICB9KTtcblxuICAgICAgLy8gVEhFTlxuICAgICAgdGVzdC50aHJvd3MoKCkgPT4gZW5kcG9pbnQuYWRkVG9Qb2xpY3kobmV3IFBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgIGFjdGlvbnM6IFsnczM6R2V0T2JqZWN0JywgJ3MzOkxpc3RCdWNrZXQnXSxcbiAgICAgICAgcmVzb3VyY2VzOiBbJyonXVxuICAgICAgfSkpLCAvYFByaW5jaXBhbGAvKTtcblxuICAgICAgdGVzdC5kb25lKCk7XG4gICAgfSxcblxuICAgICdpbXBvcnQvZXhwb3J0Jyh0ZXN0OiBUZXN0KSB7XG4gICAgICAvLyBHSVZFTlxuICAgICAgY29uc3Qgc3RhY2syID0gbmV3IFN0YWNrKCk7XG5cbiAgICAgIC8vIFdIRU5cbiAgICAgIGNvbnN0IGVwID0gR2F0ZXdheVZwY0VuZHBvaW50LmZyb21HYXRld2F5VnBjRW5kcG9pbnRJZChzdGFjazIsICdJbXBvcnRlZEVuZHBvaW50JywgJ2VuZHBvaW50LWlkJyk7XG5cbiAgICAgIC8vIFRIRU5cbiAgICAgIHRlc3QuZGVlcEVxdWFsKGVwLnZwY0VuZHBvaW50SWQsICdlbmRwb2ludC1pZCcpO1xuICAgICAgdGVzdC5kb25lKCk7XG4gICAgfSxcblxuICAgICd3b3JrcyB3aXRoIGFuIGltcG9ydGVkIHZwYycodGVzdDogVGVzdCkge1xuICAgICAgLy8gR0lWRU5cbiAgICAgIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG4gICAgICBjb25zdCB2cGMgPSBWcGMuZnJvbVZwY0F0dHJpYnV0ZXMoc3RhY2ssICdWUEMnLCB7XG4gICAgICAgIHZwY0lkOiAnaWQnLFxuICAgICAgICBwcml2YXRlU3VibmV0SWRzOiBbJzEnLCAnMicsICczJ10sXG4gICAgICAgIHByaXZhdGVTdWJuZXRSb3V0ZVRhYmxlSWRzOiBbJ3J0MScsICdydDInLCAncnQzJ10sXG4gICAgICAgIGF2YWlsYWJpbGl0eVpvbmVzOiBbJ2EnLCAnYicsICdjJ11cbiAgICAgIH0pO1xuXG4gICAgICAvLyBUSEVOXG4gICAgICB2cGMuYWRkR2F0ZXdheUVuZHBvaW50KCdHYXRld2F5JywgeyBzZXJ2aWNlOiBHYXRld2F5VnBjRW5kcG9pbnRBd3NTZXJ2aWNlLlMzIH0pO1xuXG4gICAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZSgnQVdTOjpFQzI6OlZQQ0VuZHBvaW50Jywge1xuICAgICAgICBTZXJ2aWNlTmFtZTogeyAnRm46OkpvaW4nOiBbJycsIFsnY29tLmFtYXpvbmF3cy4nLCB7IFJlZjogJ0FXUzo6UmVnaW9uJyB9LCAnLnMzJ11dIH0sXG4gICAgICAgIFZwY0lkOiAnaWQnLFxuICAgICAgICBSb3V0ZVRhYmxlSWRzOiBbJ3J0MScsICdydDInLCAncnQzJ10sXG4gICAgICAgIFZwY0VuZHBvaW50VHlwZTogJ0dhdGV3YXknLFxuICAgICAgfSkpO1xuXG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9LFxuXG4gICAgJ3Rocm93cyB3aXRoIGFuIGltcG9ydGVkIHZwYyB3aXRob3V0IHJvdXRlIHRhYmxlcyBpZHMnKHRlc3Q6IFRlc3QpIHtcbiAgICAgIC8vIEdJVkVOXG4gICAgICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuICAgICAgY29uc3QgdnBjID0gVnBjLmZyb21WcGNBdHRyaWJ1dGVzKHN0YWNrLCAnVlBDJywge1xuICAgICAgICB2cGNJZDogJ2lkJyxcbiAgICAgICAgcHJpdmF0ZVN1Ym5ldElkczogWycxJywgJzInLCAnMyddLFxuICAgICAgICBhdmFpbGFiaWxpdHlab25lczogWydhJywgJ2InLCAnYyddXG4gICAgICB9KTtcblxuICAgICAgdGVzdC50aHJvd3MoKCkgPT4gdnBjLmFkZEdhdGV3YXlFbmRwb2ludCgnR2F0ZXdheScsIHsgc2VydmljZTogR2F0ZXdheVZwY0VuZHBvaW50QXdzU2VydmljZS5TMyB9KSwgL3JvdXRlIHRhYmxlLyk7XG5cbiAgICAgIHRlc3QuZG9uZSgpO1xuICAgIH1cbiAgfSxcblxuICAnaW50ZXJmYWNlIGVuZHBvaW50Jzoge1xuICAgICdhZGQgYW4gZW5kcG9pbnQgdG8gYSB2cGMnKHRlc3Q6IFRlc3QpIHtcbiAgICAgIC8vIEdJVkVOXG4gICAgICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuICAgICAgY29uc3QgdnBjID0gbmV3IFZwYyhzdGFjaywgJ1ZwY05ldHdvcmsnKTtcblxuICAgICAgLy8gV0hFTlxuICAgICAgdnBjLmFkZEludGVyZmFjZUVuZHBvaW50KCdFY3JEb2NrZXInLCB7XG4gICAgICAgIHNlcnZpY2U6IEludGVyZmFjZVZwY0VuZHBvaW50QXdzU2VydmljZS5FQ1JfRE9DS0VSXG4gICAgICB9KTtcblxuICAgICAgLy8gVEhFTlxuICAgICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2UoJ0FXUzo6RUMyOjpWUENFbmRwb2ludCcsIHtcbiAgICAgICAgU2VydmljZU5hbWU6IHtcbiAgICAgICAgICAnRm46OkpvaW4nOiBbXG4gICAgICAgICAgICAnJyxcbiAgICAgICAgICAgIFtcbiAgICAgICAgICAgICAgJ2NvbS5hbWF6b25hd3MuJyxcbiAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgIFJlZjogJ0FXUzo6UmVnaW9uJ1xuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAnLmVjci5ka3InXG4gICAgICAgICAgICBdXG4gICAgICAgICAgXVxuICAgICAgICB9LFxuICAgICAgICBWcGNJZDoge1xuICAgICAgICAgIFJlZjogJ1ZwY05ldHdvcmtCMjU4RTgzQSdcbiAgICAgICAgfSxcbiAgICAgICAgUHJpdmF0ZURuc0VuYWJsZWQ6IHRydWUsXG4gICAgICAgIFNlY3VyaXR5R3JvdXBJZHM6IFtcbiAgICAgICAgICB7XG4gICAgICAgICAgICAnRm46OkdldEF0dCc6IFtcbiAgICAgICAgICAgICAgJ1ZwY05ldHdvcmtFY3JEb2NrZXJTZWN1cml0eUdyb3VwN0M5MUQzNDcnLFxuICAgICAgICAgICAgICAnR3JvdXBJZCdcbiAgICAgICAgICAgIF1cbiAgICAgICAgICB9XG4gICAgICAgIF0sXG4gICAgICAgIFN1Ym5ldElkczogW1xuICAgICAgICAgIHtcbiAgICAgICAgICAgIFJlZjogJ1ZwY05ldHdvcmtQcml2YXRlU3VibmV0MVN1Ym5ldDA3QkExNDNCJ1xuICAgICAgICAgIH0sXG4gICAgICAgICAge1xuICAgICAgICAgICAgUmVmOiAnVnBjTmV0d29ya1ByaXZhdGVTdWJuZXQyU3VibmV0NUU0MTg5RDYnXG4gICAgICAgICAgfSxcbiAgICAgICAgXSxcbiAgICAgICAgVnBjRW5kcG9pbnRUeXBlOiAnSW50ZXJmYWNlJ1xuICAgICAgfSkpO1xuXG4gICAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZSgnQVdTOjpFQzI6OlNlY3VyaXR5R3JvdXAnLCB7XG4gICAgICAgIEdyb3VwRGVzY3JpcHRpb246ICdWcGNOZXR3b3JrL0VjckRvY2tlci9TZWN1cml0eUdyb3VwJyxcbiAgICAgICAgVnBjSWQ6IHtcbiAgICAgICAgICBSZWY6ICdWcGNOZXR3b3JrQjI1OEU4M0EnXG4gICAgICAgIH1cbiAgICAgIH0pKTtcblxuICAgICAgdGVzdC5kb25lKCk7XG4gICAgfSxcblxuICAgICdpbXBvcnQvZXhwb3J0Jyh0ZXN0OiBUZXN0KSB7XG4gICAgICAvLyBHSVZFTlxuICAgICAgY29uc3Qgc3RhY2syID0gbmV3IFN0YWNrKCk7XG5cbiAgICAgIC8vIFdIRU5cbiAgICAgIGNvbnN0IGltcG9ydGVkRW5kcG9pbnQgPSBJbnRlcmZhY2VWcGNFbmRwb2ludC5mcm9tSW50ZXJmYWNlVnBjRW5kcG9pbnRBdHRyaWJ1dGVzKHN0YWNrMiwgJ0ltcG9ydGVkRW5kcG9pbnQnLCB7XG4gICAgICAgIHNlY3VyaXR5R3JvdXBzOiBbU2VjdXJpdHlHcm91cC5mcm9tU2VjdXJpdHlHcm91cElkKHN0YWNrMiwgJ1NHJywgJ3NlY3VyaXR5LWdyb3VwLWlkJyldLFxuICAgICAgICB2cGNFbmRwb2ludElkOiAndnBjLWVuZHBvaW50LWlkJyxcbiAgICAgICAgcG9ydDogODBcbiAgICAgIH0pO1xuICAgICAgaW1wb3J0ZWRFbmRwb2ludC5jb25uZWN0aW9ucy5hbGxvd0RlZmF1bHRQb3J0RnJvbUFueUlwdjQoKTtcblxuICAgICAgLy8gVEhFTlxuICAgICAgZXhwZWN0KHN0YWNrMikudG8oaGF2ZVJlc291cmNlKCdBV1M6OkVDMjo6U2VjdXJpdHlHcm91cEluZ3Jlc3MnLCB7XG4gICAgICAgIEdyb3VwSWQ6ICdzZWN1cml0eS1ncm91cC1pZCdcbiAgICAgIH0pKTtcbiAgICAgIHRlc3QuZGVlcEVxdWFsKGltcG9ydGVkRW5kcG9pbnQudnBjRW5kcG9pbnRJZCwgJ3ZwYy1lbmRwb2ludC1pZCcpO1xuXG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9LFxuXG4gICAgJ3dpdGggZXhpc3Rpbmcgc2VjdXJpdHkgZ3JvdXBzJyh0ZXN0OiBUZXN0KSB7XG4gICAgICAvLyBHSVZFTlxuICAgICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcbiAgICAgIGNvbnN0IHZwYyA9IG5ldyBWcGMoc3RhY2ssICdWcGNOZXR3b3JrJyk7XG5cbiAgICAgIC8vIFdIRU5cbiAgICAgIHZwYy5hZGRJbnRlcmZhY2VFbmRwb2ludCgnRWNyRG9ja2VyJywge1xuICAgICAgICBzZXJ2aWNlOiBJbnRlcmZhY2VWcGNFbmRwb2ludEF3c1NlcnZpY2UuRUNSX0RPQ0tFUixcbiAgICAgICAgc2VjdXJpdHlHcm91cHM6IFtTZWN1cml0eUdyb3VwLmZyb21TZWN1cml0eUdyb3VwSWQoc3RhY2ssICdTRycsICdleGlzdGluZy1pZCcpXVxuICAgICAgfSk7XG5cbiAgICAgIC8vIFRIRU5cbiAgICAgIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlKCdBV1M6OkVDMjo6VlBDRW5kcG9pbnQnLCB7XG4gICAgICAgIFNlY3VyaXR5R3JvdXBJZHM6IFsnZXhpc3RpbmctaWQnXSxcbiAgICAgIH0pKTtcblxuICAgICAgdGVzdC5kb25lKCk7XG4gICAgfSxcbiAgICAnc2VjdXJpdHkgZ3JvdXAgaGFzIGluZ3Jlc3MgYnkgZGVmYXVsdCcodGVzdDogVGVzdCkge1xuICAgICAgLy8gR0lWRU5cbiAgICAgIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG4gICAgICBjb25zdCB2cGMgPSBuZXcgVnBjKHN0YWNrLCAnVnBjTmV0d29yaycpO1xuXG4gICAgICAvLyBXSEVOXG4gICAgICB2cGMuYWRkSW50ZXJmYWNlRW5kcG9pbnQoJ1NlY3JldHNNYW5hZ2VyRW5kcG9pbnQnLCB7XG4gICAgICAgIHNlcnZpY2U6IEludGVyZmFjZVZwY0VuZHBvaW50QXdzU2VydmljZS5TRUNSRVRTX01BTkFHRVIsXG4gICAgICB9KTtcblxuICAgICAgLy8gVEhFTlxuICAgICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2VMaWtlKCdBV1M6OkVDMjo6U2VjdXJpdHlHcm91cCcsIHtcbiAgICAgICAgU2VjdXJpdHlHcm91cEluZ3Jlc3M6IFtcbiAgICAgICAgICB7XG4gICAgICAgICAgICBDaWRySXA6IHsgXCJGbjo6R2V0QXR0XCI6IFsgXCJWcGNOZXR3b3JrQjI1OEU4M0FcIiwgXCJDaWRyQmxvY2tcIiBdIH0sXG4gICAgICAgICAgICBGcm9tUG9ydDogNDQzLFxuICAgICAgICAgICAgSXBQcm90b2NvbDogXCJ0Y3BcIixcbiAgICAgICAgICAgIFRvUG9ydDogNDQzXG4gICAgICAgICAgfVxuICAgICAgICBdXG4gICAgICB9LCApKTtcblxuICAgICAgdGVzdC5kb25lKCk7XG4gICAgfSxcbiAgICAnbm9uLUFXUyBzZXJ2aWNlIGludGVyZmFjZSBlbmRwb2ludCcodGVzdDogVGVzdCkge1xuICAgICAgLy8gR0lWRU5cbiAgICAgIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG4gICAgICBjb25zdCB2cGMgPSBuZXcgVnBjKHN0YWNrLCAnVnBjTmV0d29yaycpO1xuXG4gICAgICAvLyBXSEVOXG4gICAgICB2cGMuYWRkSW50ZXJmYWNlRW5kcG9pbnQoJ1lvdXJTZXJ2aWNlJywge1xuICAgICAgICBzZXJ2aWNlOiBuZXcgSW50ZXJmYWNlVnBjRW5kcG9pbnRTZXJ2aWNlKFwiY29tLmFtYXpvbmF3cy52cGNlLnVzLWVhc3QtMS52cGNlLXN2Yy11dWRkbHJscmJhc3RydHN2Y1wiLCA0NDMpXG4gICAgICB9KTtcblxuICAgICAgLy8gVEhFTlxuICAgICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2UoJ0FXUzo6RUMyOjpWUENFbmRwb2ludCcsIHtcbiAgICAgICAgU2VydmljZU5hbWU6IFwiY29tLmFtYXpvbmF3cy52cGNlLnVzLWVhc3QtMS52cGNlLXN2Yy11dWRkbHJscmJhc3RydHN2Y1wiLFxuICAgICAgICBQcml2YXRlRG5zRW5hYmxlZDogZmFsc2VcbiAgICAgIH0pKTtcblxuICAgICAgdGVzdC5kb25lKCk7XG4gICAgfSxcbiAgICAnbWFya2V0cGxhY2UgcGFydG5lciBzZXJ2aWNlIGludGVyZmFjZSBlbmRwb2ludCcodGVzdDogVGVzdCkge1xuICAgICAgLy8gR0lWRU5cbiAgICAgIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG4gICAgICBjb25zdCB2cGMgPSBuZXcgVnBjKHN0YWNrLCAnVnBjTmV0d29yaycpO1xuXG4gICAgICAvLyBXSEVOXG4gICAgICB2cGMuYWRkSW50ZXJmYWNlRW5kcG9pbnQoJ1lvdXJTZXJ2aWNlJywge1xuICAgICAgICBzZXJ2aWNlOiB7bmFtZTogXCJjb20uYW1hem9uYXdzLnZwY2UudXMtZWFzdC0xLnZwY2Utc3ZjLW1rdHBsYWNlc3Zjd3ByZG5zXCIsXG4gICAgICAgICAgICAgICAgICBwb3J0OiA0NDMsXG4gICAgICAgICAgICAgICAgICBwcml2YXRlRG5zRGVmYXVsdDogdHJ1ZX1cbiAgICAgIH0pO1xuXG4gICAgICAvLyBUSEVOXG4gICAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZSgnQVdTOjpFQzI6OlZQQ0VuZHBvaW50Jywge1xuICAgICAgICBTZXJ2aWNlTmFtZTogXCJjb20uYW1hem9uYXdzLnZwY2UudXMtZWFzdC0xLnZwY2Utc3ZjLW1rdHBsYWNlc3Zjd3ByZG5zXCIsXG4gICAgICAgIFByaXZhdGVEbnNFbmFibGVkOiB0cnVlXG4gICAgICB9KSk7XG5cbiAgICAgIHRlc3QuZG9uZSgpO1xuICAgIH1cbiAgfVxufTtcbiJdfQ==