import { IBucket } from "@aws-cdk/aws-s3";
import { Resource } from "@aws-cdk/core";
import { OperatingSystemType } from "./machine-image";
/**
 * Options when constructing UserData for Linux
 */
export interface LinuxUserDataOptions {
    /**
     * Shebang for the UserData script
     *
     * @default "#!/bin/bash"
     */
    readonly shebang?: string;
}
/**
 * Options when downloading files from S3
 */
export interface S3DownloadOptions {
    /**
     * Name of the S3 bucket to download from
     */
    readonly bucket: IBucket;
    /**
     * The key of the file to download
     */
    readonly bucketKey: string;
    /**
     * The name of the local file.
     *
     * @default Linux   - /tmp/bucketKey
     *          Windows - %TEMP%/bucketKey
     */
    readonly localFile?: string;
}
/**
 * Options when executing a file.
 */
export interface ExecuteFileOptions {
    /**
     * The path to the file.
     */
    readonly filePath: string;
    /**
     * The arguments to be passed to the file.
     *
     * @default No arguments are passed to the file.
     */
    readonly arguments?: string;
}
/**
 * Instance User Data
 */
export declare abstract class UserData {
    /**
     * Create a userdata object for Linux hosts
     */
    static forLinux(options?: LinuxUserDataOptions): UserData;
    /**
     * Create a userdata object for Windows hosts
     */
    static forWindows(): UserData;
    /**
     * Create a userdata object with custom content
     */
    static custom(content: string): UserData;
    static forOperatingSystem(os: OperatingSystemType): UserData;
    /**
     * Add one or more commands to the user data
     */
    abstract addCommands(...commands: string[]): void;
    /**
     * Add one or more commands to the user data that will run when the script exits.
     */
    abstract addOnExitCommands(...commands: string[]): void;
    /**
     * Render the UserData for use in a construct
     */
    abstract render(): string;
    /**
     * Adds commands to download a file from S3
     *
     * @returns: The local path that the file will be downloaded to
     */
    abstract addS3DownloadCommand(params: S3DownloadOptions): string;
    /**
     * Adds commands to execute a file
     */
    abstract addExecuteFileCommand(params: ExecuteFileOptions): void;
    /**
     * Adds a command which will send a cfn-signal when the user data script ends
     */
    abstract addSignalOnExitCommand(resource: Resource): void;
}
