"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
const connections_1 = require("./connections");
/**
 * Peer object factories (to be used in Security Group management)
 *
 * The static methods on this object can be used to create peer objects
 * which represent a connection partner in Security Group rules.
 *
 * Use this object if you need to represent connection partners using plain IP
 * addresses, or a prefix list ID.
 *
 * If you want to address a connection partner by Security Group, you can just
 * use the Security Group (or the construct that contains a Security Group)
 * directly, as it already implements `IPeer`.
 */
class Peer {
    /**
     * Create an IPv4 peer from a CIDR
     */
    static ipv4(cidrIp) {
        return new CidrIPv4(cidrIp);
    }
    /**
     * Any IPv4 address
     */
    static anyIpv4() {
        return new AnyIPv4();
    }
    /**
     * Create an IPv6 peer from a CIDR
     */
    static ipv6(cidrIp) {
        return new CidrIPv6(cidrIp);
    }
    /**
     * Any IPv6 address
     */
    static anyIpv6() {
        return new AnyIPv6();
    }
    /**
     * A prefix list
     */
    static prefixList(prefixListId) {
        return new PrefixList(prefixListId);
    }
    constructor() {
    }
}
exports.Peer = Peer;
/**
 * A connection to and from a given IP range
 */
class CidrIPv4 {
    constructor(cidrIp) {
        this.cidrIp = cidrIp;
        this.canInlineRule = true;
        this.connections = new connections_1.Connections({ peer: this });
        if (!core_1.Token.isUnresolved(cidrIp)) {
            const cidrMatch = cidrIp.match(/^(\d{1,3}\.){3}\d{1,3}(\/\d+)?$/);
            if (!cidrMatch) {
                throw new Error(`Invalid IPv4 CIDR: "${cidrIp}"`);
            }
            if (!cidrMatch[2]) {
                throw new Error(`CIDR mask is missing in IPv4: "${cidrIp}". Did you mean "${cidrIp}/32"?`);
            }
        }
        this.uniqueId = cidrIp;
    }
    /**
     * Produce the ingress rule JSON for the given connection
     */
    toIngressRuleConfig() {
        return { cidrIp: this.cidrIp };
    }
    /**
     * Produce the egress rule JSON for the given connection
     */
    toEgressRuleConfig() {
        return { cidrIp: this.cidrIp };
    }
}
/**
 * Any IPv4 address
 */
class AnyIPv4 extends CidrIPv4 {
    constructor() {
        super("0.0.0.0/0");
    }
}
/**
 * A connection to a from a given IPv6 range
 */
class CidrIPv6 {
    constructor(cidrIpv6) {
        this.cidrIpv6 = cidrIpv6;
        this.canInlineRule = true;
        this.connections = new connections_1.Connections({ peer: this });
        if (!core_1.Token.isUnresolved(cidrIpv6)) {
            const cidrMatch = cidrIpv6.match(/^([\da-f]{0,4}:){2,7}([\da-f]{0,4})?(\/\d+)?$/);
            if (!cidrMatch) {
                throw new Error(`Invalid IPv6 CIDR: "${cidrIpv6}"`);
            }
            if (!cidrMatch[3]) {
                throw new Error(`CIDR mask is missing in IPv6: "${cidrIpv6}". Did you mean "${cidrIpv6}/128"?`);
            }
        }
        this.uniqueId = cidrIpv6;
    }
    /**
     * Produce the ingress rule JSON for the given connection
     */
    toIngressRuleConfig() {
        return { cidrIpv6: this.cidrIpv6 };
    }
    /**
     * Produce the egress rule JSON for the given connection
     */
    toEgressRuleConfig() {
        return { cidrIpv6: this.cidrIpv6 };
    }
}
/**
 * Any IPv6 address
 */
class AnyIPv6 extends CidrIPv6 {
    constructor() {
        super("::/0");
    }
}
/**
 * A prefix list
 *
 * Prefix lists are used to allow traffic to VPC-local service endpoints.
 *
 * For more information, see this page:
 *
 * https://docs.aws.amazon.com/AmazonVPC/latest/UserGuide/vpc-endpoints.html
 */
class PrefixList {
    constructor(prefixListId) {
        this.prefixListId = prefixListId;
        this.canInlineRule = false;
        this.connections = new connections_1.Connections({ peer: this });
        this.uniqueId = prefixListId;
    }
    toIngressRuleConfig() {
        return { sourcePrefixListId: this.prefixListId };
    }
    toEgressRuleConfig() {
        return { destinationPrefixListId: this.prefixListId };
    }
}
//# sourceMappingURL=data:application/json;base64,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