"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const ssm = require("@aws-cdk/aws-ssm");
const core_1 = require("@aws-cdk/core");
const cxapi = require("@aws-cdk/cx-api");
/**
 * Factory functions for standard Amazon Machine Image objects.
 */
class MachineImage {
    /**
     * A Windows image that is automatically kept up-to-date
     *
     * This Machine Image automatically updates to the latest version on every
     * deployment. Be aware this will cause your instances to be replaced when a
     * new version of the image becomes available. Do not store stateful information
     * on the instance if you are using this image.
     */
    static latestWindows(version, props) {
        return new WindowsImage(version, props);
    }
    /**
     * An Amazon Linux image that is automatically kept up-to-date
     *
     * This Machine Image automatically updates to the latest version on every
     * deployment. Be aware this will cause your instances to be replaced when a
     * new version of the image becomes available. Do not store stateful information
     * on the instance if you are using this image.
     */
    static latestAmazonLinux(props) {
        return new AmazonLinuxImage(props);
    }
    /**
     * A Linux image where you specify the AMI ID for every region
     *
     * @param amiMap For every region where you are deploying the stack,
     * specify the AMI ID for that region.
     * @param props Customize the image by supplying additional props
     */
    static genericLinux(amiMap, props) {
        return new GenericLinuxImage(amiMap, props);
    }
    /**
     * A Windows image where you specify the AMI ID for every region
     *
     * @param amiMap For every region where you are deploying the stack,
     * specify the AMI ID for that region.
     * @param props Customize the image by supplying additional props
     */
    static genericWindows(amiMap, props) {
        return new GenericWindowsImage(amiMap, props);
    }
    /**
     * Look up a shared Machine Image using DescribeImages
     *
     * The most recent, available, launchable image matching the given filter
     * criteria will be used. Looking up AMIs may take a long time; specify
     * as many filter criteria as possible to narrow down the search.
     *
     * The AMI selected will be cached in `cdk.context.json` and the same value
     * will be used on future runs. To refresh the AMI lookup, you will have to
     * evict the value from the cache using the `cdk context` command. See
     * https://docs.aws.amazon.com/cdk/latest/guide/context.html for more information.
     */
    static lookup(props) {
        return new LookupMachineImage(props);
    }
}
exports.MachineImage = MachineImage;
/**
 * Select the latest version of the indicated Windows version
 *
 * This Machine Image automatically updates to the latest version on every
 * deployment. Be aware this will cause your instances to be replaced when a
 * new version of the image becomes available. Do not store stateful information
 * on the instance if you are using this image.
 *
 * The AMI ID is selected using the values published to the SSM parameter store.
 *
 * https://aws.amazon.com/blogs/mt/query-for-the-latest-windows-ami-using-systems-manager-parameter-store/
 */
class WindowsImage {
    constructor(version, props = {}) {
        this.version = version;
        this.props = props;
    }
    /**
     * Return the image to use in the given context
     */
    getImage(scope) {
        const parameterName = this.imageParameterName();
        const ami = ssm.StringParameter.valueForTypedStringParameter(scope, parameterName, ssm.ParameterType.AWS_EC2_IMAGE_ID);
        return {
            imageId: ami,
            userData: this.props.userData,
            osType: OperatingSystemType.WINDOWS,
        };
    }
    /**
     * Construct the SSM parameter name for the given Windows image
     */
    imageParameterName() {
        return '/aws/service/ami-windows-latest/' + this.version;
    }
}
exports.WindowsImage = WindowsImage;
/**
 * Selects the latest version of Amazon Linux
 *
 * This Machine Image automatically updates to the latest version on every
 * deployment. Be aware this will cause your instances to be replaced when a
 * new version of the image becomes available. Do not store stateful information
 * on the instance if you are using this image.
 *
 * The AMI ID is selected using the values published to the SSM parameter store.
 */
class AmazonLinuxImage {
    constructor(props = {}) {
        this.props = props;
        this.generation = (props && props.generation) || AmazonLinuxGeneration.AMAZON_LINUX;
        this.edition = (props && props.edition) || AmazonLinuxEdition.STANDARD;
        this.virtualization = (props && props.virtualization) || AmazonLinuxVirt.HVM;
        this.storage = (props && props.storage) || AmazonLinuxStorage.GENERAL_PURPOSE;
    }
    /**
     * Return the image to use in the given context
     */
    getImage(scope) {
        const parts = [
            this.generation,
            'ami',
            this.edition !== AmazonLinuxEdition.STANDARD ? this.edition : undefined,
            this.virtualization,
            'x86_64',
            this.storage
        ].filter(x => x !== undefined); // Get rid of undefineds
        const parameterName = '/aws/service/ami-amazon-linux-latest/' + parts.join('-');
        const ami = ssm.StringParameter.valueForTypedStringParameter(scope, parameterName, ssm.ParameterType.AWS_EC2_IMAGE_ID);
        return {
            imageId: ami,
            userData: this.props.userData,
            osType: OperatingSystemType.LINUX,
        };
    }
}
exports.AmazonLinuxImage = AmazonLinuxImage;
/**
 * What generation of Amazon Linux to use
 */
var AmazonLinuxGeneration;
(function (AmazonLinuxGeneration) {
    /**
     * Amazon Linux
     */
    AmazonLinuxGeneration["AMAZON_LINUX"] = "amzn";
    /**
     * Amazon Linux 2
     */
    AmazonLinuxGeneration["AMAZON_LINUX_2"] = "amzn2";
})(AmazonLinuxGeneration = exports.AmazonLinuxGeneration || (exports.AmazonLinuxGeneration = {}));
/**
 * Amazon Linux edition
 */
var AmazonLinuxEdition;
(function (AmazonLinuxEdition) {
    /**
     * Standard edition
     */
    AmazonLinuxEdition["STANDARD"] = "standard";
    /**
     * Minimal edition
     */
    AmazonLinuxEdition["MINIMAL"] = "minimal";
})(AmazonLinuxEdition = exports.AmazonLinuxEdition || (exports.AmazonLinuxEdition = {}));
/**
 * Virtualization type for Amazon Linux
 */
var AmazonLinuxVirt;
(function (AmazonLinuxVirt) {
    /**
     * HVM virtualization (recommended)
     */
    AmazonLinuxVirt["HVM"] = "hvm";
    /**
     * PV virtualization
     */
    AmazonLinuxVirt["PV"] = "pv";
})(AmazonLinuxVirt = exports.AmazonLinuxVirt || (exports.AmazonLinuxVirt = {}));
var AmazonLinuxStorage;
(function (AmazonLinuxStorage) {
    /**
     * EBS-backed storage
     */
    AmazonLinuxStorage["EBS"] = "ebs";
    /**
     * S3-backed storage
     */
    AmazonLinuxStorage["S3"] = "ebs";
    /**
     * General Purpose-based storage (recommended)
     */
    AmazonLinuxStorage["GENERAL_PURPOSE"] = "gp2";
})(AmazonLinuxStorage = exports.AmazonLinuxStorage || (exports.AmazonLinuxStorage = {}));
/**
 * Construct a Linux machine image from an AMI map
 *
 * Linux images IDs are not published to SSM parameter store yet, so you'll have to
 * manually specify an AMI map.
 */
class GenericLinuxImage {
    constructor(amiMap, props = {}) {
        this.amiMap = amiMap;
        this.props = props;
    }
    getImage(scope) {
        const region = core_1.Stack.of(scope).region;
        if (core_1.Token.isUnresolved(region)) {
            throw new Error(`Unable to determine AMI from AMI map since stack is region-agnostic`);
        }
        const ami = region !== 'test-region' ? this.amiMap[region] : 'ami-12345';
        if (!ami) {
            throw new Error(`Unable to find AMI in AMI map: no AMI specified for region '${region}'`);
        }
        return {
            imageId: ami,
            userData: this.props.userData,
            osType: OperatingSystemType.LINUX,
        };
    }
}
exports.GenericLinuxImage = GenericLinuxImage;
/**
 * Construct a Windows machine image from an AMI map
 *
 * Allows you to create a generic Windows EC2 , manually specify an AMI map.
 */
class GenericWindowsImage {
    constructor(amiMap, props = {}) {
        this.amiMap = amiMap;
        this.props = props;
    }
    getImage(scope) {
        const region = core_1.Stack.of(scope).region;
        if (core_1.Token.isUnresolved(region)) {
            throw new Error(`Unable to determine AMI from AMI map since stack is region-agnostic`);
        }
        const ami = region !== 'test-region' ? this.amiMap[region] : 'ami-12345';
        if (!ami) {
            throw new Error(`Unable to find AMI in AMI map: no AMI specified for region '${region}'`);
        }
        return {
            imageId: ami,
            userData: this.props.userData,
            osType: OperatingSystemType.WINDOWS,
        };
    }
}
exports.GenericWindowsImage = GenericWindowsImage;
/**
 * The OS type of a particular image
 */
var OperatingSystemType;
(function (OperatingSystemType) {
    OperatingSystemType[OperatingSystemType["LINUX"] = 0] = "LINUX";
    OperatingSystemType[OperatingSystemType["WINDOWS"] = 1] = "WINDOWS";
})(OperatingSystemType = exports.OperatingSystemType || (exports.OperatingSystemType = {}));
/**
 * A machine image whose AMI ID will be searched using DescribeImages.
 *
 * The most recent, available, launchable image matching the given filter
 * criteria will be used. Looking up AMIs may take a long time; specify
 * as many filter criteria as possible to narrow down the search.
 *
 * The AMI selected will be cached in `cdk.context.json` and the same value
 * will be used on future runs. To refresh the AMI lookup, you will have to
 * evict the value from the cache using the `cdk context` command. See
 * https://docs.aws.amazon.com/cdk/latest/guide/context.html for more information.
 */
class LookupMachineImage {
    constructor(props) {
        this.props = props;
    }
    getImage(scope) {
        // Need to know 'windows' or not before doing the query to return the right
        // osType for the dummy value, so might as well add it to the filter.
        const filters = {
            'name': [this.props.name],
            'state': ['available'],
            'image-type': ['machine'],
            'platform': this.props.windows ? ['windows'] : undefined,
        };
        Object.assign(filters, this.props.filters);
        const value = core_1.ContextProvider.getValue(scope, {
            provider: cxapi.AMI_PROVIDER,
            props: {
                owners: this.props.owners,
                filters,
            },
            dummyValue: 'ami-1234',
        }).value;
        if (typeof value !== 'string') {
            throw new Error(`Response to AMI lookup invalid, got: ${value}`);
        }
        return {
            imageId: value,
            osType: this.props.windows ? OperatingSystemType.WINDOWS : OperatingSystemType.LINUX,
            userData: this.props.userData
        };
    }
}
exports.LookupMachineImage = LookupMachineImage;
//# sourceMappingURL=data:application/json;base64,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