"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const connections_1 = require("./connections");
const ec2_generated_1 = require("./ec2.generated");
const security_group_1 = require("./security-group");
const user_data_1 = require("./user-data");
const volume_1 = require("./volume");
/**
 * Name tag constant
 */
const NAME_TAG = 'Name';
/**
 * This represents a single EC2 instance
 */
class Instance extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        this.securityGroups = [];
        if (props.securityGroup) {
            this.securityGroup = props.securityGroup;
        }
        else {
            this.securityGroup = new security_group_1.SecurityGroup(this, 'InstanceSecurityGroup', {
                vpc: props.vpc,
                allowAllOutbound: props.allowAllOutbound !== false
            });
        }
        this.connections = new connections_1.Connections({ securityGroups: [this.securityGroup] });
        this.securityGroups.push(this.securityGroup);
        core_1.Tag.add(this, NAME_TAG, props.instanceName || this.node.path);
        this.role = props.role || new iam.Role(this, 'InstanceRole', {
            assumedBy: new iam.ServicePrincipal('ec2.amazonaws.com')
        });
        this.grantPrincipal = this.role;
        const iamProfile = new iam.CfnInstanceProfile(this, 'InstanceProfile', {
            roles: [this.role.roleName]
        });
        // use delayed evaluation
        const imageConfig = props.machineImage.getImage(this);
        this.userData = props.userData || imageConfig.userData || user_data_1.UserData.forOperatingSystem(imageConfig.osType);
        const userDataToken = core_1.Lazy.stringValue({ produce: () => core_1.Fn.base64(this.userData.render()) });
        const securityGroupsToken = core_1.Lazy.listValue({ produce: () => this.securityGroups.map(sg => sg.securityGroupId) });
        const { subnets } = props.vpc.selectSubnets(props.vpcSubnets);
        let subnet;
        if (props.availabilityZone) {
            const selected = subnets.filter(sn => sn.availabilityZone === props.availabilityZone);
            if (selected.length === 1) {
                subnet = selected[0];
            }
            else {
                throw new Error('When specifying AZ there has to be exactly on subnet of the given type in this az');
            }
        }
        else {
            subnet = subnets[0];
        }
        this.instance = new ec2_generated_1.CfnInstance(this, 'Resource', {
            imageId: imageConfig.imageId,
            keyName: props.keyName,
            instanceType: props.instanceType.toString(),
            securityGroupIds: securityGroupsToken,
            iamInstanceProfile: iamProfile.ref,
            userData: userDataToken,
            subnetId: subnet.subnetId,
            availabilityZone: subnet.availabilityZone,
            sourceDestCheck: props.sourceDestCheck,
            blockDeviceMappings: props.blockDevices !== undefined ? volume_1.synthesizeBlockDeviceMappings(this, props.blockDevices) : undefined,
            privateIpAddress: props.privateIpAddress
        });
        this.instance.node.addDependency(this.role);
        this.osType = imageConfig.osType;
        this.node.defaultChild = this.instance;
        this.instanceId = this.instance.ref;
        this.instanceAvailabilityZone = this.instance.attrAvailabilityZone;
        this.instancePrivateDnsName = this.instance.attrPrivateDnsName;
        this.instancePrivateIp = this.instance.attrPrivateIp;
        this.instancePublicDnsName = this.instance.attrPublicDnsName;
        this.instancePublicIp = this.instance.attrPublicIp;
        this.applyUpdatePolicies(props);
    }
    /**
     * Add the security group to the instance.
     *
     * @param securityGroup: The security group to add
     */
    addSecurityGroup(securityGroup) {
        this.securityGroups.push(securityGroup);
    }
    /**
     * Add command to the startup script of the instance.
     * The command must be in the scripting language supported by the instance's OS (i.e. Linux/Windows).
     */
    addUserData(...commands) {
        this.userData.addCommands(...commands);
    }
    /**
     * Adds a statement to the IAM role assumed by the instance.
     */
    addToRolePolicy(statement) {
        this.role.addToPolicy(statement);
    }
    /**
     * Apply CloudFormation update policies for the instance
     */
    applyUpdatePolicies(props) {
        if (props.resourceSignalTimeout !== undefined) {
            this.instance.cfnOptions.creationPolicy = {
                ...this.instance.cfnOptions.creationPolicy,
                resourceSignal: {
                    timeout: props.resourceSignalTimeout && props.resourceSignalTimeout.toISOString(),
                }
            };
        }
    }
}
exports.Instance = Instance;
//# sourceMappingURL=data:application/json;base64,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