"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const _1 = require(".");
const instance_1 = require("./instance");
const machine_image_1 = require("./machine-image");
const port_1 = require("./port");
/**
 * This creates a linux bastion host you can use to connect to other instances or services in your VPC.
 * The recommended way to connect to the bastion host is by using AWS Systems Manager Session Manager.
 *
 * The operating system is Amazon Linux 2 with the latest SSM agent installed
 *
 * You can also configure this bastion host to allow connections via SSH
 *
 * @experimental
 */
class BastionHostLinux extends core_1.Construct {
    constructor(scope, id, props) {
        var _a, _b, _c, _d;
        super(scope, id);
        this.stack = core_1.Stack.of(scope);
        this.instance = new instance_1.Instance(this, 'Resource', {
            vpc: props.vpc,
            availabilityZone: props.availabilityZone,
            securityGroup: props.securityGroup,
            instanceName: (_a = props.instanceName, (_a !== null && _a !== void 0 ? _a : 'BastionHost')),
            instanceType: (_b = props.instanceType, (_b !== null && _b !== void 0 ? _b : _1.InstanceType.of(_1.InstanceClass.T3, _1.InstanceSize.NANO))),
            machineImage: (_c = props.machineImage, (_c !== null && _c !== void 0 ? _c : machine_image_1.MachineImage.latestAmazonLinux({ generation: _1.AmazonLinuxGeneration.AMAZON_LINUX_2 }))),
            vpcSubnets: (_d = props.subnetSelection, (_d !== null && _d !== void 0 ? _d : {})),
        });
        this.instance.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: [
                'ssmmessages:*',
                'ssm:UpdateInstanceInformation',
                'ec2messages:*'
            ],
            resources: ['*'],
        }));
        this.instance.addUserData('yum install -y https://s3.amazonaws.com/ec2-downloads-windows/SSMAgent/latest/linux_amd64/amazon-ssm-agent.rpm');
        this.connections = this.instance.connections;
        this.role = this.instance.role;
        this.grantPrincipal = this.instance.role;
        this.instanceId = this.instance.instanceId;
        this.instancePrivateIp = this.instance.instancePrivateIp;
        this.instanceAvailabilityZone = this.instance.instanceAvailabilityZone;
        this.instancePrivateDnsName = this.instance.instancePrivateDnsName;
        this.instancePublicIp = this.instance.instancePublicIp;
        this.instancePublicDnsName = this.instance.instancePublicDnsName;
        new core_1.CfnOutput(this, 'BastionHostId', {
            description: 'Instance ID of the bastion host. Use this to connect via SSM Session Manager',
            value: this.instanceId,
        });
    }
    /**
     * Allow SSH access from the given peer or peers
     *
     * Necessary if you want to connect to the instance using ssh. If not
     * called, you should use SSM Session Manager to connect to the instance.
     */
    allowSshAccessFrom(...peer) {
        peer.forEach(p => {
            this.connections.allowFrom(p, port_1.Port.tcp(22), 'SSH access');
        });
    }
}
exports.BastionHostLinux = BastionHostLinux;
//# sourceMappingURL=data:application/json;base64,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