"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CustomCloud9Ssm = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
const cdk = require("@aws-cdk/core");
const ec2 = require("@aws-cdk/aws-ec2");
const cloud9 = require("@aws-cdk/aws-cloud9");
const ssm = require("@aws-cdk/aws-ssm");
const lambda = require("@aws-cdk/aws-lambda");
const iam = require("@aws-cdk/aws-iam");
const yaml = require('yaml');
const fs = require('fs');
/**
 * @experimental
 */
class CustomCloud9Ssm extends cdk.Construct {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id);
        let cloud9Env;
        // Create the Cloud9 environment using the default configuration
        if (!props.cloud9Ec2Props) {
            cloud9Env = new cloud9.Ec2Environment(this, 'Cloud9Ec2Environment', {
                vpc: new ec2.Vpc(this, id + '-VPC', {
                    maxAzs: 2
                })
            });
        }
        // Create the Cloud9 environment using the received props
        else {
            cloud9Env = new cloud9.Ec2Environment(this, 'Cloud9Ec2Environment', props.cloud9Ec2Props);
        }
        // Create a Role for the EC2 instance and an instance profile with it
        this.ec2Role = new iam.Role(this, 'Ec2Role', {
            assumedBy: new iam.ServicePrincipal('ec2.amazonaws.com'),
            roleName: id + '-CustomCloud9SsmEc2Role',
            managedPolicies: [
                iam.ManagedPolicy.fromManagedPolicyArn(this, id + '-SsmManagedPolicy', 'arn:aws:iam::aws:policy/AmazonSSMManagedInstanceCore')
            ]
        });
        const instanceProfile = new iam.CfnInstanceProfile(this, 'Ec2InstanceProfile', {
            roles: [this.ec2Role.roleName]
        });
        // Create the SSM Document using the default configuration
        if (!props.ssmDocumentProps) {
            let content = fs.readFileSync(CustomCloud9Ssm.DEFAULT_DOCUMENT_FILE_NAME, 'utf8');
            const ssmDocumentProps = {
                documentType: 'Command',
                content: yaml.parse(content),
                name: id + '-' + CustomCloud9Ssm.DEFAULT_DOCUMENT_NAME
            };
            this.document = new ssm.CfnDocument(this, 'SsmDocument', ssmDocumentProps);
            this.resizeEBSTo(CustomCloud9Ssm.DEFAULT_EBS_SIZE);
        }
        // Create the SSM Document using the received props
        else {
            if (!props.ssmDocumentProps.name) {
                throw new Error("The document name must be specified.");
            }
            this.document = new ssm.CfnDocument(this, 'SsmDocument', props.ssmDocumentProps);
        }
        // Create an SSM Association to apply the document configuration
        new ssm.CfnAssociation(this, 'SsmAssociation', {
            name: this.document.name,
            targets: [
                {
                    key: 'tag:aws:cloud9:environment',
                    values: [cloud9Env.environmentId]
                }
            ]
        });
        // Create the Lambda function that attaches the instance profile to the EC2 instance
        let code = fs.readFileSync(CustomCloud9Ssm.ATTACH_PROFILE_FILE_NAME, 'utf8');
        const lambdaFunction = new lambda.Function(this, 'LambdaFunction', {
            runtime: lambda.Runtime.PYTHON_3_6,
            code: lambda.Code.fromInline(code),
            handler: 'index.handler',
            timeout: cdk.Duration.seconds(60)
        });
        // Give permissions to the function to execute some APIs
        lambdaFunction.addToRolePolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                'ec2:DescribeInstances',
                'ec2:AssociateIamInstanceProfile',
                'ec2:ReplaceIamInstanceProfileAssociation',
                'ec2:RebootInstances',
                'iam:ListInstanceProfiles',
                'iam:PassRole'
            ],
            resources: ['*']
        }));
        // Create the Custom Resource that invokes the Lambda function
        new cdk.CustomResource(this, 'CustomResource', {
            serviceToken: lambdaFunction.functionArn,
            properties: {
                cloud9_env_id: cloud9Env.environmentId,
                profile_arn: instanceProfile.attrArn
            }
        });
    }
    /**
     * (experimental) Adds one or more steps to the content of the SSM Document.
     *
     * @param steps : YAML formatted string containing one or more steps to be added to the mainSteps section of the SSM Document.
     * @experimental
     */
    addDocumentSteps(steps) {
        // Add the mainSteps section if it doesn't exist
        if (!('mainSteps' in this.document.content)) {
            this.document.content['mainSteps'] = [];
        }
        // Add the new step
        this.document.content['mainSteps'].push(...yaml.parse(steps));
    }
    /**
     * (experimental) Adds one or more parameters to the content of the SSM Document.
     *
     * @param parameters : YAML formatted string containing one or more parameters to be added to the parameters section of the SSM Document.
     * @experimental
     */
    addDocumentParameters(parameters) {
        // Add the parameters section if it doesn't exist
        if (!('parameters' in this.document.content)) {
            this.document.content['parameters'] = {};
        }
        // Add the new parameter
        this.document.content['parameters'] = Object.assign({}, this.document.content['parameters'], yaml.parse(parameters));
    }
    /**
     * (experimental) Adds a step to the SSM Document content that resizes the EBS volume of the EC2 instance.
     *
     * Attaches the required policies to ec2Role.
     *
     * @param size : size in GiB to resize the EBS volume to.
     * @experimental
     */
    resizeEBSTo(size) {
        let steps = fs.readFileSync(CustomCloud9Ssm.RESIZE_STEP_FILE_NAME, 'utf8');
        steps = steps.replace('{{ size }}', String(size));
        // Add the resizing step
        this.addDocumentSteps(steps);
        // Grant permission to the EC2 instance to execute the statements included in the SSM Document
        this.ec2Role.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                'ec2:DescribeInstances',
                'ec2:ModifyVolume',
                'ec2:DescribeVolumesModifications'
            ],
            resources: ['*']
        }));
    }
}
exports.CustomCloud9Ssm = CustomCloud9Ssm;
_a = JSII_RTTI_SYMBOL_1;
CustomCloud9Ssm[_a] = { fqn: "@cdk-use-cases/custom-cloud9-ssm.CustomCloud9Ssm", version: "1.0.3" };
CustomCloud9Ssm.DEFAULT_EBS_SIZE = 100;
CustomCloud9Ssm.DEFAULT_DOCUMENT_FILE_NAME = `${__dirname}/assets/default_document.yml`;
CustomCloud9Ssm.RESIZE_STEP_FILE_NAME = `${__dirname}/assets/resize_ebs_step.yml`;
CustomCloud9Ssm.ATTACH_PROFILE_FILE_NAME = `${__dirname}/assets/profile_attach.py`;
CustomCloud9Ssm.DEFAULT_DOCUMENT_NAME = 'CustomCloudSsm-SsmDocument';
//# sourceMappingURL=data:application/json;base64,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