"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.MirrorRepository = void 0;
const aws_codecommit_1 = require("aws-cdk-lib/aws-codecommit");
const constructs_1 = require("constructs");
const context_1 = require("../util/context");
const aws_codebuild_1 = require("aws-cdk-lib/aws-codebuild");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const customNodejsFunction_1 = require("./customNodejsFunction");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const path = require("path");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const custom_resources_1 = require("aws-cdk-lib/custom-resources");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
class MirrorRepository extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.codeCommitRepository = this.createCodeCommitRepository();
        const { mirrorProject, triggerMirrorFunctionUrl, } = this.createRepositoryMirroring(props.repoTokenParam, props.repository, this.codeCommitRepository);
        this.createWebhook(props.repoTokenParam, props.repository, triggerMirrorFunctionUrl);
        this.triggerInitialMirror(mirrorProject);
    }
    createCodeCommitRepository() {
        return new aws_codecommit_1.Repository(this, 'Repository', {
            repositoryName: context_1.getProjectName(this),
        });
    }
    createRepositoryMirroring(repoTokenParam, repository, codeCommit) {
        const sourceRepositoryDomain = repository.host === 'github' ? 'github.com' : 'bitbucket.org';
        const mirrorProject = new aws_codebuild_1.Project(this, 'RepositoryMirrorProject', {
            projectName: `${aws_cdk_lib_1.Stack.of(this).stackName}-mirrorRepository`,
            timeout: aws_cdk_lib_1.Duration.minutes(20),
            environment: {
                buildImage: aws_codebuild_1.LinuxBuildImage.STANDARD_6_0,
                computeType: aws_codebuild_1.ComputeType.SMALL,
            },
            environmentVariables: {
                REPO_TOKEN: {
                    type: aws_codebuild_1.BuildEnvironmentVariableType.PARAMETER_STORE,
                    value: repoTokenParam.parameterName,
                },
            },
            buildSpec: aws_codebuild_1.BuildSpec.fromObject({
                version: '0.2',
                phases: {
                    install: {
                        commands: [
                            'pip install git-remote-codecommit',
                        ],
                    },
                    build: {
                        commands: [
                            `git clone --mirror https://x-token-auth:$REPO_TOKEN@${sourceRepositoryDomain}/${repository.name}.git repository`,
                            'cd repository',
                            `git push --mirror ${codeCommit.repositoryCloneUrlGrc}`,
                        ],
                    },
                },
            }),
        });
        codeCommit.grantPullPush(mirrorProject);
        repoTokenParam.grantRead(mirrorProject);
        const webhookSecret = aws_cdk_lib_1.Fn.select(2, aws_cdk_lib_1.Fn.split('/', aws_cdk_lib_1.Stack.of(this).stackId));
        const triggerMirrorFunction = new customNodejsFunction_1.CustomNodejsFunction(this, 'TriggerMirrorFunction', {
            code: aws_lambda_1.Code.fromAsset(path.join(__dirname, '..', 'lambda', 'mirrorRepository')),
            timeout: aws_cdk_lib_1.Duration.seconds(30),
            environment: {
                CODEBUILD_PROJECT_NAME: mirrorProject.projectName,
                SECRET: webhookSecret,
            },
        });
        triggerMirrorFunction.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: ['codebuild:StartBuild'],
            resources: [mirrorProject.projectArn],
        }));
        const triggerMirrorFunctionUrl = triggerMirrorFunction.addFunctionUrl({
            authType: aws_lambda_1.FunctionUrlAuthType.NONE,
        });
        return {
            mirrorProject,
            triggerMirrorFunctionUrl: `${triggerMirrorFunctionUrl.url}?secret=${webhookSecret}`,
        };
    }
    createWebhook(repoTokenParam, repository, webhookUrl) {
        const setupWebhooksFunction = new customNodejsFunction_1.CustomNodejsFunction(this, 'SetupWebhooksFunction', {
            code: aws_lambda_1.Code.fromAsset(path.join(__dirname, '..', 'lambda', 'setupWebhooks')),
            timeout: aws_cdk_lib_1.Duration.seconds(30),
        });
        repoTokenParam.grantRead(setupWebhooksFunction);
        const provider = new custom_resources_1.Provider(this, 'WebhookProvider', {
            onEventHandler: setupWebhooksFunction,
            logRetention: aws_logs_1.RetentionDays.ONE_MONTH,
        });
        new aws_cdk_lib_1.CustomResource(this, 'Webhook', {
            serviceToken: provider.serviceToken,
            properties: {
                StackName: aws_cdk_lib_1.Stack.of(this).stackName,
                RepositoryHost: repository.host,
                RepositoryName: repository.name,
                RepositoryTokenParamName: repoTokenParam.parameterName,
                WebhookUrl: webhookUrl,
            },
        });
    }
    triggerInitialMirror(mirrorProject) {
        new custom_resources_1.AwsCustomResource(this, 'TriggerInitialMirror', {
            onCreate: {
                service: 'CodeBuild',
                action: 'startBuild',
                parameters: {
                    projectName: mirrorProject.projectName,
                },
                physicalResourceId: custom_resources_1.PhysicalResourceId.of('1'),
            },
            policy: custom_resources_1.AwsCustomResourcePolicy.fromSdkCalls({
                resources: [mirrorProject.projectArn],
            }),
        });
    }
}
exports.MirrorRepository = MirrorRepository;
//# sourceMappingURL=data:application/json;base64,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