"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.FeatureBranchBuilds = void 0;
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_codebuild_1 = require("aws-cdk-lib/aws-codebuild");
const aws_events_1 = require("aws-cdk-lib/aws-events");
const targets = require("aws-cdk-lib/aws-events-targets");
const aws_events_targets_1 = require("aws-cdk-lib/aws-events-targets");
const constructs_1 = require("constructs");
const customNodejsFunction_1 = require("./customNodejsFunction");
const path = require("path");
const notificationsTopic_1 = require("./notificationsTopic");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const context_1 = require("../util/context");
class FeatureBranchBuilds extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const source = aws_codebuild_1.Source.codeCommit({ repository: props.codeCommitRepository });
        const deployProject = this.createDeployProject(source, props.codeBuild, props.commands, props.codeCommitRepository, props.repository.defaultBranch);
        this.createDeployNotifications(deployProject, props.repository, props.repositoryTokenParam);
        this.failuresTopic = this.createBuildFailuresTopic(deployProject);
        this.createDestroyProject(source, props.codeBuild, props.commands, props.codeCommitRepository, props.repository.defaultBranch);
    }
    createDeployProject(source, codeBuild, commands, repository, defaultBranch) {
        const deployProject = new aws_codebuild_1.Project(this, 'DeployProject', {
            projectName: `${aws_cdk_lib_1.Stack.of(this).stackName}-featureBranch-deploy`,
            source,
            timeout: codeBuild.timeout,
            environment: codeBuild.buildEnvironment,
            vpc: codeBuild.vpc,
            securityGroups: codeBuild.securityGroups,
            subnetSelection: codeBuild.subnetSelection,
            cache: codeBuild.cache,
            buildSpec: aws_codebuild_1.BuildSpec.fromObject({
                version: '0.2',
                phases: {
                    install: {
                        commands: [
                            'ENV_NAME=$(echo ${BRANCH_NAME} | awk \'{print tolower($0)}\')',
                            ...(commands.preInstall || []),
                            ...(commands.install || []),
                        ],
                    },
                    build: {
                        commands: [
                            ...(commands.buildAndTest || []),
                            ...commands.deployEnvironment,
                        ],
                    },
                },
            }),
        });
        codeBuild.rolePolicy?.forEach(policy => deployProject.addToRolePolicy(policy));
        this.grantAssumeCDKRoles(deployProject);
        repository.onCommit('OnBranchCommit', this.createProjectTriggerOptions(deployProject, defaultBranch, true));
        return deployProject;
    }
    createDeployNotifications(deployProject, repository, repoTokenParam) {
        const deployStatusFunction = new customNodejsFunction_1.CustomNodejsFunction(this, 'DeployStatus', {
            code: aws_lambda_1.Code.fromAsset(path.join(__dirname, '..', 'lambda', 'featureBranchDeployStatus')),
            environment: {
                REPOSITORY_HOST: repository.host,
                REPOSITORY_NAME: repository.name,
                REPOSITORY_TOKEN_PARAM_NAME: repoTokenParam.parameterName,
            },
        });
        repoTokenParam.grantRead(deployStatusFunction);
        deployProject.onStateChange('OnDeployStateChange', {
            target: new targets.LambdaFunction(deployStatusFunction),
        });
    }
    createBuildFailuresTopic(deployProject) {
        const failuresTopic = new notificationsTopic_1.NotificationsTopic(this, 'FeatureBranchBuildFailuresTopic', {
            projectName: context_1.getProjectName(this),
            notificationName: 'featureBranchBuildFailures',
        });
        // for better visibility, use EventBridge Rules instead of CodeStar Notifications that are generated with project.notifyOnBuildFailed()
        deployProject.onBuildFailed('OnFeatureBuildFailure', {
            target: new targets.SnsTopic(failuresTopic.topic),
        });
        return failuresTopic.topic;
    }
    createDestroyProject(source, codeBuild, commands, repository, defaultBranch) {
        const destroyProject = new aws_codebuild_1.Project(this, 'DestroyProject', {
            projectName: `${aws_cdk_lib_1.Stack.of(this).stackName}-featureBranch-destroy`,
            source,
            timeout: codeBuild.timeout,
            environment: codeBuild.buildEnvironment,
            vpc: codeBuild.vpc,
            securityGroups: codeBuild.securityGroups,
            subnetSelection: codeBuild.subnetSelection,
            cache: codeBuild.cache,
            buildSpec: aws_codebuild_1.BuildSpec.fromObject({
                version: '0.2',
                phases: {
                    install: {
                        commands: [
                            'ENV_NAME=$(echo ${BRANCH_NAME} | awk \'{print tolower($0)}\')',
                            ...(commands.preInstall || []),
                            ...(commands.install || []),
                        ],
                    },
                    build: {
                        commands: [
                            ...commands.destroyEnvironment,
                        ],
                    },
                },
            }),
        });
        codeBuild.rolePolicy?.forEach(policy => destroyProject.addToRolePolicy(policy));
        this.grantAssumeCDKRoles(destroyProject);
        repository.onReferenceDeleted('OnBranchRemoval', this.createProjectTriggerOptions(destroyProject, defaultBranch));
        return destroyProject;
    }
    createProjectTriggerOptions(targetProject, defaultBranch, withSourceVersion = false) {
        return {
            eventPattern: {
                detail: {
                    referenceType: ['branch'],
                    referenceName: [
                        { 'anything-but': [defaultBranch] },
                    ],
                    referenceFullName: [
                        { 'anything-but': { prefix: 'refs/remotes/' } },
                    ],
                },
            },
            target: new aws_events_targets_1.CodeBuildProject(targetProject, {
                event: aws_events_1.RuleTargetInput.fromObject({
                    sourceVersion: withSourceVersion ? aws_events_1.EventField.fromPath('$.detail.commitId') : undefined,
                    environmentVariablesOverride: [
                        {
                            name: 'BRANCH_NAME',
                            value: aws_events_1.EventField.fromPath('$.detail.referenceName'),
                            type: 'PLAINTEXT',
                        },
                    ],
                }),
            }),
        };
    }
    grantAssumeCDKRoles(project) {
        const qualifier = aws_cdk_lib_1.Stack.of(this).synthesizer.bootstrapQualifier || 'hnb659fds';
        project.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: ['sts:AssumeRole'],
            resources: [
                `arn:aws:iam::${aws_cdk_lib_1.Aws.ACCOUNT_ID}:role/cdk-${qualifier}-deploy-role-${aws_cdk_lib_1.Aws.ACCOUNT_ID}-${aws_cdk_lib_1.Aws.REGION}`,
                `arn:aws:iam::${aws_cdk_lib_1.Aws.ACCOUNT_ID}:role/cdk-${qualifier}-file-publishing-role-${aws_cdk_lib_1.Aws.ACCOUNT_ID}-${aws_cdk_lib_1.Aws.REGION}`,
                `arn:aws:iam::${aws_cdk_lib_1.Aws.ACCOUNT_ID}:role/cdk-${qualifier}-image-publishing-role-${aws_cdk_lib_1.Aws.ACCOUNT_ID}-${aws_cdk_lib_1.Aws.REGION}`,
                `arn:aws:iam::${aws_cdk_lib_1.Aws.ACCOUNT_ID}:role/cdk-${qualifier}-lookup-role-${aws_cdk_lib_1.Aws.ACCOUNT_ID}-${aws_cdk_lib_1.Aws.REGION}`,
            ],
        }));
    }
}
exports.FeatureBranchBuilds = FeatureBranchBuilds;
//# sourceMappingURL=data:application/json;base64,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