"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Volume = exports.EbsDeviceVolumeType = exports.BlockDeviceVolume = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const crypto = require("crypto");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_kms_1 = require("@aws-cdk/aws-kms");
const core_1 = require("@aws-cdk/core");
const ec2_generated_1 = require("./ec2.generated");
/**
 * Describes a block device mapping for an EC2 instance or Auto Scaling group.
 *
 * @stability stable
 */
class BlockDeviceVolume {
    /**
     * @param ebsDevice EBS device info.
     * @param virtualName Virtual device name.
     * @stability stable
     */
    constructor(ebsDevice, virtualName) {
        this.ebsDevice = ebsDevice;
        this.virtualName = virtualName;
    }
    /**
     * Creates a new Elastic Block Storage device.
     *
     * @param volumeSize The volume size, in Gibibytes (GiB).
     * @param options additional device options.
     * @stability stable
     */
    static ebs(volumeSize, options = {}) {
        return new this({ ...options, volumeSize });
    }
    /**
     * Creates a new Elastic Block Storage device from an existing snapshot.
     *
     * @param snapshotId The snapshot ID of the volume to use.
     * @param options additional device options.
     * @stability stable
     */
    static ebsFromSnapshot(snapshotId, options = {}) {
        return new this({ ...options, snapshotId });
    }
    /**
     * Creates a virtual, ephemeral device.
     *
     * The name will be in the form ephemeral{volumeIndex}.
     *
     * @param volumeIndex the volume index.
     * @stability stable
     */
    static ephemeral(volumeIndex) {
        if (volumeIndex < 0) {
            throw new Error(`volumeIndex must be a number starting from 0, got "${volumeIndex}"`);
        }
        return new this(undefined, `ephemeral${volumeIndex}`);
    }
}
exports.BlockDeviceVolume = BlockDeviceVolume;
_a = JSII_RTTI_SYMBOL_1;
BlockDeviceVolume[_a] = { fqn: "@aws-cdk/aws-ec2.BlockDeviceVolume", version: "1.94.0" };
/**
 * Supported EBS volume types for blockDevices.
 *
 * @stability stable
 */
var EbsDeviceVolumeType;
(function (EbsDeviceVolumeType) {
    EbsDeviceVolumeType["STANDARD"] = "standard";
    EbsDeviceVolumeType["IO1"] = "io1";
    EbsDeviceVolumeType["IO2"] = "io2";
    EbsDeviceVolumeType["GP2"] = "gp2";
    EbsDeviceVolumeType["GP3"] = "gp3";
    EbsDeviceVolumeType["ST1"] = "st1";
    EbsDeviceVolumeType["SC1"] = "sc1";
    /**
     * General purpose SSD volume (GP2) that balances price and performance for a wide variety of workloads.
     */
    EbsDeviceVolumeType["GENERAL_PURPOSE_SSD"] = "gp2";
    /**
     * General purpose SSD volume (GP3) that balances price and performance for a wide variety of workloads.
     */
    EbsDeviceVolumeType["GENERAL_PURPOSE_SSD_GP3"] = "gp3";
    /**
     * Highest-performance SSD volume (IO1) for mission-critical low-latency or high-throughput workloads.
     */
    EbsDeviceVolumeType["PROVISIONED_IOPS_SSD"] = "io1";
    /**
     * Highest-performance SSD volume (IO2) for mission-critical low-latency or high-throughput workloads.
     */
    EbsDeviceVolumeType["PROVISIONED_IOPS_SSD_IO2"] = "io2";
    /**
     * Low-cost HDD volume designed for frequently accessed, throughput-intensive workloads.
     */
    EbsDeviceVolumeType["THROUGHPUT_OPTIMIZED_HDD"] = "st1";
    /**
     * Lowest cost HDD volume designed for less frequently accessed workloads.
     */
    EbsDeviceVolumeType["COLD_HDD"] = "sc1";
    /**
     * Magnetic volumes are backed by magnetic drives and are suited for workloads where data is accessed infrequently, and scenarios where low-cost
     * storage for small volume sizes is important.
     */
    EbsDeviceVolumeType["MAGNETIC"] = "standard";
})(EbsDeviceVolumeType = exports.EbsDeviceVolumeType || (exports.EbsDeviceVolumeType = {}));
/**
 * Common behavior of Volumes. Users should not use this class directly, and instead use ``Volume``.
 */
class VolumeBase extends core_1.Resource {
    /**
     * Grants permission to attach this Volume to an instance.
     *
     * CAUTION: Granting an instance permission to attach to itself using this method will lead to
     * an unresolvable circular reference between the instance role and the instance.
     * Use {@link IVolume.grantAttachVolumeToSelf} to grant an instance permission to attach this
     * volume to itself.
     *
     * @stability stable
     */
    grantAttachVolume(grantee, instances) {
        const result = aws_iam_1.Grant.addToPrincipal({
            grantee,
            actions: ['ec2:AttachVolume'],
            resourceArns: this.collectGrantResourceArns(instances),
        });
        if (this.encryptionKey) {
            // When attaching a volume, the EC2 Service will need to grant to itself permission
            // to be able to decrypt the encryption key. We restrict the CreateGrant for principle
            // of least privilege, in accordance with best practices.
            // See: https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/EBSEncryption.html#ebs-encryption-permissions
            const kmsGrant = this.encryptionKey.grant(grantee, 'kms:CreateGrant');
            kmsGrant.principalStatement.addConditions({
                Bool: { 'kms:GrantIsForAWSResource': true },
                StringEquals: {
                    'kms:ViaService': `ec2.${core_1.Stack.of(this).region}.amazonaws.com`,
                    'kms:GrantConstraintType': 'EncryptionContextSubset',
                },
            });
        }
        return result;
    }
    /**
     * Grants permission to attach the Volume by a ResourceTag condition.
     *
     * If you are looking to
     * grant an Instance, AutoScalingGroup, EC2-Fleet, SpotFleet, ECS host, etc the ability to attach
     * this volume to **itself** then this is the method you want to use.
     *
     * This is implemented by adding a Tag with key `VolumeGrantAttach-<suffix>` to the given
     * constructs and this Volume, and then conditioning the Grant such that the grantee is only
     * given the ability to AttachVolume if both the Volume and the destination Instance have that
     * tag applied to them.
     *
     * @stability stable
     */
    grantAttachVolumeByResourceTag(grantee, constructs, tagKeySuffix) {
        const tagValue = this.calculateResourceTagValue([this, ...constructs]);
        const tagKey = `VolumeGrantAttach-${tagKeySuffix !== null && tagKeySuffix !== void 0 ? tagKeySuffix : tagValue.slice(0, 10).toUpperCase()}`;
        const grantCondition = {};
        grantCondition[`ec2:ResourceTag/${tagKey}`] = tagValue;
        const result = this.grantAttachVolume(grantee);
        result.principalStatement.addCondition('ForAnyValue:StringEquals', grantCondition);
        // The ResourceTag condition requires that all resources involved in the operation have
        // the given tag, so we tag this and all constructs given.
        core_1.Tags.of(this).add(tagKey, tagValue);
        constructs.forEach(construct => core_1.Tags.of(construct).add(tagKey, tagValue));
        return result;
    }
    /**
     * Grants permission to detach this Volume from an instance CAUTION: Granting an instance permission to detach from itself using this method will lead to an unresolvable circular reference between the instance role and the instance.
     *
     * Use {@link IVolume.grantDetachVolumeFromSelf} to grant an instance permission to detach this
     * volume from itself.
     *
     * @stability stable
     */
    grantDetachVolume(grantee, instances) {
        const result = aws_iam_1.Grant.addToPrincipal({
            grantee,
            actions: ['ec2:DetachVolume'],
            resourceArns: this.collectGrantResourceArns(instances),
        });
        // Note: No encryption key permissions are required to detach an encrypted volume.
        return result;
    }
    /**
     * Grants permission to detach the Volume by a ResourceTag condition.
     *
     * This is implemented via the same mechanism as {@link IVolume.grantAttachVolumeByResourceTag},
     * and is subject to the same conditions.
     *
     * @stability stable
     */
    grantDetachVolumeByResourceTag(grantee, constructs, tagKeySuffix) {
        const tagValue = this.calculateResourceTagValue([this, ...constructs]);
        const tagKey = `VolumeGrantDetach-${tagKeySuffix !== null && tagKeySuffix !== void 0 ? tagKeySuffix : tagValue.slice(0, 10).toUpperCase()}`;
        const grantCondition = {};
        grantCondition[`ec2:ResourceTag/${tagKey}`] = tagValue;
        const result = this.grantDetachVolume(grantee);
        result.principalStatement.addCondition('ForAnyValue:StringEquals', grantCondition);
        // The ResourceTag condition requires that all resources involved in the operation have
        // the given tag, so we tag this and all constructs given.
        core_1.Tags.of(this).add(tagKey, tagValue);
        constructs.forEach(construct => core_1.Tags.of(construct).add(tagKey, tagValue));
        return result;
    }
    collectGrantResourceArns(instances) {
        const stack = core_1.Stack.of(this);
        const resourceArns = [
            `arn:${stack.partition}:ec2:${stack.region}:${stack.account}:volume/${this.volumeId}`,
        ];
        const instanceArnPrefix = `arn:${stack.partition}:ec2:${stack.region}:${stack.account}:instance`;
        if (instances) {
            instances.forEach(instance => resourceArns.push(`${instanceArnPrefix}/${instance === null || instance === void 0 ? void 0 : instance.instanceId}`));
        }
        else {
            resourceArns.push(`${instanceArnPrefix}/*`);
        }
        return resourceArns;
    }
    calculateResourceTagValue(constructs) {
        const md5 = crypto.createHash('md5');
        constructs.forEach(construct => md5.update(core_1.Names.uniqueId(construct)));
        return md5.digest('hex');
    }
}
/**
 * Creates a new EBS Volume in AWS EC2.
 *
 * @stability stable
 */
class Volume extends VolumeBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _c, _d, _e, _f, _g;
        super(scope, id, {
            physicalName: props.volumeName,
        });
        this.validateProps(props);
        const resource = new ec2_generated_1.CfnVolume(this, 'Resource', {
            availabilityZone: props.availabilityZone,
            autoEnableIo: props.autoEnableIo,
            encrypted: props.encrypted,
            kmsKeyId: (_c = props.encryptionKey) === null || _c === void 0 ? void 0 : _c.keyArn,
            iops: props.iops,
            multiAttachEnabled: (_d = props.enableMultiAttach) !== null && _d !== void 0 ? _d : false,
            size: (_e = props.size) === null || _e === void 0 ? void 0 : _e.toGibibytes({ rounding: core_1.SizeRoundingBehavior.FAIL }),
            snapshotId: props.snapshotId,
            volumeType: (_f = props.volumeType) !== null && _f !== void 0 ? _f : EbsDeviceVolumeType.GENERAL_PURPOSE_SSD,
        });
        resource.applyRemovalPolicy(props.removalPolicy);
        this.volumeId = resource.ref;
        this.availabilityZone = props.availabilityZone;
        this.encryptionKey = props.encryptionKey;
        if (this.encryptionKey) {
            // Per: https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/EBSEncryption.html#ebs-encryption-requirements
            const principal = new aws_kms_1.ViaServicePrincipal(`ec2.${core_1.Stack.of(this).region}.amazonaws.com`, new aws_iam_1.AccountRootPrincipal()).withConditions({
                StringEquals: {
                    'kms:CallerAccount': core_1.Stack.of(this).account,
                },
            });
            const grant = this.encryptionKey.grant(principal, 
            // Describe & Generate are required to be able to create the CMK-encrypted Volume.
            'kms:DescribeKey', 'kms:GenerateDataKeyWithoutPlainText');
            if (props.snapshotId) {
                // ReEncrypt is required for when re-encrypting from an encrypted snapshot.
                (_g = grant.principalStatement) === null || _g === void 0 ? void 0 : _g.addActions('kms:ReEncrypt*');
            }
        }
    }
    /**
     * Import an existing EBS Volume into the Stack.
     *
     * @param scope the scope of the import.
     * @param id the ID of the imported Volume in the construct tree.
     * @param attrs the attributes of the imported Volume.
     * @stability stable
     */
    static fromVolumeAttributes(scope, id, attrs) {
        class Import extends VolumeBase {
            constructor() {
                super(...arguments);
                this.volumeId = attrs.volumeId;
                this.availabilityZone = attrs.availabilityZone;
                this.encryptionKey = attrs.encryptionKey;
            }
        }
        // Check that the provided volumeId looks like it could be valid.
        if (!core_1.Token.isUnresolved(attrs.volumeId) && !/^vol-[0-9a-fA-F]+$/.test(attrs.volumeId)) {
            throw new Error('`volumeId` does not match expected pattern. Expected `vol-<hexadecmial value>` (ex: `vol-05abe246af`) or a Token');
        }
        return new Import(scope, id);
    }
    /**
     * @stability stable
     */
    validateProps(props) {
        var _c, _d, _e;
        if (!(props.size || props.snapshotId)) {
            throw new Error('Must provide at least one of `size` or `snapshotId`');
        }
        if (props.snapshotId && !core_1.Token.isUnresolved(props.snapshotId) && !/^snap-[0-9a-fA-F]+$/.test(props.snapshotId)) {
            throw new Error('`snapshotId` does match expected pattern. Expected `snap-<hexadecmial value>` (ex: `snap-05abe246af`) or Token');
        }
        if (props.encryptionKey && !props.encrypted) {
            throw new Error('`encrypted` must be true when providing an `encryptionKey`.');
        }
        if (props.volumeType &&
            [
                EbsDeviceVolumeType.PROVISIONED_IOPS_SSD,
                EbsDeviceVolumeType.PROVISIONED_IOPS_SSD_IO2,
            ].includes(props.volumeType) &&
            !props.iops) {
            throw new Error('`iops` must be specified if the `volumeType` is `PROVISIONED_IOPS_SSD` or `PROVISIONED_IOPS_SSD_IO2`.');
        }
        if (props.iops) {
            const volumeType = (_c = props.volumeType) !== null && _c !== void 0 ? _c : EbsDeviceVolumeType.GENERAL_PURPOSE_SSD;
            if (![
                EbsDeviceVolumeType.PROVISIONED_IOPS_SSD,
                EbsDeviceVolumeType.PROVISIONED_IOPS_SSD_IO2,
                EbsDeviceVolumeType.GENERAL_PURPOSE_SSD_GP3,
            ].includes(volumeType)) {
                throw new Error('`iops` may only be specified if the `volumeType` is `PROVISIONED_IOPS_SSD`, `PROVISIONED_IOPS_SSD_IO2` or `GENERAL_PURPOSE_SSD_GP3`.');
            }
            // Enforce minimum & maximum IOPS:
            // https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ebs-volume.html
            const iopsRanges = {};
            iopsRanges[EbsDeviceVolumeType.GENERAL_PURPOSE_SSD_GP3] = { Min: 3000, Max: 16000 };
            iopsRanges[EbsDeviceVolumeType.PROVISIONED_IOPS_SSD] = { Min: 100, Max: 64000 };
            iopsRanges[EbsDeviceVolumeType.PROVISIONED_IOPS_SSD_IO2] = { Min: 100, Max: 64000 };
            const { Min, Max } = iopsRanges[volumeType];
            if (props.iops < Min || props.iops > Max) {
                throw new Error(`\`${volumeType}\` volumes iops must be between ${Min} and ${Max}.`);
            }
            // Enforce maximum ratio of IOPS/GiB:
            // https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ebs-volume-types.html
            const maximumRatios = {};
            maximumRatios[EbsDeviceVolumeType.GENERAL_PURPOSE_SSD_GP3] = 500;
            maximumRatios[EbsDeviceVolumeType.PROVISIONED_IOPS_SSD] = 50;
            maximumRatios[EbsDeviceVolumeType.PROVISIONED_IOPS_SSD_IO2] = 500;
            const maximumRatio = maximumRatios[volumeType];
            if (props.size && (props.iops > maximumRatio * props.size.toGibibytes({ rounding: core_1.SizeRoundingBehavior.FAIL }))) {
                throw new Error(`\`${volumeType}\` volumes iops has a maximum ratio of ${maximumRatio} IOPS/GiB.`);
            }
        }
        if (props.enableMultiAttach) {
            const volumeType = (_d = props.volumeType) !== null && _d !== void 0 ? _d : EbsDeviceVolumeType.GENERAL_PURPOSE_SSD;
            if (![
                EbsDeviceVolumeType.PROVISIONED_IOPS_SSD,
                EbsDeviceVolumeType.PROVISIONED_IOPS_SSD_IO2,
            ].includes(volumeType)) {
                throw new Error('multi-attach is supported exclusively on `PROVISIONED_IOPS_SSD` and `PROVISIONED_IOPS_SSD_IO2` volumes.');
            }
        }
        if (props.size) {
            const size = props.size.toGibibytes({ rounding: core_1.SizeRoundingBehavior.FAIL });
            // Enforce minimum & maximum volume size:
            // https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ebs-volume.html
            const sizeRanges = {};
            sizeRanges[EbsDeviceVolumeType.GENERAL_PURPOSE_SSD] = { Min: 1, Max: 16384 };
            sizeRanges[EbsDeviceVolumeType.GENERAL_PURPOSE_SSD_GP3] = { Min: 1, Max: 16384 };
            sizeRanges[EbsDeviceVolumeType.PROVISIONED_IOPS_SSD] = { Min: 4, Max: 16384 };
            sizeRanges[EbsDeviceVolumeType.PROVISIONED_IOPS_SSD_IO2] = { Min: 4, Max: 16384 };
            sizeRanges[EbsDeviceVolumeType.THROUGHPUT_OPTIMIZED_HDD] = { Min: 125, Max: 16384 };
            sizeRanges[EbsDeviceVolumeType.COLD_HDD] = { Min: 125, Max: 16384 };
            sizeRanges[EbsDeviceVolumeType.MAGNETIC] = { Min: 1, Max: 1024 };
            const volumeType = (_e = props.volumeType) !== null && _e !== void 0 ? _e : EbsDeviceVolumeType.GENERAL_PURPOSE_SSD;
            const { Min, Max } = sizeRanges[volumeType];
            if (size < Min || size > Max) {
                throw new Error(`\`${volumeType}\` volumes must be between ${Min} GiB and ${Max} GiB in size.`);
            }
        }
    }
}
exports.Volume = Volume;
_b = JSII_RTTI_SYMBOL_1;
Volume[_b] = { fqn: "@aws-cdk/aws-ec2.Volume", version: "1.94.0" };
//# sourceMappingURL=data:application/json;base64,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