# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['pynbs']

package_data = \
{'': ['*']}

setup_kwargs = {
    'name': 'pynbs',
    'version': '1.0.0b1',
    'description': 'A simple python library to read and write .nbs files from Note Block Studio',
    'long_description': "# pynbs\n\n[![PyPI](https://img.shields.io/pypi/v/pynbs.svg)](https://pypi.org/project/pynbs/)\n[![PyPI - Python Version](https://img.shields.io/pypi/pyversions/pynbs.svg)](https://pypi.org/project/pynbs/)\n\n> A simple python library to read and write [.nbs files](https://opennbs.org/nbs)\n> from [Open Note Block Studio](https://opennbs.org/). Compatible with\n> python 2 and 3.\n\n`pynbs` makes it possible to easily iterate over Note Block Studio songs.\n\n```python\nimport pynbs\n\nfor tick, chord in pynbs.read('demo_song.nbs'):\n    print(tick, [note.key for note in chord])\n```\n\nYou can also use `pynbs` to generate new songs programmatically.\n\n```python\nimport pynbs\n\nnew_file = pynbs.new_file(song_name='Hello world')\nnew_file.notes.extend([\n    pynbs.Note(tick=i, layer=0, instrument=0, key=i + 35) for i in range(10)\n])\n\nnew_file.save('new_file.nbs')\n```\n\n## Installation\n\nThe package can be installed with `pip`.\n\n```bash\n$ pip install pynbs\n```\n\nThe latest release follows the latest version of the NBS file format\n[specification](https://opennbs.org/nbs)\n(version 5). However, it also allows you to load and save files in any of\nthe older versions.\n\n## Basic usage\n\n### Reading files\n\nYou can use the `read()` function to read and parse a specific NBS file.\n\n```python\ndemo_song = pynbs.read('demo_song.nbs')\n```\n\nThe `read()` function returns a `pynbs` file object. These objects have several\nattributes that mirror the binary structure of NBS files.\n\n#### Header\n\nThe first attribute is `header`, the file header. It contains information about\nthe file.\n\n```python\nheader = demo_song.header\n```\n\nAttribute                   | Type    | Details\n:---------------------------|:--------|:------------------------------------------------\n`header.version`            | `int`   | The NBS version this file was saved on.\n`header.default_instruments`| `int`   | The amount of instruments from vanilla Minecraft in the song.\n`header.song_length`        | `int`   | The length of the song, measured in ticks.\n`header.song_layers`        | `int`   | The ID of the last layer with at least one note block in it.\n`header.song_name`          | `str`   | The name of the song.\n`header.song_author`        | `str`   | The author of the song.\n`header.original_author`    | `str`   | The original song author of the song.\n`header.description`        | `str`   | The description of the song.\n`header.tempo`              | `float` | The tempo of the song.\n`header.auto_save`          | `bool`  | Whether auto-saving has been enabled.\n`header.auto_save_duration` | `int`   | The amount of minutes between each auto-save.\n`header.time_signature`     | `int`   | The time signature of the song.\n`header.minutes_spent`      | `int`   | The amount of minutes spent on the project.\n`header.left_clicks`        | `int`   | The amount of times the user has left-clicked.\n`header.right_clicks`       | `int`   | The amount of times the user has right-clicked.\n`header.blocks_added`       | `int`   | The amount of times the user has added a block.\n`header.blocks_removed`     | `int`   | The amount of times the user has removed a block.\n`header.song_origin`        | `str`   | The file name of the original MIDI or schematic.\n`header.loop`               | `bool`  | Whether the song should loop back to the start after ending.\n`header.max_loop_count`     | `int`   | The amount of times to loop. 0 = infinite.\n`header.loop_start`         | `int`   | The tick the song will loop back to at the end of playback.\n\n> For more information about all these fields, check out the [official specification](https://hielkeminecraft.github.io/OpenNoteBlockStudio/nbs).\n\n#### Notes\n\nThe `notes` attribute holds a list of all the notes of the song in order.\n\n```python\nfirst_note = demo_song.notes[0]\n```\n\nAttribute         | Type  | Details\n:---------------- |:------|:------------------------------------------------\n`note.tick`       | `int` | The tick at which the note plays.\n`note.layer`      | `int` | The ID of the layer in which the note is placed.\n`note.instrument` | `int` | The ID of the instrument.\n`note.key`        | `int` | The key of the note. (between 0 and 87)\n`note.velocity`   | `int` | The velocity of the note. (between 0 and 100)\n`note.panning`    | `int` | The stereo panning of the note. (between -100 and 100)\n`note.pitch`      | `int` | The detune of the note, in cents. (between -1200 and 1200)\n\n#### Layers\n\nThe `layers` attribute holds a list of all the layers of the song in order.\n\n```python\nfirst_layer = demo_song.layers[0]\n```\n\nAttribute         | Type  | Details\n:-----------------|:------|:------------------------\n`layer.id`        | `int` | The ID of the layer.\n`layer.name`      | `str` | The name of the layer.\n`layer.lock`      | `bool`| Whether the layer is locked.\n`layer.volume`    | `int` | The volume of the layer.\n`layer.panning`   | `int` | The stereo panning of the layer.\n\n#### Instruments\n\nThe `instruments` attribute holds a list of all the custom instruments of the\nsong in order.\n\n```python\nfirst_custom_instrument = demo_song.instruments[0]\n```\n\nAttribute              | Type   | Details\n:----------------------|:-------|:----------------------------------------------------------\n`instrument.id`        | `int`  | The ID of the instrument.\n`instrument.name`      | `str`  | The name of the instrument.\n`instrument.file`      | `str`  | The name of the sound file of the instrument.\n`instrument.pitch`     | `int`  | The pitch of the instrument. (between 0 and 87)\n`instrument.press_key` | `bool` | Whether the piano should automatically press keys with the instrument when the marker passes them.\n\n### Iterating over songs\n\nIterating over a `pynbs` file object yields consecutively all the chords of the song with\nthe associated tick.\n\n```python\nfor tick, chord in demo_song:\n    ...\n```\n\n`chord` is a list of all the notes that play during the tick `tick`.\n\n### Creating new files\n\nYou can create new files using the `new_file()` function. The function lets\nyou specify header attributes with keyword arguments.\n\n```python\nnew_file = pynbs.new_file(song_name='Hello world')\n```\n\nThe function returns a new `pynbs` file object that you can now edit\nprogrammatically.\n\n### Saving files\n\nYou can use the `save()` method to encode and write the file to a specified\nlocation.\n\n```python\nnew_file.save('new_file.nbs')\n```\n\nBy default, the file will be saved in the latest NBS version available.\nTo save the file in an older version, you can use the `version` parameter:\n\n```python\n# This will save the song in the classic format.\nnew_file.save('new_file.nbs', version=0)\n```\n\n(Keep in mind some of the song properties may be lost when saving in older versions.)\n\n### Upgrading old files\n\nWhile `pynbs` is up-to-date with the latest version of the Open Note Block Studio\nspecification, all previous versions — including the original file format — are still\nsupported by the `read()` function, making it possible to bulk upgrade songs to the\nmost recent version:\n\n```python\nimport glob\nimport pynbs\n\nfor old_file in glob.glob('*.nbs'):\n    pynbs.read(old_file).save(old_file)\n```\n\n---\n\nLicense - [MIT](https://github.com/OpenNBS/pynbs/blob/master/LICENSE)\n",
    'author': 'Valentin Berlier',
    'author_email': 'berlier.v@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/OpenNBS/pynbs',
    'packages': packages,
    'package_data': package_data,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
