import { RemovalPolicy } from 'aws-cdk-lib';
import * as cf from 'aws-cdk-lib/aws-cloudfront';
import * as lambda from 'aws-cdk-lib/aws-lambda';
import { Construct } from 'constructs';
/**
 * Represents a MicroApps Edge to Origin Function
 */
export interface IMicroAppsEdgeToOrigin {
    /**
     * The edge to origin function for API Gateway Request Origin Edge Lambda
     *
     * The generated `config.yml` is included in the Lambda's code.
     */
    readonly edgeToOriginFunction: lambda.Function | cf.experimental.EdgeFunction;
    /**
     * Configuration of the edge to origin lambda functions
     */
    readonly edgeToOriginLambdas: cf.EdgeLambda[];
}
/**
 * Properties to initialize an instance of `MicroAppsEdgeToOrigin`.
 */
export interface MicroAppsEdgeToOriginProps {
    /**
     * RemovalPolicy override for child resources
     *
     * @default - per resource default
     */
    readonly removalPolicy?: RemovalPolicy;
    /**
     * Optional asset name root
     *
     * @example microapps
     * @default - resource names auto assigned
     */
    readonly assetNameRoot?: string;
    /**
     * Optional asset name suffix
     *
     * @example -dev-pr-12
     * @default none
     */
    readonly assetNameSuffix?: string;
    /**
     * Path prefix on the root of the API Gateway Stage
     *
     * @example dev/
     * @default none
     */
    readonly rootPathPrefix?: string;
    /**
     * Adds an X-Forwarded-Host-Header when calling API Gateway
     *
     * Can only be trusted if `signingMode` is enabled, which restricts
     * access to API Gateway to only IAM signed requests.
     *
     * Note: if true, creates OriginRequest Lambda @ Edge function for API Gateway Origin
     * @default true
     */
    readonly addXForwardedHostHeader?: boolean;
    /**
     * Replaces Host header (which will be the Edge domain name) with the Origin domain name
     * when enabled.  This is necessary when API Gateway has not been configured
     * with a custom domain name that matches the exact domain name used by the CloudFront
     * Distribution AND when the OriginRequestPolicy.HeadersBehavior is set
     * to pass all headers to the origin.
     *
     * Note: if true, creates OriginRequest Lambda @ Edge function for API Gateway Origin
     * @default true
     */
    readonly replaceHostHeader?: boolean;
    /**
     * Requires IAM auth on the API Gateway origin if not set to 'none'.
     *
     * 'sign' - Uses request headers for auth.
     * 'presign' - Uses query string for auth.
     *
     * If enabled,
     *
     * Note: if 'sign' or 'presign', creates OriginRequest Lambda @ Edge function for API Gateway Origin
     * @default 'sign'
     */
    readonly signingMode?: 'sign' | 'presign' | 'none';
    /**
     * Origin region that API Gateway will be deployed to, used
     * for the config.yml on the Edge function to sign requests for
     * the correct region
     *
     * @default undefined
     */
    readonly originRegion?: string;
    /**
     * DynamoDB Table Name for apps/versions/rules.
     *
     * Must be a full ARN as this can be cross region.
     *
     * Implies that 2nd generation routing is enabled.
     */
    readonly tableRulesArn?: string;
    /**
     * Enable invoking API Gateway from the Edge Lambda
     *
     * @default false
     */
    readonly setupApiGatewayPermissions?: boolean;
}
export interface GenerateEdgeToOriginConfigOptions {
    readonly originRegion: string;
    readonly signingMode: 'sign' | 'presign' | '';
    readonly addXForwardedHostHeader: boolean;
    readonly replaceHostHeader: boolean;
    readonly tableName?: string;
    readonly rootPathPrefix?: string;
}
/**
 * Create a new MicroApps Edge to Origin Function w/ `config.yml`
 */
export declare class MicroAppsEdgeToOrigin extends Construct implements IMicroAppsEdgeToOrigin {
    /**
     * Generate the yaml config for the edge lambda
     * @param props
     * @returns
     */
    static generateEdgeToOriginConfig(props: GenerateEdgeToOriginConfigOptions): string;
    private _edgeToOriginFunction;
    get edgeToOriginFunction(): lambda.Function | cf.experimental.EdgeFunction;
    private _edgeToOriginLambdas;
    get edgeToOriginLambdas(): cf.EdgeLambda[];
    constructor(scope: Construct, id: string, props: MicroAppsEdgeToOriginProps);
    /**
     * Hash the stack name to make the EdgeFunction parameter name unique
     *
     * @param stack
     * @returns
     */
    private hashStackName;
    private createEdgeFunction;
}
//# sourceMappingURL=MicroAppsEdgeToOrigin.d.ts.map